/*
** mailcap.c
**	A library to access the "mailcap" files as described in: RFC 1524
**	"A User Agent Configuration Mechanism For Multimedia Mail Format
**	Information" (N Borenstein)
**
** Copyright 1995 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include <stdio.h>
#include <ctype.h>
#include <string.h>

#include <stdlib.h>
#if SYSV_INCLUDES
#	include <malloc.h>
#	include <memory.h>
extern char *getenv();
#else
#if ANSI_INCLUDES
#	include <stddef.h>
#	include <stdlib.h>
#else
extern char *malloc();
extern void free();
extern char *getenv();
#endif
#endif

#include <sys/stat.h>
#include <pwd.h>
#include <X11/Intrinsic.h>

#include "filename.h"
#include "mailcap.h"

#undef tables_h
#define ENUM_TABLE_GENERATION
#include "tables.h"

static char mailcap_environment[] = "MAILCAPS";
static char mailcap_path[] =
	"~/.mailcap:/etc/mailcap:/usr/etc/mailcap:/usr/local/etc/mailcap";

typedef struct MailcapChunk
    {
	XrmQuark tag;
	unsigned char size;
	char string[1];		/* Max length < 255 * sizeof(MailcapChunk) */
    } MailcapChunk;

static XrmQuark wild_card, test_field, needsterminal, copiousoutput;
static MailcapChunk *mailcap;	/* mailcap information */

typedef struct MailcapFile
    {
	struct MailcapFile *next;
	int size;
	/* struct hack name[1], must be the last field */
	char name[1];
    } MailcapFile;

/*
** The program reads all mailcap files into a single allocated buffer
** and builds the internal parsed binary data (MailcapChunk) database
** into the same buffer (overwriting the source text in the process).
** It assumed (hopefully correctly), that the binary form is more
** compact than the textual form. To be safe, SAFE_CHUNKS defines an
** extra allocation that is made.
*/
#define SAFE_CHUNKS 20

/*
** Next
**	Advance pointer to the next character (and unfold continuation lines)
*/
static char *Next(char *s)
    {
	if (*s)
	    {
		++s;
		while (*s == '\\' && s[1] == '\n')
			s += 2;
	    }
	return s;
    }

/*
** Scan
**	If the next input char matches the given character, then skip
**	that character and any following white space (except NEWLINE).
*/
static int Scan(char **s, int c)
    {
	if (**s == c)
	    {
		while ((c = *(*s = Next(*s))) != '\n' && c && c <= ' ')
			/* Nothing here */;
		return 1;
	    }
	return 0;
    }

/*
** ParseQuark
**	extract a token and return a XrmQuark
*/
static XrmQuark ParseQuark(char **ptr)
    {
	char *r = *ptr;
	char *s, *str;
	int save;
	XrmQuark quark;

	if (*r == '"')
	    {
		/* Quoted String */
		for (str = s = r++; *r && *r != '"'; r = Next(r))
			*s++ = isupper(*r) ? tolower(*r) : *r;
		Scan(&r, '"');
	    }
	else
	    {
		for (str = s = r; isalnum(*r) || *r == '-'; r = Next(r))
			*s++ = isupper(*r) ? tolower(*r) : *r;
		if (*r <= ' ' && *r != '\n')
			Scan(&r, *r);
	    }
	save = *s;
	*s = 0;
	quark = XrmStringToQuark(str);
	*s = save;
	*ptr = r;
	return quark;
    }

/*
** MakeChunk
*/
static MailcapChunk init_chunk;

static MailcapChunk *MakeChunk(MailcapChunk *chunk)
    {
	chunk += chunk->size;
	*chunk = init_chunk;
	chunk->size = 1;
	return chunk;
    }

/*
** AddStringToChunk
*/
static char *AddStringToChunk(char *r, MailcapChunk *chunk)
    {
	char *s = chunk->string;
	
	while (*r && *r != '\n' && *r != ';')
	    {
		if (*r == '\\' && r[1])
			*s++ = *r++;
		*s++ = *r++;
	    }
	*s++ = 0;
	chunk->size = ((s - (char *)chunk) + sizeof(MailcapChunk) - 1)
		/ sizeof(MailcapChunk);
	return r;
    }	

/*
** MailcapLoad
**	load mailcap information into memory (all mailcaps found from
**	the path will be concatenated!!!)
*/
static void MailcapLoad()
    {

	char *path, *r;
	MailcapFile *files, **last;
	MailcapChunk *chunk, *entry;

	int total, n;

	if (mailcap)
		free((char *)mailcap);
	mailcap = NULL;

	wild_card = XrmStringToQuark("*");
	test_field = XrmStringToQuark("test");
	needsterminal = XrmStringToQuark("needsterminal");
	copiousoutput = XrmStringToQuark("copiousoutput");

	if (!(path = getenv(mailcap_environment)))
		path = mailcap_path;
	/*
	** Extract files form the search path and find their sizes
	*/
	for (total = 1, last = &files; ; )
	    {
		struct stat info;
		char *name;

		if (!(r = strchr(path, ':')))
			r = path + strlen(path);
		name = FileName(path, r, r, r);
		if (name && stat(name, &info) == 0)
		    {
			*last = (MailcapFile *)
				malloc(sizeof(MailcapFile) + strlen(name));
			if (*last == NULL)
				break;
			strcpy((*last)->name, name);
			(*last)->size = info.st_size;
			total += (*last)->size + 1;
			last = &(*last)->next;
		    }
		if (*r == 0)
			break;
		path = ++r;
	    }
	*last = NULL;

	/*
	** Allocate space for all mailcap files and read them in
	** (make sure each of them terminates with NEWLINE!)
	*/
	total += sizeof(MailcapChunk) * SAFE_CHUNKS + 1;
	mailcap = (MailcapChunk *)malloc(total);
	if (mailcap == NULL)
		return;	/* total copout.. */
	for (r = (char *)&mailcap[SAFE_CHUNKS], total = 0; files != NULL; )
	    {
		MailcapFile *this = files;
		FILE *fp;

		if (this->size > 0 && (fp = fopen(this->name, "rb")) != NULL)
		    {
			n = fread(r + total, 1, this->size, fp);
			if (n > 0)
			    {
				total += n;
				if (r[total-1] == '\\')
					--total; /* nuke '\' if last char! */
				r[total++] = '\n';
			    }
			fclose(fp);
		    }
		files = this->next;
		free((char *)this);
	    }
	r[total++] = '\n';
	r[total] = 0;
	/*
	** "Compile" mailcap into binary
	*/
	chunk = mailcap;
	*chunk = init_chunk;
	r = (char *)&mailcap[SAFE_CHUNKS];
	for (; *r; ++r)
	    {
		if (*r <= ' ')
			continue;
		if (*r == '#')
		    {
			r = strchr(r, '\n'); /* Guaranteed to succeed!! */
			continue;
		    }
		entry = MakeChunk(chunk);
		chunk = MakeChunk(entry);
		entry->tag = ParseQuark(&r);
		if (Scan(&r, '/'))
			if (Scan(&r, '*'))
				chunk->tag = wild_card;
			else
				chunk->tag = ParseQuark(&r);
		else
			chunk->tag = wild_card;
		/*
		** Look for next ';' (and skip any garbage in between)
		*/
		while (*r && *r != '\n' && !Scan(&r, ';'))
			r = Next(r);
		r = AddStringToChunk(r, chunk);
		while (Scan(&r, ';'))
		    {
			chunk = MakeChunk(chunk);
			chunk->tag = ParseQuark(&r);
			if (Scan(&r, '='))
				r = AddStringToChunk(r, chunk);
		    }
		entry->size = (chunk - entry) + chunk->size;
		/*
		** Look for end of line (and skip any unparsed garbage)
		*/
		while (*r && *r != '\n')
			r = Next(r);
	    }
	/* Make terminator (with ZERO size!) */
	chunk = MakeChunk(chunk);
	*chunk = init_chunk;
    }
/*
** GetAttribute
**	retrieve a named attribute string from the current MsgBody.
**
**	[This function shows that message structure is not exactly
**	 as convenient as it could be... --msa]
*/

typedef struct AttributeMap
    {
	char *key;
	char *(*get)(char *, MsgBody);
    } AttributeMap;

static char *Pick(char **list, char *def, char *name)
    {
	char *s, *v;

	if (list != NULL) 
		while ((s = *list++) && (v = *list++))
			if (strcmp(s, name) == 0)
				return v;
	return def;
    }

static char *GetAttribute_type(char *name, MsgBody body)
    {
	name = string_ContType[(int)body->type];
	if (body->type == CT_unknown)
		name = Pick(body->params, name, "x-eb-type");
	return name;
    }

static char *GetAttribute_subtype(char *name, MsgBody body)
    {
	name = string_SubType[(int)body->subtype];
	if (body->subtype == ST_unknown)
		name = Pick(body->params, name, "x-eb-subtype");
	return name;
    }

static char *GetAttribute_charset(char *name, MsgBody body)
    {
	name = string_Charset[(int)body->charset];
	if (body->charset == CS_unknown)
		name = Pick(body->params, name, "x-eb-charset");
	return name;
    }

static char *GetAttribute_params(char *name, MsgBody body)
    {
	return Pick(body->params, "", name);
    }

static AttributeMap attribute_map[] =
    {
	{"type",	GetAttribute_type},
	{"subtype", 	GetAttribute_subtype},
	{"charset", 	GetAttribute_charset},
	{NULL,		GetAttribute_params},
    };

static char *GetAttribute(MsgBody body, char *name, int n)
    {
	char token[100], *s;
	int i;
	AttributeMap *map;
	
	if (n <= 0 || n >= sizeof(token))
		return "";
	for (s = token, i = 0; i < n; ++i)
		*s++ = isupper(name[i]) ? tolower(name[i]) : name[i];
	*s = 0;
	for (map = attribute_map; map->key; map++)
		if (strcmp(token, map->key) == 0)
			break;
	return (*map->get)(token, body);
    }	

/*
*/
typedef struct DynamicString
    {
	int size, count;
	char *string;
    } DynamicString;

static void AddString(DynamicString *str, char *s, int n)
    {
	int size;

	if (n == 0)
		n = strlen(s);
	size = str->count + n;
	if (size >= str->size)
	    {
		str->size = size + 100;
		if (str->string)
			str->string = (char *)realloc(str->string, str->size);
		else
			str->string = (char *)malloc(str->size);
	    }
	memcpy(str->string + str->count, s, n);
	str->count = size;
	str->string[size] = 0;
    }

static char *Format
	(char *prefix, char *f, Message msg, MsgBody body,char *name,int *used)
    {
	DynamicString buf;
	buf.size = buf.count = 0;
	buf.string = NULL;

	if (prefix)
		AddString(&buf, prefix, 0);
	for ( ; *f; ++f)
	    {
		if (*f == '\\' && f[1])
		    {
			AddString(&buf, ++f, 1);
			continue;
		    }
		else if (*f != '%')
		    {
			AddString(&buf, f, 1);
			continue;
		    }
		if (*++f == 't')
		    {
			AddString(&buf, GetAttribute(body, "type", 4), 0);
			AddString(&buf, "/", 1);
			AddString(&buf, GetAttribute(body, "subtype", 7), 0);
		    }
		else if (*f == 's')
		    {
			AddString(&buf, name, 0);
			*used += 1;
		    }
		else if (*f == '{')
		    {
			char *m = ++f;

			while (*f)
				if (*f++ == '}')
				    {
					AddString(&buf,
						  GetAttribute(body,m,f-m-1),
						  0);
					break;
				    }
			--f;
		    }
	    }
	return buf.string;
    }

static int TestOk(MailcapChunk *entry, Message msg, MsgBody body, char *name)
    {
	MailcapChunk *end = entry + entry->size;
	char *cmd;
	int result;

	for (++entry; entry < end; entry += entry->size)
		if (entry->tag == test_field)
		    {
			cmd = Format("",entry->string,msg,body,name,&result);
			result = system(cmd);
			free(cmd);
			return !result;
		    }
	return 1;
    }

static int HasFlag(MailcapChunk *entry, XrmQuark flag)
    {
	MailcapChunk *end = entry + entry->size;

	for (++entry; entry < end; entry += entry->size)
		if (entry->tag == flag)
			return 1;
	return 0;
    }

/*
** MailcapView
**	returns the command string and an optional filename. If filename
**	is present, the bodypart content must be written into it before
**	executing the command. If filename is not returned, the bodypart
**	content is assumed to be fed via the standard input.
**
**	Returned command string and filename are allocated with 'malloc'
**	and must be released by 'free'. Just a temporary name is returned,
**	file is not created! Calling program must create and delete the
**	file.
*/
char *MailcapView(Message msg, MsgBody body, char **filename, int *copious)
    {
	XrmQuark q_type = XrmStringToQuark(GetAttribute(body,"type",4));
	XrmQuark q_subtype = XrmStringToQuark(GetAttribute(body,"subtype",7));
	MailcapChunk *entry;
	char *f = NULL, *name = tmpnam((char *)NULL);
	int use_name = 0;

	name = strcpy((char *)malloc(strlen(name)+1), name);
	if (mailcap == NULL)
		MailcapLoad();
	for (entry = mailcap; entry->size; entry += entry->size)
		if (entry[0].tag == q_type &&
		    (entry[1].tag == wild_card || entry[1].tag == q_subtype) &&
		    TestOk(entry, msg, body, name))
		    {
			char *p = NULL;

			if (HasFlag(entry, needsterminal))
				p = "xterm -e ";
			f = Format(p,entry[1].string,msg,body,name,&use_name);
			*copious = HasFlag(entry, copiousoutput);
			break;
		    }
	if (filename == NULL)
		free(name);
	else if (!use_name)
	    {
		free(name);
		*filename = NULL;
	    }
	else
		*filename = name;
	return f;
    }


