/*
** MamSave.c
**
**	This module contains the implementation of MamSaveMessage
**
**	Content-Type parameters x-eb-transfer-type, x-eb-transfer-subtype
**	and filename are only stored if current_mime == 0 (MAM format).
**
** Copyright 1993-1995 by Markku Savela and
**	Technical Research Centre of Finland
*/

#include "common.h"
#include "Mfm.h"
#include "Mam.h"
#include "MamCommon.h"
#include "MamCoding.h"

#undef tables_h		/* Force re-include */
#define ENUM_TABLE_GENERATION
#include "tables.h"

static Message current_message = NULL;
static char *current_buffer = NULL;
static long current_length = 0;
static FILE *current_file = NULL;
static char *current_boundary = NULL; /* Just to pass it to content-type */
static Encoding current_encoding = ENC_undefined; /*  -"- */
static int current_mime = 0; /* 0=MAM Format, 1=MIME out format */

typedef void (*HeaderHandler)(MsgBody, char *, void *);

typedef struct HeaderControl
    {
	char *name;		/* Header field name as written out */
	HeaderHandler handler;	/* Write function for the field */
	/*
	** The location of the value:
	*/
#	define LOC_BODY 1	/* Current MsgBody */
#	define LOC_HEADER 2	/* Current MsgHeader */
	int location;
	int offset;		/* ..to value in selected structure */
    } HeaderControl;

typedef struct StateName
    {
	char *name;
	unsigned int value;
    } StateName;

static StateName state_name[] = 
    {
	{"new",		Msg_New},
	{"read",	Msg_Read},
	{"draf",	Msg_Draft},
	{"sent",	Msg_Sent},
	{"modified",	Msg_Modified},
	{"replied-to",	Msg_Replied_To},
	{"is-archived",	Msg_Is_Archived},
	{"awaiting_reply", Msg_Awaiting_Reply},
	{"tagged",	Msg_Tagged},
	{NULL,		Msg_No_State},
    };

static void WriteIncludedContent(MsgBody body, Encoding old, Encoding new)
    {
	char *path;
	FILE *f;
	char buf[3*18*20]; /* Size must be evenly divisible by 3 for base64,
			      magic 18 is just attempting to get full lines
			      knowing that Mam_to64 breaks after 71 characters.
			      (It doesn't matter if guess 18 is wrong, it's
			      just trying to get neater output) */
	int len;
	
	path = MamContentFile(body->content->filename, current_message);
	if (path == NULL || (f = fopen(path, "rb")) == NULL)
		return;
	while ((len = fread(buf, 1, sizeof(buf), f)) > 0)
		if (old != new && new == ENC_quotedPrintable)
			Mam_toqp((unsigned char *)buf,
				 (unsigned char *)&buf[len], current_file);
		else if (old != new && new == ENC_base64)
			Mam_to64((unsigned char *)buf,
				 (unsigned char *)&buf[len], current_file);
		else
			fwrite(buf, 1, len, current_file);
	fclose(f);
    }

static void WriteContent(MsgBody body, Encoding old, Encoding new)
    {
	BodyContent content = body->content;
	if (content == NULL)
		return;
	if (content->filename && content->length == -1)
	    {
		if (current_mime || content->start_offset == -1)
			WriteIncludedContent(body, old, new);
	    }
	else if (content->length > 0 && content->start_offset >= 0)
		fwrite(current_buffer + content->start_offset,
		       1, content->length, current_file);
    }

static void WriteQuotedString(char *prefix, char *s)
    {
	char buf[1000], *p;
	register int c;

	if (prefix)
		fprintf(current_file, "%s", prefix);
	p = buf;
	*p++ = '"';
	if (s)
		for (; (c = *s++) != 0; *p++ = c)
		    {
			if (p > &buf[sizeof(buf)-3]) /* No room for '\x"' ? */
			    {
				fwrite(buf, 1, p - buf, current_file);
				p = buf;
			    }
			if (c == '"' || c == '\\')
				*p++ = '\\';
		    }
	*p++ = '"';
	fwrite(buf, 1, p - buf, current_file);
    }

static void RFC822_Unknown(MsgBody body, char *f, void *data)
    {
	char **list = *(char ***)data;
	char *s, *v;

	if (current_mime == 0) /* Write only in MAM format! */
		if (list != NULL)
			while ((s = *list++) && (v = *list++))
				fprintf(current_file, "%s: %s\n", s, v);
    }

static void RFC822_List(MsgBody body, char *f, void *data)
    {
	char **ptr = *(char ***)data;

	if (ptr && *ptr != NULL)
	    {
		fprintf(current_file, "%s: %s", f, *ptr);
		while (*++ptr)
			fprintf(current_file, ", %s", *ptr);
		fprintf(current_file, "\n");
	    }
    }

static void RFC822_String(MsgBody body, char *f, void *data)
    {
	if (*(char **)data != NULL)
		fprintf(current_file, "%s: %s\n", f, *(char **)data);
    }

static void RFC822_Sensitivity(MsgBody body, char *f, void *data)
    {
	if (*(Sensitivity *)data)
		fprintf(current_file, "%s: %s\n", f,
			string_Sensitivity[(int)*(Sensitivity *)data]);
    }

static void RFC822_Importance(MsgBody body, char *f, void *data)
    {
	if (*(Importance *)data)
		fprintf(current_file, "%s: %s\n", f,
			string_Importance[(int)*(Importance *)data]);
    }

static void RFC822_Time(MsgBody body, char *f, void *data)
    {
    }

static void RFC822_ContentTransferEncoding(MsgBody body, char *f, void *data)
    {
	if (current_encoding != ENC_sevenBit)
		fprintf(current_file, "%s: %s\n", f,
			string_Encoding[(int)current_encoding]);
    }

static char *Pick(char **list, char *def, char *name)
    {
	char *s, *v;

	if (list != NULL) 
		while ((s = *list++) && (v = *list++))
			if (strcmp(s, name) == 0)
				return v;
	return def;
    }

static void RFC822_ContentType(MsgBody body, char *f, void *data)
    {
	char **list, *s, *v;
	char *type = string_ContType[(int)body->type];
	char *subtype = string_SubType[(int)body->subtype];

	/*
	** If the type or subtype is unknown, put back the original
	** type string, if recorded in params.
	*/
	if (body->type == CT_unknown)
		type = Pick(body->params, type, "x-eb-type");
	if (body->subtype == ST_unknown)
		subtype = Pick(body->params, subtype, "x-eb-subtype");
	fprintf(current_file, "%s: %s/%s", f, type, subtype);
	if (body->charset != CS_us_ascii)
		fprintf(current_file, "; charset=%s",
			body->charset == CS_unknown ?
			Pick(body->params, "x-unknown", "x-eb-charset") :
			string_Charset[(int)body->charset]);
	if (!current_mime)
	    {
		if (body->transfer_type != CT_unknown &&
		    body->transfer_type != body->type)
			fprintf(current_file, "; x-eb-transfer-type=%s",
				string_ContType[(int)body->transfer_type]);
		if (body->transfer_subtype != ST_unknown &&
		    body->transfer_subtype != body->subtype)
			fprintf(current_file, "; x-eb-transfer-subtype=%s",
				string_SubType[(int)body->transfer_subtype]);
		if (body->content && body->content->filename)
			WriteQuotedString("; filename=",
					  body->content->filename);
	    }
	if (current_boundary)
		WriteQuotedString("; boundary=", current_boundary);
	if ((list = body->params) != NULL)
		while ((s = *list++) && (v = *list++))
			/*
			** Do not output "x-eb*" paramaters (they have
			** already been handled above).
			*/
			if (memcmp("x-eb", s, 4) != 0)
			    {
				fprintf(current_file, "; %s=", s);
				WriteQuotedString((char *)NULL, v);
			    }
	fwrite("\n", 1, 1, current_file);
    }

/*
** rfc822_header
**	This array determines which header fields are actually written
**	out and the order in which it happens.
*/
#define TO_BODY(x)	LOC_BODY,offsetof(struct _MsgBody, x)
#define TO_HEADER(x)	LOC_HEADER,offsetof(struct _MsgHeader, x)

static HeaderControl rfc822_header[] =
    {
	{"From",	RFC822_String,	TO_HEADER(from)},
	{"To",		RFC822_List,	TO_HEADER(to)},
	{"Cc",		RFC822_List,	TO_HEADER(cc)},
	{"Bcc",		RFC822_List,	TO_HEADER(bcc)},
	{"Date",	RFC822_String,	TO_HEADER(date)},
	{"Subject",	RFC822_String,	TO_HEADER(subject)},
	{"Sender",	RFC822_String,	TO_HEADER(sender)},
	{"In-Reply-To",	RFC822_String,	TO_HEADER(in_reply_to)},
	{"Expiry-Date",	RFC822_Time,	TO_HEADER(expires_at)},
	{"Reply-By",	RFC822_Time,	TO_HEADER(reply_by)},
	{"Importance",	RFC822_Importance, TO_HEADER(importance)},
	{"Sensitivity",	RFC822_Sensitivity, TO_HEADER(sensitivity)},
	{"References",	RFC822_List,	TO_HEADER(related_to)},
	{"Obsoletes",	RFC822_List,	TO_HEADER(obsoletes)},
	{"Message-Id",	RFC822_String,	TO_HEADER(id)},
	{"Content-Type",RFC822_ContentType, TO_BODY(type)},
	{"Content-Transfer-Encoding",
			RFC822_ContentTransferEncoding, TO_BODY(encoding)},
	{"Content-ID",	RFC822_String,	TO_BODY(id)},
	{"Content-Description",
			RFC822_String,	TO_BODY(description)},
	{"Comments",	RFC822_String,	TO_HEADER(comments)},
	{"Keywords",	RFC822_String,	TO_HEADER(keywords)},
	{"",		RFC822_Unknown,	TO_HEADER(unknownfields)},
	{NULL,		NULL,		TO_HEADER(unknownfields)}, /* DUMMY */
    };

static void WriteHeaders(MsgBody body)
    {
	HeaderControl *h;

	for (h = &rfc822_header[0]; h->name != NULL; ++h)
	    {
		char *data = NULL;

		switch (h->location)
		    {
		    case LOC_BODY:
			data = (char *)body;
			break;
		    case LOC_HEADER:
			data = (char *)body->header;
			break;
		    default:
			break;
		    }
		if (data != NULL)
			(*h->handler)(body,h->name,(void *)(data+h->offset));
	    }
	fwrite("\n", 1, 1, current_file); /* end of headers */
    }

static void WriteBody(MsgBody body, char *oldboundary)
    {
	static int seq = 0;
	char newboundary[100];

	while (body)
	    {
		if (oldboundary)
			fprintf(current_file, "\n--%s\n", oldboundary);
		if (body->type == CT_multipart)
		    {
			if (seq == 0)
				srand((unsigned int)time((time_t *)NULL));
			sprintf(newboundary, "%d_=Mam-%s=_%d%x", ++seq,
				mam_version, rand(), rand());
			current_boundary = newboundary;/* ...for ContType! */
		    }
		else
			current_boundary = NULL;
		/*
		** If the externally stored body content is to be stored
		** within the message header, then there may be a need to
		** change the content transfer encoding. (Only check this
		** if inclusion (start_offset == -1) is requested, does not
		** change existing encodings otherwise, even if they might
		** be inappropriate within header). Only following is needed:
		**
		**	8bit	-> quoted printable
		**	binary	-> base64
		*/
		current_encoding = body->encoding;
		if (body->content && body->content->filename &&
		    (body->content->start_offset == -1 || current_mime))
			if (body->encoding == ENC_binary)
				current_encoding = ENC_base64;
			else if (body->encoding == ENC_eightBit)
				current_encoding = ENC_quotedPrintable;
		WriteHeaders(body);
		if (body->child == NULL)
			/* Actual content is only written on leaf nodes! */
			WriteContent(body, body->encoding, current_encoding);
		WriteBody(body->child,
			  body->type == CT_multipart ? newboundary : NULL);
		body = body->next;
	    }
	if (oldboundary)
		fprintf(current_file, "\n--%s--\n", oldboundary);
    }

static void WriteMessage(Message msg)
    {
	StateName *h;

	if (current_mime == 0)
	    {
		fprintf(current_file, "x-eb-state:");
		for (h = state_name; h->name; ++h)
			if (h->value & msg->state)
			    {	
				fprintf(current_file, "%s ", h->name);
				if ((h->value & Msg_Tagged) && msg->tag)
					WriteQuotedString((char *)NULL, msg->tag);
			    }
		fwrite("\n", 1, 1, current_file);
	    }
	else
		fprintf(current_file, "Mime-Version: 1.0\n");
	WriteBody(msg->body, (char *)NULL);
    }

int MamSaveMessage (char *name, Message msg)
    {
	int result;

	if (msg == NULL)
		return MamError_NOMESSAGE;
	if (name == NULL && (name = msg->name) == NULL)
		return MamError_NOPATH;
	current_buffer = NULL;
	current_file = NULL;
	current_message = msg;
	current_mime = 0;
	/*
	** Setup general error exit handling
	*/
	if ((result = setjmp(mam_error_exit)) == 0)
	    {
		if (msg->name != NULL)
			MamReadHeader(msg->name, &current_buffer,
				      &current_length);
		if ((current_file = MamMakeHeader(name)) == NULL)
			result = MamError_NOFILE;
		else
		    {
			WriteMessage(msg);
			result = MamError_SUCCES;
		    }
	    }
	if (current_buffer != NULL)
		free(current_buffer);
	if (current_file != NULL)
		fclose(current_file);
	return result;
    }

int MamMimeMessage (char *name, Message msg)
    {
	int result;

	if (msg == NULL)
		return MamError_NOMESSAGE;
	current_buffer = NULL;
	current_file = NULL;
	current_message = msg;
	current_mime = 1;
	/*
	** Setup general error exit handling
	*/
	if ((result = setjmp(mam_error_exit)) == 0)
	    {
		if (msg->name != NULL)
			MamReadHeader(msg->name, &current_buffer,
				      &current_length);

		current_file = name ? fopen(name, "wb"): stdout;
		if (current_file == NULL)
			result = MamError_NOFILE;
		else
		    {
			WriteMessage(msg);
			result = MamError_SUCCES;
		    }
	    }
	if (current_buffer != NULL)
		free(current_buffer);
	if (current_file != NULL && current_file != stdout)
		fclose(current_file);
	return result;
    }

