/*
** MamCommon
**	Miscellaneous private Mam utility functions. These are only for
**	internal use from other mam functions.
**
** Copyright 1993-1995 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include "common.h"
#include "Mfm.h"
#include "Mam.h"
#include "MamCommon.h"

#undef tables_h		/* Force re-include */
#define ENUM_TABLE_GENERATION
#include "tables.h"

/*
** mam_error_exit
**	is always primed at entry to MamLoadMessage and can be called
**	at any with longjmp if the load should be aborted. This will
**	cancel all memory allocations done and close all open files.
*/
jmp_buf mam_error_exit;

char mam_version[] = MAM_VERSION;

static char hdrname[] = {'/', 'H', '.', 'm', 's', 'g', '\0', };

void *MamCalloc(int n, int size)
    {
	void *m = calloc(n, size);
	if (m == NULL)
		longjmp(mam_error_exit, MamError_MEMORY);
	return m;
    }

void *MamMalloc(int size)
    {
	void *m = malloc(size);
	if (m == NULL)
		longjmp(mam_error_exit, MamError_MEMORY);
	return m;
    }

/*
** MamHeaderInfo
**	Construct the actual filename of the header file into a
**	static buffer and find out the size of the header file.
**
**	Return 0 on succes (filename and length have been loaded),
**	Return non-zero on failure (filename and length have not been
**	modified). This value is 'errno' or some other error indication.
*/
int MamHeaderInfo
	(char *name, char **filename, long *offset,long *length, time_t *mtime)
    {
	static char fname[500];
	struct stat info;
	char *s;

	if (strlen(name) + sizeof(hdrname) > sizeof(fname))
		return MamError_LONGNAME;
	strcpy(fname, name);
	/*
	** Check for the special case where the header/message is in
	** in a file (like folder file). In that case the 'name' ends
	** with string "/ offset length".
	*/
	if ((s = strrchr(fname, '/')) != NULL && s[1] == ' ')
	    {
		*s = 0;
		if (!stat(fname, &info) && (info.st_mode & S_IFMT) != S_IFDIR)
		    {
			*filename = fname;
			*offset = atol(s+1);
			s = strchr(s+2, ' ');
			*length = s ? atol(s+1) : 0L;
			if (mtime)
				*mtime = info.st_mtime;
			return 0;
		    }
		*s = '/';
	    }
	if (stat(fname, &info))
		return errno ? errno : MamError_ERRNO;
	if ((info.st_mode & S_IFMT) == S_IFDIR)
	    {
		strcat(fname, hdrname);
		if (stat(fname, &info))
			return errno ? errno : MamError_ERRNO;
	    }
	*filename = fname;
	*offset = 0L;
	*length = info.st_size;
	if (mtime)
		*mtime = info.st_mtime;
	return 0;
    }

/*
** MamReadSection
**	Read a section from a file into memory (to buffer that is
**	allocated). Returns a pointer to the buffer or NULL, if the
**	read fails. Also, returns the actual number of bytes read
**	into 'actual'.
**
**	The calling program must release the buffer with free after
**	it is no more needed.
*/
char *MamReadSection(char *name, long offset, long length, long *actual)
    {
	FILE *fp;
	char *buffer;

	*actual = 0;
	if ((fp = fopen(name, "rb")) == NULL)
		return NULL;
	if (offset > 0 && fseek(fp, offset, 0))
		return NULL;
	buffer = (char *)MamMalloc(length+2);
	*actual = fread(buffer, 1, length, fp);
	fclose(fp);
	if (*actual <= 0)
	    {
		free(buffer);
		buffer = NULL;
	    }
	return buffer;
    }

/*
** MamReadHeader
**	Read the message header information into memory (allocate
**	memory for it and return pointer and length).
*/
int MamReadHeader(char *name, char **header, long *length)
    {
	char *fname;
	int result;
	long offset;

	*header = NULL;
	*length = 0;
	if ((result = MamHeaderInfo(name,&fname,&offset,length,
				    (time_t *)NULL))!=0)
		return result;
	*header = MamReadSection(fname, offset, *length, length);
	if (*length <= 0 || *header == NULL)
		return MamError_READ;
	return 0;
    }

/*
** MamMakeHeader
**	Start writing a header file. 
*/
FILE *MamMakeHeader(char *name)
    {
	struct stat info;
	char fname[500];
	char *t;

	if (stat(name, &info))
	    {
		/*
		** Message does not exist, create a directory by
		** default.
		*/
		if (mkdir(name, MAM_DEFAULT_DIRMODE) || stat(name, &info))
			longjmp(mam_error_exit, errno);
		MfmSetContentType(name, string_EBFileType[EB_message]);
	    }
	else if ((info.st_mode & S_IFMT) == S_IFDIR &&
		 ((t = MfmGetContentType(name)) == NULL ||
		 strcmp(t, string_EBFileType[EB_message]) != 0))
		/*
		** name exists and is a directory which is not
		** typed as "eurobridge/message".
		*/
		return NULL;
	if (strlen(name) + sizeof(hdrname) > sizeof(fname))
		longjmp(mam_error_exit, MamError_LONGNAME);
	strcpy(fname, name);
	if ((info.st_mode & S_IFMT) == S_IFDIR)
		strcat(fname, hdrname);
	return fopen(fname, "wb");
    }

