/*
** MamCache.c - Message Folder caching support
**
** Copyright 1993-1995 by Markku Savela and
**	Technical Research Centre of Finland
*/

#include "common.h"
#include "Mam.h"
#include "MamCommon.h"
#include "message.h"

/*
** CACHE FILE FORMAT
**
**	The format is *BINARY* octet stream format, even if it will in
**	most cases look like a normal text file. The BNF like syntax
**	for the stream can be expressed with
**
**	<Cache>		::= <TimeStamp> <NL> <CacheData>
**	<CacheData>	::= <empty> | <CacheEntry> <CacheData>
**	<CacheEntry>	::= <QFile> ":" <QData> <NL>
**
**	<QFile>		::=	Quoted octect string with quoting rules
**				":" -> "\:" and "\" -> "\\". This is the
**				filename within folder.
**
**	<QData>		::=	Quoted octed string with quoting rules
**				<NL> -> "\n" and "\" -> "\\". This is the
**				user supplied cache information.
**
**	<NL>		::=	'\n' (0x0A)
**
**	<TimeStamp>	::=	Sequence of digits representing time value
**				as returned by time function. The cache
**				information reflects the information only
**				up to this point (Note that the writing
**				time of the cache file itself is not a
**				reliable time stamp, it is always same or
**				greater than this value).
**
** Whether or not the cache file is readable by different architectures
** depends on the nature of the user supplied cache data. If this data is
** a plain text string, the cached information is also portable. If this
** data is a binary C structure, the cache information is usable only on
** same machine architecture and application.
*/

/*
** CacheBlock
**	is a building block for cache information. The cache information
**	is first collected and constructed into linked list(s).
*/
typedef struct CacheBlock
    {
	struct CacheBlock *next;
	MamCacheRec data;
    } CacheBlock;

/*
** free_blocks
**	is a storage of the unused (but allocated) CacheBlock structures.
**	CacheBlocks are allocated with malloc as needed, but are never
**	released. The same CacheBlocks will be reused on calls to
**	MamCacheAccess.
*/
static CacheBlock *free_blocks = NULL;
/*
** cache_valid
**	is set to non-zero at the start of MamCacheAccess. It is set to
**	zero on any error that would cause the inmemory cache to be
**	incorrect (mostly malloc fail). The zero value prevents writing
**	of the cache to the disk, regardless of the mode parameter value.
**	(zero cache_valid make MamCache_UPDATE not write).
*/
static cache_valid = 0;
/*
** cache_data, cache_length
**	describe the pointer to the contiguous memory are which contains
**	the old cache information. This buffer is attached to the sentinel
**	element of the cache array.
*/
static char *cache_data;
static int cache_length;

/*
** old_cache
**	is a list of cache entries retrieved from the old cache file.
**	When the folder is scanned, the entries that still exist will
**	be moved to the new_cache list. At the end of scan, this list
**	will contain only the entries for which the referenced message
**	does not exist anymore.
*/
static CacheBlock *old_cache = NULL;
static CacheBlock **old_cache_last = &old_cache;
static int old_count = 0;
static time_t old_stamp = 0;

/*
** new_cache
**	is a list of cache entries that will reflect the current state.
*/
static CacheBlock *new_cache = NULL;
static CacheBlock **new_cache_last = &new_cache;
static int new_count = 0;	/* Total number of entries */
static int new_changed = 0;	/* Number of changed or new entries */
static time_t new_stamp = 0;

/*
** GetCacheBlock
**	Take new CacheBlock into use from free_blocks, or if that is
**	empty, then allocate a new entry from the system.
*/
CacheBlock *GetCacheBlock(void)
    {
	static CacheBlock init_CacheBlock;
	
	CacheBlock *b = free_blocks;

	if ((b = free_blocks) != NULL)
		free_blocks = free_blocks->next;
	else if ((b = (CacheBlock *)malloc(sizeof(CacheBlock))) == NULL)
	    {
		cache_valid = 0; /* malloc error, prevent UPDATE */
		return NULL;
	    }
	*b = init_CacheBlock;
	return b;
    }

/*
** FreeCacheBlocks
*/
static void FreeCacheBlocks(CacheBlock *b)
    {
	CacheBlock *first = b, *last;
	if (first)
	    {
		do
		    {
			if (b->data.malloc_name)
				free(b->data.name);
			if (b->data.malloc_data)
				free(b->data.data);
			last = b;
			b = b->next;
		    }
		while (b);
		last->next = free_blocks;
		free_blocks = first;
	    }
    }

/*
** FilePath
**	Concatenate folder name with filename and return a pointer
**	to full path. This pointer is to internal static area and
**	is valid until next call to this function.
**
**	Normally folder,name are concatenated with '/', but if checkdir
**	is non-zero and folder is a file (not directory), then concatenate
**	folder and name without any extra characters.
*/
static char *FilePath(char *folder, char *name, int checkdir)
    {
	static char filename[MAM_PATH_SIZE];
	struct stat info;
	int n = strlen(folder);

	if (n + strlen(name) + 2 > sizeof(filename))
		return NULL;
	strcpy(filename, folder);
	while (n > 0 && filename[n-1] == '/')
		filename[--n] = 0; /* Zap trailing '/'s, if any */
	if (!checkdir|| stat(folder,&info) || (info.st_mode & S_IFMT)==S_IFDIR)
		filename[n++] = '/';
	strcpy(&filename[n], name);
	return filename;
    }

/*
** CacheMessage
*/
static void CacheMessage(char *folder, char *name, MamCacheData conv)
    {
	char *path = FilePath(folder, name, FALSE);
	char *header;
	long offset, length;
	time_t mtime;
	CacheBlock *b, **p;

	if (MamHeaderInfo(path, &header, &offset, &length, &mtime))
		return; /* Treat as non-existent if cannot access header */
	for (p = &old_cache; (b = *p) != NULL; p = &b->next)
		if (strcmp(b->data.name, name) == 0)
		    {
			*p = b->next;	/* Remove from old_cache */
			old_count -= 1;
			if (mtime < old_stamp)
				goto add_it; /* No change, just add to new */
			else
				goto refresh;
		    }
	if ((b = GetCacheBlock()) == NULL)
		return; /* ..out of memory!! */
	if ((b->data.name = strdup(name)) == NULL)
		cache_valid = 0;
	else
		b->data.malloc_name = TRUE;
    refresh:
	new_changed += 1;
	if (b->data.malloc_data)
		free(b->data.data);
	if (conv)
	    {
		Message msg = MamLoadMessage(path);

		b->data.data = (*conv)(msg, &b->data.length);
		b->data.malloc_data = (b->data.data != NULL);
		FreeMessage(msg);
	    }
	else
	    {
		b->data.data = NULL;
		b->data.length = 0;
		b->data.malloc_data = FALSE;
	    }
	b->data.changed = TRUE;
    add_it:
	*new_cache_last = b;
	new_cache_last = &b->next;
	new_count += 1;
	b->next = NULL;
    }

/*
** ParseCache
*/
static void ParseCache(char *s, char *e)
    {
	char *p, *r;
	int c;
	CacheBlock *b;

	old_stamp = atol(s);
	if ((s = strchr(s, '\n')) == NULL)
		return;
	s += 1;
	while (p = s, s < e && (b = GetCacheBlock()))
	    {
		*old_cache_last = b;
		old_cache_last = &b->next;
		b->next = NULL;
		old_count += 1;

		/* Extract filename up to ':' */

		for (r=p; s < e && (c = *s++) != ':'; *r++ = c)
			if (c == '\\' && s < e && (c = *s++) == 'n')
				c = '\n';
		*r = '\0';
		b->data.name = p;
		b->data.malloc_name = FALSE;

		/* Extract the cache information up to NEWLINE */

		for (r = p = s; s < e && (c = *s++) != '\n'; *r++ = c)
			if (c == '\\' && s < e && (c = *s++) == 'n')
				c = '\n';
		b->data.data = p;
		b->data.length = r-p;
		b->data.malloc_data = FALSE;
	    }
    }

/*
** LoadCache
*/
static void LoadCache(char *folder, char *cache)
    {
	char *path = FilePath(folder, cache, TRUE);
	struct stat info;
	int len;
	FILE *fp;

	if (cache_data)
		free(cache_data);
	cache_data = NULL;
	old_cache = NULL;
	old_count = 0;
	old_cache_last = &old_cache;
	if (path != NULL &&
	    stat(path, &info) == 0 &&
	    (cache_data = (char *)malloc(info.st_size+1)) != NULL)
	    {
		cache_length = info.st_size;
		cache_data[cache_length] = 0;
		if ((fp = fopen(path, "rb")) != NULL)
		    {
			if ((len = fread(cache_data, 1, cache_length, fp)) > 0)
				ParseCache(cache_data, cache_data + len);
			fclose(fp);
		    }
	    }
    }
/*
** WriteQuoted
**
**	Write out sequence of quoted data and terminate it with q.
**	The quoting is controlled by q
**
**	q == 0,	Flush out the buffer. The function maintains
**		internal buffer and before closing the output,
**		one flush call must be done,
**
**	q != 0,	q represents the octet value that must be quoted.
**		(q == ':' or q == '\n'). The quoting rules are
**
**			'\\' -> '\\' '\\'
**			':'  -> '\\' ':'
**			'\n' -> '\\' '\n'
**			
*/
static void WriteQuoted(FILE *fp, char *s, int n, int q)
    {
	static char buf[1000];
	static char *p = buf;

	int c;

	if (q == 0)
	    {
		/* Flush Out */
		if (p > &buf[0])
			fwrite(buf, 1, p - buf, fp);
		p = buf;
		return;
	    }
	else if (s)
	    {
		char *e = s + n;

		while (s < e)
		    {
			c = *s++;
			if (p > &buf[sizeof(buf)-3])
			    {
				fwrite(buf, 1, p - buf, fp);
				p = buf;
			    }
			if (c == q || c == '\\')
			    {
				*p++ = '\\';
				if (c == '\n')
					c = 'n';
			    }
			*p++ = c;
		    }
	    }
	*p++ = q;
    }

/*
** SaveCache
*/
static void SaveCache(char *folder, char *cache)
    {
	char *path = FilePath(folder, cache, TRUE);
	FILE *fp;
	if (path != NULL && (fp = fopen(path, "wb")) != NULL)
	    {
		CacheBlock *b;
		fprintf(fp, "%ld\n", (long)new_stamp);
		for (b = new_cache; b && b->data.name; b = b->next)
		    {
			WriteQuoted(fp, b->data.name,strlen(b->data.name),':');
			WriteQuoted(fp, b->data.data, b->data.length, '\n');
		    }
		WriteQuoted(fp, (char *)NULL, 0, 0);
		fclose(fp);
	    }
	
    }

/*
** MamCacheAccess
*/
MamCache MamCacheAccess(char *folder, 
			char *cache_name,
			MamCacheData conv, MamCacheMode mode)
    {
	CacheBlock *b;
	MamCache cache;

	cache_valid = 1;	/* Assume ininitially all goes well.. */
	new_cache = NULL;
	new_cache_last = &new_cache;
	new_count = 0;
	new_changed = 0;
	(void)time(&new_stamp);

	if (folder == NULL || cache_name == NULL)
		return NULL;
	LoadCache(folder, cache_name);
	if (mode == MamCache_SCAN || mode == MamCache_UPDATE)
	    {
		MamFolder fdr;
		char *name;

		fdr = MamOpenFolder(folder, (MamFilter)NULL, (void *)NULL);
		while ((name = MamNextMessage(fdr)) != NULL)
			CacheMessage(folder, name, conv);
		MamCloseFolder(fdr);
		FreeCacheBlocks(old_cache);
		/*
		** Update cache on disk if requested, and if information
		** has been changed/added (new_changed > 0) or deleted
		** (old_count > 0).
		*/
		if (mode == MamCache_UPDATE &&
		    cache_valid && (new_changed || old_count))
			SaveCache(folder, cache_name);
	    }
	else
	    {
		new_cache = old_cache;
		new_count = old_count;
	    }
	cache = (MamCache)malloc((new_count+1)*sizeof(*cache));
	if (cache)
	    {
		MamCache c;
		for (c = cache, b = new_cache; b; b = b->next)
		    {
			*c++ = b->data;
			b->data.malloc_name = FALSE; /* Do not release yet */
			b->data.malloc_data = FALSE; /* Do not release yet */
		    }
		c->name = NULL;
		c->data = cache_data;
		c->length = cache_length;
		c->malloc_data = (cache_data != NULL);
		cache_data = NULL;
	    }
	FreeCacheBlocks(new_cache);
	return cache;
    }

/*
** MamCacheFree
*/
void MamCacheFree(MamCache data)
    {
	if (data)
	    {
		MamCache c;

		for (c = data; c->name; c++)
		    {
			if (c->malloc_name)
				free((void *)c->name);
			if (c->malloc_data)
				free((void *)c->data);
		    }
		/*
		** Release the cache buffer (from sentinel) and
		** the MamCache self.
		*/
		if (c->malloc_data)
			free((void *)c->data);
		free((void *)data);
	    }
    }

