#ifndef Mam_h
#define Mam_h
/*
** Mam.h - Message Archive and Management
**
** Copyright 1993-1995 by Markku Savela and
**	Technical Research Centre of Finland
*/
#include "typedefs.h"

/*
** ERRORS
**
**	MamError enumeration describes error returns specific to the
**	Mam. These values are all negative, positive non-zero returns
**	are system errors and refer to the errno.
*/
typedef enum
    {
	MamError_SUCCES=0,	/* No Error */
	MamError_MEMORY=-1,	/* Memory allocation error */
	MamError_NOPATH=-2,	/* No filename supplied */
	MamError_READ=-3,	/* Failed in reading the message header */
	MamError_LONGNAME=-4,	/* Message name/path is too long */
	MamError_NOMESSAGE=-5,	/* No Message structure supplied */
	MamError_NOFILE=-6,	/* Header File not available */
	MamError_ERRNO=-7,	/* errno was zero after an error! */
	MamError_NOT_MSG=-8,	/* name is not of type 'eurobridge/message` */
	MamError_NOT_FDR=-9,	/* name is not of type 'eurobridge/folder' */
	MamError_IS_MSG=-10,	/* name is not to be 'eurobridge/message' */
	MamError_EXISTS=-11	/* name already exists, cannot overwrite */
    } MamError;


typedef struct MamFolderRec *MamFolder;
typedef int (*MamFilter)(char *, void *);

/*
**	Open a mail message and return an in-memory structure (Message)
**	representing the mail message.
**
**	Returns NULL if load fails.
*/
Message MamLoadMessage
	(char *path	/* Message file path */
	 );
/*
**	Read the body part content in binary form into an allocated
**	memory buffer and return an address of this buffer.
**
**	This function will undo all content transfer encodings, if
**	present. The content in the returned buffer will be exactly
**	in the format indicated by the content type.
**
**	Returns a buffer which must be released by 'free' when no
**	more used.
*/
char *MamGetContent
	(Message msg,	/* The message into which the MsgBody belongs */
	 MsgBody body,	/* The message body from which content is retrieved */
	 long *length	/* Will contain the length of the content in bytes */ 
	 );

/*
**	Read the body part content in binary form into an allocated
**	memory buffer and return an address of this buffer without
**	doing any transfer encoding decoding.
*/
char *MamGetRawContent
	(Message msg,	/* The message into which the MsgBody belongs */
	 MsgBody body,	/* The message body from which content is retrieved */
	 long *length	/* Will contain the length of the content in bytes */ 
	 );
/*
**	Save message structure onto disk and tag it with MFM as
**	"eurobridge/message".
**
**	Returns zero on success, and non-zero otherwise. The MAM specific
**	errors (MamError) have negative values, positive error returns
**	refer to system errors (errno).
*/
int MamSaveMessage
	(char *path,	/* Message file path. */
	 Message msg	/* Message structure to save */
	 );
/*
**	Dump message structure in MIME format to a file or standard
**	output if path==NULL.
**
**	Returns zero on success, and non-zero otherwise. The MAM specific
**	errors (MamError) have negative values, positive error returns
**	refer to system errors (errno).
*/
int MamMimeMessage
	(char *path,	/* Message file path, if NULL then stdout */
	 Message msg	/* Message structure to save */
	 );
/*
**	A quick routine to access only the 'x-eb-state' content.
**
**	This function assumes that this header line is the first
**	line in the file and that all the flags fit into first
**	300 characters (this means the user defined 'tag' string
**	should be written out last).
**
**	Returns Msg_No_State, if the file could not be opened or if it
**	didn't contain any x-eb-state line.
*/
MsgState MamPeekMessage
	(char *path	/* Message file path */
	 );
/*
**	Create a message directory for the message.
**
** Returns
**	zero on success. Negative error codes are defined by MamError,
**	positive errors are errno values.
*/
int MamMakeMessage
	(char *path	/* Message file path */
	 );

/*
**	Move message into new location. The move will fail if the
**	new location exists and is typed as "eurobridge/message". If
**	new location is a directory, then the message is moved into
**	this direcory. If new location does not exist, the message
**	will be renamed to this name.
**
**	After successful move operation, the MFM information is
**	updated.
*/
int MamMoveMessage
	(char *path,	/* Message file path */
	 char *newpath	/* New location file path */
	 );
/*
**	Copy message into new location. The copy will fail if the
**	new location exists or is a directory typed as "eurobridge/message".
**	If new location is a directory, then the message is copied into
**	this direcory. If new location does not exist, the message
**	will be copied to this name.
**
**	After successful copy operation, the MFM information is
**	updated.
*/
int MamCopyMessage
	(char *path,	/* Message file path */
	 char *newpath	/* New location file path */
	 );
/*
**	Delete message.
*/
int MamDeleteMessage
	(char *path	/* Message file path */
	 );
/*
**	Return the actual filename corresponding a name which is used in
**	content of a msg as filename parameter value in the content-type
**	specificication.
**
**	Returns a pointer to static area which is valid until next call
**	to this function, or returns NULL in case of any error condition.
*/
char *MamContentFile
	(char *filename,/* filename parameter value */
	 Message msg	/* associated message structure */
	 );

/*
**	Initialize directory scanning for messages (files of the type
**	"eurobridge/message"). The filter function will be called for
**	each potential message and it must indicate by the return value
**	whether this message is included (non-zero) or not (zero). The
**	client data is passed as such to the filter.
**
**	The returned folder must be close with MamCloseFolder.
**
** Returns
**	a pointer to the MamFolderRec structure or NULL, if opening
**	fails for some reason.
*/	
MamFolder MamOpenFolder
	(char *path,		/* Folder file path */
	 MamFilter filter,	/* Filter function to select the messages */
	 void *data		/* Client data for the filter function */
	 );
/*
**	Access the next message name from the folder.
**
** Returns
**	a pointer to a folder relative message filename or NULL, if
**	all messages have been processed.
*/
char *MamNextMessage
	(MamFolder folder	/* Folder data structure */
	 );
/*
**	Close the folder (release the allocated data structures)
*/
void MamCloseFolder
	(MamFolder folder	/* Folder data structure */
	 );

/*
**	Create a folder directory and set the type as "eurobridge/folder".
**
**	Returns zero on success.
*/
int MamCreateFolder
	(char *path	/* Folder file path */
	 );
/*
**	Delete a folder directory. This succeeds only if the folder
**	is empty.
**
**	Returns zero on success.
*/
int MamDeleteFolder
	(char *path	/* Folder file path */
	 );
/*
**	Move folder into new location. The move will fail if the
**	new location exists or is a directory typed as "eurobridge/message".
**	If new location is a directory, then the message is moved into
**	this direcory. If new location does not exist, the message
**	will be renamed to this name.
**
**	After successful move operation, the MFM information is
**	updated.
*/
int MamMoveFolder
	(char *path,	/* Folder directory path */
	 char *newpath	/* New location path */
	 );

int MamSetLocation(char *, char *);
char *MamGetLocation(char *);

/*
** MamCaching support
*/
typedef enum
    {
	MamCache_ACCESS,/* Access cached information only, do not scan */
	MamCache_SCAN,	/* Access cached information, scan for changes */
	MamCache_UPDATE	/* Access cached information, scan and update cache */
    }  MamCacheMode;

typedef struct MamCacheRec
    {
	char *name;	/* Message filename in folder */
	char *data;	/* User defined cache data */
	int length;	/* Length of the cache data (bytes) */
	/*
	** changed
	**	is True, if the message has been modified since the cache
	**	was saved last time [This means either modified or totally
	**	new message]. Note that if the folder is a file, any change
	**	in the middle of the folder will cause all of the messages
	**	after it to be marked as changed (because 'offset' will
	**	change).
	*/
	unsigned int changed:1;
	/*
	** malloc_name, malloc_data
	**	is True, when the 'name' string or 'data' has been
	**	separately allocated with 'malloc'. If False, then the
	**	pointer points to the internal common buffer, which will
	**	get released in MamCacheFree. (The real start address and
	**	length of this internal buffer is stored into the
	**	sentinel 'data' and 'length' fields).
	*/
	unsigned int malloc_name:1;
	unsigned int malloc_data:1;
    } MamCacheRec, *MamCache;

/*
**	Function to which gets called by MamCacheAccess to convert message
**	information into cached information.
**
**	Returns a pointer to dynamically (malloc) allocated memory area
**	which contains the cached information. Can also return a NULL,
**	which represents empty cache information.
*/
typedef char *(*MamCacheData)
	(Message msg,	/* Message from which to generate cache information,
			   can also be NULL. */
	 int *length	/* The length of the cache information in bytes. */
	 );

/*
**	MamCacheAccess will return the current uptodate list of
**	<name, cache-info> pairs. It will read the cache file from
**	folder, noting the last update time. Then it will execute a
**	directory scan+stat search within folder, comparing the
**	modification times of files to the cache file time. It will
**	regenerate the cache data for every message that has been
**	modified after the last cache write. It will also elimitate
**	references to non-existing messages from the cache.
**
**	If it needed to change any of the cached information, it will
**	write a new cache file out at the end of the scan, if parameter
**	update has non-zero value.
**
** Returns
**	a pointer to an array of MamCacheRec structures. The end of the
**	array is indicated by a sentinel structure which has NULL name.
**	A NULL return indicates an error condition, an empty folder is
**	indicated by array containing only the sentinel record.
**
** Warning:
**	The ordering of entries in the returned array may be different
**	on two consecutive calls.
**
** Bugs:
**	No attempt is made to handle simultaneous cache update by
**	multiple processes.
*/
MamCache MamCacheAccess
	(char *path,		/* Folder path */
	 char *cache_name,	/* Cache file name (within folder) */
	 MamCacheData conv,	/* User defined Message -> cached data
				   converter function. If NULL, then NULL
				   cache data is assumed. */
	 MamCacheMode mode	/* Access/Scan/Update mode */
	 );
/*
**	Release the data structure returned by MamCacheAccess function.
*/
void MamCacheFree
	(
	 MamCache info	/* Cache information */
	 );
#endif



