/* --------------------------------- window.c ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Manage the different windows.
*/

#include "fly.h"


extern void FAR
windows_term (void)
{
	int	i;

	for (i = 0; i < NHDD; ++i) {
		if (!(st.hdd[i].flags & HDF_ON))
			continue;
		st.hdd[i].type = HDT_NONE;
		st.hdd[i].flags = 0;
		DEL0 (st.hdd[i].view.window);
		DEL0 (st.hdd[i].view.viewport);
		st.hdd[i].view.screen = 0;
	}
}

static void NEAR
window_set (int i, int type, int orgx, int orgy, int maxx, int maxy)
{
	WINDOW	*w;

	if (i < 0)
		w = CW;
	else if (T(w = NEW (st.hdd[i].view.window))) {
		st.hdd[i].type = type;
		st.hdd[i].flags = HDF_ON;
		st.hdd[i].view.viewport = 0;
		st.hdd[i].view.screen = CS;
	} else
		return;
	w->orgx = orgx;
	w->orgy = orgy;
	w->maxx = maxx;
	w->maxy = maxy;
	if (i < 0 && VIS_STEREOSCOPIC == st.stereo)
		set_small_frame ();
}

extern void FAR
windows_set (void)
{
	windows_term ();

	if (WIN_LANDSCAPE == st.windows) {
		window_set (-1, 0,         FONE/2,   FONE/3,   FONE/2, FONE/3);
		window_set ( 0, HDT_REAR,  FONE/24*5,FONE/6*5,FONE/24*5,FONE/6);
		window_set ( 1, HDT_PANEL, FONE/24*13,FONE/6*5,FONE/8, FONE/6);
		window_set ( 2, HDT_RADAR, FONE/6*5, FONE/6*5, FONE/6, FONE/6);
	} else if (WIN_PORTRATE == st.windows) {
		window_set (-1, 0,         FONE/3,   FONE/2,   FONE/3, FONE/2);
		window_set ( 0, HDT_REAR,  FONE/6*5, FONE/3,   FONE/6, FONE/3);
		window_set ( 1, HDT_RADAR, FONE/6*5, FONE/6*5, FONE/6, FONE/6);
	} else if (WIN_SQUARE == st.windows) {
		window_set (-1, 0,         FONE/8*3, FONE/2, FONE/8*3, FONE/2);
		window_set ( 0, HDT_REAR,  FONE/8*7, FONE/6,   FONE/8, FONE/6);
		window_set ( 1, HDT_PAN,   FONE/8*7, FONE/6*3, FONE/8, FONE/6);
		window_set ( 2, HDT_RADAR, FONE/8*7, FONE/6*5, FONE/8, FONE/6);
	} else if (WIN_WIDE == st.windows) {
		window_set (-1, 0,         FONE/2,   FONE/3,   FONE/2, FONE/3);
		window_set ( 0, HDT_REAR,  FONE/8,   FONE/6*5, FONE/8, FONE/6);
		window_set ( 1, HDT_PANEL, FONE/8*3, FONE/6*5, FONE/8, FONE/6);
		window_set ( 2, HDT_PAN,   FONE/8*5, FONE/6*5, FONE/8, FONE/6);
		window_set ( 3, HDT_RADAR, FONE/8*7, FONE/6*5, FONE/8, FONE/6);
	} else if (WIN_PANORAMA == st.windows) {
		window_set (-1, 0,         FONE/6*3, FONE/3,   FONE/6, FONE/3);
		window_set ( 0, HDT_LEFT,  FONE/6*1, FONE/3,   FONE/6, FONE/3);
		window_set ( 1, HDT_RIGHT, FONE/6*5, FONE/3,   FONE/6, FONE/3);
		window_set ( 2, HDT_REAR,  FONE/8,   FONE/6*5, FONE/8, FONE/6);
		window_set ( 3, HDT_PANEL, FONE/8*3, FONE/6*5, FONE/8, FONE/6);
		window_set ( 4, HDT_PAN,   FONE/8*5, FONE/6*5, FONE/8, FONE/6);
		window_set ( 5, HDT_RADAR, FONE/8*7, FONE/6*5, FONE/8, FONE/6);
	} else if (WIN_ETHER == st.windows) {
		window_set (-1, 0,		FCON(0.50),	FCON(.30),
						FCON(0.485),	FCON(.28));
		window_set ( 0, HDT_STORES,	FCON(0.17),	FCON(.80),
						FCON(0.17),	FCON(.20));
		window_set ( 1, HDT_RADAR,	FCON(0.51),	FCON(.80),
						FCON(0.17),	FCON(.20));
		window_set ( 2, HDT_PAN,	FCON(0.84),	FCON(.87),
						FCON(0.16),	FCON(.13));
		window_set ( 3, HDT_NONE,	FCON(0.84),	FCON(.67),
						FCON(0.16),	FCON(.07));
		window_set ( 4, HDT_NONE,	FCON(0.50),	FCON(.30),
						FCON(0.50),	FCON(.30));
		if (IS_PLANE(CC))
			EE(CC)->hdd |= HDD_COMPASS|HDD_SQRCOMPASS|HDD_ORTCOMPASS;
	} else {
		window_set (-1, 0,         FONE/2,   FONE/2,   FONE/2, FONE/2);
	}
}

extern void FAR
set_screen (int sizex, int sizey)
{
	if (CS->device->sizex && CS->device->sizey) {
		CS->device->lengx = muldiv (CS->device->lengx, sizex,
						CS->device->sizex);
		CS->device->lengy = muldiv (CS->device->lengy, sizey,
						CS->device->sizey);
	}

	CS->device->sizex = sizex;
	CS->device->sizey = sizey;

	if (st.windows)
		CS->sizex = (CS->device->sizex-3)/2*2+1;
	else
		CS->sizex = CS->device->sizex/4*3-2;
	CS->minx = sizex - CS->sizex - 2;

	CS->sizey = (CS->device->sizey-3)/2*2+1;
	CS->miny = 1;
}

extern void FAR
adjust_viewports (void)
{
	int		t;
	OBJECT		*v;
	VIEWPORT	*vp;
/*
 * Adjust all viewports to main window's aspect ratio. This adjustment is
 * necessary because the main window and the main viewport are correctly
 * adjusted at all times. When selecting an external-view the viewports are
 * swapped, and if the new viewport has the wrong aspect ratio then the
 * main view is distorted. Note that HDD displays dynamicaly adjust the
 * aspect ratios just before using it [views.c: show_view()].
*/
	for (v = CO; v; v = v->next) {
		if (T(vp = v->viewport) && vp != CP) {
			vp->maxx  = CP->maxx;
			vp->maxy  = CP->maxy;
			vp->distz = CP->distz;
			vp->x     = CP->x;
			vp->y     = CP->y;
			t = vp->zoom - CP->zoom;
			vp->zoom  = CP->zoom;
			zoom (vp, t);
		}
	}
}

extern void FAR
set_viewport (void)
{
	int		tx, ty;

	if (CS->device->lengx == 0 || CS->device->lengy == 0) {
		LogPrintf ("null viewport\n");
		die ();
	}
	tx = fmul (CS->sizex, CW->maxx);
	tx = muldiv (FONE, tx, CS->device->sizex);
	ty = fmul (CS->sizey, CW->maxy);
	ty = muldiv (FONE, ty, CS->device->sizey);
	if (CS->device->lengx > CS->device->lengy)
		ty = muldiv (ty, CS->device->lengy, CS->device->lengx);
	else
		tx = muldiv (tx, CS->device->lengx, CS->device->lengy);

	CP->maxx = tx;			/* maps world to viewport */
	CP->maxy = ty;
	CP->distz = 1*VONE*VONE;
#if 0
	if (tx < ty)
		CP->z = 2*tx;		/* 45 degrees view in x */
	else
		CP->z = 2*ty;		/* 45 degrees view in y */
#else
	CP->z = 2*tx;			/* always 45 degrees view in x */
#endif
	CP->x = 0*FONE;			/* viewer at center */
	CP->y = 0*FONE;
	tx = CP->zoom;
	CP->zoom = 0;
	zoom (CP, tx);

	adjust_viewports ();
}

extern void FAR
set_textwin (void)
{
	int	FreePixels, FreeCols, border, color;

	if (CT)
		Gr->CloseTextWindow (CT);
	else if (!NEW (CT)) {
		LogPrintf ("out of memory [%s(%u)]\n", __FILE__, __LINE__);
		die ();
	}

	FreePixels = CS->device->sizex - (CS->sizex + 2);
	FreeCols = FreePixels / CS->device->FontWidth;
	if (FreeCols < 21) {		/* windows overlap */
		CT->minx = 1;
		CT->miny = CS->miny+1;
		CT->sizex = 20*CS->device->FontWidth;
		CT->sizey = CS->sizey-2;
		border = -1;
		color = st.black;
	} else {
		CT->minx = 0;
		CT->miny = 0;
		CT->sizex = FreePixels-1;
		CT->sizey = CS->device->sizey;
		border = st.lgray;
		color = st.tbg;
	}

	if (st.flags & SF_BLANKER) {
		border = -1;
		color = st.black;
	}

	CT->device = CS->device;
	CT->FgColor = st.tfg;
	CT->BgColor = color;
	CT->BoColor = border;
	CT->FontWidth = CT->device->FontWidth;
	CT->FontHeight = CT->device->FontHeight;

	Gr->OpenTextWindow (CT);
}

static MENU MenuWin[] = {
	{'0', "text"},		/*  0 */
	{'1', "full"},		/*  1 */
	{'2', "landscape"},	/*  2 */
	{'3', "portrate"},	/*  3 */
	{'4', "square"},	/*  4 */
	{'5', "wide"},		/*  5 */
	{'6', "panorama"},	/*  6 */
	{'e', "ether"},		/*  7 */
	{'c', "configure"},	/*  8 */
{'\0', 0}};

extern int FAR
menu_windows (void)
{
	int	sel, i, ch, ret;
	Ushort	flags;
	HMSG	*m;

	for (ret = 0; !ret;) {
		sel = menu_open (MenuWin, st.windows);

		switch (sel) {
		case MENU_ABORTED:
		case MENU_FAILED:
			ret = 1;
			break;
		default:
			flags = st.flags;
			st.flags |= SF_SIMULATING;

			reset_page (1);

			st.windows = sel;

			set_screen (CS->device->sizex, CS->device->sizey);
			windows_set ();
			set_textwin ();
			set_viewport ();

			show_fixed (0);

			st.flags = flags;
			sel = 8;	/* could just fall through... */
			break;
		case 8:
			if (st.flags1 & SF_EXTVIEW) {
				MsgEPrintf (50, "Exit ExtView!");
				break;
			}
			st.flags1 |= SF_IDENT;
			for (;;) {
				m = MsgWPrintf (0, "Which window?");
				ch = mgetch ();
				msg_del (m);
				if (KF_ESC == ch || K_ENTER == ch)
					break;
				ch -= '0';
				if (ch < 0 || ch > NHDD ||
				    (ch && !(st.hdd[ch-1].flags & HDF_ON)))
					continue;
				i = ch ? st.hdd[ch-1].type : st.extview;
				i = menu_view (i);
				if (i >= 0) {
					if (ch)
						st.hdd[ch-1].type = i;
					else
						st.extview = i;
				}
			}
			st.flags1 &= ~SF_IDENT;
			ret = 1;
			break;
		}

		if (MENU_FAILED != sel)
			menu_close ();
	}

	return (0);
}

#define	ZOOMIN	20643
#define	ZOOMOUT	13004

extern void FAR
zoom (VIEWPORT *vp, int zoom)
{
	if (!vp || !zoom)
		return;

	for (; zoom > 0; --zoom) {
#if 0
		if (vp->distz > VMAX/2)
			break;
		if (vp->z < ZOOMOUT) {
#endif
#if 1
		if (vp->z < ZOOMOUT && vp->distz < VMAX/2) {
#endif
			++vp->zoom;
			vp->z = fmul (vp->z, ZOOMIN);
			vp->distz = fmul (vp->distz, ZOOMIN);
		} else if (vp->maxx > FONE/128 && vp->maxy > FONE/128) {
			++vp->zoom;
			vp->maxx = fmul (vp->maxx, ZOOMOUT);
			vp->maxy = fmul (vp->maxy, ZOOMOUT);
#if 0
			vp->distz = fmul (vp->distz, ZOOMIN);
#endif
		}
#if 1
		else
			break;
#endif
	}

	for (; zoom < 0; ++zoom) {
#if 0
		if (vp->distz < VONE)
			break;
#endif
		if (vp->maxx < ZOOMOUT && vp->maxy < ZOOMOUT) {
			--vp->zoom;
			vp->maxx = fmul (vp->maxx, ZOOMIN);
			vp->maxy = fmul (vp->maxy, ZOOMIN);
#if 0
			vp->distz = fmul (vp->distz, ZOOMOUT);
		} else if (vp->z > FONE/128) {
#endif
#if 1
		} else if (vp->z > FONE/128 &&  vp->distz > VONE) {
#endif
			--vp->zoom;
			vp->z = fmul (vp->z, ZOOMOUT);
			vp->distz = fmul (vp->distz, ZOOMOUT);
		}
#if 1
		else
			break;
#endif
	}
}

#undef ZOOMIN
#undef ZOOMOUT

extern void FAR
clear_text (void)
{
	Gr->TextClear ();
	show_fixed (0);
	Gr->SetTextPos (1, 1);
}
