/* --------------------------------- pointer.c ------------------------------ */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Pointing devices manager.
*/

#include "fly.h"


extern struct PtrDriver NEAR* FAR PtrDrivers[];

extern int FAR
pointers_init (void)
{return (0);}

extern void FAR
pointers_term (void)
{}

static int def_opt[NOPTS] = {
	1,		/* +ve x */
	0,		/* x axis is channel 0 */
	1,		/* +ve y */
	1		/* y axis is channel 1 */
};

extern POINTER * FAR
pointer_select (char *name)
{
	int			l, n;
	char			*options;
	int			opt[NOPTS];
	POINTER			*ptr;
	struct PtrDriver	NEAR* FAR* p;

	memcpy (opt, def_opt, sizeof (opt));

	if (!name) {
		p = PtrDrivers;				/* use default */
		goto ret;
	}

	options = strchr (name, ':');
	if (options) {		/* ":+x+y" */
		n = options - name;			/* name length */
		++options;
		l = strlen (options);			/* options length */

		if (l >= 1) {
			if (options[0] == '-')		/* x direction */
				opt[0] = -1;
			else if (options[0] == '+')
				opt[0] = 1;
			else
				return (0);
		}
		if (l >= 2) {
			if (options[1] == 'x')		/* x select */
				opt[1] = 0;
			else if (options[1] == 'y')
				opt[1] = 1;
			else
				return (0);
		}

		if (l >= 3) {
			if (options[2] == '-')		/* y direction */
				opt[2] = -1;
			else if (options[2] == '+')
				opt[2] = 1;
			else
				return (0);
		}
		if (l >= 4) {
			if (options[3] == 'x')		/* y select */
				opt[3] = 0;
			else if (options[3] == 'y')
				opt[3] = 1;
			else
				return (0);
			if (opt[1] == opt[3])		/* exclusive */
				return (0);
		}
	} else
		n = strlen (name);

	for (p = PtrDrivers; *p; ++p) {
		if (!strnicmp ((*p)->name, name, n) && !(*p)->name[n])
			break;
	}
ret:
	if (!*p)
		return (0);

	if (!NEW (ptr))
		return (0);
	ptr->control = *p;
	ptr->name    = (*p)->name;
	memcpy (ptr->opt, opt, sizeof (ptr->opt));
	return (ptr);
}

extern POINTER * FAR
pointer_release (POINTER *ptr)
{
	ptr->control->Term (ptr);
	DEL (ptr);
	return (0);
}

extern void FAR
std_key (POINTER *p, int key)
{
	int	i;

	switch (key) {
	case KF_POWER_AB:
		if (p->l[3] < 75) {
			p->l[3] = 75;		/* full power first */
			break;
		}
		++p->l[3];			/* ignite AB now */
				/* fall through */
	case KF_POWER_UP:
		i = (p->l[3] > 75) ? 100 : 75;
		p->l[3] += 5;
		if (p->l[3] > i)
			p->l[3] = i;
		break;

	case KF_POWER_DOWN:
		p->l[3] -= 5;
		if (p->l[3] < 0)
			p->l[3] = 0;
		break;

	case KF_POWER_0:
		p->l[3] = 0;
		break;

	case KF_POWER_100:
		p->l[3] = 75;
		break;

	case KF_LEVEL:
		++p->b[0];
		break;

	case KF_ORIGIN:
		++p->b[1];
		break;

	case KF_FIRE:
		++p->b[2];
		break;

	case KF_RESET_ROLL:
		++p->b[4];
		break;

	case KF_STABLE:
		++p->b[3];
		break;

	case KF_FRUDLEFT:		/* rudder left */
		p->l[2] += 10;
		if (p->l[2] > 100)
			p->l[2] = 100;
		break;
	case KF_FRUDRITE:		/* rudder right */
		p->l[2] -= 10;
		if (p->l[2] < -100)
			p->l[2] = -100;
		break;
	case KF_FRUDCNTR:		/* rudder center */
		p->l[2] = 0;
		break;

	case ']':			/* flaps: more */
		p->l[6] += 10;
		if (p->l[6] > 100)
			p->l[6] = 100;
		break;
	case '[':			/* flaps: less */
		p->l[6] -= 10;
		if (p->l[6] < 0)
			p->l[6] = 0;
		break;

	case '}':			/* spoilers: more */
		p->l[7] += 10;
		if (p->l[7] > 100)
			p->l[7] = 100;
		break;
	case '{':			/* spoilers: less */
		p->l[7] += 10;
		if (p->l[7] > 100)
			p->l[7] = 100;
		break;

	case ')':			/* wheel brakes: more */
		p->l[9] += 10;
		if (p->l[9] > 100)
			p->l[9] = 100;
		break;
	case '(':			/* wheel brakes: less */
		p->l[9] -= 10;
		if (p->l[9] < 0)
			p->l[9] = 0;
		break;

	case '>':			/* speed brakes: more */
		p->l[8] += 25;
		if (p->l[8] > 100)
			p->l[8] = 100;
		break;
	case '<':			/* speed brakes: less */
		p->l[8] -= 25;
		if (p->l[8] < 0)
			p->l[8] = 0;
		break;

	case '+':			/* speed brakes on/off */
		if (p->l[8])
			p->l[8] = 0;
		else

			p->l[8] = 100;
		break;

	case 'b':			/* wheels brakes on/off */
		if (p->l[9])
			p->l[9] = 0;
		else
			p->l[9] = 100;
		break;

	case 'g':
		++p->b[5];		/* landing gear up/down */
		break;

	case ' ':
		++p->b[6];		/* release radar lock */
		break;

/* Ctl Arrows used for trim.
*/
	case KF_ZRIGHT:			/* trim: right */
		++p->l[4];
		if (p->l[4] > 100)
			p->l[4] = 100;
		break;
	case KF_ZLEFT:			/* trim: left */
		--p->l[4];
		if (p->l[4] < -100)
			p->l[4] = -100;
		break;
	case KF_ZUP:			/* trim: nose down */
		--p->l[5];
		if (p->l[5] < -100)
			p->l[5] = -100;
		break;
	case KF_ZDOWN:			/* trim: nose up */
		++p->l[5];
		if (p->l[5] > 100)
			p->l[5] = 100;
		break;
	case '\\':			/* trim: reset */
		p->l[4] = p->l[5] = 0;
		break;
	}
}

/* Select Pointing Device
*/

extern int FAR
menu_ptrs (void)
{
	MENU	*MenuPtr;
	int	sel, i, n, nEntries, EntrySize;
	short	flags;
	char	*oldptr, newptr[256];
	POINTER	*ptr;

	for (nEntries = 0; PtrDrivers[nEntries]; ++nEntries);
		;
	EntrySize = 20;

	n = (nEntries+1) * sizeof (*MenuPtr);
	if (!(MenuPtr = (MENU *) memory_alloc (n)))
		return (1);

	sel = MENU_FAILED;
	for (i = 0; i < nEntries; ++i)
		if (!(MenuPtr[i].text = (char *) memory_alloc (EntrySize)))
			goto ret;

	if (T(oldptr = strchr (st.ptrname, '=')))
		++oldptr;
	else
		oldptr = st.ptrname;
	sel = 0;
	for (i = 0; i < nEntries; ++i) {
		MenuPtr[i].letter = (Uchar)menuch[i];
		strcpy (MenuPtr[i].text, PtrDrivers[i]->name);
		if (!stricmp (PtrDrivers[i]->name, oldptr))
			sel = i;
	}

	sel = menu_open (MenuPtr, sel);

	oldptr = st.ptrname;

	switch (sel) {
	case MENU_ABORTED:
	case MENU_FAILED:
		break;
	default:
		strcpy (newptr, PtrDrivers[sel]->name);
		strcat (newptr, ":");
		i = strlen (newptr);
		getstr ("pointer options:", newptr + i, sizeof (newptr) - i);

		flags = st.flags;
		st.flags |= SF_SIMULATING;
		if (T(ptr = CV->pointer))
			CV->pointer = pointer_release (ptr);
		for (;;) {
			ptr = pointer_select (newptr);
			if (ptr && !ptr->control->Init (ptr, newptr)) {
				CV->pointer = ptr;
				st.ptrname = xfree (st.ptrname);
				st.ptrname = xstrdup (newptr);
				break;
			}
			MsgEPrintf (-100, "pointer init failed");
			ptr = pointer_select (oldptr);
			if (ptr && !ptr->control->Init (ptr, 0)) {
				CV->pointer = ptr;
				break;
			}
			MsgEPrintf (-100, "old pointer init failed");
			if (oldptr) {
				ptr = pointer_select (NULL);
				if (ptr && !ptr->control->Init (ptr, 0)) {
					CV->pointer = ptr;
					st.ptrname = xfree (st.ptrname);
					st.ptrname = xstrdup (ptr->name);
					break;
				}
			}
			LogPrintf ("default pointer init failed\n");
			die ();
		}
		st.flags = flags;
		break;
	}

ret:
	for (i = 0; i < nEntries; ++i)
		if (MenuPtr[i].text)
			memory_free (MenuPtr[i].text, EntrySize);

	memory_free (MenuPtr, n);

	if (MENU_FAILED == sel)
		return (1);

	menu_close ();
	return (0);
}


/* From here on: buttons stuff.
*/

#define BTN_POSITION	0x01
#define BTN_DEBOUNCE	0x02
#define BTN_RELEASE	0x04

#define LASTBTN		p->opt[19]		/* last button pressed */

/* read button debounce definition.
*/
extern void FAR
get_btn (POINTER *p, char *options)
{
	char	*s;
	int	n;

	for (n = 0; n < rangeof (p->d); ++n)
		p->d[n] = BTN_DEBOUNCE;

	if (T(s = get_arg (options, "ndb="))) {
		while ((n = opt36 (*s++)) >= 0)
			if (n < rangeof (p->d))
				p->d[n] &= ~BTN_DEBOUNCE;
	}

	if (T(s = get_arg (options, "rel="))) {
		while ((n = opt36 (*s++)) >= 0)
			if (n < rangeof (p->d))
				p->d[n] |= BTN_RELEASE;
	}
}

/* handle a button press with debouncing.
*/
extern void FAR
do_btn (POINTER *p, int button, int state)
{
	Ushort	btn[1];

	state = state ? BTN_POSITION : 0;

	if (!(BTN_DEBOUNCE & p->d[button]))		/* no debounce */
		;
	else if ((BTN_POSITION & p->d[button]) != state)	/* new state */
		p->d[button] ^= BTN_POSITION;
	else						/* no change */
		return;

	if (state)
		btn[0] = K_BTN;
	else if (BTN_RELEASE & p->d[button])
		btn[0] = K_RLS;
	else
		return;
	btn[0] ^= st.btnmode | menuch[button];
	mac_interpret (btn, rangeof (btn));
	LASTBTN = button;
}

extern void FAR
do_btns (POINTER *p, char *btn, int size)
{
	int	i;

	for (i = 0; i < size; ++i)
		if (!btn[i])
			do_btn (p, i, 0);	/* button release */

	for (i = 0; i < size; ++i)
		if (btn[i])
			do_btn (p, i, 1);	/* button press */
}

/* Set buttons mode.
*/

static MENU FAR MenuBtn[] = {
	{'0', "off"},
	{'1', "on"},
	{'2', "toggle"},
	{'a', "Alt"},		/*  3 */
	{'c', "Ctrl"},		/*  4 */
	{'s', "Shift"},		/*  5 */
	{'p', "Special"},	/*  6 */
	{'x', "New"},		/*  7 */
	{'=', "Set"},		/*  8 */
	{'d', "Debounce"},	/*  9 */
	{'r', "Release"},	/* 10 */
{'\0', 0}};

extern int FAR
menu_btn (void)
{
	int	sel, quit, ch;
	Ushort	i, j;
	HMSG	*m;
	POINTER *p;

	SetOption (0, 2);		/* default mode: toggle */
	sel = 2;
	for (quit = 0; !quit;) {
		sel = menu_open (MenuBtn, sel);
		switch (sel) {
		case MENU_ABORTED:
		default:
			quit = 1;
			break;
		case 0:
		case 1:
		case 2:
			SetOption (0, sel);
			break;
		case 3:
			SetOption (&st.btnmode, K_ALT);
			break;
		case 4:
			SetOption (&st.btnmode, K_CTRL);
			break;
		case 5:
			SetOption (&st.btnmode, K_SHIFT);
			break;
		case 6:
			SetOption (&st.btnmode, K_SPECIAL);
			break;
		case 7:
			st.btnmode = 0;
			SetOption (0, 1);
			break;
		case 8:
			m = MsgEPrintf (0, "buttons mode[%x]?",
							(int)st.btnmode);
			ch = mgetch ();
			msg_del (m);
			if (-1 != (ch = opt36(ch)) && ch < 16)
				st.btnmode = ch << 8;
			quit = 1;
			break;
		case 9:
		case 10:
			quit = 1;
			if (F(p = CC->pointer))
				break;
			m = MsgEPrintf (0, "enter button name:");
			ch = mgetch ();
			msg_del (m);
			ch = opt36 (ch);
			if (ch >= 0) {
				j = (9 == sel) ? BTN_DEBOUNCE : BTN_RELEASE;
				i = p->d[ch];
				SetOption (&i, j);
				p->d[ch] = i;
				MsgWPrintf (10, "button %u %s is %s", ch,
					(9 == sel) ? "Debounce" : "Release",
					(i & j) ? "On" : "Off");
			}
			break;
		}
		if (MENU_FAILED != sel)
			menu_close ();
	}
	return (0);
}
#undef NO_DEBOUNCE
#undef LASTBTN
