/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <pcap-int.h>
#include <X11/Intrinsic.h>
#include "xip.h"
#include "XmgFancy.h"
#include "pkt.h"
#include "lay_ether.h"
#include "lay_pcap.h"
#include "lay_null.h"
#include "xippktvec.h"

extern XtAppContext		app_context;
extern Widget			toplevel;
extern Widget			pktbox;
extern t_boolean		capflow;
extern t_boolean		big_packets;

extern char			*pcap_dev;
extern char			hard_filter[];
extern int			pcap_snaplen;
extern t_boolean		pcap_promisc;
extern int			pcap_to_ms;
extern t_boolean		pcap_optimize;
extern char			*pcap_fname;
extern int			pcap_cnt;

pcap_t				*pcap;
int				user_dropped = 0;

/* opens pcap, possibly opening "offline" if pcap_fname is not NULL.
   If pcap_dev is NULL, it performs pcap_lookupdev(3).
   If there is a problem it prints the appropriate error message then
   exits(2) since the use of libpcap is crucial in such a program. */
pcap_t				*open_pcap(VOID_DECL)
{
  char				errbuf[BUFSIZ];
  pcap_t			*pcap;

  if (pcap_fname)
    {
      if ((pcap = pcap_open_offline(pcap_fname,
				    errbuf)) == NULL)
	{
	  fprintf(stderr,"pcap_open_offline: %s\n",errbuf);
	  exit(1);
	}
    }
  else
    {
      if (!pcap_dev)
	if ((pcap_dev = pcap_lookupdev(errbuf)) == NULL)
	  {
	    fprintf(stderr,"%s\n",errbuf);
	    exit(1);
	  }
      if ((pcap = pcap_open_live(pcap_dev,
				 pcap_snaplen,
				 pcap_promisc,
				 pcap_to_ms,
				 errbuf)) == NULL)
	{
	  fprintf(stderr,"pcap_open_live: %s\n",errbuf);
	  exit(1);
	}
    }
  return (pcap);
}

/* compiles the bpf program and sets it to the main pcap structure. 
   It exits(2) if something goes wrong. */
VOID_FUNC			set_hard_filter(pcap,hard_filter)
pcap_t				*pcap;
char				*hard_filter;
{
  struct bpf_program		bp;

  if (pcap_compile(pcap,
                   &bp,
                   hard_filter,
                   pcap_optimize,
                   0) < 0)
    {
      fprintf(stderr,"problems in pcap_compile: %s\n",hard_filter);
      exit(1);
    }
  if (pcap_setfilter(pcap,&bp) < 0)
    {
      fprintf(stderr,"can't set pcap_setfilter\n");
      exit(1);
    }
}

struct				s_singleton
{
  struct pcap_pkthdr		*pp;
  char				*buf;
};

/* transforms a pcap "link type" to a "lay" msg_proc.
   Returns 0 if OK, -ERR_NOENT if no equivalent is found. */
t_status			lay_get_from_pcap(pcap_link_type,mp)
int				pcap_link_type;
t_msg_proc			*mp;		/* "Lay" msg_proc */
{
  switch (pcap_link_type)
    {
    case DLT_EN10MB:
      (*mp) = &lay_ether_msg;
      return (0);
    case DLT_SLIP:
      (*mp) = &lay_slip_msg;
      return (0);
    case DLT_NULL:
      (*mp) = &lay_null_msg;
      return (0);
    case DLT_PPP:
      (*mp) = &lay_ppp_msg;
      return (0);
    case DLT_FDDI:
      (*mp) = &lay_fddi_msg;
      return (0);
    case DLT_IEEE802:
      (*mp) = &lay_ether_msg;
      return (0);
#ifdef DLT_ATM_RFC1483
    case DLT_ATM_RFC1483:
      (*mp) = &lay_atm_rfc1483_msg;
      return (0);
#endif
    }
  return (-ERR_NOENT);
}

/* fills out a "pkt" structure against pcap information.
   It uses lay_get_from_pcap(3).
   Returns 0 if OK, Might return various errors. */
t_status			getpkt(buf,len,caplen,ts,pcap,pkt)
char				*buf;
int				len;
int				caplen;
struct timeval			*ts;
pcap_t				*pcap;
t_pkt				*pkt;
{
  t_status			status;
  
  if ((status = lay_get_from_pcap(pcap_datalink(pcap),&(pkt->mp))) < 0)
    return (status);
  pkt->buf = buf;
  pkt->len = caplen;
  pkt->netlen = len;
  FBCOPY(ts,&(pkt->ts),sizeof (struct timeval));
  return (0);
}

/* is a pcap_handler. 
   If the global "capflow" is TRUE, then it creates a graphical packet in
   using XipCreatePkt(3) else it increments the "user_dropped" global. */
VOID_FUNC			read_pcap_handler(s,pp,buf)
struct s_singleton		*s;
struct pcap_pkthdr		*pp;
char				*buf;
{
  t_status			status;
  t_boolean			match;
  t_pkt				pkt;
  t_pkt				*npkt;

  if (capflow)
    {
      if ((status = getpkt(buf,
			   pp->len,
			   pp->caplen,
			   &(pp->ts),
			   pcap,
			   &pkt)) < 0)
	{
	  err_print(-status,"pkt_get");
	  return ;
	}
      if ((npkt = xip_pkt_dup(&pkt,
			      &status)) == NULL)
	{
	  XmgErrPrint(-status,"xip_pkt_dup");
	  return ;
	}
      XipCreatePkt(big_packets?toplevel:pktbox,
		   npkt,
		   TRUE);
    }
  else
    user_dropped++;
}

/* is a XtInputCallbackProc.
   It reads the pcap file descriptor. */
VOID_FUNC			read_pcap(closure,fd,id)
XtPointer			closure;
int				*fd;
XtInputId			*id;
{
  struct s_singleton		s;
  
  if (pcap_read(pcap,
		1,
		(pcap_handler)read_pcap_handler,
		(u_char *)(&s)) < 0)
    {
    }
}

/* initializes the pcap part of xip.
   It needs "app_context" to be initialized since it uses XtAppAddInput(3).
   It pcap_fname is not NULL, it loops on the file. */
VOID_FUNC			xipcap_init(VOID_DECL)
{
  pcap = open_pcap();
  set_hard_filter(pcap,hard_filter);
  if (pcap_fname)
    {
      pcap_loop(pcap,
		pcap_cnt,
		(pcap_handler)read_pcap_handler,
		NULL);
    }
  else
    {
      XtAppAddInput(app_context,
		    pcap->fd,
		    (XtPointer)XtInputReadMask,
		    read_pcap,
		    NULL);
    }
}
