/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include "layer.h"
#include "typ_msg.h"
#include "typ_8.h"

/* is a t_msg_proc.
   Manages characters. It recognizes ASCII characters (1 byte), 
   hexadecimal definition (2 bytes), octal definitions (3 bytes). 
   (Using strtoul(3) for hex and octal.) */
t_status		typ_char_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("char");
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u8		c;
	
	if (ed->b.len < sizeof (t_s8))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&c,sizeof (t_s8));
	switch ((int)(ed->data))
	  {
	  case CHAR_C:
	    return (str_cat_char(bs->str,
				 bs->max_len,
				 c));
	  case CHAR_XX:
	    return (str_cat_fmt_va(bs->str,
				   bs->max_len,
				   "%02x",
				   (t_u8)c));
	  case CHAR_OOO:
	    return (str_cat_fmt_va(bs->str,
				   bs->max_len,
				   "%03o",
				   (t_u8)c));
	  }
	return (-ERR_INVAL);
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	int		len;
	t_u8		c;

	if (ed->b.len < sizeof (t_s8))
	  return (-ERR_TRUNC);
	len = strlen(str);
	switch (len)
	  {
	  case CHAR_C:
	    c = str[0];
	    break ;
	  case CHAR_XX:
	    c = strtoul(str,NULL,16);
	    break ;
	  case CHAR_OOO:
	    c = strtoul(str,NULL,8);
	    break ;
	  default:
	    return (-ERR_INVAL);
	  }
	FBCOPY(&c,ed->b.buf,sizeof (c));
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages signed integers of one byte. */
t_status		typ_s8_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("s8");
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_s8		c;

	if (ed->b.len < sizeof (t_s8))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&c,sizeof (t_s8));
	return (long_to_str((signed long)c,
			    layer_base,
			    bs->str,
			    bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_s8		c;

	if (ed->b.len < sizeof (t_s8))
	  return (-ERR_TRUNC);
	c = atoi(str);
	FBCOPY(&c,ed->b.buf,sizeof (t_s8));
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages unsigned integers of one byte. */
t_status		typ_u8_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("u8");
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u8		c;

	if (ed->b.len < sizeof (t_u8))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&c,sizeof (t_u8));
	return (ulong_to_str((unsigned long)c,
			     layer_base,
			     bs->str,
			     bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u8		c;

	if (ed->b.len < sizeof (t_u8))
	  return (-ERR_TRUNC);
	c = atoi(str);
	FBCOPY(&c,ed->b.buf,sizeof (t_u8));
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages unsigned integers of one byte taken from a t_assoc array.
   Requires a t_assoc array as ed->data. */
t_status		typ_u8assoc_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  t_status		status;

  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("u8assoc");
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u8		c;
	t_assoc		*assocs;
	t_assoc		*assoc;

	assocs = (t_assoc *)(ed->data);
	if (ed->b.len < sizeof (t_u8))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&c,sizeof (t_u8));
	if (assoc = assoc_str_int_from_right(assocs,(int)c))
	  return (str_cat_str(bs->str,
			      bs->max_len,
			      assoc->left));
	else
	  return (ulong_to_str((unsigned long)c,
			       layer_base,
			       bs->str,
			       bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u8		c;
	t_assoc		*assocs;
	t_assoc		*assoc;

	assocs = (t_assoc *)(ed->data);
	if (ed->b.len < sizeof (t_u8))
	  return (-ERR_TRUNC);
	if (assoc = assoc_str_ptr_from_left(assocs,str))
	  c = (t_u8)(t_u32)(assoc->right);
	else
	  c = atoi(str);
	FBCOPY(&c,ed->b.buf,sizeof (t_u8));
	return (0);
      }
    case TYP_GET_CHOICES:
      {
	TYP_GET_CHOICES_ARGS(ed,vec_str);
	t_assoc		*assocs;
	t_assoc		*assoc;
	
	assocs = (t_assoc *)(ed->data);
	assoc = assocs;
	while (assoc->left)
	  {
	    if ((status = vec_str_add(vec_str,assoc->left)) < 0)
	      return (status);
	    assoc++;
	  }
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages unsigned integers of one byte bitfields.
   Requires a t_bit_field structure as ed->data. */
t_status		typ_u8bitfield_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("u8bitfield");
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u8		u8;
	t_bit_field	*bf;

	if (ed->b.len < sizeof (u8))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u8,sizeof (u8));
	bf = (t_bit_field *)(ed->data);
	u8 = *bit_field_u8_get(&u8,bf->from,bf->to);
	return (ulong_to_str((unsigned long)u8,
			     layer_base,
			     bs->str,
			     bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u8		u8;
	t_u8		value;
	t_bit_field	*bf;

	if (ed->b.len < sizeof (u8))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u8,sizeof (u8));
	bf = (t_bit_field *)(ed->data);
	value = atoi(str);
	bit_field_u8_set(&u8,bf->from,bf->to,value);
	FBCOPY(&u8,ed->b.buf,sizeof (u8));
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages unsigned integers of one byte bitfields taken from a t_assoc array.
   Requires a t_bit_field_assocs_data structure as ed->data. 
   Do not try to generalize more! */
t_status			typ_u8bitfieldassoc_msg(msg,arg1,arg2)
t_msg				msg;
VOID_PTR			arg1;
VOID_PTR			arg2;
{
  t_status			status;

  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("u8bitfieldassoc");
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u8			u8;
	t_bit_field_assocs_data	*bfad;
	t_assoc			*assoc;

	if (ed->b.len < sizeof (u8))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u8,sizeof (u8));
	bfad = (t_bit_field_assocs_data *)(ed->data);
	u8 = *bit_field_u8_get(&u8,
				 bfad->bit_field->from,
				 bfad->bit_field->to);
	if (assoc = assoc_str_int_from_right(bfad->assocs,(int)u8))
	  return (str_cat_str(bs->str,
			      bs->max_len,
			      assoc->left));
	else
	  return (ulong_to_str((unsigned long)u8,
			       layer_base,
			       bs->str,
			       bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u8			u8;
	t_u8			value;
	t_bit_field_assocs_data	*bfad;
	t_assoc			*assoc;

	if (ed->b.len < sizeof (u8))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u8,sizeof (u8));
	bfad = (t_bit_field_assocs_data *)(ed->data);
	if (assoc = assoc_str_ptr_from_left(bfad->assocs,str))
	  value = (t_u8)(t_u32)(assoc->right);
	else
	  value = atoi(str);
	bit_field_u8_set(&u8,
			  bfad->bit_field->from,
			  bfad->bit_field->to,
			  value);
	FBCOPY(&u8,ed->b.buf,sizeof (u8));
	return (0);
      }
    case TYP_GET_CHOICES:
      {
	TYP_GET_CHOICES_ARGS(ed,vec_str);
	t_bit_field_assocs_data	*bfad;
	t_assoc			*assoc;
	
	bfad = (t_bit_field_assocs_data *)(ed->data);
	assoc = bfad->assocs;
	while (assoc->left)
	  {
	    if ((status = vec_str_add(vec_str,assoc->left)) < 0)
	      return (status);
	    assoc++;
	  }
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages unsigned integers of one byte masks.
   Requires a t_mask_def structure as ed->data. 
   Masks are separated by "|". */
t_status		typ_u8mask_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  t_status		status;

  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("u8mask");
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u8		c;
	t_mask_def	*mask_defs;
	
	mask_defs = (t_mask_def *)(ed->data);
	if (ed->b.len < sizeof (t_u8))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&c,sizeof (t_u8));
	return (mask_to_str(mask_defs,
                            c,
                            "|",
			    layer_base,
                            bs->str,
                            bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u8		c;
	t_mask_def	*mask_defs;
	t_vec		*vec_str;
	t_mask		mask;

	mask_defs = (t_mask_def *)(ed->data);
	if (ed->b.len < sizeof (t_u8))
	  return (-ERR_TRUNC);
	if ((vec_str = LAYER_VEC_NEW(&status)) == NULL)
          return (status);
        if ((status = vec_str_split(vec_str,str,'|')) < 0)
          {
            vec_str_delete(vec_str);
            return (status);
          }
        mask = 0;
        if ((status = mask_from_vec_str(mask_defs,
                                        vec_str,
                                        &mask)) < 0)
          {
            vec_str_delete(vec_str);
            return (status);
          }
        c = (t_u8)mask;
        FBCOPY(&c,ed->b.buf,sizeof (t_u8));
        vec_str_delete(vec_str);
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	
