/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include "layer.h"
#include "typ_msg.h"
#include "typ_32.h"

t_u32			safe_ntohl(p_nu32)
t_u32			*p_nu32;	/* Pointer to network unsigned long */
{
  t_u32			a_u32;
  
  FBCOPY(p_nu32,&a_u32,sizeof (t_u32));
  return (UNSAFE_NTOHL(a_u32));
}

VOID_FUNC		safe_htonl(a_u32,p_nu32)
t_u32			a_u32;		/* Aligned host unsigned long */
t_u32			*p_nu32;	/* Pointer to network unsigned long */
{
  a_u32 = UNSAFE_HTONL(a_u32);  
  FBCOPY(&a_u32,p_nu32,sizeof (t_u32));
}

#define TYP_OFF_32_GENERIC \
case TYP_OFF:\
{\
  TYP_OFF_ARGS(ed,off);\
\
  (*off) = 4;\
  return (0);\
}

/* is a t_msg_proc.
   Manages network signed longs. */
t_status		typ_n32_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("n32");
      TYP_OFF_32_GENERIC;
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_s32		s32;
	
	if (ed->b.len < sizeof (s32))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&s32,sizeof (s32));
	s32 = UNSAFE_NTOHL(s32);
	return (long_to_str((signed long)s32,
			    layer_base,
			    bs->str,
			    bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_s32		s32;
	t_status	status;	
	int		basenum;
	signed long	num;

	if (ed->b.len < sizeof (s32))
	  return (-ERR_TRUNC);
	if ((status = long_from_str(str,
				    &basenum,
				    &num)) < 0)
	  return (status);
	s32 = (t_s32)num;
	s32 = UNSAFE_HTONL(s32);
	FBCOPY(&s32,ed->b.buf,sizeof (s32));
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages network unsigned longs */
t_status		typ_nu32_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("nu32");
      TYP_OFF_32_GENERIC;
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u32		u32;

	if (ed->b.len < sizeof (u32))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u32,sizeof (u32));
	u32 = UNSAFE_NTOHL(u32);
	return (ulong_to_str((unsigned long)u32,
			     layer_base,
			     bs->str,
			     bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u32		u32;
	t_status	status;	
	int		basenum;
	unsigned long	num;

	if (ed->b.len < sizeof (u32))
	  return (-ERR_TRUNC);
	if ((status = ulong_from_str(str,
				     &basenum,
				     &num)) < 0)
	  return (status);
	u32 = (t_u32)num;
	u32 = UNSAFE_HTONL(u32);
	FBCOPY(&u32,ed->b.buf,sizeof (u32));
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages unsigned longs */
t_status		typ_u32_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("u32");
      TYP_OFF_32_GENERIC;
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u32		u32;

	if (ed->b.len < sizeof (u32))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u32,sizeof (u32));
	return (ulong_to_str((unsigned long)u32,
			     layer_base,
			     bs->str,
			     bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u32		u32;
	t_status	status;	
	int		basenum;
	unsigned long	num;

	if (ed->b.len < sizeof (u32))
	  return (-ERR_TRUNC);
	if ((status = ulong_from_str(str,
				     &basenum,
				     &num)) < 0)
	  return (status);
	u32 = (t_u32)num;
	FBCOPY(&u32,ed->b.buf,sizeof (u32));
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages network unsigned longs associations.
   Requires a t_assoc array as
   ed->data. Is able to lookup tokens in this array as base for
   extracting or inserting strings. */
t_status		typ_nu32assoc_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  t_status		status;

  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("nu32assoc");
      TYP_OFF_32_GENERIC;
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u32		u32;
	t_assoc		*assocs;
	t_assoc		*assoc;

	assocs = (t_assoc *)(ed->data);
	if (ed->b.len < sizeof (u32))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u32,sizeof (u32));
	u32 = UNSAFE_NTOHL(u32);
	if (assoc = assoc_str_int_from_right(assocs,(int)u32))
	  return (str_cat_str(bs->str,
			      bs->max_len,
			      assoc->left));
	else
	  return (ulong_to_str((unsigned long)u32,
			       layer_base,
			       bs->str,
			       bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u32		u32;
	t_status	status;	
	t_assoc		*assocs;
	t_assoc		*assoc;

	assocs = (t_assoc *)(ed->data);
	if (ed->b.len < sizeof (u32))
	  return (-ERR_TRUNC);
	if (assoc = assoc_str_ptr_from_left(assocs,str))
	  u32 = (t_u32)(t_u32)(assoc->right);
	else
	  {
	    int			basenum;
	    unsigned long	num;

	    if ((status = ulong_from_str(str,
					 &basenum,
					 &num)) < 0)
	      return (status);
	    u32 = (t_u32)num;
	  }
	u32 = UNSAFE_HTONL(u32);
	FBCOPY(&u32,ed->b.buf,sizeof (u32));
	return (0);
      }
    case TYP_GET_CHOICES:
      {
	TYP_GET_CHOICES_ARGS(ed,vec_str);
	t_assoc		*assocs;
	t_assoc		*assoc;
	
	assocs = (t_assoc *)(ed->data);
	assoc = assocs;
	while (assoc->left)
	  {
	    if ((status = vec_str_add(vec_str,assoc->left)) < 0)
	      return (status);
	    assoc++;
	  }
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages unsigned longs associations.
   Requires a t_assoc array as
   ed->data. Is able to lookup tokens in this array as base for
   extracting or inserting strings. */
t_status		typ_u32assoc_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  t_status		status;

  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("u32assoc");
      TYP_OFF_32_GENERIC;
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u32		u32;
	t_assoc		*assocs;
	t_assoc		*assoc;

	assocs = (t_assoc *)(ed->data);
	if (ed->b.len < sizeof (u32))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u32,sizeof (u32));
	if (assoc = assoc_str_int_from_right(assocs,(int)u32))
	  return (str_cat_str(bs->str,
			      bs->max_len,
			      assoc->left));
	else
	  return (ulong_to_str((unsigned long)u32,
			       layer_base,
			       bs->str,
			       bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u32		u32;
	t_status	status;	
	t_assoc		*assocs;
	t_assoc		*assoc;

	assocs = (t_assoc *)(ed->data);
	if (ed->b.len < sizeof (u32))
	  return (-ERR_TRUNC);
	if (assoc = assoc_str_ptr_from_left(assocs,str))
	  u32 = (t_u32)(t_u32)(assoc->right);
	else
	  {
	    int			basenum;
	    unsigned long	num;

	    if ((status = ulong_from_str(str,
					 &basenum,
					 &num)) < 0)
	      return (status);
	    u32 = (t_u32)num;
	  }
	FBCOPY(&u32,ed->b.buf,sizeof (u32));
	return (0);
      }
    case TYP_GET_CHOICES:
      {
	TYP_GET_CHOICES_ARGS(ed,vec_str);
	t_assoc		*assocs;
	t_assoc		*assoc;
	
	assocs = (t_assoc *)(ed->data);
	assoc = assocs;
	while (assoc->left)
	  {
	    if ((status = vec_str_add(vec_str,assoc->left)) < 0)
	      return (status);
	    assoc++;
	  }
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	

/* is a t_msg_proc.
   Manages network unsigned longs masks.
   It requires a t_mask_def structure as ed->data. Masks but be 
   separated by "|" */
t_status		typ_u32mask_msg(msg,arg1,arg2)
t_msg			msg;
VOID_PTR		arg1;
VOID_PTR		arg2;
{
  t_status		status;

  switch (msg)
    {
      TYP_CLASS_GENERIC;
      TYP_NAME_GENERIC("u32mask");
      TYP_OFF_32_GENERIC;
    case TYP_EXTRACT:
      {
	TYP_EXTRACT_ARGS(ed,bs);
	t_u32		u32;
	t_mask_def	*mask_defs;
	
	mask_defs = (t_mask_def *)(ed->data);
	if (ed->b.len < sizeof (t_u32))
	  return (-ERR_TRUNC);
	FBCOPY(ed->b.buf,&u32,sizeof (t_u32));
	return (mask_to_str(mask_defs,
                            u32,
                            "|",
			    layer_base,
                            bs->str,
                            bs->max_len));
      }
    case TYP_INSERT:
      {
	TYP_INSERT_ARGS(ed,str);
	t_u32		u32;
	t_mask_def	*mask_defs;
	t_vec		*vec_str;
	t_mask		mask;

	mask_defs = (t_mask_def *)(ed->data);
	if (ed->b.len < sizeof (t_u32))
	  return (-ERR_TRUNC);
	if ((vec_str = LAYER_VEC_NEW(&status)) == NULL)
          return (status);
        if ((status = vec_str_split(vec_str,str,'|')) < 0)
          {
            vec_str_delete(vec_str);
            return (status);
          }
        mask = 0;
        if ((status = mask_from_vec_str(mask_defs,
                                        vec_str,
                                        &mask)) < 0)
          {
            vec_str_delete(vec_str);
            return (status);
          }
        u32 = (t_u32)mask;
        FBCOPY(&u32,ed->b.buf,sizeof (t_u32));
        vec_str_delete(vec_str);
	return (0);
      }
    }
  return (-ERR_NOMETHOD);
}	
