/*
 * This file is a part of the mg project.
 * Copyright (C) 1998 Martin Gall
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 *
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "typ_16.h"
#include "typ_port.h"
#include "lay_ip.h"
#include "lay_udp.h"
#include "lay_data.h"
#include "lay_rip.h"
#include "lay_dns.h"
#include "lay_tracert.h"
#include "lay_tftp.h"
#include "lay_rpc.h"

t_boolean		lay_udp_perform_sum = TRUE;
t_boolean		lay_udp_guess_mode = TRUE;
t_id			*lay_udp_sub_id = NULL;

/* initializes lay_udp_sub_id (see udp_sub(3)).
   Returns 0 if OK. Might return various errors. */
t_status		lay_udp_init(VOID_DECL)
{
  t_status		status;

  if ((lay_udp_sub_id = LAYER_TINY_ID_NEW(&status)) == NULL)
    return (status);
  return (0);
}

/* deletes lay_udp_sub_id */
VOID_FUNC		lay_udp_destroy(VOID_DECL)
{
  id_delete(lay_udp_sub_id);
}

/* computes an udp checksum.
   It is portable and deals with alignment.
   Returns the checksum. */
t_u16			udp_cksum(ip)
struct s_ip		*ip;
{
   t_udp		*udp;
   t_u16		*sh;
   unsigned long	sum;
   int			len;
   int			i;
   
   udp = (t_udp *)(((char *)ip) + ip_get_hl(ip) * 4);
   len = safe_ntohs(&udp->ulen);
   sum = 0;
   sh = (t_u16 *)(&(ip->src));
   i = 0;
   while (i < UDP_ALEN)
     {
       t_u16		u16;

       FBCOPY(sh,&u16,sizeof (u16)); /* ALIGNMENT */
       sum += UNSAFE_HTONS(u16);
       sh++;
       i++;
     }
   sh = (t_u16 *)udp;
   sum += ip->p + len;
   if (len & 0x1)
   {
      ((char *)udp)[len] = 0;
      len += 1;
   }
   len /= 2;
   i = 0;
   while (i < len)
     {
       t_u16		u16;

       FBCOPY(sh,&u16,sizeof (u16)); /* ALIGNMENT */
       sum += UNSAFE_HTONS(u16);
       sh++;
       i++;
     }
   sum = (sum >> 16) + (sum & 0xffff);
   sum += (sum >> 16);
   return ((t_u16)(~sum & 0xffff));
}

/* computes udp checksum. 
   Note: It deals with alignment. 
   It calls udp_cksum(3). */
int			udp_compute_sum(ip)
struct s_ip		*ip;
{
   t_udp		*udp;
   t_u16		zero;

   udp = (t_udp *)((char *)ip + ip_get_hl(ip) * 4);
   zero = 0;
   FBCOPY(&zero,&udp->sum,sizeof (zero)); /* ALIGNMENT */
   safe_htons(udp_cksum(ip),&udp->sum);
}

/* sets udp protocol to a ip header */
VOID_FUNC		udp_set_proto(ip)
t_ip			*ip;
{
  ip_set_p(ip,IP_PROTO_UDP);
}

/* sets udp source port.
   Note: deals with alignment. */
VOID_FUNC		udp_set_sport(udp,sport)
t_udp			*udp;
int			sport;
{
  safe_htons((t_u16)(sport),&udp->sport);
}

/* gets udp source port.
   Note: deals with alignment. */
int			udp_get_sport(udp)
t_udp			*udp;
{
  return (safe_ntohs(&udp->sport));
}

/* sets udp destination port.
   Note: deals with alignment. */
VOID_FUNC		udp_set_dport(udp,dport)
t_udp			*udp;
int			dport;
{
  safe_htons((t_u16)(dport),&udp->dport);
}

/* gets udp destination port.
   Note: deals with alignment. */
int			udp_get_dport(udp)
t_udp			*udp;
{
  return (safe_ntohs(&udp->dport));
}

/* sets udp length.
   Note: deals with alignment. */
VOID_FUNC		udp_set_ulen(udp,ulen)
t_udp			*udp;
{
  safe_htons(ulen,&udp->ulen);
}

/* gets udp length.
   Note: deals with alignment. */
int			udp_get_ulen(udp)
t_udp			*udp;
{
  return (safe_ntohs(&udp->ulen));
}

/* sets udp checksum.
   Note: deals with alignment. */
VOID_FUNC		udp_set_sum(udp,sum)
t_udp			*udp;
int			sum;
{
  safe_htons(sum,&udp->sum);
}

/* gets udp checksum.
   Note: deals with alignment. */
int			udp_get_sum(udp)
t_udp			*udp;
{
  return (safe_ntohs(&udp->sum));
}	

/* gets udp sub layer.
   It is used by lay_udp_msg(3).
   If global lay_udp_guess_mode is TRUE, it recognizes its sub layer
   in using source or destination port. Currently it recognizes ip, dns,
   tftp, rpc and Van Jacobson's traceroute. It also checks for
   and entry in lay_udp_sub_id (allowing some new layers being 
   added by plugins).
   Returns 0 if OK. Might return -ERR_TRUNC. */
t_status		udp_sub(buf,len,sub_mp)
char			*buf;
int			len;
t_msg_proc		*sub_mp;
{
  t_udp			*udp;

  LAYER_UDP_CHECK(udp,buf,len);
  if (lay_udp_guess_mode)
    {
      t_u16		dport;
      t_u16		sport;

      dport = udp_get_dport(udp);
      switch (dport)
	{
	case RIP_PORT:
	  (*sub_mp) = lay_rip_msg;
	  return (0);
	case DNS_PORT:
	  (*sub_mp) = lay_dns_msg;
	  return (0);
	case TFTP_PORT:
	  (*sub_mp) = lay_tftp_msg;
	  return (0);
	case RPC_PORT:
	  (*sub_mp) = lay_rpc_msg;
	  return (0);
	}
      sport = udp_get_sport(udp);
      switch (sport)
	{
	case RIP_PORT:
	  (*sub_mp) = lay_rip_msg;
	  return (0);
	case DNS_PORT:
	  (*sub_mp) = lay_dns_msg;
	  return (0);
	case TFTP_PORT:
	  (*sub_mp) = lay_tftp_msg;
	  return (0);
	case RPC_PORT:
	  (*sub_mp) = lay_rpc_msg;
	  return (0);
	}
      if ((dport >= TRACERT_MAGIC && dport < (TRACERT_MAGIC + 256)) ||
	  (sport >= TRACERT_MAGIC && sport < (TRACERT_MAGIC + 256)))
	(*sub_mp) = lay_tracert_msg;
      if (lay_udp_sub_id)
	{
	  t_hash_elt	*he;

	  if (he = id_get(lay_udp_sub_id,
			  (VOID_PTR)(t_u32)dport))
	    {
	      (*sub_mp) = (t_msg_proc)(he->value);
	      return (0);
	    }
	  else
	    if (he = id_get(lay_udp_sub_id,
			    (VOID_PTR)(t_u32)sport))
	      {
		return (0);
		(*sub_mp) = (t_msg_proc)(he->value);
	      }
	}
    }
  (*sub_mp) = lay_data_msg;
  return (0);
}

/* performs an udp checksum.
   This procedure is called by lay_udp_msg(3) and corresponds to the 
   meta-checksum-mechanism of pkt_sum(3).
   It checks the standard header len (UDP_HLEN) and calls udp_compute_sum(3).
   Returns 0 if OK. Returns -ERR_BADLEN if the given length is different
   from the one retrieved by udp_get_ulen(3). It also check for
   ip header (given by up_buf and up_len) in the same way as ip_sum(3).
   Might of course return -ERR_TRUNC. */
t_status		udp_sum(buf,len,up_buf,up_len)
char			*buf;
int			len;
char			*up_buf;
int			up_len;
{
  t_udp			*udp;
  t_ip			*ip;

  LAYER_UDP_CHECK(udp,buf,len);
  if (udp_get_ulen(udp) != len)
    return (-ERR_BADLEN);
  LAYER_IP_CHECK(ip,up_buf,up_len);
  if (ip_get_len(ip) != up_len)
    return (-ERR_BADLEN);
  if (ip_get_hl(ip) * 4 < IP_MINHLEN)
    return (-ERR_BADHEADER);
  udp_compute_sum(ip);
  return (0);
}

/* sets zero to udp checksum.
   It is used by lay_udp_msg(3) if lay_udp_perform_sum is FALSE.
   Performs the same checks as in udp_sum(3)
   Returns 0 if OK. */
t_status		udp_zero_sum(buf,len,up_buf,up_len)
char			*buf;
int			len;
char			*up_buf;
int			up_len;
{
  t_udp			*udp;
  t_ip			*ip;

  LAYER_UDP_CHECK(udp,buf,len);
  LAYER_IP_CHECK(ip,up_buf,up_len);
  udp->sum = 0;
  return (0);
}

t_status		udp_adapt_len(buf,len)
char			*buf;
int			len;
{
  t_udp			*udp;

  LAYER_UDP_CHECK(udp,buf,len);
  udp_set_ulen(udp,len);
  return (0);
}

t_field				udp_fields[] = 
{
  {"sport",	OFFSET(t_udp *,sport),	typ_nu16_msg,		NULL},
  {"Sport",	OFFSET(t_udp *,sport),	typ_port_resolved_msg,	"udp"},
  {"dport",	OFFSET(t_udp *,dport),	typ_nu16_msg,		NULL},
  {"Dport",	OFFSET(t_udp *,dport),	typ_port_resolved_msg,	"udp"},
  {"ulen",	OFFSET(t_udp *,ulen),	typ_nu16_msg,		NULL},
  {"sum",	OFFSET(t_udp *,sum),	typ_nu16_msg,		NULL},
  NULL_FIELD
};

char				*udp_itmpl = "\n\
<!--udp_itmpl-->\n\
<table width=100%%%% bgcolor=\"%%udpColor%%\">\n\
<tr>\n\
<td>\n\
<small>\n\
<a href=\"extract(udp[%i%])\">[Extract]</a>\n\
<a href=\"trunc(udp[%i%])\">[Trunc]</a>\n\
<a href=\"trunc(udp[%i%])\">[Paste]</a>\n\
<a href=\"adaptlen(udp[%i%])\">[Adapt len]</a>\n\
<a href=\"sum(udp[%i%])\">[Sum]</a>\n\
 udp\n\
</small>\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=50%%%%>\n\
<a href=\"setfield(udp[%i%].Sport)\">%%udp[%i%].Sport%%</a>\n\
(<a href=\"setfield(udp[%i%].sport)\">%%udp[%i%].sport%%</a>)\n\
</td>\n\
<td align=center width=50%%%%>\n\
<a href=\"setfield(udp[%i%].Dport)\">%%udp[%i%].Dport%%</a>\n\
(<a href=\"setfield(udp[%i%].dport)\">%%udp[%i%].dport%%</a>)\n\
</td>\n\
</tr>\n\
<tr>\n\
<td align=center width=50%%%%>\n\
<a href=\"setfield(udp[%i%].ulen)\">%%udp[%i%].ulen%%</a>\n\
</td>\n\
<td align=center width=50%%%%>\n\
<a href=\"setfield(udp[%i%].sum)\">%%udp[%i%].sum%%</a>\n\
</td>\n\
</tr>\n\
</table>\n\
";

/* is a t_msg_proc.
   Manages udp layers */
t_status			lay_udp_msg(msg,arg1,arg2)
t_msg				msg;
VOID_PTR			arg1;
VOID_PTR			arg2;
{
  t_status			status;

  switch (msg)
    {
      LAY_CLASS_GENERIC;
      LAY_NAME_ID_GENERIC(&lay_udp_msg,"udp");
      LAY_GET_FIELD_GENERIC(udp_fields);
      LAY_SET_FIELD_GENERIC(udp_fields);
      LAY_GET_FIELDS_GENERIC(udp_fields);
      LAY_GET_FIELD_TYP_GENERIC(udp_fields);
      LAY_GET_ITMPL_GENERIC(&lay_udp_msg,udp_itmpl);
    case LAY_OFF:
      {
	LAY_OFF_ARGS(b,off);
	
	(*off) = UDP_HLEN;
	return (0);
      }
    case LAY_SUB:
      {
	LAY_SUB_ARGS(b,mp);
	
	return (udp_sub(b->buf,b->len,mp));
      }
    case LAY_SUM:
      {
	LAY_SUM_ARGS(sd,unused);
	
	if (lay_udp_perform_sum)
	  return (udp_sum(sd->b.buf,
			  sd->b.len,
			  sd->up.buf,
			  sd->up.len));
	else
	  return (udp_zero_sum(sd->b.buf,
			       sd->b.len,
			       sd->up.buf,
			       sd->up.len));
      }
    case LAY_ADAPT_LEN:
      {
	LAY_ADAPT_LEN_ARGS(b,unused);

	return (udp_adapt_len(b->buf,b->len));
      }
    }
  return (-ERR_NOMETHOD);
}
