/*
 * xdefmap.c
 *
 * Copyright (C) 1995 by Helmut Hoenig
 * All rights reserved.
 *
 * This software may be freely copied, modified and redistributed without
 * fee for non-commerical purposes provided that this copyright notice is
 * preserved intact on all copies and modified copies.
 *
 * There is no warranty or other guarantee of fitness of this software.
 * It is provided solely "as is". The author disclaim all
 * responsibility and liability with respect to this software's usage
 * or its effect upon hardware or computer systems.
 */

#include <stdio.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#define DEFAULT_SIZE_RGB   4
#define DEFAULT_SIZE_BW    32

static Display *dpy;
static int     scr;
static Window  win;
static Atom    map_type=0;    /* selected stanard colormap */
static int     verbose=0;
static int     sync=0;        /* work sychronous */
static int     size=-1;       /* size of colormap or window window */
static int     pos =-1;       /* where to open the information window */
static int     wait_flag=0;   /* let pixels allocated */
static int     set_flag=0;    /* set unused pixels */
static char *color1 = "white";
static char *color2 = "black";
static int     xcms_flag=0;   /* let pixels allocated */
static FILE    *out_fp;       /* output file for xcmsdb (default if stdout) */
static FILE    *in_fp;        /* input file for rgb.txt file */
static FILE    *join_fp;      /* input file for xcmsdb-merge */

#define  MAXCOLS  256
#define  MAGIC    256

enum { Unknown, FreePixel, PrivatePixel, SharedPixel };
static char    pixel_usage[MAXCOLS];

Atom  map[] = {
   XA_RGB_GRAY_MAP,  XA_RGB_BEST_MAP,  XA_RGB_DEFAULT_MAP,
   XA_RGB_RED_MAP,   XA_RGB_GREEN_MAP, XA_RGB_BLUE_MAP };
char  *map_string[] = {
   "gray", "best", "default", "red", "green", "blue"
};

/* -------------------------------------------------------------------------- *
   check, if pixel is read-only by trying to allocate exactly its rgb-value
 * -------------------------------------------------------------------------- */
static int check_pixel(i)
   int   i;
{
   if (pixel_usage[i]==Unknown) {
      XColor   col;
      pixel_usage[i] = PrivatePixel;

   /*
    * Check if pixel is shared, which is found out by allocating it's
    * current value, if the same pixel is returned, it must be shared!
    * (It has directly to be freed again to be clean...)
    */
      col.pixel = i;
      XQueryColor(dpy,DefaultColormap(dpy,scr),&col); /* query RGB-value */

      col.flags = DoRed|DoGreen|DoBlue;            /* try to allocate it */
      if (XAllocColor(dpy,DefaultColormap(dpy,scr),&col)) {
         if (i==col.pixel)                            /* same pixel -> shared */
            pixel_usage[i]=SharedPixel;
         XFreeColors(dpy,DefaultColormap(dpy,scr),&col.pixel,1,0L);
      }
   }
   return pixel_usage[i];
}

/* -------------------------------------------------------------------------- *
   mapusage: checks each available pixel for it's state and gives a summary
 * -------------------------------------------------------------------------- */

void mapusage(silent)
   int   silent;
 {
long  i;
int   free_count=0;
int   private_count=0;
int   shared_count=0;

/*
 * allocate all available pixels
 * (to get direct control for allocating special, but shared pixels)
 */
   memset( pixel_usage, 0, sizeof(pixel_usage) );
   while(XAllocColorCells(dpy,DefaultColormap(dpy,scr),True,0L,0,&i,1)) {
      if (i>=MAXCOLS) {
         fprintf(stderr,"program not suited for displays with more than %d colors\n", MAXCOLS );
         exit(-1);
      }
      pixel_usage[i]=FreePixel;
      free_count++;
   }

	if (!silent && verbose) {
      printf( "DefaultColormap Occupancy:\n" );
	}
   for (i=0;i<MAXCOLS;i++) {
      if (pixel_usage[i]==Unknown) {
         if (check_pixel(i)==SharedPixel)    shared_count++;
         else                                   private_count++;
      }
      if (!silent && verbose) {
      	if (pixel_usage[i]==FreePixel) {
      		printf( "  %02x: unused\n", i );
      	}
      	else {
    		  	XColor	xcol;
      		xcol.pixel = i;
      		XQueryColor(dpy,DefaultColormap(dpy,scr),&xcol);
      		printf( "  %02x: %-8s %04x %04x %04x\n", i,
      			(pixel_usage[i]==SharedPixel)?"shared":"private",
      			xcol.red, xcol.green, xcol.blue );
      	}
      }
   }

   if (!silent) {
      printf( "DefaultColormap Summary:\n" );
      printf( "  unused   :%5d entries\n", free_count );
      printf( "  shared   :%5d entries\n", shared_count );
      printf( "  private  :%5d entries\n", private_count );
      printf( "  total    :%5d entries\n", free_count+shared_count+private_count );
   }
}

/* -------------------------------------------------------------------------- *
   show_mapinfo: query the available standard colormap properties and
                 print their data
 * -------------------------------------------------------------------------- */

void show_mapinfo() {
XStandardColormap *info_field;
int count;
int i,j,none=1;

   printf( "Installed Colormaps:\n" );
   for (i=0;i<sizeof(map)/sizeof(Atom);i++) {
      if (XGetRGBColormaps(dpy,RootWindow(dpy,scr),&info_field,&count,map[i])) {
         none=0;
         for (j=0;j<count;j++) {
            printf( "  %-12s:  red:%3d/%-3dgreen:%3d/%-3dblue:%3d/%-3d  "
                     "\n                 base: %02x  ",
               map_string[i],
               info_field[j].red_max+1,   info_field[j].red_mult,
               info_field[j].green_max+1, info_field[j].green_mult,
               info_field[j].blue_max+1,  info_field[j].blue_mult,
               info_field[j].base_pixel );
            if (info_field[j].colormap==DefaultColormap(dpy,scr)) {
               printf( "map: default    " );
            }
            else {
               printf( "map: 0x%08lx ", info_field[j].colormap );
            }
            switch(info_field[j].killid) {
            case 0:
               printf( "owner: none (joined)\n" );
               break;
            case 1:
               printf( "owner: map\n" );
               break;
            default:
               printf( "owner: 0x%08x\n", info_field[j].killid );
               break;
            }
         }
         XFree(info_field);
      }
   }
   if (none)   printf( "- None -\n" );
}

/* -------------------------------------------------------------------------- *
   show_visualinfo: give information on the supported visuals
 * -------------------------------------------------------------------------- */

static char *visual_class[] = {
	"StaticGray",  "GrayScale", "StaticColor",
	"PseudoColor", "TrueColor", "DirectColor"
};

static int splitmask( mask, num, mult )
	unsigned long	mask;
	int				*num;
	int				*mult;
{
	*mult = 1;
	while( ! (mask & 0x01) ) {
		mask  >>= 1;
		*mult <<= 1;
	}
	*num  = 1;
	while( mask & 0x01 ) {
		mask >>= 1;
		*num <<= 1;
	}
	return (mask==0);
}

static void print_col( name, mask )
	char				*name;
	unsigned long	mask;
{
int num,mult;

	if (splitmask(mask,&num,&mult))
			printf( "%s:%3d/%-3d", name, num, mult );
	else	printf( "%s: 0x%04x ", name, mask );

}

void show_visualinfo() {
int i;
XVisualInfo	visual_info;

	printf( "Supported Visuals:\n" );
	for (i=0;i<sizeof(visual_class)/sizeof(char*);i+=(i==4)?-3:2) {
		if (XMatchVisualInfo(dpy,scr,DefaultDepth(dpy,scr),i,&visual_info)) {
			int num=1;
			int mul=visual_info.colormap_size;
			int j;
								/* rem: special handling for i==5 (DirectColor) */
			for (j=visual_info.bits_per_rgb;j>0 && (mul>1 || i==5);j--) {
				mul /= 2;
				num *= 2;
			}
			printf( "  %-12s:  ", visual_class[i],
				(visual_info.visual==DefaultVisual(dpy,scr))?"(default)":"" );
			switch(i) {
			case 0:
			case 1: {
					if (i)	printf( "%d of %d\n", visual_info.colormap_size, num );
					else		printf( "grey: %d/%d\n", num, mul );
					break;
			}
			case 2:
			case 4:
					print_col( "red",   visual_info.red_mask );
					print_col( "green", visual_info.green_mask );
					print_col( "blue",  visual_info.blue_mask );
					printf("\n");
					break;
			case 3:
					printf( "%d of %dx%dx%d\n",
								visual_info.colormap_size, num,num,num );
					break;
			case 5: {
					int cols;
					splitmask( visual_info.red_mask, &cols, &mul );
					printf( "%dof%d x ", cols, num );
					splitmask( visual_info.green_mask, &cols, &mul );
					printf( "%dof%d x ", cols, num );
					splitmask( visual_info.blue_mask, &cols, &mul );
					printf( "%dof%d\n", cols, num );
				}
			}

		}
	}
}

/* -------------------------------------------------------------------------- *
   check_for_swap: the best map and the delete map can be joined in which
                   case the client of the first map allocated the colors.
                   The other client-id is set  back to zero. If the first
                   map is deleted again, the second map gets the killid of
                   the first so that the colors stay allocated.
 * -------------------------------------------------------------------------- */

void check_for_swap(delete_map,swap_map)
   Atom  delete_map;
   Atom  swap_map;
{
XStandardColormap *delete_map_info;
XStandardColormap *swap_map_info;
int count;

   if (!XGetRGBColormaps(dpy,RootWindow(dpy,scr),
                     &delete_map_info,&count,delete_map)) {
         return;     /* no map to delete */
   }
   if (!XGetRGBColormaps(dpy,RootWindow(dpy,scr),
                     &swap_map_info,&count,swap_map)) {
         XFree(delete_map_info);
         return;     /* no map to swap */
   }

   if (  swap_map_info[0].killid     == 0
      && swap_map_info[0].colormap   == delete_map_info[0].colormap
      && swap_map_info[0].base_pixel == delete_map_info[0].base_pixel) {
      /* the maps were really joined, so the swap map should get the owner-id */
      swap_map_info[0].killid = delete_map_info[0].killid;
      delete_map_info[0].killid = 0;
      XSetRGBColormaps(dpy,RootWindow(dpy,scr),&swap_map_info[0],1,swap_map);
      XSetRGBColormaps(dpy,RootWindow(dpy,scr),&delete_map_info[0],1,delete_map);

      if (verbose) {
         printf("ownership of cells transfered to similar map\n" );
      }
   }
   XFree(delete_map_info);
   XFree(swap_map_info);
}

/* -------------------------------------------------------------------------- *
   join_map_with: try to find an existing map with same colors
 * -------------------------------------------------------------------------- */

int join_map_with(map_info,other_type)
   XStandardColormap *map_info;
   Atom              other_type;
{
XStandardColormap *other_map_info;
int count;

   if (!XGetRGBColormaps(dpy,RootWindow(dpy,scr),
                     &other_map_info,&count,other_type)) {
         return 0;      /* no map to join colors with */
   }
   if (  map_info->colormap   == other_map_info[0].colormap
      && map_info->red_max    == other_map_info[0].red_max
      && map_info->red_mult   == other_map_info[0].red_mult
      && map_info->green_max  == other_map_info[0].green_max
      && map_info->green_mult == other_map_info[0].green_mult
      && map_info->blue_max   == other_map_info[0].blue_max
      && map_info->blue_mult  == other_map_info[0].blue_mult ) {
         map_info->base_pixel = other_map_info[0].base_pixel;
         map_info->killid     = 0L;       /* special - no one to delete */
         return 1;
   }
   return 0;
}

/* -------------------------------------------------------------------------- *
   delete_map: delete the desired map by killing the client
 * -------------------------------------------------------------------------- */

void delete_map(map_type)
   Atom  map_type;
{
XStandardColormap *best_map_info_field;
int count;

   if (map_type==XA_RGB_BEST_MAP)   check_for_swap(map_type,XA_RGB_DEFAULT_MAP);
   if (map_type==XA_RGB_DEFAULT_MAP)check_for_swap(map_type,XA_RGB_BEST_MAP);

   if (XGetRGBColormaps(dpy,RootWindow(dpy,scr),&best_map_info_field,&count,map_type)) {
      switch(best_map_info_field[0].killid) {
      case 0:
         /* actually shouldn't be freed */
         break;
      case 1:
         XFreeColormap(dpy,best_map_info_field[0].colormap);
         break;
      default:
         if (best_map_info_field[0].killid!=win) {
            XKillClient(dpy,best_map_info_field[0].killid);
            if (verbose)    printf("XKillClient(dpy,0x%08x)\n",best_map_info_field[0].killid);
         }
         break;
      }
      XDeleteProperty(dpy,RootWindow(dpy,scr),map_type);
      XSync(dpy,0);
      if (verbose) {
         printf("old standard map removed\n" );
      }

      set_palette( color1, color2 );
      XFree(best_map_info_field);
   }
}

/* -------------------------------------------------------------------------- *
   free_unused: All colors might be allocated during the setup of the map.
                They have to be freed in case of errors
 * -------------------------------------------------------------------------- */

void free_unused() {
long i;
   for (i=0;i<MAXCOLS;i++) {
      if (pixel_usage[i]==FreePixel) {
         XFreeColors(dpy,DefaultColormap(dpy,scr),&i,1,0L);
      }
   }
}

/* -------------------------------------------------------------------------- *
   setup_best_map: Universal function, that sets up the desired standard map.
                   If the exist an old map, it will automatically by deleted.
 * -------------------------------------------------------------------------- */

int setup_best_map(map_type,rmax,gmax,bmax)
   Atom  map_type;
   int   rmax,gmax,bmax;
{
XStandardColormap best_map_info;
unsigned long i;
int      ncols;                  /* number of pixels neccessary */
unsigned long  pixel[256];       /* all available pixels */
int            first;
XColor         col[MAXCOLS];     /* color-values to be used */
int      rw_count=0;

   delete_map(map_type);

   switch( map_type ) {
   case XA_RGB_BEST_MAP:
   case XA_RGB_DEFAULT_MAP:
         ncols = rmax*gmax*bmax;
         break;
   case XA_RGB_RED_MAP:
         ncols = rmax;
         break;
   case XA_RGB_GREEN_MAP:
         ncols = gmax;
         break;
   case XA_RGB_BLUE_MAP:
         ncols = bmax;
         break;
   case XA_RGB_GRAY_MAP:
         ncols = rmax+gmax+bmax;
         break;
   }
/*
 * fill the colormap structure
 */
   best_map_info.colormap   = DefaultColormap(dpy,scr);
   best_map_info.red_max    = 0;
   best_map_info.red_mult   = 0;
   best_map_info.green_max  = 0;
   best_map_info.green_mult = 0;
   best_map_info.blue_max   = 0;
   best_map_info.blue_mult  = 0;
   for (i=0;i<ncols;i++) {
   	col[i].pixel = 0;		/* to get it 'Checker'ed */
      col[i].red   = col[i].green = col[i].blue =
            65535 * i / (ncols-1);
      col[i].flags = DoRed | DoGreen | DoBlue;
      col[i].pad   = 0;		/* to get it 'Checker' */
   }
   switch( map_type ) {
   case XA_RGB_BEST_MAP:
   case XA_RGB_DEFAULT_MAP:
      best_map_info.red_max    = rmax-1;
      best_map_info.red_mult   = 1;
      best_map_info.green_max  = gmax-1;
      best_map_info.green_mult = rmax;
      best_map_info.blue_max   = bmax-1;
      best_map_info.blue_mult  = rmax*gmax;
      if (join_map_with(&best_map_info,
            (map_type==XA_RGB_BEST_MAP)?XA_RGB_DEFAULT_MAP:XA_RGB_BEST_MAP)) {
         /*
          * shortcut: setup the map pointing to the other map
          */
            XSetRGBColormaps(dpy,RootWindow(dpy,scr),&best_map_info,1,map_type);
            XFlush(dpy);
            if (verbose)   printf( "joining with other map\n" );
            return 1;   /* to skip closedown mode ... */
      }

      for (i=0;i<ncols;i++) {
         col[i].red   = 65535 * ((i/best_map_info.red_mult)%(best_map_info.red_max+1))
                  / best_map_info.red_max;
         col[i].green = 65535 * ((i/best_map_info.green_mult)%(best_map_info.green_max+1))
                  / best_map_info.green_max;
         col[i].blue  = 65535 * ((i/best_map_info.blue_mult)%(best_map_info.blue_max+1))
                  / best_map_info.blue_max;
      }
      break;
   case XA_RGB_GRAY_MAP:
      best_map_info.red_max    = rmax-1;
      best_map_info.red_mult   = 1;
      best_map_info.green_max  = gmax-1;
      best_map_info.green_mult = 1;
      best_map_info.blue_max   = bmax-1;
      best_map_info.blue_mult  = 1;
      break;
   case XA_RGB_RED_MAP:
      best_map_info.red_max    = rmax-1;
      best_map_info.red_mult   = 1;
      for (i=0;i<ncols;i++) {
         col[i].green = col[i].blue = 0;
      }
      break;
   case XA_RGB_GREEN_MAP:
      best_map_info.green_max  = gmax-1;
      best_map_info.green_mult = 1;
      for (i=0;i<ncols;i++) {
         col[i].red   = col[i].blue = 0;
      }
      break;
   case XA_RGB_BLUE_MAP:
      best_map_info.blue_max   = bmax-1;
      best_map_info.blue_mult  = 1;
      for (i=0;i<ncols;i++) {
         col[i].red   = col[i].green = 0;
      }
      break;
   }

/*
 * grab the server so the chance is better for getting successive cells
 * as other application are also allocating cells at startup-time.
 * (Actually results are best if no other application runs before this one !!)
 * Anyway, the investigations about private and shared colors shouldn't be
 * interrupted by other applications.
 */
   XGrabServer(dpy);

/*
 * allocate all available pixels
 * (to get direct control for allocating special, but shared pixels)
 */
   memset( pixel_usage, 0, sizeof(pixel_usage) );
   while(XAllocColorCells(dpy,DefaultColormap(dpy,scr),True,0L,0,&i,1)) {
      if (i>=MAXCOLS) {
      	XUngrabServer(dpy);
         fprintf(stderr,"program not suited for displays with more than %d colors\n", MAXCOLS );
         exit(-1);
      }
      pixel_usage[i]=FreePixel;
   }

/*
 * find successive entries, if there are pixels of other application in the
 * range, it has to be checked if they are shareable and fit into the map.
 */
   for (first=0;first<MAXCOLS-ncols;first++) {
      for (i=0;i<ncols;i++) {
         int used=check_pixel(first+i);
         if (used==PrivatePixel)          break;
         if (used==SharedPixel) {
            XColor current;
            current=col[i];
            if (XAllocColor(dpy,DefaultColormap(dpy,scr),&current)) {
               XFreeColors(dpy,DefaultColormap(dpy,scr),&current.pixel,1,0L);
               if (first+i!=current.pixel)      break;
            }
            else  break;
         }
      }
      if (i==ncols)  break;
/*    first=first+i; */
   }
   if (first==MAXCOLS-ncols) {
   	XUngrabServer(dpy);
   	free_unused();
      fprintf( stderr, "unable to allocate %d successive pixels for map\n",ncols);
      return -1;
   }

   best_map_info.base_pixel = first;
   best_map_info.visualid   = DefaultVisual(dpy,scr)->visualid;
   best_map_info.killid     = win;     /* kill window to kill standard map */

/*
 * allocate successive entries by freeing a single entry and allocating
 * the necessary rgb-value read-only. If a different pixel is found, a
 * read/write-pixel has to be allocated to get the rgb-value stored at the
 * desired pixel.
 */
   for (i=0;i<ncols;i++) {
      if (pixel_usage[first+i]==FreePixel) {
         XColor   col_bak;

         col[i].pixel   = first+i;
         col[i].flags   = DoRed | DoGreen | DoBlue;
         col_bak        = col[i];

         XFreeColors(dpy,DefaultColormap(dpy,scr),&col[i].pixel,1,0L);
         if (!XAllocColor(dpy,DefaultColormap(dpy,scr),&col[i])
            || col[i].pixel!=i+first) {
               unsigned long  pixel2;
               if (!XAllocColorCells(dpy,DefaultColormap(dpy,scr),True,0L,0,&pixel2,1)
                  || pixel2!=i+first ) {
                  XUngrabServer(dpy);
                  free_unused();
                  fprintf(stderr,"internal error: can't reallocate pixel %d\n", first+i );
                  return -3;
               }
               col_bak.pixel = pixel2;
               XStoreColor(dpy,DefaultColormap(dpy,scr),&col_bak);
               if (verbose) {
               	XUngrabServer(dpy);		/* just for not getting stuck */
                  printf( "%d: rgb:%04x/%04x/%04x read/write\n",
                           ++rw_count, col_bak.red, col_bak.green, col_bak.blue );
                  XGrabServer(dpy);
               }
               pixel_usage[i+first]=PrivatePixel;
         }
         else  pixel_usage[i+first]=SharedPixel;
      }
   }

/*
 * all unused (but allocated) pixels can now be freed again
 */
   free_unused();
   XUngrabServer(dpy);

   if (verbose)   printf( "map start at pixel %d\n", first );


/*
 * setup the standard colormap structure
 */
   XSetRGBColormaps(dpy,RootWindow(dpy,scr),&best_map_info,1,map_type);
   XFlush(dpy);

   return 0;
}

/*----------------------------------------------------------------------------*/

#define  Width    DisplayWidth(dpy,scr)
#define  Height   DisplayHeight(dpy,scr)
#define  Cells    DisplayCells(dpy,scr)

/* -------------------------------------------------------------------------- *
   set_palette: set all unused colors to a color from <from> to <to>
 * -------------------------------------------------------------------------- */

int set_palette( from, to )
   char  *from;
   char  *to;
{
int i,j;
XColor   color1, color2;
XColor   color;
long  pixel[MAXCOLS];

   XParseColor( dpy, DefaultColormap(dpy,scr), from, &color1 );
   XParseColor( dpy, DefaultColormap(dpy,scr), to  , &color2 );

/*
 * allocate unused cells of the colormap for read/write
 */
   i = 0;
   while(XAllocColorCells( dpy, DefaultColormap(dpy,scr), False,
         0L, 0, &pixel[i++], 1 ));
/*
 * write color into the colormap
 */
   for (j=0;j<i-1;j++) {
	 if (set_flag) {
      color.red  =color1.red   + (int)(color2.red-color1.red)*j/i;
      color.green=color1.green + (int)(color2.green-color1.green)*j/i;
      color.blue =color1.blue  + (int)(color2.blue-color1.blue)*j/i;
    }
    else {
      color.red  =color1.red   + (int)(color2.red-color1.red)*pixel[j]/255;
      color.green=color1.green + (int)(color2.green-color1.green)*pixel[j]/255;
      color.blue =color1.blue  + (int)(color2.blue-color1.blue)*pixel[j]/255;
    }
    color.flags=DoRed | DoBlue | DoGreen;
    color.pixel=pixel[j];
    XStoreColor( dpy, DefaultColormap(dpy,scr), &color );
   }


/*
 * wait and keep all the colors allocated, when desired
 */
   if (wait_flag) {
      sleep(100000);    /* just wait and keep the colors allocated ... */
   }
/*
 * free the pixels again
 */
   XFreeColors(dpy,DefaultColormap(dpy,scr),pixel,i-1,0L);
   XSync( dpy, 0 );
   return 0;
}

/*----------------------------------------------------------------------------*/

#define  TITLE_STRING   "DEFAULT-COLORMAP"

/* -------------------------------------------------------------------------- *
   find_palette: searches for a window already used to show a palette
                 by checking the window-names
 * -------------------------------------------------------------------------- */

Window find_palette( window )
   Window   window;
{
Window   parent, root;
Window   *child;
int      nchildren,i;
Window   ret=0;
char     *name_ret;

   if (XFetchName(dpy,window,&name_ret)) {
      if (!strcmp(name_ret,TITLE_STRING)) {
         XFree(name_ret);
         return window;
      }
      XFree(name_ret);
   }

   if (XQueryTree(dpy,window,&root,&parent,&child,&nchildren)) {
      for (i=0;i<nchildren;i++) {
         ret=find_palette( child[i] );
         if (ret)    break;
      }
      XFree(child);
   }
   return ret;
}

/* -------------------------------------------------------------------------- *
   show_palette: opens a palette-window at the desired position. The program
                 stays in the event-loop until the window is deleted.
 * -------------------------------------------------------------------------- */

void show_palette() {
int      i,first,last,found;
Window   window;
XEvent   event;
GC gc;
XSetWindowAttributes attrib;


   gc = XCreateGC( dpy, RootWindow(dpy,scr), 0L, NULL );

	found=0;
   while( window=find_palette( RootWindow(dpy,scr) ) ) {
      if (size<0) {
         Window   root;    /* Inherit size information */
         int      x,y;
         unsigned w,h,b,d;

         XGetGeometry(dpy,window,&root,&x,&y,&w,&h,&b,&d);
         if (w<h)    size=w;
         else        size=h;
      }
      XDestroyWindow(dpy,window);
      found++;
   }
   if (!found || set_flag)			set_palette( color1, color2 );
   if (pos==4)    exit(0);             /* just hiding the old palette */
   if (size<0)    size=12;

   window = XCreateSimpleWindow( dpy, RootWindow(dpy,scr),
         (pos==1)?(Width-size):0,
         (pos==2)?(Height-size):0,
         (pos&1)?size:Width, (pos&1)?Height:size,
         0, WhitePixel(dpy,scr), BlackPixel(dpy,scr));

   attrib.override_redirect = 1;
   XChangeWindowAttributes( dpy, window, CWOverrideRedirect, &attrib );
   XStoreName(dpy,window,TITLE_STRING);

   XSelectInput( dpy, window, ExposureMask|StructureNotifyMask );
   XMapRaised( dpy, window );

   do {
      XNextEvent( dpy, &event );
      switch( event.type ) {
      case Expose:
         if (pos&1) {
            first = event.xexpose.y * Cells / Height;
            last  = (event.xexpose.y+event.xexpose.height) * Cells / Height + 1;
            for ( i=first; i<=last; i++ ) {
               XSetForeground( dpy, gc, i );
               XFillRectangle( dpy, window, gc,
                  0, i*Height/Cells,
                  size, (i+1)*Height/Cells-i*Height/Cells );
            }
         }
         else {
            first = event.xexpose.x * Cells / Width;
            last  = (event.xexpose.x+event.xexpose.width) * Cells / Width + 1;
            for ( i=first; i<=last; i++ ) {
               XSetForeground( dpy, gc, i );
               XFillRectangle( dpy, window, gc,
                  i*Width/Cells, 0,
                  (i+1)*Width/Cells-i*Width/Cells, size );
            }
         }
         break;
      case DestroyNotify:
         exit(0);
      } /* end switch */
   } while (1);
}

/*----------------------------------------------------------------------------*/

/* -------------------------------------------------------------------------- *
   query_rgb:  query the rgb-values if the installed pixels of the given map.
 * -------------------------------------------------------------------------- */

static XColor  *colors=0L;

int query_rgb(map_type)
   Atom  map_type;
{
int   i,j;
int count;
XStandardColormap *info_field;
int   ncols;   /* number of colors of the map */

   if (XGetRGBColormaps(dpy,RootWindow(dpy,scr),&info_field,&count,map_type)) {
      for (j=0;j<count;j++) {
         if (info_field[j].colormap==DefaultColormap(dpy,scr)) {
            /* map is installed */
            int   rmax = info_field[j].red_max   + 1;
            int   gmax = info_field[j].green_max + 1;
            int   bmax = info_field[j].blue_max  + 1;

            switch( map_type ) {
            case XA_RGB_BEST_MAP:
            case XA_RGB_DEFAULT_MAP:
                  ncols = rmax*gmax*bmax;
                  break;
            case XA_RGB_RED_MAP:
                  ncols = rmax;
                  break;
            case XA_RGB_GREEN_MAP:
                  ncols = gmax;
                  break;
            case XA_RGB_BLUE_MAP:
                  ncols = bmax;
                  break;
            case XA_RGB_GRAY_MAP:
                  ncols = rmax+gmax+bmax;
                  break;
            }

            /* allocate colorfield, if not already done */
            if (!colors) {
               colors=(XColor*)malloc(MAXCOLS*sizeof(XColor));
               memset(colors,0,MAXCOLS*sizeof(XColor));
            }

            /* for all pixels contained in the current standard map */
            for (i=0;i<ncols;i++) {
               unsigned int pixel = info_field[j].base_pixel + i;

               /* if rgb-value wasn't already queried, do it now */
               if (pixel<MAXCOLS && colors[i].flags==0) {
                  colors[i].pixel = pixel;
                  colors[i].flags = DoRed|DoGreen|DoBlue;
                  XQueryColor(dpy,DefaultColormap(dpy,scr),&colors[i]);
                  colors[i].flags = DoRed|DoGreen|DoBlue;
               }
            }
            XFree(info_field);
            return 1;
         }
      }
      XFree(info_field);
   }
   return 0;
}

/* -------------------------------------------------------------------------- *
   query_rgbs: query the rgb-values of the selected map of of all maps,
               if there wasn't any map selected.
 * -------------------------------------------------------------------------- */

int query_rgbs() {
   if (map_type)  return query_rgb(map_type);      /* query selected */
   else {
      int   i,erg;
      erg=0;
      for (i=0;i<sizeof(map)/sizeof(Atom);i++)     /* loop over all maps */
         erg+=query_rgb(map[i]);
      return erg;                                  /* 0 - no map found */
   }
}


/* -------------------------------------------------------------------------- *
   round_rgb: find closest rgb-value of one in the standard colormaps
 * -------------------------------------------------------------------------- */

int round_rgb(red,green,blue)
   unsigned int   *red;
   unsigned int   *green;
   unsigned int   *blue;
{
int   i,min_i;
long  dist;
long  min_dist;

   if (!colors) {
      if (!query_rgbs()) {
         fprintf( stderr,"couldn't find any installed standard colormaps\n" );
         exit(-1);
      }
   }

   min_i=-1;
   if (colors) {
      for (i=0;i<MAXCOLS;i++) {
         if (colors[i].flags) {
            long  rd = ((long)colors[i].red   - (long)*red)/4;
            long  gd = ((long)colors[i].green - (long)*green)/4;
            long  bd = ((long)colors[i].blue  - (long)*blue)/4;
            long  dist=rd*rd+gd*gd+bd*bd;

            if (min_i<0 || dist<min_dist) {
               min_dist = dist;
               min_i    = i;
            }
         }
      }
   }

   if (min_i>=0) {
         *red   = colors[min_i].red;
         *green = colors[min_i].green;
         *blue  = colors[min_i].blue;
         return 1;
   }
   else  return 0;
}

/* -------------------------------------------------------------------------- */

typedef struct alias_entry {
   char                 *name;
   char                 *alias;
   struct alias_entry   *next;
} AliasEntry;

static AliasEntry *first=0l;

/* -------------------------------------------------------------------------- *
   add_alias: inserts current alias into a list of alias-entry, which will
              be written into the database-file at the end of the program.
 * -------------------------------------------------------------------------- */

void add_alias(name,value)
   char  *name;
   char  *value;
{
AliasEntry  *new_alias;

   new_alias = (AliasEntry*)malloc(sizeof(AliasEntry));
   new_alias->name  = strdup(name);
   new_alias->alias = strdup(value);
   if (!first || strcmp(first->name,name)>0 ) {
      /* add as start  of the list */
         new_alias->next   = first;
         first             = new_alias;
   }
   else {
      /* insert in sorted list of entries */
         AliasEntry  *current=first;

         while(current->next&&strcmp(current->next->name,name)<0) {
            current = current->next;
         }
         new_alias->next = current->next;
         current->next   = new_alias;
   }
}

/* -------------------------------------------------------------------------- *
   scan_rgb: checks the current input line for an existing colorname
             and scans the rgb-value (when in rgb.txt format) or querys
             a value from the server
 * -------------------------------------------------------------------------- */

char *scan_rgb( linebuffer, red_out, green_out, blue_out )
   char           *linebuffer;
   unsigned int   *red_out;
   unsigned int   *green_out;
   unsigned int   *blue_out;
{
char  *str;

   if (linebuffer[0]=='!')    return 0;      /* comment leader */

   if (sscanf(linebuffer,"%d %d %d", red_out, green_out, blue_out)==3) {
      /* line in style of rgb.txt */
         *red_out   |= (*red_out<<8);        /* expand to 16 bit */
         *green_out |= (*green_out<<8);
         *blue_out  |= (*blue_out<<8);
         str = linebuffer;
         while( *str && (isspace(*str) || isdigit(*str)) )  str++;
         return str;
   }
   else {
      XColor   col;

      if (XParseColor(dpy,DefaultColormap(dpy,scr),linebuffer,&col)) {
         *red_out   = col.red;
         *green_out = col.green;
         *blue_out  = col.blue;
         return linebuffer;
      }
   }
   return 0;
}

/* -------------------------------------------------------------------------- *
   write_xcms_data: writes the collected entry into the database-file
 * -------------------------------------------------------------------------- */

static void write_xcms_data() {
int i;
time_t   current=time(NULL);
AliasEntry  *entry=first;

   fprintf( out_fp, "/*\n" );
   fprintf( out_fp, " * This color name database file for CSO was automatically created\n" );
   fprintf( out_fp, " * on the following machine:\n" );
   fprintf( out_fp, " *\n" );
   fprintf( out_fp, " * name of display:       %s\n", DisplayString(dpy) );
   fprintf( out_fp, " * version number:        %d.%d\n", ProtocolVersion(dpy), ProtocolRevision(dpy) );
   fprintf( out_fp, " * vendor string:         %s\n", ServerVendor(dpy) );
   fprintf( out_fp, " * vendor release number: %d\n", VendorRelease(dpy) );
   fprintf( out_fp, " * date of conversion:    %s", asctime(localtime(&current)) );
   fprintf( out_fp, " *\n" );
   fprintf( out_fp, " * The file was created with xdefmap by H. Hoenig (Aug-3-95)\n" );
   fprintf( out_fp, " */\n\n" );

   fprintf( out_fp, "XCMS_COLORDB_START 0.1\n");
   while( entry ) {
      fprintf( out_fp, "%-25s\t%s\n", entry->name, entry->alias );
      entry=entry->next;
   }
   fprintf( out_fp,"XCMS_COLORDB_END\n");
}

/* -------------------------------------------------------------------------- *
   create_database: top-function for creating (scanning/rounding/writing)
                    the xcms-database.
 * -------------------------------------------------------------------------- */

void create_database()
{
char  buffer[1000];
unsigned int   red,green,blue;
char           *name;

   while(fgets(buffer,sizeof(buffer),in_fp)) {
      buffer[strlen(buffer)-1]='\0';   /* add end of string */
      if (name=scan_rgb(buffer,&red,&green,&blue)) {
         char  alias[80];
         round_rgb(&red,&green,&blue);
         sprintf(alias,"rgb:%04x/%04x/%04x",red,green,blue);
         add_alias(name,alias);
      }
   }

   write_xcms_data();
}

/* -------------------------------------------------------------------------- *
   write_ppm_data: ppm-file containing colors of all shared pixels
 * -------------------------------------------------------------------------- */

void write_ppm_data(bit8)
	int	bit8;
{
int shared_count=0;
int i;

   for (i=0;i<MAXCOLS;i++) {
      if (pixel_usage[i]==Unknown) {
         if (check_pixel(i)==SharedPixel)    shared_count++;
      }
   }

	fprintf( out_fp, "P3\n" );
	fprintf( out_fp, "1 %d\n", shared_count );
	fprintf( out_fp, "%u\n", bit8?255:65535 );

   for (i=0;i<MAXCOLS;i++) {
     	if (pixel_usage[i]==SharedPixel) {
  		  	XColor	xcol;
     		xcol.pixel = i;
     		XQueryColor(dpy,DefaultColormap(dpy,scr),&xcol);
			if (bit8)
	     		fprintf( out_fp, "%d %d %d\n",
					xcol.red>>8, xcol.green>>8, xcol.blue>>8 );
			else
     			fprintf( out_fp, "%u %u %u\n",
					xcol.red, xcol.green, xcol.blue );
      }
   }
}

void write_ppm_std_data(bit8)
	int	bit8;
{
int shared_count=0;
XStandardColormap *info_field;
int count;
int i,j;
int r,g,b;

   for (i=0;i<sizeof(map)/sizeof(Atom);i++) {
      if (XGetRGBColormaps(dpy,RootWindow(dpy,scr),&info_field,&count,map[i])) {
         for (j=0;j<count;j++) {
         	shared_count+=( (info_field[j].red_max+1)*(info_field[j].green_max+1)*(info_field[j].blue_max+1) );
         }
         XFree(info_field);
      }
   }

	fprintf( out_fp, "P3\n" );
	fprintf( out_fp, "1 %d\n", shared_count );
	fprintf( out_fp, "%u\n", bit8?255:65535 );

   for (i=0;i<sizeof(map)/sizeof(Atom);i++) {
      if (XGetRGBColormaps(dpy,RootWindow(dpy,scr),&info_field,&count,map[i])) {
         for (j=0;j<count;j++) {
         	for (b=0;b<=info_field[j].blue_max;b++) {
         		for (g=0;g<=info_field[j].green_max;g++) {
         			for (r=0;r<=info_field[j].red_max;r++) {
  		  					XColor	xcol;
         				xcol.pixel=info_field[j].base_pixel+r*info_field[j].red_mult+g*info_field[j].green_mult+b*info_field[j].blue_mult;
				     		XQueryColor(dpy,DefaultColormap(dpy,scr),&xcol);
							if (bit8)
     							fprintf( out_fp, "%d %d %d\n",
									xcol.red>>8, xcol.green>>8, xcol.blue>>8 );
							else
     							fprintf( out_fp, "%u %u %u\n",
									xcol.red, xcol.green, xcol.blue );
         			}
         		}
         	}
         }
         XFree(info_field);
      }
   }
}

/*----------------------------------------------------------------------------*/

static void usage() {
   printf( "usage: xdefmap [-display host:dpy] option\n" );
   printf("        setup selected standard colormap in the default colormap\n" );
   printf("    or  visualize/set current color allocation\n\n" );
   printf("Author     : Helmut Hoenig, April-2-96 (V1.3)   (Helmut.Hoenig@hub.de)\n\n" );

   printf("    To install/uninstall a default map:\n" );
   printf("        -best        : make the RGB_BEST_MAP\n" );
   printf("        -default     : make the RGB_DEFAULT_MAP\n" );
   printf("        -gray        : make the RGB_GRAY_MAP\n" );
   printf("        -red         : make the RGB_RED_MAP\n" );
   printf("        -green       : make the RGB_GREEN_MAP\n" );
   printf("        -blue        : make the RGB_BLUE_MAP\n" );
   printf("        -delete name : remove a standard colormap\n" );
   printf("        [-s] <n>     : number of intensities\n" );
   printf("        <r><g><b>    : colordistribution (intensity levels for R,G,B)\n" );

   printf("    To visualize palette:\n" );
   printf("        -top -right -bottom -left    : show palette at desired border\n" );
   printf("        -hide                        : hide palette again\n" );

   printf("    To set unused entries:\n" );
   printf("        -set [<color1>] [<color2>]   : preset unallocated entries\n" );
   printf("        -wait                        : keeps entries allocated\n" );

   printf("    To create XCMSDB alias databases:\n" );
   printf("        -lookup             : lookup names from stdin\n" );
   printf("        -parse [<rgb-file>] : parse file in format of rgb.txt\n" );
   printf("        -f <newfile>        : write results to file (instead of stdout)\n" );
/* printf("        -join <database>    : join results with given database\n" ); */

   printf("    To write map-file in ppm-format:\n" );
   printf("        -stdmap <ppmfile>   : To write mapfile colors in standard colormaps\n" );
   printf("        -map <ppmfile>      : To write mapfile of all shared colors\n" );
   printf("        -8bit / -16bit      : Use 8/16 bit in color intensities\n" );
   printf("    Others:\n" );
   printf("        -q         : query colormap information\n" );
   printf("        -v         : verbose\n\n" );

   exit(-1);
}

#define BUF_LEN   256

static XErrorHandler old_handler;

static int x_error( dpy, error_event )
   Display *dpy;
   XErrorEvent *error_event;
{
char buffer[BUF_LEN];

   switch( error_event->error_code ) {
   case BadValue:
      if (verbose) {
         fprintf( stderr, "IGNORED error_code 2: BadValue\n" );
      }
      return(0);

   default:
      fprintf( stderr, "*****\n" );
      XGetErrorText( dpy, error_event->error_code, buffer, BUF_LEN );
      fprintf( stderr, "%s\n", buffer );
      fprintf( stderr, "request_code %d\n", error_event->request_code );
      fprintf( stderr, "XError occured in -request %d\n", error_event->serial );
      fprintf( stderr, "*****\n" );
      exit(0);
   }
   return(0);
}

void main(argc,argv)
   int   argc;
   char  **argv;
{
int      i,j;
char     *dispname="";
int      rmax=-1,gmax=-1,bmax=-1;
XEvent   event;
int      delete_flag=0;
int      query_flag=0;
int      ppm_flag=0;
int      bit8_flag=1;

   in_fp   = stdin;
   out_fp  = stdout;
   join_fp = 0L;

   for (i=1;i<argc;i++) {
      if (!strcmp(argv[i],"-dpy"))        dispname=argv[++i];
      else if (!strcmp(argv[i],"-v"))        verbose=1;
      else if (!strcmp(argv[i],"-sync"))     sync=1;
      else if (!strcmp(argv[i],"-delete"))   delete_flag=1;
      else if (!strcmp(argv[i],"-q"))        query_flag=1;
      else if (!strcmp(argv[i],"-s"))        size=atoi(argv[++i]);

      else if (strcmp(argv[i],"-top")==0)       pos=0;
      else if (strcmp(argv[i],"-right")==0)     pos=1;
      else if (strcmp(argv[i],"-bottom")==0)    pos=2;
      else if (strcmp(argv[i],"-left")==0)      pos=3;
      else if (strcmp(argv[i],"-hide")==0)      pos=4;
      else if (strcmp(argv[i],"-set")==0) {
         set_flag = 1;
         if (i<argc-1&&argv[i+1][0]!='-') {
            color1 = argv[++i];
            if (i<argc-1&&argv[i+1][0]!='-') {
               color2 = argv[++i];
            }
         }
      }
      else if (strcmp(argv[i],"-wait")==0) {
         set_flag = 1;
         wait_flag=1;
      }
      else if (strcmp(argv[i],"-lookup")==0)    xcms_flag=1;
      else if (strcmp(argv[i],"-parse")==0) {
         char  *rgb_file = (i<argc-1&&argv[i+1][0]!='-')?
                           argv[++i]:"/usr/lib/X11/rgb.txt";
         xcms_flag=1;
         in_fp = fopen( rgb_file, "r" );
         if (!in_fp) {
            fprintf( stderr, "unable to open file %s\n", argv[i] );
            exit(-1);
         }
      }
      else if (strcmp(argv[i],"-join")==0) {
         xcms_flag=1;
         join_fp = fopen( argv[++i], "r" );
         if (!join_fp) {
            fprintf( stderr, "unable to open file %s\n", argv[i] );
            exit(-1);
         }
      }
      else if (strcmp(argv[i],"-f")==0) {
         out_fp = fopen( argv[++i], "w" );
         if (!out_fp) {
            fprintf( stderr, "can't open file %s for writing\n", argv[i] );
            exit(-1);
         }
      }
      else if (strcmp(argv[i],"-map")==0||strcmp(argv[i],"-stdmap")==0) {
         ppm_flag = (strcmp(argv[i],"-stdmap"))?1:2;
      	if (strcmp(argv[i+1],"-")) {
	         out_fp = fopen( argv[++i], "w" );
	         if (!out_fp) {
  	         	fprintf( stderr, "can't open file %s for writing\n", argv[i] );
  	         	exit(-1);
  	         }
  	      }
  	      else	out_fp=stdout;
      }
      else if (strcmp(argv[i],"-8bit")==0) { bit8_flag = 1; }
      else if (strcmp(argv[i],"-16bit")==0) { bit8_flag = 0; }
      else if (atoi(argv[i])>0) {
         int len=strlen(argv[i]);

         if (len!=3)    size=atoi(argv[i]);
         else {
            rmax = argv[i][0]-'0';
            if (rmax<0 || rmax>9)      rmax=0;
            gmax = argv[i][1]-'0';
            if (gmax<0 || gmax>9)      gmax=0;
            bmax = argv[i][2]-'0';
            if (bmax<0 || bmax>9)      bmax=0;
            size=atoi(argv[i]);
         }
      }
      else {
         for (j=sizeof(map)/sizeof(Atom)-1;j>=0;j--) {
            if (!strcasecmp(argv[i],map_string[j])||!strcasecmp(&argv[i][1],map_string[j])) {
               map_type=map[j];
               break;
            }
         }
         if (j<0)    usage();
      }
   }
   if (pos>=0) {                    /* if the window should get mapped  */
      if (fork()>0)     exit(0);    /* the parent forks and quits       */
   }

   old_handler = XSetErrorHandler(x_error);
   dpy = XOpenDisplay(dispname);
   if (!dpy) {
      fprintf(stderr,"can't open display '%s'\n", dispname);
      exit(-1);
   }
   scr = DefaultScreen(dpy);
   if (xcms_flag) {
      create_database();
      exit(0);
   }

   if (query_flag) {
      mapusage(0);
      show_mapinfo();
      show_visualinfo();
      exit(0);
   }
   if (ppm_flag) {
   	if (ppm_flag==1)			write_ppm_data(bit8_flag);
   	else if (ppm_flag==2)	write_ppm_std_data(bit8_flag);
   	exit(0);
   }

   if (pos>=0) {
      show_palette();
   }
   if (set_flag) {
      set_palette( color1, color2 );
      exit(0);
   }

   if (map_type==0)                           usage();
   if (sync)   XSynchronize(dpy,1);

   if (delete_flag) {
      delete_map( map_type );
      exit(0);
   }

   if (DefaultVisual(dpy,scr)->class!=PseudoColor) {
      fprintf(stderr,"Visual class not 'PseudoColor': standard colormaps not installed.");
      exit(-1);
   }

   win = XCreateSimpleWindow(dpy,RootWindow(dpy,scr),0,0,1,1,0,0,0);
   XStoreName(dpy,win,"XID for Standard Colormap");

   switch( map_type ) {
   case XA_RGB_BEST_MAP:
   case XA_RGB_DEFAULT_MAP:
         size = (size<0)?DEFAULT_SIZE_RGB:size;
         if (rmax<0) rmax=gmax=bmax=size;
         break;
   case XA_RGB_RED_MAP:
         rmax = (size<0)?DEFAULT_SIZE_BW:size;
         gmax = bmax = 0;
         break;
   case XA_RGB_GREEN_MAP:
         rmax = bmax = 0;
         gmax = (size<0)?DEFAULT_SIZE_BW:size;
         break;
   case XA_RGB_BLUE_MAP:
         rmax = gmax = 0;
         bmax = (size<0)?DEFAULT_SIZE_BW:size;
         break;
   case XA_RGB_GRAY_MAP:
         size = (size<0)?DEFAULT_SIZE_BW:size;
         gmax = size*152/256;
         bmax = size*29/256;
         rmax = size-gmax-bmax;
         break;
   }

   if (setup_best_map(map_type,rmax,gmax,bmax))       exit(-1);

   XSetCloseDownMode(dpy,RetainPermanent);   /*let the colors be allocated ...*/
   XSync(dpy,0);
}
