/*

Copyright 1990 by Janet Carson

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose is hereby granted without fee, provided that
the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.  The author makes no representations about the
suitability of this software for any purpose.  It is provided "as is"
without express or implied warranty.

*/

/*
 *  Window shutdown program by Janet L. Carson, Baylor College of Medicine.
 *  Version 1.0, placed in /contrib on 2/12/90.
 *
 *  Please send comments or fixes to jcarson@bcm.tmc.edu
 */

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <stdio.h>

Display *d;
Atom wm_delete_window;
Atom wm_protocols;
Atom wm_state;
int err_occurred = False;

/*
 *  I'm probably going to get some BadWindow errors as I kill clients
 *  which have multiple windows open and then try to kill them again
 *  on another of their windows.  I'm just going to plow right through!
 *  The flag is set back to false in recurse_tree and kill_tree...
 */

int err_handler(dpy, err)
   Display *dpy;
   XErrorEvent *err;
{
    err_occurred = True;
    return;
}

/*
 *  Looking for properties...
 */

int has_property(w, prop)
   Window w;
   Atom prop;
{
   int nprops, j, retval = 0;
   Atom *list = XListProperties(d, w, &nprops);

   if (err_occurred)
       return 0;

   for (j = 0; j < nprops; j++)
   {
       if (list[j] == prop)
       {
           retval = 1;
           break;
       }
   }

   if (nprops)
      XFree(list);

   return retval;
}

int has_wm_state(w)
   Window w;
{
   return has_property(w, wm_state);
}

int has_wm_protocols(w)
   Window w;
{
   return has_property(w, wm_protocols);
}

/*
 *  Send a WM_PROTOCOLS WM_DELETE_WINDOW message to a window
 */

void send_delete_message(w)
   Window w;
{
   XClientMessageEvent xclient;

   xclient.type = ClientMessage;
   xclient.send_event = True;
   xclient.display = d;
   xclient.window = w;
   xclient.message_type = wm_protocols;
   xclient.format = 32;
   xclient.data.l[0] = wm_delete_window;

   XSendEvent(d, w, False, 0, &xclient);
}

/*
 *  To shutdown a top level window:  if the window participates
 *  in WM_DELETE_WINDOW, let the client shut itself off.  Otherwise,
 *  do an XKillClient on it.
 */

void handle_top_level(w)
   Window w;
{
   Atom *prots;
   int nprots, j;

   if (has_wm_protocols(w))
   {
       XGetWMProtocols(d, w, &prots, &nprots);

       if (err_occurred)
           return;

       for (j = 0; j < nprots; j++)
          if (prots[j] == wm_delete_window)
          {
             send_delete_message(w);
             break;
          }

       if (j == nprots)  /* delete window not found */
          XKillClient(d, w);

       XFree(prots);
   }
   else
      XKillClient(d, w);
}

/*
 *  recurse_tree: look for top level windows to kill all the way down
 *  the window tree.  This pass is "nice"--I'll use delete_window protocol
 *  if the window supports it.  If I get an error in the middle, I'll start
 *  over again at the same level, because reparenting window managers throw
 *  windows back up to the root...
 */

void recurse_tree(w)
   Window w;
{
   Window root, parent, *kids;
   int nkids, j;
   int wm_state;

   for (;;)
   {
      XQueryTree(d, w, &root, &parent, &kids, &nkids);
      if (err_occurred)
      {
           err_occurred = False;
           return;
      }

      for (j = 0; j < nkids; j++)
      {
          wm_state = has_wm_state(kids[j]);
   
          if (err_occurred)
              break;
   
          if (wm_state)
          {
             handle_top_level(kids[j]);
             if (err_occurred)
                 break;
          }
          else
             recurse_tree(kids[j]);
      }

      XFree(kids);

      /* when I get all the way through a level without an error, I'm done  */

      if (err_occurred)
          err_occurred = False;
      else
          return;

   }
}

/*
 *  This is the second pass--anything left gets an XKillClient!
 */

void kill_tree(w)
   Window w;
{
   Window root, parent, *kids;
   int nkids, j;

   for (;;)
   {
      XQueryTree(d, w, &root, &parent, &kids, &nkids);
      if (err_occurred)
      {
           err_occurred = False;
           return;
      }
   
      for (j = 0; j < nkids; j++)
      {
           XKillClient(d, kids[j]);
           if (err_occurred)
             break;
      }

      XFree(kids);

      /* when I get all the way through a level without an error, I'm done  */

      if (err_occurred)
          err_occurred = False;
      else
          return;
   }
}

/*
 *  Main program
 */

main(argc, argv)
   int argc;
   char *argv[];
{
    char *display_string = XDisplayName(NULL);
    int j;

    /* see if -display is set */

    for (j = 1; j < argc - 1; j++)
    {
        if (argv[j][0] == '-' &&
            argv[j][1] == 'd')
          display_string = argv[j + 1];
    }

    /* open display */

    d = XOpenDisplay(display_string);

    if (d == NULL)
    {
       fprintf(stderr, "Could not open display.");
       exit(1);
    }

    /* init global variables */

    wm_state = XInternAtom(d, "WM_STATE", False);
    wm_protocols = XInternAtom(d, "WM_PROTOCOLS", False);
    wm_delete_window = XInternAtom(d, "WM_DELETE_WINDOW", False);

    /* synchronize -- so I'm aware of errors immediately */

    XSynchronize(d, True);

    /* use my error handler from here on out */

    (void) XSetErrorHandler(err_handler);

    /* start looking for windows to kill -- be nice on pass 1 */

    for (j = 0; j < ScreenCount(d); j++)
    {
        recurse_tree(RootWindow(d, j));
    }

    /* wait for things to clean themselves up */

    sleep(5);

    /* this will forcibly kill anything that's still around --
       this second pass may or may not be needed... */

    for (j = 0; j < ScreenCount(d); j++)
    {
        kill_tree(RootWindow(d, j));
    }
}
