/* ----------------------------------------------------------------------------
 * Copyright (C) 1995-1999 by Karim Kaschani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * to rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 * --------------------------------------------------------------------------*/

#include "xlogin.h"
#include "parse.h"
#include "strutils.h"
#include "net.h"
#include "xhost.h"
#include "connect.h"
#include "help.h"
#include "xLogin.xbm"
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>






/* -------------------------------------------------------- global variables */

static char *defTerminals[] = {			/* default terminals         */
	"xterm",
#ifdef AIX
	"aixterm",
#endif
#ifdef HPUX
	"hpterm",
#endif
#ifdef SUN
	"shelltool",
	"cmdtool",
#endif
	NULL
};

static struct session_type defSessions[] = {	/* default sessions          */
	{ "telnet", TRUE  },
	{ "rlogin", TRUE  },
	{ "ftp",    FALSE },
	{ NULL,     FALSE }
};

static struct host_data	*host[MAX_HOSTS];	/* host setting structure    */
static struct s_type	sTable;			/* session table structure   */
static struct t_type	tTable;			/* terminal table structure  */

static char     	*ETC_HOSTS = "/etc/hosts";
static char     	*RC_PATH = "/.xLogin";
static char     	*RC_FILE = "/xLoginrc";
static char     	*EX_FILE = "/xLoginrc.ex";
static char     	*LOG_FILE = "/log";
static char		rc_file[MAX_CHARS];	/* resource file             */
static char		ex_file[MAX_CHARS];	/* sample resource file      */
static char		rc_path[MAX_CHARS];	/* resource path             */
static char		log_file[MAX_CHARS];	/* error logging file        */
static Widget		app_shell;		/* ApplicationShell          */
static XtAppContext	app_context; 		/* App Context               */
static Widget		auto_toggle;		/* automatic server access   */
static Widget		enable_toggle;		/* enable server access      */
static Widget		disable_toggle;		/* disable server access     */
static Widget		s_option;		/* session option menu       */
static Widget		t_option;		/* terminal option menu      */
static Widget		selectionBox;		/* selection box             */
static Widget		selectionField;		/* selection field           */
static int		host_number = 0;	/* number of hosts           */
static int		hIndex;			/* selected host index       */
static char		*username;		/* username                  */
static char		*inetAddress;		/* curr. host internet addr. */
static int		tID = 0;		/* current terminal number   */
static int		sID = 0;		/* current session number    */
static int		etcHosts = FALSE;
static int		firstTime = TRUE;	/* flag to suppress calling  *
						 * toggle button callbacks   *
						 * twice		     */
static int		selection = TRUE;	/* selection status          */
static int		otherHost = FALSE;	/* flag to indicate if an    *
						 * unlisted host address has *
						 * been entered              */

static char *fallbackResources[] = {
    "xLogin*fontList: 7x14",
    "xLogin*foreground: black",
    "xLogin*background: grey80",
    "xLogin*topShadowColor: grey90",
    "xLogin*optionPulldown*foreground: red",
    "xLogin*optionPulldown*background: grey70",
    "xLogin*promptDialog*highlightColor: red",
    "xLogin*promptDialog*background: grey70",
    "xLogin*serverAccess*foreground: black",
    "xLogin*serverAccess*background: grey70",
    "xLogin*serverAccess*selectColor: red",
    "xLogin*menuBar*foreground: black",
    "xLogin*menuBar*background: grey70",
    "xLogin*menuBar*messageForm*foreground: black",
    "xLogin*menuBar*messageForm*background: grey80",
    "xLogin*menuBar*messageForm*messageText*foreground: black",
    "xLogin*menuBar*messageForm*messageText*background: grey70",
    "xLogin*fileMenu.Log.accelerator: Ctrl<Key>l",
    "xLogin*fileMenu.Log.acceleratorText: Ctrl+L",
    "xLogin*fileMenu.Report.accelerator: Ctrl<Key>r",
    "xLogin*fileMenu.Report.acceleratorText: Ctrl+R",
    "xLogin*fileMenu.Close.accelerator: Ctrl<Key>c",
    "xLogin*fileMenu.Close.acceleratorText: Ctrl+C",
    "xLogin*syncMenu.LogFiles.accelerator: Shift Ctrl<Key>s",
    "xLogin*syncMenu.LogFiles.acceleratorText: [Shift]Ctrl+S",
    "xLogin*syncMenu.XHostTable.accelerator: Ctrl<Key>s",
    "xLogin*syncMenu.XHostTable.acceleratorText: Ctrl+S",
    "xLogin*selectionBox*highlightColor: red",
    "xLogin*selectionField.foreground: black",
    "xLogin*selectionField.background: grey70",
    0
};





/* ----------------------------------------------------------------------------
 * PExit - terminate parent process
 * ------------------------------------------------------------------------- */

void PExit(int signo)
{
	fprintf(stderr, "PExit: Terminating parent process due to signal %d.\n",
	        signo);

	XtUnrealizeWidget(app_shell);
	XtAppPending(app_context);
	XCloseDisplay(display);

	exit(0);
}





/* ----------------------------------------------------------------------------
 * suffixcmp - compare suffices
 * ------------------------------------------------------------------------- */

int suffixcmp(char *string, char *suffix, int sep)
{
   char	*pos;

   if (string == NULL || suffix == NULL || sep == 0)
      return(FALSE);
   if (strlen(string) < strlen(suffix))
      return(FALSE);
   if ((pos = strrchr(string, sep)) == NULL)
      return(FALSE);
   if (strcmp(++pos, suffix) != 0)
      return(FALSE);
      
   return(TRUE);
}





/* ----------------------------------------------------------------------------
 * initDatabases - initialize data bases
 * ------------------------------------------------------------------------- */

void initDatabases()
{
   DIR		*dirp;
   FILE 	*filep;
   mode_t	mode;
   char 	buffer[MAX_CHARS];
   int		i, l;

   umask(S_IWGRP | S_IWOTH);

   username    = (char *) malloc(sizeof(char) * MAX_CHARS);
   inetAddress = (char *) malloc(sizeof(char) * MAX_CHARS);

   /* ----------------------------------------------- get name of local host */

   hostname = getLHostName(hostname);

   /* --------------------------------------- set-up terminal/session tables */

   i = 0;

   while (defTerminals[i] != NULL && i < MAX_ITEMS) {
         l = (int) strlen(defTerminals[i]);
         tTable.name[i] = (char *) malloc(sizeof(char)*(l+1));
         strcpy(tTable.name[i], defTerminals[i]);
         i++;
   }

   if (i < MAX_ITEMS)
      tTable.name[i] = NULL;

   i = 0;

   while (defSessions[i].name != NULL && i < MAX_ITEMS) {
         l = (int) strlen(defSessions[i].name);
         sTable.name[i] = (char *) malloc(sizeof(char)*(l+1));
         strcpy(sTable.name[i], defSessions[i].name);
         sTable.access[i] = defSessions[i].access;
         i++;
   }

   if (i < MAX_ITEMS)
      sTable.name[i] = NULL;

   /* ------------------------------------------------------ check resources */

   (void) strcpy(rc_path, getenv("HOME"));
   (void) strcat(rc_path, RC_PATH);
   (void) strcpy(ex_file, rc_path);
   (void) strcat(ex_file, EX_FILE);

   if ((dirp = opendir(rc_path)) != NULL) {
      (void) closedir(dirp);
   } else {
      mode = S_IRWXU | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
      if (mkdir(rc_path, mode) != 0) {
         perror("ERROR: ");
      }
   }

   if (access(ex_file, F_OK) != 0) {
      if ((filep = fopen(ex_file, "w")) != NULL) {
         fprintf(filep,"%s", xLoginrcText);
         fclose(filep);
      } else {
         perror("ERROR: ");
      }
   }

   (void) strcpy(log_file, rc_path);
   (void) strcat(log_file, LOG_FILE);

   (void) unlink(log_file);
   (void) freopen(log_file, "a", stderr);
   (void) freopen(log_file, "a", stdout);

   setbuf(stderr, NULL);
   setbuf(stdout, NULL);

   (void) strcpy(rc_file,rc_path);
   (void) strcat(rc_file,RC_FILE);

   if(!(filep = fopen(rc_file, "r"))) {
      etcHosts = TRUE;
      (void) strcpy(rc_file,ETC_HOSTS);
      if (!(filep = fopen(rc_file, "r"))) {
         perror("ERROR: ");
         exit(2);
      }
   }

   /* ------------------------------------------------------------ read data */

   l = 0;

   while (host_number < MAX_HOSTS) {
         if (!fgets(buffer, MAX_CHARS, filep)) break;
         
         l++;
         
         if ((buffer[0] != '#') && (buffer[0] != '\n')) {
	    if ((host[host_number] = parseLine(rc_file, buffer, l)) != NULL) {
 
               /* ------------------------------------- update session table */

	       if (host[host_number]->session != NULL) {
	          i = 0;

	          while (i < MAX_ITEMS) {
	                if (sTable.name[i] != NULL && StrCmp(sTable.name[i], 
	                    host[host_number]->session) != 0)
		           i++;
		        else
		           break;
		  }

	          if (i < MAX_ITEMS) {
	             if (sTable.name[i] == NULL) {

	                /* new entry */
	             
	                sTable.name[i] = host[host_number]->session;
	                if (host[host_number]->access == UNDEF)
	                   sTable.access[i] = TRUE;
	                else
	                   sTable.access[i] = host[host_number]->access;
	             } else {
	             
	                /* old entry */

	                if (sTable.access[i] != host[host_number]->access &&
	                    host[host_number]->access != UNDEF)
	                   sTable.access[i] = TRUE;
	             }
	       
	             host[host_number]->sn = i;
	          } else {
	             host[host_number]->sn = UNDEF;
	             fprintf(stderr, "initDatabases: ");
	             fprintf(stderr, "ITEM OVERFLOW in session menu -");
	             fprintf(stderr, " IGNORED.\n");
	          }
	       } else
	          host[host_number]->sn = UNDEF;

               /* ------------------------------------ update terminal table */

	       if (host[host_number]->terminal != NULL) {
	          i = 0;

	          while (i < MAX_ITEMS) {
	                if (tTable.name[i] != NULL && StrCmp(tTable.name[i], 
	                    host[host_number]->terminal) != 0)
		           i++;
		        else
		           break;
		  }
	    
	          if (i < MAX_ITEMS) {
	             if (tTable.name[i] == NULL)
	                tTable.name[i] = host[host_number]->terminal;
	    
	             host[host_number]->tn = i;
	          } else {
	             host[host_number]->tn = UNDEF;
	             fprintf(stderr, "initDatabases: ");
	             fprintf(stderr, "ITEM OVERFLOW in terminal menu -");
	             fprintf(stderr, " IGNORED.\n");
	          }
	       } else
	          host[host_number]->tn = UNDEF;

               host_number++;
            }
         }
   }

   /* ------------------------------------------- handle host table overflow */

   while (host_number >= MAX_HOSTS && fgets(buffer, MAX_CHARS, filep)) {
      l++;
      fprintf(stderr, "%s: HOST TABLE OVERFLOW - ignoring line %d: %s",
              rc_file, l, buffer);
   }

   /* --------------------------- initialize default host entry if necessary */

   if (host_number == 0) {
      fprintf(stderr, "initDatabases: ");
      fprintf(stderr, "NO HOST ENTRY FOUND -");
      fprintf(stderr, " using default host instead.\n");

      host[host_number] = (struct host_data *) malloc(sizeof(struct host_data));
      initHostData(host[host_number]);
      host[host_number]->access  = UNDEF;
      host[host_number]->userreq = FALSE;
      host[host_number]->addr = (char *) malloc(sizeof(char) * 10);
      strcpy(host[host_number]->addr, "127.0.0.1");
      host[host_number]->alias = (char *) malloc(sizeof(char) * 10);
      strcpy(host[host_number]->alias, "localhost");
      host[host_number]->tOptions[0] = (char *) malloc(sizeof(char) * 3);
      strcpy(host[host_number]->tOptions[0], "-e");

      host_number++;
   }

   (void) fclose(filep);
}





/* ----------------------------------------------------------------------------
 * closeCB - close message form callback
 * ------------------------------------------------------------------------- */

void closeCB(Widget w, int client_data, caddr_t call_data)
{
    XtDestroyWidget(XtParent(w));
}





/* ----------------------------------------------------------------------------
 * CreateMessage - create a message form
 * ------------------------------------------------------------------------- */

void CreateMessage(Widget parent, char *message, char *title,
                   int rows, int columns, int line)
{
   Arg			args[20];
   register int		n;
   Widget		close_button;
   Widget		message_box;
   Widget		message_text;
   Widget		separator_line;
   XmString		title_string;
   XmString		button_string;
   Position		X, Y, x;
   static Position	y;
   Dimension		W, H, w, h;
 
   /* --------------------------------------------------- create message_box */

   button_string  = XmStringCreateLtoR("Close", XmSTRING_DEFAULT_CHARSET);
   title_string   = XmStringCreateLtoR(title, XmSTRING_DEFAULT_CHARSET);
 
   n = 0;
   XtSetArg (args[n], XmNdialogTitle, title_string);  n++;
   XtSetArg (args[n], XmNdefaultPosition, False);  n++;
   message_box = XmCreateFormDialog(XtParent(parent), "messageForm", args, n);

   /* ------------------------------------------- create 'Close' push button */

   n = 0;
   XtSetArg(args[n], XmNhighlightThickness, 0);  n++;
   XtSetArg(args[n], XmNlabelString, button_string); n++;
   XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNtopAttachment, XmATTACH_NONE);  n++;
   XtSetArg(args[n], XmNbottomAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNleftOffset, 5);  n++;
   XtSetArg(args[n], XmNrightOffset, 5);  n++;
   XtSetArg(args[n], XmNtopOffset, 5);  n++;
   XtSetArg(args[n], XmNbottomOffset, 5);  n++;
   close_button = XmCreatePushButtonGadget(message_box, "close", args, n);
   XtAddCallback(close_button, XmNactivateCallback, (XtCallbackProc)closeCB,
                 NULL);
   XtManageChild(close_button);

   n = 0;
   XtSetArg(args[n], XmNdefaultButton, close_button);  n++;
   XtSetValues(message_box, args, n);

   /* ------------------------------------------------ create separator line */

   n = 0;
   XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNtopAttachment, XmATTACH_NONE);  n++;
   XtSetArg(args[n], XmNbottomAttachment, XmATTACH_WIDGET);  n++;
   XtSetArg(args[n], XmNbottomWidget, close_button);  n++;
   XtSetArg(args[n], XmNleftOffset, 5);  n++;
   XtSetArg(args[n], XmNrightOffset, 5);  n++;
   XtSetArg(args[n], XmNtopOffset, 5);  n++;
   XtSetArg(args[n], XmNbottomOffset, 5);  n++;
   separator_line = XmCreateSeparator(message_box, "separator", args, n);

   XtManageChild(separator_line);
   
   /* ------------------------------------------------- create scrolled text */

   n = 0;
   XtSetArg(args[n], XmNrows, rows);  n++;
   XtSetArg(args[n], XmNcolumns, columns);  n++;
   XtSetArg(args[n], XmNtraversalOn, False); n++;

   if (columns >= line) {
      XtSetArg(args[n], XmNscrollHorizontal, False);  n++;
      XtSetArg(args[n], XmNwordWrap, True);  n++;
   } else {
      XtSetArg(args[n], XmNscrollHorizontal, True);  n++;
      XtSetArg(args[n], XmNwordWrap, False);  n++;
   }

   XtSetArg(args[n], XmNscrollVertical, True);  n++;
   XtSetArg(args[n], XmNeditable, False);  n++;
   XtSetArg(args[n], XmNeditMode, XmMULTI_LINE_EDIT);  n++;
   XtSetArg(args[n], XmNvalue, message);  n++;
   XtSetArg(args[n], XmNleftAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNrightAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNtopAttachment, XmATTACH_FORM);  n++;
   XtSetArg(args[n], XmNbottomAttachment, XmATTACH_WIDGET);  n++;
   XtSetArg(args[n], XmNbottomWidget, separator_line);  n++;
   XtSetArg(args[n], XmNleftOffset, 5);  n++;
   XtSetArg(args[n], XmNrightOffset, 5);  n++;
   XtSetArg(args[n], XmNtopOffset, 5);  n++;
   XtSetArg(args[n], XmNbottomOffset, 5);  n++;
   message_text = XmCreateScrolledText(message_box, "messageText", args, n);

   XtManageChild(message_text);

   XmStringFree(title_string);
   XmStringFree(button_string);

   /* -------------------------------- Set trough color of scrollbar widgets */

   if (DefaultDepthOfScreen(XDefaultScreenOfDisplay(display)) > 1)
   {
	Widget		hsbar, vsbar;
	Pixel		fg, bg;

	XtVaGetValues(message_text,
	              XmNforeground, &fg,
	              XmNbackground, &bg,
	              NULL);

	XtVaGetValues(XtParent(message_text),
	              XmNverticalScrollBar, &vsbar,
	              NULL);

	XtVaSetValues(vsbar,
	              XmNtroughColor, bg,
	              NULL);

	if (columns < line) {
	   XtVaGetValues(XtParent(message_text),
	                 XmNhorizontalScrollBar, &hsbar,
	                 NULL);

	   XtVaSetValues(hsbar,
	                 XmNtroughColor, bg,
	                 NULL);
	}
   }

   /* ---------------------------------------------------- place message box */

   XtVaSetValues(XtParent(message_box), XmNmappedWhenManaged, False, NULL);
   XtManageChild(message_box);

   XtVaGetValues(app_shell,
                 XmNx, &X,
                 XmNy, &Y,
                 XmNwidth, &W,
                 XmNheight, &H,
                 NULL);

   XtVaGetValues(message_box,
                 XmNwidth, &w,
                 XmNheight, &h,
                 NULL);

   x = X+W/2-w/2;                   /* center the message box in x direction */
   y += H/20;                       /* continuously increment the y position */

   /* message box off screen ? */

   if (x < 0)
      x = W/5;
   if (x+w > WidthOfScreen(XtScreen(message_box)))
      x = WidthOfScreen(XtScreen(message_box)) - w - W/5;
   
   /* message box off toplevel shell ? */
 
   if ((y < Y + H/10) || (y + h > Y+H-H/10))
      y = Y + H/10;
   
   XtVaSetValues(message_box,
                 XmNx, x,
                 XmNy, y,
                 NULL);

   XtMapWidget(XtParent(message_box));
   XtVaSetValues(XtParent(message_box), XmNmappedWhenManaged, True, NULL);
}





/* ----------------------------------------------------------------------------
 * LogCB - display log file callback
 * ------------------------------------------------------------------------- */

void LogCB(Widget parent)
{
   int			l;
   char			*log, line[MAX_CHARS];
   struct stat		filestat;
   FILE			*filep;

   /* -------------------------------------------------------- open log file */

   if ((filep = fopen(log_file, "r")) != NULL) {

      if (stat(log_file, &filestat) == 0) {
         l = filestat.st_size;
         log = (char *) malloc(sizeof(char) * (l+1));

         strcpy(log, "");

         /* -------------------------------------------------- read log file */

         while (fgets(line, MAX_CHARS, filep) != NULL) {
               strcat(log, line);
         }

         CreateMessage(parent, log, "Log File", 10, 75, MAX_CHARS);
         free(log);
      } else {
         perror("ERROR: ");
      }
      
      fclose(filep);
   } else {
      perror("ERROR: ");
   }
}





/* ----------------------------------------------------------------------------
 * ReportCB - display connection report callback
 * ------------------------------------------------------------------------- */

void ReportCB(Widget parent)
{
   int			i,l,n;
   char			filename[MAX_CHARS];
   struct dirent	*content;
   DIR			*dirp;
   FILE			*filep;
   char			addr[MAX_CHARS];	/* host address           */
   char			*name;			/* host name              */
   int			no;			/* number of connections  */
   char			access[MAX_CHARS];	/* server access          */
   char			*listing;		/* connection report list */
   XHostData		*xhosts;		/* xhosts table           */
   int			l_stat;			/* log file status        */
   int			x_stat;			/* xhost table status     */
   char			syncstat[MAX_CHARS];	/* synchronization status */

   /* ----------------------------------------------------------- title line */

   listing = (char *) malloc(sizeof(char) * 2 * MAX_CHARS);
   strcpy(listing, "   Host Address  ");
   strcat(listing, "      Host Name  ");
   strcat(listing, "Connections  ");
   strcat(listing, "Server Access    ");
   strcat(listing, "     Status\n");
   strcat(listing, "-----------------");
   strcat(listing, "-----------------");
   strcat(listing, "-------------");
   strcat(listing, "-----------------");
   strcat(listing, "-----------\n");

   /* ---------------------------------------------------- query XHost table */

   xhosts = getXHosts(display, &n);

   /* ----------------------------------------- scan ~/.xLogin for log files */

   dirp = opendir(rc_path);

   i = 1;
   while (dirp != NULL && (content = readdir(dirp)) != NULL) {
         if ((strcmp(content->d_name, ".") != 0) && 
             (strcmp(content->d_name, "..") != 0) &&
             (content->d_name[0] == '.') &&
             suffixcmp(content->d_name, hostname, '@')) {

	    /* ---------------------------------------- extract host address */
             
            (void) strcpy(filename, content->d_name);
            l = strlen(filename);
            StrNCpy(addr, &filename[1], l-2-strlen(hostname));

            /* -------------------------------------- get official host name */

	    name = getRHostName(addr);

            /* ---------------------------------- read number of connections */

            strcpy(filename, rc_path);
            strcat(filename, "/");
            strcat(filename, content->d_name);
            filep = fopen(filename, "r+");
            fscanf(filep, "%d\n", &no);
            fclose(filep);

            /* ------------------------------------ get server access status */

            switch (StatXHost(addr, rc_path)) {
                   case AUTOMATIC: strcpy(access, "automatic");
                                   l_stat = TRUE;
                                   break;
                   case ENABLE   : strcpy(access, "enabled");
                                   l_stat = TRUE;
                                   break;
                   case DISABLE  : strcpy(access, "disabled");
                                   l_stat = FALSE;
                                   break;
                   default       : fprintf(stderr, "report: UNEXPECTED TAG\n");
            }

            /* ---------------------------------- get synchronization status */

            x_stat = FALSE;

            for (l=0; l<n; l++) {
                 if (StrCmp(addr, xhosts[l].addr) == 0) {
                    removeXHost(xhosts, l, n);
                    x_stat = TRUE;
                    break;
                 }
            }

            if ((l_stat && x_stat) || (! l_stat && ! x_stat)) {
               strcpy(syncstat, "synchronous");
            } else {
               strcpy(syncstat, "asynchronous");
            }

            /* ---------------------------------------- extend report string */

	    listing = (char *) realloc(listing,
	                               sizeof(char) * (2+i) * MAX_CHARS);

	    sprintf(filename, "%15s  %15s  %9d  %15s  %13s\n",
	            addr, name, no, access, syncstat);
	    strcat(listing, filename);

	    free(name);
	    i++;
         }
   }

   /* ----------------------------- continue with unconsidered xhost entries */

   l = 0;
   while (l < n && xhosts[l].addr != NULL) {

         /* ----------------------------------------- get official host name */

	 name = getRHostName(xhosts[l].addr);

         /* -------------------------- set access and synchronization status */

         strcpy(access, "enabled");
         strcpy(syncstat, "unmanaged");

         /* ------------------------------------------- extend report string */

	 listing = (char *) realloc(listing,
	                            sizeof(char) * (2+i) * MAX_CHARS);

	 sprintf(filename, "%15s  %15s         --  %15s  %13s\n",
	         xhosts[l].addr, name, access, syncstat);
	 strcat(listing, filename);

	 free(name);
         l++;
   }

   CreateMessage(parent, listing, "Connection Report", 10, 75, 75);
   free(listing);
   freeXHosts(xhosts, n);
}





/* ----------------------------------------------------------------------------
 * SyncLogFilesCB - synchronize (clean) logging files with current xhost table
 * ------------------------------------------------------------------------- */

void SyncLogFilesCB(Widget parent)
{
   Bool			status;			/* access control status  */
   int			l, n;
   char			filename[MAX_CHARS];
   struct dirent	*content;
   DIR			*dirp;
   FILE			*filep;
   char			l_addr[MAX_CHARS];	/* log file host address  */
   int			l_no;			/* number of connections  */
   int			l_stat;			/* log file status        */
   XHostData		*xhosts;		/* xhosts table           */
   int			x_stat;			/* xhost table status     */

   /* ------------------------------------------------ enable access control */

   XEnableAccessControl(display);
   fprintf(stderr, "SyncLogFilesCB: X server access control enabled\n");

   /* ---------------------------------------------------- query XHost table */

   xhosts = getXHosts(display, &n);

   /* ----------------------------------------- scan ~/.xLogin for log files */

   dirp = opendir(rc_path);

   while (dirp != NULL && (content = readdir(dirp)) != NULL) {
         if ((strcmp(content->d_name, ".") != 0) && 
             (strcmp(content->d_name, "..") != 0) &&
             (content->d_name[0] == '.') &&
             suffixcmp(content->d_name, hostname, '@')) {

	    /* ---------------------------------------- extract host address */
             
            (void) strcpy(filename, content->d_name);
            l = strlen(filename);
            StrNCpy(l_addr, &filename[1], l-2-strlen(hostname));

            /* ---------------------------------- read number of connections */

            strcpy(filename, rc_path);
            strcat(filename, "/");
            strcat(filename, content->d_name);
            filep = fopen(filename, "r+");
            fscanf(filep, "%d\n", &l_no);
            fclose(filep);

            /* ----------------------------------------- get log file status */

            l_stat = StatXHost(l_addr, rc_path);

            if (l_stat == ENABLE || (l_stat == AUTOMATIC && l_no > 0)) {
               l_stat = TRUE;
            } else {
               l_stat = FALSE;
            }

            /* ------------------------ get corresponding xhost table status */

            x_stat = FALSE;

            for (l=0; l<n; l++) {
                 if (strcmp(l_addr, xhosts[l].addr) == 0) {
                    x_stat = TRUE;
                    break;
                 }
            }

            /* ---------------------------------------------- clean log file */

            if ((l_stat == TRUE) && (x_stat == FALSE)) {
               fprintf(stderr,
                       "SyncLogFilesCB: asynchronous status of %s detected\n",
                       l_addr);
               fprintf(stderr,
                       "SyncLogFilesCB: removing log file '%s'\n",
                       filename);
               (void) unlink(filename);
            }
         }
   }

   freeXHosts(xhosts, n);
}





/* ----------------------------------------------------------------------------
 * SyncXHostTableCB - synchronize xhost table  with logging files
 * ------------------------------------------------------------------------- */

void SyncXHostTableCB(Widget parent)
{
   int			l, n;
   char			filename[MAX_CHARS];
   struct dirent	*content;
   DIR			*dirp;
   FILE			*filep;
   char			l_addr[MAX_CHARS];	/* log file host address  */
   int			l_no;			/* number of connections  */
   int			l_stat;			/* log file status        */
   XHostData		*xhosts;		/* xhosts table           */

   /* ------------------------------------------------ enable access control */

   XEnableAccessControl(display);
   fprintf(stderr, "SyncXHostTableCB: X server access control enabled\n");

   /* ---------------------------------------------------- reset XHost table */
   
   xhosts = getXHosts(display, &n);

   fprintf(stderr, "SyncXHostTableCB: resetting xhost table\n");

   for (l=0; l<n; l++) {
       fprintf(stderr,
               "SyncXHostTableCB: ... removing '%s' from xhost table\n",
               xhosts[l].addr);
       updateXHost(xhosts[l].addr, FALSE);
   }

   /* ----------------------------------------- scan ~/.xLogin for log files */

   dirp = opendir(rc_path);

   while (dirp != NULL && (content = readdir(dirp)) != NULL) {
         if ((strcmp(content->d_name, ".") != 0) && 
             (strcmp(content->d_name, "..") != 0) &&
             (content->d_name[0] == '.') &&
             suffixcmp(content->d_name, hostname, '@')) {

	    /* ---------------------------------------- extract host address */
             
            (void) strcpy(filename, content->d_name);
            l = strlen(filename);
            StrNCpy(l_addr, &filename[1], l-2-strlen(hostname));

            /* ---------------------------------- read number of connections */

            strcpy(filename, rc_path);
            strcat(filename, "/");
            strcat(filename, content->d_name);
            filep = fopen(filename, "r+");
            fscanf(filep, "%d\n", &l_no);
            fclose(filep);

            /* ----------------------------------------- get log file status */

            l_stat = StatXHost(l_addr, rc_path);

            /* --------------------------------------------- set xhost table */

            if (l_stat == ENABLE || (l_stat == AUTOMATIC && l_no > 0)) {
               fprintf(stderr,
                       "SyncXHostTableCB: adding '%s' to xhost table\n",
                       l_addr);
               updateXHost(l_addr, TRUE);
            }
         }
   }

   freeXHosts(xhosts, n);
}





/* ----------------------------------------------------------------------------
 * MenuCB - callback from PushButtons in PulldownMenus
 * ------------------------------------------------------------------------- */

void MenuCB (Widget w, XtPointer client_data, XtPointer call_data) 
{
	switch ((int) client_data) {
		case MENU_EXIT: 
		        PExit(0);
		        break;
		case MENU_LOG: 
		        LogCB(w);
		        break;
		case MENU_REPORT: 
		        ReportCB(w);
		        break;
		case MENU_SYNC_LOGFILES: 
		        SyncLogFilesCB(w);
		        break;
		case MENU_SYNC_XHOSTTABLE: 
		        SyncXHostTableCB(w);
		        break;
		default:
			printf("MenuCB: UNEXPECTED TAG.\n");
			break;
	}

}






/* ----------------------------------------------------------------------------
 * HelpCB - callback for help button
 * ------------------------------------------------------------------------- */

void HelpCB (Widget w, XtPointer client_data, XtPointer call_data) 
{
        int	index = (int) client_data;

        CreateMessage(w, HelpText[index], HelpTitle[index], 15, 75, 75);
}





/* ----------------------------------------------------------------------------
 * UpdateSelectionList - update host table selection list
 * ------------------------------------------------------------------------- */

void UpdateSelectionList(int pos) 
{
	if (pos > 0) {

	   /* -------------------------------------------- select item 'pos' */

	   XmListSelectPos(XmSelectionBoxGetChild(selectionBox,
	                   XmDIALOG_LIST), pos, False);
	} else

	   /* ------------------------------------------- deselect all items */

	   XmListDeselectAllItems(XmSelectionBoxGetChild(selectionBox,
	                          XmDIALOG_LIST));
}





/* ----------------------------------------------------------------------------
 * UpdateSelectionField - update host address selection field text
 * ------------------------------------------------------------------------- */

void UpdateSelectionField(char *text) 
{
	Arg			al[10];
	int			ac;

	/* ---------------------------------------- set selection field text */

	ac = 0;
	XtSetArg (al[ac], XmNvalue, text); ac++;
	XtSetValues(selectionField, al, ac);
}





/* ----------------------------------------------------------------------------
 * UpdateOptionMenu - update terminal and session option menu
 * ------------------------------------------------------------------------- */

void UpdateOptionMenu(int terminal, int session) 
{
	Arg			al[10];
	int			ac;

	/* ---------------------------------------------- update terminal id */

	if (terminal != UNDEF) {
	   tID = terminal;

	   ac = 0;
	   XtSetArg(al[ac], XmNmenuHistory, tTable.button[tID]); ac++;
	   XtSetValues(t_option, al, ac);
	}

	/* ----------------------------------------------- update session id */
	
	if (session != UNDEF) {
	   sID = session;

	   ac = 0;
	   XtSetArg(al[ac], XmNmenuHistory, sTable.button[sID]); ac++;
	   XtSetValues(s_option, al, ac);
	}
}





/* ----------------------------------------------------------------------------
 * UpdateToggleButton - update server access toggle buttons
 * ------------------------------------------------------------------------- */

void UpdateToggleButton(int access)
{
	switch(access) {
	      case AUTOMATIC: XmToggleButtonSetState(auto_toggle, 
	                                             True, False);
	                      XmToggleButtonSetState(enable_toggle, 
	                                             False, False);
	                      XmToggleButtonSetState(disable_toggle, 
	                                             False, False);
	                      break;
	      case ENABLE   : XmToggleButtonSetState(auto_toggle,
	                                             False, False);
	                      XmToggleButtonSetState(enable_toggle, 
	                                             True, False);
	                      XmToggleButtonSetState(disable_toggle, 
	                                             False, False);
	                      break;
	      case DISABLE  : XmToggleButtonSetState(auto_toggle,
	                                             False, False);
	                      XmToggleButtonSetState(enable_toggle, 
	                                             False, False);
	                      XmToggleButtonSetState(disable_toggle, 
	                                             True, False);
	                      break;
	      default       : fprintf(stderr, 
	                      "UpdateToggleButton: UNEXPECTED TAG\n");
	}
}





/* ----------------------------------------------------------------------------
 * TerminalCB - callback for terminal option button
 * ------------------------------------------------------------------------- */

void TerminalCB (Widget w, XtPointer client_data, XtPointer call_data) 
{
	tID = (int) client_data;
}





/* ----------------------------------------------------------------------------
 * SessionCB - callback for session option button
 * ------------------------------------------------------------------------- */

void SessionCB (Widget w, XtPointer client_data, XtPointer call_data) 
{
	sID = (int) client_data;
}





/* ----------------------------------------------------------------------------
 * UserCB - callback for user dialg
 * ------------------------------------------------------------------------- */

void UserCB (Widget w, XtPointer client_data, XtPointer call_data) 
{
	char *text;
	int *status             = (int *) client_data;
	XmCommandCallbackStruct *cb = (XmCommandCallbackStruct *) call_data;

	if (cb->reason == XmCR_OK) {
	   *status = DIALOG_OK;

	   XmStringGetLtoR(cb->value, XmFONTLIST_DEFAULT_TAG, &text);

	   if (text && *text)
	      (void) strcpy(username, text);
	   else
	      (void) strcpy(username, getenv("LOGNAME"));

	   XtFree(text);

	} else
	   *status = DIALOG_CANCEL;
}





/* ----------------------------------------------------------------------------
 * SelectionCB - Process callback from Buttons in SelectionBox
 * ------------------------------------------------------------------------- */

void SelectionCB (Widget w, XtPointer client_data, XtPointer call_data) 
{
	Arg			al[10];
	int			ac = 0;
	XmString		label, title;
	char			text[MAX_CHARS];
	Widget			promptDialog;
	Widget			button;
	int			status = DIALOG_OK;
	int			currAccess;
	char			*selectionFieldText;
        XmListCallbackStruct    *cb = (XmListCallbackStruct *) call_data;

	switch ((int) client_data) {

		case DIALOG_STARTTEXT:

		     /* ------------------------- deselect entire host table */

		     UpdateSelectionList(0);
		     selection = FALSE;
		     otherHost = TRUE;

		     break;

		case DIALOG_ENDTEXT:

		     /* ---------------------------- read selection field */

		     selectionFieldText = XmTextGetString(selectionField);
		     (void) strcpy(inetAddress, selectionFieldText);
		     XtFree(selectionFieldText);

		     /* ------- update selection field and toggle buttons */

		     if (substAddr(&inetAddress) < 0) {
	                fprintf(stderr, "SelectionCB: ");
	                fprintf(stderr, "CANNOT RESOLVE HOST ADDRESS '%s'",
	                        inetAddress);
		        inetAddress = getLHostName(inetAddress);
		        (void) substAddr(&inetAddress);
	                fprintf(stderr, " - USING LOCAL HOST ADDRESS '%s'",
	                   	inetAddress);
	                fprintf(stderr, " INSTEAD.\n");
	             }

		     UpdateSelectionField(inetAddress);
		     UpdateToggleButton(StatXHost(inetAddress, rc_path));
		     
		     otherHost = FALSE;
		     break;

		case DIALOG_LIST:

		     /* ---------------------------------- get selected host */

		     hIndex = cb->item_position - 1;
		     (void) strcpy(inetAddress, host[hIndex]->addr);

		     /* update selection field, option menus, toggle buttons */

		     if (substAddr(&inetAddress) < 0) {
	                fprintf(stderr, "SelectionCB: ");
	                fprintf(stderr, "CANNOT RESOLVE HOST ADDRESS '%s'",
	                        inetAddress);
		        inetAddress = getLHostName(inetAddress);
		        (void) substAddr(&inetAddress);
	                fprintf(stderr, " - USING LOCAL HOST ADDRESS '%s'",
	                   	inetAddress);
	                fprintf(stderr, " INSTEAD.\n");
	             }

		     UpdateSelectionField(inetAddress);
		     UpdateOptionMenu(host[hIndex]->tn, host[hIndex]->sn);
		     UpdateToggleButton(StatXHost(inetAddress, rc_path));
		     selection = TRUE;

		     break;

		case DIALOG_LOGIN:

	             /* ----- make sure to handle a carriage return properly */

	             if (otherHost == TRUE)
	                SelectionCB(w, (XtPointer) DIALOG_ENDTEXT, NULL);

		     /* ---------------------------------- set access status */

		     if (StrCmp(sTable.name[sID], host[hIndex]->session) == 0 &&
		         selection == TRUE && host[hIndex]->access != UNDEF)
		        currAccess = host[hIndex]->access;
		     else
		        currAccess = sTable.access[sID]; /* default */

		     /* -------------------------------------- get user name */
		
		     if (host[hIndex]->userreq && selection == TRUE &&
		         StrCmp(sTable.name[sID], host[hIndex]->session) == 0) {
			(void) strcpy(text, "Enter username:");
			label = XmStringCreateLtoR(text,
                                XmSTRING_DEFAULT_CHARSET);
			(void) strcpy(text, "xLogin dialog");
			title = XmStringCreateLtoR(text,
                                XmSTRING_DEFAULT_CHARSET);

			ac = 0;
        		XtSetArg(al[ac], XmNdialogTitle, title);  ac++;
			XtSetArg(al[ac], XmNselectionLabelString, label); ac++;
			XtSetArg(al[ac], XmNdialogStyle,
			           XmDIALOG_FULL_APPLICATION_MODAL); ac++;
			promptDialog = XmCreatePromptDialog(w, "promptDialog",
			               al, ac);

			XmStringFree(label);
			XmStringFree(title);

			XtAddCallback(promptDialog, XmNokCallback, UserCB, 
			              &status);
			XtAddCallback(promptDialog, XmNcancelCallback, UserCB, 
			              &status);

			button = XmSelectionBoxGetChild(promptDialog,
			         XmDIALOG_HELP_BUTTON);
			XtUnmanageChild(button);

			XtManageChild(promptDialog);

			status = DIALOG_WAIT;

			while (status == DIALOG_WAIT)
			      XtAppProcessEvent(app_context, XtIMAll);

			XtDestroyWidget(promptDialog);
		     }

		     if (status == DIALOG_OK)
			(void) Connect(inetAddress, host[hIndex],
			               tTable.name[tID], sTable.name[sID],
			               username, rc_path, currAccess,
			               selection);

		     break;

		case DIALOG_ADDHOST:
		     if (firstTime) {
			firstTime = FALSE;
		     } else {
			firstTime = TRUE;
			(void) ForceXHost(inetAddress, rc_path, ADD, LOCK);
		     }
		     break;

		case DIALOG_DELHOST:
		     if (firstTime) {
			firstTime = FALSE;
		     } else {
			firstTime = TRUE;
			(void) ForceXHost(inetAddress, rc_path, DEL, LOCK);
		     }
		     break;

		case DIALOG_RELEASE:
		     if (firstTime) {
			firstTime = FALSE;
		     } else {
			firstTime = TRUE;
			(void) ForceXHost(inetAddress, rc_path, 0, UNLOCK);
		     }
		     break;

		default:
		     fprintf(stderr,"SelectionCB: UNEXPECTED TAG.\n");
		     break;
	}
}





/* ----------------------------------------------------------------------------
 * CreateMenuBar - Create MenuBar in MainWindow
 * ------------------------------------------------------------------------- */

Widget CreateMenuBar (Widget parent)
{
	Widget		menuBar;
	Widget		cascade;
	Widget		fileMenu;
	Widget		syncMenu;
	Widget		helpMenu;
	Widget		button;
	Widget		separator_line;
	Arg		al[10];
	register int	ac;
	int		i;
        XmString        label;


	/* -------------------------------------------------- Create MenuBar */

	ac = 0;
	menuBar = XmCreateMenuBar(parent, "menuBar", al, ac);

	/* -------------------------------------- Create "File" PulldownMenu */

	ac = 0;
	fileMenu = XmCreatePulldownMenu(menuBar, "fileMenu", al, ac);

	ac = 0;
	button = XmCreatePushButton(fileMenu, "Log", al, ac);
	XtAddCallback(button, XmNactivateCallback, MenuCB, (XtPointer)MENU_LOG);
	XtManageChild(button);

	ac = 0;
	button = XmCreatePushButton(fileMenu, "Report", al, ac);
	XtAddCallback(button, XmNactivateCallback, MenuCB, (XtPointer)MENU_REPORT);
	XtManageChild(button);

	ac = 0;
	separator_line = XmCreateSeparator(fileMenu, "menuSeparator", al, ac);

	XtManageChild(separator_line);

	ac = 0;
	button = XmCreatePushButton(fileMenu, "Close", al, ac);
	XtAddCallback(button, XmNactivateCallback, MenuCB, (XtPointer)MENU_EXIT);
	XtManageChild(button);

	ac = 0;
	XtSetArg(al[ac], XmNsubMenuId, fileMenu);  ac++;
	cascade = XmCreateCascadeButton(menuBar, "File", al, ac);
	XtManageChild(cascade);

	/* -------------------------------------- Create "Sync" PulldownMenu */

	ac = 0;
	syncMenu = XmCreatePulldownMenu(menuBar, "syncMenu", al, ac);

	label = XmStringCreateLtoR((char *) "... log files", 
	        XmSTRING_DEFAULT_CHARSET);

	ac = 0;
	XtSetArg(al[ac], XmNlabelString, label);  ac++;
	button = XmCreatePushButton(syncMenu, "LogFiles", al, ac);
	XtAddCallback(button, XmNactivateCallback, MenuCB,
		     (XtPointer)MENU_SYNC_LOGFILES);
	XtManageChild(button);

	label = XmStringCreateLtoR((char *) "... xhost table", 
	        XmSTRING_DEFAULT_CHARSET);

	ac = 0;
	XtSetArg(al[ac], XmNlabelString, label);  ac++;
	button = XmCreatePushButton(syncMenu, "XHostTable", al, ac);
	XtAddCallback(button, XmNactivateCallback, MenuCB,
		     (XtPointer)MENU_SYNC_XHOSTTABLE);
	XtManageChild(button);

	ac = 0;
	XtSetArg(al[ac], XmNsubMenuId, syncMenu);  ac++;
	cascade = XmCreateCascadeButton(menuBar, "Sync", al, ac);
	XtManageChild(cascade);

	/* -------------------------------------- Create "Help" PulldownMenu */

	ac = 0;
	helpMenu = XmCreatePulldownMenu(menuBar, "helpMenu", al, ac);

	for (i=0; i<NUMHELP; i++) {
	    label = XmStringCreateLtoR((char *) HelpTitle[i], 
	            XmSTRING_DEFAULT_CHARSET);

	    ac = 0;
	    XtSetArg(al[ac], XmNlabelString, label);  ac++;
	    button = XmCreatePushButton(helpMenu, "helpText", al, ac);
	    XtAddCallback(button, XmNactivateCallback, HelpCB, (XtPointer) i);
	    XtManageChild(button);
            XmStringFree(label);
	}
 
	ac = 0;
	XtSetArg(al[ac], XmNsubMenuId, helpMenu);  ac++;
	cascade = XmCreateCascadeButton(menuBar, "Help", al, ac);
	XtManageChild(cascade);
 
	ac = 0;
	XtSetArg(al[ac], XmNmenuHelpWidget, cascade);  ac++;
	XtSetValues(menuBar, al, ac);

	return(menuBar);
}





/* ----------------------------------------------------------------------------
 * CreateSelectionBox - Create top level SelectionBox
 * ------------------------------------------------------------------------- */

Widget CreateSelectionBox(Widget parent)
{
	Widget		list;
	Widget		text;
	Widget		kid[10];
	Arg		al[10];
	register int	ac;
	register int	i;
	XmString	list_item[MAX_HOSTS];
	XmStringCharSet	charset = (XmStringCharSet) XmSTRING_DEFAULT_CHARSET;
	Widget		hsbar, vsbar;
	XrmValue	pixel_data;
	Pixel		bg, fg;

	/* ------------------------------------------- Set up items for List */

	for ( i = 0;  i < host_number;  i++ )
		list_item[i] = XmStringCreateLtoR(host[i]->alias, charset);

	/* ------------------------------------ Create toplevel SelectionBox */

	ac = 0;
	XtSetArg(al[ac], XmNshadowThickness, 1);  ac++;
	XtSetArg(al[ac], XmNshadowType, XmSHADOW_OUT);  ac++;
	XtSetArg(al[ac], XmNtextString, list_item[0]);  ac++;
	XtSetArg(al[ac], XmNlistItems, list_item);  ac++;
	XtSetArg(al[ac], XmNlistItemCount, host_number);  ac++;
	XtSetArg(al[ac], XmNlistLabelString, 
			 XmStringCreateLtoR ("Host Table", charset));  ac++;
	XtSetArg(al[ac], XmNokLabelString,
	                 XmStringCreateLtoR ("Connect ...", 
	                 XmSTRING_DEFAULT_CHARSET));  ac++;
	selectionBox = XmCreateSelectionBox(parent, "selectionBox", al, ac);
	XtAddCallback(selectionBox, XmNokCallback, SelectionCB,
		     (XtPointer) DIALOG_LOGIN);

	/* ------------------------ Register callbacks for SelectionBox list */

	list = XmSelectionBoxGetChild(selectionBox, XmDIALOG_LIST);
	XtAddCallback(list, XmNbrowseSelectionCallback, SelectionCB,
	             (XtPointer) DIALOG_LIST);

	/* -------------------------------------- Unmanage unneeded children */

	i = 0;
	kid[i++] = XmSelectionBoxGetChild(selectionBox, XmDIALOG_SEPARATOR);
	kid[i++] = XmSelectionBoxGetChild(selectionBox, XmDIALOG_CANCEL_BUTTON);
	kid[i++] = XmSelectionBoxGetChild(selectionBox, XmDIALOG_APPLY_BUTTON);
	kid[i++] = XmSelectionBoxGetChild(selectionBox, XmDIALOG_HELP_BUTTON);
	kid[i++] = XmSelectionBoxGetChild(selectionBox, XmDIALOG_TEXT);
	kid[i++] = XmSelectionBoxGetChild(selectionBox, XmDIALOG_SELECTION_LABEL);
	XtUnmanageChildren(kid, i);

	return(selectionBox);
}





/* ----------------------------------------------------------------------------
 * EnhanceSelectionBox - enhance selection box
 * ------------------------------------------------------------------------- */

Widget EnhanceSelectionBox(Widget parent)
{
	Widget		box;
	Widget		frame, form;
	Widget		label, selectionLabel;
	Widget		radio_buttn;
	Widget		s_pulldown, t_pulldown;
	XmString	title;
	Arg		al[10];
	register int	ac;
	int		i;

	/* ------------------------------------------- Create outer Form box */

	ac = 0;
	box = XmCreateForm(parent, "work_area", al, ac);
	XtManageChild(box);

	/* ---------------------------- Create separate selection text field */

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_POSITION);  ac++;
	selectionLabel = XmCreateLabel(box, "Host Address", al, ac);
	XtManageChild(selectionLabel);

	ac = 0;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM);  ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_FORM);  ac++;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET);  ac++;
	XtSetArg(al[ac], XmNtopWidget, selectionLabel);  ac++;
	selectionField = XmCreateTextField(box, "selectionField", al, ac);
	XtAddCallback(selectionField, XmNfocusCallback, SelectionCB,
	             (XtPointer) DIALOG_STARTTEXT);
	XtAddCallback(selectionField, XmNlosingFocusCallback, SelectionCB,
	             (XtPointer) DIALOG_ENDTEXT);
	XtManageChild(selectionField);

	/* ---------------------------------- Set colors of recessed widgets */

	if (DefaultDepthOfScreen(XDefaultScreenOfDisplay(display)) > 1)
	{
		Widget		list;
		Widget		text;
		Widget		hsbar, vsbar;
		Pixel		bg, fg;

		list = XmSelectionBoxGetChild(selectionBox, XmDIALOG_LIST);
		text = XmSelectionBoxGetChild(selectionBox, XmDIALOG_TEXT);

		XtVaGetValues(XtParent(list),
		              XmNhorizontalScrollBar, &hsbar,
		              XmNverticalScrollBar, &vsbar,
		              NULL);

		XtVaGetValues(selectionField,
		              XmNforeground, &fg,
		              XmNbackground, &bg,
		              NULL);

		XtVaSetValues(list,
		              XmNforeground, fg,
		              XmNbackground, bg,
		              NULL);

		XtVaSetValues(text,
		              XmNforeground, fg,
		              XmNbackground, bg,
		              NULL);

		XtVaSetValues(hsbar,
		              XmNtroughColor, bg,
		              NULL);

		XtVaSetValues(vsbar,
		              XmNtroughColor, bg,
		              NULL);
	}

	/* ----------------------------------------- Create option menu form */

	ac = 0;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM);  ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_FORM);  ac++;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET);  ac++;
	XtSetArg(al[ac], XmNtopWidget, selectionField);  ac++;
	XtSetArg(al[ac], XmNtopOffset, 15);  ac++;
	form = XmCreateForm(box, "form", al, ac);
	XtManageChild(form);

	/* ---------------------------- Create option menu for terminal type */

	title = XmStringCreateLtoR("Terminal:", XmSTRING_DEFAULT_CHARSET);
 
	ac = 0;
	t_pulldown = XmCreatePulldownMenu(form, "optionPulldown", al, ac);

	for (i = 0; tTable.name[i] != NULL && i < MAX_ITEMS; i++) {
	    ac = 0;
	    tTable.button[i] = XmCreatePushButton(t_pulldown, tTable.name[i],
	                                          al, ac);
	    XtAddCallback(tTable.button[i], XmNactivateCallback, TerminalCB,
	                 (XtPointer) i);
	    XtManageChild(tTable.button[i]);
	}

	ac = 0;
	XtSetArg(al[ac], XmNlabelString, title);  ac++;
	XtSetArg(al[ac], XmNsubMenuId, t_pulldown); ac++;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_POSITION);  ac++;
	XtSetArg(al[ac], XmNtopPosition, 5);  ac++;
	t_option = XmCreateOptionMenu(form, "option", al, ac);
	XmStringFree(title);

	XtManageChild(t_option);

	/* ----------------------------- Create option menu for session type */

	title = XmStringCreateLtoR("Session :", XmSTRING_DEFAULT_CHARSET);
 
	ac = 0;
	s_pulldown = XmCreatePulldownMenu(form, "optionPulldown", al, ac);

	for (i = 0; sTable.name[i] != NULL && i < MAX_ITEMS; i++) {
	    ac = 0;
	    sTable.button[i] = XmCreatePushButton(s_pulldown, sTable.name[i], 
	                                          al, ac);
	    XtAddCallback(sTable.button[i], XmNactivateCallback, SessionCB,
	                 (XtPointer) i);
	    XtManageChild(sTable.button[i]);
	}

	ac = 0;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET);  ac++;
	XtSetArg(al[ac], XmNtopWidget, t_option);  ac++;
	XtSetArg(al[ac], XmNlabelString, title);  ac++;
	XtSetArg(al[ac], XmNsubMenuId, s_pulldown); ac++;
	s_option = XmCreateOptionMenu(form, "option", al, ac);
	XmStringFree(title);

	XtManageChild(s_option);

	/* - Create RadioBox and dialog style toggles for server access type */

	ac = 0;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM);  ac++;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET);  ac++;
	XtSetArg(al[ac], XmNtopWidget, form);  ac++;
	XtSetArg(al[ac], XmNtopOffset, 15);  ac++;
	label = XmCreateLabel(box, "Server Access", al, ac);
	XtManageChild(label);

	ac = 0;
	XtSetArg(al[ac], XmNshadowType, XmSHADOW_ETCHED_IN );  ac++;
	XtSetArg(al[ac], XmNleftAttachment, XmATTACH_FORM);  ac++;
	XtSetArg(al[ac], XmNrightAttachment, XmATTACH_FORM);  ac++;
	XtSetArg(al[ac], XmNtopAttachment, XmATTACH_WIDGET);  ac++;
	XtSetArg(al[ac], XmNtopWidget, label);  ac++;
	frame = XmCreateFrame(box, "serverAccess", al, ac);
	XtManageChild(frame);

	ac = 0;
	XtSetArg(al[ac], XmNentryClass, xmToggleButtonGadgetClass);  ac++;
	radio_buttn = XmCreateRadioBox(frame, "label", al, ac);
	XtManageChild(radio_buttn);

	ac = 0;
	XtSetArg(al[ac], XmNshadowThickness, 0);  ac++;
	auto_toggle =
		XmCreateToggleButtonGadget(radio_buttn, "Automatic", al, ac);
	XtAddCallback(auto_toggle, XmNvalueChangedCallback, SelectionCB,
		     (XtPointer) DIALOG_RELEASE);
	XtManageChild(auto_toggle);

	ac = 0;
	XtSetArg(al[ac], XmNshadowThickness, 0);  ac++;
	enable_toggle =
		XmCreateToggleButtonGadget(radio_buttn, "Enable", al, ac);
	XtAddCallback(enable_toggle, XmNvalueChangedCallback, SelectionCB,
		     (XtPointer) DIALOG_ADDHOST);
	XtManageChild(enable_toggle);

	ac = 0;
	XtSetArg(al[ac], XmNshadowThickness, 0);  ac++;
	disable_toggle =
		XmCreateToggleButtonGadget(radio_buttn, "Disable", al, ac);
	XtAddCallback(disable_toggle, XmNvalueChangedCallback, SelectionCB,
		     (XtPointer) DIALOG_DELHOST);
	XtManageChild(disable_toggle);

	return(box);
}





/* ----------------------------------------------------------------------------
 * main - Initialize toolkit
 *        Create MainWindow and subwidgets
 *        Realize toplevel widgets
 *        Process events
 * ------------------------------------------------------------------------- */

int main(int argc, char **argv)
{
	Widget			main_win;
	Widget			menuBar;
	Widget			work_area;
	Widget			form;
	Pixmap			icon;
	Arg			al[10];
	register int		ac;
	XmListCallbackStruct	cb;

        (void) strcpy(argv[0],"xLogin");

        signal(SIGINT, PExit);
        signal(SIGTERM, PExit);
        signal(SIGCLD, SIG_IGN);

	/* ----------------------------- Initialize toolkit and open display */

	XtToolkitInitialize();
	app_context = XtCreateApplicationContext();

	XtAppSetFallbackResources(app_context, fallbackResources);
	
	display = XtOpenDisplay(app_context, NULL, argv[0], "xLogin",
			NULL, 0, &argc, argv);
	if (!display)
	{
		XtWarning("xLogin: can't open display, exiting ...");
		exit(0);
	}

        /* -------------------------------------------------- Read databases */

        initDatabases();

	/* ------------------------------------------- Synchronize log files */

	SyncLogFilesCB(NULL);

	/* ----------------------------------------- Create ApplicationShell */

	app_shell = XtAppCreateShell(NULL, "xLogin",
			applicationShellWidgetClass, display, NULL, 0);

	/* ----------------------------------------------- Create MainWindow */

	ac = 0;
	XtSetArg(al[ac], XmNscrollingPolicy, XmAPPLICATION_DEFINED);  ac++;
	main_win = XmCreateMainWindow(app_shell, "main_win", al, ac);
	XtManageChild(main_win);

	/* ---------------------------------------------- create icon bitmap */

	icon = XCreateBitmapFromData(display, 
	       RootWindowOfScreen(XtScreen(app_shell)),
	       (char *) xlogin_bits, xlogin_width, xlogin_height);

	XtVaSetValues(app_shell, XmNiconPixmap, icon, 0);

	/* ------------------------------------ Create MenuBar in MainWindow */

	menuBar = CreateMenuBar(main_win);
	XtManageChild(menuBar);

	/* ------------------------------------ Create toplevel SelectionBox */

	work_area = CreateSelectionBox(main_win);
	XtManageChild(work_area);

	/* -------------------------------- Create work area in SelectionBox */

	form = EnhanceSelectionBox(work_area);
	XtManageChild(form);

	/* ----------------------------------------- Set areas of MainWindow */

	XmMainWindowSetAreas(main_win, menuBar, NULL, NULL, NULL, work_area);

	/* ---------------------------------------- Realize toplevel widgets */

	XtRealizeWidget(app_shell);

	/* ------------- Fake List callback to initialize selected item data */

	cb.item_position = 1;
	SelectionCB(NULL, (XtPointer) DIALOG_LIST, (XtPointer) &cb);

	/* -------------------------------------------------- Process events */

	XtAppMainLoop(app_context);
}

