/* ----------------------------------------------------------------------------
 * Copyright (C) 1995-1999 by Karim Kaschani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * to rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 * --------------------------------------------------------------------------*/

#include "xlogin.h"
#include "strutils.h"





/* ----------------------------------------------------------------------------
 * initHostData - initialize host_data struct
 * --------------------------------------------------------------------------*/

void initHostData(struct host_data *h)
{
	int	i;

	h->access  = 0;
	h->userreq = 0;
	h->sn      = 0;
	h->tn      = 0;
	
	h->addr     = NULL;
	h->alias    = NULL;
	h->session  = NULL;
	h->terminal = NULL;

	for(i = 0; i < MAX_OPTIONS; i++) {
	   h->sOptions[i] = NULL;
	   h->tOptions[i] = NULL;
	}
}





/* ----------------------------------------------------------------------------
 * getword - get next word of input string
 * ------------------------------------------------------------------------- */

char *getword(char **s, int eol, char *file, int line)
{
	int		l;			/* length of next word	  */
	char		*w;			/* beginning of next word */
	char		*word;			/* next word		  */
	struct utsname	*u_name;

	/* ------------------------------------ check for valid input string */

	if (*s == NULL)
	   return (NULL);

	/* ------------------------------------- find beginning of next word */

	w = *s;

	while (*w != eol && *w != '\n' && isspace((int) *w))
	      w++;

	/* ----------------------------------- calculate length of next word */

	if (*w != eol && *w != '\n') {
	   if (*w == '"') {				/* compound word     */
	      w++;					/* skip leading '"'  */
	      if ((*s = strpbrk(w, "\"")) != NULL) {
	         l = (int) strlen(w) - (int) strlen(*s);
	         (*s)++;				/* skip trailing '"' */
	      } else {
	         fprintf(stderr, "%s: SYNTAX ERROR in line %d ", file, line);
	         fprintf(stderr, "- IGNORED: \"%s", w);
	         return(NULL);
	      }
	   } else {					/* no compound word  */
	      l = 0;
	      *s = w;
	      while (**s != eol && **s != '\n' && !isspace((int) **s)) {
	            (*s)++;
	            l++;
	      }
	   }

	   /* ----------------------------------------- substitute variables */

	   word = (char *) malloc(sizeof(char) * (l+1));
	   (void) StrNCpy(word, w, l);

	   /* ........................................... name of local host */

	   if (StrCmp(word, "%H") == 0) {
	      u_name = (struct utsname *) malloc(sizeof(struct utsname));

	      if (uname(u_name) != -1) {
	         word = (char *) realloc(word,
	                sizeof(char) * (strlen(u_name->nodename)+1));
	         (void) strcpy(word, u_name->nodename);
	      } else {
	         fprintf(stderr, "%s: ERROR - failed to get hostname ", file);
	         fprintf(stderr, "in line %d, using 'localhost' instead.\n",
	                 line);
	         word = (char *) realloc(word, sizeof(char) * 10);
	         (void) strcpy(word, "localhost");
	      }

	      free(u_name);
	   }

	   /* ............................ host:display struct of local host */

	   if (StrCmp(word, "%D") == 0) {

	      /* ........... is hostname missing in hostname:display struct? */

	      if (DisplayString(display)[0] == ':') {
	         u_name = (struct utsname *) malloc(sizeof(struct utsname));

	         /* ..................................... get local hostname */

	         if (uname(u_name) != -1) {
	            word = (char *) realloc(word,
	                   sizeof(char) * (strlen(u_name->nodename)+1));
	            (void) strcpy(word, u_name->nodename);

	            /* .............................. resolve its ip address */

	            if (substAddr(&word) < 0) {
	               fprintf(stderr,
	                       "%s: ERROR - failed to resolve ip address ",
	                       file);
	               fprintf(stderr,
	                       "in line %d, using '0.0.0.0' instead.\n",
	                       line);
	               word = (char *) realloc(word, sizeof(char) * 8);
	               (void) strcpy(word, "0.0.0.0");
	            }
	         } else {
	            fprintf(stderr, "%s: ERROR - failed to get hostname ",
	                    file);
	            fprintf(stderr, "in line %d, using 'localhost' instead.\n",
	                    line);
	            word = (char *) realloc(word, sizeof(char) * 10);
	            (void) strcpy(word, "localhost");
	         }

	         free(u_name);
	      } else {
	         (void) strcpy(word, "");
	      }

	      /* ............................ append hostname:display struct */
	   
              word = (char *) realloc(word, sizeof(char) *
                     (strlen(DisplayString(display)) + strlen(word) + 1));
	      (void) strcat(word, DisplayString(display));
	   }

	   return (word);
	}

	return (NULL);
}





/* ----------------------------------------------------------------------------
 * parseLine - parse input line and fill in 'host' structure
 * ------------------------------------------------------------------------- */

struct host_data *parseLine(char *file, char *string, int line)
{
	int			l, n, status = SUCCESS;
	static char		tmpstr[MAX_CHARS];
	char			*p1, *p2, *w;
	struct host_data	*host;

	host   = (struct host_data *) malloc(sizeof(struct host_data));
	initHostData(host);

	/* -------------------------- insert blanks before '}' and after `{' */

	p1 = string;
	(void) strcpy(tmpstr, "");
	
	while ((p2 = strpbrk(p1, "{}")) != NULL) {
	      l = (int) strlen(p1) - (int) strlen(p2);
	      
	      if (strncmp(p2, "{", 1) == 0) {		/* found '{' */
	         l++;
	         (void) strncat(tmpstr, p1, l);
	         (void) strcat(tmpstr, " ");
	      } else {					/* found '}' */
	         (void) strncat(tmpstr, p1, l);
	         (void) strcat(tmpstr, " }");
	      }

	      p1 = p2;
	      p1++;
	}

	(void) strcat(tmpstr, p1);

	/* ------------------------------------------- get and analyze words */

	host->access  = UNDEF;
	host->userreq = FALSE;

	p1 = &tmpstr[0];

	for (n = 0; n < 2; n++) {
	    if ((w = getword(&p1, '\0', file, line)) == NULL) break; 
	      
	    switch (n) {
		   case 0 : host->addr = w;
		            if (strpbrk(w, "{}") != NULL)
			       status = FAIL;
		            break;
		   case 1 : host->alias = w;
			    if (strpbrk(w, "{}") != NULL)
			       status = FAIL;
		            break;
	    }
	}

	/* -------- make sure we got at least 'host->addr' and 'host->alias' */

	if (n < 2)
	   status = FAIL;

	while ((w = getword(&p1, '\0', file, line)) != NULL) {

	      /* --------------------------------- found 'Session' substring */

	      if (StrCmp(w, "S{") == 0) {
	         if (strpbrk(p1, "}") != NULL) {
	            n = 0;

	            while ((w = getword(&p1, '}', file, line)) != NULL) {
	                  n++;

	                  switch (n) {

				 /* ----------------------------------- name */
	                  
		                 case 1 : host->session = w;
		                          break;

				 /* ------------------ need for server access */
		                          
		                 case 2 : if (StrCmp(w, "ON") == 0)
		                             host->access = TRUE;
		                          else if (StrCmp(w , "OFF") == 0)
		                             host->access = FALSE;
		                          else {
			                     status = FAIL;
		                          }
		                          free(w);
		                          break;

				 /* -------------------------------- options */
		                          
			         default: if (n-3 < MAX_OPTIONS) {
			                     host->sOptions[n-3] = w;
			                     if (StrCmp(w, "%U") == 0)
			                        host->userreq = TRUE;
			                  } else {
			                     fprintf(stderr, "%s: TOO MANY OPTIONS in line %d", file, line);
			                     fprintf(stderr, " - IGNORED: %s\n", w);
			                     free(w);
			                  }
		                          break;
	                  }
	            }

	            if (p1 == NULL) break;

	            p1++;
	            p1++;
	         } else {
		    status = FAIL;
		    free(w);
	            break;
	         }
	         
	      /* -------------------------------- found 'Terminal' substring */

	      } else if (StrCmp(w, "T{") == 0) {
	         if (strpbrk(p1, "}") != NULL) {
	            n = 0;

	            while ((w = getword(&p1, '}', file, line)) != NULL) {
	                  n++;

	                  switch (n) {

				 /* ----------------------------------- name */
		                 
		                 case 1 : host->terminal = w;
		                          break;

				 /* -------------------------------- options */

			         default: if (n-2 < MAX_OPTIONS) {
			                     host->tOptions[n-2] = w;
			                  } else {
			                     fprintf(stderr, "%s: TOO MANY OPTIONS in line %d", file, line);
			                     fprintf(stderr, " - IGNORED: %s\n", w);
			                     free(w);
			                  }
		                          break;
	                  }
	            }

	            if (p1 == NULL) break;

	            p1++;
	            p1++;
	         } else {
		    status = FAIL;
		    free(w);
	            break;
	         }
	      
	      /* in order to accept entries from '/etc/hosts', too, *
	       * other substrings are simply ignored                */

	      } else {
		 fprintf(stderr, "%s: SYNTAX ERROR in line %d", file, line);
		 fprintf(stderr, " - IGNORED: %s %s", w, p1);
	         free(w);
	      }
	}

	/* ------------------------------------------------- SUCCESS or FAIL */

	if (status == FAIL) {

	   fprintf(stderr, "%s: SYNTAX ERROR in line %d", file, line);
	   fprintf(stderr, " - IGNORED: %s", string);

	   /* -------------------------------------------------- free memory */
	
	   if (host->addr != NULL) {
	      free(host->addr);
	      host->addr = NULL;
	   }
	   if (host->alias != NULL) {
	      free(host->alias);
	      host->alias = NULL;
	   }
	   if (host->session != NULL) {
	      free(host->session);
	      host->session = NULL;
	   }
	   if (host->terminal != NULL) {
	      free(host->terminal);
	      host->terminal = NULL;
	   }
	   n = 0;
	   while (n < MAX_OPTIONS) {
	         if (host->sOptions[n] != NULL) {
	            free(host->sOptions[n]);
	            host->sOptions[n] = NULL;
	            n++;
	         } else
	            break;
	   }
	   n = 0;
	   while (n < MAX_OPTIONS) {
	         if (host->tOptions[n] != NULL) {
	            free(host->tOptions[n]);
	            host->tOptions[n] = NULL;
	            n++;
	         } else
	            break;
	   }
	   free(host);
	   host = NULL;
	} else {

	   /* -------------------- at least terminal option '-e' must be set */
	
	   if (host->tOptions[0] == NULL) {
	      host->tOptions[0] = (char *) malloc(sizeof(char)*3);
	      (void) strcpy(host->tOptions[0], "-e");
	   }
	}

	return (host);
}




 
