/* ----------------------------------------------------------------------------
 * Copyright (C) 1995-1999 by Karim Kaschani
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * to rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHOR BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. 
 * --------------------------------------------------------------------------*/

#include "xlogin.h"
#include "parse.h"
#include "strutils.h"
#include "xhost.h"





/* ----------------------------------------------------------------- globals */

static pid_t	tPID;		/* terminal pid			*/
static int	XAccess;	/* current server access status	*/
static char	*Host;		/* curr. host internet address  */
static char	*rcPath;	/* resource path		*/





/* ----------------------------------------------------------------------------
 * CExit - terminate child process
 * ------------------------------------------------------------------------- */

void CExit(int signo)
{
	fprintf(stderr, "CExit: Terminating child process due to signal %d.\n",
	        signo);

	if (kill(tPID, SIGINT) != 0)
	   perror("ERROR: ");

	if (XAccess)
	   (void) ExecXHost(Host, rcPath, DEL);

	exit(0);
}





/* ----------------------------------------------------------------------------
 * Connect - set up connection
 * ------------------------------------------------------------------------- */

int Connect(char *address, struct host_data *host, char *terminal,
            char *session, char *username, char *rc_path, int Access,
            int validOptions)
{
	int	i, j;
	pid_t	pid;
	char	*execArgs[2*MAX_OPTIONS + 6];	/* argv[] to be executed
						   1  : terminal name
						   2-n:	terminal options
						   n+1:	session name
						   n+2:	host address
						   n+3: NULL                */

	/* --------------------------------------- initialization of globals */

	rcPath  = rc_path;
	XAccess = Access;

	/* -------------------------------------- set-up execution arguments */

	i = j = 0;

	execArgs[i] = terminal; i++;

	if (StrCmp(terminal, host->terminal) == 0 && validOptions == TRUE) {

	   /* get all terminal options */
	
	   while (j < MAX_OPTIONS) {
	         if (host->tOptions[j] != NULL) {
	            execArgs[i] = host->tOptions[j]; i++;
	            j++;
	         } else
	            break;
	   }
	} else {

	   /* get only last terminal option (probably '-e') */
	
	   while (j < MAX_OPTIONS) {
	         if (host->tOptions[j] != NULL)
	            j++;
	         else
	            break;
	   }
	   execArgs[i] = host->tOptions[j-1]; i++;
	}

	j = 0;

	execArgs[i] = session; i++;

	if (StrCmp(session, host->session) == 0 && validOptions == TRUE) {

	   /* get all session options */
	
	   while (j < MAX_OPTIONS) {
	         if (host->sOptions[j] != NULL) {
	            if (StrCmp(host->sOptions[j], "%U") == 0) {
	               execArgs[i] = username; i++;
	            } else {
	               execArgs[i] = host->sOptions[j]; i++;
	            }
	            j++;
	         } else
	            break;
	   }
	}

	if (validOptions == TRUE) {
	   execArgs[i] = host->addr; i++;
	} else {
	   execArgs[i] = address; i++;
	}

	execArgs[i] = NULL;

	i = 0;
	fprintf(stderr, "Connect:");
	while (execArgs[i] != NULL) {
	      fprintf(stderr, " %s", execArgs[i]);
	      i++;
	}
	fprintf(stderr, "\n");

	/* ---------------------------------------------- fork child process */

	if ((pid = fork()) < 0) {
	   perror("ERROR: ");
	   return(FAIL);
	} else {
	   if (pid == 0) { /* child process */

              signal(SIGINT, CExit);
              signal(SIGTERM, CExit);
	      signal(SIGCLD, SIG_DFL);

	      /* ------------------------------------------------- add xhost */
 
	      if (XAccess)
	         (void) ExecXHost(address, rcPath, ADD);

	      /* ---------------------------------------------- open session */

	      if ((pid = fork()) < 0) {
	         perror("ERROR: ");
	         return(FAIL);
	      } else {
	         if (pid == 0) { /* child process */
		    if (execvp(terminal, execArgs)) {
	               perror("ERROR: ");
	               exit(2);
	            }
	         } else { /* parent process */
	            tPID = pid;
	            Host = address;
	            if (waitpid(pid, NULL, 0) < 0) {
	               perror("ERROR: ");
	               return(FAIL);
	            }
	         }
	      }

	      /* ---------------------------------------------- delete xhost */
 
	      if (XAccess)
	         (void) ExecXHost(address, rcPath, DEL);

	      exit(0);
	   }
	}

	return (SUCCESS);
}
