/*----------------------------------------------------------------------------
--
--  Module:           xitXmStr.c
--
--  Project:          xit  - X Internal Toolkit
--  System:           xit  - X Internal Toolkit
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    XmString routines.
--
--  Filename:         xitXmStr.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-10-20
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitXmStr.c, Version: 1.1, Date: 95/02/18 15:10:53";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Xm.h>
#include <Xm/Text.h>

#include "System.h"

#include "xitTools.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#ifndef _POSIX_SOURCE
#  define  const
#endif


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static int
  compareStrings( const void  *string1,
                  const void  *string2 );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XmString
  xitStringConcatAndFree( XmString  to_the_string,
                          XmString  string )
{

  /* Variables. */
  XmString  new_string;


  /* Code. */

  new_string = XmStringConcat( to_the_string, string );

  XmStringFree( to_the_string );
  XmStringFree( string );

  return( new_string );

} /* xitStringConcatAndFree */


/*----------------------------------------------------------------------*/

char
  *xitStringGetLabel( Widget  widget )
{

  /* Variables. */
  char      *char_ref;
  char      *label_ref;
  Arg       args[ 5 ];
  Cardinal  n;
  XmString  xstr;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNlabelString, &xstr ); n++;
  XtGetValues( widget, args, n );

  if( XmStringGetLtoR( xstr, CS, &char_ref ) ) {
    XmStringFree( xstr );

    label_ref = SysNewString( char_ref );
    XtFree( char_ref );

    return( label_ref );
  }


  return( NULL );

} /* xitStringGetLabel */


/*----------------------------------------------------------------------*/

char
  *xitStringGetString( XmString         xstr,
                       XmStringCharSet  char_set )
{

  /* Variables. */
  char  *char_ref;
  char  *label_ref;


  /* Code. */

  if( XmStringGetLtoR( xstr, char_set, &char_ref ) ) {

    label_ref = SysNewString( char_ref );
    XtFree( char_ref );

    return( label_ref );
  }


  return( NULL );

} /* xitStringGetString */


/*----------------------------------------------------------------------*/

char
  *xitStringGetText( Widget  widget )
{

  /* Variables. */
  char      *char_ref;
  char      *label_ref;


  /* Code. */

  char_ref = XmTextGetString( widget );

  label_ref = SysNewString( char_ref );
  XtFree( char_ref );


  return( label_ref );

} /* xitStringGetText */


/*----------------------------------------------------------------------*/

void
  xitStringSetLabel( Widget  widget,
                     char    *char_ref )
{

  /* Variables. */
  Arg       args[ 5 ];
  Cardinal  n;
  XmString  xstr;


  /* Code. */

  xstr = XmStringCreateLtoR( char_ref, CS );

  n = 0;
  XtSetArg( args[ n ], XmNlabelString, xstr ); n++;
  XtSetValues( widget, args, n );  

  XmStringFree( xstr );


  return;

} /* xitStringSetLabel */


/*----------------------------------------------------------------------*/

void
  xitStringSetTextFromFile( Widget  widget,
                            char    *file_name )
{

  /* Variables. */
  int          status;
  char         *text_ref;
  FILE         *file_ref;
  struct stat  file_info;


  /* Code. */

  /* Fetch the file. */
  status = stat( file_name, &file_info );
  if( status != 0 )
    return;

  if( ! S_ISREG( file_info.st_mode ) )
    return;

  file_ref = fopen( file_name, "r" );
  if( file_ref == NULL )
    return;

  /* Allocate space for the contents of the file. */
  text_ref = SysMalloc( file_info.st_size + 1 );

  /* Read the contents. */
  status = fread( text_ref, sizeof( char ), file_info.st_size + 1, file_ref );

  if( status != 0 ) {

    XmTextPosition  position;

    *(text_ref + file_info.st_size) = '\0';

    /* Update the text fieled. */
    position = XmTextGetInsertionPosition( widget );
    XmTextReplace( widget, position, position, text_ref );

    XmTextShowPosition( widget, position );

  } /* if */


  /* Clean up. */
  SysFree( text_ref );
  fclose( file_ref );


  return;

} /* xitStringSetTextFromFile */


/*----------------------------------------------------------------------*/

void
  xitSortStringList( XmString  *list,
                     int       elements )
{

  /* Type definitions. */
  typedef  char  *str_ref;


  /* Variables. */
  int   index;
  char  **strings_table;


  /* Code. */

  if( elements <= 0 )
    return;

  /* Allocate space for a string array. */
  strings_table = (char **) SysMalloc( elements * sizeof( str_ref ) );

  /* Fill the string table. */
  for( index = 0; index < elements; index++ )
    *(strings_table + index) = xitStringGetString( *(list + index), CS );


  /* Use the predefined quicksort. */
  qsort( (void *) strings_table,
         (size_t) elements, sizeof( str_ref ),
         compareStrings );


  /* Allocate the sorted strings (and free the old). */
  for( index = 0; index < elements; index++ ) {
    XmStringFree( *(list + index) );

    *(list + index) = XmStringCreateLtoR( *(strings_table + index), CS );

    SysFree( *(strings_table + index) );
  }

  SysFree( strings_table );


  return;

} /* xitSortStringList */


/*----------------------------------------------------------------------*/

static int
  compareStrings( const void  *string1,
                  const void  *string2 )
{

  /* Variables. */
  char  **cmp_str1 = (char **) string1;
  char  **cmp_str2 = (char **) string2;


  /* Code. */

  return( strcmp( *cmp_str1, *cmp_str2 ) );

} /* compareStrings */


