/*----------------------------------------------------------------------------
--
--  Module:           xitColorSelect
--
--  Project:          xit   - X Internal Toolkit
--  System:           <>
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    The field select widget contains a single line Text widget. To the
--    right of the field there is a button. By clicking on the button, 
--    a menu with colors is presented. If a color is selected from the menu,
--    the background of the text field is changed to the selected color.
--
--  Filename:         xitColorSel.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-05-28
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xitColorSel.c, Version: 1.1, Date: 95/02/18 15:10:21";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/ArrowB.h>
#include <Xm/CascadeB.h>
#include <Xm/DrawnB.h>
#include <Xm/DrawingA.h>
#include <Xm/PushB.h>
#include <Xm/PushBG.h>
#include <Xm/RowColumn.h>
#include <Xm/Text.h>

#include "System.h"

#include "xitTools.h"
#include "xitColorSel.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record associated with the widget. */
typedef struct {
  XtPointer       client_data;
  int             columns;
  int             elements;
  int             item_selected;
  int             max_length;
  Dimension       button_width;
  Dimension       text_width;
  Widget          field;
  Widget          menu;
  Widget          menuBr;
  Widget          select_button;
  XmString        select_string;
  String          *list;
  XtCallbackProc  call_proc;
} SelectRec, *SelectRecPtr;



/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Resource list for items. */
static XtResource  item_resources[] = {

  { XmNlabelString, XmCLabelString, XmRXmString, sizeof( XmString ),
    XtOffset( SelectRecPtr, select_string ), 
    XmRString, "-" },

  { XmNcolumns, XmCColumns, XmRInt, sizeof( int ),
    XtOffset( SelectRecPtr, columns ), 
    XmRString, "10" },

  { XmNitems, XmCItems, XmRStringTable, sizeof( String * ),
    XtOffset( SelectRecPtr, list ), 
    XmRImmediate, NULL },

  { XmNitemCount, XmCItemCount, XmRInt, sizeof( int ),
    XtOffset( SelectRecPtr, elements ), 
    XmRString, "0" },

};


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  callCallback( SelectRecPtr  rec,
                XEvent        *event );

static void
  destroyCB( Widget        widget,
             SelectRecPtr  rec,
             XtPointer     call_data );

static void
  getColorInfo( SelectRecPtr  rec,
                Pixel         *pixel,
                XColor        *color,
                char          *rgb_string );

static void
  menuSelectCB( Widget                     widget,
                SelectRecPtr               rec,
                XmRowColumnCallbackStruct  *call_data );

static void
  resizeCB( Widget               widget,
            SelectRecPtr         rec,
            XmAnyCallbackStruct  *call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

Widget
  xitCreateColorSelect( Widget          parent,
                        String          name,
                        ArgList         add_args,
                        Cardinal        num_args,
                        XtCallbackProc  proc,
                        XtPointer       client_data )
{

  /* Variables. */
  Boolean        ok;
  int            index;
  char           buffer[ 50 ];
  Arg            args[ 6 ];
  ArgList        merge_args;
  Cardinal       n;
  Dimension      select_height;
  Dimension      text_height;
  Dimension      text_width;
  Pixel          bottom_color;
  Pixel          color;
  Pixel          top_color;
  SelectRecPtr   rec;
  Widget         manager;
  Widget         menuPb;
  XmString       xstr;

  static XIT_TEXT_STRUCT text_buffer[] = {
    { "",  NULL, 1, True },
  };


  /* Code. */

  /* Initialize widget record, used in callbacks. */
  rec = SysNew( SelectRec );

  rec -> call_proc   = proc;
  rec -> client_data = client_data;

  /* Drawing area as manager. */
  n = 0;
  XtSetArg( args[ n ], XmNmarginHeight, 0 ); n++;
  XtSetArg( args[ n ], XmNmarginWidth,  0 ); n++;
  XtSetArg( args[ n ], XmNuserData,     rec ); n++;
  manager = XmCreateDrawingArea( parent, name, args, n );

  XtAddCallback( manager, XmNresizeCallback,  
                 (XtCallbackProc) resizeCB, (XtPointer) rec );
  XtAddCallback( manager, XmNdestroyCallback,
                 (XtCallbackProc) destroyCB, (XtPointer) rec );


  /* Initialize the rest of the callback record. */
  XtGetSubresources( manager, (XtPointer) rec, name, COLOR_SELECT_CLASS,
                     item_resources, XtNumber( item_resources ),
                     add_args, num_args );


  /* Text display. */
  sprintf( buffer, "%sTx", name );

  text_buffer[ 0 ].name = buffer;
  rec -> field = xitCreateText( manager, &text_buffer[ 0 ] );

  n = 0;
  XtSetArg( args[ n ], XmNeditMode, XmSINGLE_LINE_EDIT ); n++;
  XtSetArg( args[ n ], XmNrecomputeSize, False ); n++;
  XtSetArg( args[ n ], XmNeditable, False ); n++;
  XtSetArg( args[ n ], XmNcursorPositionVisible, False ); n++;
  XtSetArg( args[ n ], XmNmaxLength, rec -> columns ); n++;
  XtSetArg( args[ n ], XmNcolumns, (short ) rec -> columns ); n++;

  merge_args = XtMergeArgLists( add_args, num_args, args, n );

  XtSetValues( rec -> field, merge_args, num_args + n );

  XtManageChild( rec -> field );

  SysFree( merge_args );


  /* Swap bottom and top shadow colors. */
  n = 0;
  XtSetArg( args[ n ], XmNtopShadowColor,    &top_color ); n++;
  XtSetArg( args[ n ], XmNbottomShadowColor, &bottom_color ); n++;
  XtGetValues( rec -> field, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNtopShadowColor,    bottom_color ); n++;
  XtSetArg( args[ n ], XmNbottomShadowColor, top_color ); n++;
  XtSetValues( rec -> field, args, n );


  /* Create the menubar and menu cascades. */
  sprintf( buffer, "%sBr", name );

  n = 0;
  XtSetArg( args[ n ], XmNmarginWidth,     0 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight,    0 ); n++;
  rec -> menuBr = XmCreateMenuBar( manager, buffer, args, n );

  XtManageChild( rec -> menuBr );


  /* Create the pull-down menu to use. */
  sprintf( buffer, "%sPd", name );

  n = 0;
  rec -> menu = XmCreatePulldownMenu( rec -> menuBr, buffer, args, n );

  XtAddCallback( rec -> menu, XmNentryCallback, 
                 (XtCallbackProc) menuSelectCB, (XtPointer) rec );


  /* The cascade button. */
  sprintf( buffer, "%sCa", name );

  if( rec -> select_string == NULL )
    xstr = XmStringCreate( "-", CS );
  else
    xstr = rec -> select_string;

  n = 0;
  XtSetArg( args[ n ], XmNshadowThickness, 0 ); n++;
  XtSetArg( args[ n ], XmNmarginHeight,    0 ); n++;
  XtSetArg( args[ n ], XmNsubMenuId,       rec -> menu ); n++;
  XtSetArg( args[ n ], XmNlabelString,     xstr ); n++;
  rec -> select_button = XmCreateCascadeButton( rec -> menuBr, buffer, 
                                                args, n );
  if( rec -> select_string == NULL )
    XmStringFree( xstr );

  XtManageChild( rec -> select_button );


  /* Assign all the labels in the menu. */
  xstr = XmStringCreate( "     ", CS );

  for( index = 0; index < rec -> elements; index++ ) {
    sprintf( buffer, "%dPb", index + 1 );

    ok = xitAllocNamedColor( rec -> menu, rec -> list[ index ], &color );
    if( ! ok )
      color = WhitePixelOfScreen( XtScreen( rec -> menu ) );

    n = 0;
    XtSetArg( args[ n ], XmNlabelString, xstr ); n++;
    XtSetArg( args[ n ], XmNbackground,  color ); n++;
    menuPb = XmCreatePushButton( rec -> menu, buffer, args, n );

    XtManageChild( menuPb );

    XtAddCallback( menuPb, XmNactivateCallback, 
                   (XtCallbackProc) menuSelectCB, (XtPointer) index );

    /* The default color is the first color. */
    if( index == 0 ) {
      n = 0;
      XtSetArg( args[ n ], XmNbackground,  color ); n++;
      XtSetValues( rec -> field, args, n );
    }

  } /* loop */

  XmStringFree( xstr );


  /* Align text and select button. */
  n = 0;
  XtSetArg( args[ n ], XmNheight, &text_height ); n++;
  XtGetValues( rec -> field, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNwidth, &text_width ); n++;
  XtGetValues( rec -> field, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNheight, &select_height ); n++;
  XtGetValues( rec -> select_button, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNheight, &rec -> button_width ); n++;
  XtGetValues( rec -> menuBr, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNx, text_width + 2 ); n++;
  XtSetArg( args[ n ], XmNy, (text_height - select_height) / 2 - 2 ); n++;
  XtSetValues( rec -> menuBr, args, n );


  return( manager );

} /* xitCreateFieldSelect */


/*----------------------------------------------------------------------*/

Widget
  xitColorSelectGetChild( Widget                  widget,
                          xitColorSelectChildren  child )
{

  /* Variables. */
  Arg           args[ 2 ];
  Cardinal      n;
  SelectRecPtr  rec;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return( NULL );

  switch( child ) {
    case xitCOLOR_SELECT_TEXT_FIELD:
      return( rec -> field );

    case xitCOLOR_SELECT_BUTTON:
      return( rec -> select_button );

    default:
      break;
  } /* switch */


  return( NULL );

} /* xitColorSelectGetChild */


/*----------------------------------------------------------------------*/

Boolean
  xitColorSelectGetColor( Widget  widget,
                          Pixel   *pixel,
                          XColor  *color,
                          char    *rgb_string )
{

  /* Variables. */
  Arg           args[ 2 ];
  Cardinal      n;
  SelectRecPtr  rec;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return( False );

  getColorInfo( rec,
                pixel, color, rgb_string );


  return( True );

} /* xitColorSelectGetColor */


/*----------------------------------------------------------------------*/

Boolean
  xitColorSelectGetIndex( Widget  widget,
                          int     *selected_index )
{

  /* Variables. */
  int           index;
  char          buffer[ 50 ];
  Arg           args[ 5 ];
  Cardinal      n;
  Pixel         bg_color;
  Pixel         sel_color;
  SelectRecPtr  rec;
  Widget        tempW;
  XColor        dummy_color;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return( False );


  /* The color we have selected. */
  getColorInfo( rec, &sel_color, &dummy_color, buffer );

  /* Search the color in the range of colors. */
  for( index = 0; index < rec -> elements; index++ ) {

    sprintf( buffer, "%dPb", index + 1 );

    tempW = XtNameToWidget( rec -> menu, buffer );

    n = 0;
    XtSetArg( args[ n ], XmNbackground, &bg_color ); n++;
    XtGetValues( tempW, args, n );

    if( bg_color == sel_color ) {
      *selected_index = index;
      return( True );
    }

  } /* loop */


  return( False );

} /* xitColorSelectGetIndex */


/*----------------------------------------------------------------------*/

Boolean
  xitColorSelectGetPixel( Widget  widget,
                          Pixel   *pixel )
{

  /* Variables. */
  Boolean  ok;
  char     rgb_string[ 50 ];
  XColor   color;


  /* Code. */

  ok = xitColorSelectGetColor( widget, pixel, &color, rgb_string );


  return( ok );

} /* xitColorSelectGetPixel */


/*----------------------------------------------------------------------*/

void
  xitColorSelectSetColor( Widget   widget,
                          char     *new_color,
                          Boolean  call )
{

  /* Variables. */
  Boolean       ok;
  Arg           args[ 5 ];
  Cardinal      n;
  Pixel         color;
  SelectRecPtr  rec;


  /* Code. */

  if( new_color == NULL || strlen( new_color ) == 0 )
    return;

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return;

  /* Allocate the new color. */
  ok = xitAllocNamedColor( widget, new_color, &color );

  if( ! ok )
    color = WhitePixelOfScreen( XtScreen( widget ) );

  n = 0;
  XtSetArg( args[ n ], XmNbackground, color ); n++;
  XtSetValues( rec -> field, args, n );

  if( call )
    callCallback( rec, NULL );


  return;

} /* xitColorSelectSetColor */


/*----------------------------------------------------------------------*/

void
  xitColorSelectSetIndex( Widget   widget,
                          int      set_index,
                          Boolean  call )
{

  /* Variables. */
  char          buffer[ 50 ];
  Arg           args[ 5 ];
  Cardinal      n;
  Pixel         bg_color;
  SelectRecPtr  rec;
  Widget        tempW;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return;


  if( set_index < 0 || set_index > rec -> elements - 1 )
    return;


  /* Fetch color for the index we selected. */
  sprintf( buffer, "%dPb", set_index + 1 );

  tempW = XtNameToWidget( rec -> menu, buffer );

  n = 0;
  XtSetArg( args[ n ], XmNbackground, &bg_color ); n++;
  XtGetValues( tempW, args, n );


  /* Set the color. */
  n = 0;
  XtSetArg( args[ n ], XmNbackground, bg_color ); n++;
  XtSetValues( rec -> field, args, n );


  /* Call callback? */
  if( call )
    callCallback( rec, NULL );


  return;

} /* xitColorSelectSetIndex */


/*----------------------------------------------------------------------*/

void
  xitColorSelectSetMenu( Widget  widget,
                         char    *new_menu[],
                         int     elements )
{

  /* Variables. */
  Boolean       ok;
  int           index;
  char          buffer[ 50 ];
  Arg           args[ 5 ];
  Cardinal      n;
  Widget        menuPb;
  Widget        tempW;
  Pixel         color;
  SelectRecPtr  rec;
  XmString      xstr;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return;


  /* New menu buttons. */
  for( index = 0; index < elements; index++ ) {
    sprintf( buffer, "%dPb", index + 1 );

    xstr = XmStringCreate( "     ", CS );

    menuPb = XtNameToWidget( rec -> menu, buffer );
    if( menuPb == NULL )
      menuPb = XmCreatePushButton( rec -> menu, buffer, args, 0 );

    ok = xitAllocNamedColor( rec -> menu, new_menu[ index ], &color );
    if( ! ok )
      color = WhitePixelOfScreen( XtScreen( rec -> menu ) );

    n = 0;
    XtSetArg( args[ n ], XmNlabelString, xstr ); n++;
    XtSetArg( args[ n ], XmNbackground,  color ); n++;
    XtSetValues( menuPb, args, n );

    XtManageChild( menuPb );

    XmStringFree( xstr );
  } /* loop */

  for( index = elements; index < rec -> elements; index++ ) {
    sprintf( buffer, "%dPb", index + 1 );

    tempW = XtNameToWidget( rec -> menu, buffer );
    if( tempW != NULL )
      XtUnmanageChild( tempW );
  }

  rec -> elements = elements;


  return;

} /* xitColorSelectSetMenu */


/*----------------------------------------------------------------------*/

void
  xitColorSelectSetPixel( Widget   widget,
                          Pixel    pixel,
                          Boolean  call )
{

  /* Variables. */
  Arg           args[ 5 ];
  Cardinal      n;
  SelectRecPtr  rec;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNuserData, &rec ); n++;
  XtGetValues( widget, args, n );

  if( rec == NULL )
    return;

  /* Set new color. */
  n = 0;
  XtSetArg( args[ n ], XmNbackground, pixel ); n++;
  XtSetValues( rec -> field, args, n );

  if( call )
    callCallback( rec, NULL );


  return;

} /* xitColorSelectSetPixel */


/*----------------------------------------------------------------------*/

static void
  callCallback( SelectRecPtr  rec,
                XEvent        *event )
{

  /* Variables. */
  xitColorSelectCallbackStruct  cb;


  /* Code. */

  if( rec -> call_proc != NULL ) {

    getColorInfo( rec,
                  &cb.pixel, &cb.color, cb.rgb_string );

    cb.reason   = XmCR_VALUE_CHANGED;
    cb.event    = event;

    (* rec -> call_proc) ( rec -> field, 
                           (XtPointer) rec -> client_data, (XtPointer) &cb );

  } /* if */


  return;

} /* callCallback */


/*----------------------------------------------------------------------*/

static void
  getColorInfo( SelectRecPtr  rec,
                Pixel         *pixel,
                XColor        *color,
                char          *rgb_string )
{

  /* Variables. */
  Arg       args[ 5 ];
  Cardinal  n;


  /* Code. */

  /* Selected pixel. */
  n = 0;
  XtSetArg( args[ n ], XmNbackground, pixel ); n++;
  XtGetValues( rec -> field, args, n );

  /* Selected color strcture. */
  color -> pixel = (unsigned long) *pixel;
  XQueryColor( XtDisplay( rec -> field ),
               DefaultColormapOfScreen( XtScreen( rec -> field ) ),
               color );

  /* Selected RGB string. */
  sprintf( rgb_string, "#%04.4hx%04.4hx%04.4hx",
           color -> red, color -> green, color -> blue );


  return;

} /* getColorInfo */


/*----------------------------------------------------------------------*/

static void
  destroyCB( Widget        widget,
             SelectRecPtr  rec,
             XtPointer     call_data )
{

  /* Code. */

  /* Free the allocated record. */
  SysFree( rec );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void
  menuSelectCB( Widget                     widget,
                SelectRecPtr               rec,
                XmRowColumnCallbackStruct  *call_data )
{

  /* Variables. */
  Arg       args[ 5 ];
  Cardinal  n;
  Pixel     color;


  /* Code. */

  if( call_data -> reason != XmCR_ACTIVATE )
    return;


  /* Fetch the selected color. */
  n = 0;
  XtSetArg( args[ n ], XmNbackground, &color ); n++;
  XtGetValues( call_data -> widget, args, n );


  /* Set the new selected color. */
  n = 0;
  XtSetArg( args[ n ], XmNbackground, color ); n++;
  XtSetValues( rec -> field, args, n );


  callCallback( rec, call_data -> event );


  return;

} /* menuSelectCB */


/*----------------------------------------------------------------------*/

static void
  resizeCB( Widget               widget,
            SelectRecPtr         rec,
            XmAnyCallbackStruct  *call_data )
{

  /* Variables. */
  Arg        args[ 5 ];
  Cardinal   n;
  Dimension  height;
  Dimension  select_height;
  Dimension  text_height;
  Dimension  text_width;
  Dimension  width;
  Position   new_y;


  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNwidth, &width ); n++;
  XtSetArg( args[ n ], XmNheight, &height ); n++;
  XtGetValues( widget, args, n );

  text_width = width - rec -> button_width - 2;

  n = 0;
  XtSetArg( args[ n ], XmNheight, &text_height ); n++;
  XtSetArg( args[ n ], XmNwidth,  &text_width ); n++;
  XtGetValues( rec -> field, args, n );

  if( height > text_height )
    new_y = (height - text_height) / 2;
  else
    new_y = 0;

  /* Stop resizing the drawing area now. */
  n = 0;
  XtSetArg( args[ n ], XmNresizePolicy, XmRESIZE_NONE ); n++;
  XtSetValues( widget, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNheight, &select_height ); n++;
  XtGetValues( rec -> select_button, args, n );

  /* Resize the children. */
  n = 0;
  XtSetArg( args[ n ], XmNy,     new_y ); n++;
  XtSetArg( args[ n ], XmNwidth, text_width ); n++;
  XtSetValues( rec -> field, args, n );

  new_y = new_y + (text_height - select_height) / 2 - 2;

  n = 0;
  XtSetArg( args[ n ], XmNx, text_width + 2 ); n++;
  XtSetArg( args[ n ], XmNy, new_y ); n++;
  XtSetValues( rec -> menuBr, args, n );


  return;

} /* resizeCB */
