/*----------------------------------------------------------------------------
--
--  Module:           XmUbTimeSlider
--
--  Project:          XmUb - Ulle's Motif widgets
--  System:           
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    This is the implementation of the widget.
--
--    For the geometry management, the text widget is set to the width it
--    requests, and the scale widths are adjusted depending on the current
--    width of the widget.
--
--  Filename:         XmUbTimeSl.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-09-21
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: XmUbTimeSl.c, Version: 1.1, Date: 95/02/18 15:10:16";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <Xm/XmP.h>

#include <Xm/Scale.h>
#include <Xm/Text.h>

/* Rogge's tools. */
#include "TimDate.h"

/* Private widget header file. */
#include "XmUbTimeSlP.h"

/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

#ifdef MAX
#undef MAX
#endif

#ifdef MIN
#undef MIN
#endif

#define MAX( x, y )  ( ( x ) > ( y ) ? ( x ) : ( y ) )
#define MIN( x, y )  ( ( x ) < ( y ) ? ( x ) : ( y ) )

#define XmUbTS_CHILD_ERROR  -1

#define TEXT_BUFFER_LENGTH  20

/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

typedef struct {

  Dimension  width;
  Dimension  height;
  Position   x;
  Position   y;

} KidDimensionRec;

/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Internal Motif functions (how do we know about this....?) */
extern void
  _XmBackgroundColorDefault();


/* The time slider resource list. */
static XtResource resources[] = {
  {
    XmNactivateCallback,
    XmCCallback,
    XmRCallback, sizeof( XtCallbackList ),
    XtOffset( XmUbTimeSliderWidget, sl.activate_callback ),
    XmRCallback,
    NULL
  },
  {
    XmNvalueChangedCallback,
    XmCCallback,
    XmRCallback, sizeof( XtCallbackList ),
    XtOffset( XmUbTimeSliderWidget, sl.value_changed_callback ),
    XmRCallback,
    NULL
  },
  {
    XmNmarginHeight,
    XmCMarginHeight,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbTimeSliderWidget, sl.margin_height ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmNmarginWidth,
    XmCMarginWidth,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbTimeSliderWidget, sl.margin_width ),
    XtRImmediate,
    (XtPointer) 0
  },
  {
    XmUbNtslRecomputeHeight,
    XmUbCTslRecomputeHeight,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbTimeSliderWidget, sl.recompute_height ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNtslRecomputeWidth,
    XmUbCTslRecomputeWidth,
    XtRBoolean, sizeof( Boolean ),
    XtOffset( XmUbTimeSliderWidget, sl.recompute_width ),
    XtRImmediate,
    (XtPointer) True
  },
  {
    XmUbNtslSpacing,
    XmCSpacing,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbTimeSliderWidget, sl.slider_spacing ),
    XtRImmediate,
    (XtPointer) 4
  },
  {
    XmUbNtslTextSpacing,
    XmCSpacing,
    XtRDimension, sizeof( Dimension ),
    XtOffset( XmUbTimeSliderWidget, sl.text_spacing ),
    XtRImmediate,
    (XtPointer) 8
  },

};  /* resources */




/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

/* Core methods. */
static void
  ChangeManaged( Widget  widget );

static void
  DeleteChild( Widget  widget );

static void 
  Destroy( Widget   widget );

static XtGeometryResult
  GeometryManager( Widget            widget,
                   XtWidgetGeometry  *request,
                   XtWidgetGeometry  *reply );

static void
  Initialize( Widget     treq,
              Widget     tnew,
              ArgList    args,
              Cardinal   *num_args );

static void
  InsertChild( Widget  widget );

static XtGeometryResult
  QueryGeometry( Widget             widget,
                 XtWidgetGeometry  *proposed,
                 XtWidgetGeometry  *answer );

static void
  Resize( Widget    widget );

static Boolean 
  SetValues( Widget     current,
             Widget     request,
             Widget     new,
             ArgList    args,
             Cardinal   *num_args );


/* Internal functions. */


static void
  ActivateCB( Widget                tw,
              XmUbTimeSliderWidget  tbox,
              XmAnyCallbackStruct   *call_data );

/* If not in the child list, returns XmUbTS_CHILD_ERROR. */
static int
  ChildIndex( XmUbTimeSliderWidget  tsl,
              Widget                child );

static Widget
  CreateSlider( XmUbTimeSliderWidget  tsl,
                char                  *name,
                int                   maximum );

/* Fills in the width and height fields in the kids dimension recs. */
static void
  GetChildPrefSizes( XmUbTimeSliderWidget  tsl,
                     Widget                initiator,
                     XtWidgetGeometry      *request,
                     KidDimensionRec       sizes[] );


/* The sizes array must have been prepared by GetChildPrefSizes before this
   function is called. */
static void
  GetOwnPreferredSize( XmUbTimeSliderWidget  tsl,
                       KidDimensionRec       sizes[],
                       Dimension             *pref_width,
                       Dimension             *pref_height );

/* The sizes array  must have been processed by GetChildPrefSizes 
   before DoLayout is called. */
static void
  DoLayout( XmUbTimeSliderWidget  tsl,
            Widget                initiator,
            XtWidgetGeometry      *request,
            KidDimensionRec       sizes[] );

static void 
  KidsPreferredGeometry( Widget            kid,
                         Widget            initiator,
                         XtWidgetGeometry  *request,
                         XtWidgetGeometry  *desired );

static void
  ScaleValueChangedCB( Widget                 scale,
                       XmUbTimeSliderWidget   tsl,
                       XmScaleCallbackStruct  *call_data );

/* ResizeIfNeeded fills in the sizes array. Does not have to be initialized. */
static Boolean
  ResizeIfNeeded( XmUbTimeSliderWidget  tsl,
                  KidDimensionRec    sizes[] );

static void
  SelectAllTextCB( Widget                tw,
                   XmUbTimeSliderWidget  tsl,
                   XmAnyCallbackStruct   *call_data );

static void
  TextChangedCB( Widget                textw,
                 XmUbTimeSliderWidget  tsl,
                 XmAnyCallbackStruct   *call_data );

static void
  UpdateScales( XmUbTimeSliderWidget  tsl,
                int                   hour,
                int                   minute );

static void
  UpdateTextField( XmUbTimeSliderWidget  tsl );

static void
  ValueChangedCB( Widget                tw,
                  XmUbTimeSliderWidget  tsl,
                  XmAnyCallbackStruct   *call_data );

static void
  WarningNoResourceChange( XmUbTimeSliderWidget  tsl,
                           String                resource );

/*----------------------------------------------------------------------------
--  Initialization of the class record.
----------------------------------------------------------------------------*/

/* This initialization has to be done after the methods have been declared. */
XmUbTimeSliderClassRec xmUbTimeSliderClassRec = {

  { /* Core class fields. */
    /* superclass */                    (WidgetClass) &xmManagerClassRec,
    /* class_name */                    "XmUbTimeSlider",
    /* widget_size */                   sizeof( XmUbTimeSliderRec ),
    /* class_initialize */              NULL,
    /* class_part_initialize */         NULL,
    /* class_inited */                  False,
    /* initialize */                    Initialize,
    /* initialize_hook */               NULL,
    /* realize */                       XtInheritRealize, 
    /* actions */                       NULL,
    /* num_actions */                   0,
    /* resources */                     resources,
    /* num_resources */                 XtNumber( resources ),
    /* xrm_class */                     NULLQUARK,
    /* compress_motion */               True,
    /* compress_exposure */             True,
    /* compress_enterleave */           True,
    /* visible_interest */              False,
    /* destroy */                       Destroy,
    /* resize */                        Resize,
    /* expose */                        NULL,
    /* set_values */                    SetValues,
    /* set_values_hook */               NULL,
    /* set_values_almost */             XtInheritSetValuesAlmost,
    /* get_values_hook */               NULL,
    /* accept_focus */                  NULL,
    /* version */                       XtVersion,
    /* callback_private */              NULL,
    /* tm_table */                      NULL,
    /* query_geometry */                QueryGeometry,
    /* display_accelerator */           XtInheritDisplayAccelerator,
    /* extension */                     NULL
  },
  { /* Composite class part. */
    /* geometry_manager */              GeometryManager,
    /* change_managed */                ChangeManaged,
    /* insert_child */                  InsertChild,
    /* delete_child */                  DeleteChild,
    /* extension */                     NULL
  },
  { /* Constraint class fields. */
    /* subresources */                  NULL,
    /* subresource_count */             0,
    /* constraint_size */               sizeof( XmUbTimeSliderConstraintsRec ),
    /* initialize */                    NULL,
    /* destroy */                       NULL,
    /* set_values */                    NULL,
    /* extension */                     NULL
  },
  { /* XmManager class part. */
    /* translations */                  NULL,
    /* get_resources */                 NULL,
    /* num_get_resources */             0,
    /* get_constraint_resources */      NULL,
    /* num_get_constraint_resources */  0,
    /* extension */                     NULL
  },
  { /* Time slider class part. */
    /* extension */                     NULL
  },

}; 


/* Class record pointer. */
WidgetClass 
  xmUbTimeSliderWidgetClass = (WidgetClass) &xmUbTimeSliderClassRec;



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

static void
  ActivateCB( Widget                tw,
              XmUbTimeSliderWidget  tsl,
              XmAnyCallbackStruct   *call_data )
{
  /* Variables. */
  XmUbTimeSliderCallbackStruct  cb;

  /* Code. */

  if( tsl -> sl.activate_callback == NULL )
    return;

  /* Set up callback structure. */

  cb.reason      = XmCR_ACTIVATE;
  cb.event       = call_data -> event;
  cb.child_index = ChildIndex( tsl, tw );
  cb.child       = tw;

  XtCallCallbackList( (Widget) tsl, tsl -> sl.activate_callback,
                      (XtPointer) &cb );


  return;

} /* ActivateCB */


/*----------------------------------------------------------------------*/

static void
  ChangeManaged( Widget  widget )
{
  /* Variables. */
  Boolean               layout_done;
  KidDimensionRec       kids_sizes[ NO_INTERNAL_CHILDREN ];
  XmUbTimeSliderWidget  tsl;


  /* Code. */

  tsl = (XmUbTimeSliderWidget) widget;

  /* ResizeIfNeeded checks the size of all children and resizes the widget. */
  layout_done = ResizeIfNeeded( tsl, kids_sizes );

  /* Do the layout. */
  if( !layout_done )
    DoLayout( tsl, NULL, NULL, kids_sizes );


  return;

} /* ChangeManaged */


/*----------------------------------------------------------------------*/

static int
  ChildIndex( XmUbTimeSliderWidget  tsl,
              Widget                child )
{
  /* Variables. */
  int  index;

  /* Code. */

  for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ ){

    if( tsl -> sl.internal_children[ index ] == child )
      return index;

  } /* for */

 
  /* Specified child not found. */
  return XmUbTS_CHILD_ERROR;

} /* ChildIndex */


/*----------------------------------------------------------------------*/

static void
  CreateInternalWidgets( XmUbTimeSliderWidget  tsl )
{
  /* Variables. */
  Arg           args[ 5 ];
  short         columns;
  int           index;
  Cardinal      n;
  char          *name_buffer;
  char          text_buffer[ TEXT_BUFFER_LENGTH ];
  TIM_TIME_REF  time;
  String        wname;

  /* Code. */

  /* Get the name of the "parent" widget. */
  wname = XtName( tsl );

  name_buffer = XtMalloc( strlen( wname ) + 4 );

  /* Create the sliders. */

  /* Hour slider. */
  sprintf( name_buffer, "%sHSL", wname );

  tsl -> sl.internal_children[ XmUbTS_CHILD_HOUR_SLIDER ] =
    CreateSlider( tsl, name_buffer, 23 );

  /* Minute slider. */
  sprintf( name_buffer, "%sMSL", wname );

  tsl -> sl.internal_children[ XmUbTS_CHILD_MINUTE_SLIDER ] =
    CreateSlider( tsl, name_buffer, 59 );

  /* Create the text field. */
  sprintf( name_buffer, "%sTX", wname );
  
  /* Find out how many columns wide we should make the widget. */
  time = TimMakeTime( 1990, 1, 1, 23, 59, 59 );
  TimFormatTime( time, text_buffer, TEXT_BUFFER_LENGTH );
  columns = (short) ( strlen( text_buffer ) + 1 );

  n = 0;
  XtSetArg( args[ n ], XmNcolumns, columns ); n++;
  XtSetArg( args[ n ], XmNmaxLength, columns ); n++;
  tsl -> sl.internal_children[ XmUbTS_CHILD_TEXT_FIELD ] =
    XmCreateText( (Widget) tsl, name_buffer, args, n );
  
  XtAddCallback( tsl -> sl.internal_children[ XmUbTS_CHILD_TEXT_FIELD ], 
                 XmNvalueChangedCallback,
                 (XtCallbackProc) TextChangedCB, (XtPointer) tsl );

  XtAddCallback( tsl -> sl.internal_children[ XmUbTS_CHILD_TEXT_FIELD ], 
                 XmNactivateCallback,
                 (XtCallbackProc) ActivateCB, (XtPointer) tsl );

  XtAddCallback( tsl -> sl.internal_children[ XmUbTS_CHILD_TEXT_FIELD ], 
                 XmNvalueChangedCallback,
                 (XtCallbackProc) ValueChangedCB, (XtPointer) tsl );

  XtAddCallback( tsl -> sl.internal_children[ XmUbTS_CHILD_TEXT_FIELD ], 
                 XmNfocusCallback, 
                 (XtCallbackProc) SelectAllTextCB, (XtPointer) tsl );

  /* Manage the created children. */
  for( index = XmUbTS_FIRST_CHILD; index < NO_INTERNAL_CHILDREN; index ++ ){
    if( tsl -> sl.internal_children[ index ] != NULL )
      XtManageChild( tsl -> sl.internal_children[ index ] );
  }


  XtFree( name_buffer );


  return;

} /* CreateInternalWidgets */


/*----------------------------------------------------------------------*/

static Widget
  CreateSlider( XmUbTimeSliderWidget  tsl,
                char                  *name,
                int                   maximum )
{
  /* Variables. */
  Arg       args[ 5 ];
  Cardinal  n;
  Widget    scale;

  /* Code. */

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNshowValue, False ); n++;
  XtSetArg( args[ n ], XmNminimum, 0 ); n++;
  XtSetArg( args[ n ], XmNmaximum, maximum ); n++;
  scale = XmCreateScale( (Widget) tsl, name, args, n );

  XtAddCallback( scale, XmNvalueChangedCallback,
                 (XtCallbackProc) ScaleValueChangedCB, (XtPointer) tsl );
  XtAddCallback( scale, XmNdragCallback,
                 (XtCallbackProc) ScaleValueChangedCB, (XtPointer) tsl );


  return( scale );

} /* CreateSlider */


/*----------------------------------------------------------------------*/

static void
  DeleteChild( Widget  widget )
{

  /* Variables. */
  int                   index;
  XmUbTimeSliderWidget  tsl;

  /* Code. */

  tsl = (XmUbTimeSliderWidget) XtParent( widget );

  /* Clear the internal reference. */
  for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ ){
    if( tsl -> sl.internal_children[ index ] == widget ){
      tsl -> sl.internal_children[ index ] = NULL;
      break;
    }
  }

  /* Perform the actual operation */
  (* ( (CompositeWidgetClass) (xmUbTimeSliderWidgetClass ->
     core_class.superclass) ) -> composite_class.delete_child ) ( widget );


  return;

} /* DeleteChild */


/*----------------------------------------------------------------------*/

static void 
  Destroy( Widget   widget )
{
  /* Code. */

  /* Remove callbacks. */
  XtRemoveAllCallbacks( widget, XmNactivateCallback );
  XtRemoveAllCallbacks( widget, XmNvalueChangedCallback );


  return;

} /* Destroy */


/*----------------------------------------------------------------------*/

static void
  DoLayout( XmUbTimeSliderWidget  tsl,
            Widget                initiator,
            XtWidgetGeometry      *request,
            KidDimensionRec       sizes[] )
{
  /* Variables. */
  int        index;
  Widget     kid;
  int        no_managed;
  Dimension  scale_width;
  Position   scale_x_pos;
  Position   scale_y_offset;
  Dimension  total_height;
  Position   y_pos;

  /* Code. */

  /* Start with the text widget, let it keep its requested size. */
  if( ( sizes[ XmUbTS_CHILD_TEXT_FIELD ].width + tsl -> sl.margin_width ) >
      tsl -> core.width )
    sizes[ XmUbTS_CHILD_TEXT_FIELD ].x = 0;
  else
    sizes[ XmUbTS_CHILD_TEXT_FIELD ].x = (Position) ( tsl -> core.width - 
      tsl -> sl.margin_width - sizes[ XmUbTS_CHILD_TEXT_FIELD ].width );

  /* Get the total height of the scales. */
  no_managed   = 0;
  total_height = 0;
  for( index = XmUbTS_CHILD_HOUR_SLIDER; 
       index < XmUbTS_CHILD_MINUTE_SLIDER + 1; index ++ ){

    /* If not managed, width and height are zero. */
    if( sizes[ index ].width > 0 ){
      no_managed ++;
      total_height += sizes[ index ].height;
    }
  } /* for */

  if( no_managed > 1 )
    total_height += tsl -> sl.slider_spacing;

  /* Center the scales and the text field vertically. */
  if( total_height > sizes[ XmUbTS_CHILD_TEXT_FIELD ].height ){
    sizes[ XmUbTS_CHILD_TEXT_FIELD ].y = (Position) ( tsl -> sl.margin_height +
      ( total_height - sizes[ XmUbTS_CHILD_TEXT_FIELD ].height ) / 2 );
    scale_y_offset = 0;
  } else {
    sizes[ XmUbTS_CHILD_TEXT_FIELD ].y = 
      (Position) ( tsl -> sl.margin_height );
    scale_y_offset = (Position) ( sizes[ XmUbTS_CHILD_TEXT_FIELD ].height - 
                                  total_height ) / 2;
  }
  
  /* Let the scales do with the width that is left. */
  if( ( sizes[ XmUbTS_CHILD_TEXT_FIELD ].width + 2 * tsl -> sl.margin_width ) >
      tsl -> core.width ){
    scale_width = 1;
    scale_x_pos = 0;

  } else {

    scale_width = tsl -> core.width - sizes[ XmUbTS_CHILD_TEXT_FIELD ].width -
      2 * tsl -> sl.margin_width;
    scale_x_pos = tsl -> sl.margin_width;
  }

  y_pos = tsl -> sl.margin_height + scale_y_offset;
  for( index = XmUbTS_CHILD_HOUR_SLIDER; 
       index < XmUbTS_CHILD_MINUTE_SLIDER + 1; index ++ ){

    kid = tsl -> sl.internal_children[ index ];

    if( ( kid != NULL ) && XtIsManaged( kid ) ){
      sizes[ index ].width = scale_width;
      sizes[ index ].x     = scale_x_pos;
      sizes[ index ].y     = y_pos;

      y_pos += (Position) ( sizes[ index ].height + tsl -> sl.slider_spacing );
    }
  } /* for */


  /* Configure the children. */
  /* All positions and dimensions are now in the sizes array. */
  for( index = XmUbTS_FIRST_CHILD; index < NO_INTERNAL_CHILDREN; index ++ ){

    kid = tsl -> sl.internal_children[ index ];

    if( ( kid != NULL ) && XtIsManaged( kid ) )
      XtConfigureWidget( kid, sizes[ index ].x, sizes[ index ].y,
        sizes[ index ].width, sizes[ index ].height, 
        kid -> core.border_width );
  }


  return;

} /* DoLayout */


/*----------------------------------------------------------------------*/

static XtGeometryResult
  GeometryManager( Widget            widget,
                   XtWidgetGeometry  *request,
                   XtWidgetGeometry  *reply )
{

  XtWidgetGeometry      own_request;
  Dimension             old_width, old_height;
  Dimension             pref_height;
  Dimension             pref_width;
  KidDimensionRec       kids_sizes[ NO_INTERNAL_CHILDREN ];
  XtGeometryResult      result;
  XmUbTimeSliderWidget  tsl;

  /* Code. */

  tsl = (XmUbTimeSliderWidget) XtParent( widget );

  /* Find out how big the widget would be if the resize were allowed. */
  GetChildPrefSizes( tsl, NULL, request, kids_sizes );
  GetOwnPreferredSize( tsl, kids_sizes, &pref_width, &pref_height );

  /* If no change in dimensions, allow the request. */
  if( ( pref_width == tsl -> core.width ) && 
      ( pref_height == tsl -> core.height )){
    DoLayout( tsl, widget, request, kids_sizes );
    return XtGeometryYes;
  }

  /* We must ask our parent to resize us. */
  own_request.request_mode = CWWidth | CWHeight;
  own_request.width  = pref_width;
  own_request.height = pref_height;

  /* Save dimensions. */
  old_width  = tsl -> core.width;
  old_height = tsl -> core.height;

  tsl -> sl.resize_called = False;

  /* We are not interested in any compromise geometry. */
  result = XtMakeGeometryRequest( (Widget) tsl, &own_request, NULL );

  /* Reset to old dimensions if request not granted. */
  if( result != XtGeometryYes ){
    tsl -> core.width  = old_width;
    tsl -> core.height = old_height;

  } else {
    if( !tsl -> sl.resize_called )
      Resize( (Widget) tsl );
  }

  /* Always grant child's request. */
  return XtGeometryYes;

} /* GeometryManager */


/*----------------------------------------------------------------------*/

static void
  GetChildPrefSizes( XmUbTimeSliderWidget  tsl,
                     Widget                initiator,
                     XtWidgetGeometry      *request,
                     KidDimensionRec       sizes[] )
{
  /* Variables. */
  XtWidgetGeometry  desired;
  int               index;
  Widget            kid;

  /* Code. */

  /* Initialize. */
  for( index = 0; index < NO_INTERNAL_CHILDREN; index ++ ){
    sizes[ index ].width  = 0;
    sizes[ index ].height = 0;
    sizes[ index ].x      = 0;
    sizes[ index ].y      = 0;
  }

  /* Get the preferred sizes for the children. */
  for( index = XmUbTS_FIRST_CHILD; index < NO_INTERNAL_CHILDREN; index ++ ){

    kid = tsl -> sl.internal_children[ index ];

    if( ( kid != NULL ) && XtIsManaged( kid ) ){

      KidsPreferredGeometry( kid, initiator, request, &desired );

      sizes[ index ].width  = desired.width;
      sizes[ index ].height = desired.height;

    }

  } /* for */

  
  return;

} /* GetChildPrefSizes */


/*----------------------------------------------------------------------*/

static void
  GetOwnPreferredSize( XmUbTimeSliderWidget  tsl,
                       KidDimensionRec       sizes[],
                       Dimension             *pref_width,
                       Dimension             *pref_height )
{
  /* Variables. */
  int        index;
  Widget     kid;
  Dimension  max_scale_width = 0;
  int        no_managed = 0;
  Dimension  total_scale_height = 0;

  /* Code. */

  /* Check out the dimension of the scales. */
  for( index = XmUbTS_CHILD_HOUR_SLIDER; 
       index < XmUbTS_CHILD_MINUTE_SLIDER + 1; index ++ ){

    kid = tsl -> sl.internal_children[ index ];

    if( ( kid != NULL ) && XtIsManaged( kid ) ){

      total_scale_height += sizes[ index ].height;

      if( sizes[ index ].width > max_scale_width )
        max_scale_width = sizes[ index ].width;

      no_managed ++;
    }
  } /* for */

  if( no_managed > 1 )
    total_scale_height += tsl -> sl.slider_spacing;

  /* We may not be allowed to request the size the children would like. */
  if( tsl -> sl.recompute_width )
    *pref_width  = max_scale_width + sizes[ XmUbTS_CHILD_TEXT_FIELD ].width +
                   2 * tsl -> sl.margin_width + tsl -> sl.text_spacing;
  else 
    *pref_width = tsl -> core.width;

  if( tsl -> sl.recompute_height )
    *pref_height = MAX( total_scale_height, 
                        sizes[ XmUbTS_CHILD_TEXT_FIELD ].height ) +
                   2 * tsl -> sl.margin_height;
  else
    *pref_height = tsl -> core.height;

  
  return;

} /* GetOwnPreferredSize */


/*----------------------------------------------------------------------*/

static void
  Initialize( Widget     treq,
              Widget     tnew,
              ArgList    args,
              Cardinal   *num_args )
{
  /* Variables. */
  int                   index;
  XmUbTimeSliderWidget  new;

  /* Code. */

  new = (XmUbTimeSliderWidget) tnew;

  /* Initialize private fields. */
  for( index = XmUbTS_FIRST_CHILD; index < NO_INTERNAL_CHILDREN; index ++ )
    new -> sl.internal_children[ index ] = NULL;

  /* Start size if none supplied. */
  if( new -> core.width == 0 )
    new -> core.width = 300;
  else
    new -> sl.recompute_width = False;

  if( new -> core.height == 0 )
    new -> core.height = 100;
  else
    new -> sl.recompute_height = False;

  /* Create the internal widgets. */
  new -> sl.internal_widgets_created = False;

  CreateInternalWidgets( new );

  new -> sl.internal_widgets_created = True;

  /* Set the initial value in the text field. */
  UpdateTextField( new );


  return;

} /* Initialize */


/*----------------------------------------------------------------------*/

static void
  InsertChild( Widget  widget )
{

  /* Variables. */
  Cardinal              num_params;
  String                params[ 1 ];
  XmUbTimeSliderWidget  tsl;

  /* Code. */

  tsl = (XmUbTimeSliderWidget) XtParent( widget );

  /* We do not allow the application to create children. */
  if( tsl -> sl.internal_widgets_created ){

    params[ 0 ] = XtClass( (Widget) tsl ) -> core_class.class_name;
    num_params  = 1;
    XtAppErrorMsg( XtWidgetToApplicationContext( widget ),
                   "childError", "number", "WidgetError",
                   "Applications cannot add children to %s widgets.",
                   params, &num_params );
  }

  /* This is an internal child. Adding it is OK. */
  (* ( (CompositeWidgetClass) (xmUbTimeSliderWidgetClass ->
     core_class.superclass) ) -> composite_class.insert_child ) ( widget );


  return;

} /* InsertChild */


/*----------------------------------------------------------------------*/

static void 
  KidsPreferredGeometry( Widget            kid,
                         Widget            initiator,
                         XtWidgetGeometry  *request,
                         XtWidgetGeometry  *desired )
{

  /* Code. */

  if( ( kid == initiator ) && ( request != NULL ) ){
    /* The initiator should not be queried. */
    if( request -> request_mode & CWWidth )
      desired -> width = request -> width;
    else
      desired -> width = (Dimension) kid -> core.width;

    if( request -> request_mode & CWHeight )
      desired -> height = request -> height;
    else
      desired -> height = (Dimension) kid -> core.height;

  } else
    (void) XtQueryGeometry( kid, NULL, desired );


  return;

} /* KidsPreferredGeometry */


/*----------------------------------------------------------------------*/

static void
  Resize( Widget    widget )
{
  /* Variables. */
  KidDimensionRec       kids_sizes[ NO_INTERNAL_CHILDREN ];
  XmUbTimeSliderWidget  tsl;

  /* Code. */

  tsl = (XmUbTimeSliderWidget) widget;

  tsl -> sl.resize_called = True;

  /* We have to get the preferred size of the children before we organize
     the layout. */
  GetChildPrefSizes( tsl, NULL, NULL, kids_sizes );
  DoLayout( tsl, NULL, NULL, kids_sizes );


  return;

} /* Resize */


/*----------------------------------------------------------------------*/

static Boolean
  ResizeIfNeeded( XmUbTimeSliderWidget  tsl,
                  KidDimensionRec       sizes[] )
{

  /* Variables. */
  Boolean           layout_done;
  Dimension         pref_height;
  Dimension         pref_width;
  XtWidgetGeometry  request;
  XtGeometryResult  result;

  /* Code. */

  /* Initialize. */
  layout_done = False;

  /* Get the preferred dimensions of the time box widget. */
  GetChildPrefSizes( tsl, NULL, NULL, sizes );
  GetOwnPreferredSize( tsl, sizes, &pref_width, &pref_height );

  /* If we want the same dimensions, no resizing is needed. */
  if(( pref_width  == tsl -> core.width ) &&
     ( pref_height == tsl -> core.height ))
    return False;

  /* Dimensions are different. Try to resize. */
  request.request_mode = CWWidth | CWHeight;

  request.width  = pref_width;
  request.height = pref_height;

  tsl -> sl.resize_called = False;

  do {

    result = XtMakeGeometryRequest( (Widget) tsl, &request, &request );

  } while( result == XtGeometryAlmost );

  if( result == XtGeometryNo )
    return False;

  /* Resize done. Core fields have already been updated. */
  return( tsl -> sl.resize_called );

} /* ResizeIfNeeded */


/*----------------------------------------------------------------------*/

static XtGeometryResult
  QueryGeometry( Widget             widget,
                 XtWidgetGeometry  *proposed,
                 XtWidgetGeometry  *answer )
{

  KidDimensionRec       kids_sizes[ NO_INTERNAL_CHILDREN ];
  Dimension             pref_height;
  Dimension             pref_width;
  XmUbTimeSliderWidget  tsl;

  /* Code. */


  tsl = (XmUbTimeSliderWidget) widget;

  /* Get dimensions that we *really* want. */
  GetChildPrefSizes( tsl, NULL, NULL, kids_sizes );
  GetOwnPreferredSize( tsl, kids_sizes, &pref_width, &pref_height );

  answer -> request_mode = CWWidth | CWHeight;
  answer -> width  = pref_width;
  answer -> height = pref_height;


  if( proposed == NULL ){
    /* This is a query for the requested geometry. */

    if(( answer -> height == (int) tsl -> core.height ) &&
       ( answer -> width  == (int) tsl -> core.width ))
      return XtGeometryNo;
    else
      return XtGeometryAlmost;
  }

  /* The parent supplied a geometry suggestion. */
  if(( answer -> height == proposed -> height ) &&
     ( answer -> width  == proposed -> width ))
    return XtGeometryYes;

  if( ( proposed -> height <= 1 ) ||
      ( proposed -> width  <= 1 ) )
    /* That's too small ! */
    return XtGeometryNo;


  /* Only a compromise left. */
  return XtGeometryAlmost;

} /* QueryGeometry */


/*----------------------------------------------------------------------*/

static void
  ScaleValueChangedCB( Widget                 scale,
                       XmUbTimeSliderWidget   tsl,
                       XmScaleCallbackStruct  *call_data )
{
  /* Code. */

  UpdateTextField( tsl );


  return;

} /* ScaleValueChangedCB */


/*----------------------------------------------------------------------*/

static void
  SelectAllTextCB( Widget                tw,
                   XmUbTimeSliderWidget  tsl,
                   XmAnyCallbackStruct   *call_data )
{
  /* Variables. */
  XmTextPosition  last;

  /* Code. */

  /* Select all text in the text widget. */

  last = XmTextGetLastPosition( tw );

  if( last > 0 )
    XmTextSetSelection( tw, 0, last, CurrentTime );

  /* Set the insertion position to the start of the widget. */
  XmTextSetInsertionPosition( tw, 0 );


  return;

} /* SelectAllTextCB */


/*----------------------------------------------------------------------*/

static Boolean 
  SetValues( Widget     current,
             Widget     request,
             Widget     new,
             ArgList    args,
             Cardinal   *num_args )
{
#ifdef Differs
#undef Differs
#endif

#define Differs( field )  ( curW -> field != newW -> field )

  /* Variables. */
  XmUbTimeSliderWidget  curW;
  Arg                   local_args[ 2 ];
  Cardinal              n;
  XmUbTimeSliderWidget  newW;
  char                  *newstr;
  Boolean               redisplay = False;
  Boolean               visual_changed = False;

  /* Code. */

  curW = (XmUbTimeSliderWidget) current;
  newW = (XmUbTimeSliderWidget) new;

  /* Width and height. */
  /* Resizing is handled higher up. */

  if( Differs( sl.recompute_width ) && newW -> sl.recompute_width )
    visual_changed = True;
  else if( Differs( core.width ) )
    newW -> sl.recompute_width = False;

  if( Differs( sl.recompute_height ) && newW -> sl.recompute_height )
    visual_changed = True;
  else if( Differs( core.height ) )
    newW -> sl.recompute_height = False;

  /* Margins and spacings. */
  if( Differs( sl.slider_spacing ) ||
      Differs( sl.text_spacing )   ||
      Differs( sl.margin_width )   ||
      Differs( sl.margin_height ) )
    visual_changed = True;

  /* These changes affect the child widgets. */
  if( Differs( manager.foreground ) ||
      Differs( core.background_pixel ) ){

    n = 0;

    if( Differs( manager.foreground ) ){
      XtSetArg( local_args[ n ], XmNforeground, 
                                 newW -> manager.foreground ); n++;
    }
    if( Differs( core.background_pixel ) ){
      XtSetArg( local_args[ n ], XmNbackground, 
                                 newW -> core.background_pixel ); n++;
    }

    XtSetValues( newW -> sl.internal_children[ XmUbTS_CHILD_HOUR_SLIDER ],
                 local_args, n );
    XtSetValues( newW -> sl.internal_children[ XmUbTS_CHILD_MINUTE_SLIDER ],
                 local_args, n );
    XtSetValues( newW -> sl.internal_children[ XmUbTS_CHILD_TEXT_FIELD ],
                 local_args, n );
  }


  if( visual_changed ){

    KidDimensionRec  kids_sizes[ NO_INTERNAL_CHILDREN ];
    Boolean          resized;

    /* Code. */

    resized = ResizeIfNeeded( newW, kids_sizes );

    /* If the widget was resized, the layout has already been done. */
    if( !resized )
      DoLayout( newW, NULL, NULL, kids_sizes );

  }

  return( visual_changed );

#undef Differs

} /* SetValues */


/*----------------------------------------------------------------------*/

static void
  TextChangedCB( Widget                textw,
                 XmUbTimeSliderWidget  tsl,
                 XmAnyCallbackStruct   *call_data )
{
  /* Variables. */
  int              hour;
  int              minute;
  TIM_STATUS_TYPE  status;
  String           str;
  TIM_TIME_REF     time;

  /* Code. */

  /* Get the current text string. */
  str = XmTextGetString( 
          XmUbTimeSliderGetChild( (Widget) tsl, XmUbTS_CHILD_TEXT_FIELD ) );

  /* Get the hour and minute values. */
  status = TimMakeTimeFromString( &time, str );

  if( status == TIM_OK ){

    hour   = TimHour( time );
    minute = TimMinute( time );

    UpdateScales( tsl, hour, minute );  
  } 

  /* Cleanup.*/
  XtFree( str );


  return;

} /* TextChangedCB */


/*----------------------------------------------------------------------*/

static void
  UpdateScales( XmUbTimeSliderWidget  tsl,
                int                   hour,
                int                   minute )
{
  /* Variables. */
  Widget  scale;

  /* Code. */

  /* Set the scale values. */
  scale = XmUbTimeSliderGetChild( (Widget) tsl, XmUbTS_CHILD_HOUR_SLIDER );
  if( scale != NULL )
    XmScaleSetValue( scale, hour );

  scale = XmUbTimeSliderGetChild( (Widget) tsl, XmUbTS_CHILD_MINUTE_SLIDER );
  if( scale != NULL )
    XmScaleSetValue( scale, minute );


  return;

} /* UpdateScales */


/*----------------------------------------------------------------------*/

static void
  UpdateTextField( XmUbTimeSliderWidget  tsl )
{
  /* Variables. */
  int           hour = 0;
  int           minute = 0;
  char          text_buffer[ TEXT_BUFFER_LENGTH ];
  TIM_TIME_REF  time;
  Widget        tmp_widget;

  /* Code. */

  /* Get the value from the sliders one at a time. */

  tmp_widget = XmUbTimeSliderGetChild( (Widget) tsl, 
                                       XmUbTS_CHILD_HOUR_SLIDER );
  if( tmp_widget != NULL )
    XmScaleGetValue( tmp_widget, &hour );

  tmp_widget = XmUbTimeSliderGetChild( (Widget) tsl, 
                                       XmUbTS_CHILD_MINUTE_SLIDER );
  if( tmp_widget != NULL )
    XmScaleGetValue( tmp_widget, &minute );


  /* Create a time string to display in the text field. */
  time = TimMakeTime( 1990, 1, 1, hour, minute, 0 );
  TimFormatTime( time, text_buffer, TEXT_BUFFER_LENGTH );

  XmTextSetString( 
    XmUbTimeSliderGetChild( (Widget) tsl, XmUbTS_CHILD_TEXT_FIELD ), 
    text_buffer );

 
  return;

} /* UpdateTextField */


/*----------------------------------------------------------------------*/

static void
  ValueChangedCB( Widget                tw,
                  XmUbTimeSliderWidget  tsl,
                  XmAnyCallbackStruct   *call_data )
{
  /* Variables. */
  XmUbTimeSliderCallbackStruct  cb;

  /* Code. */

  if( tsl -> sl.value_changed_callback == NULL )
    return;

  /* Set up callback structure. */

  cb.reason      = XmCR_VALUE_CHANGED;
  cb.event       = call_data -> event;
  cb.child_index = ChildIndex( tsl, tw );
  cb.child       = tw;

  XtCallCallbackList( (Widget) tsl, tsl -> sl.value_changed_callback,
                      (XtPointer) &cb );


  return;

} /* ValueChangedCB */


/*----------------------------------------------------------------------*/

static void
  WarningNoResourceChange( XmUbTimeSliderWidget  tsl,
                           String                resource )
{
  /* Variables. */
  Cardinal  num_params;
  String    params[ 2 ];

  /* Code. */

  params[ 0 ] = resource;
  params[ 1 ] = XtClass( tsl ) -> core_class.class_name;
  num_params  = 2;
  XtAppWarningMsg( XtWidgetToApplicationContext( (Widget) tsl ),
		   "resourceError", "setValues", "WidgetError",
		   "Resource %s may not be changed in %s widgets.",
		   params, &num_params );

  return;

} /* WarningNoResourceChange */


/*----------------------------------------------------------------------*/

Widget  
  XmUbCreateTimeSlider( Widget    parent,
                        String    name,
                        ArgList   arglist,
                        Cardinal  argcount )
{

  /* Code. */

  return XtCreateWidget( name, xmUbTimeSliderWidgetClass, 
                         parent, arglist, argcount );

} /* XmUbCreateTimeSlider */


/*----------------------------------------------------------------------*/

Widget 
  XmUbTimeSliderGetChild( Widget  widget,
                          int     child )
{

  /* Variables. */
  XmUbTimeSliderWidget  tsl;

  /* Code. */

  tsl = (XmUbTimeSliderWidget) widget;


  return( tsl -> sl.internal_children[ child ] );

} /* XmUbTimeSliderGetChild */


/*----------------------------------------------------------------------*/

time_t
  XmUbTimeSliderGetTime( Widget  widget )
{
  /* Variables. */
  Arg           args[ 1 ];
  int           hour = 0;
  int           minute = 0;
  Cardinal      n;
  TIM_TIME_REF  time;
  Widget        tmp_widget;

  /* Code. */

  /* Get the value from the sliders one at a time. */

  tmp_widget = XmUbTimeSliderGetChild( widget, XmUbTS_CHILD_HOUR_SLIDER );
  if( tmp_widget != NULL )
    XmScaleGetValue( tmp_widget, &hour );

  tmp_widget = XmUbTimeSliderGetChild( widget, XmUbTS_CHILD_MINUTE_SLIDER );
  if( tmp_widget != NULL )
    XmScaleGetValue( tmp_widget, &minute );

  /* Create the time from the fetched values. */
  time = TimMakeTime( 1970, 1, 1, hour, minute, 0 );


  return time;

} /* XmUbTimeSliderGetTime */


/*----------------------------------------------------------------------*/

char
  *XmUbTimeSliderGetTimeString( Widget  widget )
{
  /* Variables. */
  Widget  textw;

  /* Code. */
  textw = XmUbTimeSliderGetChild( widget, XmUbTS_CHILD_TEXT_FIELD );
  if( textw == NULL )
    return( NULL );


  return( XmTextGetString( textw ) );

} /* XmUbTimeSliderGetTimeString */


/*----------------------------------------------------------------------*/

void
  XmUbTimeSliderSetTime( Widget  widget,
                         time_t  time )
{
  /* Variables. */
  int                   hour;
  int                   minute;
  XmUbTimeSliderWidget  tsl;

  /* Code. */

  tsl = (XmUbTimeSliderWidget) widget;

  hour   = TimHour( time );
  minute = TimMinute( time );

  UpdateScales( tsl, hour, minute );

  /* If the widget is not mapped, the value changed callback will 
     not be called. */
  UpdateTextField( tsl );

  return;

} /* XmUbTimeSliderSetTime */
