/*----------------------------------------------------------------------------
--
--  Module:           xtmInitDb
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Create a complete XDiary database in the directory given. If a database
--    already exists but is incomplete, the missing parts are added.
--
--    Usage:
--      XDinitDb [dbDir]
--
--    where:
--      dbDir  : directory to contain the XDiary database.
--
--  Filename:         xtmInitDb.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-09-21
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmInitDb.c, Version: 1.1, Date: 95/02/18 16:06:32";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "System.h"
#include "xtmDbTools.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Name of program. */
#define PROGRAM_NAME   "xdiary"


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of program. */
static char *program_name;

/* Name of module. */
static char  *module_name = "xtmInitDb";



/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  displayUsage();

static int
  noCaseStrcmp( char  *buffer1,
                char  *buffer2 );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

main( int argc, char *argv[] )
{

  /* Variables. */
  Boolean                do_file_lock = True;
  int                    arg_ref;
  int                    status;
  char                   buffer[ 100 ];
  char                   directory[ 200 ];
  char                   filename[ 200 ];
  char                   *char_ref;
  XTM_DB_CREATE_REQUEST  create_request;
  XTM_DB_STATUS          db_status;
  struct passwd          *password_ref;
    

  /* Code. */

  /* Fetch the name of the program. */
  program_name = PROGRAM_NAME;

  
  /* Initialization. */
  SysInitializeEnvironment();

  
  /* Fetch qualifiers. */
  arg_ref = 1;
  while( arg_ref < argc && argv[ arg_ref ][ 0 ] == '-' ) {

    if( noCaseStrcmp( argv[ arg_ref ], "-usage" ) == 0 ) {
      displayUsage();
      exit( 0 );

    } else if( noCaseStrcmp( argv[ arg_ref ], "-help" ) == 0 ) {
      displayUsage();
      exit( 0 );

    } else if( noCaseStrcmp( argv[ arg_ref ], "-h" ) == 0 ) {
      displayUsage();
      exit( 0 );

    } else if( noCaseStrcmp( argv[ arg_ref ], "-nofilelock" ) == 0 ) {
      do_file_lock = False;

    } else if( noCaseStrcmp( argv[ arg_ref ], "-version" ) == 0 ) {
      printf( "%s: Version: %s\n", program_name, VERSION_ID );
      exit( 0 );

    } /* if */

    arg_ref++;

  } /* while */

  /* Check arguments. */
  if( (argc - arg_ref) < 1 ) {
    char_ref = getenv( "HOME" );

    if( char_ref != NULL )
      sprintf( directory, "%s/%s", char_ref, "Diary" );
    else
      sprintf( directory, "./%s", "Diary" );
  } else {
    strcpy( directory, argv[ arg_ref++ ] );
  }


  /* Use file locking? */
  xtmDbUseFileLock( do_file_lock );


  /* Does the main directory exist? */
  sprintf( filename, "%s", directory );

  status = access( filename, (R_OK | W_OK | F_OK) );

  if( status != 0 ) {
    status = mkdir( filename, (S_IRWXU | S_IXGRP | S_IXOTH) );

    if( status != 0 ) {
      fprintf( stderr, "%s: Cannot create directory %s.\n",
               program_name, filename );
      exit( 1 );
    }
  } /* if */


  /* Does the message directory exist? */
  sprintf( filename, "%s/Message", directory );

  status = access( filename, (R_OK | W_OK | F_OK) );

  if( status != 0 ) {
    status = mkdir( filename, (S_IRWXU | S_IXGRP | S_IXOTH) );

    if( status != 0 ) {
      fprintf( stderr, "%s: Cannot create directory %s.\n",
               program_name, filename );
      exit( 1 );
    }
  } /* if */


  /* Does the private directory exist? */
  sprintf( filename, "%s/Private", directory );

  status = access( filename, (R_OK | W_OK | F_OK) );

  if( status != 0 ) {
    status = mkdir( filename, S_IRWXU );

    if( status != 0 ) {
      fprintf( stderr, "%s: Cannot create directory %s.\n",
               program_name, filename );
      exit( 1 );
    }
  } /* if */


  /* Initialize the database(s) (create if not exist). */
  create_request.directory = directory;

  create_request.database = XTM_DB_ENTRY_DB;
  db_status = xtmDbCreateDatabase( &create_request );

  if( db_status != XTM_DB_OK ) {
    fprintf( stderr, "%s: Cannot create database %s.\n",
             program_name, XTM_DB_ENTRY_DB );

    exit( 1 );
  }

  create_request.database = XTM_DB_DATE_DB;
  db_status = xtmDbCreateDatabase( &create_request );

  if( db_status != XTM_DB_OK ) {
    fprintf( stderr, "%s: Cannot create database %s.\n",
             program_name, XTM_DB_DATE_DB );

    exit( 1 );
  }

  create_request.database = XTM_DB_STAND_ENTRY_DB;
  db_status = xtmDbCreateDatabase( &create_request );

  if( db_status != XTM_DB_OK ) {
    fprintf( stderr, "%s: Cannot create database %s.\n",
             program_name, XTM_DB_STAND_ENTRY_DB );

    exit( 1 );
  }

  create_request.database = XTM_DB_PRIV_ENTRY_DB;
  db_status = xtmDbCreateDatabase( &create_request );

  if( db_status != XTM_DB_OK ) {
    fprintf( stderr, "%s: Cannot create database %s.\n",
             program_name, XTM_DB_PRIV_ENTRY_DB );

    exit( 1 );
  }


  /* Create the support files. */
  db_status = xtmDbInitializeAuxFiles( directory );

  if( db_status != XTM_DB_OK ) {
    fprintf( stderr, "%s: Cannot create support files in %s.\n",
             program_name, XTM_DB_PRIV_ENTRY_DB );

    exit( 1 );
  }


  /* Fetch username. */
  password_ref = getpwuid( getuid() );
  if( password_ref != NULL )
    strcpy( buffer, password_ref -> pw_name );
  else
    strcpy( buffer, "<user>" );

  printf( 
    "\n"
    "The directory %s now contains a complete\n"
    "XDiary (%s) database.\n"
    "\n"
    "If you are using the Andrew File System (AFS), execute the following\n"
    "command to restrict access to the diary database:\n"
    " $ find %s -type d -exec fs setacl -clear {} %s all ';'\n"
    "\n",
    directory, VERSION_ID, directory, buffer );

  exit( 0 );

} /* main */


/*----------------------------------------------------------------------*/

static void
  displayUsage()
{

  printf( 
    "\n"
    "%s (%s): Initialize a diary database.\n"
    "\n"
    "Create a complete XDiary database in the directory given.\n"
    "If a database already exists but is incomplete, the missing parts\n"
    "are added.\n"
    "\n"
    "Usage:\n"
    "  %s dbDir [flags]\n"
    "\n"
    "where:\n"
    "  dbDir  : Directory containg XDiary database.\n"
    "\n"
    "Flags:\n"
    "  -help            : Display this help.\n"
    "  -h               : See above.\n"
    "  -noFileLock      : Don't use any file locking.\n"
    "  -usage           : Display this help.\n"
    "  -version         : Display the current version.\n"
    "\n",
    program_name, VERSION_ID, program_name );

  return;

} /* displayUsage */


/*----------------------------------------------------------------------*/

static int
  noCaseStrcmp( char  *buffer1,
                char  *buffer2 )
{

  /* Variables. */
  char  *char_ref1;
  char  *char_ref2;


  /* Code. */

  if( strlen( buffer1 ) != strlen( buffer2 ) )
    return( strcmp( buffer1, buffer2 ) );

  char_ref1 = buffer1;
  char_ref2 = buffer2;

  while( *char_ref1 != '\0' ) {
    if( tolower( *char_ref1 ) < tolower( *char_ref2 ) )
      return( -1 );
    else if( tolower( *char_ref1 ) > tolower( *char_ref2 ) )
      return( 1 );

    char_ref1++;
    char_ref2++;
  }

  return( 0 );

} /* noCaseStrcmp */
