/*----------------------------------------------------------------------------
--
--  Module:           xtmAfsAclBs
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Base functions for ACLs in AFS.
--
--  Filename:         xtmAfsAclBs.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-10-15
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmAfsAclBs.c, Version: 1.1, Date: 95/02/18 15:58:53";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include <afs/cellconfig.h>
#include <afs/ptint.h>
#include <afs/venus.h>

#include "System.h"
#include "LstLinked.h"

#include "xtmAfsAclBs.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmAfsAclBase";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_AF_STATUS
  xtmAfInitialize()
{

  /* Variables. */
  int  status;


  /* Code. */

  /* Initialize connection to server. */
  status = pr_Initialize( 1, AFSCONF_CLIENTNAME, 0 );

  if( status != 0 )
    return( XTM_AF_ERROR );


  return( XTM_AF_OK );

} /* xtmAfInitialize */


/*----------------------------------------------------------------------*/

XTM_AF_STATUS
  xtmAfAccess( char    *filename,
               UINT32  rights )
{

  /* Variables. */
  int               status;
  struct ViceIoctl  io_record;


  /* Code. */

  /* Can the caller access the file as specified. */
  io_record.in       = (char *) &rights;
  io_record.in_size  = sizeof( UINT32 );
  io_record.out      = NULL;
  io_record.out_size = 0;

  status = pioctl( filename, VIOCACCESS, &io_record, 0 );
  if( status != 0 )
    return( XTM_AF_ERROR );


  return( XTM_AF_OK );

} /* xtmAfAccess */


/*----------------------------------------------------------------------*/

void
  xtmAfAclFlagToString( UINT32  flags,
                        char    *acl_string )
{

  /* Variables. */
  *acl_string = '\0';


  /* Code. */

  if( flagIsSet( flags, XTM_AF_FLAG_READ ) )
    strcat( acl_string, "r" );

  if( flagIsSet( flags, XTM_AF_FLAG_LOOKUP ) )
    strcat( acl_string, "l" );

  if( flagIsSet( flags, XTM_AF_FLAG_INSERT ) )
    strcat( acl_string, "i" );

  if( flagIsSet( flags, XTM_AF_FLAG_DELETE ) )
    strcat( acl_string, "d" );

  if( flagIsSet( flags, XTM_AF_FLAG_WRITE ) )
    strcat( acl_string, "w" );

  if( flagIsSet( flags, XTM_AF_FLAG_LOCK ) )
    strcat( acl_string, "k" );

  if( flagIsSet( flags, XTM_AF_FLAG_ADMINISTER ) )
    strcat( acl_string, "a" );


  return;

} /* xtmAfAclFlagToString */


/*----------------------------------------------------------------------*/

UINT32
  xtmAfAclStringToFlag( char  *acl_string )
{

  /* Variables. */
  UINT32  flags = 0;


  /* Code. */

  if( strchr( acl_string, 'r' ) != NULL )
    flagSet( flags, XTM_AF_FLAG_READ );

  if( strchr( acl_string, 'l' ) != NULL )
    flagSet( flags, XTM_AF_FLAG_LOOKUP );

  if( strchr( acl_string, 'i' ) != NULL )
    flagSet( flags, XTM_AF_FLAG_INSERT );

  if( strchr( acl_string, 'd' ) != NULL )
    flagSet( flags, XTM_AF_FLAG_DELETE );

  if( strchr( acl_string, 'w' ) != NULL )
    flagSet( flags, XTM_AF_FLAG_WRITE );

  if( strchr( acl_string, 'k' ) != NULL )
    flagSet( flags, XTM_AF_FLAG_LOCK );

  if( strchr( acl_string, 'a' ) != NULL )
    flagSet( flags, XTM_AF_FLAG_ADMINISTER );


  return( flags );

} /* xtmAfAclStringToFlag */


/*----------------------------------------------------------------------*/

int
  xtmAfFetchId( char  *long_id )
{

  /* Variables. */
  int       id;
  int       ids[ 10 ];
  int       status;
  idlist    list_ids;
  namelist  list_name;


  /* Code. */

  list_ids.idlist_len = 0;
  list_ids.idlist_val = 0;

  list_name.namelist_val = (prname *) SysMalloc( 1 * PR_MAXNAMELEN );
  list_name.namelist_len = 1;

  strcpy( list_name.namelist_val[ 0 ], long_id );

  /* Fetch ID for this user. */
  status = pr_NameToId( &list_name, &list_ids );

  SysFree( list_name.namelist_val );

  id = list_ids.idlist_val[ 0 ];
  if( list_ids.idlist_val != NULL )
    SysFree( list_ids.idlist_val );

  if( status != 0 )
    return( XTM_AF_NOID );

  return( id );

} /* xtmAfFetchId */


/*----------------------------------------------------------------------*/

XTM_AF_STATUS
  xtmAfIsAfsFile( char  *filename )
{

  /* Variables. */
  int               rights;
  int               status;
  struct ViceIoctl  io_record;


  /* Code. */

  /* Make a simple read check. */
  rights = XTM_AF_FLAG_READ;

  io_record.in       = (char *) &rights;
  io_record.in_size  = sizeof( UINT32 );
  io_record.out      = NULL;
  io_record.out_size = 0;

  status = pioctl( filename, VIOCACCESS, &io_record, 0 );
  if( status != 0 && status != 0 )
    return( XTM_AF_ERROR );


  return( XTM_AF_OK );

} /* xtmAfIsAfsFile */


/*----------------------------------------------------------------------*/

XTM_AF_STATUS
  xtmAfIsAfsMountPoint( char  *dir_path,
                        char  *file_name )
{

  /* Variables. */
  int               status;
  char              buffer[ PATH_MAX + 1 ];
  struct ViceIoctl  io_record;


  /* Code. */

  /* Make a simple mount point check. */
  io_record.in       = file_name;
  io_record.in_size  = strlen( file_name ) + 1;
  io_record.out      = buffer;
  io_record.out_size = PATH_MAX;

  status = pioctl( dir_path, VIOC_AFS_STAT_MT_PT, &io_record, 1 );
  if( status != 0 )
    return( XTM_AF_ERROR );


  return( XTM_AF_OK );

} /* xtmAfIsAfsMountPoint */


/*----------------------------------------------------------------------*/

XTM_AF_STATUS
  xtmAfGetAclForFile( char           *filename,
                      LST_DESC_TYPE  *acl_list )
{

  /* Variables. */
  int               char_read;
  int               index;
  int               no_negative_acl;
  int               no_positive_acl;
  int               rights;
  int               status;
  char              acl[ 5000 ];
  char              id[ 1000 ];
  char              *char_ref;
  LST_STATUS        lst_status;
  XTM_AF_ACL_INFO   acl_info;
  struct ViceIoctl  io_record;


  /* Code. */

  *acl_list = NULL;

  /* Go and get the acl. */
  io_record.in       = NULL;
  io_record.in_size  = 0;
  io_record.out      = acl;
  io_record.out_size = sizeof( acl );

  status = pioctl( filename, VIOCGETAL, &io_record, 0 );
  if( status != 0 )
    return( XTM_AF_ERROR );


  *acl_list = LstLinkNew( sizeof( XTM_AF_ACL_INFO ), NULL );

  /* Nomber of entries. */
  char_read = strlen( acl );
  sscanf( acl, "%d %d%n", &no_positive_acl, &no_negative_acl, &char_read );
  char_ref = &acl[ char_read ];


  /* The positive rights. */
  for( index = 0; index < no_positive_acl; index++ ) {
    char_read = strlen( char_ref );
    sscanf( char_ref, "%s %d%n", id, &rights, &char_read );

    /* Check the length of the id. */
    if( strlen( id ) < XTM_AF_MAX_ID_LENGTH ) {

      /* Save the information. */
      acl_info.positive = True;
      acl_info.rights   = rights;

      strcpy( acl_info.id, id );

      lst_status = LstLinkInsertLast( *acl_list, &acl_info );

    } /* if */

    char_ref = char_ref + char_read;
  } /* loop */


  /* The negative rights. */
  for( index = 0; index < no_negative_acl; index++ ) {
    char_read = strlen( char_ref );
    sscanf( char_ref, "%s %d%n", id, &rights, &char_read );

    /* Check the length of the id. */
    if( strlen( id ) < XTM_AF_MAX_ID_LENGTH ) {

      /* Save the information. */
      acl_info.positive = False;
      acl_info.rights   = rights;

      strcpy( acl_info.id, id );

      lst_status = LstLinkInsertLast( *acl_list, &acl_info );

    } /* if */

    char_ref = char_ref + char_read;
  } /* loop */

  return( XTM_AF_OK );

} /* xtmAfGetAclForFile */


/*----------------------------------------------------------------------*/

XTM_AF_STATUS
  xtmAfSetAclForFile( char           *filename,
                      LST_DESC_TYPE  acl_list )
{

  /* Variables. */
  int               acl_minus_count = 0;
  int               acl_plus_count = 0;
  int               status;
  char              acl[ 5000 ];
  char              acl_minus[ 5000 ];
  char              acl_plus[ 5000 ];
  char              buffer[ XTM_AF_MAX_ID_LENGTH + 20 ];
  LST_STATUS        lst_status;
  XTM_AF_ACL_INFO   acl_info;
  struct ViceIoctl  io_record;


  /* Code. */

  /* Create the correct AFS ACL string. */
  acl_minus[ 0 ] = '\0';
  acl_plus[  0 ] = '\0';

  lst_status = LstLinkCurrentFirst( acl_list );

  while( lst_status == LST_OK ) {
    lst_status = LstLinkGetCurrent( acl_list, &acl_info );

    /* Positive acl? */
    if( acl_info.positive ) {
      sprintf( buffer, "%s\t%d\n", acl_info.id, acl_info.rights );
      strcat( acl_plus, buffer );

      acl_plus_count++;
    }

    /* Negative acl? */
    if( ! acl_info.positive ) {
      sprintf( buffer, "%s\t%d\n", acl_info.id, acl_info.rights );
      strcat( acl_minus, buffer );

      acl_minus_count++;
    }

    /* Next entry. */
    lst_status = LstLinkCurrentNext( acl_list );
  } /* while */


  /* The complete ACL. */
  sprintf( acl, "%d\n%d\n%s%s", 
           acl_plus_count, acl_minus_count, acl_plus, acl_minus );

  /* Set the acl. */
  io_record.in       = acl;
  io_record.in_size  = strlen( acl ) + 1;
  io_record.out      = NULL;
  io_record.out_size = 0;

  status = pioctl( filename, VIOCSETAL, &io_record, 0 );
  if( status != 0 )
    return( XTM_AF_ERROR );


  return( XTM_AF_OK );

} /* xtmAfSetAclForFile */
