/*----------------------------------------------------------------------------
--
--  Module:           xtmSelHidden
--
--  Project:          XDiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Select hidden (or normal) entry to display.
--
--  Filename:         xtmSelHidden.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1993-08-15
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmSelHidden.c, Version: 1.1, Date: 95/02/18 15:52:48";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>
#include <Xm/List.h>
#include <Xm/RowColumn.h>

#include "System.h"
#include "Message.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmDbMisc.h"
#include "xtmFields.h"
#include "xtmFormat.h"
#include "xtmHelp.h"
#include "xitError.h"
#include "xitFieldSel.h"
#include "xitTools.h"
#include "XmUbTimeB.h"
#include "xtmSelHidden.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the include window. */
#define dateLa             dataLocalW[  0 ]
#define dateRc             dataLocalW[  1 ]
#define dateTb             dataLocalW[  2 ]
#define entryLi            dataLocalW[  3 ]
#define introLa            dataLocalW[  4 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Select record. */
typedef struct {

  /* Entry we have selected. */
  int  entry_selected;

  /* We are selecting from this calendar. */
  char  select_from_cal[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* The select window. */
  Widget  selectW;

  /* Shadow entries. */
  LST_DESC_TYPE  entries;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Callback to inform our creator of specific actions. */
  void              *user_data;
  XTM_SH_ACTION_CB  actionCB;

} SEL_REC, *SEL_REC_REF;


/* Shadow entry record. */
typedef struct {

  /* Entry ID. */
  UINT32  entry_id;

  /* Calendar. */
  char  calendar[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* Date for entry. */
  TIM_TIME_REF  date_stamp;

  /* Time for entry. */
  TIM_TIME_REF  time_stamp;
  
} SHADOW_ENTRY_REC;



/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmSelectHidden";

/* IDs for the help windows. */
static char  *window_id = "SelectHidden";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void
  applyCB( Widget       widget,
           SEL_REC_REF  sel_ref,
           XtPointer    call_data );

static void
  closeCB( Widget       widget,
           SEL_REC_REF  sel_ref,
           XtPointer    call_data );

static Widget
  createSelectWindow( SEL_REC_REF  sel_ref,
                      Widget       parent,
                      Boolean      app_modal );

static void
  destroyCB( Widget       widget,
             SEL_REC_REF  sel_ref,
             XtPointer    call_data );

static void
  editCB( Widget       widget,
          SEL_REC_REF  sel_ref,
          XtPointer    call_data );

static void
  entrySelectCB( Widget                widget,
                 SEL_REC_REF           sel_ref,
                 XmListCallbackStruct  *call_data );

static void 
  helpCB( Widget       widget,
          SEL_REC_REF  sel_ref,
          XtPointer    call_data );

static void
  selectFromThisDate( SEL_REC_REF   sel_ref,
                      TIM_TIME_REF  from_date );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_SH_HANDLE
  xtmShInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent,
                   Boolean               app_modal,
                   XTM_SH_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  SEL_REC_REF  sel_ref;


  /* Code. */

  /* Create and initialize our private data. */
  sel_ref = SysNew( SEL_REC );
  if( sel_ref == NULL )
    return( NULL );

  sel_ref -> appl_data_ref  = appl_data_ref;
  sel_ref -> entries        = LstLinkNew( sizeof( SHADOW_ENTRY_REC ), NULL );
  sel_ref -> entry_selected = 0;
  sel_ref -> actionCB       = actionCB;
  sel_ref -> user_data      = user_data;


  /* Create the select window. */
  sel_ref -> selectW = createSelectWindow( sel_ref, parent, app_modal );
  if( sel_ref -> selectW == NULL ) {
    SysFree( sel_ref );

    return( NULL );
  }


  return( (XTM_SH_HANDLE) sel_ref );

} /* xtmShInitialize */


/*----------------------------------------------------------------------*/

void
  xtmShDestroy( XTM_SH_HANDLE  sel_handle )
{

  /* Variables. */
  SEL_REC_REF  sel_ref;


  /* Code. */

  if( sel_handle == NULL )
    return;

  /* Our private data. */
  sel_ref = (SEL_REC_REF) sel_handle;


  /* Destroy the window. */
  XtDestroyWidget( sel_ref -> selectW );


  return;

} /* xtmShDestroy */


/*----------------------------------------------------------------------*/

void
  xtmShDisplaySelectWindow( XTM_SH_HANDLE  sel_handle,
                            char           *select_from_cal,
                            TIM_TIME_REF   start_at_date )
{

  /* Variables. */
  SEL_REC_REF  sel_ref;


  /* Code. */

  if( sel_handle == NULL )
    return;

  /* Our private data. */
  sel_ref = (SEL_REC_REF) sel_handle;

  /* The calendar to select from. */
  strcpy( sel_ref -> select_from_cal, select_from_cal );

  /* Start selecting from this date. */
  selectFromThisDate( sel_ref, start_at_date );

  /* Make sure the window is displayed. */
  XtManageChild( sel_ref -> selectW );


  return;

} /* xtmShDisplaySelectWindow */


/*----------------------------------------------------------------------*/

static Widget
  createSelectWindow( SEL_REC_REF  sel_ref,
                      Widget       parent,
                      Boolean      app_modal )
{

  /* Variables. */
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  dataLocalW[ 5 ];
  Widget                  selFd;
  Widget                  workFo;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", applyCB, NULL },
    { "", editCB,    NULL },
    { "", closeCB, NULL },
    { "", helpCB,  NULL },
  };


  /* Code. */

  custom_data_ref = sel_ref -> appl_data_ref -> custom_data;

  /* Set message strings. */
  action_buttons[ 0 ].label = msgGetText( MXDI_APPLY_BUTTON );
  action_buttons[ 0 ].data  = sel_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_EDIT_BUTTON );
  action_buttons[ 1 ].data  = sel_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_CLOSE_BUTTON );
  action_buttons[ 2 ].data  = sel_ref;
  action_buttons[ 3 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 3 ].data  = sel_ref;

  /* Create a form dialog with buttons. */
  selFd = xitCreateFormDialog( parent, "SelHidFd",
                               1, 0,
                               action_buttons,
                               XtNumber( action_buttons ) );

  XtAddCallback( selFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) sel_ref );


  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_SEL_HIDDEN_TITLE ) ); n++;
  XtSetValues( XtParent( selFd ), args, n );

  n = 0;
  XtSetArg( args[ n ], XmNdeleteResponse, XmUNMAP ); n++;
  XtSetValues( selFd, args, n );

  if( app_modal ) {
    n = 0;
    XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
    XtSetValues( selFd, args, n );
  }


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( selFd, "SelHidFdFo" );


  /* What are we doing? */
  introLa = xitCreateLabel( workFo, "IntroLa", 
                            msgGetText( MXDI_SEL_HIDDEN_LABEL ), -1 );


  /* List with entries. */
  n = 0;
  XtSetArg( args[ n ], XmNlistSizePolicy, XmCONSTANT ); n++;
  XtSetArg( args[ n ], XmNscrollBarDisplayPolicy, XmSTATIC ); n++;
  XtSetArg( args[ n ], XmNselectionPolicy, XmSINGLE_SELECT ); n++;
  XtSetArg( args[ n ], XmNlistMarginHeight, 5 ); n++;
  XtSetArg( args[ n ], XmNlistMarginWidth, 5 ); n++;
  entryLi = XmCreateScrolledList( workFo, "EntryLi", args, n );

  XtAddCallback( entryLi, XmNsingleSelectionCallback,
                 (XtCallbackProc) entrySelectCB, (XtPointer) sel_ref );
  XtAddCallback( entryLi, XmNdefaultActionCallback,
                 (XtCallbackProc) entrySelectCB, (XtPointer) sel_ref );


  /* Select entries this date. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  dateRc = XmCreateRowColumn( workFo, "DateRc", args, n );

  dateLa = xitCreateLabel( dateRc, "DateLa", 
                           msgGetText( MXDI_DATE_LABEL ), -1 );

  dateTb = xtmFlCreateDateField( dateRc, "DateTb" );
  XtAddCallback( dateTb,  XmNactivateCallback, 
                 (XtCallbackProc) applyCB, (XtPointer) sel_ref );


  /* Put the Parts together. */
  xitAttachWidget( introLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( dateRc,
                   XmATTACH_WIDGET, introLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,    XmATTACH_NONE, NULL );
  xitAttachWidget( XtParent( entryLi ),
                   XmATTACH_WIDGET, dateRc, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,   XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( introLa,             args, n );
  XtSetValues( dateRc,              args, n );
  XtSetValues( XtParent( entryLi ), args, n );

  /* Manage the children. */
  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the initial sizes. */
  xitSetSizeFormDialog( selFd, True );

  /* Make the final attachments. */
  xitAttachWidget( XtParent( entryLi ),
                   XmATTACH_WIDGET, dateRc, XmATTACH_FORM, NULL,
                   XmATTACH_FORM,   NULL,   XmATTACH_FORM, NULL );

  return( selFd );

} /* createSelectWindow */


/*----------------------------------------------------------------------*/

static void
  selectFromThisDate( SEL_REC_REF   sel_ref,
                      TIM_TIME_REF  from_date )
{

  /* Variables. */
  char    buffer[ 50 ];
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( sel_ref -> selectW, "SelHidFdFo" );

  /* Fill in the date we start with. */
  tempW = XtNameToWidget( mainW, "DateRc.DateTb" );
  XmUbTimeBoxSetStartDate( tempW, from_date );

  /* Just as Apply. */
  applyCB( NULL, sel_ref, NULL );


  return;

} /* selectFromThisDate */


/*----------------------------------------------------------------------*/

static void
  applyCB( Widget       widget,
           SEL_REC_REF  sel_ref,
           XtPointer    call_data )
{

  /* Variables. */
  Boolean                 ok;
  int                     index;
  int                     list_index = 0;
  UINT32                  flags;
  char                    buffer[ 100 ];
  char                    entry_line[ 256 ];
  char                    *char_ref;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  mainW;
  Widget                  tempW;
  XmString                list_items[ 500 ];
  LST_DESC_TYPE           entries;
  LST_STATUS              lst_status;
  SHADOW_ENTRY_REC        shadow_entry;
  TIM_TIME_REF            from_date;
  XTM_CD_CAL_INFO         db_info;
  XTM_DB_ALL_ENTRY_DEF    entry_record;
  XTM_DB_STATUS           db_status;
  XTM_DM_FILTER_REC       filter_rec;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  custom_data_ref = sel_ref -> appl_data_ref -> custom_data;

  mainW = XtNameToWidget( sel_ref -> selectW, "SelHidFdFo" );


  /* Fetch the date to display. */
  tempW = XtNameToWidget( mainW, "DateRc.DateTb" );

  ok = xtmFoFetchDate( mainW, tempW, XTM_FO_START_DATE, False, &from_date );
  if( ! ok )
    return;


  /* Calendar to display. */
  (void) xtmCdFetchNamedDb( custom_data_ref -> cal_db_handle, 
                            sel_ref -> select_from_cal,
                            &db_info, NULL );


  /* Empty the shadow list. */
  LstLinkClear( sel_ref -> entries );
  sel_ref -> entry_selected = 0;


  /* Fetch entries for this day. */
  filter_rec.flags = 0;
  flagSet( filter_rec.flags, XTM_DM_FLAG_ALL );

  filter_rec.text_string[ 0 ] = '\0';
  filter_rec.tag_string[  0 ] = '\0';

  flags = XTM_DB_FETCH_ALL_STANDING;

  db_status = xtmDmFetchFilteredEntriesRange( 
                sel_ref -> appl_data_ref,
                sel_ref -> select_from_cal,
                sel_ref -> appl_data_ref -> context, 
                sel_ref -> selectW,
                from_date, from_date, flags, &filter_rec,
                &entries );

  if( db_status != XTM_DB_OK ) {
    xtmDmDeleteEntriesList( entries );

    return;
  }


  /* Assign entries to the list and add to shadow list. */
  lst_status = LstLinkCurrentFirst( entries );
  while( lst_status == LST_OK ) {

    /* Fetch the record. */
    lst_status = LstLinkGetCurrent( entries, &entry_record );


    /* Build the list entry. */    
    entry_line[ 0 ] = '\0';

    /* Note mark or start time. */
    if( entry_record.entry.entry_type == XTM_DB_DAY_NOTE )
      sprintf( buffer, "%s", msgGetText( MXDI_TYPE_NOTE ) );
    else
      xtmFoFormatTime( entry_record.entry.time_stamp,
                       buffer, sizeof( buffer ) );

    strcat( entry_line, buffer );
    strcat( entry_line, "  " );

    /* Calendar. */
    sprintf( buffer, "(%s)", entry_record.db_name );

    strcat( entry_line, buffer );
    strcat( entry_line, "  " );

    /* Entry text */
    if( entry_record.all_text != NULL )
      sprintf( buffer, "%-.50s...", entry_record.all_text );
    else
      sprintf( buffer, "%-.50s...", entry_record.entry.text );

    strcat( entry_line, buffer );


    /* Convert any \n to blanks. */
    char_ref = entry_line;

    while( *char_ref != '\0' ) {
      if( *char_ref == '\n' )
        *char_ref = ' ';
      char_ref++;
    }


    /* Add to list. */
    if( list_index < 500 ) {
      list_items[ list_index ] = XmStringCreate( entry_line, CS );
      list_index++;
    }


    /* Create and save shadow entry. */
    shadow_entry.entry_id   = entry_record.entry.id;
    shadow_entry.date_stamp = from_date;
    shadow_entry.time_stamp = entry_record.entry.time_stamp;
    strcpy( shadow_entry.calendar, entry_record.db_name );

    (void) LstLinkInsertLast( sel_ref -> entries, &shadow_entry );


    /* Next entry. */
    lst_status = LstLinkCurrentNext( entries );

  } /* while */


  /* We don't need this anymore. */
  xtmDmDeleteEntriesList( entries );


  /* Set the entries list. */
  tempW = XtNameToWidget( mainW, "EntryLiSW.EntryLi" );

  n = 0;
  XtSetArg( args[ n ], XmNitems, list_items ); n++;
  XtSetArg( args[ n ], XmNitemCount, list_index ); n++;
  XtSetArg( args[ n ], XmNselectedItemCount, 0 ); n++;
  XtSetValues( tempW, args, n );

  /* Free allocated memory. */
  for( index = 0; index < list_index; index++ )
    XmStringFree( list_items[ index ] );


  /* Call callback function? */
  if( sel_ref -> actionCB != NULL )
    (* sel_ref -> actionCB)( XTM_SH_REASON_APPLY,
                             0, "", (TIM_TIME_REF) 0, (TIM_TIME_REF) 0,
                             sel_ref -> user_data );


  return;

} /* applyCB */


/*----------------------------------------------------------------------*/

static void
  closeCB( Widget       widget,
           SEL_REC_REF  sel_ref,
           XtPointer    call_data )
{

  /* Code. */

  /* Call callback function? */
  if( sel_ref -> actionCB != NULL )
    (* sel_ref -> actionCB)( XTM_SH_REASON_CLOSE,
                             0, "", (TIM_TIME_REF) 0, (TIM_TIME_REF) 0,
                             sel_ref -> user_data );

  /* Hide the window. */
  XtUnmanageChild( sel_ref -> selectW );


  return;

} /* closeCB */


/*----------------------------------------------------------------------*/

static void
  destroyCB( Widget       widget,
             SEL_REC_REF  sel_ref,
             XtPointer    call_data )
{

  /* Code. */

  /* Call callback function? */
  if( sel_ref -> actionCB != NULL )
    (* sel_ref -> actionCB)( XTM_SH_REASON_DESTROY,
                             0, "", (TIM_TIME_REF) 0, (TIM_TIME_REF) 0,
                             sel_ref -> user_data );


  SysFree( sel_ref -> entries );
  SysFree( sel_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void
  editCB( Widget       widget,
          SEL_REC_REF  sel_ref,
          XtPointer    call_data )
{

  /* Variables. */
  LST_STATUS        lst_status;
  SHADOW_ENTRY_REC  entry;

  
  /* Code. */

  if( sel_ref -> entry_selected < 1 ||
      sel_ref -> entry_selected > LstLinkElements( sel_ref -> entries ) )
    return;

  /* Fetch information about the entry. */
  lst_status = LstLinkCurrentPosition( sel_ref -> entries,
                                       sel_ref -> entry_selected );
  if( lst_status != LST_OK )
    return;

  (void) LstLinkGetCurrent( sel_ref -> entries, &entry );


  /* Call callback function? */
  if( sel_ref -> actionCB != NULL )
    (* sel_ref -> actionCB)( XTM_SH_REASON_EDIT,
                             entry.entry_id, entry.calendar, 
                             entry.date_stamp, entry.time_stamp,
                             sel_ref -> user_data );

  return;

} /* editCB */


/*----------------------------------------------------------------------*/

static void
  entrySelectCB( Widget                widget,
                 SEL_REC_REF           sel_ref,
                 XmListCallbackStruct  *call_data )
{

  /* Code. */

  sel_ref -> entry_selected = call_data -> item_position;

  if( call_data -> reason == XmCR_DEFAULT_ACTION )
    editCB( NULL, sel_ref, NULL );


  return;

} /* entrySelectCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget       widget,
          SEL_REC_REF  sel_ref,
          XtPointer    call_data )
{

  /* Code. */

  xtmHlDisplayHelp( sel_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX, window_id, "" );

  return;

} /* helpCB */
