/*----------------------------------------------------------------------------
--
--  Module:           xtmFormat
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Various format routines.
--
--  Filename:         xtmFormat.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-01-07
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmFormat.c, Version: 1.1, Date: 95/02/18 15:52:21";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <string.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmDbTools.h"
#include "xitError.h"
#include "xitTools.h"
#include "XmUbTimeB.h"
#include "xtmFormat.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmFormat";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

Boolean
  xtmFoFetchDate( Widget             parentW,
                  Widget             dateW,
                  XTM_FO_FETCH_WHAT  fetch_what,
                  Boolean            empty_allowed,
                  TIM_TIME_REF       *use_time )
{

  /* Variables. */
  XmUbTimeBoxStatus  status;


  /* Code. */

  *use_time = 0;

  if( fetch_what == XTM_FO_START_DATE )
    status = XmUbTimeBoxGetStartDate( dateW, use_time );
  else
    status = XmUbTimeBoxGetEndDate( dateW, use_time );

  if( status == TBOX_EMPTY && empty_allowed ) {
    *use_time = 0;
    return( True );
  }

  if( status == TBOX_OK )
    return( True );

  *use_time = 0;

  if( parentW != NULL )
    (void) xitCreateErrorDialog( 
             parentW, "ErrorDialog",
             msgGetText( MXDI_ERROR_MESSAGE_LABEL ),
             msgGetText( MXDI_INVALID_DATE ),
             NULL, NULL );


  return( False );

} /* xtmFoFetchDate */


/*----------------------------------------------------------------------*/

Boolean
  xtmFoFetchMinuteTime( char     *char_ref,
                        Boolean  empty_allowed,
                        Widget   parent,
                        int      max_minutes,
                        int      *return_minutes )
{

  /* Variables. */
  Boolean  found = False;
  int      days;
  int      hours;
  int      items;
  int      minutes;
  char     *msg_ref;
  char     *text_ref;
  Widget   tempW;


  /* Code. */

  *return_minutes = -1;

  /* Empty field? */
  text_ref = char_ref;
  while( isspace( *text_ref ) )
    text_ref++;

  if( *text_ref == '\0' )
    return( empty_allowed );

  /* Start looking. */
  if( ! found ) {
    items = sscanf( char_ref, "%d %d:%d", &days, &hours, &minutes );
    if( items == 3 )
      found = True;
  }

  if( ! found ) {
    items = sscanf( char_ref, "%d:%d", &hours, &minutes );
    if( items == 2 ) {
      found = True;
      days  = 0;
    }
  }

  if( ! found ) {
    items = sscanf( char_ref, "%d", &minutes );
    if( items == 1 ) {
      found = True;
      days  = 0;
      hours = 0;
    }
  }

  /* Did we find anything? */
  if( found &&
      days >= 0 &&
      hours >= 0 && hours <= 23 && 
      minutes >= 0 && minutes <= 59 ) {
    minutes = days * 24 * 60 + hours * 60 + minutes;

    if( minutes <= max_minutes ) {
      *return_minutes = minutes;

      return( True );
    }
  }

  if( parent != NULL ) {
    if( found )
      msg_ref = msgGetText( MXDI_TIME_OUTSIDE_LIMITS );
    else
      msg_ref = msgGetText( MXDI_INVALID_MINUTE_TIME );

    tempW = xitCreateErrorDialog( 
              parent, "ErrorDialog",
              msgGetText( MXDI_ERROR_MESSAGE_LABEL ),
              msg_ref,
              NULL, NULL );
  }


  return( False );

} /* xtmFoFetchMinuteTime */


/*----------------------------------------------------------------------*/

Boolean
  xtmFoFetchTime( Widget             parentW,
                  Widget             timeW,
                  XTM_FO_FETCH_WHAT  fetch_what,
                  Boolean            empty_allowed,
                  TIM_TIME_REF       *use_time )
{

  /* Variables. */
  XmUbTimeBoxStatus  status;


  /* Code. */

  *use_time = 0;

  if( fetch_what == XTM_FO_START_TIME )
    status = XmUbTimeBoxGetStartTime( timeW, use_time );
  else
    status = XmUbTimeBoxGetEndTime( timeW, use_time );

  if( status == TBOX_EMPTY && empty_allowed ) {
    *use_time = 0;
    return( True );
  }

  if( status == TBOX_OK )
    return( True );

  *use_time = 0;

  if( parentW != NULL )
    (void) xitCreateErrorDialog( 
             parentW, "ErrorDialog",
             msgGetText( MXDI_ERROR_MESSAGE_LABEL ),
             msgGetText( MXDI_INVALID_ENTRY_TIME ),
             NULL, NULL );


  return( False );

} /* xtmFoFetchTime */


/*----------------------------------------------------------------------*/

void
  xtmFoFormatDate( TIM_TIME_REF  use_time,
                   char          *buffer,
                   int           buffer_size )
{

  /* Code. */

  TimFormatDate( use_time, buffer, buffer_size );


  return;

} /* xtmFoFormatDate */


/*----------------------------------------------------------------------*/

void
  xtmFoFormatEntryFlags( XTM_GL_CUSTOM_DATA_REF  custom_data,
                         XTM_DB_ALL_ENTRY_REF    entry_record,
                         char                    *flags_buffer,
                         int                     flags_buffer_size )
{

  /* Variables. */
  char  buffer[ 100 ];
  char  tmp_buffer[ 50 ];


  /* Code. */

  *flags_buffer = '\0';
  buffer[ 0 ]   = '\0';

  /* Get the flags. */
  if( flagIsSet( entry_record -> entry.flags, XTM_DB_FLAG_ALARM ) ) {
    sprintf( tmp_buffer, "%c", custom_data -> alarm_marker );
    strcat( buffer, tmp_buffer );
  }

  if( flagIsSet( entry_record -> entry.flags, 
                 (XTM_DB_FLAG_ACTION_SCRIPT | XTM_DB_FLAG_ACTION_TEXT) ) ) {
    sprintf( tmp_buffer, "%c", custom_data -> action_alarm_marker );
    strcat( buffer, tmp_buffer );
  }

  if( flagIsSet( entry_record -> entry.flags, XTM_DB_FLAG_ARCHIVED ) ) {
    sprintf( tmp_buffer, "%c", custom_data -> archive_marker );
    strcat( buffer, tmp_buffer );
  }

  if( flagIsSet( entry_record -> entry.flags, XTM_DB_FLAG_IMPORTANT ) ) {
    sprintf( tmp_buffer, "%c", custom_data -> important_marker );
    strcat( buffer, tmp_buffer );
  }

  if( flagIsSet( entry_record -> entry.flags, XTM_DB_FLAG_PRIVATE ) ) {
    sprintf( tmp_buffer, "%c", custom_data -> private_marker );
    strcat( buffer, tmp_buffer );
  }

  if( flagIsSet( entry_record -> entry.flags, XTM_DB_FLAG_INCLUDE ) ) {
    sprintf( tmp_buffer, "%c", custom_data -> include_marker );
    strcat( buffer, tmp_buffer );
  }

  if( entry_record -> entry.entry_category == XTM_DB_REP_ENTRY_LIST ) {
    sprintf( tmp_buffer, "%c", custom_data -> standing_marker );
    strcat( buffer, tmp_buffer );
  }

  if( entry_record -> entry.entry_type == XTM_DB_DAY_NOTE ) {
    if( flagIsSet( entry_record -> entry.flags, XTM_DB_FLAG_NOTE_DONE ) ) {
      sprintf( tmp_buffer, "%c", custom_data -> note_done_marker );
      strcat( buffer, tmp_buffer );
    } else {
      if( custom_data -> note_not_done_marker != 32 ) {
        sprintf( tmp_buffer, "%c", custom_data -> note_not_done_marker );
        strcat( buffer, tmp_buffer );
      }
    }
  }

  /* Save the result. */
  if( strlen( buffer ) < flags_buffer_size )
    strcpy( flags_buffer, buffer );


  return;

} /* xtmFoFormatEntryFlags */


/*----------------------------------------------------------------------*/

void
  xtmFoFormatEntryTimes( XTM_DB_ALL_ENTRY_REF  entry_record,
                         char                  *time_buffer,
                         int                   time_buffer_size )
{

  /* Variables. */
  char          buffer[ 100 ];
  char          buffer1[ 50 ];
  char          buffer2[ 50 ];
  TIM_TIME_REF  start_time;
  TIM_TIME_REF  end_time;


  /* Code. */

  strcpy( time_buffer, "" );

  if( entry_record -> entry.entry_type == XTM_DB_DAY_NOTE )
    return;

  start_time = entry_record -> entry.time_stamp;
  end_time   = start_time;

  TimAddMinutes( &end_time, (int) entry_record -> entry.duration );

  xtmFoFormatTime( start_time, buffer1, sizeof( buffer1 ) );
  xtmFoFormatTime( end_time,   buffer2, sizeof( buffer2 ) );

  if( entry_record -> entry.duration > 0 )
    sprintf( buffer, "%s - %s", buffer1, buffer2 );
  else
    sprintf( buffer, "%s", buffer1 );

  /* Save the result. */
  if( strlen( buffer ) < time_buffer_size )
    strcpy( time_buffer, buffer );


  return;

} /* xtmFoFormatEntryTimes */


/*----------------------------------------------------------------------*/

char
  *xtmFoFormatText( char  *format_text,
                    int   indent,
                    int   max_lines,
                    int   max_line_length )
{

  /* Variables. */
  Boolean    line_written = False;
  int        SEGMENT_SIZE = 10000;
  int        length;
  int        lines_read;
  int        text_buffer_size;
  char       buffer[ 100 ];
  char       format_buffer[ 20 ];
  char       line[ 100 ];
  char       *char_ref;
  char       *eol_ref;
  char       *text_buffer;


  /* Code. */

  if( format_text == NULL || *format_text == '\0' )
    return( NULL );

  /* Format for the message line. */
  sprintf( format_buffer, "%%-%d.%ds%%-%d.%ds", 
           indent, indent, max_line_length, max_line_length );

  char_ref   = format_text;
  lines_read = 0;

  if( max_lines < 1 )
    max_lines = 1;

  text_buffer      = SysMalloc( SEGMENT_SIZE );
  *text_buffer     = '\0';
  text_buffer_size = SEGMENT_SIZE;

  while( *char_ref != '\0' && lines_read < max_lines ) {

    eol_ref = strchr( char_ref, '\n' );
    if( eol_ref == NULL )
      eol_ref = (char_ref + strlen( char_ref ) - 1);
    else
      eol_ref--;

    /* The length of the string cannot exceed n characters. */
    length = eol_ref - char_ref + 1;
    if( length > max_line_length )
      length = max_line_length;

    strncpy( buffer, char_ref, length );
    buffer[ length ] ='\0';

    sprintf( line, format_buffer, "", buffer );

    /* Extend the text buffer? */
    if( strlen( text_buffer ) + strlen( line ) + 10 > text_buffer_size ) {
      text_buffer_size = text_buffer_size + SEGMENT_SIZE;

      text_buffer = SysRealloc( text_buffer, text_buffer_size );
    }

    /* Add a linefeed between the lines, not before and not after. */
    if( line_written )
      strcat( text_buffer, "\n" );
    else
      line_written = True;

    strcat( text_buffer, line );

    lines_read++;

    if( *(eol_ref + 1) == '\n' )
      char_ref = eol_ref + 2;
    else
      char_ref = eol_ref + 1;

  } /* while */


  return( text_buffer );

} /* xtmFoFormatText */


/*----------------------------------------------------------------------*/

void
  xtmFoFormatTime( TIM_TIME_REF  use_time,
                   char          *buffer,
                   int           buffer_size )
{

  /* Code. */

  TimFormatTime( use_time, buffer, buffer_size );


  return;

} /* xtmFoFormatTime */


/*----------------------------------------------------------------------*/

void
  xtmFoFormatFullTime( TIM_TIME_REF  use_time,
                       char          *buffer,
                       int           buffer_size )
{

  /* Variables. */
  char  date_buffer[ 50 ];
  char  day_buffer[ 50 ];
  char  full_time_buffer[ 50 ];
  char  time_buffer[ 50 ];


  /* Code. */

  *buffer = '\0';

  /* The date. */
  TimFormatFullDate( use_time, date_buffer, sizeof( date_buffer ) );

  /* The time. */
  TimFormatTime( use_time, time_buffer, sizeof( time_buffer ) );

  /* The day. */
  TimFormatStrTime( use_time, "%a", day_buffer, sizeof( day_buffer ) );

  /* The full date. */
  sprintf( full_time_buffer, "%s\n%s, %s",
           date_buffer, day_buffer, time_buffer );

  if( strlen( full_time_buffer ) < buffer_size )
    strcpy( buffer, full_time_buffer );


  return;

} /* xtmFoFormatFullTime */
