/*----------------------------------------------------------------------------
--
--  Module:           xtmFields
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Simplified creation of various tyoes of entry fields.
--
--  Filename:         xtmFields.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1990-12-16
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmFields.c, Version: 1.1, Date: 95/02/18 15:52:20";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <X11/Intrinsic.h>

#include <Xm/Xm.h>

#include "System.h"
#include "LstLinked.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmDbTools.h"
#include "xtmTags.h"
#include "xitColorSel.h"
#include "xitError.h"
#include "xitFieldSel.h"
#include "xitTools.h"
#include "xtmFields.h"
#include "XmUbTimeB.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Pre-defined entry tags in a menu, how many? */
#define MAX_MENU_TAGS  50


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmFields";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  actionDateCB( Widget               widget,
                int                  id,
                XmAnyCallbackStruct  *call_data );

static void 
  actionDateTimeCB( Widget               widget,
                    int                  id,
                    XmAnyCallbackStruct  *call_data );

static void 
  actionRangeCB( Widget               widget,
                 int                  id,
                 XmAnyCallbackStruct  *call_data );

static void 
  actionTimeCB( Widget               widget,
                int                  id,
                XmAnyCallbackStruct  *call_data );

static Widget
  createTimeBox( Widget        parentW,
                 char          *name,
                 int           format,
                 XmUbMenuItem  *menu,
                 int           num_menu );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

Widget
  xtmFlCreateColorField( Widget          parent,
                         char            *name,
                         int             width,
                         char            *colors,
                         XtCallbackProc  changedCB,
                         void            *cb_data )
{

  /* Variables. */
  int           char_read;
  int           index;
  int           items;
  int           tag_count;
  char          buffer[ 50 ];
  char          *char_ref;
  Arg           args[ 10 ];
  Cardinal      n;
  String        tag_array[ MAX_MENU_TAGS ];
  Widget        colorFs;


  /* Code. */

  char_ref  = colors;
  tag_count = 0;

  do {
    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    items = sscanf( char_ref, "%s%n", buffer, &char_read );
    if( items != 1 )
      break;

    tag_array[ tag_count ] = SysNewString( buffer );
    tag_count++;

    char_ref = char_ref + char_read;
  } while( tag_count < MAX_MENU_TAGS );

  n = 0;
  XtSetArg( args[ n ], XmNitems,     tag_array ); n++;
  XtSetArg( args[ n ], XmNitemCount, tag_count ); n++;
  XtSetArg( args[ n ], XmNcolumns,   width     ); n++;
  colorFs = xitCreateColorSelect( parent, name, args, n, changedCB, cb_data );

  for( index = 0; index < tag_count; index++ )
    SysFree( tag_array[ index ] );


  return( colorFs );

} /* xtmFlCreateColorField */


/*----------------------------------------------------------------------*/

Widget
  xtmFlCreateDateField( Widget   parent,
                        char     *name )
{

  /* Variables. */
  Widget  dateTb;

  static XmUbMenuItem  menu[] = {
    { XmUbITEM_PUSH_BUTTON, "", NULL, XmUbTB_ACTION_DATE_NOW, NULL },
    { XmUbITEM_SEPARATOR, "", NULL, (XtCallbackProc) NULL, (XtPointer) -1 },
    { XmUbITEM_PUSH_BUTTON, "", NULL,
      (XtCallbackProc) actionDateCB, (XtPointer) 0 },
  };


  /* Code. */

  /* The selection menu. */
  menu[ 0 ].label =
    XmStringCreate( msgGetText( MXDI_DATE_TODAY_LABEL ), CS );
  menu[ 2 ].label = 
    XmStringCreate( msgGetText( MXDI_SELECT_DATE_LABEL ), CS );


  /* Create the date box. */
  dateTb = createTimeBox( parent, name, 
                          XmUbTB_FORMAT_D, menu, XtNumber( menu ) );

  XmStringFree( menu[ 0 ].label );
  XmStringFree( menu[ 2 ].label );


  return( dateTb );

} /* xtmFlCreateDateField */


/*----------------------------------------------------------------------*/

Widget
  xtmFlCreateDateTimeField( Widget   parent,
                            char     *name )
{

  /* Variables. */
  Widget  dateTb;

  static XmUbMenuItem  menu[] = {
    { XmUbITEM_PUSH_BUTTON, "", NULL, XmUbTB_ACTION_DATE_NOW, NULL },
    { XmUbITEM_SEPARATOR, "", NULL, (XtCallbackProc) NULL, (XtPointer) -1 },
    { XmUbITEM_PUSH_BUTTON, "", NULL, XmUbTB_ACTION_TIME_NOW, NULL },
    { XmUbITEM_PUSH_BUTTON, "", NULL,
      (XtCallbackProc) actionDateTimeCB, (XtPointer) 0 },
    { XmUbITEM_SEPARATOR, "", NULL, (XtCallbackProc) NULL, (XtPointer) -1 },
    { XmUbITEM_PUSH_BUTTON, "", NULL,
      (XtCallbackProc) actionDateTimeCB, (XtPointer) 1 },
    { XmUbITEM_PUSH_BUTTON, "", NULL,
      (XtCallbackProc) actionDateTimeCB, (XtPointer) 2 },
  };


  /* Code. */

  /* The selection menu. */
  menu[ 0 ].label =
    XmStringCreate( msgGetText( MXDI_DATE_TODAY_LABEL ), CS );
  menu[ 2 ].label =
    XmStringCreate( msgGetText( MXDI_TIME_NOW_LABEL ), CS );
  menu[ 3 ].label =
    XmStringCreate( msgGetText( MXDI_TIME_NOON_LABEL ), CS );
  menu[ 5 ].label = 
    XmStringCreate( msgGetText( MXDI_SELECT_DATE_LABEL ), CS );
  menu[ 6 ].label = 
    XmStringCreate( msgGetText( MXDI_SELECT_TIME_LABEL ), CS );

  /* Create the date box. */
  dateTb = createTimeBox( parent, name, 
                          XmUbTB_FORMAT_DT, menu, XtNumber( menu ) );

  XmStringFree( menu[ 0 ].label );
  XmStringFree( menu[ 2 ].label );
  XmStringFree( menu[ 3 ].label );
  XmStringFree( menu[ 5 ].label );
  XmStringFree( menu[ 6 ].label );


  return( dateTb );

} /* xtmFlCreateDateTimeField */


/*----------------------------------------------------------------------*/

Widget
  xtmFlCreateDbField( Widget          parent,
                      char            *name,
                      XTM_CD_HANDLE   cal_db_handle,
                      Boolean         only_can_write,
                      XtCallbackProc  changedCB,
                      void            *cb_data )
{

  /* Variables. */
  int       index;
  int       tag_count;
  char      *char_ref;
  char      *db_names;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    dbFs;
  XmString  tag_array[ MAX_MENU_TAGS ];


  /* Code. */

  /* Loop through all databases. */
  (void) xtmCdFetchDbNames( cal_db_handle, &db_names );
  char_ref  = db_names;
  tag_count = 0;


  do {

    int              char_read;
    char             db_name[ XTM_GL_MAX_CAL_NAME + 1 ];
    XTM_CD_CAL_INFO  db_info;

    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    sscanf( char_ref, "%s%n", db_name, &char_read );
    char_ref = char_ref + char_read;


    /* Fetch information about the database. */
    (void) xtmCdFetchNamedDb( cal_db_handle, db_name,
                              &db_info, NULL );

    if( ! only_can_write ||
        flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_WRITE ) ) {
      tag_array[ tag_count ] = XmStringCreate( db_info.short_name, CS );

      tag_count++;
    }

  } while( True );

  if( db_names != NULL )
    SysFree( db_names );


  /* The list is always sorted. */
  xitSortStringList( tag_array, tag_count );

  n = 0;
  XtSetArg( args[ n ], XmNitems, tag_array ); n++;
  XtSetArg( args[ n ], XmNitemCount, tag_count ); n++;
  XtSetArg( args[ n ], XmNmaxLength, 15 ); n++;
  XtSetArg( args[ n ], XmNcolumns, 15 ); n++;
  XtSetArg( args[ n ], XmNeditable, False ); n++;
  XtSetArg( args[ n ], XmNcursorPositionVisible, False ); n++;
  dbFs = xitCreateFieldSelect( parent, name, args, n, changedCB, cb_data );

  /* Free occupied space. */
  for( index = 0; index < tag_count; index++ )
    XmStringFree( tag_array[ index ] );


  return( dbFs );

} /* xtmFlCreateDbField */


/*----------------------------------------------------------------------*/

Widget
  xtmFlCreateRangeField( Widget  parent,
                         char    *name )
{

  /* Variables. */
  Widget  rangeTb;

  static XmUbMenuItem  menu[] = {
    { XmUbITEM_PUSH_BUTTON, "", NULL, XmUbTB_ACTION_THIS_WEEK, NULL },
    { XmUbITEM_PUSH_BUTTON, "", NULL, XmUbTB_ACTION_THIS_MONTH, NULL },
    { XmUbITEM_SEPARATOR, "", NULL,   (XtCallbackProc) NULL, (XtPointer) -1 },
    { XmUbITEM_PUSH_BUTTON, "", NULL, XmUbTB_ACTION_PLUS_ONE_WEEK, NULL },
    { XmUbITEM_PUSH_BUTTON, "", NULL, XmUbTB_ACTION_PLUS_TWO_WEEKS, NULL },
    { XmUbITEM_PUSH_BUTTON, "", NULL, XmUbTB_ACTION_PLUS_ONE_MONTH, NULL },
    { XmUbITEM_SEPARATOR, "", NULL,   (XtCallbackProc) NULL, (XtPointer) -1 },
    { XmUbITEM_PUSH_BUTTON, "", NULL,
      (XtCallbackProc) actionRangeCB, (XtPointer) 0 },
    { XmUbITEM_PUSH_BUTTON, "", NULL,
      (XtCallbackProc) actionRangeCB, (XtPointer) 1 },
  };


  /* Code. */

  /* The selection menu. */
  menu[ 0 ].label =
    XmStringCreate( msgGetText( MXDI_DATE_THIS_WEEK_LABEL ), CS );
  menu[ 1 ].label =
    XmStringCreate( msgGetText( MXDI_DATE_THIS_MONTH_LABEL ), CS );
  menu[ 3 ].label = 
    XmStringCreate( msgGetText( MXDI_PLUS_ONE_WEEK_LABEL ), CS );
  menu[ 4 ].label = 
    XmStringCreate( msgGetText( MXDI_PLUS_TWO_WEEKS_LABEL ), CS );
  menu[ 5 ].label = 
    XmStringCreate( msgGetText( MXDI_PLUS_ONE_MONTH_LABEL ), CS );
  menu[ 7 ].label = 
    XmStringCreate( msgGetText( MXDI_SELECT_START_DATE_LABEL ), CS );
  menu[ 8 ].label = 
    XmStringCreate( msgGetText( MXDI_SELECT_END_DATE_LABEL ), CS );

  /* Create the date range box. */
  rangeTb = createTimeBox( parent, name, 
                           XmUbTB_FORMAT_DD, menu, XtNumber( menu ) );

  XmStringFree( menu[ 0 ].label );
  XmStringFree( menu[ 1 ].label );
  XmStringFree( menu[ 3 ].label );
  XmStringFree( menu[ 4 ].label );
  XmStringFree( menu[ 5 ].label );
  XmStringFree( menu[ 7 ].label );
  XmStringFree( menu[ 8 ].label );


  return( rangeTb );

} /* xtmFlCreateRangeField */


/*----------------------------------------------------------------------*/

Widget
  xtmFlCreateSelField( Widget          parent,
                       char            *name,
                       int             width,
                       char            *selections,
                       char            sel_separator,
                       Boolean         sort,
                       XtCallbackProc  changedCB,
                       void            *cb_data )
{

  /* Variables. */
  int       char_read;
  int       index;
  int       items;
  int       max_length = 0;
  int       tag_count;
  char      buffer[ 250 ];
  char      default_value[ 250 ];
  char      match_buffer[ 50 ];
  char      *char_ref;
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    selFs;
  XmString  tag_array[ MAX_MENU_TAGS ];


  /* Code. */

  sprintf( match_buffer, "%%[^%c]%c%%n", sel_separator, sel_separator );

  char_ref           = selections;
  tag_count          = 0;
  default_value[ 0 ] = '\0';

  do {
    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    items = sscanf( char_ref, match_buffer, buffer, &char_read );
    if( items != 1 )
      break;

    if( strlen( buffer ) > max_length )
      max_length = strlen( buffer );

    if( tag_count == 0 )
      strcpy( default_value, buffer );

    tag_array[ tag_count ] = XmStringCreate( buffer, CS );
    tag_count++;

    char_ref = char_ref + char_read;
  } while( tag_count < MAX_MENU_TAGS );

  if( width == 0 )
    width = max_length + 1;

  /* The list is sorted? */
  if( sort )
    xitSortStringList( tag_array, tag_count );

  n = 0;
  XtSetArg( args[ n ], XmNitems, tag_array ); n++;
  XtSetArg( args[ n ], XmNitemCount, tag_count ); n++;
  XtSetArg( args[ n ], XmNmaxLength, width ); n++;
  XtSetArg( args[ n ], XmNcolumns, width ); n++;
  selFs = xitCreateFieldSelect( parent, name, args, n, changedCB, cb_data );

  for( index = 0; index < tag_count; index++ )
    XmStringFree( tag_array[ index ] );

  xitFieldSelectSetCurrent( selFs, default_value, False );


  return( selFs );

} /* xtmFlCreateSelField */


/*----------------------------------------------------------------------*/

Widget
  xtmFlCreateTagsField( Widget          parent,
                        char            *name,
                        int             width,
                        char            *tags,
                        XTM_TG_HANDLE   tag_handle,
                        XtCallbackProc  changedCB,
                        void            *cb_data )
{

  /* Variables. */
  int       char_read;
  int       index;
  int       items;
  int       tag_count;
  char      tag[ 50 ];
  char      *char_ref;
  Arg       args[ 10 ];
  Cardinal  n;
  Pixel     bg_array[ MAX_MENU_TAGS ];
  Pixel     fg_array[ MAX_MENU_TAGS ];
  Widget    selFs;
  XmString  tag_array[ MAX_MENU_TAGS ];


  /* Code. */

  char_ref  = tags;
  tag_count = 0;

  do {
    while( isspace( *char_ref ) )
      char_ref++;

    if( *char_ref == '\0' )
      break;

    char_read = strlen( char_ref );
    items = sscanf( char_ref, "%s%n", tag, &char_read );
    if( items != 1 )
      break;

    bg_array[ tag_count ] = None;
    fg_array[ tag_count ] = None;


    /* Color for this tag? */
    (void) xtmTgFetchTagColor( tag_handle, tag,
                               &fg_array[ tag_count ],
                               &bg_array[ tag_count ] );


    tag_array[ tag_count ] = XmStringCreate( tag, CS );
    tag_count++;

    char_ref = char_ref + char_read;
  } while( tag_count < MAX_MENU_TAGS );


  n = 0;
  XtSetArg( args[ n ], XmNbgItems, bg_array ); n++;
  XtSetArg( args[ n ], XmNfgItems, fg_array ); n++;
  XtSetArg( args[ n ], XmNitems, tag_array ); n++;
  XtSetArg( args[ n ], XmNitemCount, tag_count ); n++;
  XtSetArg( args[ n ], XmNmaxLength, width ); n++;
  XtSetArg( args[ n ], XmNcolumns, width ); n++;
  selFs = xitCreateFieldSelect( parent, name, args, n, changedCB, cb_data );

  for( index = 0; index < tag_count; index++ )
    XmStringFree( tag_array[ index ] );


  return( selFs );

} /* xtmFlCreateTagsField */


/*----------------------------------------------------------------------*/

Widget
  xtmFlCreateTimeField( Widget  parent,
                        char    *name )
{

  /* Variables. */
  Widget  timeTb;

  static XmUbMenuItem  menu[] = {
    { XmUbITEM_PUSH_BUTTON, "", NULL, XmUbTB_ACTION_TIME_NOW, NULL },
    { XmUbITEM_PUSH_BUTTON, "", NULL,
      (XtCallbackProc) actionTimeCB, (XtPointer) 0 },
    { XmUbITEM_SEPARATOR, "", NULL, (XtCallbackProc) NULL, (XtPointer) -1 },
    { XmUbITEM_PUSH_BUTTON, "", NULL,
      (XtCallbackProc) actionTimeCB, (XtPointer) 1 },
  };


  /* Code. */

  /* The selection menu. */
  menu[ 0 ].label =
    XmStringCreate( msgGetText( MXDI_TIME_NOW_LABEL ), CS );
  menu[ 1 ].label =
    XmStringCreate( msgGetText( MXDI_TIME_NOON_LABEL ), CS );
  menu[ 3 ].label = 
    XmStringCreate( msgGetText( MXDI_SELECT_TIME_LABEL ), CS );

  /* Create the time box. */
  timeTb = createTimeBox( parent, name, 
                          XmUbTB_FORMAT_T, menu, XtNumber( menu ) );

  XmStringFree( menu[ 0 ].label );
  XmStringFree( menu[ 1 ].label );
  XmStringFree( menu[ 3 ].label );


  return( timeTb );

} /* xtmFlCreateTimeField */


/*----------------------------------------------------------------------*/

static Widget
  createTimeBox( Widget        parentW,
                 char          *name,
                 int           format,
                 XmUbMenuItem  *menu,
                 int           num_menu )
{

  /* Variables. */
  char      *abbrev_day;
  char      *abbrev_week;
  char      *abbrev_month;
  char      *abbrev_year;
  Arg       args[ 25 ];
  Cardinal  n;
  Widget    dateTb;
  XmString  cancel_xstr;
  XmString  month_xstr;
  XmString  ok_xstr;
  XmString  week_xstr;
  XmString  year_xstr;


  /* Code. */

  /* Abbreviation strings. */
  abbrev_day = msgGetText( MXDI_DAY_ABBREV );
  abbrev_week = msgGetText( MXDI_WEEK_ABBREV );
  abbrev_month = msgGetText( MXDI_MONTH_ABBREV );
  abbrev_year = msgGetText( MXDI_YEAR_ABBREV );

  /* Label strings. */
  cancel_xstr = XmStringCreate( msgGetText( MXDI_CANCEL_BUTTON ), CS );
  month_xstr = XmStringCreate( msgGetText( MXDI_MONTH_LABEL ), CS );
  ok_xstr = XmStringCreate( msgGetText( MXDI_OK_BUTTON ), CS );
  week_xstr = XmStringCreate( msgGetText( MXDI_WEEK_NUMBER_LABEL ), CS );
  year_xstr = XmStringCreate( msgGetText( MXDI_YEAR_LABEL ), CS );


  /* Create the time box. */
  n = 0;
  XtSetArg( args[ n ], XmNcancelLabelString, cancel_xstr ); n++;
  XtSetArg( args[ n ], XmNokLabelString, ok_xstr ); n++;
  XtSetArg( args[ n ], XmUbNtboxCodeDay, abbrev_day ); n++;
  XtSetArg( args[ n ], XmUbNtboxCodeMonth, abbrev_month ); n++;
  XtSetArg( args[ n ], XmUbNtboxCodeWeek, abbrev_week ); n++;
  XtSetArg( args[ n ], XmUbNtboxCodeYear, abbrev_year ); n++;
  XtSetArg( args[ n ], XmUbNtboxDateSelectionTitle, " " ); n++;
  XtSetArg( args[ n ], XmUbNtboxFormat, format ); n++;
  XtSetArg( args[ n ], XmUbNtboxMenuItems, menu ); n++;
  XtSetArg( args[ n ], XmUbNtboxMenuPosition, XmUbPOSITION_LAST ); n++;
  XtSetArg( args[ n ], XmUbNtboxMonthLabel, month_xstr ); n++;
  XtSetArg( args[ n ], XmUbNtboxNumItems, num_menu ); n++;
  XtSetArg( args[ n ], XmUbNtboxWeekLabel, week_xstr ); n++;
  XtSetArg( args[ n ], XmUbNtboxYearLabel, year_xstr ); n++;
  dateTb = XmUbCreateTimeBox( parentW, name, args, n );

  XmStringFree( cancel_xstr );
  XmStringFree( month_xstr );
  XmStringFree( ok_xstr );
  XmStringFree( week_xstr );
  XmStringFree( year_xstr );


  return( dateTb );

} /* createTimeBox */


/*----------------------------------------------------------------------*/

static void 
  actionDateCB( Widget               widget,
                int                  id,
                XmAnyCallbackStruct  *call_data )
{

  /* Variables. */
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    tboxW;


  /* Code. */

  /* Get the timebox widget. It is in the userdata of the button. */
  n = 0;
  XtSetArg( args[ n ], XmNuserData, &tboxW ); n++;
  XtGetValues( widget, args, n );

  /* Date dialog. */
  if( id == 0 )
    XmUbTimeBoxPopupDateDialog( tboxW, XmUbTB_CHILD_START_DATE );


  return;

} /* actionDateCB */


/*----------------------------------------------------------------------*/

static void 
  actionDateTimeCB( Widget               widget,
                    int                  id,
                    XmAnyCallbackStruct  *call_data )
{

  /* Variables. */
  Arg           args[ 10 ];
  Cardinal      n;
  Widget        tboxW;
  TIM_TIME_REF  use_time;


  /* Code. */

  /* Get the timebox widget. It is in the userdata of the button. */
  n = 0;
  XtSetArg( args[ n ], XmNuserData, &tboxW ); n++;
  XtGetValues( widget, args, n );

  /* Take action. */
  if( id == 0 )
    use_time = TimMakeTime( 1970, 1, 1, 12, 0, 0 );

  /* Set the timebox field. */
  if( id == 0 ) {
    use_time = TimMakeTime( 1970, 1, 1,
                            TimHour( use_time ), TimMinute( use_time ), 0 );

    XmUbTimeBoxSetStartTime( tboxW, use_time );
  }

  /* Time dialog. */
  if( id == 1 )
    XmUbTimeBoxPopupDateDialog( tboxW, XmUbTB_CHILD_START_DATE );
  if( id == 2 )
    XmUbTimeBoxPopupTimeDialog( tboxW, XmUbTB_CHILD_START_TIME );


  return;

} /* actionDateTimeCB */


/*----------------------------------------------------------------------*/

static void 
  actionRangeCB( Widget               widget,
                 int                  id,
                 XmAnyCallbackStruct  *call_data )
{

  /* Variables. */
  Arg       args[ 10 ];
  Cardinal  n;
  Widget    tboxW;


  /* Code. */

  /* Get the timebox widget. It is in the userdata of the button. */
  n = 0;
  XtSetArg( args[ n ], XmNuserData, &tboxW ); n++;
  XtGetValues( widget, args, n );

  /* Date dialog. */
  if( id == 0 )
    XmUbTimeBoxPopupDateDialog( tboxW, XmUbTB_CHILD_START_DATE );

  if( id == 1 )
    XmUbTimeBoxPopupDateDialog( tboxW, XmUbTB_CHILD_END_DATE );


  return;

} /* actionRangeCB */


/*----------------------------------------------------------------------*/

static void 
  actionTimeCB( Widget               widget,
                int                  id,
                XmAnyCallbackStruct  *call_data )
{

  /* Variables. */
  Arg           args[ 10 ];
  Cardinal      n;
  Widget        tboxW;
  TIM_TIME_REF  use_time;


  /* Code. */

  /* Get the timebox widget. It is in the userdata of the button. */
  n = 0;
  XtSetArg( args[ n ], XmNuserData, &tboxW ); n++;
  XtGetValues( widget, args, n );

  /* Take action. */
  if( id == 0 )
    use_time = TimMakeTime( 1970, 1, 1, 12, 0, 0 );

  /* Set the timebox field. */
  if( id == 0 ) {
    use_time = TimMakeTime( 1970, 1, 1,
                            TimHour( use_time ), TimMinute( use_time ), 0 );

    XmUbTimeBoxSetStartTime( tboxW, use_time );
  }

  /* Time dialog. */
  if( id == 1 )
    XmUbTimeBoxPopupTimeDialog( tboxW, XmUbTB_CHILD_START_TIME );


  return;

} /* actionTimeCB */

