/*----------------------------------------------------------------------------
--
--  Module:           xtmEditStand
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Edit the standing data for an XDiary entry.
--
--  Filename:         xtmEditStand.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1992-10-10
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmEditStand.c, Version: 1.1, Date: 95/02/18 15:52:18";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Protocols.h>

#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/RowColumn.h>
#include <Xm/SeparatoG.h>
#include <Xm/Text.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmDbTools.h"
#include "xtmEditSkip.h"
#include "xtmFields.h"
#include "xtmFormat.h"
#include "xtmHelp.h"
#include "xtmTools.h"
#include "xitError.h"
#include "xitFieldSel.h"
#include "xitTools.h"
#include "xitTune.h"
#include "XmUbTimeB.h"
#include "xtmEditStand.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the standing window. */
#define betweenLa          dataLocalW[  0 ]
#define dayInMonthFs       dataLocalW[  1 ]
#define dayInMonthLa       dataLocalW[  2 ]
#define dayInMonthRc       dataLocalW[  3 ]
#define dayInMonthTb       dataLocalW[  4 ]
#define noWorkFr           dataLocalW[  5 ]
#define noWorkFs           dataLocalW[  6 ]
#define noWorkLa           dataLocalW[  7 ]
#define noWorkRc           dataLocalW[  8 ]
#define rangeTb            dataLocalW[  9 ]
#define skipWeekPb         dataLocalW[ 10 ]
#define timeRc             dataLocalW[ 11 ]
#define valIntEveryFs      dataLocalW[ 12 ]
#define valIntEveryLa      dataLocalW[ 13 ]
#define valIntEveryTx      dataLocalW[ 14 ]
#define valIntFr           dataLocalW[ 15 ]
#define valIntRc           dataLocalW[ 16 ]
#define valOnDaysFr        dataLocalW[ 17 ]
#define valOnDaysRc        dataLocalW[ 18 ]
#define valRc              dataLocalW[ 19 ]
#define validDaysLa        dataLocalW[ 20 ]
#define valIntEveryRc      dataLocalW[ 21 ]
#define valIntLa           dataLocalW[ 22 ]
#define anchorSp           dataLocalW[ 23 ]
#define doActionRc         dataLocalW[ 24 ]
#define typeRb             dataLocalW[ 25 ]
#define typeDayTb          dataLocalW[ 26 ]
#define typeIntTb          dataLocalW[ 27 ]
#define typeFr             dataLocalW[ 28 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Record describing one instance of the preference window. */
typedef struct {

  /* Flags for the standing entry. */
  UINT32  entry_flags;

  /* Data for the skip weeks window. */
  UINT32  skip_week[ 2 ];

  /* Standing window. */
  Widget  standW;

  /* Default date to use. */
  TIM_TIME_REF  default_date;

  /* What type of entry is it? */
  XTM_DB_ALL_ENTRY_TYPES  entry_type;

  /* Handle for the skip weeks window. */
  XTM_EK_HANDLE  skip_handle;

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Callback and user data to inform our creator of specific actions. */
  XTM_ES_ACTION_CB  actionCB;
  void              *user_data;

} STAND_REC, *STAND_REC_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmEditStand";

/* IDs for the help windows. */
static char  *stand_window_id = "EntryStand";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static void 
  cancelCB( Widget         widget,
            STAND_REC_REF  stand_ref,
            XtPointer      call_data );

static Widget
  createStandWindow( Widget         parent,
                     STAND_REC_REF  stand_ref );

static void 
  dayInMonthCB( Widget                        widget,
                STAND_REC_REF                 stand_ref,
                XmToggleButtonCallbackStruct  *call_data );

static void 
  destroyCB( Widget         widget,
             STAND_REC_REF  stand_ref,
             XtPointer      call_data );

static Boolean
  fetchStandValues( STAND_REC_REF          stand_ref,
                    XTM_ES_GET_STAND_DATA  *values_ref );

static void 
  helpCB( Widget         widget,
          STAND_REC_REF  stand_ref,
          XtPointer      call_data );

static void 
  okCB( Widget         widget,
        STAND_REC_REF  stand_ref,
        XtPointer      call_data );

static void
  rangeActionCB( char  *range_str,
                 void  *user_data );

static void
  setDefaultStandValues( STAND_REC_REF  stand_ref );

static void
  setStandValues( STAND_REC_REF          stand_ref,
                  XTM_ES_SET_STAND_DATA  *values_ref );

static void 
  skipCB( Widget         widget,
          STAND_REC_REF  stand_ref,
          XtPointer      call_data );

static void 
  skipActionCB( XTM_EK_REASON         reason,
                void                  *user_data,
                XTM_EK_GET_SKIP_DATA  *values_ref );

static void 
  typeIntCB( Widget                        widget,
             STAND_REC_REF                 stand_ref,
             XmToggleButtonCallbackStruct  *call_data );

static void 
  typeDayCB( Widget                        widget,
             STAND_REC_REF                 stand_ref,
             XmToggleButtonCallbackStruct  *call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

XTM_ES_HANDLE
  xtmEsInitialize( XTM_GL_BASE_DATA_REF  appl_data_ref,
                   Widget                parent,
                   XTM_ES_ACTION_CB      actionCB,
                   void                  *user_data )
{

  /* Variables. */
  STAND_REC_REF  stand_ref;


  /* Code. */

  /* Create and initialize our private data. */
  stand_ref = SysNew( STAND_REC );
  if( stand_ref == NULL )
    return( NULL );

  stand_ref -> appl_data_ref = appl_data_ref;
  stand_ref -> actionCB      = actionCB;
  stand_ref -> user_data     = user_data;
  stand_ref -> standW        = NULL;
  stand_ref -> skip_handle   = NULL;


  /* Create the base window. */
  stand_ref -> standW = createStandWindow( parent, stand_ref );
  if( stand_ref -> standW == NULL ) {
    SysFree( stand_ref );

    return( NULL );
  }


  return( (XTM_ES_HANDLE) stand_ref );

} /* xtmEsInitialize */


/*----------------------------------------------------------------------*/

void
  xtmEsDestroy( XTM_ES_HANDLE  stand_handle )
{

  /* Variables. */
  STAND_REC_REF  stand_ref;


  /* Code. */

  if( stand_handle == NULL )
    return;

  /* Our private data. */
  stand_ref = (STAND_REC_REF) stand_handle;

  /* Get rid of all windows. */
  if( stand_ref -> standW != NULL )
    XtDestroyWidget( stand_ref -> standW );


  return;

} /* xtmEsDestroy */


/*----------------------------------------------------------------------*/

void
  xtmEsEditStanding( XTM_ES_HANDLE          stand_handle,
                     XTM_ES_SET_STAND_DATA  *values_ref )
{

  /* Variables. */
  STAND_REC_REF  stand_ref;


  /* Code. */

  /* Our private data. */
  stand_ref = (STAND_REC_REF) stand_handle;

  stand_ref -> entry_flags = values_ref -> flags;

  /* Set the window data. */
  setStandValues( stand_ref, values_ref );


  /* Make sure the window is visible. */
  XtManageChild( stand_ref -> standW );


  return;

} /* xtmEsEditStanding */


/*----------------------------------------------------------------------*/

static Widget
  createStandWindow( Widget         parent,
                     STAND_REC_REF  stand_ref )
{

  /* Variables. */
  int                     char_read;
  int                     index;
  char                    buffer[ 80 ];
  char                    label[ 20 ];
  char                    *day_name_ref;
  Arg                     args[ 10 ];
  Cardinal                n;
  Dimension               width;
  Widget                  dataLocalW[ 29 ];
  Widget                  standFd;
  Widget                  tempW;
  Widget                  valDaysBu[ 10 ];
  Widget                  workFo;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static XIT_PUSH_STRUCT action_button_def[] = {
    { "SkipWeekPb", "", "", True, NULL },
  };

  static XIT_TEXT_STRUCT text_buffer_def[] = {
    { "EveryTx",    NULL, 1, True },
  };

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "",   okCB,     NULL },
    { NULL, NULL,     NULL },
    { "",   cancelCB, NULL },
    { NULL, NULL,     NULL },
    { "",   helpCB,   NULL },
  };


  /* Code. */

  custom_data_ref = stand_ref -> appl_data_ref -> custom_data;

  /* Text items. */
  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = stand_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 2 ].data  = stand_ref;
  action_buttons[ 4 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 4 ].data  = stand_ref;

  action_button_def[ 0 ].title = msgGetText( MXDI_SKIP_BUTTON );


  /* Create a form dialog with buttons. */
  standFd = xitCreateFormDialog( parent, "StandFd",
                                 1, 0,
                                 action_buttons, 
                                 XtNumber( action_buttons ) );

  XtAddCallback( standFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) stand_ref );

  /* We want to know when this window is closed. */
  {
    Atom  wm_delete_window;

    wm_delete_window = XmInternAtom( XtDisplay( standFd ),
                                     "WM_DELETE_WINDOW", False );

    XmAddWMProtocols( XtParent( standFd ), &wm_delete_window, 1 );
    XmAddWMProtocolCallback( XtParent( standFd ), wm_delete_window, 
                             (XtCallbackProc) cancelCB,
                             (XtPointer) stand_ref );
  } /* block */


  n = 0;
  XtSetArg( args[ n ], XmNdialogStyle, XmDIALOG_APPLICATION_MODAL ); n++;
  XtSetValues( standFd, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_STAND_TITLE ) ); n++;
  XtSetValues( XtParent( standFd ), args, n );


  /* Container for the contents of the window. */
  workFo = XtNameToWidget( standFd, "StandFdFo" );


  /* Standing entry is valid from/to. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  timeRc = XmCreateRowColumn( workFo, "TimeRc", args, n );

  betweenLa = xitCreateLabel( timeRc, "BetweenLa", 
                              msgGetText( MXDI_BETWEEN_DATES ), -1 );

  rangeTb = xtmFlCreateRangeField( timeRc, "RangeTb" );


  /* Vhat type of repeating entry do we have? */
  typeFr = XmCreateFrame( workFo, "TypeFr", args, 0 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNradioBehavior, True ); n++;
  typeRb = XmCreateRowColumn( typeFr, "TypeRb", args, n );

  typeDayTb = xitCreateToggleButton( 
                typeRb, "TypeDayTb", 
                msgGetText( MXDI_REPEATED_TYPE_DAY_LABEL ), False );

  XtAddCallback( typeDayTb, XmNvalueChangedCallback, 
                 (XtCallbackProc) typeDayCB, (XtPointer) stand_ref );

  typeIntTb = xitCreateToggleButton( 
                typeRb, "TypeIntTb", 
                msgGetText( MXDI_REPEATED_TYPE_INT_LABEL ), False );

  XtAddCallback( typeIntTb, XmNvalueChangedCallback, 
                 (XtCallbackProc) typeIntCB, (XtPointer) stand_ref );


  /* Entry is valid on specific days. */
  valOnDaysFr = XmCreateFrame( workFo, "ValOnDaysFr", args, 0 );

  n = 0;
  valOnDaysRc = XmCreateRowColumn( valOnDaysFr, "ValOnDaysRc", args, n );

  /* Valid on the days. */
  validDaysLa = xitCreateLabel( valOnDaysRc, "ValidDaysLa", 
                                msgGetText( MXDI_STAND_DAYS_VALID_LABEL ),
                                -1 );

  /* Valid days. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking, XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNnumColumns, 2 ); n++;
  valRc = XmCreateRowColumn( valOnDaysRc, "ValRc", args, n );

  day_name_ref = msgGetText( MXDI_WEEKDAYS_MO_FIRST );

  for( index = 0; index < XtNumber( valDaysBu ); index++ ) {

    if( index < 7 ) {
      char_read = strlen( day_name_ref );
      sscanf( day_name_ref, "%s%n", label, &char_read );
      day_name_ref = day_name_ref + char_read;
    } else {
      strcpy( label, " " );
    }

    sprintf( buffer, "ValDays%dBu", index + 1 );

    valDaysBu[ index ] = xitCreateToggleButton( valRc, buffer, label, False );

    if( index >= 7 ) {
      n = 0;
      XtSetArg( args[ n ], XmNmappedWhenManaged, False ); n++;
      XtSetValues( valDaysBu[ index ], args, n );
    }

  }

  /* Valid on specific day in the month. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  dayInMonthRc = XmCreateRowColumn( valOnDaysRc, "DayInMonthRc", args, n );

  dayInMonthTb = xitCreateToggleButton( 
                   dayInMonthRc, "DayInMonthTb", 
                   msgGetText( MXDI_EVERY_IN_MONTH_LABEL ), False );

  XtAddCallback( dayInMonthTb, XmNvalueChangedCallback, 
                 (XtCallbackProc) dayInMonthCB, (XtPointer) stand_ref );

  dayInMonthFs = xtmFlCreateSelField(
                   dayInMonthRc, "DayInMonthFs",
                   10, msgGetText( MXDI_VALID_ON_DAY_IN_MONTH ),
                   ',', False,
                   (XtCallbackProc) NULL, (void *) stand_ref );

  tempW = xitFieldSelectGetChild( dayInMonthFs, xitFIELD_SELECT_TEXT_FIELD );

  n = 0;
  XtSetArg( args[ n ], XmNeditable, False ); n++;
  XtSetArg( args[ n ], XmNcursorPositionVisible, False ); n++;
  XtSetValues( tempW, args, n );

  dayInMonthLa = xitCreateLabel( dayInMonthRc, "DayInMonthLa", 
                                  msgGetText( MXDI_DAY_IN_MONTH_LABEL ), -1 );


  /* Entry is valid for a specific interval. */
  valIntFr = XmCreateFrame( workFo, "ValIntFr", args, 0 );

  n = 0;
  valIntRc = XmCreateRowColumn( valIntFr, "ValIntRc", args, n );


  valIntLa = xitCreateLabel( valIntRc, "ValIntLa", 
                             msgGetText( MXDI_VALID_ON_INT_DAYS_LABEL ),
                             -1 );

  /* Entry is valid every n. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  valIntEveryRc = XmCreateRowColumn( valIntRc, "ValIntEveryRc", args, n );

  valIntEveryLa = xitCreateLabel( valIntEveryRc, "", 
                                  msgGetText( MXDI_INT_EVERY_LABEL ), -1 );

  valIntEveryTx = xitCreateTextCols( valIntEveryRc, &text_buffer_def[ 0 ], 3 );

  n = 0;
  XtSetArg( args[ n ], XmNmaxLength, 3 );  n++;
  XtSetValues( valIntEveryTx, args, n );  

  valIntEveryFs = xtmFlCreateSelField(
                    valIntEveryRc, "ValIntEveryFs",
                    18, msgGetText( MXDI_VALID_EVERY ),
                    ',', False,
                    (XtCallbackProc) NULL, (void *) stand_ref );

  tempW = xitFieldSelectGetChild( valIntEveryFs, xitFIELD_SELECT_TEXT_FIELD );

  n = 0;
  XtSetArg( args[ n ], XmNeditable, False ); n++;
  XtSetArg( args[ n ], XmNcursorPositionVisible, False ); n++;
  XtSetValues( tempW, args, n );


  /* What do we do on workdays? */
  noWorkFr = XmCreateFrame( workFo, "NoWorkFr", args, 0 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  noWorkRc = XmCreateRowColumn( noWorkFr, "NoWorkRc", args, n );

  noWorkLa = xitCreateLabel( noWorkRc, "", 
                             msgGetText( MXDI_ACTION_NO_WORK_LABEL ), -1 );

  noWorkFs = xtmFlCreateSelField(
               noWorkRc, "NoWorkFs",
               15, msgGetText( MXDI_ACTION_DO_NO_WORK ),
               ',', False,
               (XtCallbackProc) NULL, (void *) stand_ref );

  tempW = xitFieldSelectGetChild( noWorkFs, xitFIELD_SELECT_TEXT_FIELD );

  n = 0;
  XtSetArg( args[ n ], XmNeditable, False ); n++;
  XtSetArg( args[ n ], XmNcursorPositionVisible, False ); n++;
  XtSetValues( tempW, args, n );


  /* Skip weeks window. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  doActionRc = XmCreateRowColumn( workFo, "DoActionRc", args, n );

  skipWeekPb = xitCreatePushButton( doActionRc, &action_button_def[ 0 ] );

  XtAddCallback( skipWeekPb, XmNactivateCallback, 
                 (XtCallbackProc) skipCB, (XtPointer) stand_ref );


  /* Create anchors. */
  n = 0;
  XtSetArg( args[ n ], XmNseparatorType,  XmNO_LINE );  n++;
  XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_POSITION );  n++;
  XtSetArg( args[ n ], XmNleftPosition,   50 );  n++;
  anchorSp = XmCreateSeparatorGadget( workFo, "AnchorSp", args, n );


  /* Put the elements together. */
  xitAttachWidget( timeRc,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( typeFr,
                   XmATTACH_WIDGET, timeRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( valOnDaysFr,
                   XmATTACH_WIDGET, typeFr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE, NULL );
  xitAttachWidget( valIntFr,
                   XmATTACH_WIDGET, typeFr, XmATTACH_WIDGET, valOnDaysFr,
                   XmATTACH_NONE,   NULL,   XmATTACH_NONE,   NULL );
  xitAttachWidget( noWorkFr,
                   XmATTACH_WIDGET, valOnDaysFr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,        XmATTACH_NONE, NULL );
  xitAttachWidget( doActionRc,
                   XmATTACH_WIDGET, noWorkFr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,     XmATTACH_NONE, NULL );

  n = 0;
  XtSetArg( args[ n ], XmNleftOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNtopOffset,    10 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset,  5 ); n++;
  XtSetValues( timeRc,      args, n );
  XtSetValues( typeFr,      args, n );
  XtSetValues( valOnDaysFr, args, n );
  XtSetValues( valIntFr,    args, n );
  XtSetValues( noWorkFr,    args, n );
  XtSetValues( doActionRc,  args, n );


  /* Manage the widgets. */
  XtManageChildren( valDaysBu, XtNumber( valDaysBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );


  /* Set the initial sizes. */
  xitSetSizeFormDialog( standFd, False );


  /* Attach the type and non-work frames in the middle. */
  n = 0;
  XtSetArg( args[ n ], XmNwidth, &width ); n++;
  XtGetValues( typeFr, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_WIDGET ); n++;
  XtSetArg( args[ n ], XmNleftWidget, anchorSp ); n++;
  XtSetArg( args[ n ], XmNleftOffset, (-1) * width / 2 ); n++;
  XtSetValues( typeFr, args, n );


  n = 0;
  XtSetArg( args[ n ], XmNwidth, &width ); n++;
  XtGetValues( noWorkFr, args, n );

  n = 0;
  XtSetArg( args[ n ], XmNleftAttachment, XmATTACH_WIDGET ); n++;
  XtSetArg( args[ n ], XmNleftWidget, anchorSp ); n++;
  XtSetArg( args[ n ], XmNleftOffset, (-1) * width / 2 ); n++;
  XtSetValues( noWorkFr, args, n );


  return( standFd );

} /* createStandWindow */


/*----------------------------------------------------------------------*/

static Boolean
  fetchStandValues( STAND_REC_REF          stand_ref,
                    XTM_ES_GET_STAND_DATA  *values_ref )
{

  /* Variables. */
  Boolean               ok;
  int                   index;
  int                   temp;
  char                  buffer[ 50 ];
  char                  *char_ref;
  Widget                baseW;
  Widget                mainW;
  Widget                tempW;
  TIM_TIME_REF          time_temp;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  appl_data_ref = stand_ref -> appl_data_ref;

  mainW = XtNameToWidget( stand_ref -> standW, "StandFdFo" );


  /* Default values. */
  values_ref -> from        = 0;
  values_ref -> to          = 0;
  values_ref -> every_n     = 0;
  values_ref -> valid_every = 0;

  for( index = 0; index < 7; index++ )
    values_ref -> valid_days[ index ] = False;

  values_ref -> flags = stand_ref -> entry_flags;

  flagClear( values_ref -> flags, XTM_DB_FLAG_SE_DAY_IN_MONTH );
  flagClear( values_ref -> flags, XTM_DB_FLAG_SE_1ST );
  flagClear( values_ref -> flags, XTM_DB_FLAG_SE_2ND );
  flagClear( values_ref -> flags, XTM_DB_FLAG_SE_3RD );
  flagClear( values_ref -> flags, XTM_DB_FLAG_SE_1ST_LAST );
  flagClear( values_ref -> flags, XTM_DB_FLAG_SE_2ND_LAST );
  flagClear( values_ref -> flags, XTM_DB_FLAG_SE_3RD_LAST );
  flagClear( values_ref -> flags, XTM_DB_FLAG_SE_NWDAY_PREV );
  flagClear( values_ref -> flags, XTM_DB_FLAG_SE_NWDAY_NEXT );
  flagClear( values_ref -> flags, XTM_DB_FLAG_SE_NWDAY_SKIP );


  /* Fetch the date range. */
  tempW = XtNameToWidget( mainW, "TimeRc.RangeTb" );

  ok = xtmFoFetchDate( mainW, tempW, XTM_FO_START_DATE, True, &time_temp );
  if( ! ok )
    return( False );

  if( time_temp > 0 )
    values_ref -> from = time_temp;

  ok = xtmFoFetchDate( mainW, tempW, XTM_FO_END_DATE, True, &time_temp );
  if( ! ok )
    return( False );

  if( time_temp > 0 )
    values_ref -> to = time_temp;


  /* Make sure there is a valid range of dates. */
  if( values_ref -> from != 0 && values_ref -> to != 0 &&
      values_ref -> to < values_ref -> from ) {
    temp               = values_ref -> to;
    values_ref -> to   = values_ref -> from;
    values_ref -> from = temp;
  }


  /* Only for standing entries. */
  if( stand_ref -> entry_type != XTM_DB_TYPE_STAND_APPOINTMENT &&
      stand_ref -> entry_type != XTM_DB_TYPE_STAND_NOTE )
    return( True );


  /* Selected days in week? */
  tempW = XtNameToWidget( mainW, "TypeFr.TypeRb.TypeDayTb" );
  if( XmToggleButtonGetState( tempW ) ) {

    /* Valid days. */
    baseW = XtNameToWidget( mainW, "ValOnDaysFr.ValOnDaysRc.ValRc" );

    for( index = 0; index < 7; index++ ) {
      sprintf( buffer, "ValDays%dBu", index + 1 );
      tempW = XtNameToWidget( baseW, buffer );

      if( XmToggleButtonGetState( tempW ) )
        values_ref -> valid_days[ index ] = True;
    }


    /* Valid the nth weekday in month? */
    baseW = XtNameToWidget( mainW, "ValOnDaysFr.ValOnDaysRc.DayInMonthRc" );
    tempW = XtNameToWidget( baseW, "DayInMonthTb" );

    if( XmToggleButtonGetState( tempW ) ) {

      flagSet( values_ref -> flags, XTM_DB_FLAG_SE_DAY_IN_MONTH );

      tempW = XtNameToWidget( baseW, "DayInMonthFs" );
      (void) xitFieldSelectGetIndex( tempW, &index );

      if( index == 0 )
        flagSet( values_ref -> flags, XTM_DB_FLAG_SE_1ST );
      else if( index == 1 )
        flagSet( values_ref -> flags, XTM_DB_FLAG_SE_2ND );
      else if( index == 2 )
        flagSet( values_ref -> flags, XTM_DB_FLAG_SE_3RD );
      else if( index == 3 )
        flagSet( values_ref -> flags, XTM_DB_FLAG_SE_3RD_LAST );
      else if( index == 4 )
        flagSet( values_ref -> flags, XTM_DB_FLAG_SE_2ND_LAST );
      else if( index == 5 )
        flagSet( values_ref -> flags, XTM_DB_FLAG_SE_1ST_LAST );

    } /* if */

    /* For day in month, a day must be selected. */
    ok = False;

    for( index = 0; index < 7; index++ ) {
      if( values_ref -> valid_days[ index ] )
        ok = True;
    }

    if( ! ok ) {
      xitErMessage( stand_ref -> standW, XIT_ER_ERROR,
                    module_name, "fetchStandValues",
                    msgGetText( MXDI_NO_DAYS_IN_WEEK_SEL ) );

      return( False );
    }

  } /* if */


  /* Valid every n? */
  tempW = XtNameToWidget( mainW, "TypeFr.TypeRb.TypeIntTb" );
  if( XmToggleButtonGetState( tempW ) ) {

    baseW = XtNameToWidget( mainW, "ValIntFr.ValIntRc.ValIntEveryRc" );

    tempW = XtNameToWidget( baseW, "EveryTx" );
    char_ref = xitStringGetText( tempW );

    index = sscanf( char_ref, "%d", &temp );
    SysFree( char_ref );

    if( index != 1 ) {
      xitErMessage( stand_ref -> standW, XIT_ER_ERROR,
                    module_name, "fetchStandValues",
                    msgGetText( MXDI_ENTRY_INTERVAL_MISSING ) );

      return( False );
    }

    values_ref -> every_n = temp;


    if( values_ref -> every_n > 0 ) {
      tempW = XtNameToWidget( baseW, "ValIntEveryFs" );

      (void) xitFieldSelectGetIndex( tempW, &index );

      if( index == 4 )
        values_ref -> valid_every = XTM_DB_VALID_YEAR;
      else if( index == 3 )
        values_ref -> valid_every = XTM_DB_VALID_MONTH_LAST;
      else if( index == 2 )
        values_ref -> valid_every = XTM_DB_VALID_MONTH;
      else if( index == 1 )
        values_ref -> valid_every = XTM_DB_VALID_WEEK;
      else
        values_ref -> valid_every = XTM_DB_VALID_DAY;

      for( index = 0; index < 7; index++ )
        values_ref -> valid_days[ index ] = False;
    }

    /* If valid every n, we must have a start date. */
    if( values_ref -> from == 0 ) {
      xitErMessage( stand_ref -> standW, XIT_ER_ERROR,
                    module_name, "fetchStandValues",
                    msgGetText( MXDI_START_DATE_MISSING ) );

      return( False );
    }

  } /* if */



  /* Action if non-workday. */
  baseW = XtNameToWidget( mainW, "NoWorkFr.NoWorkRc" );
  tempW = XtNameToWidget( baseW, "NoWorkFs" );

  (void) xitFieldSelectGetIndex( tempW, &index );

  if( index == 1 )
    flagSet( values_ref -> flags, XTM_DB_FLAG_SE_NWDAY_SKIP );
  else if( index == 2 )
    flagSet( values_ref -> flags, XTM_DB_FLAG_SE_NWDAY_NEXT );
  else if( index == 3 )
    flagSet( values_ref -> flags, XTM_DB_FLAG_SE_NWDAY_PREV );


  return( True );

} /* fetchStandValues */


/*----------------------------------------------------------------------*/

static void
  setDefaultStandValues( STAND_REC_REF  stand_ref )
{

  /* Variables. */
  int     index;
  char    buffer[ 256 ];
  Widget  baseW;
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  mainW = XtNameToWidget( stand_ref -> standW, "StandFdFo" );


  /* Valid from/to. */
  baseW = XtNameToWidget( mainW, "TimeRc" );

  tempW = XtNameToWidget( baseW, "RangeTb" );
  XmUbTimeBoxSetStartDateString( tempW, "" );
  XmUbTimeBoxSetEndDateString( tempW, "" );
 

  /* What kind of repeated? */
  baseW = XtNameToWidget( mainW, "TypeFr.TypeRb" );

  tempW = XtNameToWidget( baseW, "TypeDayTb" );
  XmToggleButtonSetState( tempW, False, False );

  tempW = XtNameToWidget( baseW, "TypeIntTb" );
  XmToggleButtonSetState( tempW, False, False );


  /* Valid days. */
  baseW = XtNameToWidget( mainW, "ValOnDaysFr.ValOnDaysRc.ValRc" );

  for( index = 0; index < 7; index++ ) {
    sprintf( buffer, "ValDays%dBu", index + 1 );

    tempW = XtNameToWidget( baseW, buffer );
    XmToggleButtonSetState( tempW, False, False );
  }

  baseW = XtNameToWidget( mainW, "ValOnDaysFr.ValOnDaysRc.DayInMonthRc" );

  tempW = XtNameToWidget( baseW, "DayInMonthFs" );
  xitFieldSelectSetIndex( tempW, 0, False );

  tempW = XtNameToWidget( baseW, "DayInMonthTb" );
  XmToggleButtonSetState( tempW, True,  False );
  XmToggleButtonSetState( tempW, False, True );


  /* Valid every n something. */
  baseW = XtNameToWidget( mainW, "ValIntFr.ValIntRc.ValIntEveryRc" );

  tempW = XtNameToWidget( baseW, "EveryTx" );
  XmTextSetString( tempW, " " );

  tempW = XtNameToWidget( baseW, "ValIntEveryFs" );
  xitFieldSelectSetIndex( tempW, 0, False );


  /* Action if non-workday. */
  baseW = XtNameToWidget( mainW, "NoWorkFr.NoWorkRc" );

  tempW = XtNameToWidget( baseW, "NoWorkFs" );
  xitFieldSelectSetIndex( tempW, 0, False );


  /* Everything is valid. */
  tempW = XtNameToWidget( mainW, "TypeFr" );
  XtSetSensitive( tempW, True );

  tempW = XtNameToWidget( mainW, "ValOnDaysFr" );
  XtSetSensitive( tempW, True );

  tempW = XtNameToWidget( mainW, "ValIntFr" );
  XtSetSensitive( tempW, True );

  tempW = XtNameToWidget( mainW, "NoWorkFr" );
  XtSetSensitive( tempW, True );

  tempW = XtNameToWidget( mainW, "DoActionRc" );
  XtSetSensitive( tempW, True );


  return;

} /* setDefaultStandValues */


/*----------------------------------------------------------------------*/

static void
  setStandValues( STAND_REC_REF          stand_ref,
                  XTM_ES_SET_STAND_DATA  *values_ref )
{

  /* Variables. */
  Boolean               set_value;
  Boolean               valid;
  int                   index;
  char                  buffer[ 80 ];
  Widget                baseW;
  Widget                mainW;
  Widget                tempW;
  XTM_GL_BASE_DATA_REF  appl_data_ref;


  /* Code. */

  mainW = XtNameToWidget( stand_ref -> standW, "StandFdFo" );

  stand_ref -> entry_type     = values_ref -> entry_type;
  stand_ref -> default_date   = values_ref -> default_date;
  stand_ref -> skip_week[ 0 ] = values_ref -> skip_week[ 0 ];
  stand_ref -> skip_week[ 1 ] = values_ref -> skip_week[ 1 ];


  /* Default values. */
  setDefaultStandValues( stand_ref );

 
  /* Valid from/to. */
  tempW = XtNameToWidget( mainW, "TimeRc.RangeTb" );
  if( values_ref -> from == 0 )
    XmUbTimeBoxSetStartDateString( tempW, "" );
  else
    XmUbTimeBoxSetStartDate( tempW, values_ref -> from );

  tempW = XtNameToWidget( mainW, "TimeRc.RangeTb" );
  if( values_ref -> to == 0 )
    XmUbTimeBoxSetEndDateString( tempW, "" );
  else
    XmUbTimeBoxSetEndDate( tempW, values_ref -> to );


  /* For sticky notes, this is it! */
  if( values_ref -> entry_type == XTM_DB_TYPE_STICKY_NOTE ) {
    tempW = XtNameToWidget( mainW, "TypeFr" );
    XtSetSensitive( tempW, False );

    tempW = XtNameToWidget( mainW, "ValOnDaysFr" );
    XtSetSensitive( tempW, False );

    tempW = XtNameToWidget( mainW, "ValIntFr" );
    XtSetSensitive( tempW, False );

    tempW = XtNameToWidget( mainW, "NoWorkFr" );
    XtSetSensitive( tempW, False );

    tempW = XtNameToWidget( mainW, "DoActionRc" );
    XtSetSensitive( tempW, False );

    return;
  }


  /* Selected days in week? */
  if( values_ref -> every_n == 0 ) {

    /* Valid days. */
    baseW = XtNameToWidget( mainW, "ValOnDaysFr.ValOnDaysRc.ValRc" );
    valid = False;

    for( index = 0; index < 7; index++ ) {
      sprintf( buffer, "ValDays%dBu", index + 1 );
      tempW = XtNameToWidget( baseW, buffer );

      if( ! values_ref -> valid_days[ index ] ) {
        XmToggleButtonSetState( tempW, False, False );
      } else {
        XmToggleButtonSetState( tempW, True, False );
        valid = True;
      }
    } /* loop */

    if( ! valid && values_ref -> from != 0 ) {
      sprintf( buffer, "ValDays%dBu", 
               TimIndexOfDayInIsoWeek( values_ref -> from ) );

      tempW = XtNameToWidget( baseW, buffer );
      XmToggleButtonSetState( tempW, True, False );
    }


    /* Valid the nth weekday in month? */
    baseW = XtNameToWidget( mainW, "ValOnDaysFr.ValOnDaysRc.DayInMonthRc" );

    if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_SE_DAY_IN_MONTH ) ) {
      tempW = XtNameToWidget( baseW, "DayInMonthFs" );

      if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_SE_1ST ) )
        xitFieldSelectSetIndex( tempW, 0, False );
      else if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_SE_2ND ) )
        xitFieldSelectSetIndex( tempW, 1, False );
      else if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_SE_3RD ) )
        xitFieldSelectSetIndex( tempW, 2, False );
      else if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_SE_3RD_LAST ) )
        xitFieldSelectSetIndex( tempW, 3, False );
      else if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_SE_2ND_LAST ) )
        xitFieldSelectSetIndex( tempW, 4, False );
      else if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_SE_1ST_LAST ) )
        xitFieldSelectSetIndex( tempW, 5, False );

      tempW = XtNameToWidget( baseW, "DayInMonthTb" );
      XmToggleButtonSetState( tempW, True, True );

    } else {
      tempW = XtNameToWidget( baseW, "DayInMonthTb" );
      XmToggleButtonSetState( tempW, False, True );

    } /* if */

  } /* if */


  /* Valid every n something. */
  if( values_ref -> every_n > 0 ) {

    baseW = XtNameToWidget( mainW, "ValIntFr.ValIntRc.ValIntEveryRc" );

    sprintf( buffer, "%d", values_ref -> every_n );
    tempW = XtNameToWidget( baseW, "EveryTx" );
    XmTextSetString( tempW, buffer );

    tempW = XtNameToWidget( baseW, "ValIntEveryFs" );

    if( values_ref -> valid_every == XTM_DB_VALID_YEAR )
      xitFieldSelectSetIndex( tempW, 4, False );
    else if( values_ref -> valid_every == XTM_DB_VALID_MONTH_LAST )
      xitFieldSelectSetIndex( tempW, 3, False );
    else if( values_ref -> valid_every == XTM_DB_VALID_MONTH )
      xitFieldSelectSetIndex( tempW, 2, False );
    else if( values_ref -> valid_every == XTM_DB_VALID_WEEK )
      xitFieldSelectSetIndex( tempW, 1, False );
    else
      xitFieldSelectSetIndex( tempW, 0, False );

  } /* if */


  /* Action if non-workday. */
  baseW = XtNameToWidget( mainW, "NoWorkFr.NoWorkRc" );
  tempW = XtNameToWidget( baseW, "NoWorkFs" );

  if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_SE_NWDAY_SKIP ) )
    xitFieldSelectSetIndex( tempW, 1, False );
  else if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_SE_NWDAY_NEXT ) )
    xitFieldSelectSetIndex( tempW, 2, False );
  else if( flagIsSet( values_ref -> flags, XTM_DB_FLAG_SE_NWDAY_PREV ) )
    xitFieldSelectSetIndex( tempW, 3, False );


  /* What kind of repeated? */
  baseW = XtNameToWidget( mainW, "TypeFr.TypeRb" );
  if( values_ref -> every_n == 0 )
    tempW = XtNameToWidget( baseW, "TypeDayTb" );
  else
    tempW = XtNameToWidget( baseW, "TypeIntTb" );

  XmToggleButtonSetState( tempW, True, True );


  return;

} /* setStandValues */


/*----------------------------------------------------------------------*/

static void 
  cancelCB( Widget         widget,
            STAND_REC_REF  stand_ref,
            XtPointer      call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( stand_ref -> actionCB != NULL )
    (* stand_ref -> actionCB)( XTM_ES_REASON_CANCEL,
                               stand_ref -> user_data, NULL );

  XtUnmanageChild( stand_ref -> standW );


  return;

} /* cancelCB */


/*----------------------------------------------------------------------*/

static void 
  dayInMonthCB( Widget                        widget,
                STAND_REC_REF                 stand_ref,
                XmToggleButtonCallbackStruct  *call_data )
{

  /* Variables. */
  Widget  baseW;
  Widget  mainW;
  Widget  temp1W;
  Widget  temp2W;


  /* Code. */

  mainW = XtNameToWidget( stand_ref -> standW, "StandFdFo" );
  baseW = XtNameToWidget( mainW, "ValOnDaysFr.ValOnDaysRc.DayInMonthRc" );

  temp1W = XtNameToWidget( baseW, "DayInMonthFs" );
  temp2W = XtNameToWidget( baseW, "DayInMonthLa" );

  if( call_data -> set ) {
    XtSetSensitive( temp1W, True );
    XtSetSensitive( temp2W, True );
  } else {
    XtSetSensitive( temp1W, False );
    XtSetSensitive( temp2W, False );
  }


  return;

} /* typeDayCB */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget         widget,
             STAND_REC_REF  stand_ref,
             XtPointer      call_data )
{

  /* Code. */

  /* Do we have a user action callback registered? */
  if( stand_ref -> actionCB != NULL )
    (* stand_ref -> actionCB)( XTM_ES_REASON_DESTROY,
                               stand_ref -> user_data, NULL );

  /* Skip window? */
  if( stand_ref -> skip_handle != NULL )
    xtmEkDestroy( stand_ref -> skip_handle );

  SysFree( stand_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget         widget,
          STAND_REC_REF  stand_ref,
          XtPointer      call_data )
{

  /* Code. */

  xtmHlDisplayHelp( stand_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    stand_window_id, "" );


  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void 
  okCB( Widget         widget,
        STAND_REC_REF  stand_ref,
        XtPointer      call_data )
{

  /* Variables. */
  Boolean                ok;
  XTM_ES_GET_STAND_DATA  values;


  /* Code. */

  ok = fetchStandValues( stand_ref, &values );
  if( ! ok )
    return;


  /* Values from the skip week window. */
  values.skip_week[ 0 ] = stand_ref -> skip_week[ 0 ];
  values.skip_week[ 1 ] = stand_ref -> skip_week[ 1 ];


  /* Do we have a user action callback registered? */
  if( stand_ref -> actionCB != NULL )
    (* stand_ref -> actionCB)( XTM_ES_REASON_OK,
                               stand_ref -> user_data,
                               &values );

  XtUnmanageChild( stand_ref -> standW );


  return;

} /* okCB */


/*----------------------------------------------------------------------*/

static void 
  skipCB( Widget         widget,
          STAND_REC_REF  stand_ref,
          XtPointer      call_data )
{

  /* Variables. */
  XTM_EK_SET_SKIP_DATA  values;


  /* Code. */

  /* Create the skip window? */
  if( stand_ref -> skip_handle == NULL )
    stand_ref -> skip_handle = xtmEkInitialize( stand_ref -> appl_data_ref,
                                                stand_ref -> standW,
                                                skipActionCB,
                                                (void *) stand_ref );

  /* Display the skip window. */
  values.default_date   = stand_ref -> default_date;
  values.skip_week[ 0 ] = stand_ref -> skip_week[ 0 ];
  values.skip_week[ 1 ] = stand_ref -> skip_week[ 1 ];

  xtmEkEditSkip( stand_ref -> skip_handle,
                 &values );

  return;

} /* skipCB */


/*----------------------------------------------------------------------*/

static void 
  skipActionCB( XTM_EK_REASON         reason,
                void                  *user_data,
                XTM_EK_GET_SKIP_DATA  *values_ref )
{

  /* Variables. */
  STAND_REC_REF  stand_ref;


  /* Code. */

  if( user_data == NULL )
    return;

  /* Our private data. */
  stand_ref = (STAND_REC_REF) user_data;


  /* Is the skip window destroyed? */
  if( reason == XTM_EK_REASON_DESTROY ) {
    stand_ref -> skip_handle = NULL;

    return;
  }


  /* OK confirmation? */
  if( reason == XTM_EK_REASON_OK ) {
    stand_ref -> skip_week[ 0 ] = values_ref -> skip_week[ 0 ];
    stand_ref -> skip_week[ 1 ] = values_ref -> skip_week[ 1 ];

    return;
  }


  return;

} /* skipActionCB */


/*----------------------------------------------------------------------*/

static void 
  typeDayCB( Widget                        widget,
             STAND_REC_REF                 stand_ref,
             XmToggleButtonCallbackStruct  *call_data )
{

  /* Variables. */
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  /* Only react when buttons are selected. */
  if( ! call_data -> set )
    return;


  mainW = XtNameToWidget( stand_ref -> standW, "StandFdFo" );

  tempW = XtNameToWidget( mainW, "ValOnDaysFr" );
  XtSetSensitive( tempW, True );

  tempW = XtNameToWidget( mainW, "ValIntFr" );
  XtSetSensitive( tempW, False );


  return;

} /* typeDayCB */


/*----------------------------------------------------------------------*/

static void 
  typeIntCB( Widget                        widget,
             STAND_REC_REF                 stand_ref,
             XmToggleButtonCallbackStruct  *call_data )
{

  /* Variables. */
  Widget  mainW;
  Widget  tempW;


  /* Code. */

  /* Only react when buttons are selected. */
  if( ! call_data -> set )
    return;


  mainW = XtNameToWidget( stand_ref -> standW, "StandFdFo" );

  tempW = XtNameToWidget( mainW, "ValIntFr" );
  XtSetSensitive( tempW, True );

  tempW = XtNameToWidget( mainW, "ValOnDaysFr" );
  XtSetSensitive( tempW, False );


  return;

} /* typeIntCB */

