/*----------------------------------------------------------------------------
--
--  Module:           xtmDuplEntry
--
--  Project:          Xdiary
--  System:           xtm - X Desktop Calendar
--    Subsystem:      <>
--    Function block: <>
--
--  Description:
--    Taken an entry and duplicate it, i.e. define it on other days.
--    Only normal appointments and notes can be duplicated, standing
--    appointments and notes cannot.
--
--  Filename:         xtmDuplEntry.c
--
--  Authors:          Roger Larsson, Ulrika Bornetun
--  Creation date:    1991-05-03
--
--
--  (C) Copyright Ulrika Bornetun, Roger Larsson (1995)
--      All rights reserved
--
--  Permission to use, copy, modify, and distribute this software and its
--  documentation for any purpose and without fee is hereby granted,
--  provided that the above copyright notice appear in all copies. Ulrika
--  Bornetun and Roger Larsson make no representations about the usability
--  of this software for any purpose. It is provided "as is" without express
--  or implied warranty.
----------------------------------------------------------------------------*/

/* SCCS module identifier. */
static char SCCSID[] = "@(#) Module: xtmDuplEntry.c, Version: 1.1, Date: 95/02/18 15:52:12";


/*----------------------------------------------------------------------------
--  Include files
----------------------------------------------------------------------------*/

#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/RowColumn.h>
#include <Xm/ToggleB.h>

#include "System.h"
#include "Message.h"
#include "TimDate.h"

#include "msgXdiary.h"
#include "xtmGlobal.h"
#include "xtmCalDb.h"
#include "xtmDbTools.h"
#include "xtmFields.h"
#include "xtmFormat.h"
#include "xtmHelp.h"
#include "xtmTools.h"
#include "xtmUpdate.h"
#include "xitError.h"
#include "xitFieldSel.h"
#include "xitTools.h"
#include "XmUbTimeB.h"
#include "xtmDuplEntry.h"


/*----------------------------------------------------------------------------
--  Macro definitions
----------------------------------------------------------------------------*/

/* Local widgets in the duplicate window. */
#define databaseFs          dataLocalW[  0 ]
#define databaseLa          dataLocalW[  1 ]
#define dateSelLa           dataLocalW[  2 ]
#define dateSelRc           dataLocalW[  3 ]
#define dbRc                dataLocalW[  4 ]
#define messageLa           dataLocalW[  5 ]
#define rangeTb             dataLocalW[  6 ]
#define valDaysRc           dataLocalW[  7 ]
#define valFr               dataLocalW[  8 ]
#define valLa               dataLocalW[  9 ]
#define valRc               dataLocalW[ 10 ]


/*----------------------------------------------------------------------------
--  Type declarations
----------------------------------------------------------------------------*/

/* Information about the entry to duplicate. */
typedef struct {

  /* Application wide data. */
  XTM_GL_BASE_DATA_REF  appl_data_ref;

  /* Entry being duplicated. */
  XTM_DB_ALL_ENTRY_REF  use_entry_ref;

  /* Duplicate entry from/to. */
  TIM_TIME_REF  from_date;
  TIM_TIME_REF  to_date;

  /* Duplicate entry to database. */
  char  db_name[ XTM_GL_MAX_CAL_NAME + 1 ];

  /* The duplicate window. */
  Widget  duplW;

} ENTRY_INFO, *ENTRY_INFO_REF;


/*----------------------------------------------------------------------------
--  Global definitions
----------------------------------------------------------------------------*/

/* Name of module. */
static char  *module_name = "xtmDuplicateEntry";

/* IDs for the help windows. */
static char  *dupl_window_id = "Duplicate";


/*----------------------------------------------------------------------------
--  Function prototypes
----------------------------------------------------------------------------*/

static Widget
  createDuplWindow( Widget          parent,
                    ENTRY_INFO_REF  entry_ref );

static void 
  destroyCB( Widget          widget,
             ENTRY_INFO_REF  entry_ref,
             XtPointer       call_data );

static void
  duplicateCB( Widget          widget,
               ENTRY_INFO_REF  entry_ref,
               XtPointer       call_data );

static void 
  helpCB( Widget          widget,
          ENTRY_INFO_REF  entry_ref,
          XtPointer       call_data );

static void
  okCB( Widget          widget,
        ENTRY_INFO_REF  entry_ref,
        XtPointer       call_data );



/*----------------------------------------------------------------------------
--  Functions
----------------------------------------------------------------------------*/

void 
  xtmDpDuplicateEntry( XTM_GL_BASE_DATA_REF  appl_data_ref,
                       Widget                parent,
                       char                  *db_name,
                       UINT32                entry_id )
{

  /* Variables. */
  Boolean                 ok;
  char                    appointment_buffer[ 50 ];
  char                    buffer[ 256 ];
  Widget                  mainW;
  Widget                  tempW;
  ENTRY_INFO_REF          entry_ref;
  TIM_TIME_REF            tmp_time;
  XTM_DB_ALL_ENTRY_REF    use_entry_ref;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_STATUS           db_status;
  XTM_CD_CAL_INFO         db_info;


  /* Code. */

  /* Fetch database information. */
  ok = xtmCdFetchNamedDb( appl_data_ref -> custom_data -> cal_db_handle,
                          db_name,
                          &db_info, NULL );
  if( ! ok )
    return;

  /* We need read access. */
  if( ! flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_READ ) ) {
    xitErMessage( parent, XIT_ER_ERROR, 
                  module_name, "xtmDpDuplicateEntry",
                  msgGetText( MXDI_ERRMSG_NO_ACCESS_DB ) );

    return;
  }


  /* Create and initialize our private data. */
  entry_ref = SysNew( ENTRY_INFO );
  if( entry_ref == NULL )
    return;

  entry_ref -> appl_data_ref = appl_data_ref;
  entry_ref -> use_entry_ref = SysNew( XTM_DB_ALL_ENTRY_DEF );

  entry_ref -> use_entry_ref -> all_text = NULL;


  /* Open the database and fetch the entry. */
  ok = xtmDmOpenDatabase( appl_data_ref, db_name, XTM_DB_FLAG_MODE_READ,
                          &database );
  if( ! ok )
    raise exception;


  /* Fetch the entry. */
  db_status = xtmDbFetchEntry( &database, entry_id, 
                               entry_ref -> use_entry_ref, 
                               &entry_ref -> use_entry_ref -> all_text );

  xtmDbCloseEntryDb( &database );


  if( db_status != XTM_DB_OK ) {
    xitErMessage( parent, XIT_ER_ERROR,
                  module_name, "xtmDpDuplicateEntry",
                  msgGetText( MXDI_NO_ENTRY ) );

    raise exception;
  }

  use_entry_ref = entry_ref -> use_entry_ref;

  /* Standing/Sticky entries cannot be duplicated. */
  if( use_entry_ref -> entry.entry_category == XTM_DB_REP_ENTRY_LIST ||
      use_entry_ref -> entry.entry_category == XTM_DB_STICKY_LIST ) {

    xitErMessage( parent, XIT_ER_ERROR,
                  module_name, "xtmDpDuplicateEntry",
                  msgGetText( MXDI_NO_DB_SELECTED ) );

    raise exception;
  }


  /* Create the duplicate window. */
  entry_ref -> duplW = createDuplWindow( parent, entry_ref );


  /* Depending on the type of the entry, take different actions. */
  switch( use_entry_ref -> entry.entry_type ) {
    case XTM_DB_DAY_NOTE:
      sprintf( appointment_buffer, "%s", 
               msgGetText( MXDI_SIMPLE_NOTE_FLAG ) );
      break;

    case XTM_DB_DAY_ENTRY:
      xtmFoFormatTime( use_entry_ref -> entry.time_stamp,
                       appointment_buffer, sizeof( appointment_buffer ) );
      break;
  } /* switch */


  mainW = XtNameToWidget( entry_ref -> duplW, "DuplFdFo" );


  /* Label with what is moved. */
  sprintf( buffer, msgGetText( MXDI_DUPLICATE_APPOINTMENT ), 
           appointment_buffer );

  tempW = XtNameToWidget( mainW, "MessageLa" );
  xitStringSetLabel( tempW, buffer );


  /* The fields containing from and to date. */
  tmp_time = use_entry_ref -> entry.date_stamp;
  TimAddDays( &tmp_time, 1 );

  tempW = XtNameToWidget( mainW, "DateSelRc.RangeTb" );
  XmUbTimeBoxSetStartDate( tempW, tmp_time );
  XmUbTimeBoxSetEndDate( tempW, tmp_time );


  /* Database name. */
  tempW = XtNameToWidget( mainW, "DbRc.DatabaseFs" );
  if( flagIsSet( db_info.operations, XTM_DB_FLAG_MODE_WRITE ) )
    xitFieldSelectSetCurrent( tempW, use_entry_ref -> db_name, False );
  else
    xitFieldSelectSetCurrent( tempW, "", False );


  /* Make the form visible. */
  XtManageChild( entry_ref -> duplW );


  return;


  /* Exception handler. */
  exception:
    SysFree( entry_ref -> use_entry_ref -> all_text );
    SysFree( entry_ref -> use_entry_ref );
    SysFree( entry_ref );

    return;

} /* xtmDpDuplicateEntry */


/*----------------------------------------------------------------------*/

static Widget
  createDuplWindow( Widget          parent,
                    ENTRY_INFO_REF  entry_ref )
{

  /* Variables. */
  int                     char_read;
  int                     index;
  char                    buffer[ 50 ];
  char                    label[ 20 ];
  char                    *day_name_ref;
  Arg                     args[ 10 ];
  Cardinal                n;
  Widget                  dataLocalW[ 11 ];
  Widget                  duplFd;
  Widget                  valDaysBu[ 7 ];
  Widget                  workFo;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;

  static XIT_ACTION_AREA_ITEM  action_buttons[] = {
    { "", okCB,   NULL },
    { "", NULL,   NULL },
    { "", helpCB, NULL },
  };


  /* Code. */

  appl_data_ref   = entry_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;


  /* Text items. */
  action_buttons[ 0 ].label = msgGetText( MXDI_OK_BUTTON );
  action_buttons[ 0 ].data  = entry_ref;
  action_buttons[ 1 ].label = msgGetText( MXDI_CANCEL_BUTTON );
  action_buttons[ 1 ].data  = entry_ref;
  action_buttons[ 2 ].label = msgGetText( MXDI_HELP_BUTTON );
  action_buttons[ 2 ].data  = entry_ref;


  /* Create a form dialog with buttons. */
  duplFd = xitCreateFormDialog( parent, "DuplFd",
                                1, 0,
                                action_buttons,
                                XtNumber( action_buttons ) );

  n = 0;
  XtSetArg( args[ n ], XmNtitle, msgGetText( MXDI_DUPLICATE_TITLE ) ); n++;
  XtSetValues( XtParent( duplFd ), args, n );

  XtAddCallback( duplFd, XmNdestroyCallback, 
                 (XtCallbackProc) destroyCB, (XtPointer) entry_ref );

  /* Fetch container for the contents of the window. */
  workFo = XtNameToWidget( duplFd, "DuplFdFo" );


  /* Label indication what is moved. */
  messageLa = xitCreateLabel( workFo, "MessageLa", "", -1 );


  /* Date selection. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking,     XmPACK_TIGHT ); n++;
  dateSelRc = XmCreateRowColumn( workFo, "DateSelRc", args, n );

  dateSelLa = xitCreateLabel( dateSelRc, "DateSelLa", 
                              msgGetText( MXDI_BETWEEN_DATES ), -1 );


  /* Date range selector. */
  rangeTb = xtmFlCreateRangeField( dateSelRc, "RangeTb" );


  /* Valid days form. */
  valFr = XmCreateFrame( workFo, "ValFr", args, 0 );
  valRc = XmCreateRowColumn( valFr, "ValRc", args, 0 );

  valLa = xitCreateLabel( valRc, "ValLa", 
                          msgGetText( MXDI_VALID_DAYS_LABEL ), -1 );

  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  XtSetArg( args[ n ], XmNpacking,     XmPACK_COLUMN ); n++;
  XtSetArg( args[ n ], XmNnumColumns,  2 ); n++;
  valDaysRc  = XmCreateRowColumn( valRc, "ValDaysRc",  args, n );

  day_name_ref = msgGetText( MXDI_WEEKDAYS_MO_FIRST );

  for( index = 0; index < XtNumber( valDaysBu ); index++ ) {

    char_read = strlen( day_name_ref );
    sscanf( day_name_ref, "%s%n", label, &char_read );
    day_name_ref = day_name_ref + char_read;

    sprintf( buffer, "ValDays%dBu", index + 1 );

    if( index < 5 )
      valDaysBu[ index ] = xitCreateToggleButton( valDaysRc, buffer,
                                                  label, True );
    else
      valDaysBu[ index ] = xitCreateToggleButton( valDaysRc, buffer,
                                                  label, False );
  } /* loop */


  /* Duplicate to database. */
  n = 0;
  XtSetArg( args[ n ], XmNorientation, XmHORIZONTAL ); n++;
  dbRc = XmCreateRowColumn( workFo, "DbRc", args, n );

  databaseLa = xitCreateLabel( dbRc, "DatabaseLa", 
                               msgGetText( MXDI_DATABASE_LABEL ), -1 );

  /* Database selections. */
  databaseFs = xtmFlCreateDbField( dbRc, "DatabaseFs",
                                   custom_data_ref -> cal_db_handle,
                                   True,
                                   NULL, NULL );


  /* Put the parts together. */
  xitAttachWidget( messageLa,
                   XmATTACH_FORM, NULL, XmATTACH_FORM, NULL,
                   XmATTACH_NONE, NULL, XmATTACH_NONE, NULL );
  xitAttachWidget( dateSelRc,
                   XmATTACH_WIDGET, messageLa, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( valFr,
                   XmATTACH_WIDGET, dateSelRc, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,      XmATTACH_NONE, NULL );
  xitAttachWidget( dbRc,
                   XmATTACH_WIDGET, valFr, XmATTACH_FORM, NULL,
                   XmATTACH_NONE,   NULL,  XmATTACH_NONE, NULL );


  /* Make sure there is enough space between the children. */
  n = 0;
  XtSetArg( args[ n ], XmNtopOffset,    5 ); n++;
  XtSetArg( args[ n ], XmNleftOffset,   5 ); n++;
  XtSetArg( args[ n ], XmNrightOffset,  5 ); n++;
  XtSetArg( args[ n ], XmNbottomOffset, 5 ); n++;
  XtSetValues( messageLa,  args, n );
  XtSetValues( dateSelRc,  args, n );
  XtSetValues( valFr,      args, n );
  XtSetValues( dbRc,       args, n );


  /* Manage all the children. */
  XtManageChildren( valDaysBu, XtNumber( valDaysBu ) );

  xitManageChildren( dataLocalW, XtNumber( dataLocalW ) );

  /* Set the initial sizes. */
  xitSetSizeFormDialog( duplFd, True );


  return( duplFd );

} /* createDuplWindow */


/*----------------------------------------------------------------------*/

static void 
  destroyCB( Widget          widget,
             ENTRY_INFO_REF  entry_ref,
             XtPointer       call_data )
{

  /* Code. */

  /* Release the user data. */
  if( entry_ref -> use_entry_ref != NULL ) {
    SysFree( entry_ref -> use_entry_ref -> all_text );
    SysFree( entry_ref -> use_entry_ref );
  }

  SysFree( entry_ref );


  return;

} /* destroyCB */


/*----------------------------------------------------------------------*/

static void
  duplicateCB( Widget          widget,
               ENTRY_INFO_REF  entry_ref,
               XtPointer       call_data )
{

  /* Variables. */
  Boolean                 ok;
  int                     day_number;
  UINT32                  id;
  char                    buffer[ 250 ];
  char                    entry_text[ XTM_DB_RECORD_TEXT_LEN + 1 ];
  Widget                  mainW;
  Widget                  tempW;
  Widget                  valW;
  TIM_TIME_REF            current_date;
  XTM_DB_ALL_ENTRY_REF    use_entry_ref;
  XTM_DB_ENTRY_DATABASES  database;
  XTM_DB_ID_REQUEST       id_request;
  XTM_DB_STATUS           db_status;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = entry_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( entry_ref -> duplW, "DuplFdFo" );


  /* Open the database to save the entry. */
  ok = xtmDmOpenDatabase( appl_data_ref, entry_ref -> db_name,
                          XTM_DB_FLAG_MODE_WRITE,
                          &database );
  if( ! ok )
    return;
  

  /* Start to duplicate the entry. */
  use_entry_ref = entry_ref -> use_entry_ref;
  current_date  = entry_ref -> from_date;

  /* Save the entry text (for private entries, it is overwritten). */
  strcpy( entry_text, use_entry_ref -> entry.text );

  valW = XtNameToWidget( mainW, "ValFr.ValRc.ValDaysRc" );

  while( current_date <= entry_ref -> to_date ) {

    /* Do not tuch the current day. */
    if( TimIsSameDate( current_date, 
                       use_entry_ref -> entry.date_stamp ) == TIM_YES ) {
      TimAddDays( &current_date, 1 );

      continue;
    }


    /* Duplicate entry to this day. */
    day_number = TimIndexOfDayInIsoWeek( current_date );

    sprintf( buffer, "ValDays%dBu", day_number );
    tempW = XtNameToWidget( valW, buffer );

    if( ! XmToggleButtonGetState( tempW ) ) {
      TimAddDays( &current_date, 1 );
      continue;
    }


    /* Generate new entry id. */
    id_request.directory  = database.database_dir;
    id_request.operations = 0;
    id_request.lock_file  = False;

    db_status = xtmDbGenerateId( &id_request, &id );
    if( db_status != XTM_DB_OK )
      xitErMessage( NULL, XIT_ER_FATAL, 
                    module_name, "duplicateCB",
                    msgGetText( MXDI_ERRMSG_GENERATE_ID ) );


    /* Update the master record. */
    use_entry_ref -> entry.id       = id;
    use_entry_ref -> stand_entry.id = id;

    use_entry_ref -> entry.date_stamp = current_date;
    use_entry_ref -> stand_entry.from = current_date;

    strcpy( use_entry_ref -> entry.text, entry_text );

    /* Save the entry. */
    db_status = xtmDbInsertEntry( &database,
                                  use_entry_ref, use_entry_ref -> all_text );

    if( db_status != XTM_DB_OK ) {
      xitErMessage( NULL, XIT_ER_FATAL, 
                    module_name, "duplicateCB",
                    msgGetText( MXDI_ERRMSG_INSERT_ENTRY ) );

      xtmDbCloseEntryDb( &database );

      return;
    }

    /* Next day. */
    TimAddDays( &current_date, 1 );

  } /* while */


  xtmDbCloseEntryDb( &database );

  XtDestroyWidget( entry_ref -> duplW );


  /* Update the day list/calendar. */
  xtmUpDoUpdate( (XTM_UP_CALENDAR | XTM_UP_SCHEDULE | XTM_UP_PLANNER), NULL );

  
  return;

} /* duplicateCB */


/*----------------------------------------------------------------------*/

static void 
  helpCB( Widget          widget,
          ENTRY_INFO_REF  entry_ref,
          XtPointer       call_data )
{

  /* Code. */

  xtmHlDisplayHelp( entry_ref -> appl_data_ref -> info_handle,
                    XTM_HL_WINDOW_INDEX,
                    dupl_window_id, "" );

  return;

} /* helpCB */


/*----------------------------------------------------------------------*/

static void
  okCB( Widget          widget,
        ENTRY_INFO_REF  entry_ref,
        XtPointer       call_data )
{

  /* Variables. */
  int                     index;
  Boolean                 ok;
  char                    buffer[ 50 ];
  char                    *char_ref;
  Widget                  mainW;
  Widget                  tempW;
  Widget                  valW;
  TIM_DELTA_TYPE          delta;
  XTM_GL_BASE_DATA_REF    appl_data_ref;
  XTM_GL_CUSTOM_DATA_REF  custom_data_ref;


  /* Code. */

  appl_data_ref   = entry_ref -> appl_data_ref;
  custom_data_ref = appl_data_ref -> custom_data;

  mainW = XtNameToWidget( entry_ref -> duplW, "DuplFdFo" );


  /* Fetch the date range. */
  tempW = XtNameToWidget( mainW, "DateSelRc.RangeTb" );

  ok = xtmFoFetchDate( mainW, tempW, XTM_FO_START_DATE, False,
                       &entry_ref -> from_date );
  if( ! ok )
    return;

  ok = xtmFoFetchDate( mainW, tempW, XTM_FO_END_DATE, False,
                       &entry_ref -> to_date );
  if( ! ok )
    return;


  /* Fetch selected database. */
  tempW = XtNameToWidget( mainW, "DbRc.DatabaseFs" );

  xitFieldSelectGetCurrent( tempW, &char_ref );
  if( char_ref == NULL )
    return;

  if( *char_ref == '\0' ) {
    xitErMessage( entry_ref -> duplW, XIT_ER_ERROR,
                  module_name, "okCB",
                  msgGetText( MXDI_NO_DB_SELECTED ) );

    SysFree( char_ref );
    return;
  }

  strcpy( entry_ref -> db_name, char_ref );
  SysFree( char_ref );


  /* Is the date range correct? */
  if( entry_ref -> from_date > entry_ref -> to_date ) {
    xitErMessage( entry_ref -> duplW, XIT_ER_ERROR,
                  module_name, "okCB",
                  msgGetText( MXDI_INVALID_DATE_RANGE ) );

    return;
  }

  /* Can only duplicate 65 days (safe is better ...). */
  (void) TimDelta( entry_ref -> from_date, entry_ref -> to_date, &delta );

  if( delta.days > 65 ) {
    xitErMessage( entry_ref -> duplW, XIT_ER_ERROR,
                  module_name, "okCB",
                  msgGetText( MXDI_DUPL_TO_MANY_DAYS ) );

    return;
  }


  /* Any valid days? */
  ok   = False;
  valW = XtNameToWidget( mainW, "ValFr.ValRc.ValDaysRc" );

  for( index = 0; index < 7; index++ ) {
    sprintf( buffer, "ValDays%dBu", index + 1 );
    tempW = XtNameToWidget( valW, buffer );

    if( XmToggleButtonGetState( tempW ) )
      ok = True;
  }

  if( ! ok ) {
    xitErMessage( entry_ref -> duplW, XIT_ER_ERROR,
                  module_name, "okCB",
                  msgGetText( MXDI_NO_VALID_DAYS ) );
    return;
  }


  /* Let the user confirm the action. */
  tempW = xitCreateQuestionDialog( 
            entry_ref -> duplW, "QuestionDialog",
            msgGetText( MXDI_DUPLICATE_TITLE ),
            msgGetText( MXDI_DUPLICATE_ENTRY_CONF ),
            duplicateCB, (void *) entry_ref, 
            NULL, NULL );

  return;

} /* okCB */
