# Clock Client Tcl code for tkgoodstuff
# Mark Crimmins (markcrim@umich.edu)

# These settings can be overridden in ~/.tkgoodstuffrc

setifunset Clock_analog 1
setifunset Clock_show both

# DIGITAL STUFF
setifunset ClockOrient vertical
setifunset ClockBigFont "-*-times-bold-r-*-*-24-*-*-*-*-*-*-*"
setifunset ClockSmallFont "-*-times-bold-r-*-*-14-*-*-*-*-*-*-*"
setifunset ClockForeground #7fff00
setifunset ClockBackground #402540

# ALARM STUFF
setifunset Alarm_nobeep 0
setifunset Alarm_timefont "-*-lucidatypewriter-bold-r-*-*-24-*-*-*-*-*-*-*"
setifunset Alarm_Analogflagcolor $ClockForeground
setifunset Alarm_flagfont "-*-helvetica-bold-r-*-*-14-*-*-*-*-*-*-*"
setifunset Alarm_NoAnalogflagcolor \#00ffff


# ANALOG STUFF
setifunset Analog_minsize 30
  # should we expand to a square along our biggest dimension?
setifunset Analog_expand_to_square 1
setifunset Analog_bezel 1
setifunset Analog_background $ClockBackground
setifunset Analog_hubcolor \#f81440
setifunset Analog_facecolor \#add8e6
setifunset Analog_minutecolor black
setifunset Analog_hourcolor black
setifunset Analog_bezelcolor \#f81440
setifunset Analog_tickcolor \#f81440
setifunset Analog_bigtickcolor black
  # define a list of sines and consines
  # I know tcl has the functions now but this is less computation
foreach case {
    {  0      0.00      1.00}
    {  1      0.10      0.99}
    {  2      0.21      0.98}
    {  3      0.31      0.95}
    {  4      0.41      0.91}
    {  5      0.50      0.87}
    {  6      0.59      0.81}
    {  7      0.67      0.74}
    {  8      0.74      0.67}
    {  9      0.81      0.59}
    { 10      0.87      0.50}          
    { 11      0.91      0.41}          
    { 12      0.95      0.31}          
    { 13      0.98      0.21}          
    { 14      0.99      0.10}          
    { 15      1.00      0.00}          
    { 16      0.99     -0.10}          
    { 17      0.98     -0.21}          
    { 18      0.95     -0.31}          
    { 19      0.91     -0.41}          
    { 20      0.87     -0.50}          
    { 21      0.81     -0.59}          
    { 22      0.74     -0.67}          
    { 23      0.67     -0.74}          
    { 24      0.59     -0.81}          
    { 25      0.50     -0.87}          
    { 26      0.41     -0.91}          
    { 27      0.31     -0.95}          
    { 28      0.21     -0.98}          
    { 29      0.10     -0.99}          
    { 30      0.00     -1.00}          
    { 31     -0.10     -0.99}          
    { 32     -0.21     -0.98}          
    { 33     -0.31     -0.95}          
    { 34     -0.41     -0.91}          
    { 35     -0.50     -0.87}          
    { 36     -0.59     -0.81}          
    { 37     -0.67     -0.74}          
    { 38     -0.74     -0.67}          
    { 39     -0.81     -0.59}          
    { 40     -0.87     -0.50}          
    { 41     -0.91     -0.41}          
    { 42     -0.95     -0.31}          
    { 43     -0.98     -0.21}          
    { 44     -0.99     -0.10}          
    { 45     -1.00     -0.00}          
    { 46     -0.99      0.10}          
    { 47     -0.98      0.21}          
    { 48     -0.95      0.31}          
    { 49     -0.91      0.41}          
    { 50     -0.87      0.50}          
    { 51     -0.81      0.59}          
    { 52     -0.74      0.67}          
    { 53     -0.67      0.74}          
    { 54     -0.59      0.81}          
    { 55     -0.50      0.87}          
    { 56     -0.41      0.91}          
    { 57     -0.31      0.95}          
    { 58     -0.21      0.98}          
    { 59     -0.10      0.99}          
} { 
    set sin([lindex $case 0]) [lindex $case 1]
    set cos([lindex $case 0]) [lindex $case 2]
}


# initialize various stuff
set Clock_togglestate 1
set Clock_offset 0
set Alarm_enablealarms 0
set Clock_currentfg $ClockForeground


######
# Analog clock stuff
######

proc Analog_resize {} {
    global Analog_height Analog_width Analog_xorig Analog_yorig
    global Analog_bezel Analog_bezelcolor Analog_facecolor 
    global Clock_window Analog_hubcolor Analog_tickcolor Analog_bigtickcolor
    global sin cos global Analog_hubextent Analog_minuteextent Analog_hourextent
    global Analog_minutethick Analog_hourthick Analog_minutearrowshape
    global Analog_hourarrowshape Analog_expand_to_square
    global Analog_minsize Analog_size

    update
    if { $Analog_height < $Analog_minsize } { set Analog_height $Analog_minsize }
    if { $Analog_width < $Analog_minsize } { set Analog_width $Analog_minsize }
    if { $Analog_height < $Analog_width } {
	if $Analog_expand_to_square {
	    set Analog_size [expr $Analog_width - 2] 
	} else {
	    set Analog_size [expr $Analog_height - 2]
	} 
    } else {
	if $Analog_expand_to_square {
	    set Analog_size [expr $Analog_height - 2]
	} else {
	    set Analog_size [expr $Analog_width - 2] 
	} 
    }	

    if $Analog_expand_to_square {
	$Clock_window.plate.analog config \
	    -width [expr $Analog_size + 2 ] -height [expr  $Analog_size + 2 ]
    } 

    set Analog_scale [expr $Analog_size/25 ]
    set Analog_padding [expr $Analog_scale + 1 ]
    set Analog_minutethick [ expr $Analog_scale + 2]
    set Analog_hourthick [ expr $Analog_scale + 2 + $Analog_scale/3 ]
    set Analog_tickthick [ expr $Analog_scale / 2  + 1]
    set Analog_ticklong  [ expr $Analog_scale * 3/2 ]
    set Analog_bigticklong  [ expr $Analog_scale * 2 ]
    set Analog_bigtickthick [ expr $Analog_scale + 2 ]
    set Analog_bezelthick [ expr $Analog_scale ]
    set Analog_hubextent [ expr $Analog_hourthick/2]
    
    set Analog_xorig [expr $Analog_width/2 ]
    set Analog_yorig [expr $Analog_height/2]
    set Analog_extent [expr ( ($Analog_size/2) - $Analog_padding )]
    set Analog_minuteextent [expr $Analog_extent - ($Analog_bezelthick/2)]
    set Analog_hourextent [expr ( $Analog_extent * .6 )]
    set Analog_tickextent [expr ( $Analog_extent - $Analog_ticklong)]
    set Analog_bigtickextent [expr ( $Analog_extent - $Analog_bigticklong)]
    set Analog_minutearrowshape [ list \
				      [expr $Analog_minutethick + $Analog_scale + 3] \
				      [expr $Analog_minutethick + $Analog_scale  + 3 ] \
				      0 ]
    set Analog_hourarrowshape [ list \
				    [expr $Analog_hourthick + $Analog_scale + 3] \
				    [expr $Analog_hourthick + $Analog_scale + 3] \
				    0 ]
 # For "sharper" (triangular) hands uncomment this and comment out the above
 #    set Analog_minutearrowshape [list \
 #      $Analog_minuteextent \
 #      $Analog_minuteextent \
 #      0 ]
 #    set Analog_hourarrowshape [list \
 #      $Analog_hourextent \
 #      $Analog_hourextent \
 #      0 ]
    
    $Clock_window.plate.analog delete all
    if { ! $Analog_bezel } {
	set Analog_bezelcolor $Analog_facecolor
    }
    $Clock_window.plate.analog create oval \
	[expr $Analog_xorig - ($Analog_extent) ] \
	[expr $Analog_yorig - ($Analog_extent)] \
	[expr $Analog_xorig + ($Analog_extent) -1 ] \
	[expr $Analog_yorig + ($Analog_extent) ] \
	-outline $Analog_bezelcolor -fill $Analog_facecolor \
	-width $Analog_bezelthick
    $Clock_window.plate.analog create oval \
	[ expr $Analog_xorig  - $Analog_hubextent  ] \
	[ expr $Analog_yorig  - $Analog_hubextent  ] \
	[ expr $Analog_xorig + $Analog_hubextent -1 ]\
	[ expr $Analog_yorig + $Analog_hubextent -1 ]\
	-outline $Analog_hubcolor -fill $Analog_hubcolor -tags hub
    for {set time 0 } { $time != 60 } { incr time 5 } {
	if { 0 == ($time % 15) } { 
	    set fill $Analog_bigtickcolor
	    set thick $Analog_bigtickthick
	    set ext $Analog_bigtickextent
	} else {
	    set fill $Analog_tickcolor
	    set thick $Analog_tickthick
	    set ext $Analog_tickextent
	}
	set x1 [expr $Analog_xorig + $Analog_extent * $cos($time) ]
	set y1 [expr $Analog_yorig + $Analog_extent * $sin($time) ]
	set x2 [expr $Analog_xorig + $ext * $cos($time) ]
	set y2 [expr $Analog_yorig + $ext * $sin($time) ]
	$Clock_window.plate.analog create line $x1 $y1 $x2 $y2 -fill $fill -width $thick
    }
}

proc Analog_update { } { 
    global Clock_minute Clock_hour Analog_hubextent Analog_minuteextent
    global Analog_extent sin cos Analog_xorig Analog_yorig Analog_hourextent
    global Analog_minutethick Analog_minutecolor Analog_hourthick Analog_hourcolor
    global Analog_minutearrowshape Analog_hourarrowshape Clock_window

    $Clock_window.plate.analog delete killme

    set Analog_minute $Clock_minute
    set Analog_hour [expr (5 * ($Clock_hour%12)) + ($Clock_minute / 12)]

    set x1 [expr $Analog_xorig + $Analog_hubextent * $sin($Analog_minute) ]
    set y1 [expr $Analog_yorig - $Analog_hubextent * $cos($Analog_minute) ]
    set x2 [expr $Analog_xorig + $Analog_minuteextent * $sin($Analog_minute) ]
    set y2 [expr $Analog_yorig - $Analog_minuteextent * $cos($Analog_minute) ]
    $Clock_window.plate.analog create line $Analog_xorig $Analog_yorig $x2 $y2 -arrow last -tags killme \
       -width $Analog_minutethick -fill $Analog_minutecolor \
       -arrowshape $Analog_minutearrowshape
    set x1 [expr $Analog_xorig + $Analog_hubextent * $sin($Analog_hour) ]
    set y1 [expr $Analog_yorig - $Analog_hubextent * $cos($Analog_hour) ]
    set x2 [expr $Analog_xorig + $Analog_hourextent * $sin($Analog_hour) ]
    set y2 [expr $Analog_yorig - $Analog_hourextent * $cos($Analog_hour) ]
    $Clock_window.plate.analog create line $Analog_xorig $Analog_yorig $x2 $y2 -arrow last -tags killme \
	-width $Analog_hourthick -fill $Analog_hourcolor \
	-arrowshape $Analog_hourarrowshape
    $Clock_window.plate.analog raise hub
    Alarm_set
}

#######
# Alarm stuff
#######
proc Alarm_view {} {
    global TKGcounter TKG_async_map images Alarm_timefont
    global Alarm_hour Alarm_ampm Alarm_minute
    global Clock_ampm Clock_modhour Clock_minute 

    setifunset Alarm_hour $Clock_modhour
    if {$Alarm_hour == 0} {set Alarm_hour 12}
    setifunset Alarm_minute $Clock_minute
    setifunset Alarm_ampm $Clock_ampm

    TKGDialog alarm_view \
	-wmtitle "tkgoodstuff: Alarm Settings" \
	-title "Alarm is Set for:" \
	-nodismiss \
	-buttons { 
	    { enable Enable { 
		set Alarm_enablealarms 1
		DEBUG "Enabling Alarm"
		Alarm_set
		destroy .alarm_view 
	    } 
	    }
	    { disable Disable {
		set Alarm_enablealarms 0
		DEBUG "Disabling Alarm"
		Alarm_set
		destroy .alarm_view 
	    } 
	    }
	}

    frame .alarm_view.top
    pack .alarm_view.top -after .alarm_view.title -side top -fill x -pady 10
    label .alarm_view.top.time -textvariable Alarm_timedisplay \
	-font $Alarm_timefont -relief sunken
    pack .alarm_view.top.time -side left -expand y -padx 40

    frame .alarm_view.hour 
    pack  .alarm_view.hour -side top -after .alarm_view.top \
	-expand y -anchor n -fill both
    label .alarm_view.hour.hourlabel -text "Hour:" 
    pack  .alarm_view.hour.hourlabel -side left -padx 10
    tk_optionMenu .alarm_view.hour.hourbutton Alarm_hour 12 1 2 3 4 5 6 7 8 9 10 11
    pack .alarm_view.hour.hourbutton  -side left
    tk_optionMenu .alarm_view.hour.ampmbutton Alarm_ampm am pm
    pack .alarm_view.hour.ampmbutton  -side left
    trace variable Alarm_hour w Alarm_update_display
    trace variable Alarm_ampm w Alarm_update_display
   
    frame .alarm_view.minute
    pack  .alarm_view.minute -side top -after .alarm_view.hour \
	-expand y -anchor n -fill both -padx 10
    label .alarm_view.minute.minutelabel -text "Minute:" 
    pack  .alarm_view.minute.minutelabel -side left
    scale .alarm_view.minute.minutescale -orient horizontal -length 394 -from 0 -to 59 \
	-tickinterval 10 -variable Alarm_minute -command "Alarm_update_display"
    pack .alarm_view.minute.minutescale -side top -expand y -anchor n\
	-fill x 

    frame .alarm_view.msgfrm
    pack .alarm_view.msgfrm -after .alarm_view.minute -side top -padx 10 -fill x -pady 20
    label .alarm_view.msgfrm.lbl -text "Message:"
    pack .alarm_view.msgfrm.lbl -side left
    entry .alarm_view.msgfrm.entry -textvariable Alarm_message \
	-background lightblue
    pack .alarm_view.msgfrm.entry -side left -fill x -expand y
}

proc Alarm_update_display args {
    global Alarm_ampm Alarm_hour Alarm_minute Alarm_timedisplay Alarm_enablealarms
    global TKG_async_map
    set Alarm_timedisplay "[format %2i $Alarm_hour]:[format %02i $Alarm_minute] $Alarm_ampm"
    set Alarm_minute [Clock_zerotrim $Alarm_minute]
}

proc Alarm_set {} {
    global Alarm_ampm Alarm_hour Alarm_minute Alarm_enablealarms 
    global TKG_async_map Clock_window
    global Clock_analog Alarm_Analogflagcolor Alarm_NoAnalogflagcolor Alarm_flagfont
    global ClockForeground Clock_currentfg
    if !$Alarm_enablealarms {
	catch { $Clock_window.plate.analog delete alarmflag }
	ColorConfig $Clock_window.plate $ClockForeground -
	catch "unset Alarm_minute Alarm_hour Alarm_ampm"
	return
    }
    if [winfo exists $Clock_window.plate.analog] {
	 $Clock_window.plate.analog create text 0 0 -text A -anchor nw\
	     -fill $Alarm_Analogflagcolor -font $Alarm_flagfont -tags alarmflag
    } else {
	ColorConfig $Clock_window.plate $Alarm_NoAnalogflagcolor -
	set Clock_currentfg $Alarm_NoAnalogflagcolor
    }
}

proc Alarm_alarm {} {
    global TKG_nobeep Clock_prettytime Alarm_message Alarm_enablealarms
    global Alarm_nobeep images
    TKGDialog alarm_alarm \
	-wmtitle "tkgoodstuff: Alarm" \
	-image $images/warning \
        -title "Alarm        It's $Clock_prettytime" \
        -message $Alarm_message
    ColorConfig .alarm_alarm.title black red
    if { !($TKG_nobeep || $Alarm_nobeep) } { bell }
    catch { $Clock_window.plate.analog delete alarmflag }
    set Alarm_enablealarms 0
    Alarm_set
}


########
# General Clock stuff
########

proc Clock_zerotrim { s } {
    scan $s "%d" o
    return $o
}

proc Clock_update {} {
    global Clock_minute Clock_hour Clock_modhour Clock_ampm Clock_analog
    global Clock_prettydate Clock_prettytime
    global TKG_async_map Clock_offset TKGcounter Clock_window
    global Alarm_ampm Alarm_hour Alarm_minute Alarm_enablealarms

    set timedate [ exec date "+%H %M %M %S %h %d"] 
    scan $timedate {%s %s %s %s %s %s} Clock_hour Clock_minute prettyminute seconds month day
    foreach v { Clock_hour Clock_minute seconds day } {
	set $v [Clock_zerotrim [set $v]]
    }
    set Clock_modhour [expr $Clock_hour % 12]
    set Clock_prettyhour [expr (($Clock_modhour == 0) ? 12: $Clock_modhour)]
    set Clock_ampm [expr ( ($Clock_hour > 11) ? \"pm\" : \"am\" ) ]
    set Clock_prettytime "$Clock_prettyhour:$prettyminute"
    set Clock_prettydate "$month $day"

    set Clock_offset [expr ( (($TKGcounter % 60) - $seconds) + 60 ) % 60]
    set TKG_async_map(Clock_periodic) [list 60 $Clock_offset Clock_update]

    if { [winfo exists $Clock_window.plate.analog]} {
	Analog_update
    }     

    if { $Alarm_enablealarms && \
	     "$Clock_ampm" == "$Alarm_ampm" && \
	     ($Clock_modhour == ($Alarm_hour % 12)) && \
	     "$Clock_minute" == "$Alarm_minute" } {
	Alarm_alarm
    }
}

proc Clock_timedatetoggle {} {
    global Clock_show Clock_togglestate Clock_window

    if { $Clock_show != "dateonly" } {
        if $Clock_togglestate {
	   $Clock_window.plate.time.msg config -textvariable Clock_prettydate
	} else {
           $Clock_window.plate.time.msg config -textvariable Clock_prettytime
	}
    }
    if { $Clock_show != "timeonly"} {
        if $Clock_togglestate {
	   $Clock_window.plate.date.msg config -textvariable Clock_prettytime
	} else {
           $Clock_window.plate.date.msg config -textvariable Clock_prettydate
	}
    }
    set Clock_togglestate [expr 1 - $Clock_togglestate]
    update
}

proc Clock_configure_window args {
    global Clock_show Clock_window ClockBigFont ClockSmallFont
    global ClockForeground  ClockBackground
    global Clock_currentfg Alarm_NoAnalogflagcolor Alarm_enablealarms
    global ClockOrient Clock_analog
    global Clock_window Clients
    global Analog_width Analog_height Analog_background
    global Analog_minsize

    catch { destroy $Clock_window.plate.date }
    catch { destroy $Clock_window.plate.time }
    catch { destroy $Clock_window.plate.analog }
    
    if {$ClockOrient == "horizontal"} {
	set side right
    } else {
	set side top
    }   
    set dside $side
    
    if { $Clock_analog } {
	set dside top
	setifunset Analog_width $Analog_minsize
	setifunset Analog_height $Analog_minsize
	canvas $Clock_window.plate.analog \
	    -width $Analog_width -height $Analog_height \
	    -background $Analog_background \
	    -highlightthickness 0
	pack $Clock_window.plate.analog -fill both -expand y -padx 0 -pady 0 -side $side
	Analog_resize
	bind $Clock_window.plate.analog <Configure> {
	    set Analog_width %w
	    set Analog_height %h
	    Analog_resize
	    Clock_update
	}
	set Clock_currentfg $ClockForeground
    } else {
	if $Alarm_enablealarms {
	    set Clock_currentfg $Alarm_NoAnalogflagcolor 
	}
    }
    if { $Clock_show != "neither"} {

	if {$Clock_show != "timeonly" } {
	    frame $Clock_window.plate.date
	    label $Clock_window.plate.date.msg \
		-textvariable Clock_prettydate -font $ClockSmallFont
	    pack $Clock_window.plate.date.msg -fill both -expand y
	    pack $Clock_window.plate.date -fill both -expand y -side $dside
	}
	if {$Clock_show == "dateonly"} {
	    if { $Clock_analog == 0 } {
		$Clock_window.plate.date.msg configure -font $ClockBigFont
	    }
	} else {
	    frame $Clock_window.plate.time
	    label $Clock_window.plate.time.msg \
		-textvariable Clock_prettytime -font $ClockBigFont
	    pack $Clock_window.plate.time.msg -fill both -expand y
	    pack $Clock_window.plate.time -fill both -expand y -side $dside
	}
    }
    ColorConfig $Clock_window.plate $Clock_currentfg $ClockBackground
    Clock_bind
    Clock_update
}    

proc Clock_bind {} {
    global Clock_window
    RecursiveBind $Clock_window <1> { TKGPopupPost %X %Y}
    RecursiveBind $Clock_window <3> Clock_timedatetoggle
}

proc create-Clock-window {} {
    global Clock_analog
    global Clock_prettydate Clock_prettytime 
    global Clock_window
    global TKGmenutitleforeground Clock_show ClockOrient

    # set up async processing
    set Clock_offset [expr 60 - [Clock_zerotrim [exec date +%S]]]
    set TKG_async_map(Clock_periodic) [list 60 $Clock_offset Clock_update]

    create-label-box Clock - {}

    Clock_update
    Clock_configure_window

    trace variable Clock_show w Clock_configure_window
    trace variable ClockOrient w Clock_configure_window
    trace variable Clock_analog w Clock_configure_window

    TKGPopupAddClient Clock

    .tkgpopup.clock add checkbutton \
	-label "Show analog clock" \
	-variable Clock_analog

    .tkgpopup.clock add separator

    .tkgpopup.clock add radiobutton \
	-label "Vertical orientation" \
	-variable ClockOrient -value vertical
    .tkgpopup.clock add radiobutton \
	-label "Horizontal Orientation" \
	-variable ClockOrient -value horizontal

    .tkgpopup.clock add separator

    .tkgpopup.clock add radiobutton \
	-label "Show digital time" \
	-variable Clock_show -value timeonly
    .tkgpopup.clock add radiobutton \
	-label "Show date" \
	-variable Clock_show -value dateonly
    .tkgpopup.clock add radiobutton \
	-label "Show digital time and date" \
	-variable Clock_show -value both
    .tkgpopup.clock add radiobutton \
	-label "Show neither" \
	-variable Clock_show -value neither

    TKGPopupAdd command -label "ALARM" -command Alarm_view
    TKGPopupAdd separator
}

DEBUG "Loaded Clock.tcl"
