/* Copyright (C) 1994 Groupe BULL. See file COPYRIGHT for details */
/*
 *
 * $Id: kts_main.c,v 1.4 1994/12/14 10:39:28 beust Exp $
 */

#include <stdio.h>
#include <assert.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/times.h>
#include <sys/socket.h>
#ifdef DPX20
#include <sys/select.h>
#endif
#include <fcntl.h>

#include "kts_main.h"

/*
** Debugging the the ICE library wanted?
*/
int ICEdebug = 0;

GV GlobalVars;

/*
 **---------------------------------------------------------------------------
 ** Parsing the command line
 **
 */

static void
kts_usage(char **argv)
{
   fprintf(stderr, "Usage: %s %s\n", argv[0], KTS_VERSION);
   fprintf(stderr, "\t[-help]\t\t\tThis help\n");
   fprintf(stderr, "\t[-debug <n>]\t\tSet debug level\n");
   fprintf(stderr, "\t[-icedebug]\t\tTurn on ICE debugging\n");
   exit(0);
}

static void
kts_readEnvironment(GV gv, int *argc, char ***argv)
{
   char *p, *progName = (*argv)[0];

   while ((*argc) > 1) {
      if (strcmp((*argv)[1], "-debug") == 0) {
	 gv -> debugLevel = atoi((*argv)[2]);
	 *argc -= 2;
	 *argv += 2;
      }
      else if (strcmp((*argv)[1], "-icedebug") == 0) {
	 ICEdebug = 1;
	 *argc -= 1;
	 *argv += 1;
      }
      else if (strcmp((*argv)[1], "-h") == 0 ||
	       strcmp((*argv)[1], "-help") == 0) {
	 kts_usage(*argv); /* will exit */
      }
      else {
	 fprintf(stderr, "ignored option '%s'\n", (*argv)[1]);
	 kts_usage(*argv);   /* will exit */
      }
   }
}

/*
 **---------------------------------------------------------------------------
 ** Miscellaneous ICE callbacks
 **
 */

/* callback of type IceIOErrorProc */
void
kts_errorHandler(IceConn conn)
{
   GV gv = GlobalVars;

   kts_removeRealClient(gv, IceConnectionNumber(conn));
   DL(DEBUG_MISC, printf("kts_errorHandler: disconnection detected\n"));
   longjmp(gv -> errorEnv, 0);
}



static void
kts_freeConnectionRec(Kts_Connection conn)
{
}

static void
kts_freeDeclaredKRLRec(Kts_DeclaredKRL krl)
{
}

static void
kts_freeInterestedInKRRec(Kts_InterestedInKR krl)
{
}

static Bool
kts_hostBasedAuth(char *host)
{
   return True;
}

static void
kts_watchProc(IceConn iceConn, IcePointer userData, Bool opening,
	      IcePointer *watchData)
{
   DP(printf("watch procedure called\n"));
}

/*
** These two functions are used to enter (resp. remove) a client
** in our Connection database
*/

void
kts_addVirtualClient(GV gv, char *executable, char *krs, Bool isObserved)
{
   Kts_Connection c;

   NEW(c, Kts_ConnectionRec);
   c -> isVirtual = True;
   c -> virtual.executable = kt_strdup(executable);
   c -> virtual.krs = kt_strdup(krs);
   c -> virtual.isObserved = isObserved;
   c -> fd = 0;
   DB_AddEntry(gv -> connections, c);
}

void
kts_addRealClient(GV gv, IceConn iceConn)
{
   Kts_Connection c;

   NEW(c, Kts_ConnectionRec);
   c -> isVirtual = False;
   c -> iceConn = iceConn;
   c -> fd = IceConnectionNumber(iceConn);
   c -> iceSenderCoord = (char *) 0xa4a4a4a4;
   c -> msgSendOnExit = NULL;
   c -> declaredKRL =
      DB_NewDataBase(sizeof(Kts_DeclaredKRLRec), kts_freeDeclaredKRLRec);
   c -> interestedInKR =
      DB_NewDataBase(sizeof(Kts_InterestedInKRRec),
		     kts_freeInterestedInKRRec);
   
   /*
   ** and add it to our database
   */
   DB_AddEntry(gv -> connections, c);
}

static Bool
kts_findConnectionString(Kts_Connection entry, int fd)
{
   if (False == entry -> isVirtual) {
      if (fd == IceConnectionNumber(entry -> iceConn)) return True;
      else return False;
   }
   else
      return False;
}

void
kts_removeRealClient(GV gv, int fd)
{
   DataBase connections = gv -> connections;
   DataBase handlers = gv -> handlers;
   Kts_Connection conn;
   Kt_handler handler;
   IceConn iceConn;
   int running = 1;

   conn = DB_LocateEntry(connections,
			 (void *) ((unsigned long) kts_findConnectionString),
			 (void *) ((unsigned long) fd));
   if (! conn) return;    /* connection removed already */

   iceConn = conn -> iceConn;

   /*
   ** Before giving control back to the main loop, several things to do :
   */

   /*
   ** - see if this client registered a message to be sent on exit
   ** and send it if it did
   **
   ** Note : this part could be optimized : when the message
   ** MESSAGE_SEND_ON_EXIT is received, I change it from a Kt_messageBuffer
   ** to a Kt_Message and here I have to do it the other way around
   ** in order to be able to feed it to kts_dispatchMessage. However,
   ** the opCode of the message changes from MESSAGE_SEND_ON_EXIT to
   ** CLIENT_EXITED.
   */
   if (0 != conn -> msgSendOnExit) {
      char *fillBuffer, *extra;
      Ktm_ClientExited ce;
      DP(printf("sending a message on exit\n"));
      kts_dispatchMessage(gv, conn -> msgSendOnExit,
			  conn -> msgSendOnExitKRLId,
			  IceConnectionNumber(iceConn));
      /*@@ should free some resources here */
   }

   /*
   ** - remove the now dead file descriptor from the list of open
   ** connections
   */

   DB_RemoveEntry(connections, conn);

   /*
   ** - remove the possible handling declarations from this client
   ** - broadcast a DELETE_HANDLER packet to all our connected clients
   ** so they remove it from their database too
   */

   DB_Rewind(handlers);
   while (running) {
      if (DB_EndOfDataBase(handlers)) running = 0;
      else {
	 handler = DB_NextEntry(handlers);
	 if (handler -> fd == IceConnectionNumber(iceConn)) {
	    kts_broadcastDeleteHandler(gv, handler -> iceCoord);
	    DB_RemoveEntry(handlers, handler);
	    running = 0;
	 }
      }
   }

      
}

/*
** This callback will be called before the ProtocolReply is sent.
** It is only used to set the userdata field that will be passed
** on to function kts_processMessage.
*/
Status
kts_protocolSetupProc(IceConn iceConn, int majVersion, int minVersion,
		      char *vendor, char *release, IcePointer *userData,
		      char **failureReason)
{
   *userData = (IcePointer) GlobalVars;
   return 1;
}

static void
kts_fdsPrint(fd_set fds)
{
   int i;

   printf("Descriptors: "); fflush(stdout);
   for (i=0; i<kt_getdtablesize(); i++)
      if (FD_ISSET(i, & fds)) {
	 printf("%d ", i);
	 fflush(stdout);
      }
   printf("\n");
}

IceListenObj
kts_nonLocalListenObj(int count, IceListenObj *obj)
{
    int i;
    char *s;
    for (i=0; i < count; i++) {
	s = IceGetListenConnectionString(obj[i]);
	if (NULL == strstr(s, "local")) {
	    free(s);
	    return obj[i];
	}
	else {
	    free(s);
	}
    }
    ASSERT(0);
}

int
kts_initServer(GV gv)
{
   DataBase connections;
   int i;
   int result = 0;
   char error[128];
   IceConn newConn;
   Status status;
   int count;	 /* # of listen objects */
   IceListenObj *obj, listenObj;
   fd_set fds;
   int ready, prot;
   IcePaVersionRec callbackList[] = {
      { 1, 0, (IcePaProcessMsgProc) kts_processMessage }
   };

   /*
   ** Initialize some global variables
   */
   gv -> connections =
     DB_NewDataBase(sizeof(Kts_ConnectionRec), kts_freeConnectionRec);
   ASSERT(gv -> connections);
   FD_ZERO(& gv -> defaultFdSet);

   kts_queueInit(gv);

   /*
   ** Read the protocol files
   */
   kts_readProtocolFiles(gv);

   /*
   ** Initialize ICE
   */
   status = IceListenForConnections(& count,
				   & obj,
				   128,
				   error);

   if (count != 2) {
      printf("Error: '%s'\n", error);
      exit(0);
   }

   /*
   ** Store my coordinates in well-known places so I can be found by
   ** kt_LocateServer()
   */
   kts_advertiseMyPresence(gv, count, obj);


   prot = IceRegisterForProtocolReply(PNAME, PVENDOR, PRELEASE,
			       1, callbackList,	   /* callbacks */
			       0,            /* # of auth methods supported */
			       NULL,  /* auth names */
			       NULL,		   /* auth callbacks */
			       kts_hostBasedAuth,     /* host-based auth */
			       kts_protocolSetupProc,   /* cb before Protocol Reply is sent */
			       NULL,   /* cb after Protocol Reply is sent */
			       NULL    /* IO error handler */
      );
   ASSERT(prot);

   IceSetIOErrorHandler(kts_errorHandler);
/*
   IceAddConnectionWatch(kts_watchProc, NULL);
*/

   /*
   ** Store the listen object we will be using, and
   ** assign a host-based authentication procedure to it
   */
   listenObj = kts_nonLocalListenObj(count, obj);
   IceSetHostBasedAuthProc(listenObj, kts_hostBasedAuth);
   gv -> mainProtocol = prot;
   gv -> mainFd = IceGetListenConnectionNumber(listenObj);
   gv -> mainListenObj = listenObj;

   newConn = NULL;
   /*
   ** We are always listening on the server's socket.
   ** If a new connection is opened, we don't add the fd immediately
   ** but stay in an "alternate" situation for a while. Once the
   ** connection succeeds, the fd is added to the list of all the
   ** connections
   */
      FD_ZERO(& fds);
   while (1) {
      setjmp(gv -> errorEnv);
#ifndef NO_RPC
      fds = svc_fdset;
#endif /* NO_RPC */
      status = IceConnectPending;
/*
      {
	 int i;
	 for (i=0;i<kt_getdtablesize(); i++)
	    if FD_ISSET(i, & svc_fdset) break;
	 DP(printf("default fds before %x, fd = %d\n", fds, i));
      }
*/

      /*
      ** We will listen to the fileset fds
      ** First, we put all the open connections in it that are not
      ** virtual.
      */
      connections = gv -> connections;
      DB_Rewind(connections);
      while (! DB_EndOfDataBase(connections)) {
	 Kts_Connection conn = (Kts_Connection) DB_NextEntry(connections);
	 if (False == conn -> isVirtual) FD_SET(conn -> fd, & fds);
      }

      /*
      ** Then the possible connection being opened.
      */
      if (newConn != NULL) {
	 FD_SET(IceConnectionNumber(newConn), & fds);
      }

      /*
      ** Our RPC port where we can be asked to reply the ICE port
      */
#if 0
#ifndef NO_RPC
      FD_SET(gv -> rpcFd, & fds);
#endif /* NO_RPC */
#endif

      /*
      ** And the server itself.
      */
      FD_SET(gv -> mainFd, & fds);

      ready = select(kt_getdtablesize(), & fds, NULL, NULL, NULL);

      /*
      ** Connection to the server. This is a new client. We
      ** accept the connection and will process the messages on
      ** this file descriptor until the connection is not in
      ** pending state any more.
      */
      if (ready > 0 && FD_ISSET(gv -> mainFd, & fds)) { /* conn to the server */
	 IceConnectStatus status;
	 ready--;
	 newConn = IceAcceptConnection(gv -> mainListenObj, & status);
	 if (NULL != newConn) {
	    DP(printf("kts_initServer: accepted new connection %x, fd=%d\n",
		      newConn, IceConnectionNumber(newConn)));
	    IceProcessMessages(newConn, NULL, NULL);
	 }
      }

      /*
      ** See if the incoming data belongs to an already open
      ** connection. Process it if it does
      */
      DB_Rewind(connections);
      while (! DB_EndOfDataBase(connections)) {
	 Kts_Connection conn = (Kts_Connection) DB_NextEntry(connections);
	 /*
         ** Only do the next test if the client is not virtual
         */
	 if (True != conn -> isVirtual) {  
	    if (ready > 0 && FD_ISSET(conn -> fd, & fds)) {
	       ready--;
	       IceProcessMessages(conn -> iceConn, NULL, NULL);
	    }
	 }
      }

      /*
      ** Some data on the dangling connection. It could be some
      ** prealable handshake (e.g. protocol setup). We process the
      ** the message and check out if it makes the connection
      ** switch fron IceConnectPending to IceConnectAccepted,
      ** in which case it becomes a valid connection and is added to
      ** the set we are listening to.
      */
      if (ready > 0 && NULL != newConn && FD_ISSET(IceConnectionNumber(newConn), & fds)) {
	 ready--;
	 IceProcessMessages(newConn, NULL, NULL);
	 status = IceConnectionStatus(newConn);
	 if (status == IceConnectAccepted) {
	    kts_addRealClient(gv, newConn);
	    newConn = NULL;
	 }
      }
      /*
      ** No match, then it must be a request for our RPC service.
      */
#ifndef NO_RPC
      if (ready > 0) { /*  && FD_ISSET(gv -> rpcFd, & fds)) { */
	 ready--;
	 svc_getreqset(& fds);
      }
#endif
   }

   return result;
}

static void
kts_freeHandler(Kt_handler h)
{
   SAFE_FREE(h -> krs);
   SAFE_FREE(h -> iceCoord);
}

static void
kts_signalUsr1()
{
   GV gv = GlobalVars;
   printf("Rereading protocol files\n");
   kts_readProtocolFiles(gv);
}

static void
kts_signalUsr2()
{
   GV gv = GlobalVars;
   printf("Known protocols :\n");
   kts_displayProtocols(gv);
}

static void
kts_signalPipe()
{
   signal(SIGPIPE, kts_signalPipe);
}

int
main(int argc, char **argv)
{
   int result = 0;
   GV gv;

   /*
   ** Catch some signals
   */
   signal(SIGUSR1, kts_signalUsr1);
   signal(SIGUSR2, kts_signalUsr2);
   signal(SIGPIPE, kts_signalPipe);

   /*
   ** Initialize global vars
   */
   NEW(GlobalVars, GlobalVarsRec);
   gv = GlobalVars;
   memset(gv, 0xa3, sizeof(*gv));
   gv -> serial = 6666;
   gv -> handlers = DB_NewDataBase(sizeof(Kt_handlerRec), kts_freeHandler);

   kts_readEnvironment(gv, & argc, & argv);

   if (argc == 1) {
      kts_initServer(gv);
   }
   else {
      fprintf(stderr, "Usage\n");
      result = 1;
   }

   return result;
}

