/* Copyright (C) 1994 Groupe BULL. See file COPYRIGHT for details */
/*
 *
 * $Id: kts_dispatch.c,v 1.3 1994/12/15 13:37:57 beust Exp beust $
 */

#include "regexp.h"
#include "kts_main.h"

/*
**---------------------------------------------------------------------------
** kts_forwardMessage & kts_forwardMessageBuffer
** Forward a message to a client.
** All it has to do is forward the message itself with 2 changes :
** - opCode becomes Ktp_MESSAGE_FORWARDED
** - a krSpecId field is inserted (the one that caused the delivery of
**   this message to the client)
*/

void
kts_forwardMessageBuffer(GV gv, Kts_Connection clientTo, Kt_messageBuffer mb,
		      CARD32 krSpecId, Kts_Connection clientFrom)
{
   IceConn iceConn = clientTo -> iceConn;
   char *strSenderCoord = clientFrom -> iceSenderCoord;
   Ktm_MessageForwarded pmsg;
   char *fillBuffer, *extra;
   size_t nBytesToWrite = 0;
   int l = ((mb -> length)<<3) - (sizeof(CARD32) * 2), coordLength;

   coordLength = strlen(strSenderCoord);
   IceGetHeaderExtra(iceConn, gv -> mainProtocol,
		     Ktp_MESSAGE_FORWARDED,
		     ICE_HEADER_SIZE, 0,
		     Ktm_MessageForwardedRec, pmsg, extra);

   fillBuffer = K_VARIABLE_FIELD(extra);
   WRITELONG(mb -> krlId, fillBuffer);
   WRITELONG(mb -> opLength, fillBuffer);
   WRITELONG(krSpecId, fillBuffer);
   WRITELONG(coordLength, fillBuffer);
   memcpy(fillBuffer, strSenderCoord, coordLength); fillBuffer += coordLength;
   memcpy(fillBuffer, & mb -> bytes[0], l); fillBuffer += l;
   nBytesToWrite = fillBuffer - extra;
   pmsg -> length = WORD64COUNT(nBytesToWrite);

   IceWriteData(iceConn, (pmsg -> length) << 3, extra);
   {
       fd_set writeFds;
       int result;
       struct timeval timer;
       FD_ZERO(& writeFds);
       FD_SET(IceConnectionNumber(iceConn), & writeFds);
       memset((char *) & timer, 0, sizeof(struct timeval));
       result = select(kt_getdtablesize(), NULL, & writeFds, NULL, & timer);
       if (0 == result) {
	   fprintf(stderr, "write:() descriptor not ready for writing\n");
	   return;
       }
       IceFlush(iceConn);
   }
   IceFlush(iceConn);

}

void
kts_forwardMessage(GV gv, IceConn iceConn, Kt_Message msg, CARD32 krSpecId)
{
   Ktm_MessageForwarded pmsg;
   char *extra;
   char *fillBuffer;

   IceGetHeaderExtra(iceConn, gv -> mainProtocol,
		     Ktp_MESSAGE_FORWARDED,
		     ICE_HEADER_SIZE, 0,
		     Ktm_MessageForwardedRec, pmsg, fillBuffer);
   fillBuffer = K_VARIABLE_FIELD(fillBuffer);
   fillBuffer = kt_messageStartPrepare(iceConn, msg, fillBuffer);
   IceWriteData(iceConn, sizeof(CARD32), (void *) & krSpecId);
   msg -> size += sizeof(CARD32);
   pmsg -> length = kt_messageEndPrepare(iceConn, msg);

   DP(printf("forwardMessage sending %d bytes (gves %d)\n", iceConn->outbufptr-iceConn->outbuf, pmsg->length));
   IceFlush(iceConn);
  
}


/*
**---------------------------------------------------------------------------
** kts_broadcastNewHandler & kts_sendNewHandler
** We were just informed of a new handler. Send a NEW_HANDLER message
** to all our connected clients, except the initial sender
** (who already knows he is a handler...)
** Parameter : kt client at location 'iceCoord' is a handler for
** Koalatalk Resource Specification 'krSpec'. When reaching it, we
** must use id number 'krSpecId'
** kts_sendNewHandler() achieves the same task but to *one* client
*/

void
kts_sendNewHandler(GV gv, char *krSpec, CARD32 krSpecId,
		   char *iceCoord, Kts_Connection client)
{
   size_t krsLen, cooLen;
   Ktm_NewHandler pmsg;
   char *fillBuffer, *extra;

   krsLen = strlen(krSpec);
   cooLen = strlen(iceCoord);

   IceGetHeaderExtra(client -> iceConn,gv -> mainProtocol,Ktp_NEW_HANDLER,
		     ICE_HEADER_SIZE, 0,
		     Ktm_NewHandlerRec, pmsg, extra);
   fillBuffer = extra;
   /* Fields krlId and operation not relevant here */
   WRITELONG(KT_NOT_RELEVANT, fillBuffer);
   WRITELONG(KT_NOT_RELEVANT, fillBuffer);
   WRITELONG(krSpecId, fillBuffer);
   WRITELONG(krsLen, fillBuffer);
   WRITELONG(cooLen, fillBuffer);
   memcpy(fillBuffer, krSpec, krsLen); fillBuffer += krsLen;
   memcpy(fillBuffer, iceCoord, cooLen); fillBuffer += cooLen;
   pmsg -> length = WORD64COUNT(sizeof(CARD32) * 6 + krsLen + cooLen);
   IceWriteData(client -> iceConn, (pmsg -> length) << 3, extra);
   if (IceValidIO(client -> iceConn))
      IceFlush(client -> iceConn);
   else {
      fprintf(stderr, "broken pipe?!?");
      abort();
   }
}

void
kts_broadcastNewHandler(GV gv, char *krSpec, int senderFd,
			CARD32 krSpecId, char *iceCoord)
{
   Kts_Connection client;
   DataBase db;

   /*
   ** Find the handle to the sender
   */
   db = gv -> connections;
   DB_Rewind(db);
   while (! DB_EndOfDataBase(db)) {
      size_t krsLen = strlen(krSpec);
      size_t cooLen = strlen(iceCoord);

      client = (Kts_Connection) DB_NextEntry(db);
      /*
      ** Found a client. Before sending NEW_HANDLER, make sure it is
      ** - not the sender
      ** - not virtual
      */
      if (False == client -> isVirtual) {
	 if (client -> fd != senderFd)
	    kts_sendNewHandler(gv, krSpec, krSpecId, iceCoord, client);
      }
   }
}

/*
**---------------------------------------------------------------------------
** kts_broadcastDeleteHandler
** A handler just disconnected, tell everyone to remove it from their
** bases.
*/
void
kts_broadcastDeleteHandler(GV gv, char *iceCoord)
{
   Kts_Connection client;
   char *fillBuffer, *extra;
   Ktm_DeleteHandler pmsg;
   IceConn iceConn;
   DataBase db;

   db = gv -> connections;
   DB_Rewind(db);
   while (! DB_EndOfDataBase(db)) {
      size_t len = strlen(iceCoord);

      client = DB_NextEntry(db);
      if (False == client -> isVirtual) {
	 iceConn = client -> iceConn;
	 IceGetHeaderExtra(iceConn,gv -> mainProtocol,Ktp_DELETE_HANDLER,
			   ICE_HEADER_SIZE, 0,
			   Ktm_DeleteHandlerRec, pmsg, extra);
	 fillBuffer = extra;

	 /* field krlId not relevant for this packet */
	 WRITELONG(KT_NOT_RELEVANT, fillBuffer);
	 /* opLength neither */
	 WRITELONG(KT_NOT_RELEVANT, fillBuffer);
	 /* coordinate length, then coordinates */
	 WRITELONG(len, fillBuffer);
	 memcpy(fillBuffer, iceCoord, len); fillBuffer += len;
	 pmsg -> length = WORD64COUNT(fillBuffer - extra);
	 IceWriteData(iceConn, (pmsg -> length) << 3, extra);
	 IceFlush(iceConn);
      }
   }
}

/*
**---------------------------------------------------------------------------
** kts_sendHandlerInfo
** A new client just connected to us, tell it about all the handlers
** we know.
** 'iconn' is the ICE connection to use
*/

static Bool
findClientByFd(Kts_Connection entry, int fd)
{
   if (False == entry -> isVirtual) {
      if (fd == entry -> fd) return True;
      else return False;
   }
   else
      return False;
}

void
kts_sendHandlerInfo(GV gv, IceConn iconn)
{
   DataBase db = gv -> handlers;
   Kt_handler handler;
   Kts_Connection client;
   int fd = IceConnectionNumber(iconn);

   /*
   ** Find the entry corresponding to this client given its iconn
   */
   client = DB_LocateEntry(gv -> connections,
			   (void *) ((unsigned long) findClientByFd),
			   (void *) ((unsigned long) fd));
   ASSERT(client);

   DB_Rewind(db);
   while (! DB_EndOfDataBase(db)) {
      handler = (Kt_handler) DB_NextEntry(db);
      kts_sendNewHandler(gv, handler -> krs, handler -> krSpecId,
			 handler -> iceCoord, client);
   }
}

/*
**---------------------------------------------------------------------------
** kts_dispatchMessage
** This function is incredibly inefficient, should be rewritten
** It transforms a message from MESSAGE_SENT to MESSAGE_FORWARDED and
** send this information to all interested clients
*/

static Bool
findKRL(Kts_DeclaredKRL entry, CARD32 krl)
{
   if (entry -> krlId == krl) return True;
   else return False;
}

void
kts_dispatchMessage(GV gv, Kt_messageBuffer mb, CARD32 dest, int fd)
{
   Kts_Connection clientTo, clientFrom;
   Kts_DeclaredKRL krl;
   Kts_InterestedInKR ir;
   Bool recipientFound;
   DataBase db1, db2;

   /*
   ** Find the handle for the sender
   */
   clientFrom =
      DB_LocateEntry(gv -> connections,
		     (void *) ((unsigned long) findClientByFd),
		     (void*) ((unsigned long) fd));
   ASSERT(clientFrom);

   /*
   ** Find the krlName
   */
   krl =
      DB_LocateEntry(clientFrom -> declaredKRL,
		     (void *) ((unsigned long) findKRL),
		     (void *) ((unsigned long) dest));
   ASSERT(krl);

   /*
   ** Now send to the clients that are interested in a krl
   ** matching krl -> krlName. If no recipient is found, search
   ** among the virtual clients (those declared in the protocol
   ** file) to see if one could be launched.
   */

   db1 = gv -> connections;
   DB_Rewind(db1);
   recipientFound = False;
   /*
   ** For each of our connections...
   */
   while (! DB_EndOfDataBase(db1)) {

      clientTo = (Kts_Connection) DB_NextEntry(db1);
      if (False == clientTo -> isVirtual) {
	 db2 = clientTo -> interestedInKR;
	 DB_Rewind(db2);
	 /*
	 ** ... for each of the registered patterns for this connection...
	 */
	 while (! DB_EndOfDataBase(db2)) { 
	    ir = (Kts_InterestedInKR) DB_NextEntry(db2);
	    /*
	    ** ... make the comparison
	    */
	    if (True == kt_regexpMatch(ir -> krSpec, krl -> krlName)) {
	       DP(printf("Dispatching message '%s' to fd %d who is observing '%s' (id=%x)\n",
			 krl -> krlName, clientTo -> fd,
			 ir -> krSpec, ir->krSpecId));
	       kts_forwardMessageBuffer(gv, clientTo, mb, ir -> krSpecId,
					clientFrom);
	       recipientFound = True;
	       /*@@ should dispose the message here */
	    }
	    else {
	       /* doesn't match, carry on with the next krSpec */
	    }
	 }   /* walking db2 */
	 /*
	 ** Now flush the connection before proceeding with
	 ** next client
	 */
	 IceFlush(clientTo -> iceConn);
      } /* ! isVirtual */
   }  /* while */

   /*
   ** No recipient found, check out the known protocols.
   ** If we find one, the message must be stored until the appropriate
   ** client pops up.
   */
   if (False == recipientFound) {
      Kt_Message msg = kt_messageStartDecode(mb);
      kt_messageEndDecode(msg);
      /*
      ** The last True is to specify that this krl must be observed
      ** to match.
      */
      if (True == kts_checkVirtualClients(gv, krl -> krlName, True))
	 kts_queueStoreMessage(gv, krl -> krlName, msg);
   }
}

