/* Copyright (C) 1994 Groupe BULL. See file COPYRIGHT for details */
/*
 *
 * $Id: kt_typesP.h,v 1.1 1994/10/07 16:34:43 beust Exp beust $
 *
 * Abstract types I don't want anybody to see but me (and that's
 * already one person too many) 
 * This is a _private_ file, not to be included for normal use
 */

#ifndef __KT_TYPESP_H__
#define __KT_TYPESP_H__

#include <stddef.h>     /* size_t */
#include <X11/Xmd.h>
#include "kt_lib.h"
#include "kt_protocolP.h"
#include "database.h"

#ifndef NO_RPC
#include <rpc/rpc.h>
#include <rpc/pmap_prot.h> /* struct pmaplist */
#include <rpc/pmap_clnt.h> /* protos for pmap_*() */
#include <sys/socket.h>   /* AF_INET */
#ifndef mips
#include <netinet/in.h>   /* sockaddr_in */
#endif
#include <netdb.h>        /* getservbyname() */
#include <sys/types.h>    /* FD_SET */
#endif /* NO_RPC */


/*
**
** Debugging macros
** D = Standard debug macro
** DL = Debug Level
** DP = Debug macro printf (will print file and line)
** The levels are bitwise ored, so for example if you want to have
** both PACKETS and MISC debugging messages on sterr, call ktserv with
** -debug 3
**
*/

#define READLONG(l,p) l = kt_decodeULONG(p); p += sizeof(CARD32)
#define WRITELONG(l,p) kt_encodeULONG(l,p); p += sizeof(CARD32)

#define MAGIC_BYTE '\xa4'
#define MAGIC_LONG (void *) 0xa4a4a4a4
#ifdef DEBUG
#define D(x) x
#define DL(n,x) { if (gv -> debugLevel & n) x; }
#define DP(x) { printf("--- %s,%d: ", __FILE__, __LINE__); x; }
/*
#define ASSERT(ex) 	{if (!(ex)){(void)fprintf(stderr,"Assertion failed: file \"%s\", line %d\n", __FILE__, __LINE__);}}
*/
#define ASSERT(p) assert(p)
#define SAFE_FREE(p) { if (MAGIC_LONG != ((void *) p)) free(p); p = (void *) 0xa5a5a5a5; }
#else
#define D(x) NULL
#define DL(n,x) NULL
#define DP(x) NULL
#define SAFE_FREE(p) free(p)
#define ASSERT(a) NULL
#endif
#ifdef DEBUG_FD_SET
#define	FD_SET(n, p)	{\
      ((p)->fds_bits[(n)/NFDBITS] |= (1 << ((n) % NFDBITS)));\
      DP(printf("setting fd %d\n", n)); }
#endif

#define DEBUG_PACKETS 1
#define DEBUG_MISC 2

#define WORD64COUNT(_bytes) (((unsigned int) ((_bytes) + 7)) >> 3)
#define PAD32(_bytes) ((4 - ((unsigned int) (_bytes) % 4)) % 4)
#define PAD64(_bytes) ((8 - ((unsigned int) (_bytes) % 8)) % 8)

/*
** This info is used by both ktserv and libkoalatalk.
** ktserv is an RPC service that will respond to procedure number GETPORT
** and will send the actual ICE connection it is listening to
*/
#ifndef NO_RPC
#define KTS_PROG_NUM 600000042
#define KTS_VERS_NUM 0
#define KTS_PROTO_NUM IPPROTO_TCP
#define GETPORT 1    /* number of RPC procedure to retrieve the port */
#endif /* NO_RPC */


/*
** This is the name of the variable that is passed silently to
** a client launched by the server. Its value is sent back
** to the server during the init phase so that a client can
** say "hey I was autmatically launched because I am observing
** *this* krs".
*/

#define VARIABLE_KTSERV_OPERATION "KTSERV_KRS"

/*
** Name of the host where ktserv is running. This host's portmapper
** will be consulted in order to retrieve the full ICE coordinates
*/
#define VARIABLE_KTSERV_HOST "KTSERV_HOST"

/*
** Name of the property used to identity a running server
*/
#define PROPERTY_KTSERV_COORD "KTSERV_ICE_COORD"
#define PROPERTY_WINDOW "KTSERV_WINDOW_ID"

/*
** Main type describing a Koalatalk Resource Location
*/

typedef struct _Kt_KRL {
                                       CARD32 krlId;
                                       char *krlName;
} Kt_KRLRec;


/*
** Main structure to send a message peer to peer
** A message can have any number of arguments.
** An argument is one of two types : a string or a stream of bytes.
** For the latter, the number of bytes must be specified.
** messages (db of Kt_peerMessageDescription) : the messages
**
** opLength : strlen of the operation field
** opCode : type of this message
** krSpecId : only used for MESSAGE_FORWARDED, id of the matched krSpec
** krlId : where this message was sent to
*/

typedef enum { KT_UNKNOWN, KT_STRING = 1, KT_BYTES = 2 } Kt_messageType;

typedef struct _Kt_messageArg {
   Kt_messageType type;
   union {
      struct {
         size_t length;
         void *bytes;
      } b;
      struct {
         char *string;
      } s;
   } content;
} Kt_messageArgRec, *Kt_messageArg;

typedef struct _Kt_Message {
                                  DataBase arguments;
				  char *iceSenderCoord;
                                  void *buffer;
				  void *bufferPointer;
                                  size_t size;
                                  Bool isBufferInitialized;
				  char *operation;
                                  int opLength;
				  int opCode;
				  CARD32 krlId;
                                  CARD32 krSpecId;
/*
				  CARD32 dest;
*/
				  union {
				     struct {
					char *krl;
				     } peerToPeer;
				     struct {
					char *krl;
					CARD32 requestId;
				     } request;
				     struct {
					char *krl;
					CARD32 requestId;
				     } replyToRequest;
				  } msgType;
				  
} Kt_MessageRec;

/*
** Description of a peer-to-peer connection
** iconn : IceConnection to this peer
** protocol : protocol number used to contact this host
** callbacks : called when a message arrives on a direct connection
*/

typedef struct _Kt_peerMsgCallbackType {
                                          Kt_CallbackPeerMsg cb;
                                          void *userData;
} Kt_peerMsgCallbackTypeRec, Kt_peerMsgCallbackType;

typedef struct _Kt_PeerConnection {
                                       IceConn iceConn;
                                       int protocol;
                                       Kt_peerMsgCallbackType callbacks[1];
} Kt_PeerConnectionRec;

/*
** Main structure to hold information about the connection to the
** server
**
** iceConn : ICE connection
** incomingConn : connection where we listen to direct connections
** protocol : protocol number
** strIdentity : the ice identity (toolkit-chosen)
** errorEnv : environment for longjmp in case of IO error
** customIOErrorHandler : if the caller decides to use their own handler
** krlUniqueId : id of this client
** lastDeclaredKRLName : last declared KRL name
** lastDeclaredKRLId : last declared KRL name
** lastInterestedInKRSpec : last observed/handled koalatalk resource spec
** lastCallback : last declared callback
** lastUserData : last declared userData
** directConn (db of Kt_PeerConnection): all the direct connections to myself
** directCallbacks : the callback called when a direct conn arrives
*/

typedef struct _Kt_serverConnection {
                                       IceConn iceConn;
                                       IceConn iceIncomingConn;
                                       int iceProtocol;
                                       jmp_buf errorEnv;
                                       void *customIOErrorHandler;

                                       CARD32 krlUniqueId;

                                /*
                                ** The last* fields are used by
                                ** kt_sendPacket() essentially
                                */

                                       char *lastDeclaredKRLName;
                                       CARD32 lastDeclaredKRLId;
                                       char *lastInterestedInKRSpec;
                                       CARD32 lastKRSpecId;

                                       void *lastCallback;
                                       void *lastUserData;

                                       void *directCallbacks;
} Kt_serverConnectionRec, *Kt_serverConnection;

/*
** Main type to define a handler
** callback : called when the reply from the handler comes
** userData : user data
** ksr : kr Spec for this handler
** krSpecId : the id to give to the callback
** iceCoord : ice coordinates to contact directly this handler
** fd : used by the server only
**
** Note : when information about a handler is first stored here,
** it's because the server notified us about it. So we don't know
** yet what callback will be called (if ever). In that case
** the callback field is set to CALLBACK_UNKNOWN
** Note2 : if iceCoord = NULL, then this handler is actually ourselves
*/

#define CALLBACK_UNKNOWN (void *) 0x4489
typedef struct _Kt_handler {
                                         void *callback;
					 void *userData;
					 char *krs;
					 CARD32 krSpecId;
					 char *iceCoord;
					 int fd;
} Kt_handlerRec, *Kt_handler;


/*
** Main type for the koalatalk
** serverConn : connection to the server
** interestedInCallbacks (db of Kt_interestedInCallback) : callbacks
** directConn (db of Kt_peerConnection) : direct connections
** addInput,removeInput : hooks to plug your own input handling
** mainFdSet : only used if addInput and removeInput are to their default value
  ** userFileDescriptors (db of Kt_userFileDescriptor) : user defined
** directNewConn : a temporary connection (status IceConnectPending)
** directProt : the protocol number assigned to us for direct connections
** directFd : file descriptor where we listen to direct-comms requests
** directListenObj : the listen obj (directFd comes from here)
** directCallback : to be called if a message is received
** directClosure : userData for the callback
** serial : an incremented counter for each kr/krl
** lastReceivedMessage : last message received from peer or server
** bLastMessageFromServer : True if it came from the server
*/

typedef struct _Kt_interestedInCallback {
                                       void *callback;
                                       void *userData;
                                       char *krs;
                                       CARD32 krSpecId;
                                       Bool isObserve;
} Kt_interestedInCallbackRec, *Kt_interestedInCallback;

typedef struct _Kt_directIdentity {
                                       char *strPeer;
                                       int directFd;
                                       IceListenObj directListenObj;
} *Kt_directIdentity, Kt_directIdentityRec;

/*
typedef struct _Kt_userFileDescriptor {
   int fd;
   void *f;
   void *userData;
} Kt_userFileDescriptorRec, *Kt_userFileDescriptor;
*/

typedef struct _Kt_Context {
                                       Kt_serverConnection serverConn;
                                       DataBase interestedInCallbacks;
                                       DataBase directConn;
                                       Kt_CallbackAddInput addInput;
                                       void *addInputUserData;
                                       Kt_CallbackAddInput removeInput;
                                       void *removeInputUserData;
                                       fd_set mainFdSet;
/*
				       DataBase userFileDescriptors;
*/
                                       IceConn directNewConn;
                                       DataBase directIdentities;

                                       char *directIdentity;
/*
				       int directProt;
                                       int directFd;
*/

                                       IceListenObj directListenObj;
                                       Kt_CallbackPeerMsg directCallback;
                                       void *directClosure;
				       Kt_Message lastReceivedMessage;
				       Bool bLastMessageFromServer;
				       char *lastKRL;
                                       CARD32 serial;
} Kt_ContextRec;

/*
** This is the main structure for a received message, when all we
** have is a stream of bytes.
** The address of the field 'bytes' of this structure must be passed
** to kt_messageDecode in order to obtain a Kt_Message var.
** Ktm_PeerToPeer, Ktm_MessageSent, Ktm_Request Ktm_MessageSendOnExit
** are actually identical to
** this type (which can be therefore seen as the canonical form of
** a raw message).
** Any packet contining this MESSAGE_HEADER as header must also have
** an 'operation' field in the end (of variable length, so it can't
** be put in the header)
** dest : depending on the nature of the message, indicates the destination
** opLength : length of the "operation" field
*/
typedef struct _Kt_messageBuffer {
                                       KT_HEADER;
				       char *bytes[1];
} Kt_messageBufferRec, *Kt_messageBuffer;

#define K_SIZEOF_MESSAGE_BUFFER_DATA(_mb) (_mb -> length * 8 - sizeof(_mb -> dest) - sizeof(void *))
/*
** This type defines a request.
** Each time a new handler connects, the queue is checked out, and all
** the matching requests not sent yet are removed and forwarded to the
** appropriate handler. Then the request remains in the queue but
** its status is changed to KT_SENT. When the reply comes, its status
** changes temporarily to KT_PROCESSED as long as the queue is being
** processed. When it's finished, all KT_PROCESSED requests are
** purged.
*/

typedef enum { KT_QUEUED, KT_SENT, KT_PROCESSED } Kt_requestStatus;

typedef struct _Kt_request {
                                 Kt_KRL krl;
				 Kt_Message msg;
				 CARD32 requestId;
				 Kt_Disposition disposition;
				 Kt_requestStatus status;
				 Kt_Context context;
				 Kt_CallbackHandlerMsg callback;
				 void *userData;
} *Kt_request, Kt_requestRec;


/***********************************************************************
** Prototypes for private functions used in all modules
*/

/* kt_compat.c */

int kt_getdtablesize();
char *kt_strdup(char *);

/* kt_message.c */
Kt_Message
kt_messageDecode(char *p, size_t operationLength);

/* kt_lib.c */
void
kt_sendPacket(Kt_Context context, Kt_serverConnection sconn, int opCode);

void
kt_addRequestToQueue(Kt_Context, Kt_KRL krl, Kt_Message msg, CARD32 requestId,
		     Kt_Disposition disposition,
		     Kt_CallbackHandlerMsg callback, void *userData,
		     Kt_requestStatus status);

Kt_Status
kt_sendToHandler(Kt_Context context, Kt_handler handler, Kt_KRL krl,
		 Kt_Message msg, CARD32 requestId,
		 Kt_CallbackHandlerMsg callback, void *userData);
Kt_Status
kt_sendToPeer(Kt_Context context, char *strPeer, Kt_Message msg,
	      CARD32 requestId,
	      void *callback, void *userData,
	      Kt_KRL krl, Kt_handler handler,
	      CARD32 opcode);

void
kt_messagePrepare(char *senderCoord, Kt_Message msg);

Kt_handler
kt_addHandlerDetailed(char *krSpec, CARD32 specid, char *iceCoord,
		      Kt_CallbackServerMsg callback, void *userData);

void
kt_checkOutPendingRequests(Kt_handler handler);

/* kt_mainloop.c */

void
kt_addInput(int fd, void *userData);

void
kt_removeInput(int fd, void *userData);

void
kt_mainLoop(Kt_Context context);

/* kt_encoding.c */

char *
kt_messageStartPrepare(IceConn iceConn, Kt_Message msg, char *buffer);

size_t
kt_messageEndPrepare(IceConn iceConn, Kt_Message msg);
/* Return the final length of the message in units of 8 bytes */

Kt_Message
kt_messageStartDecode(Kt_messageBuffer mb);

void
kt_messageEndDecode(Kt_Message msg);


#endif /* __KT_TYPESP_H__ */

