/* Copyright (C) 1994 Groupe BULL. See file COPYRIGHT for details */
/*
 *
 * $Id: kt_message.c,v 1.3 1994/12/15 13:38:44 beust Exp $
 */

#include <stdio.h>
#include <assert.h>
#include "kt_lib.h"
#include "kt_protocolP.h"
#include "kt_typesP.h"
#include "database.h"

/****************************************************************************
 * Private
 ****************************************************************************/

static Kt_Status
kt_messageSend(Kt_Context context, Kt_KRL krl, Kt_Message msg, CARD32 opCode)
{
   IceConn iceConn;
   Kt_serverConnection sconn;
   Kt_Status result = KT_OK;
   Kt_messageBuffer mb;
   char *fillBuffer, *operation;
   int protocol;
   
   msg -> opCode = Ktp_MESSAGE_SENT;
   msg -> krlId = krl -> krlId;

   sconn = context -> serverConn;
   if (NULL == sconn) {
      fprintf(stderr, "*** kt_MessageSend: no connection to the server\n");
      result = KT_ERR;
   }
   else {
      char *fillBuffer;
      int l;
      iceConn = sconn -> iceConn;
      protocol = sconn -> iceProtocol;


      /*
      ** Damn this is ugly. Relying on an opcode to determine
      ** the type of the message. Some genericity would have helped here
      */
      switch(opCode) {
	 case Ktp_MESSAGE_SENT : {
	    Ktm_MessageSent pmsg;
	    IceGetHeaderExtra(iceConn, protocol, opCode,
			      ICE_HEADER_SIZE, 0,
			      Ktm_MessageSentRec, pmsg, fillBuffer);
	    mb = (Kt_messageBuffer) pmsg;
	    
	    fillBuffer = K_VARIABLE_FIELD(fillBuffer);
	    fillBuffer = kt_messageStartPrepare(iceConn, msg, fillBuffer);
	 }
	 break;

	 case Ktp_MESSAGE_SEND_ON_EXIT : {
	    Ktm_MessageSendOnExit pmsg;
	    IceGetHeaderExtra(iceConn, protocol, opCode,
			      ICE_HEADER_SIZE, 0,
			      Ktm_MessageSendOnExitRec, pmsg, fillBuffer);
	    mb = (Kt_messageBuffer) pmsg;
	    
	    fillBuffer = K_VARIABLE_FIELD(fillBuffer);
	    fillBuffer = kt_messageStartPrepare(iceConn, msg, fillBuffer);
	    msg -> bufferPointer = fillBuffer;
	 }
	 break;

	 default : {
	    ASSERT(0);
	 }
      }
      l = kt_messageEndPrepare(iceConn, msg);
      mb -> length = l;

      IceFlush(iceConn);

   }

   return result;
  
}

/****************************************************************************
 * Public
 ****************************************************************************/


/*
**---------------------------------------------------------------------------
** kt_MessageSendToPeer
*/
Kt_Status
kt_MessageSendToPeer(Kt_Context context, char *strPeer, Kt_Message msg)
{
   return kt_sendToPeer(context, strPeer, msg, 42,   /*@@ id number */
			NULL, NULL,  /* no callback, no userData */
			NULL,   /* no krl */
			NULL,   /* no handler */
			Ktp_PEER_TO_PEER);
}

/*
**---------------------------------------------------------------------------
** kt_MessageNew
*/
Kt_Message
kt_MessageNew()
{
   Kt_Message result;
   char *empty = (char *) malloc(1);
   empty[0] = '\0';

   NEW(result, Kt_MessageRec);
   memset(result, 0, sizeof(*result));
   result -> isBufferInitialized = False;

   result -> arguments = DB_NewDataBase(sizeof(Kt_messageArgRec),NULL);
   result -> operation = "";
   return result;
}

/*
**---------------------------------------------------------------------------
** kt_MessageFree
*/
void
kt_MessageFree(Kt_Message msg)
{
   DataBase db;
   Kt_messageArg ma;

   if (NULL == msg) return;
   db = msg -> arguments;

   SAFE_FREE(msg -> buffer);
   SAFE_FREE(msg -> operation);
   DB_Rewind(db);
   while (! DB_EndOfDataBase(db)) {
      ma = (Kt_messageArg) DB_NextEntry(db);
      if (KT_STRING == ma -> type) {
	 SAFE_FREE(ma -> content.s.string);
      }
      else {
	 SAFE_FREE(ma -> content.b.bytes);
      }
      SAFE_FREE(ma);
   }
   DB_DestroyDataBase(msg -> arguments);

   switch(msg -> opCode) {
      case Ktp_PEER_TO_PEER : {
	 free(msg -> msgType.peerToPeer.krl);
	 break;
      }
      case Ktp_REQUEST : {
	 free(msg -> msgType.request.krl);
	 break;
      }
      case Ktp_REPLY_TO_REQUEST : {
	 free(msg -> msgType.replyToRequest.krl);
	 break;
      }
   }
   SAFE_FREE(msg);
}


/*
**---------------------------------------------------------------------------
** kt_MessageCopy
*/
Kt_Message
kt_MessageCopy(Kt_Message msg)
{
   Kt_Message result;
   DataBase db = msg -> arguments;
   Kt_messageArg ma, ma2;


   /*
   ** Regular fields to duplicate
   */
   result = kt_MessageNew();
   result -> size = msg -> size;
   result -> operation = kt_strdup(msg -> operation);
   result -> iceSenderCoord = kt_strdup(msg -> iceSenderCoord);
   result -> opCode = msg -> opCode;
   result -> krlId = msg -> krlId;

   /*
   ** Duplicate arguments
   */
   DB_Rewind(db);
   while (! DB_EndOfDataBase(db)) {
      ma = (Kt_messageArg) DB_NextEntry(db);
      NEW(ma2, Kt_messageArgRec);
      ma2 -> type = ma -> type;
      if (KT_STRING == ma -> type) {
	 ma2 -> content.s.string = kt_strdup(ma -> content.s.string);
      }
      else {
	 size_t l = ma -> content.b.length;
	 ma2 -> content.b.length = l;
	 ma2 -> content.b.bytes = (void *) malloc(l);
	 memcpy(ma2 -> content.b.bytes, ma -> content.b.bytes, l);
      }
      DB_AddEntry(result -> arguments, ma2);
   }

   switch(result -> opCode) {
      case Ktp_MESSAGE_SEND_ON_EXIT :
	 /* fallthrough */
      case Ktp_MESSAGE_SENT : {
	 break;
      }
      case Ktp_PEER_TO_PEER : {
	 result -> msgType.peerToPeer.krl = msg -> msgType.peerToPeer.krl;
	 break;
      }
      case Ktp_REQUEST : {
	 result -> msgType.request.krl = msg -> msgType.request.krl;
	 result -> msgType.request.requestId= msg -> msgType.request.requestId;
	 break;
      }
      case Ktp_REPLY_TO_REQUEST : {
	 result -> msgType.replyToRequest.krl =
	    msg -> msgType.replyToRequest.krl;
	 result -> msgType.replyToRequest.requestId =
	    msg -> msgType.replyToRequest.requestId;
	 break;
      }
      default :
	 fprintf(stderr, "*** kt_MessageCopy : unknown opcode %d\n", result->opCode);
	 ASSERT(0);
   }

   return result;
}

/*
**---------------------------------------------------------------------------
** kt_MessageOperationSet
*/
Kt_Status
kt_MessageOperationSet(Kt_Message msg,
		       char *string)
{
   if (NULL != msg -> operation && 0 != strcmp(msg -> operation, ""))
      SAFE_FREE(msg -> operation);
   msg -> operation = kt_strdup(string);
   return KT_OK;
}

/*
**---------------------------------------------------------------------------
** kt_MessageOperationGet
*/
char *
kt_MessageOperationGet(Kt_Message msg)
{
   return msg -> operation;
}


/*
**---------------------------------------------------------------------------
** kt_MessageStringAdd
*/
Kt_Status
kt_MessageStringAdd(Kt_Message msg, char *string)
{
   Kt_messageArg msgArg;

   NEW(msgArg, Kt_messageArgRec);
   msgArg -> type = KT_STRING;
   msgArg -> content.s.string = kt_strdup(string);
   DB_AddEntry(msg -> arguments, msgArg);
   return KT_OK;
}

/*
**---------------------------------------------------------------------------
** kt_MessageBytesGet
*/
Kt_Status
kt_MessageBytesAdd(Kt_Message msg, void *bytes, size_t len)
{
   Kt_messageArg newMsg;

   NEW(newMsg, Kt_messageArgRec);
   newMsg -> type = KT_BYTES;
   newMsg -> content.b.length = len;
   newMsg -> content.b.bytes = (void *) malloc(len);
   memcpy(newMsg -> content.b.bytes, bytes, len);
   DB_AddEntry(msg -> arguments, newMsg);
   return KT_OK;
}

/*
**---------------------------------------------------------------------------
** kt_MessageStringGet
*/
Kt_Status
kt_MessageStringGet(Kt_Message msg,
		    int argPos, char **returnedString)
{
   Kt_Status result = KT_OK;
   DataBase db = msg -> arguments;
   Kt_messageArg arg;
   *returnedString = "";

   ASSERT(msg);
   ASSERT(msg -> arguments);
   if (argPos >= 0 && argPos < DB_Count(msg -> arguments)) {

      DB_Rewind(db);
      arg = (Kt_messageArg) DB_NthEntry(db, argPos);
      ASSERT(arg);
      if (KT_STRING != arg -> type) {
	 fprintf(stderr, "*** kt_MessageStringGet: wrong type\n");
	 result = KT_ERR;
      }
      else {
	 *returnedString = kt_strdup(arg -> content.s.string);
      }
   }
   else {
      result = KT_ERR;
      fprintf(stderr, "*** kt_MessageStringGet: n=%d too big\n", argPos);
   }

   return result;
}

/*
**---------------------------------------------------------------------------
** kt_MessageBytesGet
*/
Kt_Status
kt_MessageBytesGet(Kt_Message msg,
		   int argPos, char **returnedBytes, size_t *returnedLength)
{
   Kt_Status result = KT_OK;
   DataBase db = msg -> arguments;
   Kt_messageArg arg;

   ASSERT(msg);
   ASSERT(msg -> arguments);

   if (argPos >= 0 && argPos < DB_Count(msg -> arguments)) {

      DB_Rewind(db);
      arg = (Kt_messageArg) DB_NthEntry(db, argPos);
      ASSERT(arg);

      if (KT_BYTES != arg -> type) {
	 fprintf(stderr, "*** kt_MessageBytesGet : wrong type\n");
	 result = KT_ERR;
      }
      else {
	 *returnedLength = arg -> content.b.length;
	 *returnedBytes = (char *) malloc(*returnedLength);
	 memcpy(*returnedBytes, arg -> content.b.bytes, *returnedLength);
      }
   }
   else {
      result = KT_ERR;
      fprintf(stderr, "*** kt_MessageBytesGet : n=%d too big\n", argPos);
   }

   return result;
}



/*
**---------------------------------------------------------------------------
** kt_MessageSend
*/

Kt_Status
kt_MessageSend(Kt_Context context, Kt_KRL krl, Kt_Message msg)
{
   return kt_messageSend(context, krl, msg, Ktp_MESSAGE_SENT);
}

/*
**---------------------------------------------------------------------------
** kt_MessageSendWithReply
*/

static Bool
findHandlerBySpec(Kt_handler h, char *krlName)
{
   return kt_regexpMatch(h -> krs, krlName);
}

CARD32
kt_MessageSendWithReply(Kt_Context context, Kt_KRL krl, Kt_Message msg,
			Kt_Disposition disposition,
			Kt_CallbackHandlerMsg callback, void *userData)
{
   extern DataBase KnownHandlers;
   DataBase db = KnownHandlers;
   Kt_handler handler;
   CARD32 result = context -> serial++;
   char *krlName = krl -> krlName;

   msg -> opCode = Ktp_REQUEST;
   handler = DB_LocateEntry(db, (void *) findHandlerBySpec, krlName);
   switch (disposition) {
      case KT_QUEUE : {
	 /*
	 ** Do we know a handler that could take care of this request?
	 ** If yes, send directly to it. Otherwise, add the request
	 ** to the queue
	 */
	 if (NULL == handler) {
	    kt_addRequestToQueue(context, krl, msg, result,
				 disposition, callback, userData, KT_QUEUED);
	 }
	 else {
	    kt_sendToHandler(context, handler, krl, msg, result,
			     callback, userData);
	 }
	 break;

	 case KT_START : {
	    if (NULL == handler) {
	       kt_addRequestToQueue(context, krl, msg, result,
				    disposition, callback, userData, KT_QUEUED);
	       context -> serverConn -> lastDeclaredKRLName = krl -> krlName;
	       kt_sendPacket(context, context -> serverConn, Ktp_LAUNCH);
	    }
	    else {
	       kt_sendToHandler(context, handler, krl, msg, result,
				callback, userData);
	    }

	 }
	 break;
      }
   }

   return result;
}

/*
**---------------------------------------------------------------------------
** kt_MessageReply
*/
Kt_Status
kt_MessageReply(Kt_Context context, Kt_Message query, Kt_Message reply)
{
   Kt_KRLRec krl;
   CARD32 requestId;

   ASSERT(query -> opCode == Ktp_REQUEST);
   kt_encodeULONG(query -> msgType.request.requestId, & requestId);
   kt_encodeULONG(query -> krlId, & krl.krlId);
   reply -> opCode = Ktp_REPLY_TO_REQUEST;
   reply -> iceSenderCoord = kt_strdup(context -> directIdentity);

   return kt_sendToPeer(context, query -> iceSenderCoord, reply, requestId,
			NULL, NULL,
			& krl,   /*@@ to be removed??? */
			NULL,   /* no handler */
			Ktp_REPLY_TO_REQUEST);
}

/*
**---------------------------------------------------------------------------
** kt_MessageArgCount
*/
int
kt_MessageArgCount(Kt_Message msg)
{
   ASSERT(msg -> arguments);
   return DB_Count(msg -> arguments);
}

/*
**---------------------------------------------------------------------------
** kt_MessageArgIsString
*/
Bool
kt_MessageArgIsString(Kt_Message msg, int argPos)
{
   Kt_messageArg arg;

   ASSERT(msg);
   ASSERT(msg -> arguments);
   ASSERT(argPos >= 0 && argPos < DB_Count(msg -> arguments));

   DB_Rewind(msg -> arguments);
   arg = DB_NthEntry(msg -> arguments, argPos);

   return (KT_STRING == arg -> type);
}

/*
**---------------------------------------------------------------------------
** kt_MessageSendOnExit
*/
Kt_Status
kt_MessageSendOnExit(Kt_Context context, Kt_KRL krl, Kt_Message msg)
{
   Kt_Status result = KT_OK;

   if (0 != krl)
      kt_messageSend(context, krl, msg, Ktp_MESSAGE_SEND_ON_EXIT);
   else
      result = KT_ERR;

   return result;
}

/*
**---------------------------------------------------------------------------
** kt_MessageSendOnExit
*/
char *
kt_MessageSender(Kt_Message msg)
{
   return msg -> iceSenderCoord;
}
