/* Copyright (C) 1994 Groupe BULL. See file COPYRIGHT for details */
/*
 *
 * $Id: kt_lib.c,v 1.4 1994/12/15 13:38:20 beust Exp beust $
 */


#include <stdio.h>
#include <stdlib.h>   /* getenv() */
#include <assert.h>
#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include "regexp.h"
#include "kt_lib.h"
#include "kt_typesP.h"
#include "kt_protocolP.h"
#include "database.h"

/*
** One instance of this type for each direct connection
*/
/* Put type here from kt_lib.h */

Kt_Context LastContext;


/**************************************************************************
 * Handling our connections
 **************************************************************************/

/*
** PendingRequests contains the list of requests that were sent but
** couldn't find a matching handler.
*/

DataBase PendingRequests;  /* db of Kt_request */

/*
** This variable is used to hold the list of all the Kt_serverConn
** this application has opened. Unfortunately, it seems mandatory
** since there is no way in the IOErrorHandler to deduce the
** Kt_serverConn related to the IceConnection.
** See the functions kt_addConnection(), kt_removeConnection()
** and kt_locateConnection()
*/

/* DataBase of Kt_serverConnectionRec (kt_typesP.h) */
static DataBase _ServerConnections = NULL;
/* DataBase of Kt_ */
/*
static DataBase _PeerConnections = NULL;
*/

/*
** Here I hold all the known handlers. This is independant from the
** connection, so a separate variable is needed
*/
/* DataBase of Kt_handler */
DataBase KnownHandlers;

/*
** conn can be either a serverConn or a PeerConnection
*/
void
kt_addConnection(DataBase db, void *conn)
{
   DB_AddEntry(db, conn);
}

static void
kt_removeConnection(DataBase db, void *kconn)
{
   DB_RemoveEntry(db, kconn);
}


Kt_PeerConnection
kt_locatePeerConnection(IceConn iceConn, DataBase db)
{
   Kt_PeerConnection result = NULL, cconn;
   Bool searching = True;

   DB_Rewind(db);
   while (True == searching) {
      if (DB_EndOfDataBase(db)) searching = False;
      else {
         cconn = (Kt_PeerConnection) DB_NextEntry(db);
         ASSERT(cconn);
         if (cconn -> iceConn == iceConn) {
            searching = False;
            result = cconn;
         }
      }
   }
   
   return result;
}

/*
** Simple regular expression matching, common to server and lib
*/
Bool
kt_regexpMatch(char *reg, char *string)
{
   Bool result = False;

   regexp *re = regcomp(reg);
   if (1 == regexec(re, string))
      result = True;

   SAFE_FREE(re);
   return result;
}

/**************************************************************************
 * ICE callbacks
 **************************************************************************/

/*
 **-----------------------------------------------------------------------
 ** kt_defaultIOHandler
 */
/* callback of type IceIOErrorProc */
static void
kt_defaultIOHandler(IceConn conn)
{
   Kt_CallbackRemoveInput removeInput;
   Kt_PeerConnection cconn = NULL;
   Kt_serverConnection sconn = NULL;
   Kt_Context context;
   /*
   ** Find which kt_Connection contains this IceConn
   */
   context = LastContext;    /*@@ damn global variable */
   removeInput = context -> removeInput;
   if (NULL != context) {
      /*
      ** First, try to see of it was a direct standalone connection
      */
      cconn = kt_locatePeerConnection(conn, context -> directConn);
      if (NULL != cconn) {
         kt_removeConnection(context -> directConn, cconn);
	 (*removeInput)(IceConnectionNumber(cconn -> iceConn),
			context -> removeInputUserData);
         sconn = context -> serverConn;
      }
      else if (NULL != context -> serverConn) {
         /*
         ** ... or the connection to the server?
         */
         sconn = context -> serverConn;
         if (sconn -> iceConn == conn) {
            D(fprintf(stderr, "server connection lost\n"));
            exit(0);
         }
         else {
            /*
            ** ... neither, then it must be a direct server-provided connection
            */
            cconn = kt_locatePeerConnection(conn, context -> directConn);
            ASSERT(cconn);
            D(fprintf(stderr, "direct connection through server lost\n"));
            ASSERT(0);
         }
      }
   }

   ASSERT(NULL != sconn || NULL != cconn);

/*
   longjmp(KoalaTalk_MainLoopJmp, 0);
*/
}

/*
 **-----------------------------------------------------------------------
 ** kt_runCallbacks
 ** A message had been received from the server, look up the possible
 ** callbacks for it and call them.
 ** Return : True if the msg was dispatched.
 */

static Bool
kt_runCallbacks(Kt_Message msg)
{
   Bool result = False;
   Kt_Context context = LastContext;
   char *fillBuffer;
   DataBase db;
   CARD32 krSpecId = msg -> krSpecId;
   ASSERT(context);

   db = context -> interestedInCallbacks;
   DB_Rewind(db);
   DP(printf("runcallbacks starting\n"));
   while (! DB_EndOfDataBase(db)) {
      Kt_interestedInCallback ic = DB_NextEntry(db);
      DP(printf("comparing %d and %d\n", krSpecId, ic->krSpecId));
      /*
      ** The message can only be a notice (a request would come from
      ** a peer)
      */
      if (True == ic -> isObserve && krSpecId == ic -> krSpecId) {
         void (*f)();

         f = ic -> callback;
         ASSERT(f);
	 DP(printf("running a callback...\n"));
         (*f)(context, ic -> userData, msg, krSpecId);
	 result = True;
      }
   }
   return result;
}

/*
**-----------------------------------------------------------------------
** kt_deleteHandler
** A handler disconnected, we must remove all the krSpec it was
** handling
*/

static Bool
findHandlerByCoord(Kt_handler handler, char *coord)
{
   if (0 == strcmp(coord, handler -> iceCoord)) return True;
   else return False;
}

void
kt_deleteHandler(char *iceCoord)
{
   DataBase db = KnownHandlers;
   size_t l;
   Kt_handler handler;

   DP(printf("removing handler '%s'\n", iceCoord));
   handler = DB_LocateEntry(db, (void *) findHandlerByCoord, iceCoord);
   ASSERT(handler);
   DB_RemoveEntry(db, handler);

   SAFE_FREE(iceCoord);
}

/*
**-----------------------------------------------------------------------
** kt_addHandlerDetailed
** The server just notified us a new handler is now connected. Put
** it into our internal table.
** kt_addHandlerDetailed will *not* copy the strings it receives.
** They must be properly allocated by the caller
*/

Kt_handler
kt_addHandlerDetailed(char *krSpec, CARD32 specid, char *iceCoord,
		      Kt_CallbackServerMsg callback, void *userData)
{
   Kt_handler result;

   /*
   ** There is no attached callback at this point
   */
   NEW(result, Kt_handlerRec);
   result -> callback = callback;
   result -> userData = userData;

   result -> krs = krSpec;
   result -> krSpecId = specid;
   result -> iceCoord = (0 == iceCoord ? 0 : kt_strdup(iceCoord));
   DB_AddEntry(KnownHandlers, result);
   DP(printf("added a handler '%s'\n", iceCoord));

   return result;
}

/*
** Encoding/Decoding long 
** @@ These two functions should be optimized (e.g. inlined) for
** better performances : they are used and abused everywhere in Koalatalk
*/
unsigned long
kt_decodeULONG(char *pt)
{
  return (pt[0] * (1 << 24) & 0xff000000) +
         (pt[1] * (1 << 16) & 0xff0000) +
         (pt[2] * (1 << 8) & 0xff00) +
         (pt[3] & 0xff);
}

void
kt_encodeULONG(unsigned long n, char *pt)
{
  pt[0] = (n >> 24) & 0xff;
  pt[1] = (n >> 16) & 0xff;
  pt[2] = (n >> 8) & 0xff;
  pt[3] = n & 0xff;
}

/*
**-----------------------------------------------------------------------
** kt_sendToPeer
** This function is used to send a message directly to a peer.
** It is used by
** kt_MessageSendToPeer()
** kt_MessageSendWithReply()
** kt_MessageReply()
** 
*/

/* Return the canonical name of this host */
static char *
kt_getCName(char *host)
{
   struct hostent *h;
   char *result;
   unsigned char l[4];
   int a,b,c,d;
   char format[10];

   h = gethostbyname(host);
   if (0 == h) {
      sscanf(host, "%d.%d.%d.%d", & a, & b, & c, & d);
      l[0] = a; l[1] = b; l[2] = c; l[3] = d;
      h = gethostbyaddr(l, 4, AF_INET);
      if (0 == h)
	 result = host;
      else
	 result = h -> h_name;
   }
   else
      result = h -> h_name;

   return result;
}

#if 0
/*
** ice1 and ice2 are two valid string Ice coordinates.
** Return True if both strings designate the same host
*/
static Bool
kt_compareIdentities(char *ice1, char *ice2)
{
   char p1[128], p2[128];
   char host1[128], host2[128], *chost1, *chost2;
   int port1, port2;

   if (0 == strlen(ice1) || 0 == strlen(ice2))
      return True;
   {
      char *p = ice1, *phost1 = host1;
      while (*p && *p != '/') p++;
      if (*p) p++;
      while (*p && *p != ':') *phost1++ = *p++;
      *phost1 = '\0';
   }
   sscanf(ice1, "%[^/]/%[^:]:%d", p1, host1, & port1);
   sscanf(ice2, "%[^/]/%[^:]:%d", p2, host2, & port2);
   chost1 = kt_getCName(host1);
   chost2 = kt_getCName(host2);

   if (0 == strcmp(p1, p2) &&
       0 == strcmp(chost1, chost2) &&
       port1 == port2)
      return True;
   else
      return False;
}
#endif

static Bool
kt_isHostMyself(Kt_Context context, char *strPeer)
{
   DataBase db = context -> directIdentities;
   DB_Rewind(db);
   while (! DB_EndOfDataBase(db)) {
      Kt_directIdentity di = DB_NextEntry(db);
      if (0 == strcmp(strPeer, di -> strPeer))
	 return True;
   }

   return False;
}

/*
** - handler is non NULL only if the peer we are
** sending to is a handler. In that case, we need to know
** what krSpecId its callback is supposed to receive.
** - callback can be of two types, either Kt_CallbackHandlerMsg
** if handler is non-NULL, or Kt_CallbackPeerMsg otherwise
*/

#define TYPE Ktm_RequestRec
static void
cedIceGetHeaderExtra(IceConn _iceConn, int _major, int _minor,
		     int _headerSize, int _extra, TYPE *_pMsg,
		     char *_pData) 
{
    if ((_iceConn->outbufptr + 
	_headerSize + ((_extra) << 3)) > _iceConn->outbufmax) 
        IceFlush (_iceConn); 
    _pMsg = (TYPE *) _iceConn->outbufptr; 
    if ((_iceConn->outbufptr + 
	_headerSize + ((_extra) << 3)) <= _iceConn->outbufmax) 
        _pData = (char *) _pMsg + _headerSize; 
    else 
        _pData = NULL; 
    _pMsg->majorOpcode = _major; 
    _pMsg->minorOpcode = _minor; 
    _pMsg->length = ((_headerSize - SIZEOF (iceMsg)) >> 3) + (_extra); 
    _iceConn->outbufptr += (_headerSize + ((_extra) << 3)); 
    _iceConn->send_sequence++;
}

Kt_Status
kt_sendToPeer(Kt_Context context, char *strPeer, Kt_Message msg,
	      CARD32 requestId,
	      void *callback, void *userData,
	      Kt_KRL krl, Kt_handler handler,
	      CARD32 opCode)
{
   IceConn iceConn;
   iceMsg *imsg;
   int protocol;
   Kt_PeerConnection peer;
   Kt_CallbackHandlerMsg cbHandler = (Kt_CallbackHandlerMsg) callback;
   Kt_CallbackPeerMsg cbPeer = (Kt_CallbackPeerMsg) callback;
   Kt_CallbackPeerMsg cbDefault = context -> directCallback;
   char *fillBuffer, *extra, *tmpBuffer;
   Kt_Status result = KT_OK;
   size_t varSize = 0;    /* size of variable portion of this message */

   /*
   ** Special cases :
   ** - strPeer == my own id : just sending a message to myself,
   **   call directly the directCallback
   ** For all these cases, I merely have to call the specified callback
   */

   if (True == kt_isHostMyself(context, strPeer)) {
      context -> bLastMessageFromServer = False;
      msg -> iceSenderCoord = kt_strdup(context -> directIdentity);
      switch(opCode) {
	 case Ktp_PEER_TO_PEER : {
	    (*cbDefault)(context, context -> directClosure, msg);
	    break;
	 }
	 case Ktp_REQUEST : {
	    msg -> msgType.request.krl = krl -> krlName;
	    msg -> msgType.request.requestId = requestId;
	    kt_addRequestToQueue(context, krl, msg,
				 requestId, KT_QUEUE,
				 callback, userData,
				 KT_SENT);
	    break;
	 }
	 case Ktp_REPLY_TO_REQUEST : {
	    msg -> msgType.replyToRequest.krl = NULL;
	    msg -> msgType.replyToRequest.requestId = requestId;
	    break;
	 }
	 default : {
	    ASSERT(0);
	 }
      }
      /*
      ** Now the message is set up just like if it had been created
      ** in kt_processDirectConnection. I can feed it to
      ** kt_DispatchMessage()
      */
      kt_DispatchMessage(context, msg);
      return result;
   }

   peer = kt_PeerOpenConnection(context, strPeer);
   if (NULL == peer) {
      result = KT_ERR;
      return result;
   }
   else {
      iceConn = kt_PeerReturnIceConnection(peer);
      protocol = kt_PeerReturnIceProtocol(peer);

      if (Ktp_PEER_TO_PEER == opCode) {
	 Ktm_PeerToPeer pmsg;
	 char *strSender = context -> directIdentity;
	 size_t l = strlen(strSender);
	 IceGetHeaderExtra(iceConn, protocol, opCode,
			   ICE_HEADER_SIZE, 0,
			   Ktm_PeerToPeerRec, pmsg, extra);
	 fillBuffer = extra;
	 fillBuffer = kt_messageStartPrepare(iceConn, msg, fillBuffer);
	 tmpBuffer = fillBuffer;
	 WRITELONG(l, fillBuffer);
	 memcpy(fillBuffer, strSender, l); fillBuffer += l;
	 pmsg -> length = WORD64COUNT(fillBuffer - extra);
	 varSize = sizeof(CARD32) + l;
	 msg -> size += varSize;
	 IceWriteData(iceConn, varSize, tmpBuffer);
	 pmsg -> length = kt_messageEndPrepare(iceConn, msg);
	 imsg = (iceMsg *) pmsg;
      }
      else if (Ktp_REQUEST == opCode) {
	 Ktm_Request rmsg = NULL;
	 char *operation = msg -> operation;
	 size_t operationl = strlen(operation);
	 char *fillBuffer;
/*
	 char *iceCoord = context -> directIdentity;
*/
	 char *iceCoord = kt_strdup(context -> directIdentity);
	 size_t icel = strlen(iceCoord);
	 char *strk = krl -> krlName;
	 size_t strl = strlen(strk);
/*
	 cedIceGetHeaderExtra(iceConn, protocol, opCode,
			   ICE_HEADER_SIZE, roundedLength,
			   rmsg, extra);
*/
	 IceGetHeaderExtra(iceConn, protocol, opCode,
			   ICE_HEADER_SIZE, 0,
			   Ktm_RequestRec, rmsg, extra);
	 fillBuffer = extra;
	 fillBuffer = kt_messageStartPrepare(iceConn, msg, fillBuffer);
	 tmpBuffer = fillBuffer;
	 WRITELONG(requestId, fillBuffer);
	 WRITELONG(strl, fillBuffer);
	 WRITELONG(icel, fillBuffer);
	 memcpy(fillBuffer, strk, strl); fillBuffer += strl;
	 memcpy(fillBuffer, iceCoord, icel); fillBuffer += icel;
	 varSize = sizeof(CARD32) * 3 + strl + icel;
	 msg -> size += varSize;
	 IceWriteData(iceConn, varSize, tmpBuffer);
	 rmsg -> length = kt_messageEndPrepare(iceConn, msg);
	 imsg = (iceMsg *) rmsg;

	 kt_addRequestToQueue(context, krl, msg,
			      requestId, KT_QUEUE,  /* dispos */
			      callback, userData,
			      KT_SENT);
			      
			      
      }
      else if (Ktp_REPLY_TO_REQUEST == opCode) {
	 Ktm_ReplyToRequest rtrMsg;
	 char *operation = msg -> operation;
	 char *fillBuffer;
	 char *ice = msg -> iceSenderCoord;
	 size_t operationl = strlen(operation);
	 size_t icel = strlen(ice);

	 IceGetHeaderExtra(iceConn, protocol, opCode,
			   ICE_HEADER_SIZE, 0,
			   Ktm_ReplyToRequestRec, rtrMsg, extra);
	 fillBuffer = extra;
	 fillBuffer = kt_messageStartPrepare(iceConn, msg, fillBuffer);
	 tmpBuffer = fillBuffer;
	 WRITELONG(requestId, fillBuffer);
	 WRITELONG(icel, fillBuffer);
	 memcpy(fillBuffer, ice, icel); fillBuffer += icel;
	 varSize = sizeof(CARD32) * 2 + icel;
	 msg -> size += varSize;
	 IceWriteData(iceConn, varSize, tmpBuffer);
	 rtrMsg -> length = kt_messageEndPrepare(iceConn, msg);
	 imsg = (iceMsg *) rtrMsg;
      }
      else ASSERT(0);


      IceFlush(iceConn);
   }

   return result;
}

Kt_Status
kt_sendToHandler(Kt_Context context, Kt_handler handler, Kt_KRL krl,
		 Kt_Message msg, CARD32 requestId,
		 Kt_CallbackHandlerMsg callback, void *userData)
{
   return kt_sendToPeer(context, handler -> iceCoord, msg, requestId,
			callback, userData, krl, handler, Ktp_REQUEST);
}

void
kt_checkOutPendingRequests(Kt_handler handler)
{
   char *krSpec = handler -> krs, *iceCoord = handler -> iceCoord;
   DataBase db = PendingRequests;
   Kt_request request;

   DB_Rewind(db);
   /*
   ** Find a request that would match the krs for this handler, and make sure
   ** it is in QUEUED status. If it is, send it to the handler
   */
   while (! DB_EndOfDataBase(db)) {
      request = DB_NextEntry(db);
      ASSERT(request -> msg -> opCode == Ktp_REQUEST);

      if (KT_QUEUED == request -> status &&
	  True == kt_regexpMatch(krSpec, request -> krl -> krlName)) {
	 kt_sendToHandler(request -> context, handler, request -> krl,
			  request -> msg,
			  request -> msg -> msgType.request.requestId,
			  (void *) request -> callback, request -> userData);
	 request -> status = KT_SENT;   /* careful, modifying the reference */
      }
   }
}

/*
 **-----------------------------------------------------------------------
 ** kt_processServerConnection
 ** Main callback called each time a message comes from the server
 */

/* callback of type IcePOprocessMsgProc */
static void
kt_processServerConnection(IceConn iceConn, void *clientData,
		 int opCode, unsigned long length, Bool swap,
		 IceReplyWaitInfo *replyWait, Bool *replyReadyRet)
{
   Kt_Context context = LastContext;    /*@@ hack */
   Kt_Message msg = 0;
   char *fillBuffer;

   if (! IceValidIO(iceConn)) {
      DP(printf("invalid iceConn to server\n"));
      return;
   }

   context -> bLastMessageFromServer = True;
   switch(opCode) {
     case Ktp_NEW_CLIENT_REPLY : {
        Ktm_NewClientReply reply;
        char *preply;
        DP(printf("processMessage: NEW_CLIENT_REPLY\n"));
        IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
                               Ktm_NewClientReplyRec, reply, preply);
     }
        break;
     case Ktp_MESSAGE_FORWARDED : {
        Ktm_MessageForwarded header;
        char *string;
        int l, *tab, coordLength;
        void *data;
        
        DP(printf("processMessage: MESSAGE_FORWARDED\n"));
        IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
                               Ktm_MessageForwardedRec, header, data);
	msg = kt_messageStartDecode((Kt_messageBuffer) header);
	fillBuffer = msg -> bufferPointer;
	READLONG(msg -> krSpecId, fillBuffer);
	READLONG(coordLength, fillBuffer);
	msg -> iceSenderCoord = (char *) malloc(coordLength + 1);
	memcpy(msg -> iceSenderCoord, fillBuffer, coordLength);
	msg -> iceSenderCoord[coordLength] = '\0';
	fillBuffer += coordLength;
	msg -> bufferPointer = fillBuffer;
	kt_messageEndDecode(msg);
     }
        break;
     case Ktp_NEW_HANDLER : {
        Ktm_NewHandler header;
        char *string;
        int l, *tab, lspec, lsender;
	CARD32 krSpecId;
        void *data;
	char *spec, *coo, *p;
	Kt_handler handler;
        
        DP(printf("processMessage: NEW_HANDLER\n"));
        IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
                               Ktm_NewHandlerRec, header, data);
	/*
	** Retrieve the data from the message
	*/
	msg = kt_messageStartDecode((Kt_messageBuffer) header);
	fillBuffer = msg -> bufferPointer;
	READLONG(krSpecId, fillBuffer);
	READLONG(lspec, fillBuffer);
	READLONG(lsender, fillBuffer);
	spec = (char *) malloc(lspec + 1);
	coo = (char *) malloc(lsender + 1);
	memcpy(spec, fillBuffer, lspec); fillBuffer += lspec;
	memcpy(coo, fillBuffer, lsender); fillBuffer += lsender;
	spec[lspec]=0; coo[lsender]=0;
	msg -> bufferPointer = fillBuffer;

        handler = kt_addHandlerDetailed(spec, krSpecId, coo,
					CALLBACK_UNKNOWN, NULL);
	DP(printf("new_handler = '%s'\n", coo));

	/*
	** Now we have a new handler, see if some of the pending
	** requests might interest it
	*/
	kt_checkOutPendingRequests(handler);

	/*
	** Necessary, so that this message doesn't get into
	** kt_DispatchMessage()
	*/
	msg = 0;
	

     }
        break;
     case Ktp_DELETE_HANDLER : {
        Ktm_DeleteHandler header;
        void *data;
        char *iceCoord;
        int lcoord, *tab;
        
        DP(printf("processMessage: DELETE_HANDLER\n"));
        IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
                               Ktm_DeleteHandlerRec, header, data);
	msg = kt_messageStartDecode((Kt_messageBuffer) header);
	fillBuffer = msg -> bufferPointer;
	READLONG(lcoord, fillBuffer);
	iceCoord = (char *) malloc(lcoord + 1);
	memcpy(iceCoord, fillBuffer, lcoord); fillBuffer += lcoord;
	iceCoord[lcoord] = '\0';
	msg -> bufferPointer = fillBuffer;

        kt_deleteHandler(iceCoord);

	/*
	** Necessary, so that this message doesn't get into
	** kt_DispatchMessage()
	*/
	msg = 0;
     }
        break;

     default :
        fprintf(stderr, "kt_processServerConnection : unknown opcode %d\n", opCode);
        ASSERT(0);
   }

   context -> lastReceivedMessage = msg;
}

/*
**-----------------------------------------------------------------------
** kt_processDirectConnection
** Main callback called each time a message is sent to us on a direct
** connection. The message is stored in context -> lastReceivedMessage
*/

/* callback of type IcePaProcessMsgProc */
void
kt_processDirectConnection(IceConn iceConn, IcePointer userData, int opCode,
			   unsigned long length, Bool swap)
{
   Kt_Message msg;
   Kt_PeerConnection pconn = NULL;
   Kt_Context context = LastContext;    /*@@ hack */
   int running = 1;
   DataBase db = context -> directConn;

   if (! IceValidIO(iceConn)) {
      DP(printf("invalid iceConn to client\n"));
      return;
   }

   context -> bLastMessageFromServer = False;
   msg = kt_MessageNew();
   /*
   ** Search this ICEConnection among our direct connections so that
   ** I can call the possible callbacks
   */
   DB_Rewind(db);
   while (1 == running) {
      Kt_PeerConnection pc;
      if (DB_EndOfDataBase(db)) running = 0;
      else {
         pc = (Kt_PeerConnection) DB_NextEntry(db);
         if (pc -> iceConn == iceConn) {
            running = 0;
            pconn = pc;
         }
      }
   }
   ASSERT(pconn);

   switch(opCode) {
      case Ktp_PEER_TO_PEER : {
         Ktm_PeerToPeer ptpMsg;
         char *pmsg, *p, *fillBuffer;
	 size_t l;

         IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
                                Ktm_PeerToPeerRec, ptpMsg, pmsg);
         msg = kt_messageStartDecode((Kt_messageBuffer) ptpMsg);
	 pmsg = msg -> bufferPointer;
	 /*
         ** Retrieve the coordinates length, and then the coord
         ** of sender
         */
	 READLONG(l, pmsg);
	 msg -> iceSenderCoord = (char *) malloc(l+1);
	 memcpy(msg -> iceSenderCoord, pmsg, l);
	 msg -> iceSenderCoord[l] = '\0';
	 msg -> bufferPointer = pmsg;
	 pmsg += l;
	 msg -> bufferPointer = pmsg;
	 kt_messageEndDecode(msg);
	 DP(printf("end decoding peer to peer, '%s'\n",
	    kt_ReturnIdentity(context)));
         break;
      }
      case Ktp_REQUEST : {
         Ktm_Request rMsg;
         char *pmsg, *operation;
	 char *krl, *senderIceCoord, *p;
	 CARD32 requestId;
	 size_t krlLength, iceCoordLength, operationLength;

         IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
                                Ktm_RequestRec, rMsg, pmsg);

	 msg = kt_messageStartDecode((Kt_messageBuffer) rMsg);
	 pmsg = msg -> bufferPointer;
	 /*
	 ** Retrieve the request Id, krlLength and iceCoordLength
	 */
	 READLONG(requestId, pmsg);
	 READLONG(krlLength, pmsg);
	 READLONG(iceCoordLength, pmsg);
	 /*
	 ** Retrieve the KRL and krlId
	 */
	 p = (char *) malloc(krlLength + 1);
	 memcpy(p, pmsg, krlLength);
	 p[krlLength] = '\0';
	 krl = p;
	 pmsg += krlLength;

	 /*
	 ** Retrieve the ICE coordinates of sender
	 */
	 p = (char *) malloc(iceCoordLength + 1);
	 memcpy(p, pmsg, iceCoordLength);
	 p[iceCoordLength] = '\0';
	 senderIceCoord = p;
	 pmsg += iceCoordLength;
	 ASSERT(0 != strlen(senderIceCoord));

	 msg -> iceSenderCoord = senderIceCoord;
	 msg -> opLength = rMsg -> opLength;
	 msg -> opCode = opCode;
	 msg -> msgType.request.krl = krl;
	 msg -> msgType.request.requestId = requestId;
	 msg -> krlId = rMsg -> krlId;
	 msg -> msgType.request.requestId = rMsg -> requestId;
	 msg -> bufferPointer = pmsg;
	 kt_messageEndDecode(msg);
         break;
      }

      case Ktp_REPLY_TO_REQUEST : {
         Ktm_ReplyToRequest rtrMsg;
	 DataBase db;
	 CARD32 requestId;
         char *pmsg, *strIceCoord;
	 size_t l, icel;

         IceReadCompleteMessage(iceConn, ICE_HEADER_SIZE,
                                Ktm_ReplyToRequestRec, rtrMsg, pmsg);
         msg = kt_messageStartDecode((Kt_messageBuffer) rtrMsg);
	 pmsg = msg -> bufferPointer;

	 /*
         ** Retrieve the requestId and the ICE coordinates of sender
         */
	 READLONG(requestId, pmsg);
	 READLONG(icel, pmsg);

	 strIceCoord = (char *) malloc(icel + 1);
	 memcpy(strIceCoord, pmsg, icel); pmsg += icel;
	 strIceCoord[icel] = '\0';

	 msg -> iceSenderCoord = strIceCoord;
	 msg -> opLength = rtrMsg -> opLength;
	 msg -> krlId = rtrMsg -> krlId;
	 msg -> msgType.replyToRequest.krl = NULL;
	 msg -> msgType.replyToRequest.requestId = requestId;

	 msg -> bufferPointer = pmsg;
	 kt_messageEndDecode(msg);

	 break;
      }

      default :
         fprintf(stderr, "unknown opcode %d\n", opCode);
	 assert(0);
   }
   msg -> opCode = opCode;

   context -> lastReceivedMessage = msg;
}

/**************************************************************************
 * Private
 **************************************************************************/

static void
freeServerConn(Kt_serverConnection kconn)
{
   SAFE_FREE(kconn);
}

static void
freeDirectConnection(Kt_PeerConnection pc)
{
   NULL;
}

static void
freeInterestedInCallback(Kt_interestedInCallback ic)
{
   SAFE_FREE(ic -> krs);
}

static void
freePendingRequest(Kt_request request)
{
   SAFE_FREE(request -> krl);
}

/*
 **-----------------------------------------------------------------------
 ** kt_setupListenConnection
 ** Create the socket where we listen to direct connections
 ** Initialize identity to a ICE-format network (e.g. "tcp/host:port")
 ** Initialize the fields directConn, directFd and directProt
 */


static Bool
kt_hostBasedAuth(char *host)
{
   return True;
}

static void
kt_setupListenConnection(Kt_Context context, char **identity)
{
   Kt_CallbackAddInput addInput = context -> addInput;
   char error[128];
   int count;
   int  prot;
   IceListenObj *obj, listenObj;
   Status status;
   IcePaVersionRec callbackList[] = {
         { 1, 0, (IcePaProcessMsgProc) kt_processDirectConnection }
   };

   status = IceListenForConnections(& count,
                                    & obj,
                                    128,
                                    error);

   if (count == 0 || count != 2) {
      printf("Error: '%s'\n", error);
      exit(0);
   }
   else {
      prot = IceRegisterForProtocolReply(PDIRECT_NAME, PVENDOR, PDIRECT_RELEASE,
                               1, callbackList,           /* callbacks */
			      0, /* # of auth methods supported */
			       NULL,                   /* authNames */
                               NULL,                   /* auth callbacks */
                               kt_hostBasedAuth,           /* host-base auth */
                               NULL,   /* cb before Protocol Reply is sent */
                               NULL,   /* cb after ProtocolReply is sent */
                               NULL    /* IO error handler */
	 );
      {
	 /*
          ** Iterate through the listen objects and enter them in the
          ** database.
          */
	 int i;
	 for (i=0; i<count; i++) {
	    Kt_directIdentityRec *di;
	    listenObj = obj[i];
	    NEW(di, Kt_directIdentityRec);
	    di -> directListenObj = listenObj;
	    di -> directFd = IceGetListenConnectionNumber(listenObj);
	    di -> strPeer = IceGetListenConnectionString(listenObj);

	    /*
            ** Add the fd for direct requests to the list of those we watch
            */
	    (*addInput)(di -> directFd, context -> addInputUserData);
	    IceSetHostBasedAuthProc(listenObj, kt_hostBasedAuth);
	    /*
            ** Advertise only the non-local descriptor as our valid
            ** identity.
            */
	    {
	       char *strId = IceGetListenConnectionString(listenObj);
	       if (NULL == strstr(strId, "local"))
		  *identity = kt_strdup(strId);
	       free(strId);
	    }

	    DB_AddEntry(context -> directIdentities, di);
	    DP(printf("addInput directFd: %d\n", di -> directFd));
	 }
      }
/*   mmh... shouldn't free that here, they are stored now
      IceFreeListenObj(count, obj);
*/
   }

   if (NULL == *identity) {
      /*
      **  Could only find local descriptors. Emit a warning and
      ** use the last found
      */
      char *strId = IceGetListenConnectionString(listenObj);
      fprintf(stderr, "kt_lib, warning : could only find local identiy '%s'\n",
	      strId);
      *identity = strId; 
   }
}

/*
 **-----------------------------------------------------------------------
 ** kt_sendPacket
 */
void
kt_sendPacket(Kt_Context context, Kt_serverConnection sconn, int opCode)
{
   IceConn iceConn;
   unsigned long fullLength;
   iceMsg *imsg = NULL;
   char *fillBuffer, *extra;

   ASSERT(sconn);
   iceConn = sconn -> iceConn;
   switch(opCode) {
      case Ktp_NEW_CLIENT : {
         Ktm_NewClient nc;
         char *id = context -> directIdentity;
	 char *krs = getenv(VARIABLE_KTSERV_OPERATION);
         int identityl = strlen(id);
	 int krsl;
         size_t extraLength;
         size_t roundedLength;

	 if (NULL == krs) krs = "";
	 krsl = strlen(krs);
/*
         extraLength = K_SIZEOF_NEW_CLIENT(id, krs);
         roundedLength = WORD64COUNT(extraLength);
         fullLength = ICE_HEADER_SIZE + extraLength;
*/

         /*
         ** Fill in the header
         */
         IceGetHeaderExtra(iceConn, sconn -> iceProtocol, opCode,
                           ICE_HEADER_SIZE, 0,
                           Ktm_NewClientRec, nc, extra);
	 fillBuffer = extra;
         /*
         ** Then  the specific information pertaining to this packet
         */
	 /* field krlId not relevant for this packet */
	 WRITELONG(KT_NOT_RELEVANT, fillBuffer);
	 /* opLength neither */
	 WRITELONG(KT_NOT_RELEVANT, fillBuffer);
         WRITELONG(identityl, fillBuffer);
         WRITELONG(krsl, fillBuffer);
         memcpy(fillBuffer, id, identityl); fillBuffer += identityl;
	 memcpy(fillBuffer, krs, krsl); fillBuffer += krsl;
	 nc -> length = WORD64COUNT(sizeof(CARD32) * 5 + identityl + krsl);
	 DP(printf("NEW_CLIENT sending krs '%s'\n", krs));
         imsg = (iceMsg *) nc;
         break;
      }
      case Ktp_KRL_DECLARE : {
         Ktm_KRLDeclare kd;
         int id = sconn -> krlUniqueId;
         int l = strlen(sconn -> lastDeclaredKRLName);
         CARD32 krlId = sconn -> lastDeclaredKRLId;
         char *krSpec = sconn -> lastDeclaredKRLName;
         /*
         ** Fill in the header
         */
         IceGetHeaderExtra(iceConn, sconn -> iceProtocol, opCode,
                           ICE_HEADER_SIZE, 0,
                           Ktm_KRLDeclareRec, kd, extra);
	 fillBuffer = extra;
         /*
         ** Then  the specific information pertaining to this packet
         */
	 WRITELONG(id, fillBuffer);
	 /* opLength not relevant */
	 WRITELONG(KT_NOT_RELEVANT ,fillBuffer);
         WRITELONG(krlId ,fillBuffer);
         WRITELONG(l ,fillBuffer);
         memcpy(fillBuffer, krSpec, l); fillBuffer += l;
	 kd -> length = WORD64COUNT(sizeof(CARD32) * 4 + l);
         imsg = (iceMsg *) kd;
         break;
      }
      case Ktp_KR_OBSERVE : {
         Ktm_KRObserve ko;
         int id = sconn -> krlUniqueId;
         CARD32 specId = sconn -> lastKRSpecId;
         char *krlName = sconn -> lastInterestedInKRSpec;
         int l = strlen(krlName);
         /*
         ** Fill in the header
         */
         IceGetHeaderExtra(iceConn, sconn -> iceProtocol, opCode,
                           ICE_HEADER_SIZE, 0,
                           Ktm_KRObserveRec, ko, extra);
	 fillBuffer = extra;
         /*
         ** Then  the specific information pertaining to this packet :
         ** dest, kr, length of message and message
         */
	 WRITELONG(id, fillBuffer);
	 /* opLength neither */
	 WRITELONG(KT_NOT_RELEVANT, fillBuffer);
         WRITELONG(specId, fillBuffer);
         WRITELONG(l, fillBuffer);
         memcpy(fillBuffer, krlName, l); fillBuffer += l;
	 ko -> length = WORD64COUNT(sizeof(CARD32) * 4 + l);
         imsg = (iceMsg *) ko;

         break;
      }
      case Ktp_KR_UNOBSERVE : {
         Ktm_KRUnObserve ku;
         int id = sconn -> krlUniqueId;
         CARD32 specId = sconn -> lastKRSpecId;

         /*
         ** Fill in the header
         */
         IceGetHeaderExtra(iceConn, sconn -> iceProtocol, opCode,
                           ICE_HEADER_SIZE, 0,
                           Ktm_KRUnObserveRec, ku, extra);
	 fillBuffer = extra;
         /*
         ** Then  the specific information pertaining to this packet :
         ** dest, kr, length of message and message
         */
	 WRITELONG(id, fillBuffer);
	 /* opLength neither */
	 WRITELONG(KT_NOT_RELEVANT, fillBuffer);
         WRITELONG(specId, fillBuffer);
	 ku -> length = WORD64COUNT(sizeof(CARD32) * 3);
         imsg = (iceMsg *) ku;

         break;
      }
      case Ktp_KR_HANDLE : {
         Ktm_KRHandle kh;
         int id = sconn -> krlUniqueId;
         CARD32 specId = sconn -> lastKRSpecId;
         char *krSpec = sconn -> lastInterestedInKRSpec;
	 char *personalId = context -> directIdentity;
         int lspec = strlen(krSpec);
	 int lpersonal = strlen(personalId);

         /*
         ** Fill in the header
         */
         IceGetHeaderExtra(iceConn, sconn -> iceProtocol, opCode,
                           ICE_HEADER_SIZE, 0,
                           Ktm_KRHandleRec, kh, extra);
	 fillBuffer = extra;
         /*
         ** Then  the specific information pertaining to this packet :
         ** id, kr, length of message, length of name, message and name
         */
         WRITELONG(id, fillBuffer);
	 /* opLength irrelevant */
         WRITELONG(KT_NOT_RELEVANT, fillBuffer);
         WRITELONG(specId, fillBuffer);
         WRITELONG(lspec, fillBuffer);
         WRITELONG(lpersonal, fillBuffer);
         memcpy(fillBuffer, krSpec, lspec); fillBuffer += lspec;
         memcpy(fillBuffer, personalId, lpersonal); fillBuffer += lpersonal;
	 kh -> length = WORD64COUNT(sizeof(CARD32) * 5 + lspec + lpersonal);
         imsg = (iceMsg *) kh;

         break;
      }
      case Ktp_LAUNCH : {
         Ktm_Launch launch;
         int id = sconn -> krlUniqueId;
         CARD32 specId = sconn -> lastKRSpecId;
         char *krl = sconn -> lastDeclaredKRLName;
	 char *personalId = context -> directIdentity;
         int krlLength = strlen(krl);
	 int lpersonal = strlen(personalId);

         /*
         ** Fill in the header
         */
         IceGetHeaderExtra(iceConn, sconn -> iceProtocol, opCode,
                           ICE_HEADER_SIZE, 0,
                           Ktm_LaunchRec, launch, extra);
	 fillBuffer = extra;
         /*
         ** Then  the specific information pertaining to this packet
         */
         WRITELONG(id, fillBuffer);
	 /* opLength irrelevant */
         WRITELONG(KT_NOT_RELEVANT, fillBuffer);
         WRITELONG(krlLength, fillBuffer);
         memcpy(fillBuffer, krl, krlLength); fillBuffer += krlLength;
	 launch -> length = WORD64COUNT(sizeof(CARD32) * 3 + krlLength);
         imsg = (iceMsg *) launch;

         break;
      }
      default :
         fprintf(stderr, "(sendPacket) unknown id %d\n", opCode);
         imsg = NULL;
   }
   assert(imsg);

   IceWriteData(iceConn, (imsg -> length << 3), extra);
   IceFlush(iceConn);
}


/*
** This function should definitely be static, but is used in kt_peer.c.
** A 'protected' keyword would help...
*/
void
_kt_openConnection(char *targetLocation, char *pchProtocol, char *pchRelease,
                  IcePoVersionRec *callbackList,
                  IceConn *returnedConn, int *returnedProtocol)
{
   char error[128];
   int majVer, minVer;
   char *vendor, *release;
   int prot, status;
   IceConn iceConn;
   Kt_serverConnection result = NULL;

   prot = IceRegisterForProtocolSetup(pchProtocol, PVENDOR, pchRelease,
                               1, callbackList,    /* callbacks */
			       0,   /* auth count */
			       NULL,            /* auth  names */
			       NULL,  /* auth callbacks */
                               NULL   /* err procedure */
      );

   ASSERT(prot);
   iceConn = IceOpenConnection(targetLocation,
                           NULL, /* don't create a new connection */
                           False,    /* no authentication needed */
			       -1,     /* major opcode? */
                           128,
                           error);

   if (NULL == iceConn) {
      fprintf(stderr, "*** _kt_openConnection: couldn't open connection to '%s'\n", targetLocation);
      fprintf(stderr, "*** Ice error : '%s'\n", error);
      *returnedProtocol = -1;
      *returnedConn = NULL;
   }
   else {
      status = IceProtocolSetup(iceConn,
                                (int) prot,
                                NULL,       /* client data */
				False,    /* no auth needed */
                                & majVer,
                                & minVer,
                                & vendor,
                                & release,
                                128, error);
      if (IceProtocolSetupFailure == status) {
         fprintf(stderr, "***  %s,%d: protocol failure: %d '%s'\n",
		 __FILE__, __LINE__,
		 status, error);
      }
      else if (IceProtocolSetupIOError == status) {
         fprintf(stderr, "*** %s,%d: %setupIOError: %d '%s'\n",
		 __FILE__, __LINE__,
		 status, error);
      }
      else {
      }

      *returnedConn = iceConn;
      *returnedProtocol = prot;
   }
}

void
kt_addRequestToQueue(Kt_Context context, Kt_KRL krl, Kt_Message msg,
		     CARD32 requestId,
		     Kt_Disposition disposition,
		     Kt_CallbackHandlerMsg callback, void *userData,
		     Kt_requestStatus status)
{
   Kt_request request;

   NEW(request, Kt_requestRec);
   NEW(request -> krl, Kt_KRLRec);
   request -> krl -> krlId = krl -> krlId;
   request -> krl -> krlName = kt_strdup(krl -> krlName);
   ASSERT(msg -> opCode == Ktp_REQUEST);
   request -> msg = kt_MessageCopy(msg);
   request -> requestId = requestId;
   request -> disposition = disposition;
   request -> callback = callback;
   request -> userData = userData;
   request -> context = context;
   request -> status = status;
   DB_AddEntry(PendingRequests, request);
}


/**************************************************************************
 * Public
 **************************************************************************/

/*
**-----------------------------------------------------------------------
** kt_Init()
*/

static void
kt_freeDirectIdentity(Kt_directIdentity di)
{
    SAFE_FREE(di -> strPeer);
}

Kt_Context
kt_Init(Kt_CallbackAddInput addInput, void *addInputUserData,
	Kt_CallbackRemoveInput removeInput, void *removeInputUserData)
{
   Kt_Context result;
   char *identity;

   if (NULL == LastContext) {    /* first call, ok */
      NEW(result, Kt_ContextRec);
      memset(result, MAGIC_BYTE, sizeof(*result));
      result -> interestedInCallbacks =
         DB_NewDataBase(sizeof(Kt_interestedInCallbackRec),
                        freeInterestedInCallback);
      result -> directConn = DB_NewDataBase(sizeof(Kt_PeerConnectionRec), NULL);
      result -> directIdentities = DB_NewDataBase(sizeof(Kt_directIdentityRec),
						  kt_freeDirectIdentity);
      result -> directIdentity = NULL;
      _ServerConnections = DB_NewDataBase(sizeof(Kt_serverConnectionRec), NULL);
      result -> serial = 101;
      result -> serverConn = NULL;
      result -> directNewConn = NULL;
      result -> directCallback = NULL;
      result -> lastReceivedMessage = NULL;

      /*
      ** Set the *input() callbacks, default to ours.
      */
      if (NULL != addInput) {
	 result -> addInput = addInput;
	 result -> addInputUserData = addInputUserData;
	 result -> removeInput = removeInput;
	 result -> removeInputUserData = removeInputUserData;
      }
      else {
	 result -> addInput = kt_addInput;    /* kt_mainloop.c */
	 result -> addInputUserData = result;
	 result -> removeInput = kt_removeInput;
	 result -> removeInputUserData = result;
      }
      FD_ZERO(& result -> mainFdSet);

      /*
      ** Global variables.
      ** This will be fixed eventually (and hopefully)
      */
      KnownHandlers = DB_NewDataBase(sizeof(Kt_handlerRec),
                                     freeInterestedInCallback);
      PendingRequests =DB_NewDataBase(sizeof(Kt_requestRec),freePendingRequest);

      /*
      ** Setup the direct connection
      */
      kt_setupListenConnection(result, & identity);
      result -> directIdentity = identity;

      /*
      ** Install the error handler used if an ICE connection breaks
      */
      IceSetIOErrorHandler(kt_defaultIOHandler);

#if 0
      /*
      ** Setup the user-defined file descriptors
      */
      result -> userFileDescriptors =
	 DB_NewDataBase(sizeof(Kt_userFileDescriptorRec), NULL);
#endif

      /*
      ** kludge alert
      ** The contexts should be stored and indexed in order for an application
      ** to be able to call kt_Init() several times. Not possible for the
      ** moment: it will return the same over and over
      */
      LastContext = result;
   }
   else {
      result = LastContext;
      fprintf(stderr, "*** kt_Init() : called several times. Beware!\n");
   }
   return result;
}

/*
**-----------------------------------------------------------------------
** kt_HandleInput
*/
void
kt_HandleInput(Kt_Context context, fd_set fds)
{
   Kt_CallbackAddInput addInput = context -> addInput;
   Kt_CallbackRemoveInput removeInput = context -> removeInput;
   Kt_serverConnection sconn = context -> serverConn;
   IceConn newConn = context -> directNewConn;
   IceProcessMessagesStatus aStatus;
   DataBase db;

   /*
   ** ... from the server?
   */
   if (NULL != sconn)
      if (FD_ISSET(IceConnectionNumber(sconn -> iceConn), & fds)) {
	 aStatus = IceProcessMessages(sconn -> iceConn, NULL, NULL);
      }

   /*
   ** ... a direct connection request?
   */
   db = context -> directIdentities;
   DB_Rewind(db);
   while (! DB_EndOfDataBase(db)) {
      Kt_directIdentity di = DB_NextEntry(db);
      if (FD_ISSET(di -> directFd, & fds)) {
	 IceAcceptStatus status;
	 newConn = context -> directNewConn =
	    IceAcceptConnection(di -> directListenObj, & status);
	 /*
          ** Add it to the list of fd's we are watching
          */
	 DP(printf("direct connection request: %d\n", IceConnectionNumber(newConn)));
	 (*addInput)(IceConnectionNumber(newConn), context -> addInputUserData);
      
	 ASSERT(newConn);
	 IceProcessMessages(newConn, NULL, NULL);
      }
   }

   /*
   ** ...or data on the dangling connection
   */
   if (NULL != newConn &&
       FD_ISSET(IceConnectionNumber(context -> directNewConn), & fds)) {
      IceProcessMessages(newConn, NULL, NULL);
      if (IceConnectAccepted == IceConnectionStatus(newConn)) {
	 Kt_PeerConnection dc;
	 NEW(dc, Kt_PeerConnectionRec);
	 memset(dc, 0, sizeof(dc));
	 dc -> iceConn = newConn;
	 DB_AddEntry(context -> directConn, dc);
/*
	 (*removeInput)(IceConnectionNumber(newConn),
			context -> removeInputUserData);
*/
	 newConn = context -> directNewConn = NULL;
      }
   }

   /*
   ** ... or an actual direct connection
   */
   db = context -> directConn;
   DB_Rewind(db);
   while (! DB_EndOfDataBase(db)) {
      Kt_PeerConnection dc;
      dc = (Kt_PeerConnection) DB_NextEntry(db);
      if (FD_ISSET(IceConnectionNumber(dc -> iceConn), & fds)) {
	 IceProcessMessages(dc -> iceConn, NULL, NULL);
      }
   }

  if (NULL != context -> lastReceivedMessage) {
     kt_DispatchMessage(context, context -> lastReceivedMessage);
     kt_MessageFree(context -> lastReceivedMessage);
     context -> lastReceivedMessage = NULL;
  }
}

/*
**-----------------------------------------------------------------------
** kt_UnInit
*/
void
kt_UnInit(Kt_Context context)
{
   Kt_serverConnection sc = context -> serverConn;

   if (0 != sc) {
      SAFE_FREE(sc -> lastDeclaredKRLName);
      SAFE_FREE(sc -> lastInterestedInKRSpec);
      SAFE_FREE(sc);
   }
   
   DB_DestroyDataBase(context -> interestedInCallbacks);
   DB_DestroyDataBase(context -> directConn);
   DB_DestroyDataBase(context -> directIdentities);
/*
   DB_DestroyDataBase(context -> userFileDescriptors);
*/
   kt_MessageFree(context -> lastReceivedMessage);
   SAFE_FREE(context -> lastKRL);
   SAFE_FREE(context -> directIdentity);
   SAFE_FREE(context);
}
/*
**-----------------------------------------------------------------------
** kt_LocateServer
*/

#ifndef NO_RPC
static bool_t
kts_xdrStringDecode(XDR *xdrs, char **string)
{
   bool_t result;
   result = xdr_string(xdrs, string, 128);
   return result;
}
#endif /* NO_RPC */

#ifndef NO_X
int
kt_xErrorHandler(Display *dis, XErrorEvent *error)
{
    printf("error handler called\n");
}
#endif /* NO_X */

void
kt_LocateServer(char ***retServers, int *retN)
{
   char host[128], *phost, tmpBuf[128];
   char *servers[24];
   int port, nServers = 0, i;
   FILE *f;

   /*
   ** See it the user supplied a KTSERV_HOST to know which portmapper
   ** we must consult. Take hostname as default.
   */
   phost = getenv(VARIABLE_KTSERV_HOST);
   if (NULL == phost)
      gethostname(host, 128);
   else
      strcpy(host, phost);

#ifndef NO_RPC
   /*
   ** Look for a server in the portmapper's tables
   */
   {   /* RPC section */
      struct timeval val;
      char *retLocation = NULL;   /* XDR will allocate for me */
      unsigned int retPort;

      CLIENT *clnt;
      struct timeval tv;
      enum clnt_stat rpcStat;
      tv.tv_sec = 1;
      tv.tv_usec = 0;
      clnt = clnt_create(host, KTS_PROG_NUM, KTS_VERS_NUM, "tcp");
      if (NULL != clnt) {
	 rpcStat = clnt_call(clnt, GETPORT,   /* proc number */
			     xdr_void, ((caddr_t) 0),  /* in parameter */
			     kts_xdrStringDecode, ((caddr_t) & retLocation),/*out parameter */
			     tv);
#if 0
			     xdr_u_int, & retPort,
#endif
	 if (RPC_SUCCESS != rpcStat) {
	    printf("*** portmapper warning : couldn't perform GETPORT\n");
	 }
	 else {
	    servers[nServers] = (char *) kt_strdup(retLocation);
	    free(retLocation);
	    nServers++;
	 }
	 clnt_destroy(clnt);
      }
      else {
	 printf("*** portmapper warning : couldn't locate ktserv service\n");
      }
      
   }  /* RPC section */
   

#endif /* NO_RPC */

#ifndef NO_X
   /*
   ** Look for a server on DISPLAY
   */
   {    /* DISPLAY section */
      char *displayName;
      Display *display;
      if (NULL != (displayName = (char *) getenv("DISPLAY"))) {
	 if (NULL != (display = XOpenDisplay(displayName))) {
	    Window serverWindow;
	    Atom atomReturn, atom;
	    int formatReturn;
	    unsigned long n, bytes;
	    unsigned char *propResult;
	    int success;
	    int (*oldHandler)();

	    /*
	    ** Install new error handler
	    */
	    oldHandler = XSetErrorHandler(kt_xErrorHandler);

	    /*
	    ** Try to find the properties
	    */
	    atom = XInternAtom(display, PROPERTY_WINDOW, True);
	    success = XGetWindowProperty(display,
					 DefaultRootWindow(display),
					 atom,
					 0L,       /* offset */
					 1,    /* length in 32 bits multiple to be retrieved */
					 False,    /* delete */
					 XA_INTEGER,
					 & atomReturn,
					 & formatReturn,
					 & n,
					 & bytes,
					 & propResult
	       );
	    if (success != Success) {
	       fprintf(stderr, "ktserv: couldn't find property %s on server %s\n",
		       display, displayName);
	    }
	    serverWindow = * (Window *) propResult; n++;
	    atom = XInternAtom(display, PROPERTY_KTSERV_COORD, True);
	    success = XGetWindowProperty(display,
					 serverWindow,
					 atom,
					 0L,       /* offset */
					 64,      /* length in 32 bits multiple to be retrieved */
					 False,    /* delete */
					 XA_STRING,
					 & atomReturn,
					 & formatReturn,
					 & n,
					 & bytes,
					 & propResult
	       );

	    if (success != Success) {
	       fprintf(stderr, "kt_LocateServer(): couldn't find property %s on server %s\n",
		       PROPERTY_KTSERV_COORD, display);
	    }
	    if (NULL != propResult) {
	       servers[nServers] = (char *) propResult; nServers++;
	       DP(printf("kt_LocateServer() : found server '%s'\n", servers[nServers-1]));
	    }
	    XSetErrorHandler(oldHandler);
	    XCloseDisplay(display);
	 }
      }
   }   /* DISPLAY section */
#endif /* NO_X */

   /*
   ** Look for a server on file
   */
   f = fopen(FILE_KTSERV_PORT, "r");
   if (NULL != f) {
      fscanf(f, "%d", & port);
      fclose(f);
      sprintf(tmpBuf, "tcp/%s:%d", host, port);
      servers[nServers] = (char *) kt_strdup(tmpBuf);
      nServers++;
   }

   ASSERT(nServers < 24);

   *retServers = (char **) malloc(sizeof(char *) * nServers);
   *retN = nServers;
   for (i=0; i < nServers; i++) {
      (*retServers)[i] = kt_strdup(servers[i]);
      free(servers[i]);
   }
}

/*
**-----------------------------------------------------------------------
** kt_OpenServerConnection
*/
Kt_Status
kt_OpenServerConnection(Kt_Context context, char *serverLocation, char *identity)
{
   Kt_CallbackAddInput addInput = context -> addInput;
   Kt_Status result = KT_OK;
   IceConn conn;
   int status;
   int majVer, minVer;
   char *kt_identity;
   int prot;
   Kt_serverConnection serverConn = NULL;
   IcePoVersionRec callbackList[] = {
      { 1, 0, (IcePoProcessMsgProc) kt_processServerConnection}
   };

   _kt_openConnection(serverLocation, PNAME, PRELEASE,
                     callbackList, & conn, & prot);

   NEW(serverConn, Kt_serverConnectionRec);
   memset(serverConn, 0, sizeof(*serverConn));

   if (NULL == conn) {
      serverConn = NULL;
      result = KT_ERR;
   }
   else {
      serverConn -> iceConn = conn;
      serverConn -> iceProtocol = prot;
      serverConn -> krlUniqueId = 1;
      serverConn -> lastDeclaredKRLId = 0;
      kt_sendPacket(context, serverConn, Ktp_NEW_CLIENT);
      kt_addConnection(_ServerConnections, serverConn);
      context -> serverConn = serverConn;

      /*
      ** Add this file descriptor to the list of those we watch
      */
      DP(printf("addinput openserverconnection: %d\n", IceConnectionNumber(serverConn -> iceConn)));
      (*addInput)(IceConnectionNumber(serverConn -> iceConn),
		  context -> addInputUserData);
   }


   return result;
}


/*
**-----------------------------------------------------------------------
** kt_CloseConnection
*/
Kt_Status
kt_CloseConnection(Kt_Context context)
{
   ASSERT(0);
   freeServerConn(context -> serverConn);
   return KT_OK;
}

/*
**-----------------------------------------------------------------------
** kt_ReturnIceServerConnection
*/
IceConn
kt_ReturnIceServerConnection(Kt_Context context)
{
   ASSERT(context -> serverConn);
   return context -> serverConn -> iceConn;
}

/*
**-----------------------------------------------------------------------
** kt_ReturnIceServerProtocol
*/
int
kt_ReturnIceServerProtocol(Kt_Context context)
{
   ASSERT(context -> serverConn);
   return context -> serverConn -> iceProtocol;
}

/*
**-----------------------------------------------------------------------
** kt_MainLoop
*/
void
kt_MainLoop(Kt_Context context)
{
   Kt_Message msg;
   Bool dispatched;

   while (1) {
      kt_NextMessage(context, & msg);
      dispatched = kt_DispatchMessage(context, msg);
      if (False == dispatched)
	 DP(printf("mainloop, not dispatched\n"));
   }
/*
   while (1)
      kt_mainLoop(context);
*/
}

/*
**-----------------------------------------------------------------------
** kt_NextMessage
*/
void
kt_NextMessage(Kt_Context context, Kt_Message *msg)
{
    kt_MessageFree(context -> lastReceivedMessage);
    context -> lastReceivedMessage = NULL;
    while (NULL == context -> lastReceivedMessage)
	kt_mainLoop(context);
    *msg = context -> lastReceivedMessage;
}

/*
**-----------------------------------------------------------------------
** kt_DispatchMessage
*/
Bool
kt_DispatchMessage(Kt_Context context, Kt_Message msg)
{
   Bool result = False;
   Bool bCallDefault = True;
   int opCode = msg -> opCode;
   DataBase db;
   Kt_CallbackPeerMsg cbPeerDefault = context -> directCallback;

   /*
   ** This boolean is not really necessary, I could have put
   ** all the opcodes in the same switch().
   ** But it improves readability to ahve two distinct cases
   ** for server messages and peer messages
   */
   if (True == context -> bLastMessageFromServer){/* last message from server*/
      switch(opCode) {
	 case Ktp_MESSAGE_FORWARDED : {
	    result = kt_runCallbacks(msg);
	    DP(printf("runCallbacks returned %d\n", result));
	    break;
	 }
	 default : {
	    DP(printf("*** unknown message in DispatchMessage %d\n", opCode));
	 }
      }
   }
   else {     /* last message was from a peer */
      Kt_CallbackHandlerMsg cbHandler;

      switch(opCode) {
	 /*
	 ** See if this client registered a callback to handle
	 ** incoming direct messages. Otherwise, do nothing
	 */
	 case Ktp_PEER_TO_PEER : {
	    if (NULL != cbPeerDefault) {
	       (*cbPeerDefault)(context, context -> directClosure, msg);
	       result = True;
	    }
	    break;
	 }

         /*
	 ** Run the list of callbacks registered with a kt_MessageSendWithReply()
	 ** and see if we can match this request with one of them. If we
	 ** can't, something is really wrong : the server thinks we are
	 ** a handler for this request and we did not. I should find a way to
	 ** bring them back to synch. Right now, this will merely raise
	 ** an exception :-)
	 */
	 case Ktp_REQUEST : {
	    db = context -> interestedInCallbacks;
	    DB_Rewind(db);
	    while (! DB_EndOfDataBase(db)) {
	       Kt_interestedInCallback ic = DB_NextEntry(db);
	       if (False == ic -> isObserve &&
		   True == kt_regexpMatch(ic -> krs,
					  msg -> msgType.request.krl)) {
		  bCallDefault = False;
		  cbHandler = ic -> callback;
		  (*cbHandler)(context, ic -> userData, msg,
			       ic -> krSpecId);
		  result = True;
		  break;
	       }
	    }
	    ASSERT(True == result);
	    break;
	 }

	 case Ktp_REPLY_TO_REQUEST : {
	    Kt_request req;
/*
	    CARD32 id = kt_decodeULONG((char *)
				& msg -> msgType.replyToRequest.requestId);
*/
	    CARD32 id = msg -> msgType.replyToRequest.requestId;

	    /*
	    ** Run through the list of pending requests to see if this reply
	    ** really belongs to us. To achieve this, a PendingRequest
	    ** must be in SENT status and have the same requestId. Same
	    ** remarks regarding the exception in previous case here.
	    */
	    db = PendingRequests;
	    DB_Rewind(db);
	    while (! DB_EndOfDataBase(db)) {
	       Kt_CallbackHandlerMsg cbHandler;
	       req = DB_NextEntry(db);
	       if (KT_SENT == req -> status && req -> requestId == id) {
		  cbHandler = (Kt_CallbackHandlerMsg) req -> callback;
		  (*cbHandler)(context, req -> userData, msg, id);
		  req -> status = KT_PROCESSED;
		  bCallDefault = False;
		  result = True;
		  break;
	       }
	    }
	    ASSERT(True == result);

	    /*
	    ** I can't remove the requests from the
	    ** database as they are being processed, this would
	    ** result in an inconsistent PendingRequests database.
	    ** The first step has been to mark them as PROCESSED, now I walk
	    ** the database again and remove them.
	    */
	    DB_Rewind(db);
	    while (! DB_EndOfDataBase(db)) {
	       req = DB_NextEntry(db);
	       if (KT_PROCESSED == req -> status)
		  DB_RemoveEntry(db, req);
	    }
	    break;
	 }
      } /* switch */

      /*
      ** Note : msg must be filled by
      ** - a call to kt_messageDecode()
      ** - an assignment of the field krlId
      ** at this point, or chaos will ensue
      */

/*
      if (True == bCallDefault && NULL != cbDefault) {
	 (*cbDefault)(context,
		      context -> directDefaultClosure,
		      msg);
	 result = True;
      }
*/
   }

   return result;
}

/*
**-----------------------------------------------------------------------
** kt_SetServerErrorHandler
*/
void
kt_SetServerErrorHandler(Kt_Context context, void *handler)
{
   ASSERT(0);
}

/*
**-----------------------------------------------------------------------
** kt_AddUserFileDescriptor
*/
#if 0
void
kt_AddUserFileDescriptor(Kt_Context context, int fd,
			 void (*callback)(), void *userData)
{
   Kt_userFileDescriptor entry;

   NEW(entry, Kt_userFileDescriptorRec);
   entry -> fd = fd;
   entry -> f = callback;
   entry -> userData = userData;
   DB_AddEntry(context -> userFileDescriptors, entry);
}
#endif

/*
**-----------------------------------------------------------------------
** kt_ReturnIdentity
*/
char *
kt_ReturnIdentity(Kt_Context context)
{
   return context -> directIdentity;
}

/*
void
IceReadData(IceConn _iceConn, size_t _bytes, char *_pData)
{
   D(printf("** icereaddata: %d bytes\n", _bytes));
   _IceRead (_iceConn, (unsigned long) (_bytes), (char *) _pData);
}
*/

