/* Copyright (C) 1994 Groupe BULL. See file COPYRIGHT for details */
/*
 *
 * $Id: kt_encoding.c,v 1.2 1994/12/15 13:38:37 beust Exp $
 */

#include <stdio.h>
#include <assert.h>
#include "kt_lib.h"
#include "kt_protocolP.h"
#include "kt_typesP.h"
#include "database.h"

/*
** A raw message is
**
** - ICE Header (2 CARD32)
** - destination krlid  (1 CARD32)
** - operation length (1 CARD32)
**
** - message-related additional information (variable)
** - argument count (1 CARD32)
** - encoded arguments (variable)
**
** - operation (length CARD8)
***/


char *
kt_messageStartPrepare2(IceConn iceConn, char *buffer,
			CARD32 krlId, size_t opLength,
			int *newSize)
{
   char *result;
   char *bufferPtr = buffer;

   unsigned long ulong;

   /*
   ** At this point, IceGetHeaderExtra has been called,
   ** so initial size = sizeof(iceMsg)
   */
   *newSize = ICE_HEADER_SIZE;

   /*
   ** Here starts the encoding of the message itself.
   ** CAREFUL! Don't change anything below without modifying
   ** kt_messageDecode (kt_message.c) appropriately. Most of all,
   ** the order in which the encoding is done is *critical*.
   */

   /*
   ** operation length, dest and krlId
   */

   ulong = krlId; WRITELONG(ulong, bufferPtr);
   ulong = opLength; WRITELONG(ulong, bufferPtr);
   /*
   ** Initialize the size of the Ice message in units of 8-bytes
   */
   *newSize += (bufferPtr - buffer);

   IceWriteData(iceConn, (bufferPtr - buffer), buffer);
   result = bufferPtr;
   return result;
}

size_t
kt_messageEndPrepare2(IceConn iceConn, char *buffer,
		      DataBase arguments, char *operation,
		      size_t *mSize)
{
   char *bufferPtr = buffer;
   CARD32 ulong;
   DataBase db = arguments;
   Kt_messageArg marg;
   size_t result = 0;
   size_t msgSize = *mSize;

   /*
   ** Determine the size
   ** First, compute the size for all the arguments
   */
   DB_Rewind(db);
   while (! DB_EndOfDataBase(db)) {
      marg = (Kt_messageArg) DB_NextEntry(db);
      msgSize += 8;   /* type and length of this argument */
      switch (marg -> type) {
	 case KT_STRING :
	    msgSize += strlen(marg -> content.s.string);
	    break;
	 case KT_BYTES :
	    msgSize += marg -> content.b.length;
	    break;
	 default :
	    ASSERT(0);
      }
      msgSize += PAD32(msgSize);
   }
   /*
   ** Then compute the size held by the operation field
   */
   msgSize += sizeof(CARD32);
   msgSize += strlen(operation);
   msgSize += PAD32(msgSize);
   

   /*
   ** Encode the number of arguments
   */
   ulong = DB_Count(db); WRITELONG(ulong, bufferPtr);
   msgSize += sizeof(ulong);
   /*
   ** And now each argument in turn
   */
   DB_Rewind(db);
   while (! DB_EndOfDataBase(db)) {
      marg = DB_NextEntry(db);
      switch (marg -> type) {
	 case KT_STRING : {
	    char *string = marg -> content.s.string;
	    size_t l = strlen(string);

	    ulong = (unsigned long) KT_STRING;
	    WRITELONG(ulong, bufferPtr);
	    ulong = (unsigned long) strlen(string);
	    WRITELONG(ulong, bufferPtr);
	    memcpy(bufferPtr, string, strlen(string)); bufferPtr += l;
	    memset(bufferPtr, MAGIC_BYTE, PAD32(l)); bufferPtr += PAD32(l);
	    break;
         }
	 case KT_BYTES : {
	    size_t size = marg -> content.b.length;
	    ulong = (unsigned long) KT_BYTES;
	    WRITELONG(ulong, bufferPtr);
	    ulong = (unsigned long) size;
	    WRITELONG(ulong, bufferPtr);
	    memcpy(bufferPtr, marg -> content.b.bytes, size); bufferPtr += size;
	    memset(bufferPtr, MAGIC_BYTE, PAD32(size)); bufferPtr += PAD32(size);
	 }
	    
	    break;
	 default :
	    ASSERT(0);
      }
   }

   /*
    ** operation...
    */
   ulong = strlen(operation);
   memcpy(bufferPtr, operation, ulong); bufferPtr += ulong;

   *mSize += (bufferPtr - buffer);
   result = *mSize - ICE_HEADER_SIZE + PAD64(*mSize);
   result = WORD64COUNT(result);

   IceWriteData(iceConn, result << 3, buffer);
   return result;
   
}
/*
** Encode data in the buffer according to the contents of the message.
** Fill the following :
** - dest
** - id
** - count of arguments
** - arguments
**
** Besides, it will initialize msg -> size and msg -> bufferPointer
** and will return the place in the buffer where the rest of the encoding
** should be stored.
*/
char *
kt_messageStartPrepare(IceConn iceConn, Kt_Message msg, char *buffer)
{
   char *result;
   char *bufferPtr = buffer;

   DataBase db = msg -> arguments;
   Kt_messageArg marg;
   unsigned long ulong;
   size_t argSize = 0;    /* size of current argument */

   ASSERT(MAGIC_BYTE != msg -> krlId);
   /*
   ** At this point, IceGetHeaderExtra has been called,
   ** so initial size = sizeof(iceMsg)
   */
   msg -> size = ICE_HEADER_SIZE;

   /*
   ** Here starts the encoding of the message itself.
   ** CAREFUL! Don't change anything below without modifying
   ** kt_messageDecode (kt_message.c) appropriately. Most of all,
   ** the order in which the encoding is done is *critical*.
   */

   /*
   ** operation length, dest and krlId
   */

/*
   ulong = strlen(msg -> operation); kt_encodeULONG(ulong, buffer); buffer += sizeof(ulong);
*/
   ulong = msg -> krlId; WRITELONG(ulong, bufferPtr);
   ulong = strlen(msg -> operation); WRITELONG(ulong, bufferPtr);
   /*
   ** Initialize the size of the Ice message in units of 8-bytes
   */
   msg -> size += bufferPtr - buffer;
   msg -> bufferPointer = bufferPtr;

   IceWriteData(iceConn, (bufferPtr - buffer), buffer);
   result = bufferPtr;
   return result;


}

/*
** Encode data in the buffer according to the contents of the message.
** Fill the ICE buffer with the following :
** - length of operation
** - operation
** - msg -> buffer
*/
size_t
kt_messageEndPrepare(IceConn iceConn, Kt_Message msg)
{
   char *bufferPtr, *buffer;
   CARD32 ulong;
   DataBase db = msg -> arguments;
   Kt_messageArg marg;
   int msgSize = msg -> size;
   size_t result = 0;


   /*
   ** Determine the size
   ** First, compute the size for all the arguments
   */
   DB_Rewind(db);
   while (! DB_EndOfDataBase(db)) {
      marg = (Kt_messageArg) DB_NextEntry(db);
      msgSize += 8;   /* type and length of this argument */
      switch (marg -> type) {
	 case KT_STRING :
	    msgSize += strlen(marg -> content.s.string);
	    break;
	 case KT_BYTES :
	    msgSize += marg -> content.b.length;
	    break;
	 default :
	    ASSERT(0);
      }
      msgSize += PAD32(msgSize);
   }
   /*
   ** Then compute the size held by the operation field
   */
   msgSize += sizeof(CARD32);
   msgSize += strlen(msg -> operation);
   msgSize += PAD32(msgSize);

   /*
   ** Here I discard the buffer argument. I should remove
   ** it eventually...
   */
   buffer = (char *) malloc(msgSize);
   bufferPtr = buffer;

   /*
   ** Encode the number of arguments
   */
   ulong = DB_Count(db); WRITELONG(ulong, bufferPtr);
   msgSize += sizeof(ulong);
   /*
   ** And now each argument in turn
   */
   DB_Rewind(db);
   while (! DB_EndOfDataBase(db)) {
      marg = DB_NextEntry(db);
      switch (marg -> type) {
	 case KT_STRING : {
	    char *string = marg -> content.s.string;
	    size_t l = strlen(string);

	    ulong = (unsigned long) KT_STRING;
	    WRITELONG(ulong, bufferPtr);
	    ulong = (unsigned long) strlen(string);
	    WRITELONG(ulong, bufferPtr);
	    memcpy(bufferPtr, string, strlen(string)); bufferPtr += l;
	    memset(bufferPtr, MAGIC_BYTE, PAD32(l)); bufferPtr += PAD32(l);
	    break;
         }
	 case KT_BYTES : {
	    size_t size = marg -> content.b.length;
	    ulong = (unsigned long) KT_BYTES;
	    WRITELONG(ulong, bufferPtr);
	    ulong = (unsigned long) size;
	    WRITELONG(ulong, bufferPtr);
	    memcpy(bufferPtr, marg -> content.b.bytes, size); bufferPtr += size;
	    memset(bufferPtr, MAGIC_BYTE, PAD32(size)); bufferPtr += PAD32(size);
	 }
	    
	    break;
	 default :
	    ASSERT(0);
      }
   }

   /*
    ** operation...
    */

   ulong = strlen(msg -> operation);
   memcpy(bufferPtr, msg -> operation, ulong); bufferPtr += ulong;

   msg -> size += (bufferPtr - buffer);
   result = msg -> size - ICE_HEADER_SIZE + PAD64(msg -> size);
   result = WORD64COUNT(result);

   IceWriteData(iceConn, (bufferPtr - buffer), buffer);
   IceWritePad(iceConn, PAD64(msg -> size));
   free(buffer);
   return result;
}


Kt_Message
kt_messageStartDecode(Kt_messageBuffer mb)
{
   DataBase db;
   Kt_Message result;
   int n;
   CARD32 krlId;
   size_t opL;
   unsigned long argCount, argType, argLength;
   char *p = (char *) & mb -> bytes;


   /*
   ** Initialize return value
   */
   result = kt_MessageNew();
   result -> arguments =
      DB_NewDataBase(sizeof(Kt_messageArgRec), NULL);  /*@@ leak */
   db = result -> arguments;


   /*
   ** Retrieve the krlId and operation length
   */


   result -> buffer = NULL;
   result -> isBufferInitialized = False;
   result -> size = 0;
   result -> bufferPointer = p;
   result -> opLength = mb -> opLength;
   result -> krlId = mb -> krlId;
   result -> opCode = mb -> minorOpcode;

   return result;

}

/*
** mb -> bufferPointer must point on the argument count at this point
*/
void
kt_messageEndDecode(Kt_Message msg)
{
   char *p = msg -> bufferPointer;
   size_t opLength;
   unsigned long argCount, argType, argLength, n;
   DataBase db = msg -> arguments;

   /*
   ** Here starts the decoding of the message itself.
   ** CAREFUL! Don't change anything here without modifying
   ** kt_messagePrepare (kt_lib.c) appropriately. Most of all,
   ** the order in which the decoding is done is *critical*.
   */

   /*
   ** Retrieve the argument count
   */
   READLONG(argCount, p);
   n = argCount;
   ASSERT(n >= 0 && n < 10);   /* safety check */

   /*
   ** Retrieve each argument in turn
   */
   while (n-- > 0) {
      Kt_messageType type;
      Kt_messageArg marg;
      char *buf;

      NEW(marg, Kt_messageArgRec);
      READLONG(argType, p);
      READLONG(argLength, p);
      type = (Kt_messageType) argType;
      ASSERT(KT_STRING == type || KT_BYTES == type);
      switch(type) {
	 case KT_STRING : {
	    buf = (char *) malloc(argLength + 1);
	    memcpy(buf, p, argLength); p += argLength;
	    p += PAD32(argLength);
	    buf[argLength] = '\0';
	    marg -> type = KT_STRING;
	    marg -> content.s.string = kt_strdup(buf);
	    DB_AddEntry(db, marg);
	    SAFE_FREE(buf);
	    break;
	 }
	 case KT_BYTES : {
	    buf = (char *) malloc(argLength);
	    memcpy(buf, p, argLength); p += argLength;
	    p += PAD32(argLength);
	    marg -> type = KT_BYTES;
	    marg -> content.b.length = argLength;
	    marg -> content.b.bytes = buf;
	    DB_AddEntry(db, marg);
	    break;
	 }
	 default :
	    ASSERT(0);
	    break;
      }
   }

   /*
   ** Convert some values before using them
   */
   DP(printf("decoding opLength from %x", msg->opLength));
   msg -> opLength = kt_decodeULONG(& msg -> opLength);
   DP(printf("to %x\n", msg->opLength));
   msg -> krlId = kt_decodeULONG(& msg -> krlId);

   msg -> operation = (char *) malloc(msg -> opLength + 1);
   memcpy(msg -> operation, p, msg -> opLength);
   msg -> operation[msg -> opLength] = '\0';
   p += msg -> opLength;
}
