//
// Copyright (C) 1994-1995 Harry Danilevsky and Andrew Renalds
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish and distribute copies of the 
// Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESSED OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// HARRY DANILEVSKY AND ANDREW RENALDS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

#include <DD++/DD.H>

XContext DropSite::contextId = 0;

DropSite::DropSite(Widget aWidget, unsigned char ops, int animationPixmapStyle_,
		   char *animationPixmapName, char *animationMaskPixmapName) :
    w(aWidget),
    operations(ops),
    animationPixmapStyle(animationPixmapStyle_),
    last(NULL)
{
  if( !contextId )
      contextId = XUniqueContext();

  // Associate "this" with the widget, so we can access it in Motif callbacks
  XSaveContext( XtDisplay( w ), 
		(XID) w,
		contextId,
		(char *) this );
  
  animationPixmap = new XPixmap( w, animationPixmapName );
  animationMaskPixmap = new XPixmap( w, animationMaskPixmapName );
}


DropSite::~DropSite()
{
  // free pixmaps
  delete animationPixmap;
  delete animationMaskPixmap;

  // clear target list
  if (last)
    {
      Node *node = last->next;
      last->next = NULL;
      last = NULL;
      while (node)
	{
	  Node *nxt = node->next;
	  delete node;
	  node = nxt;
	}
    }

  // unregister site via XmDropSiteUnregister
  if( w )  XmDropSiteUnregister(w);

  // delete context
  XDeleteContext( XtDisplay( w ),
		  (XID) w,
		  contextId );
}

DropSite& DropSite::append(DropTargetPtr dtP)
{
  if (!dtP)
    return *this;
  if (w)
    {
      dtP->dPtr->atom =  XmInternAtom( XtDisplay(w), (char *)dtP->dPtr->atomName(), False );
      // append new list item
      Node *new_last = new Node;
      if (last)
	{
	  new_last->next = last->next;
	  last = last->next = new_last;
	}
      else
	{
	  new_last->next = last = new_last;
	}
      last->data = dtP;
	  
      // How many targets registered for this site
      int nTargets = 0;
      Node *node = last;
      if (node) do { ++nTargets; node = node->next; } while (node != last);

      Atom *importList = new Atom[nTargets];
      // Fill in importList with atoms
      int n=0;
      if (last)
	{
	  node = last;
	  do
	    {
	      importList[n++] = node->next->data->dPtr->atom;
	      node = node->next;
	    }
	  while (node != last);
	}

      Arg args[10];
      n = 0;
      XtSetArg( args[n], XmNimportTargets, importList );  n++;
      XtSetArg( args[n], XmNnumImportTargets, nTargets );  n++;
      XtSetArg( args[n], XmNdropSiteOperations, operations );  n++;
      XtSetArg( args[n], XmNdropProc, ((XtCallbackProc)DropSite::dropCB) );  n++;

      XtSetArg( args[n], XmNanimationStyle, animationPixmapStyle );  n++;
      if( animationPixmapStyle == XmDRAG_UNDER_PIXMAP )
      {
        XtSetArg( args[n], XmNanimationPixmapDepth, animationPixmap->getDepth() ); n++;
        if( animationPixmap->getPixmap() )
        {
          XtSetArg( args[n], XmNanimationPixmap, animationPixmap->getPixmap() );
          n++;
        }
        if( animationMaskPixmap->getPixmap() )
        {
          XtSetArg( args[n], XmNanimationMask, animationMaskPixmap->getPixmap() );
          n++;
        }
      }

      // Re-register drop site; If I don't call XmDropSiteUnregister
      // I'll get the message "Can't register widget as a drop site more than once"
      XmDropSiteUnregister(w);
      XmDropSiteRegister( w, args, n );
      delete importList;
    }
  return *this;
}

DropSite& DropSite::remove(DropTargetPtr dtP)
{
  Node *n = last;
  if (n)
    do 
      {
	Node* f = n->next;
	if (f->data == dtP)
	  {
	    n->next = f->next;
	    if (f == last)
	      last = n;
	    delete f;
	    break;
	  }
	n = n->next;
      }
    while (n != last);

  return *this;
}


void DropSite::dropCB( Widget w, XtPointer client_data, 
		  XtPointer call_data )
{
  DropSite *dsPtr;

  XFindContext( XtDisplay( w ), 
		(XID) w,
		contextId,
		(XPointer *)&dsPtr );
  if (dsPtr)
    dsPtr->handleDrop(client_data, call_data);
}

void DropSite::transferCB( Widget, XtPointer client_data, Atom*, 
			Atom *type, XtPointer value, 
			unsigned long *length, int )
{
  DropSite *dsPtr = (DropSite*)client_data;
  // Match atom type with those of individual targets.
  int i=0;

  if (dsPtr->last)
    {
       Node *node = dsPtr->last;
       do
	 {
	   DropTargetPtr dtP = node->data;
	   if (*type == dtP->dPtr->atom)
	     {
#ifdef DD_USE_XDR
	       DDistream getstr((char *)value, *length);
#else	  
	       istrstream getstr((char *)value, *length);
#endif	  
	       dtP->dPtr->restoreFrom(getstr, *length);
	       dtP->receive(dsPtr->w);
	       break;
	     }
	   node = node->next;
	 } while (node != dsPtr->last);
    }
}


void DropSite::handleDrop( XtPointer, XtPointer call_data )
{
   Arg args[10];
   XmDropTransferEntryRec transferList[1];
   XmDropProcCallback DropData = (XmDropProcCallback)call_data;

   int n = 0;
   Atom *aPtr;

   XtVaGetValues(DropData->dragContext, XmNexportTargets, &aPtr, NULL);

   // Can we match the dropped object to the atom types of DropTarget list

   int atomMatch = False;
   int matchTarget = -1;

   if (last)
     {
       Node *node = last;
       do
	 {
	   if (node->data->dPtr->atom == *aPtr)
	     {
	       atomMatch = True;
	       break;
	     }
	   node = node->next;
	 } while (node != last);
     }

   if( DropData->dropAction != XmDROP ||
       !(DropData->operation & operations) ||
       !atomMatch)
   {
      XtSetArg( args[n], XmNtransferStatus, XmTRANSFER_FAILURE );  n++;
   }
   else
   {
     int nTargets = 1;
     // We have already identified the target atom
     transferList[0].target = *aPtr;
     // Have the 'this' pointer passed to transferProc as client data
     transferList[0].client_data = (XtPointer)this;

     XtSetArg( args[n], XmNdropTransfers, transferList );  n++;
     XtSetArg( args[n], XmNnumDropTransfers, nTargets );  n++;
     XtSetArg( args[n], XmNtransferProc,
	       ((XtSelectionCallbackProc)DropSite::transferCB) );  n++;
   }

   XmDropTransferStart( DropData->dragContext, args, n );
}
