//
// Copyright (C) 1994-1995 Harry Danilevsky and Andrew Renalds
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to
// deal in the Software without restriction, including without limitation the
// rights to use, copy, modify, merge, publish and distribute copies of the 
// Software, and to permit persons to whom the Software is furnished to do so,
// subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESSED OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// HARRY DANILEVSKY AND ANDREW RENALDS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

#include <DD++/DD.H>

XContext DragSite::contextId = 0;
char *DragSite::DefaultActionName = "StartDrag";

static struct DragOps {
  int  op;
  char *tag;
} dragOps[] = {
  { XmDROP_NOOP, "DROP_NOOP" },
  { XmDROP_LINK, "DROP_LINK" },
  { XmDROP_MOVE, "DROP_MOVE" },
  { XmDROP_COPY, "DROP_COPY" }
};

const char* DragSite::dragKey(int opCode) const
{
  char *keyCode = NULL;
  switch (opCode)
    {
    case XmDROP_NOOP:
      break;
    case XmDROP_MOVE:
      keyCode = "Shift <Btn2Down>";
      break;
    case XmDROP_COPY:
      keyCode = "<Btn2Down>";
      break;
    case XmDROP_LINK:
      keyCode = "Shift Ctrl <Btn2Down>";
      break;
    default:
      break;			// NULL will be returned for invalid code
    }
  return keyCode;
}

void DragSite::objectToDrag(Draggable& t)
{
  dPtr = &t;
  dPtr->atom =  XmInternAtom( XtDisplay(w), (char *)dPtr->atomName(), False );
}

DragSite::DragSite( Widget aWidget, const char* actionName_, Boolean setDefaultDragKeys ):
  w(aWidget), dPtr(NULL), actionName(NULL)
{
  if (w)
    {
      actionName = actionName_;
      if (!actionName)
	actionName = DragSite::DefaultActionName;

      if (!contextId)
	contextId = XUniqueContext();
      // Associate "this" with the widget, so we can access it in Motif callbacks

      XSaveContext( XtDisplay( w ), 
		    (XID) w,
		    contextId, 
		    (char *) this );
      // set actions for the app
      
      XtActionsRec dragActions;
      dragActions.string = (char *)actionName;
      dragActions.proc   = (XtActionProc)DragSite::startDrag;
      XtAppAddActions( XtDisplayToApplicationContext(XtDisplay(w)),
		       &dragActions, 1);

      if (setDefaultDragKeys)
	setDragKeys();
    }
}

  
void DragSite::setDragKeys()
{
  // set translations for the drag site

  char xlatString[1024];
  const char *keyCode;

  strcpy (xlatString, "#override ");
  for (int i = 0; i < XtNumber(dragOps); i++)
    {
      // Are we going to support this type of drag operation?
      if (keyCode = dragKey(dragOps[i].op))
	{
	  strcat (xlatString, "\n");
	  strcat (xlatString, " ");
	  strcat (xlatString, keyCode);
	  strcat (xlatString, ": ");
	  strcat (xlatString, actionName);
	  strcat (xlatString, "(" );
	  strcat (xlatString,  dragOps[i].tag);
	  strcat (xlatString, ")" );
	}
    }

  XtTranslations xlats = XtParseTranslationTable (xlatString);
  XtOverrideTranslations(w, xlats);
 
}

void DragSite::startDrag (Widget w,
		     XEvent *event,
		     String *params,
		     Cardinal *nparams)
{
  DragSite *dsPtr;

  XFindContext( XtDisplay( w ), 
		(XID) w,
		contextId, 
		(XPointer *)&dsPtr );
  if (dsPtr)
    dsPtr->dragActionProc (event, params, nparams);
}

void DragSite::dragActionProc (XEvent *event,
			  String *params,
			  Cardinal *nparams)
{
  Cardinal n, nExtra;
  Arg args[5];
  ArgList extra_args;

  if (!dPtr)			// No object to drag
    return;

  // Set DragContext widget properties in the form of Arg[] array
  // and merge them with locally set properties

  nExtra = dragContextProperties(&extra_args);

  n = 0;
  XtSetArg( args[n], XmNexportTargets, &dPtr->atom );  n++;
  XtSetArg( args[n], XmNnumExportTargets, 1 );  n++;

  // Set drag operations. The value of 'params' is one or more drag/drop operation
  // tags (e.g. "DROP_COPY")
  int dragMask = XmDROP_NOOP;
  if (*nparams)
    for (int np=0; np < *nparams; np++)
      for (int nact=0; nact < XtNumber(dragOps); nact++)
	{
	  if (!strcmp(params[np], dragOps[nact].tag))
	    {
	      dragMask |= dragOps[nact].op;
	      break;
	    }
	}
  XtSetArg( args[n], XmNdragOperations, dragMask );  n++;
  XtSetArg( args[n], XmNclientData, (XtPointer)this );  n++;
  XtSetArg( args[n], XmNconvertProc, (XtConvertSelectionProc)DragSite::DragConvertProc );  n++;

  if (nExtra)
    {
      ArgList all_args = XtMergeArgLists(args, n, extra_args, nExtra);
      XmDragStart( w, event, all_args, n+nExtra );
      XtFree((char *)extra_args);
      XtFree((char *)all_args);
    }
  else
    XmDragStart( w, event, args, n );
}

Boolean DragSite::DragConvertProc( Widget dragContextWidget, Atom*, 
			   Atom *target, Atom *typeRtn, XtPointer *valueRtn,
			   unsigned long *lengthRtn, int *formatRtn,
			   unsigned long*, XtPointer, XtRequestId* )
{
  Boolean retval = True;
  
  DragSite *dsPtr;
  XtVaGetValues(dragContextWidget,
		XmNclientData, &dsPtr,
		NULL);

  // If all pointers are non-zero and target atom type matches
  // this drag site's atom type, prepare draggable object for transfer

  if (dsPtr && dsPtr->dPtr && (*target==dsPtr->dPtr->atom))
    {
      // Determine the actual operation
      unsigned char dragMask;
      XtVaGetValues(dragContextWidget,
		    XmNdragOperations, &dragMask,
		    NULL);

      strstreambuf *putbuf = new strstreambuf(); // dynamic streambuf
      o_stream   putstr(putbuf);
      // switch on the actual operation
      switch(dragMask)
	{
	case XmDROP_COPY:
	  dsPtr->dPtr->doCopy(putstr);
	  break;
	case XmDROP_MOVE:
	  dsPtr->dPtr->doMove(putstr);
	  break;
	case XmDROP_LINK:
	  dsPtr->dPtr->doLink(putstr);
	  break;
	default:
	  break;
	}
      char *buf = XtMalloc(putstr.tellp());
      if (!buf)
	retval = False;
      else
	{
	  // copy data from stream to buf
	  memcpy(buf, putbuf->str(), putstr.tellp());
	  // putbuf->str() has frozen the internal array; to allow automatic deletion
	  // I need to call freeze(0).
	  putbuf->freeze(0);
	  *valueRtn  = (XtPointer)buf;
	  *lengthRtn = putstr.tellp();
	  *typeRtn   = dsPtr->dPtr->atom;
	  *formatRtn = dsPtr->dPtr->format();
	}
      delete putbuf;
    }
  else
    {
      // Drag object not set
      retval = False;
    }
  return retval;
}


Cardinal DragSite::dragContextProperties(ArgList*)
{
  return 0;
}
