/*
 * Copyright (C) 1994 Communications and Systems Specialists, Inc.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * GROUPE BULL BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
 * AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Communications and Systems 
 * Specialists, Inc. shall not be used in advertising or otherwise to promote 
 * the sale, use or other dealings in this Software without prior written 
 * authorization from Communications and Systems Specialists, Inc.
 */

#ifndef __Xmsg_h
#define __Xmsg_h


#include <X11/Intrinsic.h>
#include <X11/Xosdefs.h>
#include <errno.h>

#ifdef X_NOT_STDC_ENV
   extern int errno;
#endif

#ifdef _Xconst
#   define _XmsgConst	_Xconst
#else
#  if __STDC__ || defined(__cplusplus) || defined(c_plusplus) || (FUNCPROTO&4)
#     define _XmsgConst	const
#  else
#     define _XmsgConst	
#  endif
#endif

#define _XmsgString	char*


/*--------------------------
-- Define category strings 
--------------------------*/
#define XmsgCATEGORY_DEBUG	"Debug"
#define XmsgCATEGORY_ERROR	"Error"
#define XmsgCATEGORY_FATAL	"Fatal"
#define XmsgCATEGORY_INFO	"Info"
#define XmsgCATEGORY_WARNING	"Warning"


/*--------------------------------
-- Control flags for handlers
--------------------------------*/
#define XmsgCONTROL_CALL_AS_DEFAULT      (1<<0)
#define XmsgCONTROL_INSTALL_XTERRORMSG   (1<<1)
#define XmsgCONTROL_INSTALL_XTWARNINGMSG (1<<2)
#define XmsgCONTROL_INSTALL_XTERRORLL    (1<<3)
#define XmsgCONTROL_INSTALL_XTWARNINGLL  (1<<4)
#define XmsgCONTROL_INSTALL_XERROR       (1<<5)
#define XmsgCONTROL_INSTALL_XIO          (1<<6)
#define XmsgCONTROL_INSTALL_STDERR       (1<<7)
#define XmsgCONTROL_DISPATCH_ALL         (1<<8)

#define XmsgCONTROL_INSTALL_XTERROR     (XmsgCONTROL_INSTALL_XTERRORMSG | \
                                         XmsgCONTROL_INSTALL_XTERRORLL)
 
#define XmsgCONTROL_INSTALL_XTWARNING	(XmsgCONTROL_INSTALL_XTWARNINGMSG | \
                                         XmsgCONTROL_INSTALL_XTWARNINGLL)
 
#define XmsgCONTROL_INSTALL_XT		(XmsgCONTROL_INSTALL_XTERROR | \
                                         XmsgCONTROL_INSTALL_XTWARNING)
 
#define XmsgCONTROL_INSTALL_XLIB	(XmsgCONTROL_INSTALL_XERROR | \
                                         XmsgCONTROL_INSTALL_XIO)
 
#define XmsgCONTROL_INSTALL_ALL_X	(XmsgCONTROL_INSTALL_XT | \
                                         XmsgCONTROL_INSTALL_XLIB)
 
#define XmsgCONTROL_INSTALL_ALL	        (XmsgCONTROL_INSTALL_ALL_X | \
                                         XmsgCONTROL_INSTALL_STDERR)
 
/*******************************************************************
 * Type used to determine how a popup message should be positioned
 ******************************************************************/
typedef enum
{
   XmsgPositionDATABASE, /* Popup position based on entries in the error DB  */
   XmsgPositionWIDGET,   /* Popup centered over specified widget             */
   XmsgPositionSHELL,    /* Popup centered over widget's non-transient shell */
   XmsgPositionSCREEN,   /* Popup centered over screen                       */
   XmsgPositionDEFAULT   /* Popup position is implmementation specific       */
}
XmsgPosition;


/****************************************************************
 * Abstract data type used to define a message to be dispatched
 ***************************************************************/
typedef struct _XmsgMessageStruct XmsgMessageStruct;
 
/**************************
 * Xmsg method prototypes
 *************************/

/******************************************************************************
 * determines the processing for a message.
 *
 * This type of function processes a message.
 *****************************************************************************/
typedef void (*XmsgHandlerProc)( 
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr	/* An opaque message structure */
#endif
);

/******************************************************************************
 * performs exit functions if a fatal message is processed.
 *
 * This type of function is called when a fatal message is received.
 * The application will terminate immediately after this function is called.
 *****************************************************************************/
typedef void (*XmsgExitProc)( 
#if NeedFunctionPrototypes
   void 
#endif
);

/******************************************************************************
 * determines if the message forced to an associated message handler,
 * regardless of the entires in the error database.
 *
 * This type of function is associated with a message handler when the message
 * handler is installed with XmsgAddHandler().  If the function returns
 * True, the message is sent to the handler; otherwise, the error database
 * is examined to determine if the message should be routed.
 *
 * Returns True if the message should always be routed to the associated
 * message handler.
 *****************************************************************************/
typedef Boolean (*XmsgShouldDispatchProc)( 
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr	/* An opaque message structure */
#endif
);

/***********************
 * Function prototypes
 **********************/
_XFUNCPROTOBEGIN


/******************************************************************************
 * register a new message handler with the system
 *
 * XmsgAddHandler is used to register new message handlers with the Xmsg 
 * library.  If a handler already exists with the specified name, the old
 * handler is removed and returned.
 *
 * The control parameter can either be 0L or XmsgCONTROL_CALL_AS_DEFAULT, which
 * specifies that the handler should be called if no related entries are found
 * in the error database.
 *
 * Returns the previously installed handler with the name name_str or
 * NULL if a handler with the specified name was not previously installed.   
 *****************************************************************************/
XmsgHandlerProc XmsgAddHandler(
#if NeedFunctionPrototypes
   _XmsgConst _XmsgString      name_str,		/* The name of the handler */
   XmsgHandlerProc        handler_proc,         /* The message handler
						   function */
   unsigned int           control,		/* Bit-field used to specify
						   options */
   XmsgShouldDispatchProc should_dispatch_proc	/* The function to determine
						   if routing is forced or
						   NULL */
#endif 
);

/******************************************************************************
 * bind a new display connetion with the Xmsg library
 *
 * If additional display connections are opened by an application, the
 * XmsgAddDisplay() function needs to be called before any messages are
 * generated against the new display.
 *****************************************************************************/
void XmsgAddDisplay( 
#if NeedFunctionPrototypes
   Display* display_ptr 	/* The X display connection pointer */
#endif
);

/******************************************************************************
 * binds Xmsg message handlers to the X Window System message handlers
 *
 * This function sets Xmsg handlers on X Window System message handlers
 * as specified by the control parameter.  This function is normally not
 * used because it is automatically handler when the library is installed.
 * It is primarily used if new application contexts are created.
 * XmsgAppDefaultXtHandlers() does not remove Xmsg handlers that are not
 * specified in the control parameter.
 *
 * The control parameter can have the following values ORed together:
 * .TP
 * .B "XmsgCONTROL_INSTALL_ALL_X"
 *                                      All X and Xt errors and warnings are
 *					trapped.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XLIB"
 *					All X events and I/O errors are trapped.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XERROR"
 *					X error events are trapped by the Xmsg 
 *					facility.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XIO"
 *					X I/O errors are trapped by the Xmsg 
 *					facility.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XT"
 *					All Xt errors and warnings are trapped.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XTERROR"
 *					Xt high- and low-level errors are
 *					trapped.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XTERRORLL"
 *					Xt low-level errors are trapped by the 
 *					Xmsg facility.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XTERRORMSG"
 *					Xt high-level errors are trapped by the 
 *					Xmsg facility.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XTWARNING"
 *					Xt high- and low-level warnings are
 *					trapped.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XTWARNINGLL"
 *					Xt low-level warnings are trapped by 
 *					the Xmsg facility.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XTWARNINGMSG"
 *					Xt high-level warnings are trapped by 
 *					the Xmsg facility.
 *****************************************************************************/
void XmsgAppDefaultXtHandlers(
#if NeedFunctionPrototypes
   XtAppContext app_context,	/* The application context to use */	
   unsigned int control 	/* Bit-field used to specify options */
#endif
);

/******************************************************************************
 * override the error database with specified values
 *
 * This function uses a string of resources to override the error database.
 * the value_str parameter has the same format as the data parameter in the
 * function XrmGetStringDatabase().
 *****************************************************************************/
Boolean XmsgAppDirectOverrideErrorDB(
#if NeedFunctionPrototypes
   XtAppContext	     app_context,	/* The application context to use */	
   _XmsgConst _XmsgString value_str 		/* The entries to add to the error 
					   database */
#endif
);

/******************************************************************************
 * merge the contents of the specified file with the error database
 *
 * XmsgAppMergeErrorDB merges the specified resource file with the
 * error database associated with a given application context.
 *****************************************************************************/
Boolean XmsgAppMergeErrorDB(
#if NeedFunctionPrototypes
   XtAppContext      app_context,	/* The application context to use */
   _XmsgConst _XmsgString database_name_str  /* The resource file to merge     */
#endif
);

/******************************************************************************
 * copy an opaque message structure
 *
 * This function copies a message structure into heap memory.  The message
 * must be freed by the application with XmsgFreeMessage().
 * The client data associated with the original message is not retained
 * in the copy; instead, the client data in the copy is NULL.
 *****************************************************************************/
XmsgMessageStruct* XmsgCopyMessage(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr	/* Opaque message structure */
#endif
);

/******************************************************************************
 * concerts a string to a Boolean
 *
 * XmsgCvtStringToBoolean() converts a string to a Boolean.   
 * XmsgCvtStringToBoolean() is thread-safe in an X11R6 environment and 
 * supports a convenient default value if the specified string cannot be 
 * converted.
 *
 * Returns True if the string represents a Boolean True and False if the
 * string represents a Boolean False.
 *****************************************************************************/
Boolean XmsgCvtStringToBoolean(
#if NeedFunctionPrototypes
   _XmsgConst _XmsgString str,		/* The string to convert */
#  if NeedWidePrototypes
   int               default_value      /* The value to return if str cannot
					   be converted */
#  else
   Boolean           default_value      /* The value to return if str cannot
					   be converted */
#  endif
#endif
);

/******************************************************************************
 * calls the exit handler installed in the Xmsg library
 *
 * This function calls the exit handler installed in the Xxsg library.
 *****************************************************************************/
void XmsgExit( 
#if NeedFunctionPrototypes
   void 
#endif
);

/******************************************************************************
 * free an opaque message structure
 *
 * XmsgFreeMessage() is used to deallocate the memory used by an opaque
 * message structure.  It must be called to free memory allocated with
 * XmsgCopyMessage().  XmsgFreeMessage() can be called on any message
 * structure, even if it was not originally created via XmsgCopyMessage().
 *****************************************************************************/
void XmsgFreeMessage(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the application class associated with a message
 *
 * XmsgGetMsgApplClass retrieves the application class associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the value of the associated application class.
 ****************************************************************************/
_XmsgConst _XmsgString XmsgGetMsgApplClass(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the application name associated with a message
 *
 * XmsgGetMsgApplName retrieves the application name associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the value of the associated application name.
 ****************************************************************************/
_XmsgConst _XmsgString XmsgGetMsgApplName(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the value from the error database associated with a message.
 *
 * XmsgGetMsgDatabaseString retrieves a configuration value from the error
 * database associated with a message and specified keys.  The value returned
 * must not be freed or altered.  If a value cannot be found in the database,
 * NULL is returned.
 *
 * Returns the value of the associated database entry.
 ****************************************************************************/
_XmsgConst _XmsgString XmsgGetMsgDatabaseString(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr,	/* An opaque message structure */
   XrmQuark           name_quark,	/* Name key of desired value */
   XrmQuark           class_quark 	/* Class key of desired value */
#endif
);

/*****************************************************************************
 * get the delay protocol associated with a message
 *
 * XmsgGetMsgDelayProtocol retrieves the delay protocol associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the delay protocol for the message.
 ****************************************************************************/
Boolean XmsgGetMsgDelayProtocol(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the error number associated with a message
 *
 * XmsgGetMsgErrorNumber retrieves the error number associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the error number for the message.
 ****************************************************************************/
int XmsgGetMsgErrorNumber(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the message class associated with a message
 *
 * XmsgGetMsgClass retrieves the message class associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the message class for the message.
 ****************************************************************************/
_XmsgConst _XmsgString XmsgGetMsgClass(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the client data associated with a message
 *
 * XmsgGetMsgClientData retrieves the client data associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the client data for the message.
 ****************************************************************************/
_XmsgConst XtPointer XmsgGetMsgClientData(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the host associated with a message
 *
 * XmsgGetMsgHost retrieves the host associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the host for the message.
 ****************************************************************************/
_XmsgConst _XmsgString XmsgGetMsgHost(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the time associated with a message
 *
 * XmsgGetMsgLocalTime retrieves the time associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the time for the message.
 ****************************************************************************/
_XmsgConst _XmsgString XmsgGetMsgLocalTime(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the message name associated with a message
 *
 * XmsgGetMsgName retrieves the message name associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the message name for the message.
 ****************************************************************************/
_XmsgConst _XmsgString XmsgGetMsgName(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the desired popup position associated with a message
 *
 * XmsgGetMsgPosition retrieves the desired popup position associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the desired popup position for the message.
 ****************************************************************************/
XmsgPosition XmsgGetMsgPosition(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the  associated with a message
 *
 *  retrieves the  associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the  for the message.
 ****************************************************************************/
_XmsgConst _XmsgString XmsgGetMsgCategory(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the source line associated with a message
 *
 * XmsgGetMsgSourceLine retrieves the source line associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the source line for the message.
 ****************************************************************************/
unsigned int XmsgGetMsgSourceLine(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the source name associated with a message
 *
 * XmsgGetMsgSourceName retrieves the source name associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the source name for the message.
 ****************************************************************************/
_XmsgConst _XmsgString XmsgGetMsgSourceName(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the message text associated with a message
 *
 * XmsgGetMsgText retrieves the message text associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the message text for the message.
 ****************************************************************************/
_XmsgConst _XmsgString XmsgGetMsgText(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the message type associated with a message
 *
 * XmsgGetMsgType retrieves the message type associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the message type for the message.
 ****************************************************************************/
_XmsgConst _XmsgString XmsgGetMsgType(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the user login name associated with a message
 *
 * XmsgGetMsgUserLogin retrieves the user login name associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the user login name for the message.
 ****************************************************************************/
_XmsgConst _XmsgString XmsgGetMsgUserLogin(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the user name associated with a message
 *
 * XmsgGetMsgUserName retrieves the user name associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the user name for the message.
 ****************************************************************************/
_XmsgConst _XmsgString XmsgGetMsgUserName(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the widget associated with a message
 *
 * XmsgGetMsgWidget retrieves the widget associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the widget for the message.
 ****************************************************************************/
Widget XmsgGetMsgWidget(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * get the X display server associated with a message
 *
 * XmsgGetMsgXDisplayServer retrieves the X display server associated with
 * a message.  The value returned must not be freed or altered.
 *
 * Returns the X display server for the message.
 ****************************************************************************/
_XmsgConst _XmsgString XmsgGetMsgXDisplayServer(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * named message handler for writing messages to stderr.
 *
 * The standard named message handler which writes messages to stderr.
 * The handler can be automatically registered when XmsgInstall()
 * is called.  
 *
 * If the handler is uninstalled, it can be reinstalled by using the
 * function XmsgAddHandler().
 ****************************************************************************/
void XmsgHandlerStderr(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * initialize the Xmsg library
 *
 * XmsgInitialize must be called directly or indirectly (via XmsgInstall)
 * before the Xmsg library can be used.  This function can be called at any
 * time to change the default widget. 
 ****************************************************************************/
void XmsgInitialize(
#if NeedFunctionPrototypes
   Widget       w 		/* The default widget associated with 
				   widget-less messages */
#endif
);

/*****************************************************************************
 * install the Xmsg facility
 *
 * XmsgInstall is typically the first Xmsg function called.  It initializes
 * the library and installs default named message handlers.  
 *
 * The control parameter can have the following values ORed together:
 * .TP
 * .B "XmsgCONTROL_DISPATCH_ALL"
 *                                      If specified, all messages will be
 *                                      dispatched to at least one handler.
 *
 * .TP
 * .B "XmsgCONTROL_INSTALL_ALL_X"
 *                                      All X and Xt errors and warnings are
 *                                      trapped.
 * .TP
 * .B "XmsgCONTROL_INSTALL_STDERR"
 *                                      Install XmsgHandlerStderr() as a 
 *                                      message handler with the name "stderr".
 * .TP
 * .B "XmsgCONTROL_INSTALL_XLIB"
 *					All X events and I/O errors are trapped.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XERROR"
 *					X error events are trapped by the Xmsg 
 *					facility.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XIO"
 *					X I/O errors are trapped by the Xmsg 
 *					facility.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XT"
 *					All Xt errors and warnings are trapped.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XTERROR"
 *					Xt high- and low-level errors are
 *					trapped.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XTERRORLL"
 *					Xt low-level errors are trapped by the 
 *					Xmsg facility.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XTERRORMSG"
 *					Xt high-level errors are trapped by the 
 *					Xmsg facility.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XTWARNING"
 *					Xt high- and low-level warnings are
 *					trapped.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XTWARNINGLL"
 *					Xt low-level warnings are trapped by 
 *					the Xmsg facility.
 * .TP
 * .B "XmsgCONTROL_INSTALL_XTWARNINGMSG"
 *					Xt high-level warnings are trapped by 
 *					the Xmsg facility.
 ****************************************************************************/
void XmsgInstall( 
#if NeedFunctionPrototypes
   Widget       w, 		/* The default widget associated with 
				   widget-less messages */
   unsigned int control 	/* Bit-field used to specify options */
#endif
);

/*****************************************************************************
 * indicates if a specified handler is installed
 *
 * XmsgIsHandlerDefined searchs for a handler with the specified name and
 * indicates if it is found.
 *
 * Returns True if the specified handler is installed in the Xmsg facility.
 ****************************************************************************/
Boolean XmsgIsHandlerDefined(
#if NeedFunctionPrototypes
   _XmsgConst _XmsgString name_str 	/* The name of the handler to find */
#endif
);

/*****************************************************************************
 * indicates if a specified handler is installed
 *
 * XmsgQIsHandlerDefined searchs for a handler with the specified quark name 
 * and indicates if it is found.
 *
 * Returns True if the specified handler is installed in the Xmsg facility.
 ****************************************************************************/
Boolean XmsgQIsHandlerDefined(
#if NeedFunctionPrototypes
   XrmQuark name_quark   	/* The quark name of the handler to find */
#endif
);

/*****************************************************************************
 * removes a message handler
 *
 * XmsgRemoveHandler removes the specified handler from the Xmsg facility.
 *
 * Returns the previously installed handler with the specified quark name or
 * NULL if a handler with the specified name was not previously installed.   
 ****************************************************************************/
XmsgHandlerProc XmsgQRemoveHandler(
#if NeedFunctionPrototypes
   XrmQuark name_quark  	/* The quark name of the handler to remove */
#endif
);

/*****************************************************************************
 * removes a message handler
 *
 * XmsgRemoveHandler removes the specified handler from the Xmsg facility.
 *
 * Returns the previously installed handler with the specified name or
 * NULL if a handler with the specified name was not previously installed.   
 ****************************************************************************/
XmsgHandlerProc XmsgRemoveHandler(
#if NeedFunctionPrototypes
   _XmsgConst _XmsgString name_str  	/* The name of the handler to remove */
#endif
);

/*****************************************************************************
 * define an exit handler
 *
 * XmsgSetExitHandler installs a new exit handler with the Xmsg facility.
 *
 * Returns the previous exit handler.
 ****************************************************************************/
XmsgExitProc XmsgSetExitHandler(
#if NeedFunctionPrototypes
   XmsgExitProc exit_method
#endif
);

/*****************************************************************************
 * indicates if a message should always be displayed to the user
 *
 * XmsgShouldPopup is normally used as the should_dispatch_proc for message
 * handlers that display messages in a popup window.  ****************************************************************************/
Boolean XmsgShouldPopup(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr 	/* An opaque message structure */
#endif
);

/*****************************************************************************
 * translate raw message text to Xmsg cooked message text
 *
 * XmsgTranslate uses the attributes in the specified message to cook an
 * arbitrary character string.
 ****************************************************************************/
void XmsgTranslate(
#if NeedFunctionPrototypes
   XmsgMessageStruct* message_ptr,	/* An opaque message structure */
   _XmsgConst _XmsgString  buffer_in_str,    /* The raw text buffer */
   String             buffer_out_str    /* The cooked text buffer */
#endif
);

/*****************************************************************************
 * generate a debug message
 *
 * Xmsg_Debug is used to generated an Xmsg debug message.
 *
 * This function normally should not be used.  XmsgDebug() or XmsgDebugPos()
 * should be used instead.
 ****************************************************************************/
void Xmsg_Debug(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   _XmsgConst _XmsgString source_str,	/* The source code file name */
   unsigned int      source_line,	/* The source line where the message
					   is generated */
   int               error_number,	/* The Unix errno at the time the 
					   message is generated */
   XtPointer         client_data,	/* Generic data associated with
					   the message */
   XmsgPosition      position		/* The logical position for a popup */
#endif
);   

/*****************************************************************************
 * generate an error message
 *
 * Xmsg_Error is used to generated an Xmsg error message.
 *
 * This function normally should not be used.  XmsgError() or XmsgErrorPos()
 * should be used instead.
 ****************************************************************************/
void Xmsg_Error(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   _XmsgConst _XmsgString source_str,	/* The source code file name */
   unsigned int      source_line,	/* The source line where the message
					   is generated */
   int               error_number,	/* The Unix errno at the time the 
					   message is generated */
   XtPointer         client_data,	/* Generic data associated with
					   the message */
   XmsgPosition      position		/* The logical position for a popup */
#endif
);   

/*****************************************************************************
 * generate a fatal message
 *
 * Xmsg_Fatal is used to generated an Xmsg fatal message.
 *
 * This function normally should not be used.  XmsgFatal() or XmsgFatalPos()
 * should be used instead.
 ****************************************************************************/
void Xmsg_Fatal(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   _XmsgConst _XmsgString source_str,	/* The source code file name */
   unsigned int      source_line,	/* The source line where the message
					   is generated */
   int               error_number,	/* The Unix errno at the time the 
					   message is generated */
   XtPointer         client_data,	/* Generic data associated with
					   the message */
   XmsgPosition      position		/* The logical position for a popup */
#endif
);   

/*****************************************************************************
 * generate an arbitrary message
 *
 * Xmsg_Generic is used to generated an arbitrary message.
 *
 * This function is normally used to define new message catagories, either
 * with macros or wrapper functions.
 ****************************************************************************/
void Xmsg_Generic(
#if NeedFunctionPrototypes
   _XmsgConst _XmsgString category_str,	/* A NULL-terminated string specifying
					   the catagory of the message, such
					   as "Debug" or "Fatal" */
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   _XmsgConst _XmsgString source_str,	/* The source code file name */
   unsigned int      source_line,	/* The source line where the message
					   is generated */
   int               error_number,	/* The Unix errno at the time the 
					   message is generated */
   XtPointer         client_data,	/* Generic data associated with
					   the message */
   XmsgPosition      position		/* The logical position for a popup */
#endif
);   

/*****************************************************************************
 * generate an informational message
 *
 * Xmsg_Info is used to generated an Xmsg informational message.
 *
 * This function normally should not be used.  XmsgInfo() or XmsgInfoPos()
 * should be used instead.
 ****************************************************************************/
void Xmsg_Info(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   _XmsgConst _XmsgString source_str,	/* The source code file name */
   unsigned int      source_line,	/* The source line where the message
					   is generated */
   int               error_number,	/* The Unix errno at the time the 
					   message is generated */
   XtPointer         client_data,	/* Generic data associated with
					   the message */
   XmsgPosition      position		/* The logical position for a popup */
#endif
);   

/*****************************************************************************
 * generate a warning message
 *
 * Xmsg_Warning is used to generated an Xmsg warning message.
 *
 * This function normally should not be used.  XmsgWarning() or 
 * XmsgWarningPos() should be used instead.
 ****************************************************************************/
void Xmsg_Warning(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   _XmsgConst _XmsgString source_str,	/* The source code file name */
   unsigned int      source_line,	/* The source line where the message
					   is generated */
   int               error_number,	/* The Unix errno at the time the 
					   message is generated */
   XtPointer         client_data,	/* Generic data associated with
					   the message */
   XmsgPosition      position		/* The logical position for a popup */
#endif
);   


/*---------------------
-- Convienence Macros
---------------------*/
#if __C2MAN__
/*********************************************************************
 * c2man is used to automatically generate the Xmsg man pages.  Fake
 * prototypes for these macros are created to trick c2man into 
 * generating pages for the macros
 *********************************************************************/

/*****************************************************************************
 * generate a debug message
 *
 * XmsgDebug is used to generated an Xmsg debug message.
 *
 * The message is only generated if the XmsgUSE_DEBUG macro is defined
 * when the application is compiled.
 *
 * This function is implemented as a macro.
 ****************************************************************************/
void XmsgDebug(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   XtPointer         client_data 	/* Generic data associated with
					   the message */
#endif
);   

/*****************************************************************************
 * generate an error message
 *
 * XmsgError is used to generated an Xmsg error message.
 *
 * This function is implemented as a macro.
 ****************************************************************************/
void XmsgError(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   XtPointer         client_data 	/* Generic data associated with
					   the message */
#endif
);   

/*****************************************************************************
 * generate a fatal message
 *
 * XmsgFatal is used to generated an Xmsg fatal message.
 *
 * The exit handler installed in the Xmsg library will be called and the
 * application will terminate before this function returns.
 *
 * This function is implemented as a macro.
 ****************************************************************************/
void XmsgFatal(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   XtPointer         client_data 	/* Generic data associated with
					   the message */
#endif
);   

/*****************************************************************************
 * generate an informational message
 *
 * XmsgInfo is used to generated an Xmsg informational message.
 *
 * This function is implemented as a macro.
 ****************************************************************************/
void XmsgInfo(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   XtPointer         client_data 	/* Generic data associated with
					   the message */
#endif
);   

/*****************************************************************************
 * generate a warning message
 *
 * XmsgWarning is used to generated an Xmsg warning message.
 *
 * This function is implemented as a macro.
 ****************************************************************************/
void XmsgWarning(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   XtPointer         client_data 	/* Generic data associated with
					   the message */
#endif
);   

/*****************************************************************************
 * generate a debug message with a specified popup position
 *
 * XmsgDebugPos is used to generated an Xmsg debug message.  If a handler
 * is installed which will display the message in a dialog box, the 
 * logical position of the dialog can be specified.
 *
 * The message is only generated if the XmsgUSE_DEBUG macro is defined
 * when the application is compiled.
 *
 * This function is implemented as a macro.
 ****************************************************************************/
void XmsgDebugPos(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   XtPointer         client_data, 	/* Generic data associated with
					   the message */
   XmsgPosition      position		/* The logical position for a popup */	
#endif
);   

/*****************************************************************************
 * generate an error message with a specified popup position
 *
 * XmsgErrorPos is used to generated an Xmsg error message.  If a handler
 * is installed which will display the message in a dialog box, the 
 * logical position of the dialog can be specified.
 *
 * This function is implemented as a macro.
 ****************************************************************************/
void XmsgErrorPos(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   XtPointer         client_data, 	/* Generic data associated with
					   the message */
   XmsgPosition      position		/* The logical position for a popup */	
#endif
);   

/*****************************************************************************
 * generate a fatal message with a specified popup position
 *
 * XmsgFatalPos is used to generated an Xmsg fatal message.  If a handler
 * is installed which will display the message in a dialog box, the 
 * logical position of the dialog can be specified.
 *
 * The exit handler installed in the Xmsg library will be called and the
 * application will terminate before this function returns.
 *
 * This function is implemented as a macro.
 ****************************************************************************/
void XmsgFatalPos(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   XtPointer         client_data, 	/* Generic data associated with
					   the message */
   XmsgPosition      position		/* The logical position for a popup */	
#endif
);   

/*****************************************************************************
 * generate an informational message with a specified popup position
 *
 * XmsgInfoPos is used to generated an Xmsg informational message.  If a handler
 * is installed which will display the message in a dialog box, the 
 * logical position of the dialog can be specified.
 *
 * This function is implemented as a macro.
 ****************************************************************************/
void XmsgInfoPos(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   XtPointer         client_data, 	/* Generic data associated with
					   the message */
   XmsgPosition      position		/* The logical position for a popup */	
#endif
);   

/*****************************************************************************
 * generate a warning message with a specified popup position
 *
 * XmsgWarningPos is used to generated an Xmsg warning message.  If a handler
 * is installed which will display the message in a dialog box, the 
 * logical position of the dialog can be specified.
 *
 * This function is implemented as a macro.
 ****************************************************************************/
void XmsgWarningPos(
#if NeedFunctionPrototypes
   Widget            w,			/* The widget associated with the 
					   message or NULL */
   _XmsgConst _XmsgString name_str,		/* The general kind of message */
   _XmsgConst _XmsgString type_str,		/* The detailed name of the message */
   _XmsgConst _XmsgString class_str,		/* The class of the message */
   _XmsgConst _XmsgString default_str,	/* The default message text to use */
   String*           params_str_table,	/* Specifies a pointer to a list of
					   values to be substituted in the
					   message */
   Cardinal          params_str_count,	/* Specifies the number of values 
					   in params_str_table */
   XtPointer         client_data, 	/* Generic data associated with
					   the message */
   XmsgPosition      position		/* The logical position for a popup */	
#endif
);   

#else

#define XmsgDebug(w,name,type,class,def,params,count,cd)		\
   XmsgDebugPos( (w), (name), (type), (class), (def),			\
                 (params), (count), (cd), XmsgPositionDATABASE )

#define XmsgError(w,name,type,class,def,params,count,cd)		\
   XmsgErrorPos( (w), (name), (type), (class), (def),			\
                 (params), (count), (cd), XmsgPositionDATABASE )

#define XmsgFatal(w,name,type,class,def,params,count,cd)		\
   XmsgFatalPos( (w), (name), (type), (class), (def),			\
                 (params), (count), (cd), XmsgPositionDATABASE )

#define XmsgGeneric(sev,w,name,type,class,def,params,count,cd) 		\
   XmsgGenericPos( (sev), (w), (name), (type), (class), (def),		\
                   (params), (count), (cd), XmsgPositionDATABASET )

#define XmsgInfo(w,name,type,class,def,params,count,cd)			\
   XmsgInfoPos( (w), (name), (type), (class), (def),			\
                (params), (count), (cd), XmsgPositionDATABASE )

#define XmsgWarning(w,name,type,class,def,params,count,cd)		\
   XmsgWarningPos( (w), (name), (type), (class), (def),			\
                   (params), (count), (cd), XmsgPositionDATABASE )


#ifdef XmsgUSE_DEBUG
#
#  define XmsgDebugPos(w,name,type,class,def,params,count,cd,pos)	\
      Xmsg_Debug( (w), (name), (type), (class), (def),			\
                  (params), (count), __FILE__, __LINE__, (int) errno,	\
                  (cd), (pos) )
#
#else
#
#  define XmsgDebugPos(w,name,type,class,def,params,count,cd,pos)	\
      /**** XmsgDebug call not compiled into source ****/
#
#endif

#define XmsgErrorPos(w,name,type,class,def,params,count,cd,pos)		\
   Xmsg_Error( (w), (name), (type), (class), (def),			\
               (params), (count), __FILE__, __LINE__, (int) errno,	\
               (cd), (pos) )

#define XmsgFatalPos(w,name,type,class,def,params,count,cd,pos)		\
   Xmsg_Fatal( (w), (name), (type), (class), (def),			\
               (params), (count), __FILE__, __LINE__, (int) errno,	\
               (cd), (pos) )

#define XmsgGenericPos(sev,w,name,type,class,def,params,count,cd,pos)	\
   Xmsg_Generic( (sev), (w), (name), (type), (class), (def),		\
                 (params), (count), __FILE__, __LINE__, (int) errno,	\
                 (cd), (pos) )

#define XmsgInfoPos(w,name,type,class,def,params,count,cd,pos)		\
   Xmsg_Info( (w), (name), (type), (class), (def),			\
               (params), (count), __FILE__, __LINE__, (int) errno,	\
               (cd), (pos) )

#define XmsgWarningPos(w,name,type,class,def,params,count,cd,pos)	\
   Xmsg_Warning( (w), (name), (type), (class), (def),			\
                 (params), (count), __FILE__, __LINE__, (int) errno,	\
                 (cd), (pos) )
#endif

_XFUNCPROTOEND

#endif
