/* 
 * X Development Tools Library
 * $Id: QkHelp.c,v 1.1 1996/04/01 03:48:28 rwerne Exp rwerne $
 * 
 * Written by Robert Werner.
 * Copyright (c) 1995, 1996 by Software Komponents International.
 * All Rights Reserved.  See the file COPYRIGHT for details.
 * This is not public domain software.  See the file LICENSE for details.
 * There is no warranty for this software.  See NO_WARRANTY for details.
 */

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <X11/IntrinsicP.h>
#include <X11/CoreP.h>		/* To get access to the popup list          */
#include <X11/CompositeP.h>	/* To get access to the Composite internals */
#include <X11/keysym.h>
#include <X11/Xfuncs.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>


#ifdef USE_MOTIF
   #include <Xm/Xm.h>
   #include <Xm/XmStrDefs.h>
#endif

#include <Xdt/Xdt.h>
#include <Xdt/Child.h>
#include <Xdt/QkHelp.h>
#include <Xdt/ResMgmt.h>


#define XdtQkHelpGetPopupList( w )	((w)->core.popup_list)
#define XdtQkHelpGetNumPopups( w )	((w)->core.num_popups)

#define XdtQkHelpNhelp	"help"
#define XdtQkHelpNhint	"hint"
#define XdtQkHelpNtext	"text"
#define XdtQkHelpNtip	"tip"

#define XdtQkHelpCHelp	"Help"
#define XdtQkHelpCHint	"Hint"
#define XdtQkHelpCText	"Text"
#define XdtQkHelpCTip	"Tip"

#define XdtNbgColorFromParent	"bgColorFromParent"
#define XdtNbgTextureFromParent	"bgTextureFromParent"
#define XdtNhintInTip		"hintInTip"

#define XdtCBgColorFromParent	"BgColorFromParent"
#define XdtCBgTextureFromParent	"BgTextureFromParent"
#define XdtCHintInTip		"HintInTip"

#define XdtDefaultFont "-adobe-helvetica-medium-r-normal-*-*-120-75-75-p-*-iso8859-1"

#define XdtLOCK_APP_CONTEXT
#define XdtLOCK_GLOBAL
#define XdtUNLOCK_APP_CONTEXT
#define XdtUNLOCK_GLOBAL


typedef struct _XdtQkHelpAgentStruct  XdtQkHelpAgentStruct ,*XdtQkHelpAgentPtr ;
typedef struct _XdtQkHelpApplStruct   XdtQkHelpApplStruct  ,*XdtQkHelpApplPtr  ;
typedef struct _XdtQkHelpGMgrStruct   XdtQkHelpGMgrStruct  ,*XdtQkHelpGMgrPtr  ;
typedef struct _XdtQkHelpHintStruct   XdtQkHelpHintStruct  ,*XdtQkHelpHintPtr  ;
typedef struct _XdtQkHelpTipStruct    XdtQkHelpTipStruct   ,*XdtQkHelpTipPtr   ;
typedef struct _XdtQkHelpScreenStruct XdtQkHelpScreenStruct,*XdtQkHelpScreenPtr;
typedef struct _XdtQkHelpShellStruct  XdtQkHelpShellStruct ,*XdtQkHelpShellPtr ;

typedef struct _XdtQkHelpDisplayStruct
   XdtQkHelpDisplayStruct, *XdtQkHelpDisplayPtr;

struct _XdtQkHelpTipStruct
{
   String       text;
   int          textLength;
   int          textWidth;
   Pixel        background;
   Pixel        foreground;
   XFontStruct* fontRecPtr;
   Boolean      hintInTip;
   Pixel        borderColor;
   Dimension    borderWidth;
};

struct _XdtQkHelpHintStruct
{
   String       text;
   int          textLength;
   int          textWidth;
   Pixel        background;
   Pixel        foreground;
   XFontStruct* fontRecPtr;
   Boolean      bgColorFromParent;
   Boolean      bgTextureFromParent;
};

struct _XdtQkHelpGMgrStruct
{
   Widget            currentGadget;
   XdtQkHelpAgentPtr currentAgent;
   Boolean           gadgetAware;
};

struct _XdtQkHelpShellStruct
{
   XdtQkHelpScreenPtr qkHelpScreenPtr;
   Widget             w;
   Widget             hintWidget;
   Boolean            hintMapped;
   Boolean            hintTextured;
   XtIntervalId       hintActiveTimerId;
};

struct _XdtQkHelpAgentStruct
{
   XdtQkHelpScreenPtr    qkHelpScreenPtr;
   XdtQkHelpShellPtr     qkHelpShellPtr;
   Widget                w;
   XdtQkHelpTipStruct    tip;
   XdtQkHelpHintStruct   hint;
   XdtQkHelpGMgrStruct   gMgr;
};

/*
 * The following data structure represents the QkHelp info associated with
 * a given application
 */
struct _XdtQkHelpApplStruct
{
   Boolean  installed;
   
   XrmQuark appNameQuark;		/* Quark of application name  */
   XrmQuark appClassQuark;		/* Quark of application class */
   XrmQuark xdtQkHelpNameQuark;		/* Quark of "xdtQkHelp"       */
   XrmQuark xdtQkHelpClassQuark;	/* Quark of "XdtQkHelp"       */

   XContext agentContextId;		/* Hash lookup of information */
   XContext shellContextId;		/* Hash lookup of information */

   XdtQkHelpDisplayPtr qkHelpDisplayPtr;	/* List of display data */
   unsigned long       qkHelpDisplayCount;	/* Number of elements   */
   
   int                      helpBufferSize;	/* Size of the help struct */
   XtResourceList           helpResources;	/* Resources to retrieve   */
   Cardinal                 numHelpResources;	/* Num of help resources   */
   XdtQkHelpContextHelpProc contextHelpProc;
};

/*
 * The following data structure represents the QkHelp info associated with
 * a given display
 */
struct _XdtQkHelpDisplayStruct
{
   Display*      displayPtr;		/* The actual display data          */
   XtIntervalId  tipPopupTimerId;	/* Timer: inactive to active tip    */
   XtIntervalId  tipActiveTimerId;	/* Timer: active to inactive tip    */
   unsigned long tipPopupInterval;	/* Time for tipPopupTimerId         */
   unsigned long tipActiveInterval;	/* Time for tipActiveTimerId        */
   int           tipXOffset;		/* X Offset when mapping tip        */
   int           tipYOffset;		/* Y Offset when mapping tip        */
   unsigned long hintActiveInterval;	/* Time to unmap hint once inactive */
   unsigned long shellRefCount;		/* Number of shells using this      */
   
   XdtQkHelpAgentPtr   currentAgentPtr;		/* Agent with mouse cursor  */
   XdtQkHelpScreenPtr  qkHelpScreenTable;	/* Table to screen data     */
   unsigned long       qkHelpScreenCount;	/* Number of elements       */
   
   XdtQkHelpDisplayPtr next;
};

/*
 * The following data structure represents the QkHelp info associated with
 * a given screen
 */
struct _XdtQkHelpScreenStruct
{
   Screen* screenPtr;	/* The actual screen info          */
   int     screenNum;   /* The actual screen number        */
   Widget  tipShell;	/* The shell of the tip window     */
   Widget  tipWidget;	/* The text area of the tip window */
   Boolean tipMapped;   /* Is the tip window mapped?       */
   GC      gc;		/* GC for drawing on this screen   */
   
   XdtQkHelpDisplayPtr qkHelpDisplayPtr;	/* Reference to display info */
};


/*
 * Private prototypes
 */
static Boolean _AgentBind _((
   Widget             w,
   XdtQkHelpShellPtr  qh_shell_ptr ));
   
static XdtQkHelpAgentStruct* _AgentCreate _(( 
   Widget                 w,
   XdtQkHelpShellStruct*  qh_shell_ptr ));
   
static void _AgentDestroyCB _((
   Widget    w, 
   XtPointer cd, 
   XtPointer cbd ));

static void _AgentGadgetAware _((
   Widget             w,
   XdtQkHelpShellPtr  qh_shell_ptr ));
   
static void _DetermineShellAndMain _(( 
   Widget  w, 
   Widget* shell_widget,
   Widget* main_widget ));
   
static void _DisplayBind _((
   Display*             display_ptr,
   XdtQkHelpDisplayPtr* qh_display_ptr_ptr ));
   
static void _DisplayDestroy _(( 
   XdtQkHelpDisplayPtr qh_display_ptr ));
   
static void _EnterOrLeaveWindowEH _((
   Widget    w,
   XtPointer cd,
   XEvent*   event,
   Boolean*  cont ));
   
static XdtQkHelpAgentPtr _GetAgent _(( 
   Widget w ));
   
#ifdef USE_MOTIF

static void _HelpCB _((
   Widget    w,
   XtPointer cd,
   XtPointer cbd ));

#endif
   
static void _HelpCallCallback _((
   Widget         w,
   XdtStringConst help_text ));
   
static XdtStringConst _HelpGetText _(( 
   Widget w ));
   
static void _HintDeactivateTO _(( 
   XtPointer     cd, 
   XtIntervalId* id ));

static void _HintDestroyCB _((
   Widget    w, 
   XtPointer cd, 
   XtPointer cbd ));

static void _HintEnterWindow _((
   XdtQkHelpAgentPtr qh_agent_ptr ));

static void _HintExposeEH _(( 
   Widget    w, 
   XtPointer cd, 
   XEvent*   event_ptr, 
   Boolean*  bool_ptr ));

static void _HintSetup _(( 
   XdtQkHelpAgentStruct* qh_agent_ptr, 
   Widget                w, 
   String                hint_text ));

static void _InitializeGlobal _(( Display* ));

static void _InstallOnDisplayAndScreen _(( 
   Widget               w,
   XdtQkHelpDisplayPtr* qh_display_ptr_ptr,
   XdtQkHelpScreenPtr*  qh_screen_ptr_ptr ));
    
static Boolean _InstallOnWidgetTree _((
   Widget            w,
   XdtQkHelpShellPtr shell_ptr ));

static void _LeaveWindow _((
   XdtQkHelpAgentPtr qh_agent_ptr,
   Widget            w ));

static void _MotionEH _((
   Widget    w,
   XtPointer cd,
   XEvent*   event,
   Boolean*  cont ));
   
static void _ScreenBind _(( 
   Screen*             screen_ptr, 
   XdtQkHelpDisplayPtr qh_display_ptr, 
   XdtQkHelpScreenPtr* qh_screen_ptr_ptr ));

static  XdtQkHelpShellPtr _ShellBind _((
   Widget             shell,
   Widget             hint_parent,
   Widget*            hint_widget_ptr,
   XdtQkHelpScreenPtr qh_screen_ptr ));
   
static void _ShellDestroyCB _((
   Widget    w, 
   XtPointer cd, 
   XtPointer cbd ));

static void _TipDeactivateTO _(( 
   XtPointer     cd, 
   XtIntervalId* id ));

static void _TipEnterWindow _((
   XdtQkHelpAgentPtr qh_agent_ptr ));

static void _TipExposeEH _(( 
   Widget    w, 
   XtPointer cd, 
   XEvent*   event_ptr, 
   Boolean*  bool_ptr ));

static void _TipPopupTO _(( 
   XtPointer     cd, 
   XtIntervalId* not_used ));
   
static void _TipSetup _(( 
   XdtQkHelpAgentStruct* qh_agent_ptr, 
   Widget                w, 
   String                tip_text ));

/*----------------------------------------------------------------------------*/

/*
 * Global data
 */
static XdtQkHelpApplStruct _appl_data;

/*----------------------------------------------------------------------------*/

/*
 * Private functions
 */
#if NeedFunctionPrototypes
static Boolean _AgentBind(
   Widget             w,
   XdtQkHelpShellPtr  qh_shell_ptr )
#else
static Boolean _AgentBind( w, qh_shell_ptr )
   Widget             w;
   XdtQkHelpShellPtr  qh_shell_ptr;
#endif
{
   XtResource text_resource =
   {
      XdtQkHelpNtext, XdtQkHelpCText, XtRString, 
      sizeof( String ), 0, XtRImmediate, NULL
   };
  
   String                tip_text;
   String                hint_text;
   XdtQkHelpAgentStruct* qh_agent_ptr = NULL;
   Boolean               gadget_agent = FALSE;
   
   if (XtIsRectObj( w ) && !XtIsShell( w ))
   {
      XPointer data;
      int      not_found;
      
      not_found = XFindContext( 
         XtDisplayOfObject( w ),
         (XID)w,
         _appl_data.agentContextId,
         &data );
   
      if (not_found)
      {
	 XdtQGetLibraryResource( 
	    w, 
	    &tip_text, 
	    _appl_data.appNameQuark, 
	    _appl_data.appClassQuark,
	    _appl_data.xdtQkHelpNameQuark,
	    _appl_data.xdtQkHelpClassQuark,
	    XdtQkHelpNtip,
	    XdtQkHelpCTip,
	    &text_resource,
	    (Cardinal) 1 );
	 XdtQGetLibraryResource( 
	    w, 
	    &hint_text, 
	    _appl_data.appNameQuark, 
	    _appl_data.appClassQuark,
	    _appl_data.xdtQkHelpNameQuark,
	    _appl_data.xdtQkHelpClassQuark,
	    XdtQkHelpNhint,
	    XdtQkHelpCHint,
	    &text_resource,
	    (Cardinal) 1 );

	 /*
	  * tip_text should be copied (or something) since we do not own the
	  * memory
	  */
	 if (tip_text || hint_text)
	 {
            qh_agent_ptr = _AgentCreate( w, qh_shell_ptr );

            _TipSetup ( qh_agent_ptr, w, tip_text  );
            _HintSetup( qh_agent_ptr, w, hint_text );

            if (XtIsWidget( w ))
            {
   	       XtAddEventHandler( 
                  w, 
                  EnterWindowMask | LeaveWindowMask,
                  False,
                  _EnterOrLeaveWindowEH,
                  (XtPointer) qh_agent_ptr );
            }
            else
            {
               gadget_agent = TRUE;
            }             

	 }
	 
	 #ifdef USE_MOTIF
	 {
            String help_text;
            
	    XdtQGetLibraryResource( 
	       w, 
	       &help_text, 
	       _appl_data.appNameQuark, 
	       _appl_data.appClassQuark,
	       _appl_data.xdtQkHelpNameQuark,
	       _appl_data.xdtQkHelpClassQuark,
	       XdtQkHelpNhelp,
	       XdtQkHelpCHelp,
	       &text_resource,
	       (Cardinal) 1 );
	    
	    if (help_text)
	    {
	       XtAddCallback( w, XmNhelpCallback, _HelpCB, NULL );
	    }
	 }
	 #endif
      }
   }
   return gadget_agent;
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static XdtQkHelpAgentPtr _AgentCreate( 
   Widget            w,
   XdtQkHelpShellPtr qh_shell_ptr )
#else
static XdtQkHelpAgentPtr _AgentCreate( w, qh_shell_ptr )
   Widget            w;
   XdtQkHelpShellPtr qh_shell_ptr ;
#endif
{
   XdtQkHelpAgentPtr   qh_agent_ptr;

   qh_agent_ptr = XtNew( XdtQkHelpAgentStruct );
   memset( (void*) qh_agent_ptr, 0, sizeof( XdtQkHelpAgentStruct ) );
   qh_agent_ptr->w               = w;
   qh_agent_ptr->qkHelpShellPtr  = qh_shell_ptr;
   qh_agent_ptr->qkHelpScreenPtr = qh_shell_ptr->qkHelpScreenPtr;

   XSaveContext( 
      XtDisplayOfObject( w ), 
      (XID) w, 
      _appl_data.agentContextId, 
      (XPointer) qh_agent_ptr );
      
   XtAddCallback( 
      w, 
      XtNdestroyCallback, 
      _AgentDestroyCB,
      (XtPointer) qh_agent_ptr );
   
   return qh_agent_ptr;
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _AgentDestroyCB(
   Widget    w, 
   XtPointer cd, 
   XtPointer cbd )
#else
static void _AgentDestroyCB( w, cd, cbd )
   Widget    w;
   XtPointer cd;
   XtPointer cbd;
#endif
{
   XdtQkHelpAgentStruct* qh_agent_ptr = (XdtQkHelpAgentStruct*) cd;
   
   assert( qh_agent_ptr );
   
   XDeleteContext( XtDisplay( w ), (XID) w, _appl_data.agentContextId );
   
   if (qh_agent_ptr->tip.text)
   {
      XtFree( (void*) qh_agent_ptr->tip.text );
   }
   if (qh_agent_ptr->hint.text)
   {
      XtFree( (void*) qh_agent_ptr->hint.text );
   }
   XtFree( (void*) qh_agent_ptr );   
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _AgentGadgetAware(
   Widget             w,
   XdtQkHelpShellPtr  qh_shell_ptr )
#else
static void _AgentGadetAware( w, qh_shell_ptr )
   Widget             w;
   XdtQkHelpShellPtr  qh_shell_ptr;
#endif
{
   XPointer              data;
   int                   not_found;
   XdtQkHelpAgentStruct* qh_agent_ptr;
   
   not_found = XFindContext( 
      XtDisplayOfObject( w ),
      (XID)w,
      _appl_data.agentContextId,
      &data );
   
   if (not_found)
   {
      qh_agent_ptr = _AgentCreate( w, qh_shell_ptr );

      XtAddEventHandler( 
         w, 
         EnterWindowMask | LeaveWindowMask,
         False,
         _EnterOrLeaveWindowEH,
         (XtPointer) qh_agent_ptr );
   }
   else
   {
      qh_agent_ptr = (XdtQkHelpAgentStruct*) data;
   }
   
   if (!qh_agent_ptr->gMgr.gadgetAware)
   {
      XtAddEventHandler( 
	 w, 
	 PointerMotionMask,
	 False,
	 _MotionEH,
	 (XtPointer) qh_agent_ptr );
	 
      qh_agent_ptr->gMgr.gadgetAware = True;
   }
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _DetermineShellAndMain( 
   Widget  w, 
   Widget* shell_widget,
   Widget* main_widget )
#else
static void _DetermineShellAndMain( w, shell_widget, main_widget )
   Widget  w;
   Widget* shell_widget;
   Widget* main_widget;
#endif
{
   if (XtIsWMShell( w ))
   {
      WidgetList children;
      Cardinal   num_children;
      
      *shell_widget = w;
      
      XtVaGetValues( 
         w,
         XtNchildren,    &children,
         XtNnumChildren, &num_children,
         NULL );
      
      if (num_children > 0)
      {
         *main_widget = *children;
      }
      else
      {
         *main_widget = NULL;
      }
   }
   else
   {
      *shell_widget = XtParent( w );
      *main_widget  = w;
      
      while (!XtIsWMShell( *shell_widget ))
      {
         *main_widget = *shell_widget;
	 *shell_widget = XtParent( *shell_widget );
      }
   }
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _DisplayBind(
   Display*             display_ptr,
   XdtQkHelpDisplayPtr* qh_display_ptr_ptr )
#else
static void _DisplayBind( display_ptr, qh_display_ptr_ptr )
   Display*             display_ptr;
   XdtQkHelpDisplayPtr* qh_display_ptr_ptr;
#endif
{
   XdtQkHelpDisplayPtr qh_display_ptr = NULL;
   
   for (qh_display_ptr = _appl_data.qkHelpDisplayPtr;
        qh_display_ptr;
        qh_display_ptr = qh_display_ptr->next)
   {
      if (qh_display_ptr->displayPtr == display_ptr)
      {
         break;
      }
   }
   
   if (!qh_display_ptr)
   {
      int i;
      
      qh_display_ptr = (XdtQkHelpDisplayPtr)XtNew(XdtQkHelpDisplayStruct);

      qh_display_ptr->displayPtr         = display_ptr;
      qh_display_ptr->tipPopupTimerId    = (XtIntervalId) NULL;
      qh_display_ptr->tipActiveTimerId   = (XtIntervalId) NULL;
      qh_display_ptr->tipPopupInterval   = 1000;
      qh_display_ptr->tipActiveInterval  = 300;
      qh_display_ptr->tipXOffset         = 0;
      qh_display_ptr->tipYOffset         = 15;
      qh_display_ptr->hintActiveInterval = 50;
      qh_display_ptr->shellRefCount      = 0;

      qh_display_ptr->currentAgentPtr = NULL;

      qh_display_ptr->qkHelpScreenCount = ScreenCount( display_ptr );

      qh_display_ptr->qkHelpScreenTable = 
         (XdtQkHelpScreenPtr) XtMalloc( 
            sizeof( XdtQkHelpScreenStruct ) * 
               qh_display_ptr->qkHelpScreenCount );
                
      for (i = 0; i < qh_display_ptr->qkHelpScreenCount; i++)
      {
         qh_display_ptr->qkHelpScreenTable[ i ].screenPtr = 
            ScreenOfDisplay( display_ptr, i );
            
         qh_display_ptr->qkHelpScreenTable[ i ].screenNum = -1;
         qh_display_ptr->qkHelpScreenTable[ i ].tipShell  = NULL;
         qh_display_ptr->qkHelpScreenTable[ i ].tipWidget = NULL;
         qh_display_ptr->qkHelpScreenTable[ i ].tipMapped = FALSE;
         qh_display_ptr->qkHelpScreenTable[ i ].gc        = 0;
         
         qh_display_ptr->qkHelpScreenTable[ i ].qkHelpDisplayPtr = 
            qh_display_ptr;
      }
      
      qh_display_ptr->next    = _appl_data.qkHelpDisplayPtr;
      _appl_data.qkHelpDisplayPtr = qh_display_ptr;
   }
   *qh_display_ptr_ptr = qh_display_ptr;
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _DisplayDestroy( 
   XdtQkHelpDisplayPtr qh_display_ptr )
#else
static void _DisplayDestroy( qh_display_ptr )
   XdtQkHelpDisplayPtr qh_display_ptr;
#endif
{
   int                 i;
   XdtQkHelpScreenPtr  qh_screen_ptr = NULL;
   XdtQkHelpDisplayPtr prev;
   
   /*
    * First, destroy all resources associated with individual screens
    */
   for (i = 0; i < qh_display_ptr->qkHelpScreenCount; i++)
   {
      qh_screen_ptr = &qh_display_ptr->qkHelpScreenTable[ i ];
         
      if (qh_screen_ptr->screenNum == -1)
      {
         XtDestroyWidget( qh_screen_ptr->tipShell );
         XFreeGC( qh_display_ptr->displayPtr, qh_screen_ptr->gc );
      }
   }

   if (qh_display_ptr->tipPopupTimerId)
   {
      XtRemoveTimeOut( qh_display_ptr->tipPopupTimerId );
   }
   if (qh_display_ptr->tipActiveTimerId)
   {
      XtRemoveTimeOut( qh_display_ptr->tipActiveTimerId );
   }

   XtFree( (void*) qh_display_ptr->qkHelpScreenTable );
   
   if (_appl_data.qkHelpDisplayPtr == qh_display_ptr)
   {
      _appl_data.qkHelpDisplayPtr = qh_display_ptr->next;
   }
   else
   {
      prev = _appl_data.qkHelpDisplayPtr;
      while (prev->next)
      {
         if (prev->next == qh_display_ptr)
         {
            prev->next = qh_display_ptr->next;
            break;
         }
      }
   }
   XtFree( (void*) qh_display_ptr );
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _EnterOrLeaveWindowEH(
   Widget    w,
   XtPointer cd,
   XEvent*   event,
   Boolean*  cont )
#else
static void _EnterOrLeaveWindowEH( w, cd, event, cont )
   Widget    w;
   XtPointer cd;
   XEvent*   event;
   Boolean*  cont;
#endif
{
   XdtQkHelpAgentStruct* qh_agent_ptr = (XdtQkHelpAgentStruct*) cd;
   assert( qh_agent_ptr );

   if (event->type == EnterNotify)
   {
      XdtQkHelpScreenPtr  qh_screen_ptr;
      XdtQkHelpDisplayPtr qh_display_ptr;
      XCrossingEvent*     enter_event = &(event->xcrossing);
      
      if (enter_event->subwindow) return;

      qh_screen_ptr  = qh_agent_ptr->qkHelpScreenPtr;
      qh_display_ptr = qh_screen_ptr->qkHelpDisplayPtr;
      
      if (qh_agent_ptr->gMgr.gadgetAware)
      {
         Widget gadget;
         
         gadget = XdtGetGadgetChild( w, enter_event->x, enter_event->y );
         if (gadget)
         {
            if (XtIsSensitive( gadget ))
            {
               XdtQkHelpAgentStruct* gadget_agent_ptr = _GetAgent( gadget );
               
               if (gadget_agent_ptr)
               {
                  qh_agent_ptr->gMgr.currentGadget = gadget;
                  qh_agent_ptr->gMgr.currentAgent  = gadget_agent_ptr;
                  
                  qh_agent_ptr = gadget_agent_ptr;
               }
            }
         }
      }
      
      qh_display_ptr->currentAgentPtr = qh_agent_ptr;
      
      _TipEnterWindow( qh_agent_ptr );
      _HintEnterWindow( qh_agent_ptr );
   }
   else if (event->type == LeaveNotify)
   {
      XCrossingEvent* leave_event = &(event->xcrossing);
      if (leave_event->subwindow) return;
      
      if (qh_agent_ptr->gMgr.gadgetAware)
      {
         if (qh_agent_ptr->gMgr.currentAgent)
         {
            XdtQkHelpAgentStruct* gmgr_agent_ptr = qh_agent_ptr;
            
            qh_agent_ptr = qh_agent_ptr->gMgr.currentAgent;
            w = qh_agent_ptr->gMgr.currentGadget;
            
            gmgr_agent_ptr->gMgr.currentAgent = NULL;
            gmgr_agent_ptr->gMgr.currentGadget = NULL;
         }
      }
      _LeaveWindow( qh_agent_ptr, w );
   }
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static XdtQkHelpAgentPtr _GetAgent( 
   Widget w )
#else
static XdtQkHelpAgentPtr _GetAgent( w )
   Widget w;
#endif
{
   XdtQkHelpAgentPtr qh_agent_ptr = NULL;   
   XPointer          data;
   int               not_found;

   if (w)
   {
      not_found = XFindContext( 
	 XtDisplayOfObject( w ),
	 (XID)w,
	 _appl_data.agentContextId,
	 &data );

      if (!not_found)
      {
	 qh_agent_ptr = (XdtQkHelpAgentPtr) data;
      }
   }
   return qh_agent_ptr;
}

/*----------------------------------------------------------------------------*/

#ifdef USE_MOTIF

#if NeedFunctionPrototypes
static void _HelpCB(
   Widget    w,
   XtPointer cd,
   XtPointer cbd )
#else
static void _HelpCB( w, cd, cbd )
   Widget    w;
   XtPointer cd;
   XtPointer cbd;
#endif
{
   XdtStringConst help_text;

   if (_appl_data.contextHelpProc)
   {
      help_text = _HelpGetText( w );
      _HelpCallCallback( w, help_text );
   }
}

#endif

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _HelpCallCallback(
   Widget         w,
   XdtStringConst help_text )
#else
static void _HelpCallCallback( w, help_text )
   Widget         w;
   XdtStringConst help_text;
#endif
{
   XtPointer buffer;

   if (_appl_data.helpBufferSize   && 
       _appl_data.numHelpResources &&
       _appl_data.helpResources    &&
       help_text )
   {
      buffer = XtMalloc( _appl_data.helpBufferSize );

      XdtQGetLibraryResource( 
	 w, 
	 buffer, 
	 _appl_data.appNameQuark, 
	 _appl_data.appClassQuark,
	 _appl_data.xdtQkHelpNameQuark,
	 _appl_data.xdtQkHelpClassQuark,
	 XdtQkHelpNhelp,
	 XdtQkHelpCHelp,
	 _appl_data.helpResources,
	 _appl_data.numHelpResources );
   }
   else
   {
      buffer = NULL;
   }

   _appl_data.contextHelpProc( w, help_text, buffer );
   XtFree( buffer );
}
 
/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static XdtStringConst _HelpGetText( 
   Widget w )
#else
static XdtStringConst _HelpGetText( w )
   Widget w;
#endif
{
   XdtStringConst help_text;
   
   XtResource text_resource =
   {
      XdtQkHelpNtext, XdtQkHelpCText, XtRString, 
      sizeof( String ), 0, XtRImmediate, NULL
   };
  
   XdtQGetLibraryResource( 
      w, 
      &help_text, 
      _appl_data.appNameQuark, 
      _appl_data.appClassQuark,
      _appl_data.xdtQkHelpNameQuark,
      _appl_data.xdtQkHelpClassQuark,
      XdtQkHelpNhelp,
      XdtQkHelpCHelp,
      &text_resource,
      (Cardinal) 1 );

   return help_text;
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _HintDeactivateTO( 
   XtPointer     cd, 
   XtIntervalId* id )
#else
static void _HintDeactivateTO( cd, id )
   XtPointer     cd;
   XtIntervalId* id;
#endif
{
   XdtQkHelpShellStruct* qh_shell_ptr = (XdtQkHelpShellStruct*) cd;
   
   assert( qh_shell_ptr );

   qh_shell_ptr->hintActiveTimerId = (XtIntervalId) NULL;
   XtUnmapWidget( qh_shell_ptr->hintWidget );
   qh_shell_ptr->hintMapped = FALSE;
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _HintDestroyCB(
   Widget    w, 
   XtPointer cd, 
   XtPointer cbd )
#else
static void _HintDestroyCB( w, cd, cbd )
   Widget    w;
   XtPointer cd,;
   XtPointer cbd;
#endif
{
   XdtQkHelpShellStruct* qh_shell_ptr = (XdtQkHelpShellStruct*) cd;
      
   if (qh_shell_ptr->hintActiveTimerId)
   {
      XtRemoveTimeOut( qh_shell_ptr->hintActiveTimerId );
      qh_shell_ptr->hintActiveTimerId = (XtIntervalId) NULL;
   }
   qh_shell_ptr->hintMapped = FALSE;
   qh_shell_ptr->hintWidget = NULL;
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _HintEnterWindow(
   XdtQkHelpAgentPtr qh_agent_ptr )
#else
static void _HintEnterWindow( qh_agent_ptr )
   XdtQkHelpAgentPtr qh_agent_ptr;
#endif
{
   if (qh_agent_ptr->hint.text)
   {
      XdtQkHelpShellStruct* qh_shell_ptr = qh_agent_ptr->qkHelpShellPtr;
      assert( qh_shell_ptr );
      
      if (qh_shell_ptr->hintWidget)
      {
         XdtQkHelpHintStruct* qh_hint_ptr = &qh_agent_ptr->hint;
	 int height, width;

	 height = qh_hint_ptr->fontRecPtr->ascent +
	          qh_hint_ptr->fontRecPtr->descent+2;
	          
	 width  = qh_hint_ptr->textWidth + 4;

         if (qh_hint_ptr->bgColorFromParent)
         {
            XtVaGetValues(
               XtParent( qh_shell_ptr->hintWidget ),
               XtNbackground, &(qh_hint_ptr->background),
               NULL );
         }

	 XtVaSetValues( 
            qh_shell_ptr->hintWidget, 
            XtNbackground,	qh_hint_ptr->background,
            XtNwidth,		(Dimension) width,
            XtNheight,		(Dimension) height,
            NULL );

         if (qh_hint_ptr->bgTextureFromParent)
         {
            if (qh_shell_ptr->hintTextured == FALSE)
            {
               XSetWindowBackgroundPixmap( 
                  XtDisplay( qh_shell_ptr->hintWidget ), 
                  XtWindow( qh_shell_ptr->hintWidget ),
                  ParentRelative );
               qh_shell_ptr->hintTextured = TRUE;
            }
         }
         else
         {
            if (qh_shell_ptr->hintTextured == TRUE)
            {
               XSetWindowBackground( 
                  XtDisplay( qh_shell_ptr->hintWidget ), 
                  XtWindow( qh_shell_ptr->hintWidget ),
                  (unsigned long) qh_hint_ptr->background );
               qh_shell_ptr->hintTextured = FALSE;
            }
         }

         if (qh_shell_ptr->hintMapped)
         {
            if (qh_shell_ptr->hintActiveTimerId)
            {
               XtRemoveTimeOut( qh_shell_ptr->hintActiveTimerId );
               qh_shell_ptr->hintActiveTimerId = (XtIntervalId) NULL;
            }
            XClearArea( 
               XtDisplay( qh_shell_ptr->hintWidget ),
               XtWindow( qh_shell_ptr->hintWidget ),
               0, 0, 0, 0,
               True );
         }
         else
         {
            XtMapWidget( qh_shell_ptr->hintWidget );
            qh_shell_ptr->hintMapped = TRUE;
         }
      }
   }
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _HintExposeEH( 
   Widget    w, 
   XtPointer cd, 
   XEvent*   event_ptr, 
   Boolean*  bool_ptr )
#else
static void _HintExposeEH( w, cd, event_ptr, bool_ptr )
   Widget    w;
   XtPointer cd;
   XEvent*   event_ptr;
   Boolean*  bool_ptr;
#endif
{
   XdtQkHelpDisplayPtr qh_display_ptr = (XdtQkHelpDisplayPtr) cd;
   XdtQkHelpAgentStruct* qh_agent_ptr;
   
   assert( qh_display_ptr );
   
   qh_agent_ptr = qh_display_ptr->currentAgentPtr;
   
   if (qh_agent_ptr && qh_agent_ptr->hint.text)
   {
      if (event_ptr->type == Expose)
      {
         XdtQkHelpScreenPtr qh_screen_ptr = qh_agent_ptr->qkHelpScreenPtr;
         Display*           display_ptr   = qh_display_ptr->displayPtr;
         GC                 gc            = qh_screen_ptr->gc;
         
	 /*
	  * This can be optimized later to change only when needed
	  */
	 XSetFont      ( display_ptr, gc, qh_agent_ptr->hint.fontRecPtr->fid );
	 XSetForeground( display_ptr, gc, qh_agent_ptr->hint.foreground      );

	 XDrawString( 
            display_ptr, 
            XtWindow( w ),
            gc,
            2, 1 + qh_agent_ptr->hint.fontRecPtr->ascent,
            qh_agent_ptr->hint.text,
            qh_agent_ptr->hint.textLength );
      }
   }
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _HintSetup( 
   XdtQkHelpAgentStruct* qh_agent_ptr, 
   Widget                w, 
   String                hint_text )
#else
static void _HintSetup( qh_agent_ptr, w, hint_text )
   XdtQkHelpAgentStruct* qh_agent_ptr;
   Widget                w;
   String                hint_text;
#endif
{
   struct _hint_resources
   {
      Pixel        background;
      Pixel        foreground;
      XFontStruct* fontRecPtr;
      Boolean      bgColorFromParent;
      Boolean      bgTextureFromParent;
   }
   hint_resources;
   
   #define OFFSET_OF( mem ) XtOffsetOf( struct _hint_resources, mem )
   
   static XtResource resource[] =
   {
      {
         XtNbackground, XtCBackground, 
         XtRPixel, sizeof( Pixel ), OFFSET_OF( background ), 
         XtRString, XtDefaultBackground
      },{
         XtNforeground, XtCForeground, 
         XtRPixel, sizeof( Pixel ), OFFSET_OF( foreground ), 
         XtRString, XtDefaultForeground
      },{
         XtNfont, XtCFont,
         XtRFontStruct, sizeof( XFontStruct* ), OFFSET_OF( fontRecPtr ),
         XtRString, XdtDefaultFont
      },{
         XdtNbgColorFromParent, XdtCBgColorFromParent, 
         XtRBoolean, sizeof( Boolean ), OFFSET_OF( bgColorFromParent ), 
         XtRBoolean, (XtPointer) TRUE
      },{
         XdtNbgTextureFromParent, XdtCBgTextureFromParent, 
         XtRBoolean, sizeof( Boolean ), OFFSET_OF( bgTextureFromParent ), 
         XtRBoolean, (XtPointer) FALSE
      }
   };
   #undef OFFSET_OF
   
   XdtQkHelpHintStruct* qh_hint_ptr = NULL;
   
   if (hint_text && *hint_text)
   {
      qh_hint_ptr = &qh_agent_ptr->hint;
      
      qh_hint_ptr->text = XtNewString( hint_text );

      XdtQGetLibraryResource( 
	 w, 
	 &hint_resources, 
	 _appl_data.appNameQuark, 
	 _appl_data.appClassQuark,
	 _appl_data.xdtQkHelpNameQuark,
	 _appl_data.xdtQkHelpClassQuark,
	 XdtQkHelpNhint,
	 XdtQkHelpCHint,
	 resource,
	 (Cardinal) XtNumber( resource ) );

      qh_hint_ptr->background           = hint_resources.background;
      qh_hint_ptr->foreground           = hint_resources.foreground;
      qh_hint_ptr->fontRecPtr           = hint_resources.fontRecPtr;
      qh_hint_ptr->bgColorFromParent    = hint_resources.bgColorFromParent;
      qh_hint_ptr->bgTextureFromParent  = hint_resources.bgTextureFromParent;

      qh_hint_ptr->textLength = strlen( qh_hint_ptr->text );
      
      qh_hint_ptr->textWidth = XTextWidth( 
         qh_hint_ptr->fontRecPtr,
         qh_hint_ptr->text,
         qh_hint_ptr->textLength );
   }
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _InitializeGlobal( 
   Display* display_ptr )
#else
static void _InitializeGlobal( display_ptr )
   Display* display_ptr;
#endif
{
   XdtLOCK_GLOBAL;
   if (!_appl_data.installed)
   {
      String app_name;
      String app_class;
      
      XtGetApplicationNameAndClass( display_ptr, &app_name, &app_class );
      _appl_data.appNameQuark        = XrmStringToQuark( app_name    );
      _appl_data.appClassQuark       = XrmStringToQuark( app_class   );
      _appl_data.xdtQkHelpNameQuark  = XrmStringToQuark( "xdtQkHelp" );
      _appl_data.xdtQkHelpClassQuark = XrmStringToQuark( "XdtQkHelp" );

      _appl_data.agentContextId      = XUniqueContext();
      _appl_data.shellContextId      = XUniqueContext();
     
      _appl_data.qkHelpDisplayPtr   = NULL;
      _appl_data.qkHelpDisplayCount = 0;

      _appl_data.helpBufferSize   = 0;
      _appl_data.helpResources    = NULL;
      _appl_data.numHelpResources = 0;
      _appl_data.contextHelpProc  = NULL;

      _appl_data.installed = TRUE;
   }
   XdtUNLOCK_GLOBAL;
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _InstallOnDisplayAndScreen( 
   Widget               w,
   XdtQkHelpDisplayPtr* qh_display_ptr_ptr,
   XdtQkHelpScreenPtr*  qh_screen_ptr_ptr )
#else
static void _InstallOnDisplayAndScreen( w, qh_display_ptr_ptr, 
                                        qh_screen_ptr_ptr )
   Widget               w;
   XdtQkHelpDisplayPtr* qh_display_ptr_ptr;
   XdtQkHelpScreenPtr*  qh_screen_ptr_ptr;
#endif
{
   Display*            display_ptr = XtDisplay( w );
   Screen*             screen_ptr  = XtScreen ( w );
   XdtQkHelpDisplayPtr qh_display_ptr;
   XdtQkHelpScreenPtr  qh_screen_ptr;
   
   _InitializeGlobal( display_ptr );
   
   _DisplayBind( display_ptr, &qh_display_ptr );
   _ScreenBind( screen_ptr, qh_display_ptr, &qh_screen_ptr );
   
   *qh_display_ptr_ptr = qh_display_ptr;
   *qh_screen_ptr_ptr  = qh_screen_ptr;
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static Boolean _InstallOnWidgetTree(
   Widget             w,
   XdtQkHelpShellPtr  qh_shell_ptr )
#else
static Boolean _InstallOnWidgetTree( w, qh_shell_ptr )
   Widget             w;
   XdtQkHelpShellPtr  qh_shell_ptr;
#endif
{
   int     i;
   Boolean gadget_manager_needed;
   
   gadget_manager_needed = _AgentBind( w, qh_shell_ptr );
   
   if (XtIsComposite( w ))
   {
      WidgetList children;
      Cardinal   num_children;
      Boolean    add_gadget_manager = FALSE;
      
      XtVaGetValues( 
         w,
         XtNchildren,    &children,
         XtNnumChildren, &num_children,
         NULL );
         
      for (i = 0; i < num_children; i++)
      {
         Boolean gadget_installed;
         gadget_installed = _InstallOnWidgetTree( children[i], qh_shell_ptr );
         if (gadget_installed)
         {
            add_gadget_manager = gadget_installed;
         }
      }
      if (add_gadget_manager)
      {
         _AgentGadgetAware( w, qh_shell_ptr );
      }
   }

   if (XtIsWidget( w ))
   {
      Cardinal num_popups;
      Widget   popup;
      
      num_popups = XdtQkHelpGetNumPopups( w );
      for (i = 0; i < num_popups; i++)
      {
         popup = XdtQkHelpGetPopupList( w )[ i ];
         if (!XtIsWMShell( popup ))
         {
            _InstallOnWidgetTree( popup, qh_shell_ptr );
         }
      }
   }
   return gadget_manager_needed;
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _LeaveWindow(
   XdtQkHelpAgentPtr qh_agent_ptr,
   Widget            w )
#else
static void _LeaveWindow( qh_agent_ptr, w )
   XdtQkHelpAgentPtr qh_agent_ptr;
   Widget            w;
#endif
{
   XdtQkHelpShellPtr   qh_shell_ptr;
   XdtQkHelpScreenPtr  qh_screen_ptr;
   XdtQkHelpDisplayPtr qh_display_ptr;
   
   assert( qh_agent_ptr );
   
   qh_shell_ptr = qh_agent_ptr->qkHelpShellPtr;
   assert( qh_shell_ptr );
   
   qh_screen_ptr = qh_shell_ptr->qkHelpScreenPtr;
   assert( qh_screen_ptr );
   
   qh_display_ptr = qh_screen_ptr->qkHelpDisplayPtr;
   assert( qh_display_ptr );
   
   if (qh_display_ptr->currentAgentPtr)
   {
      if (qh_display_ptr->currentAgentPtr->hint.text)
      {
	 if (qh_shell_ptr->hintWidget)
	 {
	    if (qh_shell_ptr->hintActiveTimerId)
	    {
	       XtRemoveTimeOut( qh_shell_ptr->hintActiveTimerId );
	    }
	    
	    qh_shell_ptr->hintActiveTimerId = XtAppAddTimeOut( 
               XtWidgetToApplicationContext( qh_agent_ptr->w ), 
               qh_display_ptr->hintActiveInterval, 
               _HintDeactivateTO, 
               qh_shell_ptr );
	 }
      }

      if (qh_display_ptr->tipPopupTimerId)
      {
         XtRemoveTimeOut( qh_display_ptr->tipPopupTimerId );
      }

      qh_display_ptr->tipPopupTimerId = (XtIntervalId) NULL;

      XtPopdown( qh_screen_ptr->tipShell );

      qh_display_ptr->tipActiveTimerId = XtAppAddTimeOut( 
         XtWidgetToApplicationContext( qh_agent_ptr->w ), 
         qh_display_ptr->tipActiveInterval, 
         _TipDeactivateTO, 
         qh_screen_ptr );

      if (qh_display_ptr->currentAgentPtr->w == w)
      {
         qh_display_ptr->currentAgentPtr = NULL;
      }
   }
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _MotionEH(
   Widget    w,
   XtPointer cd,
   XEvent*   event,
   Boolean*  cont )   
#else
static void _MotionEH( w, cd, event, cont );
   Widget    w;
   XtPointer cd;
   XEvent*   event;
   Boolean*  cont;   
#endif
{
   Widget              gadget         = NULL;
   XdtQkHelpAgentPtr   qh_agent_ptr   = (XdtQkHelpAgentPtr) cd;
   XdtQkHelpScreenPtr  qh_screen_ptr;
   XdtQkHelpDisplayPtr qh_display_ptr;

   assert( qh_agent_ptr );
   assert( qh_agent_ptr->gMgr.gadgetAware );
   
   qh_screen_ptr = qh_agent_ptr->qkHelpScreenPtr;
   assert( qh_screen_ptr );
   
   qh_display_ptr = qh_screen_ptr->qkHelpDisplayPtr;
   assert( qh_display_ptr );

   if (event->type == MotionNotify)
   {
      if (event->xmotion.subwindow != 0) return;

      gadget = XdtGetGadgetChild( w, event->xmotion.x, event->xmotion.y );
         
      if (gadget)
      {
         if (gadget != qh_agent_ptr->gMgr.currentGadget)
         {
            if (qh_agent_ptr->gMgr.currentGadget)
            {
                _LeaveWindow( 
                   qh_agent_ptr->gMgr.currentAgent,
                   qh_agent_ptr->gMgr.currentGadget );
             
                qh_agent_ptr->gMgr.currentAgent = NULL;
                qh_agent_ptr->gMgr.currentGadget = NULL;
            }
            
            if (XtIsSensitive( gadget ))
            {
               XdtQkHelpAgentPtr gadget_agent_ptr;
               
               gadget_agent_ptr = _GetAgent( gadget );
               if (gadget_agent_ptr)
               {
                  qh_agent_ptr->gMgr.currentAgent = gadget_agent_ptr;
                  qh_agent_ptr->gMgr.currentGadget = gadget;
                  
                  qh_display_ptr->currentAgentPtr = gadget_agent_ptr;
                  
                  _TipEnterWindow( gadget_agent_ptr );
                  _HintEnterWindow( gadget_agent_ptr );
               }
            }
         }
      }
      else
      {
         if (qh_agent_ptr->gMgr.currentAgent)
         {
             _LeaveWindow( 
                qh_agent_ptr->gMgr.currentAgent,
                qh_agent_ptr->gMgr.currentGadget );
             
             qh_agent_ptr->gMgr.currentAgent = NULL;
             qh_agent_ptr->gMgr.currentGadget = NULL;

             qh_display_ptr->currentAgentPtr = qh_agent_ptr;
      
             _TipEnterWindow( qh_agent_ptr );
             _HintEnterWindow( qh_agent_ptr );
         }
      }
   }
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _ScreenBind( 
   Screen*             screen_ptr, 
   XdtQkHelpDisplayPtr qh_display_ptr, 
   XdtQkHelpScreenPtr* qh_screen_ptr_ptr )
#else
static void _ScreenBind( screen_ptr, qh_display_ptr, qh_screen_ptr_ptr )
   Screen*             screen_ptr;
   XdtQkHelpDisplayPtr qh_display_ptr;
   XdtQkHelpScreenPtr* qh_screen_ptr_ptr;
#endif
{
   int                i;
   XdtQkHelpScreenPtr qh_screen_ptr = NULL;
   
   for (i = 0; i < qh_display_ptr->qkHelpScreenCount; i++)
   {
      if (qh_display_ptr->qkHelpScreenTable[ i ].screenPtr == screen_ptr)
      {
         qh_screen_ptr = &qh_display_ptr->qkHelpScreenTable[ i ];
         
         if (qh_screen_ptr->screenNum == -1)
         {
            qh_screen_ptr->screenNum = i;
            
            assert( qh_display_ptr->displayPtr );
            qh_screen_ptr->tipShell = XtAppCreateShell( 
               "tipShell",
               "TipShell",
               overrideShellWidgetClass,
               qh_display_ptr->displayPtr,
               NULL, (Cardinal) 0 );

            XtVaSetValues( 
               qh_screen_ptr->tipShell, 
               XtNallowShellResize, TRUE, 
               NULL );
   
            qh_screen_ptr->tipWidget = XtVaCreateManagedWidget(
               "tipLabel",
               coreWidgetClass,
               qh_screen_ptr->tipShell,
               NULL );

            XtAddEventHandler( 
               qh_screen_ptr->tipWidget,
               ExposureMask,
               False,
               _TipExposeEH,
               (XtPointer) qh_display_ptr );

            qh_screen_ptr->gc = XCreateGC(
               qh_display_ptr->displayPtr,
               RootWindowOfScreen( screen_ptr ),
               0,
               NULL );
         }
      }
   }
   *qh_screen_ptr_ptr = qh_screen_ptr;
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static XdtQkHelpShellPtr _ShellBind(
   Widget             shell,
   Widget             hint_parent,
   Widget*            hint_widget_ptr,
   XdtQkHelpScreenPtr qh_screen_ptr )
#else
static XdtQkHelpShellPtr _ShellBind( shell, hint_parent, hint_widget_ptr,
                                     qh_screen_ptr )
   Widget             shell;
   Widget             hint_parent;
   Widget*            hint_widget_ptr;
   XdtQkHelpScreenPtr qh_screen_ptr;
#endif
{
   XdtQkHelpShellStruct* qh_shell_ptr;
   XPointer              data;
   int                   not_found;
   
   not_found = XFindContext( 
      XtDisplay( shell ), 
      (XID) shell,
      _appl_data.shellContextId,
      &data );
   
   if (not_found)
   {
      qh_shell_ptr = XtNew( XdtQkHelpShellStruct );

      qh_shell_ptr->qkHelpScreenPtr   = qh_screen_ptr;
      qh_shell_ptr->w                 = shell;
      qh_shell_ptr->hintWidget        = NULL;
      qh_shell_ptr->hintMapped        = FALSE;
      qh_shell_ptr->hintTextured      = FALSE;
      qh_shell_ptr->hintActiveTimerId = (XtIntervalId) NULL;

      qh_screen_ptr->qkHelpDisplayPtr->shellRefCount += 1;

      XSaveContext( 
	 XtDisplay( shell ), 
	 (XID) shell, 
	 _appl_data.shellContextId, 
	 (XPointer) qh_shell_ptr );

      XtAddCallback( 
	 shell, 
	 XtNdestroyCallback, 
	 _ShellDestroyCB,
	 (XtPointer) qh_shell_ptr );

      if (hint_widget_ptr)
      {
	 if (XtIsComposite( hint_parent ))
	 {
            Dimension h,w;
            XdtQkHelpDisplayPtr qh_display_ptr = qh_screen_ptr->qkHelpDisplayPtr;
            assert( qh_display_ptr );

	    qh_shell_ptr->hintWidget = XtVaCreateManagedWidget(
	       "hintLabel",
	       coreWidgetClass,
	       hint_parent,
	       XtNmappedWhenManaged,	FALSE,
	       NULL );

            XtVaGetValues( 
               qh_shell_ptr->hintWidget,
               XtNheight, &h,
               XtNwidth,  &w,
               NULL );

            if (!h) h = 1;
            if (!w) w = 1;

            XtVaSetValues( 
               qh_shell_ptr->hintWidget,
               XtNheight, h,
               XtNwidth,  w,
               NULL );
            
	    XtAddCallback( 
               qh_shell_ptr->hintWidget, 
               XtNdestroyCallback, 
               _HintDestroyCB,
               (XtPointer) qh_shell_ptr );

	    XtAddEventHandler( 
	       qh_shell_ptr->hintWidget,
	       ExposureMask,
	       False,
	       _HintExposeEH,
	       (XtPointer) qh_display_ptr );

            *hint_widget_ptr = qh_shell_ptr->hintWidget;
	 }
	 else
	 {
            *hint_widget_ptr = NULL;
	 }
      }
   }
   else
   {
      qh_shell_ptr = (XdtQkHelpShellPtr) data;
   }
   return qh_shell_ptr;
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _ShellDestroyCB( 
   Widget    w, 
   XtPointer cd, 
   XtPointer cbd )
#else
static void _ShellDestroyCB( w, cd, cbd )
   Widget    w;
   XtPointer cd;
   XtPointer cbd;
#endif
{
   XdtQkHelpShellStruct* qh_shell_ptr = (XdtQkHelpShellStruct*) cd;
      
   XDeleteContext( XtDisplay( w ), (XID) w, _appl_data.shellContextId );

   if (qh_shell_ptr->hintActiveTimerId)
   {
      XtRemoveTimeOut( qh_shell_ptr->hintActiveTimerId );
   }
   
   qh_shell_ptr->qkHelpScreenPtr->qkHelpDisplayPtr->shellRefCount -= 1;
   if (!qh_shell_ptr->qkHelpScreenPtr->qkHelpDisplayPtr->shellRefCount)
   {
      _DisplayDestroy( qh_shell_ptr->qkHelpScreenPtr->qkHelpDisplayPtr );
   }

   XtFree( (void*) qh_shell_ptr );
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _TipDeactivateTO( 
   XtPointer     cd, 
   XtIntervalId* id )
#else
static void _TipDeactivateTO( cd, id )
   XtPointer     cd; 
   XtIntervalId* id;
#endif
{
   XdtQkHelpScreenPtr  qh_screen_ptr = (XdtQkHelpScreenPtr) cd;
   XdtQkHelpDisplayPtr qh_display_ptr;
   
   assert( qh_screen_ptr );
   
   qh_display_ptr = qh_screen_ptr->qkHelpDisplayPtr;
   assert( qh_display_ptr );
   
   qh_display_ptr->tipActiveTimerId = (XtIntervalId) NULL;
   qh_screen_ptr->tipMapped = FALSE;
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _TipEnterWindow(
   XdtQkHelpAgentPtr qh_agent_ptr )
#else
static void _TipEnterWindow( qh_agent_ptr )
   XdtQkHelpAgentPtr qh_agent_ptr;
#endif
{
   XdtQkHelpScreenPtr qh_screen_ptr;
   
   qh_screen_ptr = (XdtQkHelpScreenPtr) qh_agent_ptr->qkHelpScreenPtr;
   assert( qh_screen_ptr );

   if (qh_agent_ptr->tip.text)
   {
      XdtQkHelpDisplayPtr qh_display_ptr = 
         (XdtQkHelpDisplayPtr) qh_screen_ptr->qkHelpDisplayPtr;
      assert( qh_display_ptr );
      
      if (qh_screen_ptr->tipMapped)
      {
         if (qh_display_ptr->tipActiveTimerId)
         {
            XtRemoveTimeOut( qh_display_ptr->tipActiveTimerId );
            qh_display_ptr->tipActiveTimerId = (XtIntervalId) NULL;
         }
         _TipPopupTO( qh_agent_ptr, NULL );
      }
      else
      {
         if (qh_display_ptr->tipPopupTimerId)
         {
            XtRemoveTimeOut( qh_display_ptr->tipPopupTimerId );
         }
         qh_display_ptr->tipPopupTimerId = XtAppAddTimeOut(
            XtWidgetToApplicationContext( qh_agent_ptr->w ), 
            qh_display_ptr->tipPopupInterval, 
            _TipPopupTO, 
            (XtPointer) qh_agent_ptr );
      }
   }
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _TipExposeEH( 
   Widget    w, 
   XtPointer cd, 
   XEvent*   event_ptr, 
   Boolean*  bool_ptr )
#else
static void _TipExposeEH( w, cd, event_ptr, bool_ptr )
   Widget    w;
   XtPointer cd;
   XEvent*   event_ptr;
   Boolean*  bool_ptr;
#endif
{
   XdtQkHelpDisplayPtr qh_display_ptr = (XdtQkHelpDisplayPtr) cd;
   assert( qh_display_ptr );
   
   if ((event_ptr->type == Expose) && (qh_display_ptr->currentAgentPtr))
   {
      XdtQkHelpAgentPtr  qh_agent_ptr = qh_display_ptr->currentAgentPtr;
      XdtQkHelpScreenPtr qh_screen_ptr;
      Display*           display_ptr;
      GC                 gc;
      
      /*
       * This can be optimized later to change only when needed
       */      
      assert( qh_agent_ptr );
      assert( qh_agent_ptr->tip.text );
      assert( qh_agent_ptr->tip.fontRecPtr->fid );
      
      qh_screen_ptr = qh_agent_ptr->qkHelpScreenPtr;
      assert( qh_screen_ptr );
      
      display_ptr = qh_display_ptr->displayPtr;
      gc = qh_screen_ptr->gc;
      
      XSetFont      ( display_ptr, gc, qh_agent_ptr->tip.fontRecPtr->fid );
      XSetForeground( display_ptr, gc, qh_agent_ptr->tip.foreground      );
      
      XDrawString( 
         display_ptr, 
         XtWindow( w ),
         gc,
         2, 1 + qh_agent_ptr->tip.fontRecPtr->ascent,
         qh_agent_ptr->tip.text,
         qh_agent_ptr->tip.textLength );
         
      if (qh_agent_ptr->tip.hintInTip)
      {
         if (qh_agent_ptr->hint.text)
         {
	    XSetFont      ( display_ptr,gc,qh_agent_ptr->hint.fontRecPtr->fid );
	    XSetForeground( display_ptr,gc,qh_agent_ptr->tip.foreground       );

	    XDrawString( 
               display_ptr, 
               XtWindow( w ),
               gc,
               2, 
               1 + qh_agent_ptr->tip.fontRecPtr->ascent + 
                  qh_agent_ptr->tip.fontRecPtr->descent + 2 +
                  qh_agent_ptr->hint.fontRecPtr->ascent,
               qh_agent_ptr->hint.text,
               qh_agent_ptr->hint.textLength );
         }
      }
   }
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _TipPopupTO( 
   XtPointer     cd, 
   XtIntervalId* not_used )
#else
static void _TipPopupTO( cd, not_used )
   XtPointer     cd;
   XtIntervalId* not_used;
#endif
{
   XdtQkHelpAgentStruct* qh_agent_ptr = (XdtQkHelpAgentStruct*) cd;
   XdtQkHelpScreenPtr    qh_screen_ptr;
   XdtQkHelpDisplayPtr   qh_display_ptr;
   XdtQkHelpTipStruct*   qh_tip_ptr;
   Bool                  bool;
   Window                root_window, child_window;
   int                   root_x, root_y, win_x, win_y;
   unsigned int          keys_buttons;
   
   assert( qh_agent_ptr );
   
   qh_screen_ptr = qh_agent_ptr->qkHelpScreenPtr;
   assert( qh_screen_ptr );
   
   qh_display_ptr = qh_screen_ptr->qkHelpDisplayPtr;
   assert( qh_display_ptr );
   
   qh_tip_ptr = &qh_agent_ptr->tip;
   
   qh_display_ptr->tipPopupTimerId = (XtIntervalId) NULL;
   
   bool = XQueryPointer(
      qh_display_ptr->displayPtr,
      RootWindowOfScreen( qh_screen_ptr->screenPtr ),
      &root_window,
      &child_window,
      &root_x,
      &root_y,
      &win_x,
      &win_y,
      &keys_buttons );

   if (bool)
   {
      int x_pos, y_pos, height, width;
      Screen* screen_ptr = qh_screen_ptr->screenPtr;
      
      height = qh_tip_ptr->fontRecPtr->ascent + 
               qh_tip_ptr->fontRecPtr->descent + 2;
               
      width  = qh_tip_ptr->textWidth + 4;
      
      if (qh_tip_ptr->hintInTip)
      {
         if (qh_agent_ptr->hint.text)
         {
            XdtQkHelpHintPtr qh_hint_ptr=(XdtQkHelpHintPtr) &qh_agent_ptr->hint;
            assert( qh_hint_ptr );
            
            height += qh_hint_ptr->fontRecPtr->ascent + 
                      qh_hint_ptr->fontRecPtr->descent + 2;
               
            if ((qh_hint_ptr->textWidth + 4) > width)
            {
               width = qh_hint_ptr->textWidth + 4;
            }
         }
      }
      
      x_pos = root_x + qh_display_ptr->tipXOffset;
      y_pos = root_y + qh_display_ptr->tipYOffset;
      
      if (x_pos < 0) 
      {
         x_pos = 0;
      }
      else if (x_pos + width > WidthOfScreen( screen_ptr ))
      {
         x_pos = WidthOfScreen( screen_ptr ) - width - 1;
      }
      
      if ((y_pos < 0) || (y_pos + height > HeightOfScreen( screen_ptr )))
      {
         y_pos = root_y - qh_display_ptr->tipYOffset - height - 1;
      }

      XtVaSetValues( 
         qh_screen_ptr->tipWidget, 
         XtNbackground,  qh_tip_ptr->background,
         XtNwidth,	 (Dimension) width,
         XtNheight,	 (Dimension) height,
         NULL );

      XtVaSetValues(
         qh_screen_ptr->tipShell, 
         XtNx,			(Position) x_pos,
         XtNy,			(Position) y_pos,
         XtNborderWidth,	qh_tip_ptr->borderWidth,
         XtNborderColor,	qh_tip_ptr->borderColor,
         NULL );

      XtPopup( qh_screen_ptr->tipShell, XtGrabNone );
      qh_screen_ptr->tipMapped = TRUE;
   }
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
static void _TipSetup( 
   XdtQkHelpAgentStruct* qh_agent_ptr, 
   Widget                w, 
   String                tip_text )
#else
static void _TipSetup( qh_agent_ptr, w, tip_text )
   XdtQkHelpAgentStruct* qh_agent_ptr; 
   Widget                w;
   String                tip_text;
#endif
{
   struct _tip_resources
   {
      Pixel        background;
      Pixel        foreground;
      XFontStruct* fontRecPtr;
      Boolean      hintInTip;
      Pixel        borderColor;
      Dimension    borderWidth;
   }
   tip_resources;
   
   #define OFFSET_OF( mem ) XtOffsetOf( struct _tip_resources, mem )
   
   static XtResource resource[] =
   {
      {
         XtNbackground, XtCBackground, 
         XtRPixel, sizeof( Pixel ), OFFSET_OF( background ), 
         XtRString, "Yellow"
      },{
         XtNforeground, XtCForeground, 
         XtRPixel, sizeof( Pixel ), OFFSET_OF( foreground ), 
         XtRString, "Black"
      },{
         XtNfont, XtCFont,
         XtRFontStruct, sizeof( XFontStruct* ), OFFSET_OF( fontRecPtr ),
         XtRString, XdtDefaultFont
      },{
         XdtNhintInTip, XdtCHintInTip,
         XtRBoolean, sizeof( Boolean ), OFFSET_OF( hintInTip ),
         XtRBoolean, FALSE
      },{
         XtNborderColor, XtCBorderColor,
         XtRPixel, sizeof( Pixel ), OFFSET_OF( borderColor ), 
         XtRString, "Black"
      },{
         XtNborderWidth, XtCBorderWidth,
         XtRDimension, sizeof( Dimension ), OFFSET_OF( borderWidth ), 
         XtRImmediate, (XtPointer) 1
      }
   };
   #undef OFFSET_OF
   
   XdtQkHelpTipStruct* qh_tip_ptr = NULL;
   
   if (tip_text && *tip_text)
   {
      qh_tip_ptr = &qh_agent_ptr->tip;
      
      qh_tip_ptr->text = XtNewString( tip_text );

      XdtQGetLibraryResource( 
	 w, 
	 &tip_resources, 
	 _appl_data.appNameQuark, 
	 _appl_data.appClassQuark,
	 _appl_data.xdtQkHelpNameQuark,
	 _appl_data.xdtQkHelpClassQuark,
	 XdtQkHelpNtip,
	 XdtQkHelpCTip,
	 resource,
	 (Cardinal) XtNumber( resource ) );

      qh_tip_ptr->background  = tip_resources.background;
      qh_tip_ptr->foreground  = tip_resources.foreground;
      qh_tip_ptr->fontRecPtr  = tip_resources.fontRecPtr;
      qh_tip_ptr->hintInTip   = tip_resources.hintInTip;
      qh_tip_ptr->borderColor = tip_resources.borderColor;
      qh_tip_ptr->borderWidth = tip_resources.borderWidth;
     
      qh_tip_ptr->textLength = strlen( qh_tip_ptr->text );
      
      qh_tip_ptr->textWidth = XTextWidth( 
         qh_tip_ptr->fontRecPtr,
         qh_tip_ptr->text,
         qh_tip_ptr->textLength );
   }
}

/*----------------------------------------------------------------------------*/

/*
 * Public functions
 */
 
/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
void XdtQkHelpActivateHelpOnWidget( 
   Widget widget )
#else
void XdtQkHelpActivateHelpOnWidget( widget )
   Widget widget;
#endif
{
   Widget         w;
   XdtStringConst help_text;

   if (_appl_data.contextHelpProc)
   {
      w = widget;
      
      while( w )
      {
         help_text = _HelpGetText( w );
         if (help_text)
         {
            _HelpCallCallback( w, help_text );
            break;
         }
         else if (XtIsWMShell( w ))
         {
            _HelpCallCallback( widget, NULL );
            break;
         }
         
         w = XtParent( w );
      }
   }
}
   
/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
void XdtQkHelpChooseAndActivateHelp( 
   Widget         widget,
   Cursor         cursor, 
   XdtWideBoolean confine_to )
#else
void XdtQkHelpChooseAndActivateHelp( widget, cursor, confine_to )
   Widget         widget;
   Cursor         cursor;
   XdtWideBoolean confine_to;
#endif
{
   XtAppContext app_context;
   Time         event_time;
   Widget       specified_widget = NULL;
   
   event_time = XtLastTimestampProcessed( XtDisplay( widget ) );
   
   XtGrabPointer( 
      widget, 
      False, 
      ButtonPressMask,
      GrabModeAsync,
      GrabModeAsync,
      confine_to ? XtWindow( widget ) : None,
      cursor,
      event_time );
      
   XtGrabKeyboard( 
      widget, 
      False, 
      GrabModeAsync,
      GrabModeAsync,
      event_time );
   
   app_context = XtWidgetToApplicationContext( widget );
   
   while (True)
   {
      XEvent event;
      
      XtAppNextEvent( app_context, &event );

      if (event.type == ButtonPress || event.type == ButtonRelease)
      {
         specified_widget =
            XdtGetDescendantAtPos( widget, event.xbutton.x, event.xbutton.y );

         event_time = event.xbutton.time;         
         break;
      }
      else if (event.type == KeyPress || event.type == KeyRelease)
      {
         KeySym    key_sym;

         key_sym = XLookupKeysym( &event.xkey, 0 )         ;
         if (key_sym != XK_Escape)
         {
            specified_widget = 
               XdtGetDescendantAtPos( widget, event.xkey.x, event.xkey.y );
         }
         event_time = event.xkey.time;         
         break;
      }
      else
      {
         XtDispatchEvent( &event );
      }
   }
   XtUngrabPointer ( widget, event_time );
   XtUngrabKeyboard( widget, event_time );
   
   XdtQkHelpActivateHelpOnWidget( specified_widget );
}

/*----------------------------------------------------------------------------*/

#if NeedFunctionPrototypes
void XdtQkHelpInstallContextHelp(
   Widget                   top,
   XtResourceList           resource_list,
   Cardinal                 num_resources,
   int                      size_of_help_struct,
   XdtQkHelpContextHelpProc context_help_proc )
#else
void XdtQkHelpInstallContextHelp( top, resource_list, num_resources,
                                  size_of_help_struct, context_help_proc )
   Widget                   top;
   XtResourceList           resource_list;
   Cardinal                 num_resources;
   int                      size_of_help_struct;
   XdtQkHelpContextHelpProc context_help_proc;
#endif
{
   _InitializeGlobal( XtDisplay( top ) );
   
   _appl_data.helpBufferSize   = size_of_help_struct;
   _appl_data.helpResources    = resource_list;
   _appl_data.numHelpResources = num_resources;
   _appl_data.contextHelpProc  = context_help_proc;
}

/*----------------------------------------------------------------------------*/
 
#if NeedFunctionPrototypes
void XdtQkHelpInstallOnShell( 
   Widget  hint_parent, 
   Widget* hint_widget_ptr )
#else
void XdtQkHelpInstallOnShell( hint_parent, hint_widget_ptr )
   Widget  hint_parent;
   Widget* hint_widget_ptr;
#endif
{
   Widget              shell_widget, main_widget;
   XdtQkHelpDisplayPtr qh_display_ptr;
   XdtQkHelpScreenPtr  qh_screen_ptr;
   
   assert( hint_parent );
      
   _InstallOnDisplayAndScreen( 
      hint_parent, 
      &qh_display_ptr, 
      &qh_screen_ptr );
      
   _DetermineShellAndMain( hint_parent, &shell_widget, &main_widget );
   
   if (main_widget)
   {
      XdtQkHelpShellPtr qh_shell_ptr = _ShellBind( 
         shell_widget, 
         hint_parent, 
         hint_widget_ptr, 
         qh_screen_ptr );
         
      _InstallOnWidgetTree( main_widget, qh_shell_ptr );
   }
}

/*----------------------------------------------------------------------------*/
 
#if NeedFunctionPrototypes
void XdtQkHelpInstallOnWidget(
   Widget w )
#else
void XdtQkHelpInstallOnWidget( w )
   Widget w;
#endif
{
   Widget              shell_widget, main_widget;
   XdtQkHelpDisplayPtr qh_display_ptr;
   XdtQkHelpScreenPtr  qh_screen_ptr;
   Boolean             gadget_manager_needed;
  
   assert( w );
      
   _InstallOnDisplayAndScreen( w, &qh_display_ptr, &qh_screen_ptr );
   _DetermineShellAndMain( w, &shell_widget, &main_widget );
   
   if (main_widget)
   {
      XdtQkHelpShellPtr qh_shell_ptr = _ShellBind( 
         shell_widget, 
         w, 
         NULL, 
         qh_screen_ptr );
         
      gadget_manager_needed = _AgentBind( w, qh_shell_ptr );
      if (gadget_manager_needed)
      {
         _AgentGadgetAware( XtParent( w ), qh_shell_ptr );
      }
   }
}

/*----------------------------------------------------------------------------*/
 
#if NeedFunctionPrototypes
void XdtQkHelpInstallOnWidgetTree(
   Widget w )
#else
void XdtQkHelpInstallOnWidgetTree( w )
   Widget w;
#endif
{
   Widget              shell_widget, main_widget;
   XdtQkHelpDisplayPtr qh_display_ptr;
   XdtQkHelpScreenPtr  qh_screen_ptr;
   
   assert( w );
      
   _InstallOnDisplayAndScreen( w, &qh_display_ptr, &qh_screen_ptr );
   _DetermineShellAndMain( w, &shell_widget, &main_widget );
   
   if (main_widget)
   {
      XdtQkHelpShellPtr qh_shell_ptr = _ShellBind( 
         shell_widget, 
         w, 
         NULL, 
         qh_screen_ptr );
         
      _InstallOnWidgetTree( w, qh_shell_ptr );
   }
}

