/*
 Copyright (C) 1999 Gerald L. Gay

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Library General Public License
 version 2 as published by the Free Software Foundation.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 Library General Public License for more details.

 You should have received a copy of the GNU Library General Public
 License along with this library; see the file COPYING.  If not,
 write to the Free Software Foundation, Inc., 675 Mass Ave,
 Cambridge, MA 02139, USA.
*/

/* Include(s) */
#include "setup.h"
#include "xarm_logo.xpm"

//************************************************************
//* I prefer to define all my resources in fallbacks.        *
//* That way, no matter what environment I'm in, I always    *
//* know that the application will look right. And I don't   *
//* have to worry about installing an app-defaults file.     *
//************************************************************

static _XtString fallbacks[] = {
    "*foreground: black",
    "setup*foreground: black",
    "*introLabel*FontList: -adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-*-*",
    "*swLabel*FontList: -adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-*-*",

    "*yesNoPrompt*FontList: -adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-*-*",
    "*yesNoYesButton*FontList: -adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-*-*",
    "*yesNoNoButton*FontList: -adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-*-*",

    "*dirPrompt*FontList: -adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-*-*",
    "*directory*FontList: -adobe-helvetica-medium-r-normal-*-14-*-*-*-*-*-*-*",
    "*directory*foreground: black",

    "*stringPanelPrompt*FontList: -adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-*-*",
    "*stringPanelEntry*FontList: -adobe-helvetica-medium-r-normal-*-14-*-*-*-*-*-*-*",
    "*stringPanelEntry*foreground: black",

    "*readyPrompt*FontList: -adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-*-*",

    "*currAction*FontList: -adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-*-*",
    "*installMonitor*FontList: -*-courier-medium-r-normal-*-10-*-*-*-*-*-*-*",
    "*installMonitor*foreground: black",
    "*waitToggle*FontList: -adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-*-*",

    "*thanksPrompt*FontList: -adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-*-*",

    "setup*appTitleLabel*background: blue",
    "setup*appTitleLabel*foreground: white",

    "*progressBar*sliderColor: blue",
    "*progressBar*foreground: white",

    "*RB*FontList: -adobe-helvetica-bold-r-normal-*-14-*-*-*-*-*-iso8859-1",
    "*nextLabel*FontList: -*-*-medium-i-normal-*-14-*-*-*-*-*-iso8859-1",
    "*backLabel*FontList: -*-*-medium-i-normal-*-14-*-*-*-*-*-iso8859-1",
    "*backButton*FontList: -*-*-bold-r-normal-*-14-*-*-*-*-*-iso8859-1",
    "*nextButton*FontList: -*-*-bold-r-normal-*-14-*-*-*-*-*-iso8859-1",
    "*Exit*FontList: -*-*-bold-r-normal-*-14-*-*-*-*-*-iso8859-1",
    "setup*titleLabel*background: blue",
    "setup*titleLabel*foreground: white",
    "setup*background: #C6C6B2B2A8A8",
    "",
    NULL
};

Application *theApp = NULL;
string       setupFile;

static char *configurePrompt = "Setup can configure the Xarm library for\n"
                               "compilation on your system. If you answer\n"
                               "Yes here, Setup will prompt for all the\n"
                               "required configuration information.\n\n"
                               "Should Setup configure Xarm:";

static char *sharedLibPrompt = "Shared libraries allow all programs that\n"
                               "use a library to dynamically load the\n"
                               "library code at run-time. This reduces the\n"
                               "size of the applications as well as the\n"
                               "amount of memory used.\n\n"
                               "Build the shared version of the Xarm and Xmext\n"
                               "libraries:";

static char *staticLibPrompt = "Static libraries are needed to build versions\n"
                               "of programs to run on systems that may not have\n"
                               "the required shared libraries, or may not support\n"
                               "shared libraries. This is useful for distributing\n"
                               "stand-alone applications, like this Setup Utility.\n\n"
                               "Build the static version of the Xarm and Xmext\n"
                               "libraries:";

static char *cdePrompt       = "The Xarm Library supports some parts of CDE, most\n"
                               "notably the Drag and Drop functionality. During\n"
                               "configuration, Xarm will check for and use some CDE\n"
                               "extensions to Motif if present.\n\n"
                               "Check for the Common Desktop Environment (CDE):";

static char *xbaePrompt      = "Xarm also provides C++ wrapper classes for the\n"
                               "Xbae Matrix and Caption widgets version 4.6.2\n"
                               "or higher.\n\n"
                               "Check for and use the Xbae widgets:";

static char *glPrompt        = "Xarm also provides C++ wrapper classes for the\n"
                               "GL Motif Widgets included in Mesa 3.1 and higher.\n"
                               "Check for and use the Mesa Motif widgets:";

static char *examplePrompt   = "The Xarm Library comes with several example\n"
                               "programs to demonstrate using the library,\n"
                               "including this Setup tool.\n\n"
                               "Build the example programs:";

static char *compilePrompt   = "After extracting and configuring the Xarm\n"
                               "Library, Setup can attempt to compile the\n"
                               "source code.\n\n"
                               "Compile Xarm:";

static char *advancedPrompt  = "This Setup Utility attempts to prompt for\n"
                               "the most commonly changed configuration\n"
                               "options. It does not ask for every possible\n"
                               "configuration variable. Advanced users may\n"
                               "enter configuration options that Setup has\n"
                               "not prompted for here. These should be\n"
                               "option strings just as you would provide to\n"
                               "configure on the command line. Setup will\n"
                               "pass them verbatim to configure.";

static char *installPrompt   = "After compiling the Xarm library, Setup can\n"
                               "attempt to install Xarm into the directory\n"
                               "specified. Note that this may not work if\n"
                               "you do not have write permissions on the target\n"
                               "directory.\n\n"
                               "Attempt to install Xarm:";

bool getMyFileName(char *name)
{

    struct stat statbuf;

    // If my name starts with a slash
    // then I was called absolutely.

    if (*name == '/') {
        if (stat(name, &statbuf) != 0) return false;
	setupFile = name;
	return true;
    }

    // See if I was called as ./xxxx (the most likely case)

    if ((strlen(name) > 2) && (*name == '.') && (*(name + 1) == '/')) {
	// Determine the absolute path because we will chdir()
	// to the desired extractDir later.
	char cwdBuf[1024];
	if (getcwd(cwdBuf, 1024) == NULL) return false;
	setupFile = cwdBuf;
	setupFile += "/";
	setupFile += (name + 2);
	if (stat(setupFile.c_str(), &statbuf) != 0) return false;
	else                                        return true;
    }

    // This is bad, the only way I could get here is if XarmSetup
    // was in the user's path. I don't know why anyone would do
    // this, but you never know!

    // If we were called out of the user's path, then we try to
    // do a "which prog-name" to get the full path.

    char cmd[1024];

    // Make sure we don't overflow the buffer!
    // The 7 is "which " plus the terminator.

    if (strlen(name) > (1024 - 7)) {
        cerr << "The program name is too long!" << endl;
	return false;
    }

    sprintf(cmd, "which %s", name);
    FILE *fp = popen(cmd, "r");

    if (fp == NULL) {
        cerr << "Setup couldn't execute the 'which' command!" << endl;
	return false;
    }

    char *result = fgets(cmd, 1024, fp);

    pclose(fp);

    if (result == NULL) {
        cerr << "Setup couldn't read the return from the 'which' command!" << endl;
        return false;
    }

    // If we didn't get an EOL in our string, then
    // the full path is bigger than our buffer!

    if (cmd[strlen(cmd) - 1] != '\n') {
        cerr << "The full path to the Setup program is too long!" << endl;
	return false;
    }

    // Whew!  That was painful.  Remove the newline and save off the name.

    cmd[strlen(cmd) - 1] = '\0';

    if (stat(cmd, &statbuf) != 0) return false;

    setupFile = cmd;
    return true;
}

int main (int argc, char **argv)
{

    if (argc == 0) {
        cerr << "Setup didn't get it's file name passed!" << endl;
	return EXIT_FAILURE;
    }

    // Try to determine our location...
    if (!getMyFileName(argv[0])) {
        cerr << "Setup couldn't deduce it's own file name" << endl
	     << "Try running as ./XarmSetup" << endl;
	return EXIT_FAILURE;
    }

    // initialize application 
    theApp = new Application("setup",argc,argv);

    // realize application
    theApp->realize();

    // start the main loop 
    theApp->mainLoop();

    // mainLoop *DOES* return!

    return EXIT_SUCCESS;
}

Application::Application(char *app_class, int &argc_in_out, char **argv_in_out):
   AppContext(app_class, NULL, 0, argc_in_out, argv_in_out, fallbacks)
{

    XarmArg args;

    title("Xarm 1.4.0 Setup");

    exitDialog = NULL;
    fatalErr   = NULL;
    warnDialog = NULL;

    // install window manager message handler(s)
    Atom proto = addWMProtocol(XA_WM_DELETE_WINDOW);
    addWMProtocolCallback(widget(), proto, (p_msg)&Application::onExit);

    // Setup Wizards are supposed to be dialog boxes.
    // Turn off all the resize functions.
    mwmDecorations( mwmDecorations() & ~MWM_DECOR_RESIZEH & ~1 );
    mwmFunctions( mwmFunctions() & ~(MWM_FUNC_RESIZE | MWM_FUNC_MAXIMIZE) & ~1 );

    logo = getPixmapFromData((const char **)xarm_logo);
    theForm = new Form(*this);

    // OK, start creating widgets!
    // You can get a good idea how the screen
    // is laid out by running the app!
    // The subpanels are in the middle right
    // area. They are drawn on top of each other.

    args.reset();
    args(XmNlabelType,      XmPIXMAP)
        (XmNlabelPixmap,    (XtArgVal)logo)
        (XmNtopOffset,      7)
        (XmNleftOffset,     7)
        (XmNbottomOffset,   3)
        (XmNleftAttachment, XmATTACH_FORM)
        (XmNtopAttachment,  XmATTACH_FORM);

    logoLabel = new Label(*theForm, args, "LogoLabel");

    args.reset();
    args(XmNorientation,      XmVERTICAL)
        (XmNleftAttachment,   XmATTACH_WIDGET)
        (XmNleftWidget,       logoLabel)
        (XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET)
        (XmNbottomWidget,     logoLabel)
        (XmNpacking,          XmPACK_TIGHT)
        (XmNmarginHeight,     0)
        (XmNborderWidth,      0)
        (XmNspacing,          0)
        (XmNshadowThickness,  0);

    rc = new RowColumn(*theForm, args, "RC");

    args.reset();
    args(XmNmarginHeight,    0)
        (XmNborderWidth,     0)
        (XmNshadowThickness, 0);

    nextLabel = new Label(*rc, args, "nextLabel");
    nextLabel->labelString("To go to the next screen, click Next");

    backLabel = new Label(*rc, args, "backLabel");
    backLabel->labelString("To go to the previous screen, click Back");

    backLabel->unmanage();
    rc->manage();

    args.reset();
    args(XmNleftAttachment,   XmATTACH_WIDGET)
        (XmNleftWidget,       logoLabel)
        (XmNtopAttachment,    XmATTACH_FORM)
        (XmNrightAttachment,  XmATTACH_FORM)
        (XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET)
        (XmNbottomWidget,     logoLabel)
        (XmNrightOffset,      5)
        (XmNtopOffset,        10)
        (XmNwidth,            400);

    panels.push_back(new WelcomePanel(*theForm, args, "WelcomePanel"));
    panels.push_back(new DirChooserPanel(*theForm, args, "DirChooserPanel"));
    panels.push_back(new YesNoPanel(*theForm, args, "ConfigurePanel",
				    " Configure Xarm", configurePrompt, true));
    panels.push_back(new YesNoPanel(*theForm, args, "BuildSharedPanel",
				    " Build Shared Libraries", sharedLibPrompt, true));
    panels.push_back(new YesNoPanel(*theForm, args, "BuildStaticPanel",
				    " Build Static Libraries", staticLibPrompt, true));
    panels.push_back(new YesNoPanel(*theForm, args, "CDEPanel",
				    " Check for CDE", cdePrompt, true));
    panels.push_back(new YesNoPanel(*theForm, args, "XbaePanel",
				    " Check for Xbae", xbaePrompt, true));
    panels.push_back(new XbaeDirPanel(*theForm, args, "XbaeDirPanel"));
    panels.push_back(new YesNoPanel(*theForm, args, "GLPanel",
				    " Check for Mesa Widgets", glPrompt, true));
    panels.push_back(new GLDirPanel(*theForm, args, "GLDirPanel"));
    panels.push_back(new YesNoPanel(*theForm, args, "ExamplePanel",
				    " Build Example Programs", examplePrompt, true));
    panels.push_back(new InstallDirPanel(*theForm, args, "InstallDirPanel"));
    panels.push_back(new StringPanel(*theForm, args, "AdvancedPanel",
				     " Enter Advanced Options", advancedPrompt));
    panels.push_back(new YesNoPanel(*theForm, args, "CompilePanel",
				    " Compile Xarm", compilePrompt, true));
    panels.push_back(new YesNoPanel(*theForm, args, "InstallPanel",
				    " Install Xarm", installPrompt, false));
    panels.push_back(new ReadyPanel(*theForm, args, "ReadyPanel"));
    panels.push_back(new InstallingPanel(*theForm, args, "InstallingPanel"));
    panels.push_back(new ThankYouPanel(*theForm, args, "ThankYouPanel"));

    panels[0]->manage();   // Manage the first panel
    sequence.push_back(0); // and prime the sequence

    args.reset();
    args(XmNtopAttachment,   XmATTACH_WIDGET)
        (XmNtopWidget,       logoLabel)
        (XmNleftAttachment,  XmATTACH_FORM)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNleftOffset,      1)
        (XmNrightOffset,     1)
        (XmNtopOffset,       10);

    sep = new Separator(*theForm, args, "sep1");

    args.reset();
    args(XmNtopAttachment,      XmATTACH_WIDGET)
        (XmNtopWidget,          sep)
        (XmNleftAttachment,     XmATTACH_FORM)
        (XmNbottomAttachment,   XmATTACH_FORM)
        (XmNleftOffset,         120)
        (XmNbottomOffset,       10)
        (XmNtopOffset,          10)
        (XmNhighlightThickness, 0);

    backButton = new PushButton(*theForm, args, "backButton");

    backButton->labelString("< Back");
    backButton->width(80);
    backButton->unmanage();
    ::addCallback(this, &Application::backCallback, backButton->widget(), XmNactivateCallback);

    args.reset();
    args(XmNtopAttachment,      XmATTACH_WIDGET)
        (XmNtopWidget,          sep)
        (XmNleftAttachment,     XmATTACH_FORM)
        (XmNbottomAttachment,   XmATTACH_FORM)
        (XmNleftOffset,         200)
        (XmNbottomOffset,       10)
        (XmNtopOffset,          10)
        (XmNhighlightThickness, 0);

    nextButton = new PushButton(*theForm, args, "nextButton");

    nextButton->labelString("Next >");
    nextButton->width(80);
    ::addCallback(this, &Application::nextCallback, nextButton->widget(), XmNactivateCallback);

    args.reset();

    args(XmNtopAttachment,      XmATTACH_WIDGET)
        (XmNtopWidget,          sep)
        (XmNrightAttachment,    XmATTACH_FORM)
        (XmNbottomAttachment,   XmATTACH_FORM)
        (XmNrightOffset,        10)
        (XmNbottomOffset,       10)
        (XmNtopOffset,          10)
        (XmNhighlightThickness, 0);

    exitButton = new PushButton(*theForm, args, "Exit");
    ::addCallback(this, &Application::onExit, exitButton->widget(), XmNactivateCallback);

    exitButton->width(80);

    allFinished = false;

    theForm->manage();

    // OK, it's kinda chessy but, I want the window
    // centered on the screen.  The problem is, you
    // can't ask it how big it is until it's drawn.
    // That means it will jump around on the screen
    // which you obviously don't want. So the only
    // other way is to know the size before-hand and
    // size it to the screen you're running on
    // (which you can ask).
    //
    // Does anyone know a better way to do this?
    //

    int xloc = WidthOfScreen(XtScreen(widget()));
    int yloc = HeightOfScreen(XtScreen(widget()));

    x((xloc - 566) / 2);
    y((yloc - 466) / 2);

}

void Application::fatal(char *mesg)
{

    if (fatalErr == NULL) {

        XarmArg args;

	args(XmNdialogStyle, XmDIALOG_APPLICATION_MODAL);

	fatalErr = new ErrorDialog(*this, args, "fatalError");
	fatalErr->dialogTitle("Fatal Error");
	fatalErr->messageString("A fatal error has occured");
	fatalErr->foreground(theForm->foreground());
	::addCallback(this, &Application::reallyExit, fatalErr->widget(), XmNokCallback);
	XtUnmanageChild(fatalErr->getChild(XmDIALOG_HELP_BUTTON));
	XtUnmanageChild(fatalErr->getChild(XmDIALOG_CANCEL_BUTTON));

    }

    fatalErr->messageString(mesg);
    fatalErr->manage();
}

void Application::warning(char *mesg)
{

    if (warnDialog == NULL) {

        XarmArg args;

	args(XmNdialogStyle, XmDIALOG_APPLICATION_MODAL);

	warnDialog = new WarningDialog(*this, args, "WarnDialog");
	warnDialog->dialogTitle("Warning");
	warnDialog->messageString("A non-fatal error has occured");
	warnDialog->foreground(theForm->foreground());
	XtUnmanageChild(warnDialog->getChild(XmDIALOG_HELP_BUTTON));
	XtUnmanageChild(warnDialog->getChild(XmDIALOG_CANCEL_BUTTON));

    }

    if (mesg != NULL) warnDialog->messageString(mesg);
    warnDialog->manage();
}

// nextCallback is where all the logic processing
// of the forms is done. I experimented with having
// the current form tell me where to go next or
// back to but I didn't like the idea of forms
// knowing about each other. The forms should do
// their job independently. It's OK for the main
// App to know about all the forms since that is who
// creates them! 

void Application::nextCallback(Widget, XtPointer, XtPointer)
{

    // This should never happen but I like to check anyway!
    if (panels.size() < 2)     return;
    if (sequence.size() == 0)  return;

    // If allFinished is true, then we could have
    // only gotten here via the Install panel.
    // In that case, we throw up the ThankYou panel
    // and unmanage the Install panel.

    if (allFinished) {
        panels[panels.size() - 2]->unmanage();
	panels[panels.size() - 1]->manage();
	nextButton->unmanage();
	nextLabel->unmanage();
	return;
    }

    // Find which form is active
    int currentForm = sequence[sequence.size() - 1];

    if (currentForm >= panels.size()) return;

    // formOK() will tell us if the data that the user
    // entered on this form is valid.
    if (!panels[currentForm]->formOK()) return;

    panels[currentForm]->unmanage();

    // If someone hit "Next", that means they can go back!
    backButton->manage();
    backLabel->manage();

    // OK, now it's time to figure out where to go...

    if (currentForm == 0) {
        // From the Welcome Panel, ask for a directory
        sequence.push_back(1);
    } else if (currentForm == 1) {
        // We're coming from the Directory Panel, ask about configuring...
        sequence.push_back(2);
    } else if (currentForm == 2) {
        // From ConfigurePanel, get the answer...
        YesNoPanel *yp = (YesNoPanel *)panels[currentForm];
	if (yp->value()) sequence.push_back(3);
	else             sequence.push_back(15);
    } else if (currentForm == 3) {
        // From Shared, continue to Static...
        sequence.push_back(4);
    } else if (currentForm == 4) {
        // From Static, do a little sanity check first.
        // We have to build at least one of shared or static.
        YesNoPanel *yp1, *yp2;
	yp1 = (YesNoPanel *)panels[3];
	yp2 = (YesNoPanel *)panels[4];
	if (!yp1->value() && !yp2->value()) {
	    warning("You must build at least one of the Shared\n"
		    "or Static versions of the Xarm Library");
	    panels[currentForm]->manage();
	    return;
	}
	// OK, now go on to CDE...
	sequence.push_back(5);
    } else if (currentForm == 5) {
        // From CDE, ask about Xbae...
        sequence.push_back(6);
    } else if (currentForm == 6) {
        // If we want Xbae, see if we want to give a directory...
        YesNoPanel *yp = (YesNoPanel *)panels[6];
	if (yp->value()) sequence.push_back(7);
	else             sequence.push_back(8);
    } else if (currentForm == 7) {
        // Ask about GL...
        sequence.push_back(8);
    } else if (currentForm == 8) {
        // If we want GL, see if we want to give a directory...
        YesNoPanel *yp = (YesNoPanel *)panels[8];
	if (yp->value()) sequence.push_back(9);
	else             sequence.push_back(10);
    } else if (currentForm == 9) {
        // From GL, ask about examples...
        sequence.push_back(10);
    } else if (currentForm == 10) {
        // From examples, ask about prefix...
        sequence.push_back(11);
    } else if (currentForm == 11) {
        // Ask if they want to give any other configure options...
        sequence.push_back(12);
    } else if (currentForm == 12) {
        // Ask if we should compile...
        sequence.push_back(13);
    } else if (currentForm == 13) {
        // From the compile panel, only ask about installing
        // if the user told us to compile!
        YesNoPanel *ynp = (YesNoPanel *)panels[13];
	if (ynp->value()) sequence.push_back(14);
	else              sequence.push_back(15);
    } else if (currentForm == 14) {
        sequence.push_back(15);
    } else if (currentForm == 15) {
        // Here we go!
        // Prepare everything and manage the InstallingPanel!
        doInstall();
	return;
    }

    // At this point, the next panel to go to will be
    // at the end of sequence...
    int nextForm = sequence[sequence.size() - 1];
    panels[nextForm]->manage();

    // If nextForm is 13, we are almost ready to go.
    // Change the nextButton and nextLabel.

    if (nextForm == 15) {
        nextButton->labelString("Finish");
	nextButton->width(80);
	nextLabel->labelString("To install the Xarm Library, click Finish");
    }
}

// backCallback is easy. We look in the sequence for the current
// and previous screens. Unmanage the current one and re-manage
// the previous one. If we end up re-managing screen 0, we
// hide backButton.

void Application::backCallback(Widget, XtPointer, XtPointer)
{

    if (panels.size() < 2)   return;
    if (sequence.size() < 2) return;

    int currentForm = sequence[sequence.size() - 1];
    int prevForm    = sequence[sequence.size() - 2];

    if (currentForm >= panels.size()) return;
    if (currentForm <= 0)             return;
    if (prevForm    >= panels.size()) return;
    if (prevForm    <  0)             return;

    // Remove the last element from the sequence.
    // We may traverse the list differently next time.
    sequence.erase(sequence.end() - 1);

    panels[currentForm]->unmanage();
    panels[prevForm]->manage();

    if (prevForm == 0) {
        backButton->unmanage();
	backLabel->unmanage();
    }

    // Fix up the next button just in case someone messed with it.
    nextButton->labelString("Next >");
    nextButton->width(80);
    nextLabel->labelString("To go to the next screen, click Next");
    nextButton->manage();
    nextLabel->manage();
}

void Application::doInstall()
{
    XarmOptions options;

    // Get all the answers
    options.extractDir    = ((DirChooserPanel *) panels[1])->value();
    options.wantConfigure = ((YesNoPanel *)      panels[2])->value();
    options.wantShared    = ((YesNoPanel *)      panels[3])->value();
    options.wantStatic    = ((YesNoPanel *)      panels[4])->value();
    options.wantCDE       = ((YesNoPanel *)      panels[5])->value();
    options.wantXbae      = ((YesNoPanel *)      panels[6])->value();
    options.xbaeDir       = ((XbaeDirPanel *)    panels[7])->value();
    options.wantGL        = ((YesNoPanel *)      panels[8])->value();
    options.glDir         = ((GLDirPanel *)      panels[9])->value();
    options.wantExamples  = ((YesNoPanel *)      panels[10])->value();
    options.installDir    = ((InstallDirPanel *) panels[11])->value();
    options.otherOptions  = ((StringPanel *)     panels[12])->value();
    options.wantCompile   = ((YesNoPanel *)      panels[13])->value();
    options.wantInstall   = ((YesNoPanel *)      panels[14])->value();

    // Unmanage the next/back buttons and labels and
    // change Exit to Cancel

    rc->unmanage();
    backButton->unmanage();
    nextButton->unmanage();
    exitButton->labelString("Cancel");
    exitButton->width(80);

    // Set up InstallingPanel...

    InstallingPanel *ip = (InstallingPanel *)panels[16];

    ip->setOptions(options);
    ip->manage();
}

void Application::onExit(Widget, XtPointer, XtPointer)
{

    if (allFinished) {
        quit();
	return;
    }

    if (exitDialog == NULL) {

        XarmArg args;

	args(XmNdialogStyle, XmDIALOG_APPLICATION_MODAL);

	exitDialog = new QuestionDialog(*this, args, "exitDialog");
	exitDialog->dialogTitle("Exit");
	exitDialog->messageString("Are you sure you want\nto abort this installation?");
	exitDialog->foreground(theForm->foreground());
	XtUnmanageChild(exitDialog->getChild(XmDIALOG_HELP_BUTTON));
	::addCallback(this, &Application::reallyExit, exitDialog->widget(), XmNokCallback);
    }

    exitDialog->manage();
}

void Application::reallyExit(Widget, XtPointer, XtPointer)
{
    // We might need to do something here if an install is in progress

    quit();
}

void Application::done()
{
    // We're all done!
    // The last panel will always be the ThankYou panel
    // and the next-to-last panel is the Install panel.

    panels[panels.size() - 2]->unmanage();
    panels[panels.size() - 1]->manage();

    allFinished = true;

    exitButton->labelString("Exit");
    exitButton->width(80);
}

void Application::almostDone()
{
    // We're all done! But the user wants to look
    // at the results for a while. So don't unmanage
    // the Install panel. Instead, put the Next button back
    // on the screen.
    // The last panel will always be the ThankYou panel
    // and the next-to-last panel is the Install panel.

    allFinished = true;

    exitButton->labelString("Exit");
    exitButton->width(80);
    nextButton->labelString("Next >");
    nextButton->width(80);
    nextLabel->labelString("To go to the next screen, click Next");
    nextButton->manage();
    nextLabel->manage();
    backLabel->unmanage();
    rc->manage();
}

SetupSubpanel::SetupSubpanel(Widget parent, XarmArg &formArgs, char *name)
{
    XarmArg args;
    string str;

    myForm = new Form(parent, formArgs, name);

    args(XmNleftAttachment,  XmATTACH_FORM)
        (XmNtopAttachment,   XmATTACH_FORM)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNalignment,       XmALIGNMENT_BEGINNING)
        (XmNrightOffset,     5)
        (XmNleftOffset,      9);

    titleLabel = new Label(*myForm, args, "titleLabel");
    titleLabel->labelString(" Set me!");
}

YesNoPanel::YesNoPanel(Widget parent,
		       XarmArg &formArgs,
		       char *name,
		       const char *title,
		       const char *text,
		       bool defAnswer) : SetupSubpanel(parent, formArgs, name)
{
    XarmArg args;

    setTitle(title);
    answer = defAnswer;

    args.reset();
    args(XmNleftAttachment,  XmATTACH_FORM)
        (XmNtopAttachment,   XmATTACH_WIDGET)
        (XmNtopWidget,       titleLabel)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNrightOffset,     5)
        (XmNleftOffset,      20)
        (XmNtopOffset,       30)
        (XmNalignment,       XmALIGNMENT_BEGINNING);

    prompt = new Label(*myForm, args, "yesNoPrompt");

    prompt->labelString(text);

    args.reset();
    args(XmNleftAttachment,     XmATTACH_FORM)
        (XmNrightAttachment,    XmATTACH_FORM)
        (XmNtopAttachment,      XmATTACH_WIDGET)
        (XmNtopWidget,          prompt)
        (XmNleftOffset,         50)
        (XmNtopOffset,          20)
        (XmNrightOffset,        5)
        (XmNhighlightThickness, 0)
        (XmNorientation,        XmVERTICAL)
        (XmNradioBehavior,      True);

    choiceBox = new RowColumn(*myForm, args, "yesNoChoiceBox");

    args.reset();
    args(XmNhighlightThickness, 0);

    yesButton = new ToggleButton(*choiceBox, args, "yesNoYesButton");
    noButton  = new ToggleButton(*choiceBox, args, "yesNoNoButton");

    if (answer) {
        yesButton->labelString("Yes (recommended)");
	noButton->labelString("No");
	yesButton->set(True);
    } else {
        yesButton->labelString("Yes");
	noButton->labelString("No (recommended)");
	noButton->set(True);
    }

    choiceBox->manage();

    ::addCallback(this, &YesNoPanel::valueTracker, yesButton->widget(), XmNvalueChangedCallback);
    ::addCallback(this, &YesNoPanel::valueTracker, noButton->widget(), XmNvalueChangedCallback);
}

void YesNoPanel::valueTracker(Widget, XtPointer, XtPointer)
{
  answer = (yesButton->set() == True) ? true : false;
}

StringPanel::StringPanel(Widget parent, XarmArg &formArgs, char *name,
			 const char *title, const char *promptText, const char *defValue)
  : SetupSubpanel(parent, formArgs, name)
{
    XarmArg args;

    setTitle(title);
    strValue = NULL;

    args.reset();
    args(XmNleftAttachment,  XmATTACH_FORM)
        (XmNtopAttachment,   XmATTACH_WIDGET)
        (XmNtopWidget,       titleLabel)
        (XmNrightAttachment, XmATTACH_FORM)
        (XmNrightOffset,     5)
        (XmNleftOffset,      40)
        (XmNtopOffset,       30)
        (XmNalignment,       XmALIGNMENT_BEGINNING);

    prompt = new Label(*myForm, args, "stringPanelPrompt");

    prompt->labelString(promptText);

    args.reset();
    args(XmNleftAttachment,     XmATTACH_FORM)
        (XmNrightAttachment,    XmATTACH_FORM)
        (XmNtopAttachment,      XmATTACH_WIDGET)
        (XmNtopWidget,          prompt)
        (XmNleftOffset,         30)
        (XmNtopOffset,          20)
        (XmNrightOffset,        5)
        (XmNhighlightThickness, 0);

    entry = new TextField(*myForm, args, "stringPanelEntry");

    // Here's something that really bugs me: I hate Text Fields
    // that leave the text cursor visible while you're not
    // entering data into it.  It's distracting and it can make
    // the text under the grey'd out cursor hard to read. So
    // I turn it off and only turn it on when the user is editing
    // the text field.

    entry->cursorPositionVisible(False);
    ::addCallback(this, &StringPanel::tfGetFocus,  entry->widget(), XmNfocusCallback);
    ::addCallback(this, &StringPanel::tfLoseFocus, entry->widget(), XmNlosingFocusCallback);

    if (defValue != NULL) entry->value(defValue);
    else                  entry->value("");
}

void StringPanel::tfGetFocus(Widget, XtPointer, XtPointer)
{
    entry->cursorPositionVisible(True);
}

void StringPanel::tfLoseFocus(Widget, XtPointer, XtPointer)
{
    entry->cursorPositionVisible(False);
}

char *StringPanel::value()
{
    if (strValue == NULL) return "";
    else                  return strValue;
}

bool StringPanel::formOK()
{

    if (strValue != NULL) {
        XtFree(strValue);
    }

    strValue = entry->value();

    // Call validate() to do any required checking on the data

    return validate();
}
