//
// Copyright 1994, Cray Research, Inc.
//                 
// Permission to use, copy, modify and distribute this software and
// its accompanying documentation (the "Software") is granted without
// fee, provided that the above copyright notice and this permission
// notice appear in all copies of the Software and all supporting
// documentation, and the name of Cray Research, Inc. not be used in
// advertising or publicity pertaining to distribution of the 
// Software without the prior specific, written permission of Cray
// Research, Inc.  The Software is a proprietary product of Cray
// Research, Inc., and all rights not specifically granted by this
// license shall remain in Cray Research, Inc.  No charge may be made
// for the use or distribution of the Software.  The Software may be
// distributed as a part of a different product for which a fee is
// charged, if (i) that product contains or provides substantial
// functionality that is additional to, or different from, the
// functionality of the Software, and (ii) no separate, special or
// direct charge is made for the Software.
//         
// THE SOFTWARE IS MADE AVAILABLE "AS IS", AND ALL EXPRESS AND
// IMPLIED WARRANTIES, INCLUDING THE IMPLIED WARRANTIES OF FITNESS
// FOR A PARTICULAR PURPOSE, MERCHANTABILITY, AND FREEDOM FROM
// VIOLATION OF THIRD PARTY INTELLECTUAL PROPERTY RIGHTS, ARE HEREBY
// DISCLAIMED AND EXCLUDED BY CRAY RESEARCH, INC.  CRAY RESEARCH,
// INC. WILL NOT BE LIABLE IN ANY EVENT FOR ANY CONSEQUENTIAL,
// SPECIAL, INCIDENTAL, OR INDIRECT DAMAGES ARISING OUT OF OR IN
// CONNECTION WITH THE PERFORMANCE OF THE SOFTWARE OR ITS USE BY ANY
// PERSON, OR ANY FAILURE OR NEGLIGENCE ON THE PART OF CRAY RESEARCH,
// INC., EXCEPT FOR THE GROSS NEGLIGENCE OR WILLFUL MISCONDUCT OF
// CRAY RESEARCH.
// 
// This License Agreement shall be governed by, and interpreted and
// construed in accordance with, the laws of the State of Minnesota,
// without reference to its provisions on the conflicts of laws, and
// excluding the United Nations Convention of the International Sale
// of Goods.
//
static void USMID() { void("%Z%%M%      %I%     %G% %U%"); }
//
// This code was initially contributed by Dean Johnson
//

//
//	Commands:
//		{COMMENT:text}
//		{XREF:text:file}
//		{POPUP:file}
//		{GRAPHIC:filename}
//		{EXEC:command:blah}
//		{INDEX:entry}
//		{TITLE:entry}
//		{MARK:entry}
//

#include <stdio.h>
#include <assert.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <Cvo/HelpParse.h++>

struct	HelpCmdTableType {
	char	*name;
	HelpCommandType	type;
};

static struct HelpCmdTableType cmd_table[] = {
	{ "TAG",	HelpTag		},
	{ "HTAG",	HelpHiddenTag	},
	{ "XREF",	HelpXref	},
	{ "INDEX",	HelpIndex	},
	{ "MARK",	HelpBookmark	},
	{ "TITLE",	HelpTitle	},
	{ "POPUP",	HelpPopup	},
	{ "COMMENT",	HelpComment	},
	{ "EXEC",	HelpExec	},
	{ "GRAPHIC",	HelpGraphic	},
};
static int	max_help_cmd_length = -1,
		table_length = -1;
static char	*cmd_buffer = NULL;

static char *
RestOfCommand(char *str,int *len)
{
	char	*cp = str,
		*tmp;

	*len = 0;
	while (*cp && *cp != '}') {
		(*len)++;
		if (*cp == '\n') {
			break;
		}
		cp++;
	}
	tmp = new char[*len + 1];
	memcpy(tmp,str,*len);	
	tmp[*len] = '\0';
	return(tmp);
}

void	
HelpUnit::AddToTable(char *file,HelpCommandType type,int location,int length,int line,int column,char *tag)
{
	HelpItem *look,
		 *last;
	HelpItem	*tmp = new HelpItem(file,type,location,length,line,column,tag);

	if (xref_table == NULL) {
		xref_table = tmp;
		last_item = tmp;
	}
	else	{
		/*
		 *	Insert it into the right place
		 */
		last = NULL;	
		for (look = xref_table; look != NULL; look = look->Next()) {
			if (location < look->Location()) {
				tmp->SetNext(look);
				if (last == NULL)
					xref_table = tmp;
				else
					last->SetNext(tmp);
				return;
			}
			else if (look->Next() == NULL) {
				//
				// Add it at the end
				//
				look->SetNext(tmp);
				last_item = tmp;
				return;
			}
			last = look;
		}
	}
}

void
HelpUnit::PrintTable()
{
	int	i;
	HelpItem	*tmp;

	for (i = 0, tmp = xref_table; tmp != NULL; tmp = tmp->Next(), i++)
		switch (tmp->Type()) {
		case HelpXref:
			printf("%d.<%d>(%d-%d) \"%*.*s\" %s <%s> (c=%d l=%d)\n",
				i,tmp->Type(), tmp->Location(),
				tmp->Location() + tmp->Length(),tmp->Length(),
				tmp->Length(), &buffer[tmp->Location()],
				tmp->File(),tmp->Tag(),tmp->Column(),
				tmp->Line());
			break;
		default:
			printf("%d.<%d>(%d-%d) %s (c=%d l=%d)\n",i,tmp->Type(),
				tmp->Location(),tmp->Location() + tmp->Length(),
				tmp->Tag(),tmp->Column(),tmp->Line());
		}
}

HelpUnit::HelpUnit(char *filename,int markup)		// constructor
{
	struct stat	buf;
	int	i,
		inf,
		status,
		slen,
		src_line = 0,
		line = 0,
		column = 0;

	last_item = NULL;
	title = NULL;
	max_lines = 0;

	//
	//	Initialize the commands table.
	//
	if (max_help_cmd_length < 0) {
		table_length = sizeof(cmd_table) / sizeof(cmd_table[0]);
		for (i = 0; i < table_length; i++) {
			slen = strlen(cmd_table[i].name);
			if (slen > max_help_cmd_length)
				max_help_cmd_length = slen;
		}
		cmd_buffer = new char[max_help_cmd_length + 2];
	}

	file_name = new char[strlen(filename) + 1];
	strcpy(file_name,filename);

	buffer = NULL;
	length = -1;	// 0 is a meaningful length
	xref_table = NULL;

	if (stat(filename,&buf) == -1) return;	// ERROR: doesn't exist!
	//
	//	Make sure that we can open it.
	//
	inf = open(filename,O_RDONLY);
	if (inf < 0) return;			// ERROR: can't open it!
	length = (int) buf.st_size;
	buffer = new char[length + 1];
	if (buffer == NULL) {
		fprintf(stderr,"Unable to allocate enough memory to hold the help file!\n");
		return;
	}
	buffer[length] = 0;
	//
	//	Read in the data.
	//
	status = read(inf,buffer,length);
	if (status == -1) {			// ERROR: couldn't read it!
		length = -1;
		delete buffer;
		buffer = NULL;
		return;
	}

	if (!markup) {
		//
		//	No markup, just get the number of lines.
		//
		for (char *cp = buffer; *cp; cp++) {
			if (*cp == '\n')
				max_lines++;
		}
		max_lines += 2;
		return;
	}

	//--------------------------------------------------------------//
	//			Parse the data				//
	//--------------------------------------------------------------//

	char	*base  = buffer,	// stays behind.
		*scout = buffer;	// looks ahead.
	int	newlen = 0,
		curloc = 0;
	char	*saveptr;
	char	*str;

	while (*scout) {
		//--------------------------------------//
		//		LITERAL			//
		//--------------------------------------//
		if (*scout == '\\') {
			scout++;	// its a literal, skip the '\'
			if (*scout == '\n') {
					fprintf(stderr,"Warning: single backslash at end of line %d. Ignoring literal translation.\n",
						src_line + 1);
				scout--;
			}
		}

		//--------------------------------------//
		//		COMMAND			//
		//--------------------------------------//
		else if (*scout == '{') {
			int len = 0;

			saveptr = scout++;
			//
			//	Get the command string.
			//
			char	*cmb = scout;
			char	*cp;
			int	curlen;

			curlen = 0;
			cp = cmd_buffer;
			for (; curlen < max_help_cmd_length && *cmb != ':'; cmb++) {
				if (!isspace(*cmb)) {
					*cp++ = *cmb;
					curlen++;
				}
			}
			*cp = '\0';		// null terminate
			if (*cmb == ':') {
				cmb++;
			}
			//
			//
			//
			HelpCommandType	type = HelpUnknown;

			for (i = 0; i < table_length; i++) {
				if (!strcmp(cmd_buffer,cmd_table[i].name)) {
					// bingo!
					type = cmd_table[i].type;
					break;
				}
			}

			scout = cmb;
			slen = 0;
			if (type != HelpUnknown) {
				str = RestOfCommand(scout,&len);
				slen = strlen(str);
				if (str[slen-1] == '\n') {
					// check for tag with <CR> in it.
					fprintf(stderr,"<CR> in tag at line %d\n",
						src_line + 1);
					type = HelpUnknown;
					//
					//	Put it into the line, as is.
					//
					scout = saveptr;
					while (*scout != '\n') {
						*base++ = *scout++;
						newlen++;
						curloc++;
					}
					src_line++;
					continue;
				}
				assert(slen != 0);
			}

			switch (type) {
			case HelpComment:
			case HelpGraphic: {
				//
				//	skip to the end of the comment.
				//
				AddToTable(NULL,type,curloc,slen,line,column,str);
			    }
				break;
			case HelpTag: {
				//
				//	skip to the end of the tag.
				//
				char	*allstr = str;

				while (isspace(*allstr)) {
					allstr++;
					slen--;
				}
				AddToTable(NULL,type,curloc,slen,line,column,str);
				strcpy(base,allstr);
				base   += slen;
				curloc += slen;
				column += slen;
				newlen += slen;
			    }
				break;
			case HelpHiddenTag: {
				//
				//	skip to the end of the comment.
				//
				char	*allstr = str;

				AddToTable(str,type,curloc,slen,line,column);
			    }
				break;
			case HelpExec:
			case HelpPopup: {
				char	*cptr;
				int string_length;

				for (cptr = str; *cptr && *cptr != ':'; cptr++);
				if (*cptr) {
					*cptr = '\0';
					cptr++;
					}
				string_length = strlen(str);
				AddToTable(NULL,type,curloc,string_length,line,column,cptr);
				strcpy(base,str);
				base   += string_length;
				curloc += string_length;
				column += string_length;
				newlen += string_length;
			    }
				break;
			case HelpXref: {
				char	*arg1 = str,
					*arg2,		// file pointer
					*arg3;		// tag pointer

				for (arg2 = arg1; *arg2 && *arg2 != ':'; arg2++);
				if (*arg2) {
					*arg2 = '\0';
					arg2++;
					}
				for (arg3 = arg2; *arg3 && *arg3 != ':'; arg3++);
				if (*arg3) {
					*arg3 = '\0';
					arg3++;
					}
				int a1_len = strlen(arg1);
				AddToTable(arg2,type,curloc,a1_len,line,
					   column,arg3);

				strcpy(base,arg1);
				base   += a1_len;
				curloc += a1_len;
				column += a1_len;
				newlen += a1_len;
			    }
				break;
			case HelpTitle: {
				char	*allstr = str;
				while (isspace(*allstr)) allstr++;
				if (title) {
					delete title;
				}
				title = new char[strlen(allstr) + 1];
				strcpy(title,allstr);
			    }
				break;
			case HelpBookmark:
				AddToTable(filename,type,curloc,strlen(str),line,0,str);
				break;
			case HelpIndex:
				AddToTable(NULL,type,curloc,strlen(str),line,0,str);
				break;
			case HelpUnknown:
				fprintf(stderr,"Unknown Command \"%s\"!\n",cmd_buffer);
				scout = saveptr;
				*base++ = *scout++;
				curloc++;
				column++;
				newlen++;
				break;
			default:		// ERROR: bogus command!
				fprintf(stderr,"Bogus Command \"%s\"!\n",
					cmd_buffer);
				break;
			}
			if (type != HelpUnknown) {
				if (*scout) scout++;	// skip close bracket
				delete str;
			}
			scout += len;
			continue;
		}
		else if (*scout == '\n') {
			line++;
			src_line++;
			column = 0;
		}

		//--------------------------------------//
		//		Do the Copy		//
		//--------------------------------------//
		if (*scout != '\n')
			column++;
		*base++ = *scout++;
		newlen++;
		curloc++;
	}
	*base = '\0';
	length = ++newlen;
	max_lines = line + 2;
}

HelpUnit::~HelpUnit()				// destructor
{
	if (file_name)
		delete [] file_name;
	delete buffer;
}

/*
 *	Write out the help data file.
 */

#define	TITLE	"{TITLE:%s}"
#define	INDEX	"{INDEX:%s}"
#define	TAG	"{TAG:%s}"
#define	HTAG	"{HTAG:%s}"
#define	XREF1	"{XREF:%s:%s}"
#define	XREF2	"{XREF:%s:%s:%s}"
#define	EXEC	"{EXEC:%s:%s}"
#define	GRAPHIC	"{GRAPHIC:%s}"
#define	COMMENT	"{COMMENT:%s}"

int
HelpUnit::Write(char *file)
{
	int	data_line = 0;
	FILE	*outfd;
	char	buffer[1024];
	char	*bufp;

	if ((outfd = fopen(file,"w")) == NULL) {
		fprintf(stderr,"Cannot open file %s\n",buffer);
		return(1);
	}	
	/*
	 *	Do the title, if any.
	 */
	if (title) {
		fprintf(outfd,TITLE,title);
	}
	/*
	 *	Do the write
	 */

	HelpItem *tmp = Xref();
	int count = 0;
	int visible;
	int advanced;

	for (bufp = Buffer(); *bufp; bufp++) {
		if (tmp && (count == tmp->Location())) {
			visible = 1;
			advanced = 0;		// pointer hasn't been bumped
			switch (tmp->Type()) {
			case HelpExec:
				memcpy(buffer,bufp,tmp->Length());
				buffer[tmp->Length()] = '\0';
				fprintf(outfd,EXEC,buffer,tmp->Tag());
				break;
			case HelpXref:
				memcpy(buffer,bufp,tmp->Length());
				buffer[tmp->Length()] = '\0';
				if (tmp->Tag() && *(tmp->Tag())) {
					fprintf(outfd,XREF2,buffer,tmp->File(),
						tmp->Tag());
				}
				else	{
					fprintf(outfd,XREF1,buffer,tmp->File());
				}
				break;
			case HelpTag:
				fprintf(outfd,TAG,tmp->Tag());
				// base count upon displayed chars
				{
				char *sp = tmp->Tag();
				while (isspace(*sp)) sp++;
				bufp += strlen(sp) - 1;
				count += strlen(sp);
				}
				advanced = 1;
				break;
			case HelpHiddenTag:
				fprintf(outfd,HTAG,tmp->Tag());
				break;
			case HelpIndex:
				fprintf(outfd,INDEX,tmp->Tag());
				visible = 0;
				break;
			case HelpComment:
				fprintf(outfd,COMMENT,tmp->Tag());
				visible = 0;
				break;
			case HelpGraphic:
				fprintf(outfd,GRAPHIC,tmp->Tag());
				visible = 0;
				break;
			}
			if (!advanced) {
				if (visible) {
					bufp  += tmp->Length() - 1;
					count += tmp->Length();
				}
				else {
					bufp--;
				}
			}
			tmp = tmp->Next();
			continue;
		}
		fprintf(outfd,"%c",*bufp);
		count++;
	}
	fclose(outfd);
	return(0);
}

void
HelpUnit::DeleteRange(int b_loc,int e_loc)
{
HelpItem	*tmp = Xref(),
		*last = NULL;
int	skip;

	for (; tmp != NULL; tmp = tmp->Next()) {
		if (e_loc < tmp->Location())
			skip = 1;
		else if (b_loc > tmp->Location() + tmp->Length())
			skip = 1;
		else	
			skip = 0;
		if (skip) {
			last = tmp;
			continue;
		}
		if (last == NULL) {
			//	At the beginning of the list
			xref_table = tmp->Next();
		} 
		else	{
			last->SetNext(tmp->Next());
			tmp = last;
		}
		delete tmp;
	}
}

HelpItem::HelpItem(char *fn, HelpCommandType typ,
                   int loc, int len,
                   int ln, int col, char *tg)
{
    if (fn) {
	file = new char[strlen(fn) + 1];
	strcpy(file,fn);
    } else {
	file = NULL;
    }
    location = loc;
    length = len;
    type = typ;
    next = NULL;
    column = col;
    line = ln;
    if (tg) {
	tag = new char[strlen(tg) + 1];
	strcpy(tag,tg);
    } else {
	tag = NULL;
    }
}

