//
// Copyright 1994, Cray Research, Inc.
//                 
// Permission to use, copy, modify and distribute this software and
// its accompanying documentation (the "Software") is granted without
// fee, provided that the above copyright notice and this permission
// notice appear in all copies of the Software and all supporting
// documentation, and the name of Cray Research, Inc. not be used in
// advertising or publicity pertaining to distribution of the 
// Software without the prior specific, written permission of Cray
// Research, Inc.  The Software is a proprietary product of Cray
// Research, Inc., and all rights not specifically granted by this
// license shall remain in Cray Research, Inc.  No charge may be made
// for the use or distribution of the Software.  The Software may be
// distributed as a part of a different product for which a fee is
// charged, if (i) that product contains or provides substantial
// functionality that is additional to, or different from, the
// functionality of the Software, and (ii) no separate, special or
// direct charge is made for the Software.
//         
// THE SOFTWARE IS MADE AVAILABLE "AS IS", AND ALL EXPRESS AND
// IMPLIED WARRANTIES, INCLUDING THE IMPLIED WARRANTIES OF FITNESS
// FOR A PARTICULAR PURPOSE, MERCHANTABILITY, AND FREEDOM FROM
// VIOLATION OF THIRD PARTY INTELLECTUAL PROPERTY RIGHTS, ARE HEREBY
// DISCLAIMED AND EXCLUDED BY CRAY RESEARCH, INC.  CRAY RESEARCH,
// INC. WILL NOT BE LIABLE IN ANY EVENT FOR ANY CONSEQUENTIAL,
// SPECIAL, INCIDENTAL, OR INDIRECT DAMAGES ARISING OUT OF OR IN
// CONNECTION WITH THE PERFORMANCE OF THE SOFTWARE OR ITS USE BY ANY
// PERSON, OR ANY FAILURE OR NEGLIGENCE ON THE PART OF CRAY RESEARCH,
// INC., EXCEPT FOR THE GROSS NEGLIGENCE OR WILLFUL MISCONDUCT OF
// CRAY RESEARCH.
// 
// This License Agreement shall be governed by, and interpreted and
// construed in accordance with, the laws of the State of Minnesota,
// without reference to its provisions on the conflicts of laws, and
// excluding the United Nations Convention of the International Sale
// of Goods.
//
static void USMID() { void("%Z%%M%	%I%	%G% %U%"); }
static void RSCID() { void("$Id: ChartHorizontal.cc,v 1.6 1994/09/21 18:18:28 prb Exp $"); }

#include <Cvo/Chart.h++>
#include <Cvo/Frame.h++>
#include <Cvo/TextBoard.h++>
#include <Cvo/Paint.h++>
#include <math.h>

CONSTRUCTORS_1ARG_PASS(Cvo_HorizontalGraph, Cvo_GraphObject,
		       "CvoHorizontalGraph", Cvo_Chart *)
CONSTRUCTORS_1ARG_PASS(Cvo_HorizontalSideGraph, Cvo_HorizontalGraph,
		       "CvoHorizontalSideGraph", Cvo_Chart *)
CVO_CREATE_REGISTER_FUNCTIONS(Cvo_HorizontalGraph)

void
Cvo_HorizontalSideGraph::_Init(Cvo_Chart *)
{
    return;
}

void
Cvo_HorizontalGraph::_Init(Cvo_Chart *)
{
    colwidth = 30;
    fudge = 0;
    coldepth = GetResourceInt("depth", "Depth", 15);
    colseparation = GetResourceInt("separation", "Separation", 7);
    mincolwidth = 1;
    mincolheight = 0.0;

    VerticalChildren();

    frame1 = new Cvo_Frame("frame1", this);
    frame1->ExpandFrame();
    frame1->HorizontalChildren();

    frame2 = new Cvo_Frame("frame2", this);
    frame2->HorizontalChildren();

    if (pixmapbacking == True)
        graph = new Cvo_Paint("grafik", frame1);
    else
        graph = new Cvo_Window("grafik", frame1);

    graph->Flatten();
    graph->ExpandFrame();
    graph->LayOpposite();
    
    graph->Register(Expose, _Cvo_GraphObject_Expose_Handler, this);

    if (pixmapbacking == True)
        graph->Register(CvoPaintChangedSizeEvent,
			_Cvo_GraphObject_Expose_Handler, this);

    cticks = new Cvo_TextBoard("ticks", frame1);
    cticks->LayOpposite();
    cticks->SetMinSize(1, 0);

    ticks = new Cvo_TextBoard("ticks", frame2);
    ticks->SetMinSize(0, 1);
    ticks->MatchWidth(graph);
    ticks->MatchX(graph);

    marklist = NULL;
    cmarklist = NULL;
}

Cvo_GraphObject *
Cvo_HorizontalGraph::CloneMe(Cvo_Object *o, Cvo_Chart *c)
{
    Cvo_HorizontalGraph *ret = new Cvo_HorizontalGraph("grafik", o, c);
    ret->colwidth = colwidth;
    ret->fudge = fudge;
    ret->coldepth = coldepth;
    ret->colseparation = colseparation;
    ret->mincolwidth = mincolwidth;
    ret->mincolheight = mincolheight;
    ret->dirty = dirty;

    return(ret);
}
    

Cvo_GraphObject *
Cvo_HorizontalSideGraph::CloneMe(Cvo_Object *o, Cvo_Chart *c)
{
    Cvo_HorizontalSideGraph *ret = new Cvo_HorizontalSideGraph("grafik", o, c);
    ret->colwidth = colwidth;
    ret->fudge = fudge;
    ret->coldepth = coldepth;
    ret->colseparation = colseparation;
    ret->mincolwidth = mincolwidth;
    ret->mincolheight = mincolheight;
    ret->dirty = dirty;

    return(ret);
}

void
Cvo_HorizontalGraph::SetText(wchar_t *s)
{
    if (marklabel) {
	marklabel->SetText(s);
	RootObject()->ToLayoutWindow()->ReLayout(1);
    } else {
	marklabel = new Cvo_Label("marklabel", this, s);
	marklabel->PlaceAfter(frame2);
	marklabel->MatchX(graph);
	marklabel->MatchWidth(graph);
	if (Mapped())
	    marklabel->Map();
    }
}

void
Cvo_HorizontalGraph::SetColumnText(wchar_t *s)
{
    int len = 0;
    wchar_t *p = s;

    while(*p++)
	len++;

    wchar_t *buf = new wchar_t[len*2];

    for(int i = 0; i < len; i++) {
	buf[i*2] = s[i];
	buf[i*2+1] = '\n';
    }

    buf[len*2-1] = '\0';

    if (cmarklabel) {
        cmarklabel->SetText(buf);
        RootObject()->ToLayoutWindow()->ReLayout(1);
    } else {
	cmarklabel = new Cvo_Label("marklabel", frame1, buf);
	cmarklabel->PlaceBefore(cticks);
	cmarklabel->MatchY(graph);
	cmarklabel->MatchHeight(graph);
	if (Mapped())
	    cmarklabel->Map();
    }

    delete[] buf;
}


Cvo_GraphMark *
Cvo_HorizontalGraph::AddMark(Cvo_ChartMark *m)
{
    Cvo_GraphMark *ret = new Cvo_GraphMark(&marklist, m);
                                           
    ret->item = ticks->AddString(m->string);
    dirty = True;
    return(ret);
}

Cvo_GraphMark *
Cvo_HorizontalGraph::AddColumnMark(Cvo_ChartMark *m)
{
    Cvo_GraphMark *ret = new Cvo_GraphMark(&cmarklist, m);
                                           
    ret->item = cticks->AddString(m->string);
    dirty = True;
    return(ret);
}


void
Cvo_HorizontalGraph::ClearMarks()
{
    while(marklist)
        RemoveMark(marklist);
    ticks->Redraw();
}

void
Cvo_HorizontalGraph::ClearColumnMarks()
{
    while(cmarklist)
        RemoveColumnMark(cmarklist);
    cticks->Redraw();
}

int
Cvo_HorizontalGraph::ComputeX(int i)
{
    return((colwidth + colseparation) * i
	   + colseparation + coldepth + 1
           + ((i < fudge) ? i : fudge));
}


void
Cvo_HorizontalGraph::DrawMarks()
{                        
    if (!scale)
        return;

    int max = graph->Width() - coldepth - 2;

    graph->SetForeground(Linecolor());

    for (Cvo_GraphMark *sm = marklist; sm; sm = sm->Next()) {
        int y = ticks->Height() - ticks->StringHeight(sm->mark->string);
        int v = (int)floor(sm->mark->value * scale + 0.5);
        int x = v + 1;
        int sw = ticks->StringWidth(sm->mark->string);

        if ((v < 0) || (v > max)) {
            x = -1; y = -1;
        } else if (v == 0) {
            x = 0;
        } else if (v < (sw/2)) {
            // Diagonal
            if (coldepth > 1)
                graph->DrawLine(v+1, graph->Height()-2,
                    v+coldepth, graph->Height()-coldepth-1);
            else if (coldepth)
                graph->DrawPoint(v, graph->Height()-2);

            // Vertical
            graph->DrawLine(v+coldepth, 1, 
                    v+coldepth, graph->Height()-coldepth-2);
					
            x = 0;
        } else if (v != max) {
            // Diagonal
            if (coldepth > 1)
                graph->DrawLine(v+1, graph->Height()-2,
                    v+coldepth, graph->Height()-coldepth-1);
            else if (coldepth)
                graph->DrawPoint(v,graph->Height()-2);

            // Vertical
            graph->DrawLine(v+coldepth, 1,
                    v+coldepth, graph->Height()-coldepth-2);
					
            x -= sw/2;
        } else {
            x -= sw/2;
        }

        if ((x+sw) > graph->Width())
            x = graph->Width() - sw;

        sm->item->SetLocation(x, y);
    }

    int wid = cticks->Width();
    for (sm = cmarklist; sm; sm = sm->Next()) {
        int y = ComputeX(int(sm->mark->value));
        int sw = cticks->StringWidth(sm->mark->string);

        y += colwidth/2 - cticks->StringHeight(sm->mark->string)/2;
        sm->item->SetLocation(wid-sw, y);
    }

    ticks->Redraw(); 
    cticks->Redraw();
}

void
Cvo_HorizontalGraph::EvaluateMarks()
{
    return;
}


void
Cvo_HorizontalGraph::EvaluateColumnMarks()
{
    if (!cticks)
        return;

    int maxx = 1;
    int maxy = 1;
    int toty = 0;

    for (Cvo_GraphMark *sm = cmarklist; sm; sm = sm->Next()) {
        int x = cticks->StringWidth(sm->mark->string);
        int y = cticks->StringHeight(sm->mark->string);

        if (x > maxx)
            maxx = x;
        if (y > maxy)
            maxy = y;
        toty += y;
    }
    cticks->SetMinSize(0,0);
    cticks->SetMinPixelSize(maxx,toty);
    SetMinColumnWidth(maxy);
}


CARD16
Cvo_HorizontalGraph::CalculateHeight()
{
    int wid = 20 + coldepth;
    int hei = coldepth + colseparation +
	      Ncolumns() * (mincolwidth + colseparation);

    int theight = cmarklabel ? cmarklabel->CalculateHeight() : 0;

    hei = (hei < theight) ? theight : hei;

    int twidth = marklabel ? marklabel->CalculateWidth() : 0;

    wid = (wid < twidth) ? twidth : wid;

    graph->SetMinSize(0,0);
    graph->SetMinPixelSize(wid, hei);

    return(0);
}


CARD16
Cvo_HorizontalGraph::CalculateWidth()
{
    return(0);
}


void
Cvo_HorizontalGraph::FillRhombus(Cvo_Window *obj, int x, int y, int w, int h, int xoff, int yoff)
{
    XPoint poly[5];

    if (!pixmapbacking)
        obj->ToXCoord(&x, &y);

    poly[0].x = x;
    poly[0].y = y;

    poly[1].x = x + w;
    poly[1].y = y + yoff;

    poly[2].x = x + w + xoff;
    poly[2].y = y + yoff + h;

    poly[3].x = x + xoff;
    poly[3].y = y + h;
    
    poly[4] = poly[0];

    if (pixmapbacking) {
        XFillPolygon(obj->Dpy(),
		     ((Cvo_Paint*)obj)->GetPixmap()->Object(),
		     obj->Gc(), poly, 5, Convex, CoordModeOrigin);
        ((Cvo_Paint *)obj)->MakeDirty();
    } else
        XFillPolygon(obj->Dpy(), obj->Object(), obj->Gc(),
             	     poly, 5, Convex, CoordModeOrigin);
}

void
Cvo_HorizontalGraph::DrawIt(XEvent *, void *)
{
    if (!Object())
        return;

    if (graph->Mapped())
        graph->Erase();

    int ncolors=Ncolors();
    int ncolumns=Ncolumns();
    double** columns=Columns();

    double mw = 0.0;

    for (int i = 0; i < ncolumns; i++) {
        double tw = 0.0;
        for (int j = 0; j < ncolors; j++)
            tw += columns[i][j];

        if (tw > mw)
            mw = tw;
    }

    if (mw < mincolheight)
        mw = mincolheight;

    int x = 0;
    int y = graph->Height() - 1;
    int h = y - coldepth;
    int w = graph->Width() - coldepth - 2;

    if (mw == 0.0)
        scale = w;
    else
        scale = w / mw;

    if (ncolumns == 0)
        return;

    colwidth = (h - colseparation - 1) / ncolumns;

    fudge = (h - colseparation - 1) - colwidth * ncolumns;

    colwidth -= colseparation;

    graph->SetForeground(Backcolor());
    graph->FillRectangle(coldepth+1, 0, w, h);
    if (coldepth) {
        FillRhombus(graph, 0, coldepth, coldepth+1, h, 0, -coldepth);
        FillRhombus(graph, coldepth, h, w+1, coldepth, -coldepth, 0);
    }

    /*   x                   w
             _____________________
            /|        7          |
          6/ |                   |
          /  |                   |
          |  |4                  |8
          |  |                   |
     h   1|  |___________________|
          |  /        5         /
          | /2                 /9
     y    |/__________________/
                    3
    */

    graph->SetForeground(Linecolor());

    if (coldepth) {
        graph->DrawLine(0, coldepth+1, 0, y);                       // Line 1
        graph->DrawLine(0, y, coldepth+1, h-1);                     // Line 2
        graph->DrawLine(0, y, w, y);                                // Line 3
        graph->DrawLine(coldepth+1, 0, coldepth+1, h-1);            // Line 4
        graph->DrawLine(coldepth+1, h-1, w+coldepth+1, h-1);        // Line 5
        graph->DrawLine(0, coldepth, coldepth, 0);                  // Line 6
        graph->DrawLine(coldepth+1, 0, w+coldepth+1, 0);            // Line 7
        graph->DrawLine(w+coldepth+1, 0, w+coldepth+1, h-1);        // Line 8
        graph->DrawLine(w+1, y, w+coldepth+1, h);                   // Line 9
    } else {
        graph->DrawLine(0, 0, 0, h-1);                              // Line 4
        graph->DrawLine(0, h, w+1, h);                              // Line 5
        graph->DrawLine(0, 0, w+1, 0);                              // Line 7
        graph->DrawLine(w+1, 0, w+1, h-1);                          // Line 8
    }

    DrawMarks();

    for (i = ncolumns - 1; i >= 0; i--) {
        double x = 0.0;
        int xx = 1;
        for (int j = 0; j < ncolors; ++j) {
            double h = columns[i][j];
            if (h) {
                int v = (int)floor((x+h)*scale);
                int he = (int)floor(x*scale);

                DrawBox(xx, ComputeX(i), (v==he)?1:v-he, Colors()[j]);
                x += h;
                xx += v-he;
            }
        }
    }
    dirty = False;
}


void
Cvo_HorizontalSideGraph::DrawIt(XEvent *, void *)
{
    if (!Object())
        return;

    if (graph->Mapped())
        graph->Erase();

    int ncolors=Ncolors();
    int ncolumns=Ncolumns();
    double** columns=Columns();

    double mw = 0.0;

    for (int i = 0; i < ncolumns; i++) {
        for (int j = 0; j < ncolors; j++) {
            double th = Columns()[i][j];
            if (th > mw)
                mw = th;
        }
    }

    if (mw < mincolheight)
        mw = mincolheight;

    int x = 0;
    int y = graph->Height() - 1;
    int h = y - coldepth;
    int w = graph->Width() - coldepth - 2;

    if (mw == 0.0)
        scale = w;
    else
        scale = w / mw;

    if (ncolumns == 0)
        return;

    colwidth = (h - colseparation - 1) / ncolumns;

    fudge = (h - colseparation - 1) - colwidth * ncolumns;

    colwidth -= colseparation;

    graph->SetForeground(Backcolor());
    graph->FillRectangle(coldepth+1, 0, w, h);
    if (coldepth) {
        FillRhombus(graph, 0, coldepth, coldepth+1, h, 0, -coldepth);
        FillRhombus(graph, coldepth, h, w+1, coldepth, -coldepth, 0);
    }

    /*   x                   w
             _____________________
            /|        7          |
          6/ |                   |
          /  |                   |
          |  |4                  |8
          |  |                   |
     h   1|  |___________________|
          |  /        5         /
          | /2                 /9
     y    |/__________________/
                    3
    */

    graph->SetForeground(Linecolor());

    if (coldepth) {
        graph->DrawLine(0, coldepth+1, 0, y);                       // Line 1
        graph->DrawLine(0, y, coldepth+1, h-1);                     // Line 2
        graph->DrawLine(0, y, w, y);                                // Line 3
        graph->DrawLine(coldepth+1, 0, coldepth+1, h-1);            // Line 4
        graph->DrawLine(coldepth+1, h-1, w+coldepth+1, h-1);        // Line 5
        graph->DrawLine(0, coldepth, coldepth, 0);                  // Line 6
        graph->DrawLine(coldepth+1, 0, w+coldepth+1, 0);            // Line 7
        graph->DrawLine(w+coldepth+1, 0, w+coldepth+1, h-1);        // Line 8
        graph->DrawLine(w+1, y, w+coldepth+1, h);                   // Line 9
    } else {
        graph->DrawLine(0, 0, 0, h-1);                              // Line 4
        graph->DrawLine(0, h, w+1, h);                              // Line 5
        graph->DrawLine(0, 0, w+1, 0);                              // Line 7
        graph->DrawLine(w+1, 0, w+1, h-1);                          // Line 8
    }

    DrawMarks();

    int ndraw=0;

    Bool *draw = new Bool[ncolors];
    for(i=0;i<ncolors;i++)
        draw[i]=False;
    for (i=0; i<ncolors; i++) {
        for (int j=0; j<ncolumns; j++) {
            if (!draw[i] && columns[j][i]) {
                draw[i]=True;
                ndraw++;
            }
	}
    }

    for (i = ncolumns - 1; i >= 0; i--) {
        int pos = ndraw-1;
        for (int j = ncolors-1; j >= 0; j--) {
            double h = columns[i][j];
            if (h) {
                int v = (int)floor(h*scale);

                DrawBox(1, ComputeX(i), v, colwidth/ndraw, pos, Colors()[j]);
            }
            if (draw[j])
                pos--;
        }
    }

    delete[] draw;

    dirty = False;
}


void
Cvo_HorizontalGraph::DrawBox (int x, int y, int h, const Cvo_Color &c)
{
    graph->SetForeground(c);

    graph->FillRectangle(x, y, h, colwidth);

    if (coldepth) {
        graph->SetForeground(c->Upper());

        if (coldepth > 1)
            FillRhombus(graph, x-1, y, h, -coldepth, coldepth , 0);
        else
            graph->DrawLine(x, y, x+h, y);


        graph->SetForeground(c->Lower());

        if (coldepth > 1)
            FillRhombus(graph, x+h, y, coldepth, colwidth, 0, -coldepth);
        else
            graph->DrawLine(x+h,y,x+h,y+colwidth-2);

        graph->SetForeground(c);
        
        if (coldepth > 1)
            graph->DrawLine(x+h, y-1, x+h+coldepth-1, y-coldepth);
        else
            graph->DrawPoint(x+h, y-1);
    }
}
    

void
Cvo_HorizontalSideGraph::DrawBox(int x, int y, int h, int width, int pos, const Cvo_Color &c)
{
    graph->SetForeground(c);

    graph->FillRectangle(x, y+width*pos, h, width);

    if (coldepth) {
        graph->SetForeground(c->Upper());

        if (coldepth>1)
            FillRhombus(graph, x-1, y+width*pos, h, -coldepth, coldepth , 0);
        else
            graph->DrawLine(x, y+width*pos, x+h, y+width*pos);


        graph->SetForeground(c->Lower());

        if (coldepth>1)
            FillRhombus(graph, x+h, y+width*pos, coldepth, width, 0, -coldepth);
        else
            graph->DrawLine(x+h,y+width*pos,x+h,y+width+width*pos-2);

        graph->SetForeground(c);
        
        if (coldepth>1)
            graph->DrawLine(x+h, y-1+width*pos,
                	    x+h+coldepth-1, y-coldepth+width*pos);
        else
            graph->DrawPoint(x+h, y-1+width*pos);
    }
}

void
Cvo_HorizontalGraph::HandleEvent(XEvent* ev, void*)
{
    static unsigned cnt=0;

    if (ev->type==ButtonPress) {
        cnt++;
        return;
    }

    if (ev->type==ButtonRelease) {
        if (--cnt==0) {
            int xo,yo;
            graph->XOrigin(&xo,&yo);
            int x = ev->xbutton.x - xo;
            int y = ev->xbutton.y - yo;
            graph->FromXCoord(&x,&y);

            if ((x >= 0) && (y >= 0)
			 && (x < graph->XWidth())
			 && (y < graph->XHeight())) {
                Cvo_ChartEvent cev;

                cev.button = ev->xbutton.button;
                cev.state = ev->xbutton.state;

                int ncolumns = Ncolumns();
                int ncolors = Ncolors();
                double** columns = Columns();

                for (int i = 0; i < ncolumns; i++) {
                    for (int j = ncolors-1; j >= 0; j--) {
                        double h = columns[i][j];
                        if (h) {
                            double xv = 0.0;
                            for (int k = 0; k < j; k++)
                                xv += columns[i][k];

                            int v = (int)floor((xv+h)*scale);
                            int he = (int)floor(xv*scale);

                            int left = he;
                            int right = v;
                            int top = ComputeX(i);
                            int bottom = top + colwidth - 1;

			    cev.bar = i;
			    cev.color = j;
			    cev.value = h;

			    if ((x >= left) && (x <= right)
					    && (y >= top) && (y <= bottom)) {
				chartob->SendEvent(CvoChartEvent, &cev);
				return;
			    }

			    if (coldepth) {
				if ((y >= (top-coldepth))
					&& (y < top)
					&& ((x+y) > (left+top-2))
					&& ((x+y) <= (right+top))) {
				    chartob->SendEvent(CvoChartEvent, &cev);
				    return;
				}

				if ((x >= right) && (x <= (right+coldepth))
					 && ((x+y) > (right+top))
					 && ((x+y) <= (right+bottom+1))) {
				    chartob->SendEvent(CvoChartEvent, &cev);
				    return;
				}
			    }
                        }
                    }
                }
            }
        }
    }
}

void
Cvo_HorizontalSideGraph::HandleEvent(XEvent *ev, void *)
{
    static unsigned cnt=0;

    if (ev->type == ButtonPress) {
	    cnt++;
	    return;
    }

    if (ev->type == ButtonRelease) {
	if (--cnt == 0) {
	    int xo, yo;
	    graph->XOrigin(&xo,&yo);
	    int x = ev->xbutton.x - xo;
	    int y = ev->xbutton.y - yo;
	    graph->FromXCoord(&x,&y);

	    if (x >= 0 && y >= 0
		       && x < graph->XWidth()
		       && y < graph->XHeight()) {
		Cvo_ChartEvent cev;

		cev.button = ev->xbutton.button;
		cev.state = ev->xbutton.state;

		int ncolumns = Ncolumns();
		int ncolors = Ncolors();
		double** columns = Columns();
		int ndraw = 0;
		Bool *draw = new Bool[ncolors];

		for (int i=0;i<ncolors;i++)
			draw[i]=False;

		for (i=0;i<ncolors;i++) {
		    for (int j=0;j<ncolumns;j++) {
			if (!draw[i] && columns[j][i]) {
			    draw[i]=True;
			    ndraw++;
			}
		    }
		}

		for (i = 0; i < ncolumns; i++) {
		    int pos = 0;
		    for (int j = 0; j < ncolors; j++) {
			double h = columns[i][j];
			if (h) {
			    int v = (int)floor(h*scale);

			    int left = 1;
			    int right = v;
			    int top = ComputeX(i) + (colwidth/ndraw) * pos;
			    int bottom = top + (colwidth/ndraw) -1;

			    cev.bar = i;
			    cev.color = j;
			    cev.value = h;

			    if (x >= left && x <= right
					  && y >= top && y <= bottom) {
				chartob->SendEvent(CvoChartEvent, &cev);
				return;
			    }

			    if (coldepth) {
				if (y >= top - coldepth && y < top
							&& x + y > left+top-2
							&& x + y <= right+top) {
				    chartob->SendEvent(CvoChartEvent, &cev);
				    return;
				}

				if (x >= right && x <= right + coldepth
					       && x + y > right + top
					       && x + y <= right + bottom + 1) {
				    chartob->SendEvent(CvoChartEvent, &cev);
				    return;
				}
			    }
			}
			if (draw[j])
			    pos++;
		    }
		}
	    }
	}
    }
}
