//
// Copyright 1994, Cray Research, Inc.
//                 
// Permission to use, copy, modify and distribute this software and
// its accompanying documentation (the "Software") is granted without
// fee, provided that the above copyright notice and this permission
// notice appear in all copies of the Software and all supporting
// documentation, and the name of Cray Research, Inc. not be used in
// advertising or publicity pertaining to distribution of the 
// Software without the prior specific, written permission of Cray
// Research, Inc.  The Software is a proprietary product of Cray
// Research, Inc., and all rights not specifically granted by this
// license shall remain in Cray Research, Inc.  No charge may be made
// for the use or distribution of the Software.  The Software may be
// distributed as a part of a different product for which a fee is
// charged, if (i) that product contains or provides substantial
// functionality that is additional to, or different from, the
// functionality of the Software, and (ii) no separate, special or
// direct charge is made for the Software.
//         
// THE SOFTWARE IS MADE AVAILABLE "AS IS", AND ALL EXPRESS AND
// IMPLIED WARRANTIES, INCLUDING THE IMPLIED WARRANTIES OF FITNESS
// FOR A PARTICULAR PURPOSE, MERCHANTABILITY, AND FREEDOM FROM
// VIOLATION OF THIRD PARTY INTELLECTUAL PROPERTY RIGHTS, ARE HEREBY
// DISCLAIMED AND EXCLUDED BY CRAY RESEARCH, INC.  CRAY RESEARCH,
// INC. WILL NOT BE LIABLE IN ANY EVENT FOR ANY CONSEQUENTIAL,
// SPECIAL, INCIDENTAL, OR INDIRECT DAMAGES ARISING OUT OF OR IN
// CONNECTION WITH THE PERFORMANCE OF THE SOFTWARE OR ITS USE BY ANY
// PERSON, OR ANY FAILURE OR NEGLIGENCE ON THE PART OF CRAY RESEARCH,
// INC., EXCEPT FOR THE GROSS NEGLIGENCE OR WILLFUL MISCONDUCT OF
// CRAY RESEARCH.
// 
// This License Agreement shall be governed by, and interpreted and
// construed in accordance with, the laws of the State of Minnesota,
// without reference to its provisions on the conflicts of laws, and
// excluding the United Nations Convention of the International Sale
// of Goods.
//
//	USMID %Z%%M%	%I%	%G% %U%
//	$Id: Ev.h++,v 1.7 1994/08/10 18:11:11 prb Exp $
//
//
// This code was initially contributed by Jim Nowicki 10-92
//
#if	!defined(_Cvo_Ev_)
#define _Cvo_Ev_

#include <Cvo/_Machine.h++>
#include <signal.h>
#include <unistd.h>
#if	!defined(_Cvo_FdSet_)
#include <Cvo/FdSet.h++>
#endif
#if	!defined(_Cvo_TimeVal_)
#include <Cvo/TimeVal.h++>
#endif
#if defined(__NEED_OSFCN__)
#include <osfcn.h>
#endif
#if defined(__NEED_SELECT_DEF__)
extern "C" int select(int, fd_set *, fd_set *, fd_set *, struct timeval *);
#endif

//
// We define our own assert macro so that assert will abort
// instead of exiting.  This way, if stderr gets lost, we still have the core
//
#if	!defined(NDEBUG)
#   include <stdio.h>
#   define _assert(ex)  ((void)((ex)? 0 : ((void)fprintf(stderr,"Assertion failed: file \"%s\", line %d\n", __FILE__, __LINE__),fflush(stderr),abort(),0)))
#   define assert(ex)   _assert(ex)
#else
#   define _assert(ex)
#   define assert(ex)
#endif

//
// Forward definitions of the event classes
//
class EvIOEvent;
class EvTimerEvent;
class EvWorkProcEvent;
class EvSignal;

//
// Define the types of the callback functions
//
typedef void (*EvIOProc)(EvIOEvent *, void *);
typedef void (*EvTimerProc)(EvTimerEvent *, void *);
typedef int  (*EvWorkProcProc)(EvWorkProcEvent *, void *);
typedef void (*EvSignalProc)(EvSignal *, void *);

//
// Define the different IO conditions
//
enum EvIOCond {
    EvInput = 0,	// First entry must be zero
    EvOutput,
    EvExcept,
    EvNumberCond	// MUST BE LAST
};

const int EV_NODELETE = 0;
const int EV_DELETE = 1;

//
// ProcEvent control options
//

#if	!defined(True)
#define	True	1		// These must match the X11 values
#define	False	0
#endif
#define EV_POLL		False
#define EV_BLOCK	True
#define EV_LOOP		(False + True + 1)

#if	defined(__SIGNAL_RETURN_INT__)
typedef void (*EV_SIG_CALLBACK_PROC)(int);
#elif	defined(__SIGNAL_RETURN_UNKNOWN__)
typedef void (*EV_SIG_CALLBACK_PROC)(...);
#else
typedef void (*EV_SIG_CALLBACK_PROC)(int, ...);
#endif

//
// Define class to keep track of event masks
//
class EvFdMasks {
 public:
    EvFdMasks();
    EvFdMasks(const EvFdSet &);
    EvFdMasks(const EvFdMasks &);
    EvFdMasks &operator=(const EvFdMasks &);
    fd_set *Inputs() const;
    fd_set *Outputs() const;
    fd_set *Excepts() const;
    int IsSet(int d, EvIOCond cond) const;
    int IsSet(int d) const;
    void Set(int d, EvIOCond cond);
    void Clr(int d, EvIOCond cond);
    void Clr(int d);
    int CopyFd(const EvFdMasks &m, int d);

 private:
    void Copy(const EvFdMasks &);
    EvFdSet masks[EvNumberCond];
};

//
// Define a class which will queue up received signals
//
const int EvNumberSignalQueues = 2;
typedef unsigned char EvSigNbr_t;

class EvSigQ {
 public:
    inline EvSigQ();
    sig_atomic_t Active() const;
    sig_atomic_t SetActive();
    void         SetInactive();
    int          Empty() const;
    int          Full() const;
    void         QuickPush(int n);
    int          QuickPop();
    int          Push(int n);
    int          Pop();

 private:
    enum {
        EVSQ_L2SIZE   = 7,
        EVSQ_SIZE     = (1 << 7),
        EVSQ_SIZEMASK = (1 << 7) - 1
    };

    sig_atomic_t active;        	// Queue active flag
    sig_atomic_t head;          	// Queue head index
    sig_atomic_t tail;          	// Queue tail index
    EvSigNbr_t   signals[EVSQ_SIZE];

    int Next(sig_atomic_t n) const;
    int Advance(sig_atomic_t &n);
};

//
// Define the event classes
//
class Ev {
 public:
    static void ProcessEvents(int);
    static void SetFlush(void (*f)());

    static int  	    nbrFds;         // Number of file descriptors
 protected:
    static int  	    eventsChanged;  // Flag of changed events
    static int  	    retryCount;     // Error retry count
    static int              nbrEvents;      // Number of pending IO events

    static EvFdMasks       *masks;              // Masks of file descriptors
    static EvFdMasks       *pendingMasks;       // Pending I/O descriptors
    static EvIOEvent       *evList;             // Current pos. in IO event list
    static EvIOEvent       *ioEvents;           // IO Event list
    static EvTimerEvent    *timers;             // Timer event list
    static EvWorkProcEvent *workProcs;          // Work proc list
    static void		  (*flush)();		// Flush proc

    EvVolatile static EvSigQ		signalQs[EvNumberSignalQueues];
    EvVolatile static sig_atomic_t	sigPending;
    static EV_SIG_CALLBACK_PROC		def_sig[NSIG+1];
    static EvSignal 			*signals[NSIG+1];
    static int 				nbrSignals;

    Ev   		   *next;	// Link to next event
    void		   *userData;	// User data for callback
    int  		    flags;	// Event flags

    enum {
        EV_ACTIVE       = 1 << 0,
        EV_DODELETE     = 1 << 1,
        EV_RETRY_MAX    = 10	    // Max number of retries
    };

    Ev();
    void Link(Ev **h);
    void Unlink(Ev **);
    friend void Ev_SignalCatcher(int);
    static void DispatchSignalHandlers();

    static void InitMasks();

 private:
    static int CheckBadFD();
    static int EvSelect(EvFdMasks *m, EvTimeVal *tv);
};

class EvIOEvent : public Ev {
 public:
    EvIOEvent(int, EvIOCond, EvIOProc, void * = 0);
    ~EvIOEvent();

    int Fd() const;
    EvIOCond Cond() const;

    int IsEvent(int f, EvIOCond c) const;
    int IsEvent(int f) const;
    int IsEvent(const EvFdMasks *m) const;
    EvIOEvent *Next();
    int Enable();
    int Disable();

 protected:
    int           fd;                       // Event file descriptor
    EvIOCond      cond;                     // Event condition
    EvIOProc      callback;                 // Callback function
    void CallCallback();
  private:
    friend class Ev;
};

class EvInputEvent : public EvIOEvent {
 public:
    EvInputEvent(int f, EvIOProc cb, void *ud = 0);
};

class EvOutputEvent : public EvIOEvent {
 public:
    EvOutputEvent(int f, EvIOProc cb, void *ud = 0);
};

class EvExceptEvent : public EvIOEvent {
 public:
    EvExceptEvent(int f, EvIOProc cb, void *ud = 0);
};

class EvTimerEvent : public Ev {
 public:
    EvTimerEvent(const EvTimeVal &tv, EvTimerProc, void * = 0, int del = 1);
    EvTimerEvent(long interval, EvTimerProc, void * = 0, int del = 1);
    ~EvTimerEvent();

    void Set(const EvTimeVal &tv, EvTimerProc cb = 0, void * = 0);
    void Set(long interval, EvTimerProc cb = 0, void * = 0);
    void Enable();
    void Disable();

    EvTimerEvent *Next();
    EvTimeVal &Time();
    void CallCallback();

 protected:
    EvTimeVal     eventTime;      // Event time
    EvTimerProc   callback;       // Callback

 private:
    friend Ev;
    void BasicSet(const EvTimeVal &, EvTimerProc = 0, void * = 0);
};

class EvWorkProcEvent : public Ev {
 public:
    EvWorkProcEvent(EvWorkProcProc, void * = 0, int del = 0);
    ~EvWorkProcEvent();
    EvWorkProcEvent *Next();
    int CallCallback();

 private:
    friend Ev;
    EvWorkProcProc      callback;
};

class EvSignal : public Ev {
 public:
    EvSignal(int, EvSignalProc, void * = 0);
    ~EvSignal();
    int Signal() const;
    void CallCallback();

 private:
    EvSignalProc callback;
    int          signbr;
};



// Inlined Methods for EvFdMasks:
inline EvFdMasks::EvFdMasks()                   { ; }
inline void EvFdMasks::Copy(const EvFdMasks &x)
    { masks[EvInput] = x.masks[EvInput];
      masks[EvOutput] = x.masks[EvOutput];
      masks[EvExcept] = x.masks[EvExcept];
    }
inline EvFdMasks::EvFdMasks(const EvFdMasks &x) { Copy(x); }
inline EvFdMasks &EvFdMasks::operator=(const EvFdMasks &x)
    { Copy(x); return *this; }

inline fd_set *EvFdMasks::Inputs() const        { return masks[EvInput]; }
inline fd_set *EvFdMasks::Outputs() const       { return masks[EvOutput]; }
inline fd_set *EvFdMasks::Excepts() const       { return masks[EvExcept]; }
inline void EvFdMasks::Set(int d, EvIOCond cnd) { masks[cnd].Set(d); }
inline void EvFdMasks::Clr(int d, EvIOCond cnd) { masks[cnd].Clr(d); }

inline int EvFdMasks::IsSet(int d, EvIOCond cond) const
    { return masks[cond].IsSet(d); }

inline int EvFdMasks::IsSet(int d) const
    { return IsSet(d, EvInput) || IsSet(d, EvOutput) || IsSet(d, EvExcept);}

inline void EvFdMasks::Clr(int d)
    { Clr(d, EvInput); Clr(d, EvOutput); Clr(d, EvExcept); }

inline int EvFdMasks::CopyFd(const EvFdMasks &m, int d)
    { int ret = 0;
      if (m.IsSet(d, EvInput))  { ret = 1; Set(d, EvInput);  }
      if (m.IsSet(d, EvOutput)) { ret = 1; Set(d, EvOutput); }
      if (m.IsSet(d, EvExcept)) { ret = 1; Set(d, EvExcept); }
      return ret;
    }


// Inlined methods for EvSigQ:
inline EvSigQ::EvSigQ()                        { active = head = tail = 0; }
inline int EvSigQ::Next(sig_atomic_t n) const  { return (n+1) & EVSQ_SIZEMASK; }
inline int EvSigQ::Advance(sig_atomic_t &n)    { return n = Next(n); }
inline sig_atomic_t EvSigQ::Active() const     { return active; }

inline void EvSigQ::SetInactive()       { active = 0; }
inline int  EvSigQ::Empty() const       { return head == tail; }
inline int  EvSigQ::Full() const        { return Next(head) == tail; }
inline void EvSigQ::QuickPush(int n)    { signals[head] = n; Advance(head); }

inline sig_atomic_t EvSigQ::SetActive()
    { sig_atomic_t oa = active; active = 1; return(oa); }
inline int  EvSigQ::QuickPop()
    { int n = signals[tail]; Advance(tail); return n; }
inline int  EvSigQ::Push(int n)
    { int r = !Full(); if (r) QuickPush(n); return r; }
inline int  EvSigQ::Pop()
    { int n=0; if (!Empty()) n = QuickPop(); return n; }


// Inlined Methods for Ev:
inline Ev::Ev()                 { next = 0; flags = EV_ACTIVE; userData = 0; }
inline void Ev::Link(Ev **h)   	{ assert(h != 0); next = *h; *h = this; }
inline void Ev::InitMasks()
    {
        if (masks == 0) {
           masks        = new EvFdMasks;
           pendingMasks = new EvFdMasks;
           nbrFds       = EvFdSet::NbrFds();
           assert(masks != 0 && pendingMasks != 0 && nbrFds != 0);
        }
    }
inline int Ev::EvSelect(EvFdMasks *m, EvTimeVal *tv)
    { return(select(nbrFds, m->Inputs(), m->Outputs(), m->Excepts(), tv)); }
inline void Ev::SetFlush(void (*f)())	    { flush = f; }


// Inlined Methods for EvIOEvent and subclasses:
inline void EvIOEvent::CallCallback()       { (*callback)(this, userData); }
inline int  EvIOEvent::Fd() const           { return fd; }
inline EvIOCond EvIOEvent::Cond() const     { return cond; }
inline int EvIOEvent::IsEvent(int f) const  { return f == fd; }
inline EvIOEvent *EvIOEvent::Next()         { return (EvIOEvent *) next; }
inline int EvIOEvent::IsEvent(int f, EvIOCond c) const
    { return f == fd && c == cond; }
inline int EvIOEvent::IsEvent(const EvFdMasks *m) const
    { return m->IsSet(fd, cond); }
inline int EvIOEvent::Enable()
    { int r = masks->IsSet(fd, cond); masks->Set(fd, cond); return(r); }
inline int EvIOEvent::Disable()
    { int r = masks->IsSet(fd, cond); masks->Clr(fd, cond); return(r); }

inline EvInputEvent::EvInputEvent(int f, EvIOProc cb, void *ud)
    : EvIOEvent(f, EvInput, cb, ud) { ; }
inline EvOutputEvent::EvOutputEvent(int f, EvIOProc cb, void *ud)
    : EvIOEvent(f, EvOutput, cb, ud) { ; }
inline EvExceptEvent::EvExceptEvent(int f, EvIOProc cb, void *ud)
    : EvIOEvent(f, EvExcept, cb, ud) { ; }


// Inlined Methods for EvTimerEvent:
inline EvTimerEvent::~EvTimerEvent()            { Unlink((Ev **) &timers); }
inline EvTimerEvent *EvTimerEvent::Next()       { return (EvTimerEvent *)next; }
inline EvTimeVal &EvTimerEvent::Time()          { return eventTime; }
inline void EvTimerEvent::CallCallback()        { (*callback)(this, userData); }
inline void EvTimerEvent::Disable()             { Unlink((Ev **) &timers); }
inline void EvTimerEvent::Enable()
    { if ((flags & EV_ACTIVE) == 0) BasicSet(eventTime); }



// Inlined Methods for EvWorkProcEvent:
inline EvWorkProcEvent::~EvWorkProcEvent()      { Unlink((Ev **) &workProcs); }
inline EvWorkProcEvent *EvWorkProcEvent::Next()
    { return (EvWorkProcEvent *) next; }
inline int EvWorkProcEvent::CallCallback()
    { return (*callback)(this, userData); }


// Inlined Methods for EvSignal:
inline int EvSignal::Signal() const     { return signbr; }
inline void EvSignal::CallCallback()    { (*callback)(this, userData); }

#endif
