/* **********************************************************************
 *
 *	database.c	Database Handling
 *	---------------------------------
 *	Functions:
 *
 *	void DBClearObjectsFromMemory()
 *	long DBReadFromFile(char *dbfilename)
 *	long DBSaveToFile(char *dbfilename)
 *	long DBCleanMemory();
 *	long DBCreateObject(int width, int height)
 *	long DBCreateExplosion1(double loc_x, double loc_y)
 *      long DBCreateExplosion2(double loc_x, double loc_y)
 *	void DBRecycleObject(long object_num)
 *      void DBEmergancySave()
 *	long DBGetTopObjectNumber()
 *      double DBGetObjectVisibility(long object_num)
 *
 *	---
 *
 *	Database IO is handled by these functions, including reading
 *	and writing database files, creating and removing database
 *	entries, and database management.   The XSW database is made up
 *	up of xsw_object[] entries.
 *
 *	DBClearObjectsFromMemory() removes any DB objects from memory,
 *	setting all of their types to XSW_OBJ_TYPE_GARBAGE.
 *
 *	DBReadFromFile() loads all the xsw objects in database file
 *	dbfilename into memory, erasing any xsw objects already in
 *	memory  Returns the topmost xsw object number relative to 0, if 20
 *	xsw objects were loaded, then returns 19.
 *
 *	DBSaveToFile() saves (writes) all xsw objects
 *	in memory into database file dbfilename.   If dbfilename exist
 *	it will be erased!!!   Returns the number of xsw objects
 *	(including those of type error and garbage) written relative to 0.
 *
 *	DBEmergancySave() attempts to make an emergancy save,
 *	writing to a db filename with .EMERGANCY.1234 prepended to
 *	the file's name, where 1234 is the pid.
 *
 *	DBCleanMemory() cleans memory, returns topmost object number
 *	relative to 0.   total_objects + 1 should be assigned this
 *	return value.
 *
 *	DBCreateObject() allocates a new space in memory for a new
 *	xsw object. The new XSW Object will be initally set to be of type
 *	XSW_OBJ_TYPE_STATIC and have an imageset of IMAGESET_CODE_GENERIC.
 *	Return value is the number for the newly created xsw object
 *	relative to 0.
 *
 *	DBRecycleObject() removes xsw object object_num from memory
 *	and frees (recycles) its number for future reuse.   If the
 *	object_num was that of an invalid xsw object, then no action
 *	will be taken.   No return.
 *
 *	DBGetTopObjectNumber() returns the number for the topmost xsw
 *	object loaded in memory.   Return is the number for the topmost
 *	xsw object loaded
 *	memory.
 *
 */

#include "../include/swserv.h"



void DBClearObjectsFromMemory()
{
	long object_num;

	for(object_num = 0; object_num < MAX_OBJECTS; object_num++)
	{
	   DBRecycleObject(object_num);
	}

	return;
}



long DBReadFromFile(char *dbfilename)
{
	/* Local variables. */
	char stringa[256];
	FILE *fp;
	char *filebuf;
	off_t filesize;
	char *strptr;
	struct stat statbuf;

	long lines_read = 0;
	long xswobject_count = -1;	/* Start at -1 since we're  */
					/* counting xsw objects	    */
					/* relative to 0.           */
	long weapons_count = 0;

	/* Does dbfilename exist? */
        if(access(dbfilename,F_OK) == -1)
        {
                return(xswobject_count);
        }


	/* Open dbfilename. */
	fp = fopen(dbfilename,"r");
	if(fp == NULL)
	{
                fprintf(stderr, "Unable to open file: %s\n",
                        dbfilename
                );
                return(xswobject_count);
	}


        /* Get statistics and stuff about the file. */
        fstat(fileno(fp), &statbuf);    /* Get stats on file. */
        filesize = statbuf.st_size;     /* Get size of file. */
        
        /* Allocate memory. */
        filebuf = malloc(filesize * sizeof(char) + 1);


	/* *************************************************** */
	/* ***        Clear DB Objects in Memory           *** */
	DBClearObjectsFromMemory();


	/* ************************************************ */
        /* ***   Begin Loading XSW Objects from DB File *** */
	while( fgets(filebuf,256,fp) != NULL )
	{
	   lines_read++;

	   /* Did we load more than MAX_OBJECTS? */
	   if( xswobject_count >= (MAX_OBJECTS - 1) )
	   {
		fprintf(stderr,
                   "Warning: Maximum of %i XSW Objects loaded.\n",
                   MAX_OBJECTS
                );
		fprintf(stderr,
                   "Warning: Additional XSW Objects not loaded.\n",
                   MAX_OBJECTS
                );
                break;
	   }


           /* Skip comments. */
           if(strstr(filebuf,"#") != NULL)
               continue;
 
           /* Skip short lines. */
           if(strlen(filebuf) < 4)
               continue;

           /* Remove newlines characters. */
           while(1)
           {
              strptr = strchr(filebuf,'\n');
              if(strptr == NULL)
                 break; 
              else  
                 *strptr='\0';
           }


	   /* ************************************************** */
	   /* ***                Read Header                 *** */
	   if(strstr(filebuf,"BeginHeader") != NULL)
	   {
                while((fgets(filebuf,256,fp)) != NULL)
                {
                   lines_read++;
           
                   /* Skip comments. */
                   if(strstr(filebuf,"#") != NULL)
                      continue;

                   /* Skip short lines. */
                   if(strlen(filebuf) < 4)
                      continue;

                   /* Remove newlines characters. */
                   while(1)
                   {
                       strptr = strchr(filebuf,'\n');
                       if(strptr == NULL)
                          break;
                       else
                          *strptr='\0';
                   }

                   /* Title */
                   if(strstr(filebuf,"Title") != NULL)
                   {
                        strptr = strstr(filebuf,"=");
                        strptr += 2;
                        strncpy(dbtitle, strptr, 256);
                   }
  

                   /* EndHeader */
                   else if(strstr(filebuf,"EndHeader") != NULL)
                   {
                        break;
                   }
               }
	   }


	   /* BeginXSWObject */
           if(strstr(filebuf,"BeginXSWObject") != NULL)
	   {
		/* Increment the XSW Object counter. */
                xswobject_count++;

		/* ************************************************ */
		/* Begin fetching the information for this object.  */
		while((fgets(filebuf,256,fp)) != NULL)
                {
                   lines_read++;

	           /* Skip comments. */
        	   if(strstr(filebuf,"#") != NULL)
                      continue;
                
	           /* Skip short lines. */
        	   if(strlen(filebuf) < 4)
                      continue;

                   /* Remove newlines characters. */
                   while(1)
                   {  
                       strptr = strchr(filebuf,'\n');
                       if(strptr == NULL)
                          break;
                       else
                          *strptr='\0';
                   }

		   /* Name. */
		   if(strstr(filebuf,"Name") != NULL)
		   {
                        strptr = strstr(filebuf,"=");
                        strptr += 2;
                        strncpy(xsw_object[xswobject_count].name, strptr,
			    MAX_NAME_LENGTH);
		   }

                   /* Password. */
                   else if(strstr(filebuf,"Password") != NULL)
                   {
                        strptr = strstr(filebuf,"=");
                        strptr += 2;
                        strncpy(xsw_object[xswobject_count].password, strptr,
			    MAX_PASSWORD_LENGTH);
                   }

                   /* Empire. */
                   else if(strstr(filebuf,"Empire") != NULL)
                   {
                        strptr = strstr(filebuf,"=");
                        strptr += 2;
			strncpy(xsw_object[xswobject_count].empire, strptr,
			    MAX_EMPIRE_LENGTH);
                   }
                   /* Type. */  
                   else if(strstr(filebuf,"Type") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].type
                        );
                   }
                   /* ImageSet. */
                   else if(strstr(filebuf,"ImageSet") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].imageset
                        );
                   }
                   /* Owner */
                   else if(strstr(filebuf,"Owner") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].owner
                        );
                   }
                   /* Size */
                   else if(strstr(filebuf,"Size") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].size
                        );
                   }

                   /* LockedOn */
                   else if(strstr(filebuf,"LockedOn") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].locked_on
                        );
                   }

                   /* InterceptingObject */
                   else if(strstr(filebuf,"InterceptingObject") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].intercepting_object
                        );
                   }
 
                   /* Window not supported on server. */

		   /* WindowWidth */
                   else if(strstr(filebuf,"WindowWidth") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].window_width
                        );
                   }
                   /* WindowHeight */  
                   else if(strstr(filebuf,"WindowHeight") != NULL)
                   {       
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].window_height
                        );
                   }
		   /* ScannerRange */
                   else if(strstr(filebuf,"ScannerRange") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].scanner_range
                        );
                   }
                   /* CoordX */
                   else if(strstr(filebuf,"CoordX") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].x
                        );
                   }
                   /* CoordY */
                   else if(strstr(filebuf,"CoordY") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].y
                        );
                   }
                   /* CoordZ (Not implmented yet) */
                   else if(strstr(filebuf,"CoordZ") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].z
                        );
                   }
                   /* ObjectHeading (in radians) */
                   else if(strstr(filebuf,"ObjectHeading") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].heading
                        );
                   }
                   /* ObjectPitch (in radians) (Not implmented yet) */
                   else if(strstr(filebuf,"ObjectPitch") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].pitch
                        );
                   }
                   /* VelocityHeading (in radians) */
                   else if(strstr(filebuf,"VelocityHeading") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].velocity_heading
                        );
                   }
                   /* VelocityMax */
                   else if(strstr(filebuf,"VelocityMax") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].velocity_max
                        );
                   }
                   /* Velocity */
                   else if(strstr(filebuf,"Velocity") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].velocity
                        );
                   }
		   /* ThrustRevState */
                   else if(strstr(filebuf,"ThrustRevState") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].thrust_rev_state
                        );
                   }
                   /* ThrustDir */
                   else if(strstr(filebuf,"ThrustDir") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].thrust_dir
                        );
                   }
                   /* ThrustPower */
                   else if(strstr(filebuf,"ThrustPower") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].thrust_power
                        );
                   }
		   /* Thrust */
                   else if(strstr(filebuf,"Thrust") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].thrust
                        );
                   }
                   /* Throttle */
                   else if(strstr(filebuf,"Throttle") != NULL)   
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].throttle
                        );
                   }
                   /* TurnRate */
                   else if(strstr(filebuf,"TurnRate") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].turnrate
                        );
                   }
                   /* BrakesState */
                   else if(strstr(filebuf,"BrakesState") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].brakes_state
                        );
                   }
                   /* BrakesPower */
                   else if(strstr(filebuf,"BrakesPower") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].brakes_power
                        );
                   }
		   /* HitPointsMax */
                   else if(strstr(filebuf,"HitPointsMax") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].hp_max
                        );
                   }
                   /* HitPoints */
                   else if(strstr(filebuf,"HitPoints") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].hp
                        );
                   }
                   /* PowerPurity */  
                   else if(strstr(filebuf,"PowerPurity") != NULL)  
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].power_purity
                        );
                   }
                   /* PowerMax */
                   else if(strstr(filebuf,"PowerMax") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].power_max
                        );
                   }
                   /* Power */
                   else if(strstr(filebuf,"Power") != NULL)   
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].power
                        );
                   }
                   /* CoreEfficency */
                   else if(strstr(filebuf,"CoreEfficency") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].core_efficency
                        );
                   }
                   /* AntimatterMax */
                   else if(strstr(filebuf,"AntimatterMax") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].antimatter_max
                        );
                   }
                   /* Antimatter */
                   else if(strstr(filebuf,"Antimatter") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].antimatter
                        );
                   }
                   /* ShieldState */   
                   else if(strstr(filebuf,"ShieldState") != NULL)   
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].shield_state
                        );
                   }
                   /* ShieldFrequency */
                   else if(strstr(filebuf,"ShieldFrequency") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].shield_frequency
                        );
                   }
		   /* SelectedWeapon */
                   else if(strstr(filebuf,"SelectedWeapon") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].selected_weapon
                        );
                   }
                   /* TotalWeapons */
                   else if(strstr(filebuf,"TotalWeapons") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].total_weapons
                        );
			if(xsw_object[xswobject_count].total_weapons >
				MAX_WEAPONS)
			{
				xsw_object[xswobject_count].total_weapons = 
					MAX_WEAPONS;
			}
                   }
                   /* BirthTime */
                   else if(strstr(filebuf,"BirthTime") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].birth_time
                        ); 
                   }
                   /* Age */
                   else if(strstr(filebuf,"Age") != NULL)
                   {  
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].age
                        );
                   }
                   /* LifeSpan */
                   else if(strstr(filebuf,"LifeSpan") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].lifespan
                        );
                   }
                   /* CloakState */
                   else if(strstr(filebuf,"CloakState") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].cloak_state
                        ); 
                   }
                   /* CloakStrength */
                   else if(strstr(filebuf,"CloakStrength") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].cloak_strength
                        );
                   }
                   /* ShieldVisibility */
                   else if(strstr(filebuf,"ShieldVisibility") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].shield_visibility
                        );
                   }
                   /* Visibility */
                   else if(strstr(filebuf,"Visibility") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].visibility
                        );
                   }
                   /* DamageControl */
                   else if(strstr(filebuf,"DamageControl") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].damage_control
                        );
                   }

                   /* Scoreing stuff. */
		   /* ScoreGPL */
                   else if(strstr(filebuf,"ScoreGPL") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].score.gpl
                        );
                   }
                   /* ScoreRMUMax */
                   else if(strstr(filebuf,"ScoreRMUMax") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].score.rmu_max
                        );
                   }
                   /* ScoreRMU */
                   else if(strstr(filebuf,"ScoreRMU") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].score.rmu
                        );
                   }
                   /* ScoreDamageGivin */
                   else if(strstr(filebuf,"ScoreDamageGivin") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].score.damage_givin
                        );
                   }
                   /* ScoreDamageRecieved */
                   else if(strstr(filebuf,"ScoreDamageRecieved") != NULL)
                   {
                        sscanf(filebuf,"%s %s %lf",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].score.damage_recieved
                        );
                   }
                   /* ScoreKills */
                   else if(strstr(filebuf,"ScoreKills") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].score.kills
                        );
                   }

		   /* Permission Settings. */
                   /* PermissionUID */
                   else if(strstr(filebuf,"PermissionUID") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].permission.uid
                        );
                   }
                   /* PermissionGID */
                   else if(strstr(filebuf,"PermissionGID") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                           &xsw_object[xswobject_count].permission.gid
                        );
                   }

		   /* Animation stuff. */
                   /* AnimationLastInterval */
                   else if(strstr(filebuf,"AnimationLastInterval") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                        &xsw_object[xswobject_count].animation.last_interval
                        );
                   }
                   /* AnimationInterval */
                   else if(strstr(filebuf,"AnimationInterval") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                        &xsw_object[xswobject_count].animation.interval
                        );
                   }
                   /* AnimationCurrentFrame */
                   else if(strstr(filebuf,"AnimationCurrentFrame") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                        &xsw_object[xswobject_count].animation.current_frame
                        );
                   }
                   /* AnimationTotalFrames */
                   else if(strstr(filebuf,"AnimationTotalFrames") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                        &xsw_object[xswobject_count].animation.total_frames
                        );
                   }
                   /* AnimationCycleCount */
                   else if(strstr(filebuf,"AnimationCycleCount") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                        &xsw_object[xswobject_count].animation.cycle_count
                        );
                   }
                   /* AnimationCycleTimes */
                   else if(strstr(filebuf,"AnimationCycleTimes") != NULL)
                   {
                        sscanf(filebuf,"%s %s %i",
                           &stringa,
                           &stringa,
                        &xsw_object[xswobject_count].animation.cycle_times
                        );
                   }


		   /* ******************************************* */
		   /* ***               Weapons               *** */
		   else if(strstr(filebuf,"BeginWeapon") != NULL)
		   {

                      while((fgets(filebuf,256,fp)) != NULL)
                      {
                          lines_read++;
         
                          /* Skip comments. */
                          if(strstr(filebuf,"#") != NULL)
                          continue;
                           
                          /* Skip short lines. */
                          if(strlen(filebuf) < 4)
                          continue;

                          /* Remove newlines characters. */
                          while(1)
                          {
                             strptr = strchr(filebuf,'\n');
                             if(strptr == NULL)
                                break;
                             else 
                                *strptr='\0';
                          }

                          /* Name */
                          if(strstr(filebuf,"Name") != NULL)
                          {
                              strptr = strstr(filebuf,"=");
                              strptr += 2;
            strncpy(xsw_object[xswobject_count].weapons[weapons_count].name,
                                  strptr, MAX_NAME_LENGTH
			      );
                          }

                          /* EmissionType */
                          else if(strstr(filebuf,"EmissionType") != NULL)
                          {
                              sscanf(filebuf,"%s %s %i",
                                 &stringa,
                                 &stringa,
             &xsw_object[xswobject_count].weapons[weapons_count].emission_type
                              );
                          } 

                          /* Type */
                          else if(strstr(filebuf,"Type") != NULL)
                          {
                              sscanf(filebuf,"%s %s %i",
                                 &stringa,
                                 &stringa,
                     &xsw_object[xswobject_count].weapons[weapons_count].type
                              );
                          }
 
                          /* Amount */
                          else if(strstr(filebuf,"Amount") != NULL)
                          {
                              sscanf(filebuf,"%s %s %i",
                                 &stringa,
                                 &stringa,
                    &xsw_object[xswobject_count].weapons[weapons_count].amount
                              );
                          }

                          /* Max */
                          else if(strstr(filebuf,"Max") != NULL)
                          {
                              sscanf(filebuf,"%s %s %i",
                                 &stringa,
                                 &stringa,
                       &xsw_object[xswobject_count].weapons[weapons_count].max
                              );
                          }

                          /* CreatePower */
                          else if(strstr(filebuf,"CreatePower") != NULL)
                          {
                              sscanf(filebuf,"%s %s %lf",
                                 &stringa,
                                 &stringa,
             &xsw_object[xswobject_count].weapons[weapons_count].create_power
                              );
                          }

 
                          /* Power */
                          else if(strstr(filebuf,"Power") != NULL)
                          {
                              sscanf(filebuf,"%s %s %lf",
                                 &stringa,
                                 &stringa,
                    &xsw_object[xswobject_count].weapons[weapons_count].power
                              );
                          }

                          /* Range */
                          else if(strstr(filebuf,"Range") != NULL)
                          {
                              sscanf(filebuf,"%s %s %lf",
                                 &stringa,
                                 &stringa,
                    &xsw_object[xswobject_count].weapons[weapons_count].range
                              );
                          }

                          /* Delay */
                          else if(strstr(filebuf,"Delay") != NULL)
                          {
                              sscanf(filebuf,"%s %s %i",
                                 &stringa,
                                 &stringa,
                    &xsw_object[xswobject_count].weapons[weapons_count].delay
                              );
                          }

                          /* LastUsed */
                          else if(strstr(filebuf,"LastUsed") != NULL)
                          {
                              sscanf(filebuf,"%s %s %i",
                                 &stringa,
                                 &stringa,
                 &xsw_object[xswobject_count].weapons[weapons_count].last_used
                              );
			  }
			  /* EndWeapon */
			  else if(strstr(filebuf,"EndWeapon") != NULL)
			  {
				/* weapons_count must be less than MAX_WEAPONS. */
				if(weapons_count < (MAX_WEAPONS - 1))
					weapons_count++;
				break;
			  }
                          /* Unsupported property */
                          else
                          {
                             fprintf(stderr,
                             "%s: Line %i: Warning: Unsupported property `%s'\n",
                                 dbfilename, lines_read, filebuf
                             );
                          }
		       }
		   }
                   /* EndXSWObject */
                   else if(strstr(filebuf,"EndXSWObject") != NULL)
                   {
			weapons_count = 0;
                        break;
                   }
		   /* Unsupported property */
		   else
		   {
			fprintf(stderr,
			   "%s: Line %i: Warning: Unsupported property `%s'\n",
                           dbfilename, lines_read, filebuf
			);
		   }
		}
	   }
	}


	free(filebuf);
        fclose(fp);   
	return(xswobject_count);
}



long DBSaveToFile(char *dbfilename)
{
	/* Local variables. */
	char stringa[256];
	FILE *fp;
	char *filebuf;
	char *strptr;

	long xswobject_count = 0;	/* Relative to 1. */
	long xswobjects_written = -1;	/* Relative to 0. */
	long weapon_count = 0;
	long bytes_written = 0;


	/* Open dbfilename for writing, overwrites existing one. */
	fp = fopen(dbfilename,"w");
	if(fp == NULL)
	{
                fprintf(stderr, "Unable to create file: %s\n",
                        dbfilename
                );
                return(xswobjects_written);
	}


	/* ************************************************ */
	/* ***               Write Header               *** */
        sprintf(stringa, "# %s Version %s - Database File\n",
	   SWSERV_NAME, SWSERV_VERSION);
        bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

        sprintf(stringa, "#\n\n");
        bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);


        sprintf(stringa, "# Header\n");
        bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

        sprintf(stringa, "BeginHeader\n");
        bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
        
        sprintf(stringa, "   Title = %s\n", dbtitle);
        bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

        sprintf(stringa, "   Version = %s\n", SWSERV_VERSION);
        bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

        sprintf(stringa, "EndHeader\n\n");
        bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
	


	/* ************************************************ */
        /* ***   Begin Writing XSW Objects to DB File   *** */
	for(xswobject_count = 0; xswobject_count < MAX_OBJECTS; xswobject_count++)
	{
	   /* Skip object entries that are garbage or error. */
           if(xsw_object[xswobject_count].type <= XSW_OBJ_TYPE_GARBAGE)
                continue;


	   /* xswobject_count is now of a valid object that needs */
	   /* to be saved, begin saving xsw object number         */
	   /* xswobject_count.                                    */

	   /* Comment (number is xswobjects_written + 1) */
	   sprintf(stringa, "# Object %i\n", (xswobjects_written + 1) );
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

	   /* BeginXSWObject */
	   sprintf(stringa, "BeginXSWObject\n");
	   bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

	   /* Name */
	   sprintf(stringa, "   Name = %s\n", xsw_object[xswobject_count].name);
	   bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* Password */
           sprintf(stringa, "   Password = %s\n",
		xsw_object[xswobject_count].password);  
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

	   /* Empire */
           sprintf(stringa, "   Empire = %s\n",
		xsw_object[xswobject_count].empire);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* Type */
           sprintf(stringa, "   Type = %i\n",
                xsw_object[xswobject_count].type);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* ImageSet */
           sprintf(stringa, "   ImageSet = %i\n",
                xsw_object[xswobject_count].imageset);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* Owner */
           sprintf(stringa, "   Owner = %i\n",
                xsw_object[xswobject_count].owner);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* Size */
           sprintf(stringa, "   Size = %i\n",
                xsw_object[xswobject_count].size);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* LockedOn */
           sprintf(stringa, "   LockedOn = %i\n",
                xsw_object[xswobject_count].locked_on);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* InterceptingObject */
           sprintf(stringa, "   InterceptingObject = %i\n",
                xsw_object[xswobject_count].intercepting_object);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
 
	   /* WindowWidth */
           sprintf(stringa, "   WindowWidth = %i\n",
                xsw_object[xswobject_count].window_width);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* WindowHeight */
           sprintf(stringa, "   WindowHeight = %i\n",
                xsw_object[xswobject_count].window_height);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* ScannerRange */
           sprintf(stringa, "   ScannerRange = %lf\n",
                xsw_object[xswobject_count].scanner_range);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* CoordX */
           sprintf(stringa, "   CoordX = %lf\n",
                xsw_object[xswobject_count].x);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* CoordY */
           sprintf(stringa, "   CoordY = %lf\n",
                xsw_object[xswobject_count].y);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* CoordZ */
           sprintf(stringa, "   CoordZ = %lf\n",
                xsw_object[xswobject_count].z);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* ObjectHeading */
           sprintf(stringa, "   ObjectHeading = %lf\n",
                xsw_object[xswobject_count].heading);  
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
                   
           /* ObjectPitch */
           sprintf(stringa, "   ObjectPitch = %lf\n",
                xsw_object[xswobject_count].pitch);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* VelocityHeading */
           sprintf(stringa, "   VelocityHeading = %lf\n",
                xsw_object[xswobject_count].velocity_heading);  
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
                   
           /* VelocityMax */
           sprintf(stringa, "   VelocityMax = %lf\n",
                xsw_object[xswobject_count].velocity_max);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
 
           /* Velocity */
           sprintf(stringa, "   Velocity = %lf\n",
                xsw_object[xswobject_count].velocity);    
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* ThrustRevState */
           sprintf(stringa, "   ThrustRevState = %i\n",
                xsw_object[xswobject_count].thrust_rev_state);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* ThrustDir */
           sprintf(stringa, "   ThrustDir = %lf\n",
                xsw_object[xswobject_count].thrust_dir);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* ThrustPower */
           sprintf(stringa, "   ThrustPower = %lf\n",
                xsw_object[xswobject_count].thrust_power);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* Thrust */
           sprintf(stringa, "   Thrust = %lf\n",
                xsw_object[xswobject_count].thrust);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
  
           /* Throttle */
           sprintf(stringa, "   Throttle = %i\n",
                xsw_object[xswobject_count].throttle);      
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* TurnRate */
           sprintf(stringa, "   TurnRate = %lf\n",
                xsw_object[xswobject_count].turnrate);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* BrakesState */
           sprintf(stringa, "   BrakesState = %i\n",
                xsw_object[xswobject_count].brakes_state);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* BrakesPower */
           sprintf(stringa, "   BrakesPower = %lf\n",
                xsw_object[xswobject_count].brakes_power);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* HitPointsMax */
           sprintf(stringa, "   HitPointsMax = %lf\n",
                xsw_object[xswobject_count].hp_max);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* HitPoints */
           sprintf(stringa, "   HitPoints = %lf\n",
                xsw_object[xswobject_count].hp);  
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
         
           /* PowerPurity */
           sprintf(stringa, "   PowerPurity = %lf\n",
                xsw_object[xswobject_count].power_purity);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* PowerMax */
           sprintf(stringa, "   PowerMax = %lf\n",
                xsw_object[xswobject_count].power_max);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* Power */
           sprintf(stringa, "   Power = %lf\n",
                xsw_object[xswobject_count].power);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* CoreEfficency */
           sprintf(stringa, "   CoreEfficency = %lf\n",
                xsw_object[xswobject_count].core_efficency);    
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* AntimatterMax */
           sprintf(stringa, "   AntimatterMax = %lf\n",
                xsw_object[xswobject_count].antimatter_max);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* Antimatter */
           sprintf(stringa, "   Antimatter = %lf\n",
                xsw_object[xswobject_count].antimatter);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* ShieldState */
           sprintf(stringa, "   ShieldState = %i\n",
                xsw_object[xswobject_count].shield_state);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* ShieldFrequency */
           sprintf(stringa, "   ShieldFrequency = %lf\n",
                xsw_object[xswobject_count].shield_frequency);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* SelectedWeapon */
           sprintf(stringa, "   SelectedWeapon = %i\n",
                xsw_object[xswobject_count].selected_weapon);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* TotalWeapons */ 
           sprintf(stringa, "   TotalWeapons = %i\n",
                xsw_object[xswobject_count].total_weapons);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* BirthTime */
           sprintf(stringa, "   BirthTime = %i\n",
                xsw_object[xswobject_count].birth_time);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* Age */
           sprintf(stringa, "   Age = %i\n",
                xsw_object[xswobject_count].age);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* LifeSpan */
           sprintf(stringa, "   LifeSpan = %i\n",
                xsw_object[xswobject_count].lifespan);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* CloakState */
           sprintf(stringa, "   CloakState = %i\n",
                xsw_object[xswobject_count].cloak_state);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* CloakStrength */
           sprintf(stringa, "   CloakStrength = %lf\n",
                xsw_object[xswobject_count].cloak_strength);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* ShieldVisibility */
           sprintf(stringa, "   ShieldVisibility = %lf\n",
                xsw_object[xswobject_count].shield_visibility);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* Visibility */
           sprintf(stringa, "   Visibility = %lf\n",
                xsw_object[xswobject_count].visibility);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* DamageControl */
           sprintf(stringa, "   DamageControl = %i\n",
                xsw_object[xswobject_count].damage_control);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

	   /* Scoring stuff. */
	   /* ScoreGPL */
           sprintf(stringa, "   ScoreGPL = %lf\n",
                xsw_object[xswobject_count].score.gpl);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* ScoreRMUMax */
           sprintf(stringa, "   ScoreRMUMax = %lf\n",
                xsw_object[xswobject_count].score.rmu_max);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* ScoreRMU */
           sprintf(stringa, "   ScoreRMU = %lf\n",
                xsw_object[xswobject_count].score.rmu); 
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
  
           /* ScoreDamageGivin */
           sprintf(stringa, "   ScoreDamageGivin = %lf\n",
                xsw_object[xswobject_count].score.damage_givin);    
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* ScoreDamageRecieved */
           sprintf(stringa, "   ScoreDamageRecieved = %lf\n",
                xsw_object[xswobject_count].score.damage_recieved); 
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* ScoreKills */
           sprintf(stringa, "   ScoreKills = %i\n",
                xsw_object[xswobject_count].score.kills); 
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

	   /* Permission stuff. */
           /* PermissionUID */
           sprintf(stringa, "   PermissionUID = %i\n",
                xsw_object[xswobject_count].permission.uid);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* PermissionGID */
           sprintf(stringa, "   PermissionGID = %i\n",
                xsw_object[xswobject_count].permission.gid); 
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* Animation stuff. */
	   /* AnimationLastInterval */
           sprintf(stringa, "   AnimationLastInterval = %i\n",
                xsw_object[xswobject_count].animation.last_interval);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
                
           /* AnimationInterval */
           sprintf(stringa, "   AnimationInterval = %i\n",
                xsw_object[xswobject_count].animation.interval);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
 
           /* AnimationCurrentFrame */
           sprintf(stringa, "   AnimationCurrentFrame = %i\n",
                xsw_object[xswobject_count].animation.current_frame);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* AnimationTotalFrames */
           sprintf(stringa, "   AnimationTotalFrames = %i\n",
                xsw_object[xswobject_count].animation.total_frames);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* AnimationCycleCount */
           sprintf(stringa, "   AnimationCycleCount = %i\n",
                xsw_object[xswobject_count].animation.cycle_count);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

           /* AnimationCycleTimes */
           sprintf(stringa, "   AnimationCycleTimes = %i\n",
                xsw_object[xswobject_count].animation.cycle_times);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

	   /* Weapons... */
	   for(weapon_count = 0; weapon_count < MAX_WEAPONS; weapon_count++)
	   {
		/* BeginWeapon */
		sprintf(stringa, "   BeginWeapon\n");
		bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

		/* Name */
                sprintf(stringa, "      Name = %s\n",
                xsw_object[xswobject_count].weapons[weapon_count].name);
                bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
 
                /* EmissionType */
                sprintf(stringa, "      EmissionType = %i\n",
                xsw_object[xswobject_count].weapons[weapon_count].emission_type);
                bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

                /* Type */
                sprintf(stringa, "      Type = %i\n",
                xsw_object[xswobject_count].weapons[weapon_count].type);
                bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

                /* Amount */
                sprintf(stringa, "      Amount = %i\n",
                xsw_object[xswobject_count].weapons[weapon_count].amount);
                bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
            
                /* Max */
                sprintf(stringa, "      Max = %i\n",
                xsw_object[xswobject_count].weapons[weapon_count].max);
                bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

		/* CreatePower */
                sprintf(stringa, "      CreatePower = %lf\n",
                xsw_object[xswobject_count].weapons[weapon_count].create_power);
                bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

                /* Power */
                sprintf(stringa, "      Power = %lf\n",
                xsw_object[xswobject_count].weapons[weapon_count].power);
                bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

                /* Range */
                sprintf(stringa, "      Range = %lf\n",
                xsw_object[xswobject_count].weapons[weapon_count].range);
                bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
           
                /* Delay */
                sprintf(stringa, "      Delay = %i\n",
                xsw_object[xswobject_count].weapons[weapon_count].delay);
                bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

                /* LastUsed */
                sprintf(stringa, "      LastUsed = %i\n",
                xsw_object[xswobject_count].weapons[weapon_count].last_used);
                bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

                /* EndWeapon */
                sprintf(stringa, "   EndWeapon\n");
                bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);
	   }

           /* EndXSWObject */
           sprintf(stringa, "EndXSWObject\n\n",
                xsw_object[xswobject_count].shield_frequency);
           bytes_written += fwrite(stringa,sizeof(char),strlen(stringa),fp);

	   xswobjects_written++;
	}


        fclose(fp);   
	return(xswobjects_written);
}



long DBCleanMemory()
{
	/* Local variables. */
	long topmostobj;


	/* Do cleaning... not coded right now, but insert code later
	 * when needed..
	 */


	/* Get topmost object. */
	topmostobj = DBGetTopObjectNumber();

	return(topmostobj);
}



long DBCreateObject(int width, int height)
{
	/* Local variables. */
	long created_object_num;
	char stringa[1024];


	/* Find garbage to create space from. */
	for(created_object_num = 0; created_object_num < MAX_OBJECTS;
	   created_object_num++)
	{
	   /* Is this xsw_object[] free? */
	   if(xsw_object[created_object_num].type != XSW_OBJ_TYPE_GARBAGE)
		continue;

	   break;
	}

        /* Did we find any available space for new objects? */
        if(created_object_num >= MAX_OBJECTS)
                return(-1);


	/* Increment global variable total_objects if created_object_num */
	/* tops it.                                                      */
	if(created_object_num >= total_objects)
	{
		total_objects = created_object_num + 1;
	}
        if(total_objects > MAX_OBJECTS)
                total_objects = MAX_OBJECTS;


	/* Got free object space, now begin creating new object. */
	sprintf(stringa, "Object %i", created_object_num);
	strncpy(xsw_object[created_object_num].name, stringa, MAX_NAME_LENGTH);
	strncpy(xsw_object[created_object_num].empire, "NONE", MAX_EMPIRE_LENGTH);
	strncpy(xsw_object[created_object_num].password, "", MAX_PASSWORD_LENGTH);
	xsw_object[created_object_num].last_updated = cur_millitime;
        xsw_object[created_object_num].type = XSW_OBJ_TYPE_STATIC;
        xsw_object[created_object_num].imageset = IMAGESET_CODE_GENERIC;
        xsw_object[created_object_num].owner = created_object_num;
        xsw_object[created_object_num].size = 1;
	xsw_object[created_object_num].locked_on = -1;
        xsw_object[created_object_num].intercepting_object = -1;

        xsw_object[created_object_num].window_width = width;
        xsw_object[created_object_num].window_height = height;
        xsw_object[created_object_num].scanner_range = 8.0;
        xsw_object[created_object_num].x = 0;
        xsw_object[created_object_num].y = 0;
        xsw_object[created_object_num].z = 0;
        xsw_object[created_object_num].heading = 0;
        xsw_object[created_object_num].pitch = 0;
        xsw_object[created_object_num].velocity = 0;
        xsw_object[created_object_num].velocity_max = 0;
        xsw_object[created_object_num].velocity_heading = 0;
        xsw_object[created_object_num].thrust_rev_state = 0;
        xsw_object[created_object_num].thrust_dir = 3.1415927;
        xsw_object[created_object_num].thrust = 0;
        xsw_object[created_object_num].thrust_power = 0;
        xsw_object[created_object_num].throttle = 0;
        xsw_object[created_object_num].turnrate = 0.003;
        xsw_object[created_object_num].brakes_state = 0;
	xsw_object[created_object_num].brakes_power = 0;
        xsw_object[created_object_num].hp = 1;
        xsw_object[created_object_num].hp_max = 1;
        xsw_object[created_object_num].power = 0;
        xsw_object[created_object_num].power_max = 0;
        xsw_object[created_object_num].power_purity = 0;
        xsw_object[created_object_num].core_efficency = 0;
        xsw_object[created_object_num].antimatter = 0;
        xsw_object[created_object_num].antimatter_max = 0;
        xsw_object[created_object_num].shield_state = 0;
        xsw_object[created_object_num].shield_frequency = 180.25;
        xsw_object[created_object_num].selected_weapon = 0;
        xsw_object[created_object_num].total_weapons = 0;
        xsw_object[created_object_num].birth_time = cur_millitime;
        xsw_object[created_object_num].age = 0;
        xsw_object[created_object_num].lifespan = -1;
        xsw_object[created_object_num].cloak_state = -1;
        xsw_object[created_object_num].cloak_strength = 1.0;
        xsw_object[created_object_num].shield_visibility = 0.0;
        xsw_object[created_object_num].visibility = 1.0;
        xsw_object[created_object_num].damage_control = 0;

        xsw_object[created_object_num].score.gpl = 0;
        xsw_object[created_object_num].score.rmu = 0;
        xsw_object[created_object_num].score.rmu_max = 1000;
        xsw_object[created_object_num].score.damage_givin = 0;
        xsw_object[created_object_num].score.damage_recieved = 0;
        xsw_object[created_object_num].score.kills = 0;

        xsw_object[created_object_num].permission.uid = DEFAULT_UID;
        xsw_object[created_object_num].permission.gid = DEFAULT_GID;

        xsw_object[created_object_num].animation.interval = 500;
        xsw_object[created_object_num].animation.last_interval = cur_millitime;
        xsw_object[created_object_num].animation.current_frame = 0;
        xsw_object[created_object_num].animation.total_frames = 1;
        xsw_object[created_object_num].animation.cycle_count = 0; 
        xsw_object[created_object_num].animation.cycle_times = 1;


	return(created_object_num);
}



/* *****************************************************************************
 *
 *   long DBCreateExplosion1(double loc_x, double loc_y)
 *
 *      Creates an explosion 1 (large explosion) object at location
 *      loc_x and loc_y.
 * 
 *      Returns the number of the object created, or -1 if an object
 *      cannot be created.
 */

long DBCreateExplosion1(double loc_x, double loc_y)
{
	long created_object_num;

	created_object_num = DBCreateObject(200, 150);
	if(created_object_num < 0)
		return(-1);


	strncpy(xsw_object[created_object_num].name, "", MAX_NAME_LENGTH);
        strncpy(xsw_object[created_object_num].empire,"", MAX_EMPIRE_LENGTH);
        xsw_object[created_object_num].type = XSW_OBJ_TYPE_ANIMATED;
        xsw_object[created_object_num].imageset = IMAGESET_CODE_EXPLOSION1;

	xsw_object[created_object_num].size = 75;
	xsw_object[created_object_num].x = loc_x;
	xsw_object[created_object_num].y = loc_y;
	xsw_object[created_object_num].animation.interval = 200;
        xsw_object[created_object_num].animation.current_frame = 0;
        xsw_object[created_object_num].animation.total_frames = 4;
        xsw_object[created_object_num].animation.cycle_count = 0;
        xsw_object[created_object_num].animation.cycle_times = 1;

	return(created_object_num);
}



/* *****************************************************************************
 *
 *   long DBCreateExplosion2(double loc_x, double loc_y)
 *
 *	Creates an explosion 2 (small explosion) object at location
 *	loc_x and loc_y.
 *
 *	Returns the number of the object created, or -1 if an object
 *	cannot be created.
 */

long DBCreateExplosion2(double loc_x, double loc_y)
{
        long created_object_num;

        created_object_num = DBCreateObject(200, 150);
        if(created_object_num < 0)
                return(-1);

  
        strncpy(xsw_object[created_object_num].name, "", MAX_NAME_LENGTH);
        strncpy(xsw_object[created_object_num].empire, "", MAX_EMPIRE_LENGTH);
        xsw_object[created_object_num].type = XSW_OBJ_TYPE_ANIMATED;
        xsw_object[created_object_num].imageset = IMAGESET_CODE_EXPLOSION2;

        xsw_object[created_object_num].size = 6;
        xsw_object[created_object_num].x = loc_x;
        xsw_object[created_object_num].y = loc_y;
        xsw_object[created_object_num].animation.interval = 200;
        xsw_object[created_object_num].animation.current_frame = 0;
        xsw_object[created_object_num].animation.total_frames = 1;
        xsw_object[created_object_num].animation.cycle_count = 0;
        xsw_object[created_object_num].animation.cycle_times = 1;
        
        return(created_object_num);
}       



void DBRecycleObject(long object_num)
{
	/* Is it already recycled? */
	if(xsw_object[object_num].type <= XSW_OBJ_TYPE_GARBAGE)
		return;

	/* Begin recycling... */
	strncpy(xsw_object[object_num].name, "", MAX_NAME_LENGTH);
	strncpy(xsw_object[object_num].empire, "", MAX_EMPIRE_LENGTH);
	xsw_object[object_num].owner = -1;

	xsw_object[object_num].type = XSW_OBJ_TYPE_GARBAGE;
	xsw_object[object_num].imageset = IMAGESET_CODE_GENERIC;


	return;
}



void DBEmergancySave()
{
	/* Local variables. */
	char stringa[1024];
	char stringb[1024];
	long save_status;


	/* Print and log incident. */
	fprintf(stderr,
	    "*** Server: Emergancy database save in progress...\n");
        LogAppendLineFormatted(primary_log_fullname,
	    "*** Server: Emergancy database save in progress...");


	/* Get new filename based on PID. */
	sprintf(stringa, "%s.EMERGANCY.%i", db_out_fullname, rootpid);
	save_status = DBSaveToFile(stringa);


	/* Check status. */
	if(save_status > -1)
	{
            fprintf(stderr, "%s: Database sucessfully saved!\n",
                stringa);
	    sprintf(stringb, "%s: Database sucessfully saved!",
		stringa);
 	    LogAppendLineFormatted(primary_log_fullname, stringb);
	}
	else
	{
            fprintf(stderr, "%s: Error occured during save!\n",
                stringa);
            sprintf(stringb, "%s: Error occured during save!",
                stringa);
            LogAppendLineFormatted(primary_log_fullname, stringb);
	}


	return;
}



long DBGetTopObjectNumber()
{
	/* Local variables. */
	long last_valid_object = -1;
	long current_object = 0;


	/* Get last valid object. */
	for(current_object = 0; current_object < MAX_OBJECTS;
	   current_object++)
	{
	   if(xsw_object[current_object].type <= XSW_OBJ_TYPE_GARBAGE)
		continue;

	   last_valid_object = current_object;
	}


	return(last_valid_object);
}



double DBGetObjectVisibility(long object_num)
{
        /* Local variables. */
	double visibility_rtn;


	/* Garbages and errors have no visibility. */
	if(xsw_object[object_num].type <= XSW_OBJ_TYPE_GARBAGE)
		return(0);


	/* Get objects visibility. */
	visibility_rtn = xsw_object[object_num].visibility;

	/* See if object is cloaked. */
	if(xsw_object[object_num].cloak_state >= 1)
		visibility_rtn -= xsw_object[object_num].cloak_strength;

	/* Visibility should not be negative. */
	if(visibility_rtn < 0)
		visibility_rtn = 0;

	return(visibility_rtn);
}
