/* ********************************************************************
 *
 *	devices.c	Device Interfacing
 *	----------------------------------
 *
 *	int JSInitialize(char *jsdevicenode, char *jscalfile)
 *	double JSGetThrottle(int jsfd, int throttle_mode)
 *	double JSGetTurn(int jsfd)
 *	int JSGetButtonState(int jsfd, int button_number)
 *
 *	---
 *
 *	These functions fetch data from the devices, like the
 *	joystick.   Some of these functions return data already
 *	formatted.
 *
 *	JSInitialize() initializes the joystick device jsdevicenode
 *	and fetches the joystick caliberation values from jscalfile.
 *	Returns the file descriptor number for the newly opened joystick
 *	device or -1 on error.
 *
 *	JSGetThrottle() returns the throttle (y-axis) value, the range of
 *	the return depends on throttle_mode.   When throttle_mode is set to
 *	0, then return is from 0 to 100.  When throttle_mode is set to 1
 *	then return is from -100 to 100.   All returns are of type double.
 *	Must call JSInitialize() first to initialize jsfd.
 *
 *	JSGetTurn() returns the turn (x-axis) value from -100 to 100 as
 *	a double.
 *	Negative values means applying left vector force,
 *	positive values means applying right vector force.
 *	Must call JSInitialize() first, to initialize jsfd.
 *
 *	JSGetButtonState() returns the status of button button_number
 *	on joystick file descriptor jsfd.   Returns 0 for unarmed, 1 for
 *	armed and -1 for error.
 *	Must call JSInitialize() first, to initialize jsfd.
 *
 */

#include "../include/xsw.h"  



int JSInitialize(char *jsdevicenode, char *jscalfile)
{
        /* Local variables. */
        int x,y,z;
	char stringa[1024];
	char *strptr;
	FILE *fp;
	char *filebuf;
        off_t filesize;
        struct stat statbuf;
        int jsfd;
	int lines_read = 0;


	/* Is jscalfile readable? */
        if(access(jscalfile, R_OK) == -1)
        {
		fprintf(stderr,
                   "Error: Cannot read joystick calibration file: `%s'\n",
			jscalfile
		);
                return(-1);
        }

	/* Open the jscalfile file. */
	fp = fopen(jscalfile, "r");
        if(fp == NULL)
        {
                fprintf(stderr,
                   "Error: Cannot read joystick calibration file: `%s'\n",
                        jscalfile
                );
                return(-1);   
        }

        /* Get statistics and stuff about the file. */
        fstat(fileno(fp), &statbuf);    /* Get stats on file. */
        filesize = statbuf.st_size;     /* Get size of file. */
        
        /* Allocate memory. */
        filebuf = malloc(filesize * sizeof(char) + 1);


	/* Begin reading each line. */
        while(fgets(filebuf,256,fp) != NULL)
        {
           lines_read++;
	
           /* Skip comments. */
           if(strstr(filebuf,"#") != NULL)
               continue;
 
           /* Skip short lines. */
           if(strlen(filebuf) < 4)
               continue;

           /* Remove newlines characters. */
           while(1)
           {
              strptr = strchr(filebuf,'\n');
              if(strptr == NULL)
                 break; 
              else  
                 *strptr='\0';
           }

	   /* ********************************************* */
	   /*               BeginJoystick1                  */
	   if(strstr(filebuf, "BeginJoystick1") != NULL)
	   {
              while(fgets(filebuf,256,fp) != NULL)
              {
                   lines_read++;
                                
                   /* Skip comments. */
                   if(strstr(filebuf,"#") != NULL)
                         continue;

                   /* Skip short lines. */
                   if(strlen(filebuf) < 4)
                         continue;        

                   /* Remove newlines characters. */
                   while(1)
                   {       
                       strptr = strchr(filebuf,'\n');
                       if(strptr == NULL)
                            break;
                       else
                            *strptr='\0';
                   }

		   /* X_Min */
	       	   if(strstr(filebuf, "X_Min") != NULL)
	           {
	                 sscanf(filebuf,"%s %s %i",
				&stringa,
				&stringa,
				&joystick_data.x_min
			 );
                	 continue;
           	   }
	           /* Y_Min */
        	   else if(strstr(filebuf, "Y_Min") != NULL)
	           {
		         sscanf(filebuf,"%s %s %i",
	                        &stringa,
                                &stringa,
                                &joystick_data.y_min
                         );
                         continue;
                   }
                   /* X_Max */
                   else if(strstr(filebuf, "X_Max") != NULL)
                   {
                         sscanf(filebuf,"%s %s %i",
                                &stringa,
                                &stringa,
                                &joystick_data.x_max
                         );
                         continue;   
                   }
                   /* Y_Max */
                   else if(strstr(filebuf, "Y_Max") != NULL)
                   {
                         sscanf(filebuf,"%s %s %i",
                                &stringa,
                                &stringa,
                                &joystick_data.y_max
                         );
                         continue;
                   }
                   /* NullZone */
                   else if(strstr(filebuf, "NullZone") != NULL)
                   {
                         sscanf(filebuf,"%s %s %i",
                                &stringa,
                                &stringa,
                                &joystick_data.nullzone
                         );
                         continue;
                   }
                   /* EndJoystick */
                   else if(strstr(filebuf, "EndJoystick") != NULL)
                   {
                         break;
                   }
                }
             }
	}

        /* Free filebuf and fclose fp. */
        free(filebuf);
        fclose(fp);


	/* Open the joystick device node jsdevicenode. */
	jsfd = open(jsdevicenode, O_RDONLY);
	if(jsfd < 0)
	{
		fprintf(stderr,
			"Error: Cannot open joystick device: `%s'\n",
			jsdevicenode
		);
		return(-1);
 	}

	/* Return the joystick file descriptor. */
	return(jsfd);
}



double JSGetThrottle(int jsfd, int throttle_mode)
{
	/* Local variables. */
        struct JS_DATA_TYPE js;
	int status = 0;
	double throttle_rtn;


        /* Check if jsfd is valid. */
        if(jsfd < 0)
                return(-1);
        

        /* Read the joystick device to get current y state. */
        status = read(jsfd, &js, JS_RETURN);
        if(status < 0)
                return(-1);


	/* Get the current raw throttle_rtn value. */
	throttle_rtn = js.y;


	switch(throttle_mode)
	{
          /* Bidirectional throttle scope, -100 to 100. */
          case 1:
            /* Invert throttle_rtn value. */
            throttle_rtn = ( joystick_data.y_max - joystick_data.y_min ) -
                throttle_rtn;

            /* Get throttle_rtn into range of 0 to 200. */
            throttle_rtn *= 200;
            throttle_rtn = throttle_rtn /
                ( joystick_data.y_max - joystick_data.y_min);

	    /* Shift to range of -100 to 100. */
	    throttle_rtn = throttle_rtn - 100;

            /* Sanitize throttle_rtn. */
            if(throttle_rtn < -100) 
                throttle_rtn = -100;
            else if(throttle_rtn > 100)
                throttle_rtn = 100;

            break;


	  /* Normal throttle scope, 0 to 100. */
	  case 0:
	    /* Invert throttle_rtn value. */
	    throttle_rtn = ( joystick_data.y_max - joystick_data.y_min ) -
		throttle_rtn;

	    /* Get throttle_rtn into range of 0 to 100. */
	    throttle_rtn *= 100;
	    throttle_rtn = throttle_rtn /
		( joystick_data.y_max - joystick_data.y_min);

            /* Sanitize throttle_rtn. */
            if(throttle_rtn < 0)
                throttle_rtn = 0;
            else if(throttle_rtn > 100)
                throttle_rtn = 100;

	    break;


          /* Default to normal throttle scope, 0 to 100. */
          default:
            /* Invert throttle_rtn value. */
            throttle_rtn = ( joystick_data.y_max - joystick_data.y_min ) -
                throttle_rtn;

            /* Get throttle_rtn into range of 0 to 100. */
            throttle_rtn *= 100;
            throttle_rtn = throttle_rtn /
                ( joystick_data.y_max - joystick_data.y_min);

            /* Sanitize throttle_rtn. */
            if(throttle_rtn < 0)
                throttle_rtn = 0;
            else if(throttle_rtn > 100)
                throttle_rtn = 100;
         
            break;
	}



	return(throttle_rtn);
}



double JSGetTurn(int jsfd)
{
        /* Local variables. */
        struct JS_DATA_TYPE js;
        int status = 0;
	double turn_rtn = 0;


        /* Check if jsfd is valid. */
        if(jsfd < 0)
                return(-1);       


        /* Read the joystick device to get current y state. */
        status = read(jsfd, &js, JS_RETURN);
        if(status < 0)
                return(-1);


        /* Get the current x value */
        turn_rtn = js.x;


 	/* Get turn_rtn in the range of -100 to 100. */
	turn_rtn *= 200;
	turn_rtn = turn_rtn / ( joystick_data.x_max - joystick_data.x_min );
	turn_rtn = turn_rtn - 100;

	/* Sanitize turn_rtn. */
	if(turn_rtn < -100)
		turn_rtn = -100;
	else if(turn_rtn > 100)
		turn_rtn = 100;


        return(turn_rtn);
}



int JSGetButtonState(int jsfd, int button_number)
{
	/* Local variables. */
	struct JS_DATA_TYPE js;
	int status = 0;
	int button_state = 0;


	/* Check if jsfd is valid. */
	if(jsfd < 0)
		return(-1);


	/* Read the joystick device to get button state. */
	status = read(jsfd, &js, JS_RETURN);
	if(status < 0)
		return(-1);


	switch(button_number)
	{
	   case 1:
		if(js.buttons & 1)
			button_state = 1;
		break;

	   case 2:
		if(js.buttons & 2)
			button_state = 1;
		break;

           case 3:
                break;

           case 4:
                break;

           default:
                break;
	}


	return(button_state);
}
