/* ***************************************************************************
 *
 *	jscal.h		Header File for XJoystick Caliberator
 *	-----------------------------------------------------
 *
 *
 */


#ifndef __XJSCAL_H__
#define __XJSCAL_H__

#include <stdio.h>

#include <unistd.h>
#include <stdlib.h>
#include <malloc.h>
#include <errno.h>
#include <ctype.h>
#include <stdarg.h>
#include <string.h>
#include <time.h>
#include <fcntl.h>

#include <sys/time.h>
#include <sys/stat.h>

#include <linux/joystick.h>

/* Xlib */
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>     /* X cursor handling. */
#include <X11/Intrinsic.h>      /* Xt. */

/* Xpm */
#include <X11/xpm.h>            /* xpm handling. */

/* Xext (X Extensions) */
#include <X11/extensions/shape.h>       /* For Shape mask. */


/* ***************************************************************************
 *
 *                        USER EDITABLE DEFINATIONS
 *
 *   You may edit these definations, read the comments carefully, so you know
 *   which settings you must edit and which to leave alone.
 */

/*
 *   Color reduction for XPM (pixmap) images:
 *
 *      0     = 0% Reduction allowed (not recommended).
 *      40000 = (Default).
 *      60000 = 100% Reduction allowed.
 */
#define COLORCLOSENESS 40000



/*
 *   Default Joystick Caliberation File:
 *
 *	This is the caliberation file to read and write joystick
 *	caliberation information to.   An alternative name for
 *	this file can be specified at the command line.
 */
#define JS_CALIBRATION_FILE ".joystick"



/*
 *   Joystick Device Nodes:
 *
 *	These are the device node names for your joysticks. Currently
 *	the Linux joystick driver only has support for joystick 1
 *	(/dev/js0) and joystick 2 (/dev/js1).
 *
 */
#define DEV_JOYSTICK_0 "/dev/js0"
#define DEV_JOYSTICK_1 "/dev/js1"
#define DEV_JOYSTICK_2 "/dev/js2"
#define DEV_JOYSTICK_3 "/dev/js3"



/*
 *   Maximum Number of Joysticks:
 *
 *	This is the maximum number of joysticks that this program
 *	will support.
 */
#define MAX_JOYSTICKS 4



/*
 *   Maximum X and Y Joystick Values:
 *
 *	Some joysticks have a very large range of X and Y axis values.
 *	Normally, joysticks have X and Y axis values from 0 to 1400.
 *	But some joysticks may produce values outside of this range.
 *	Some kind of limit must be set by this program in order to limit
 *	and truncate huge numbers.
 */
#define MIN_JS_X_AXIS_VALUE -16385
#define MAX_JS_X_AXIS_VALUE 16384
#define MIN_JS_Y_AXIS_VALUE -16385
#define MAX_JS_Y_AXIS_VALUE 16384


/*
 *   XJoystick Caliberator Loop Interval:
 *
 *	This controls the interval for the main loop of this program.
 *	Units are in microseconds.
 */
#define XJSCAL_LOOP_INTERVAL 10000



/*
 * *********************************************************************
 * *********************************************************************
 * ***      THERE IS NO NEED FOR USER EDITING BELOW THIS LINE        ***
 * *********************************************************************
 * *********************************************************************
 */


/*
 *   Master name and version strings:
 */
#define XJSCAL_NAME "XJoystick Caliberator"
#define XJSCAL_VERSION "0.04"


/*
 *   Copyright information:
 */
#define XJSCAL_COPYRIGHT \
"Copyright (C) 1997, 1998 Taura (learfox@hotmail.com)\n\
Permission to use, copy, and modify this program without\n\
restriction is hereby granted, as long as this copyright\n\
notice appears in each copy of the program source code.\n\
This program is freely distributable without licensing fees and\n\
is provided without guarantee or warrantee expressed or implied.\n\
\n"


/*
 *   Color Specifications:
 */
#define NORMAL_TEXT_COLOR_SPEC      "rgbi:1.00/1.00/1.00"
#define BOLD_TEXT_COLOR_SPEC        "rgbi:1.00/1.00/1.00"
#define READOUT_TEXT_COLOR_SPEC     "rgbi:1.00/0.18/0.10"


/*
 *   Window Attributes Mask:
 */
#define WIN_ATTR_MASK CWBackPixel|CWBorderPixel|   \
                      CWBackPixmap|CWBorderPixmap| \
                      CWBitGravity|CWWinGravity|   \
                      CWColormap|CWBackingStore|CWCursor

/*
 *   Window XEvent Mask:
 *
 *      Mask of events to watch for.
 */
#define TOPLEVEL_EVENTMASK ExposureMask| \
                           KeyPressMask|KeyReleaseMask| \
                           ButtonPressMask|ButtonReleaseMask

#define BUTTON_EVENTMASK ExposureMask| \
                         ButtonPressMask|ButtonReleaseMask| \
                         KeyPressMask|KeyReleaseMask



/* **********************************************************************
 *
 *                         GLOBAL VARIABLES
 *
 */

/*
 *   Process handling:
 *
 *      Note the rootpid is the PID of the main xpppcon process,
 *      the process that started it all.   childpid can change
 *      during execution, it will contain and child processes'
 *      PID.
 */
pid_t rootpid;
pid_t childpid;
int child_procs;


/*
 *   Current Millitime:
 */
long cur_millitime;


/*
 *   Joystick Caliberation Recording Structure:
 *
 *	When in caliberation mode, the joystick's X and Y axis minimum and
 *	maximum values are stored in these structures.
 */
typedef struct
{
	int exists;		/* 0 for exist, -1 for non existant. */
	int x_min, y_min;
	int x_max, y_max;
	unsigned int nullzone;
} js_caliberation_struct;
js_caliberation_struct js_caliberation_record[MAX_JOYSTICKS];


/*
 *    Currently Selected Joystick Device Node:
 */
char *cur_js_device_node;


/*
 *   Currently Selected Joystick File Descriptor:
 */
int cur_js_fd;
struct JS_DATA_TYPE js;


/*
 *   XLib Variables:
 *
 *	These variables are for XLib interface.
 */
Display                 *display;       /* Default display. */
int                     scr_num;        /* Default screen number. */
Screen                  *scr_ptr;       /* Default screen pointer. */
Visual                  *visual;        /* Default visual. */
int                     depth;          /* Depth. */
XSetWindowAttributes    attributes;             /* Set window attributes. */
XWindowAttributes       winattributes;          /* Return Window attributes. */
unsigned long           black_pix, white_pix;   /* Default pixels. */
GC                      the_GC;         /* Graphics context. */
XGCValues               GCvalues;       /* Graphics context values. */
Colormap                colormap;       /* Default colormap. */
XColor                  xcolor;         /* Default xcolor. */
XSizeHints              sizehints;      /* X size hints. */
XWMHints                wmhints;        /* Window manager size hints. */

/* XEvents. */
XEvent buttonpress_xevent,
       buttonrelease_xevent,
       keypress_xevent,
       keyrelease_xevent,
       expose_xevent,
       destroy_xevent;

/* XFontinfos. */
XFontStruct *normal_xfont,
            *bold_xfont;

/* XFontnames. */
char *normal_fontname;
char *bold_fontname;


/* XColors. */
XColor normal_text_xcolor,
       bold_text_xcolor,
       readout_text_xcolor;


/* Pixels numbers. */
unsigned long normal_text_pixel,
              bold_text_pixel,
              readout_text_pixel;


/*
 *   Windows:
 *
 *	These are the primary XLib Windows used in this program.
 */
Window crosshair_win,
       toplevel,
       js_coords_win,
       js_readout_win,
       js_mode_switch_win,
       js_num_switch_win,
       js_save_button_win,
       js_exit_button_win;


/*
 *   Pixmaps:
 *
 *	Pixmaps used by this program.
 */
Pixmap crosshair_pixmap,
       crosshairm_bitmap,
       toplevel_bkg_pixmap,
       exit_button0_pixmap,
       exit_button1_pixmap,
       jscoords_pixmap,
       mode_switch0_pixmap,
       mode_switch1_pixmap,
       num_switch0_pixmap,
       num_switch1_pixmap,
       num_switch2_pixmap,
       num_switch3_pixmap,
       readout_pixmap,
       save_button0_pixmap,
       save_button1_pixmap;

/*
 *   Redraw Counters:
 */
int button_draw_count;


/*
 *   Switch States:
 *
 *	js_number_switch_state:
 *		0 = Reading /dev/js0
 *		1 = Reading /dev/js1
 *              2 = Reading /dev/js2
 *              3 = Reading /dev/js3
 */
int js_number_switch_state;
int js_mode_switch_state;


/*
 *   Button States:
 */
Bool js_save_button_state,
     js_exit_button_state;


/*
 *   Full Pathnames to various files:
 */
char js_calibration_file_fullname[1024];



/* *******************************************************************
 *
 *                        GLOBAL FUNCTIONS
 */

/* In xutils.c */
extern int XuGetFontInfo(XFontStruct **xfont_info, char *xfontname);
extern unsigned long XuGetColor(char *spec, XColor xcolor_in);

/* In xpm.c */
extern Pixmap readpixmapfile(char *xpmfilename);
extern Pixmap readbitmapfile(char *xbmfilename);
extern Pixmap readpixmapdata(char **xpmdata);

/* In joystick.c */
extern int JSOpenDevice(char *js_device_to_open);
extern int JSReadCaliberationFile(char *calibfilename);
extern int JSWriteCaliberationFile(char *calibfilename);

/* In timming.c */
extern long MilliTime(void);



#endif
