/* **********************************************************************
 *
 *	update.c	Update Objects
 *	------------------------------
 *	Functions:
 *
 *	int UpdateObject(long object_num)
 *
 *	---
 *
 *	UpdateObject() calls all other functions here to
 *	update given object number object_num.   If object_num
 *	has a xsw_object[] type of XSW_OBJ_TYPE_ERROR or
 *	XSW_OBJ_TYPE_GARBAGE, nothing will be done.
 *	Returns 0 if the object was updated, 1 if nothing was done,
 *	and -1 for error.
 *
 */

#include "../include/swserv.h"



int UpdateHeading(long object_num)
{
	/* Local variables. */
        double x, y;
	double rel_heading;
	double target_heading;
	int nullzone_min, nullzone_max;
	long temp_object;


	/* Skip static objects. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_STATIC)
           return(0);

	/* Skip player objects, their heading is updated on client. */
	/* Except when intercepting.                                */
	if( (xsw_object[object_num].type == XSW_OBJ_TYPE_PLAYER) &&
	    (xsw_object[object_num].intercepting_object < 0)
	)
           return(0);


	/* Is object intercepting something? */
	if(xsw_object[object_num].intercepting_object >= 0)
	{
	    /* Get rel_heading to the object we're intercepting. */
	    temp_object = xsw_object[object_num].intercepting_object;
	    x = xsw_object[temp_object].x - xsw_object[object_num].x;
	    y = xsw_object[temp_object].y - xsw_object[object_num].y;
	    target_heading = MuLocationtoHeading(x, y);

	    rel_heading = target_heading - xsw_object[object_num].heading;

	    if( (rel_heading > 3.1415927) || (rel_heading < -3.1415927) )
	    {
	        /* rel_heading must be within the turnrate of object_num. */
	        if(rel_heading > (100 * xsw_object[object_num].turnrate))
		    rel_heading = 100 * xsw_object[object_num].turnrate;
	        else if(rel_heading < (-100 * xsw_object[object_num].turnrate))
		    rel_heading = -100 * xsw_object[object_num].turnrate;

		xsw_object[object_num].heading -= rel_heading;
	    }
	    else
	    {
                /* rel_heading must be within the turnrate of object_num. */
                if(rel_heading > (100 * xsw_object[object_num].turnrate))
                    rel_heading = 100 * xsw_object[object_num].turnrate;
                else if(rel_heading < (-100 * xsw_object[object_num].turnrate))
                    rel_heading = -100 * xsw_object[object_num].turnrate;

                xsw_object[object_num].heading += rel_heading;
	    }
	}

        /* Sanitize heading. */
        while(xsw_object[object_num].heading >= 6.2831853)
           xsw_object[object_num].heading -= 6.2831853;
        while(xsw_object[object_num].heading < 0)
           xsw_object[object_num].heading += 6.2831853;


	return(0);
}



int UpdateThrottle(long object_num)
{
        /* Skip static objects. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_STATIC)
           return(0);

        /* Skip player objects, their throttle is updated on client. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_PLAYER)
           return(0);


	return(0);
}



/* Thrust and velocity is set here. */
int UpdateThrust(long object_num)
{
	/* Local variables. */
	double x;


        /* Skip static objects. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_STATIC)
            return(0);


        /* Thrust output should be 0 if there is no antimatter left. */
        if(xsw_object[object_num].antimatter <= 0)
                xsw_object[object_num].thrust = 0;


	/* Skip player objects from this point on. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_PLAYER)
            return(0);



	/* ******************************************************** */
	/* ***            Begin Calculating Thrust              *** */

	/* Thrust is calculated by taking the coefficent of the  */
	/* throttle and multiplying that with the thrust_power.  */
	x = xsw_object[object_num].throttle;

	/* Sanitize throttle value to be within 0 to 100. */
/*
	if(x < 0)
		x = 0;
	else if(x > 100)
		x = 100;
*/

	x = ( x / 100 ) * xsw_object[object_num].thrust_power;

	xsw_object[object_num].thrust = x;



	/* Make sure velocity is not greater than velocity_max. */
	if(xsw_object[object_num].velocity > xsw_object[object_num].velocity_max)
	{
		xsw_object[object_num].velocity = xsw_object[object_num].velocity_max;
	}


	/* Velocity drag coeff, default #defined to 0.009. */
/*	if(xsw_object[object_num].velocity >= WARP_THRESHOLD) */
	   xsw_object[object_num].velocity -=
		xsw_object[object_num].velocity * WARP_DRAG_COEF;

/*	else if(xsw_object[object_num].velocity >= TRANSWARP_THRESHOLD) */
/*	   xsw_object[object_num].velocity -= TRANSWARP_DRAG_COEF;      */


	/* If velocity is such a small number, then set it 0. */
	if(xsw_object[object_num].velocity < 0.00003)
		xsw_object[object_num].velocity = 0;

	/* Thrust cannot be negative. */
	if(xsw_object[object_num].thrust < 0)
                xsw_object[object_num].thrust = 0; 



	return(0);
}



int UpdateLocation(long object_num)
{
	/* Delta coordinates for... */
	double thrust_dir, thrust_mag;
	double x_vel, y_vel, z_vel;	/* Velocity. */
	double x_thr, y_thr, z_thr;	/* Thrust. */
	double x_res, y_res, z_res;	/* Resultant. */


	/* Skip static objects. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_STATIC)
                return(0);


        /* Skip player objects, their heading is updated on client. */
	if(xsw_object[object_num].type == XSW_OBJ_TYPE_PLAYER)
		return(0);


	/* Stream weapons have location of their owner. */
	if(xsw_object[object_num].type == XSW_OBJ_TYPE_STREAMWEAPON)
	{
	    if((xsw_object[object_num].owner > -1) &&
	       (xsw_object[object_num].owner < MAX_OBJECTS))
	    {
	        xsw_object[object_num].x =
		    xsw_object[xsw_object[object_num].owner].x;
		xsw_object[object_num].y =
		    xsw_object[xsw_object[object_num].owner].y;
	    }
	    return(0);
	}



	/* ***************************************************** */
	/* ***          Begin Moving the Object              *** */

	/* Sanitize xsw_object[object_num].velocity_heading. */
	while(xsw_object[object_num].velocity_heading >= 6.2831853)
		xsw_object[object_num].velocity_heading -= 6.2831853;
	while(xsw_object[object_num].velocity_heading < 0)
		xsw_object[object_num].velocity_heading += 6.2831853;


	/* Calculate movement from velocity (momentum). */
	/* In quadrant 1? */
	if(xsw_object[object_num].velocity_heading < 1.5707963)
	{
	   x_vel = xsw_object[object_num].velocity *
		sin(xsw_object[object_num].velocity_heading);
	   y_vel = xsw_object[object_num].velocity *
		cos(xsw_object[object_num].velocity_heading);
	   z_vel = 0;
	}
	/* In Quadrant 2? */
	else if(xsw_object[object_num].velocity_heading < 3.1415927 )
	{
	   x_vel = xsw_object[object_num].velocity *
		cos(xsw_object[object_num].velocity_heading - 1.5707963);
           y_vel = xsw_object[object_num].velocity *
                sin(xsw_object[object_num].velocity_heading - 1.5707963);
	   y_vel *= -1;
           z_vel = 0;
	}
	/* In Quadrant 3? */
	else if(xsw_object[object_num].velocity_heading < 4.712389 )
	{
           x_vel = xsw_object[object_num].velocity *
                sin(xsw_object[object_num].velocity_heading - 3.1415927);
	   x_vel *= -1;
           y_vel = xsw_object[object_num].velocity *
                cos(xsw_object[object_num].velocity_heading - 3.1415927);
	   y_vel *= -1;
           z_vel = 0;
	}
        /* In Quadrant 4? */
        else
        {
           x_vel = xsw_object[object_num].velocity *
                cos(xsw_object[object_num].velocity_heading - 4.712389);
           x_vel *= -1;
           y_vel = xsw_object[object_num].velocity *
                sin(xsw_object[object_num].velocity_heading - 4.712389);
           z_vel = 0;
        }


        /* Calculate movement from thrust. */
	thrust_dir = xsw_object[object_num].heading +
		xsw_object[object_num].thrust_dir + 3.1415927;
	while(thrust_dir >= 6.2831853)
		thrust_dir -= 6.2831853;
	while(thrust_dir < 0)
		thrust_dir += 6.2831853;


	thrust_mag = xsw_object[object_num].thrust;
	if( (thrust_mag + xsw_object[object_num].velocity ) >
		xsw_object[object_num].velocity_max)
	{
	   thrust_mag = xsw_object[object_num].velocity_max -
		xsw_object[object_num].velocity;
	}

        /* In quadrant 1? */
        if(thrust_dir < 1.5707963)
        {
           x_thr = thrust_mag * sin(thrust_dir);
           y_thr = thrust_mag * cos(thrust_dir);
           z_thr = 0;  
        }
        /* In Quadrant 2? */
        else if(thrust_dir < 3.1415927 )
        {
           x_thr = thrust_mag * cos(thrust_dir - 1.5707963);
           y_thr = thrust_mag * sin(thrust_dir - 1.5707963);
           y_thr *= -1;
           z_thr = 0;
        }
        /* In Quadrant 3? */
        else if(thrust_dir < 4.712389 )
        {
           x_thr = thrust_mag * sin(thrust_dir - 3.1415927);
           x_thr *= -1;
           y_thr = thrust_mag * cos(thrust_dir - 3.1415927);
           y_thr *= -1;
           z_thr = 0;
        }
        /* In Quadrant 4? */
        else
        {
           x_thr = thrust_mag * cos(thrust_dir - 4.712389);
           x_thr *= -1;
           y_thr = thrust_mag * sin(thrust_dir - 4.712389);
           z_thr = 0;
        }


	/* Find the resultant. */
	x_res = x_vel + x_thr;
	y_res = y_vel + y_thr;
	z_res = z_vel + z_thr;


        /* Record new velocity and velocity_heading on object. */
        xsw_object[object_num].velocity_heading = MuLocationtoHeading(
                x_res,
                y_res
        );  
        xsw_object[object_num].velocity = Mu3DDistance(
                x_res,
                y_res,
                z_res  
        );


	/* Move object. */
	xsw_object[object_num].x += x_res;
	xsw_object[object_num].y += y_res;
	xsw_object[object_num].z += z_res;


	return(0);
}



int UpdateAge(long object_num)
{
        /* Age object. */
        if(xsw_object[object_num].lifespan > -1)
        {
           xsw_object[object_num].age =
                cur_millitime - xsw_object[object_num].birth_time;
        }


	return(0);
}



int UpdateMortality(long object_num)
{
	/* Local variables. */
	char stringa[256];


        /* Animation Cycle Check:                                 */
        /* If this is an animated object, check if all the frame  */
        /* cycles has cycled.   If they have, then recycl object. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_ANIMATED)
        {   
	    if(xsw_object[object_num].animation.cycle_times > 0)
	    {
                if(xsw_object[object_num].animation.cycle_count >=
                    xsw_object[object_num].animation.cycle_times)
                {
                    DBRecycleObject(object_num);
		    NetSendRecycleObject(-1, object_num);
                    return(0);
                }
	    }
        }



	/* Aging Check:                                      */
	/* Check if the object's age is older than lifespan, */
	/* If it is, then recycle it.                        */
        if(xsw_object[object_num].lifespan < 0)
                return(0);

        if(xsw_object[object_num].age >
            xsw_object[object_num].lifespan )
        {
            DBRecycleObject(object_num);
	    NetSendRecycleObject(-1, object_num);
            return(0);
        }


	return(0);
}



int UpdateAnimated(long object_num)
{
	/* Local variables. */


        /* We only do XSW Objects with animated imagesets. */
        if(xsw_object[object_num].type != XSW_OBJ_TYPE_ANIMATED)
                return(0);


	/* See which animated imageset this object has. */
	switch(xsw_object[object_num].imageset)
	{
           /* Explosion Type 1 */
           case IMAGESET_CODE_EXPLOSION1:

                /* Time to increment frame? */
                if((cur_millitime -
                   xsw_object[object_num].animation.last_interval)
                   > xsw_object[object_num].animation.interval
                )
                {
                   xsw_object[object_num].animation.current_frame++;
                   xsw_object[object_num].animation.last_interval = cur_millitime;
                }

                /* Explosion 1 has 4 frames. if we go beyond that   */
                /* then the cycle_count needs to be incremented and */
                /* the current_frame set back to 0.                 */
                if(xsw_object[object_num].animation.current_frame > 3)
                {
                   xsw_object[object_num].animation.current_frame = 0;
                   xsw_object[object_num].animation.cycle_count++;
                }
                break;

           /* Explosion Type 2 */
           case IMAGESET_CODE_EXPLOSION2:

                /* Time to increment frame? */
                if((cur_millitime -
                   xsw_object[object_num].animation.last_interval)
                   > xsw_object[object_num].animation.interval 
                )
                {
                   xsw_object[object_num].animation.current_frame++;
                   xsw_object[object_num].animation.last_interval = cur_millitime;
                }

                /* Explosion 2 has 1 frame. if we go beyond that    */
                /* then the cycle_count needs to be incremented and */
                /* the current_frame set back to 0.                 */
                if(xsw_object[object_num].animation.current_frame > 3)
                {
                   xsw_object[object_num].animation.current_frame = 0;
                   xsw_object[object_num].animation.cycle_count++;
                }
                break;


	    /* Default, set frame to 0 and do nothing else. */
	    default:

		xsw_object[object_num].animation.current_frame = 0;

		break;
	}


	return(0);
}



int UpdateWeaponsLock(long object_num)
{
	/* Local variables. */
	double x, y;
	double cur_range;
	long locked_object;


	/* Is object_num locked onto anything? */
	if(
	   (xsw_object[object_num].locked_on < 0) ||
	   (xsw_object[object_num].locked_on >= MAX_OBJECTS)
	)
		return(0);


	/* Get locked_object. */
	locked_object = xsw_object[object_num].locked_on;

	/* Get delta distances between objects. */
        x = xsw_object[locked_object].x - xsw_object[object_num].x;
        y = xsw_object[locked_object].y - xsw_object[object_num].y;

	/* Get effective scanner range from object_num to locked_object. */
        cur_range = xsw_object[object_num].scanner_range *
                          DBGetObjectVisibility(locked_object);

	/* See if delta distance is out of effective scanner range. */
        if(Mu3DDistance(x, y, 0) > cur_range)
	{
           xsw_object[object_num].locked_on = -1;
	}


	return(0);
}



int UpdateIntercept(long object_num)
{
        /* Local variables. */
        double x, y;
        double cur_range;
        long intercepting_object;
 

        /* Is object_num intercepting anything? */
        if(
           (xsw_object[object_num].intercepting_object < 0) ||
           (xsw_object[object_num].intercepting_object >= MAX_OBJECTS)
        )
                return(0);

	/* Use MatchInterceptByNumber() to see if object is in range. */
	intercepting_object = xsw_object[object_num].intercepting_object;
	if(MatchInterceptByNumber(object_num, intercepting_object) < 0)
	{
		xsw_object[object_num].intercepting_object = -1;
	}


	return(0);
}



int UpdatePowerCore(long object_num)
{
	/* Increase power in the object's power core. */
        if(xsw_object[object_num].power <
              xsw_object[object_num].power_max)
        {
           xsw_object[object_num].power +=
              xsw_object[object_num].core_efficency;

           if(xsw_object[object_num].power >
                   xsw_object[object_num].power_max)
           {
                xsw_object[object_num].power = xsw_object[object_num].power_max;
           }
        }


	return(0);
}



int UpdateAntimatter(long object_num)
{
	/* Local variables. */
	double throttle;
	double thrust_power;

        /* Skip static objects. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_STATIC)
                return(0);


	throttle = xsw_object[object_num].throttle;
	thrust_power = xsw_object[object_num].thrust_power;


        /* Consume Antimatter based on (throttle / 100) * thrust_power */
        xsw_object[object_num].antimatter -= (throttle / 100) * thrust_power;


        if(xsw_object[object_num].antimatter < 0)
                xsw_object[object_num].antimatter = 0;


	return(0);
}



int UpdateDoHit(long wobj, long tobj)
{
	/* WARNING: This function makes no checks if wobj and tobj really
	 * hit each other.   It is up to UpdateCollisionCheck() to do all
	 * the contact checking.
	 */


	/* Local variables. */
        long x, y;
        char sndbuf[MAX_NET_CMD_SIZE];

	double dmgtohull = 0;	/* Hull damage. */
	double dmgtopwr = 0;	/* Shield damage. */
	double tdmg = 0;	/* Total damage incurred. */
	int rtn_code = 0;


	/* Check if wobj has any power. */
	if(xsw_object[wobj].power <= 0)
	{
		return(-1);
	}

	/* Take away shield power and/or hitpoints from tobj. */
	if(xsw_object[tobj].shield_state == 1)
	{
	    xsw_object[tobj].power =
		xsw_object[tobj].power - xsw_object[wobj].power;
	    if(xsw_object[tobj].power < 0)
	    {
		xsw_object[tobj].hp = xsw_object[tobj].hp + xsw_object[tobj].power;
	    }
	}
	else
	{
	    xsw_object[tobj].hp =
		xsw_object[tobj].hp - xsw_object[wobj].power;
	}

	/* Sanitize power. */
	if(xsw_object[tobj].power < 0)
	{
	    xsw_object[tobj].power = 0;
	}


	/* Update score. */
	xsw_object[tobj].score.damage_recieved += xsw_object[wobj].power;
	if( (xsw_object[wobj].owner > -1) && (xsw_object[wobj].owner < MAX_OBJECTS) )
	{
	    xsw_object[xsw_object[wobj].owner].score.damage_givin +=
		xsw_object[wobj].power;
	    if(xsw_object[tobj].hp < 0)
		xsw_object[xsw_object[wobj].owner].score.kills++;
	}


	/* Send message to tobj's connections. */
	if(xsw_object[tobj].type == XSW_OBJ_TYPE_PLAYER)
	{
	    if(xsw_object[tobj].power_max > 0)
	    {
	        sprintf(sndbuf, "%i We're hit!  Shields at %.0lf%%.\n%c",
		    NET_CMD_GLOBALMESG,
		    (xsw_object[tobj].power / xsw_object[tobj].power_max) * 100,
		    '\0'
	        );
	    }
	    else
	    {
		sprintf(sndbuf, "%i We're hit!\n%c",
                    NET_CMD_GLOBALMESG,
                    '\0'
                );
	    }
	    for(x = 0; x < total_connections; x++)
	    {
	        if(connection[x].socket < 0)
		    continue;
	        if(connection[x].object_num != tobj)
		    continue;

		NetSendDataToConnection(x, sndbuf, 1);
	    }
	}

	/* Send message to wobj's owner's connection. */
	if( (xsw_object[wobj].owner > -1) && (xsw_object[wobj].owner < MAX_OBJECTS)
	)
	{
	    if( xsw_object[xsw_object[wobj].owner].type == XSW_OBJ_TYPE_PLAYER)
            {
		sprintf(sndbuf, "%i Hit %s(#%i):  DmgGiv: %.2lf\n%c",
		    NET_CMD_GLOBALMESG,
                    xsw_object[tobj].name,
		    tobj,
		    xsw_object[wobj].power,
                    '\0'
                );
                for(x = 0; x < total_connections; x++)
                {   
                    if(connection[x].socket < 0)
                        continue;
                    if(connection[x].object_num != xsw_object[wobj].owner)
                        continue;
             
                    NetSendDataToConnection(x, sndbuf, 1);
                }
            }
	}



        /* Create explosion. */
	if(xsw_object[tobj].hp < 0)
	{
            DBCreateExplosion1(
                xsw_object[tobj].x,
                xsw_object[tobj].y
            );

/* Recycle tobj? */


	    rtn_code = 1;

	}
	else
	{
	    if(xsw_object[wobj].type == XSW_OBJ_TYPE_STREAMWEAPON)
	    {
                DBCreateExplosion2(
                    xsw_object[tobj].x,
                    xsw_object[tobj].y
                );
	    }
	    else
	    {
                DBCreateExplosion2(
                    xsw_object[wobj].x,
                    xsw_object[wobj].y
                );
	    }
	}


/* Sanitize hp on tobj (since we didn't recycle it. */
if(xsw_object[tobj].hp < 0)
    xsw_object[tobj].hp = 0;



	/* Recycle the weapon object if its not a stream weapon. */
        if(xsw_object[wobj].type == XSW_OBJ_TYPE_WEAPON)
	{
            DBRecycleObject(wobj);
            NetSendRecycleObject(-1, wobj);
	}


	return(rtn_code);
}



int UpdateCollisionCheck(long object_num)
{
	/* Local variables. */
	long object_count;
	char stringa[256];
	double heading_variance;
	double distance;
	int result;


        /* ****************************************************** */
        /* ***                Colission Check                 *** */

	/* Colission check if object_num is a regular weapon. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_WEAPON)
        {

	    /* Begin checking for contact. */
            for(object_count = 0; object_count < total_objects; object_count++)
            {
		/* Skip garbage and error objects. */
                if(xsw_object[object_count].type <= XSW_OBJ_TYPE_GARBAGE)
                        continue;

                /* Skip stream weapons. */
                if(xsw_object[object_count].type == XSW_OBJ_TYPE_STREAMWEAPON)
                        continue;

                /* Skip XSW_OBJ_TYPE_AREA. */
                if(xsw_object[object_count].type == XSW_OBJ_TYPE_AREA)
                        continue;

		/* Skip XSW_OBJ_TYPE_ANIMATED. */
                if(xsw_object[object_count].type == XSW_OBJ_TYPE_ANIMATED)
                        continue;

                /* Don't hit its owner. */
                if(xsw_object[object_num].owner == object_count)
                        continue;
        
                /* Don't hit itself. */
                if(object_num == object_count)
                        continue;

                /* Did it hit something? */
                if(Mu3DInContact(object_num, object_count) > 0)
                {
		    UpdateDoHit(object_num, object_count);
                    return(1);
                }
            }            
        }



        /* Colission check if object_num is a stream weapon. */
        if(xsw_object[object_num].type == XSW_OBJ_TYPE_STREAMWEAPON)
        {

	    /* Stream weapon object must have power. */
	    if(xsw_object[object_num].power <= 0)
		return(0);

	    /* Begin checking for contact. */
            for(object_count = 0; object_count < total_objects; object_count++)
            {          
                /* Skip garbage and error objects. */
                if(xsw_object[object_count].type <= XSW_OBJ_TYPE_GARBAGE)
                        continue;

		/* Skip other stream weapons (and itself). */
		if(xsw_object[object_count].type == XSW_OBJ_TYPE_STREAMWEAPON)
			continue;

                /* Skip XSW_OBJ_TYPE_AREA. */
                if(xsw_object[object_count].type == XSW_OBJ_TYPE_AREA)    
                        continue;

                /* Skip XSW_OBJ_TYPE_ANIMATED. */
                if(xsw_object[object_count].type == XSW_OBJ_TYPE_ANIMATED)
                        continue;

                /* Don't hit its owner. */
                if(xsw_object[object_num].owner == object_count)
                        continue;


                /* Did it hit something? */
		distance = Mu3DDistance(
			xsw_object[object_count].x - xsw_object[object_num].x,
			xsw_object[object_count].y - xsw_object[object_num].y,
			0
		);
		distance *= 1000;	/* distance must be in XSW Screen units. */
		heading_variance = atan(
			xsw_object[object_count].size / distance
		);
		result = Mu3DInVectorContact(
			object_num,
			object_count,
			xsw_object[object_num].heading,
			heading_variance,
			xsw_object[object_num].size + xsw_object[object_count].size
		);
		if(result > 0)
		{
                    UpdateDoHit(object_num, object_count);

		    /* Instead of recycling the stream weapon object, */
		    /* we set its power to 0.                         */
		    xsw_object[object_num].power = 0;
		    return(1);
		}
	    }
	}


	return(0);
}



int UpdateObject(long object_num)
{
	/* Local variables. */
	int result;


	/* ************************************************** */
	/* ***          Error/Validility Checking         *** */
        /* Check if object_num is in range. */
        if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
                return(-1);

	/* Skip XSW Objects type error or garbage. */
	if(xsw_object[object_num].type <= XSW_OBJ_TYPE_GARBAGE)
                return(0);


	/* ************************************************** */
	/* ***           Begin Updating Object            *** */

	/* Update age. */
	UpdateAge(object_num);

	/* Update mortality. */
	UpdateMortality(object_num);

        /* Update animated object. */
        UpdateAnimated(object_num);

	/* Update power core. */
	UpdatePowerCore(object_num);

	/* Update heading. */
	UpdateHeading(object_num);

	/* Update throttle. */
	UpdateThrottle(object_num);

	/* Update thrust. */
	UpdateThrust(object_num);

	/* Update location (movement). */
	UpdateLocation(object_num);

	/* Update weapons lock. */
	UpdateWeaponsLock(object_num);

	/* Update (consume) antimatter. */
	UpdateAntimatter(object_num);

	/* Update check for collision. */
	result = UpdateCollisionCheck(object_num);
	/* Positive results means object_num was recycled. */
	if(result > 0)
		return(0);


	return(0);
}
