/* ***************************************************************************
 *
 *      objects.h	XSW Object and ImageSet Structures
 *      --------------------------------------------------
 */


#ifndef __XSW_OBJECTS__
#define __XSW_OBJECTS__


/*
 *   Default XSW Object Permission IDs:
 *
 *	Ligher settings give higher permissions. Default values should be
 *	a high number. 5 is a good number.
 */
#define DEFAULT_UID 5
#define DEFAULT_GID 5


/*
 *   Name and Password Lengths:
 *
 *      Includes null terminating byte.
 */
#define MAX_NAME_LENGTH       25
#define MAX_PASSWORD_LENGTH   25
#define MAX_EMPIRE_LENGTH     9


/*
 *   Maximum Values for Objects:
 *
 *      These are maximum values for objects.
 */
#define MAX_OBJECTS 5000
#define MAX_IMAGESET_FRAMES 16
#define MAX_IMAGESETS 100
#define MAX_WEAPONS 4           /* For each XSW Object. */



/*
 *   Core XSW Object Type Codes:
 *
 *      These are codes (data type long) to identify each XSW
 *      Object.   They are stored in each object's xsw_object[].type.
 *
 *      XSW_OBJ_TYPE_ERROR is similar to XSW_OBJ_TYPE_GARBAGE
 *      except that it means that xsw_object[] should not be used
 *      (think of it as a bad sector on a disk).
 *
 *      XSW_OBJ_TYPE_GARBAGE is the most common type, it means that
 *      that xsw_object[] contains nothing of value and may be used.
 */
#define XSW_OBJ_TYPE_ERROR              -1  /* Do not use. */
#define XSW_OBJ_TYPE_GARBAGE            0   /* Free to re/use. */
#define XSW_OBJ_TYPE_STATIC             1   /* Primitive, not move. */
#define XSW_OBJ_TYPE_DYNAMIC            2   /* Moves. */
#define XSW_OBJ_TYPE_CONTROLLED         3   /* Controllable. */
#define XSW_OBJ_TYPE_PLAYER             4   /* Controlled by player. */
#define XSW_OBJ_TYPE_WEAPON             5   /* Regular weapons fire object; */
                                            /* torpedo, pulse, etc. */
#define XSW_OBJ_TYPE_STREAMWEAPON       6   /* Stream weapons; phazer beams. */
#define XSW_OBJ_TYPE_HOME               10  /* Starbases, planets, etc. */
#define XSW_OBJ_TYPE_AREA               11  /* Space weather phoemonia;  */
                                            /* nebulas, plasma storms... */
#define XSW_OBJ_TYPE_ANIMATED           12  /* Explosions, wormholes... */
/* more to come... */



/*
 *   ImageSet Codes:
 *   
 *      These are image set codes for various XSW Objects models.   They
 *      are used to determine which image set to draw for each
 *      xsw_object[].imagesets[].
 */
#define IMAGESET_CODE_UNKNOWN      0    /* Question mark. */
#define IMAGESET_CODE_GENERIC      1

#define IMAGESET_CODE_BUOY         10
#define IMAGESET_CODE_HUBBLE       11	/* Hubble Space Telescope. */

#define IMAGESET_CODE_EXPLOSION1   20
#define IMAGESET_CODE_EXPLOSION2   21

#define IMAGESET_CODE_WEP_PP_G     30
#define IMAGESET_CODE_WEP_PP_Y     31
#define IMAGESET_CODE_WEP_PT_R     32

/* Stream weapons are special, they do not have image sets, and     */
/* instead have themselves drawn using XLib draw functions.         */
#define IMAGESET_CODE_WEP_STREAM_Y 40
#define IMAGESET_CODE_WEP_STREAM_G 41

#define IMAGESET_CODE_MRAD         50	/* Maqui Raider. */
#define IMAGESET_CODE_KBOP         51	/* Klingon Bird of Prey. */
#define IMAGESET_CODE_INTR         52   /* Intrepid Class. */
#define IMAGESET_CODE_KVOR         53   /* Klingon Vortcha Class. */

#define IMAGESET_CODE_STCAR        80   /* Station - Cardassian. */
/* more to come... */



/*
 *   Weapon Type Codes:
 *   
 *      These codes determine what kind/type of weapon a weapon is during
 *      the creation of that weapon.
 *      Each XSW Object may have weapons (yes even weapons may have weapons),
 *      the weapon type code is stored in xsw_object[].weapons[].type.
 *      In weapons.c WepCreate() checks the type of weapon to create. 
 *      After a weapon is created, it becomes a XSW Object and it's   
 *      xsw_object[].type becomes XSW_OBJ_TYPE_WEAPON or
 *      XSW_OBJ_TYPE_STREAMWEAPON.
 */
#define WEPTYPE_NULL            0       /* Water Pistol */

#define WEPTYPE_PHAZER_1        10      /* Stream style. */
#define WEPTYPE_PHAZER_2        11
#define WEPTYPE_PHAZER_3        12
#define WEPTYPE_PHAZER_4        13
#define WEPTYPE_PHAZER_5        14

#define WEPTYPE_DISRUPTOR_1     20      /* Stream style. */
#define WEPTYPE_DISRUPTOR_2     21      /* Single pulse. */
#define WEPTYPE_DISRUPTOR_3     22      /* Double pulse. */
#define WEPTYPE_DISRUPTOR_4     23
#define WEPTYPE_DISRUPTOR_5     24

#define WEPTYPE_ESPHERE_1       30
#define WEPTYPE_ESPHERE_2       31
#define WEPTYPE_ESPHERE_3       32
#define WEPTYPE_ESPHERE_4       33
#define WEPTYPE_ESPHERE_5       34

#define WEPTYPE_PULSE_PHAZER_1  40      /* Single pulse. */
#define WEPTYPE_PULSE_PHAZER_2  41      /* Double pulse, Maqui Raider. */
#define WEPTYPE_PULSE_PHAZER_3  42
#define WEPTYPE_PULSE_PHAZER_4  43      /* Quad pulse, Defiant Class. */
#define WEPTYPE_PULSE_PHAZER_5  44

#define WEPTYPE_PHOTON_TORP_1   60      /* Single fire. */
#define WEPTYPE_PHOTON_TORP_2   61      /* Double fire, Defiant Class. */
#define WEPTYPE_PHOTON_TORP_3   62
#define WEPTYPE_PHOTON_TORP_4   63      /* Quad fire, Vortcha Class. */
#define WEPTYPE_PHOTON_TORP_5   64
/* more to come... */



/*
 *   Weapon Emission Type Codes:
 *   
 *      These determine the firing style of the weapon.   The codes for
 *      each weapon are stored in xsw_object[].weapons[].emission_type.
 *      They are checked for by functions in draw.c and inhandler.c.   
 *   
 */  
#define WEPEMISSION_STREAM       0      /* Supply dependant on power left. */
#define WEPEMISSION_PROJECTILE   1      /* Limited supply. */
#define WEPEMISSION_PULSE        2      /* Supply dependant on power left. */



/*
 *   Database Title:
 */
char dbtitle[1024];



/*
 *   Scoring Structure:
 *
 *	Each XSW Object has this structure.
 */
typedef struct
{
    double gpl;		/* Strips of Gold Pressed Latinum. */
    double rmu;		/* Raw material units. */
    double rmu_max;	/* Maximum raw material units. */
    double damage_givin;
    double damage_recieved;
    long kills;
} score_struct;


/*
 *   Permission Structure:
 *
 *	Each XSW Object has this structure.
 */
typedef struct
{
    int uid;		/* User ID, 0 is highest. */
    int gid;		/* Group ID, 0 is highest. */
} permission_struct;


/*
 *   ImageSet Structure:
 *
 *      This structure is not within a XSW Object structure,
 *      The imageset code for an XSW Object should match the
 *      imageset number and may not exceed MAX_IMAGESETS.
 */
#ifdef X_H
typedef struct
{
    int load_state;	/* 0 = not loaded, 1 = loaded. */
    int width, height;	    /* Total width and height (not used yet) */
    int fwidth, fheight;    /* Frame width and height. */
    Pixmap image[MAX_IMAGESET_FRAMES];
    Pixmap shape_mask[MAX_IMAGESET_FRAMES];
} xsw_imageset_struct;

xsw_imageset_struct xsw_imageset[MAX_IMAGESETS];
#endif


/*
 *   Weapons Structure:
 *   
 *      Each XSW Object structure has this.
 */
typedef struct
{
 
   char name[MAX_NAME_LENGTH];
		        /* Name of weapon NO LONGER USED. The name of */
			/* weapon is now determined by its type.      */

   long type;           /* Type of weapon. */
   long emission_type;  /* Type of emission (beam or projectile) */
 
   long amount;         /* For projectile type weapons only. */
   long max;            /* For projectile type weapons only. */
 
   double power;        /* Amount of damage this weapon causes */
                        /* per hit.                            */
   double range;        /* For WEPEMISSION_STREAM type weapons only, */
                        /* in XSW Screen (pixel) units.              */
                        
   double create_power; /* Power needed to create one unit of this */
                        /* weapon.                                 */
                        
   long delay;          /* In milliseconds, time between uses. */
   long last_used;      /* In milliseconds. time last used. */   
   
} xsw_weapons_struct;



/*
 *   Animations Structure:
 *   
 *      Each XSW Object structure has this.
 */
typedef struct
{
   long interval;          /* Number of milliseconds between frames. */
   long last_interval;     /* In milliseconds, time current frame was */
                           /* incremented.                            */
   long current_frame;     /* Current frame number, starts at 0. */
   long total_frames;      /* Total number of frames, starts at 1. */
   long cycle_count;       /* Number of times frames has cycled. */  
   long cycle_times;       /* Number of times to cycle, or -1 to   */
                           /* cycle infinately                     */
} xsw_animation_struct;



/*
 *   Core Structure for XSW Objects:
 *
 *	All XSW Objects have these members.
 */
struct xsw_object_struct
{
   char name[MAX_NAME_LENGTH];
   char empire[MAX_EMPIRE_LENGTH];
   char password[MAX_PASSWORD_LENGTH];

   long last_updated;		/* In milliseconds since last update */
				/* by server.                        */
   
   long type;                   /* Type of object. */
   long imageset;               /* Imageset type code for object. */
   long owner;
   long size;                   /* Size of object in pixels. */
   
   long locked_on;      /* Scanning/locked on object, can be -1 */
                        /* for unlocked.                        */
                        
   long intercepting_object;    /* Intercepting this object,       */
                                /* can be -1 for not intercepting. */

#ifdef X_H
   Window window;               /* Object's XLib window. */
#endif
   int window_width;            /* Window height and width. */
   int window_height;
   int window_mapping;		/* 0 = unmapped, 1 = mapped. */
   
   double scanner_range;        /* Radius in XSW real units. */
   
   double x,y,z;                /* z is not implmented yet. */
   double heading;              /* In radians. */
   double pitch;                /* In radians (not implmented yet). */
   
   double velocity;             /* In XSWunits per MAIN_LOOP_INTERVAL. */
   double velocity_max;
   double velocity_heading;     /* In radians. */
   
   int thrust_rev_state;        /* 0 = off, 1 = on. */
   double thrust_dir;           /* In radians. */
                                /* -1 = backwards                  */
   double thrust;               /* Thrust being applied.   Relative */
                                /* from throttle and velocity_max.  */
   double thrust_power;         /* Full power of thrust when throttle  */
                                /* is at 100.                          */
   int throttle;                /* 0 to 100. */
   
   double turnrate;
   
   int brakes_state;            /* 0 = off, 1 = on. */
   double brakes_power;
   
   double hp;
   double hp_max;
   
   double power;
   double power_max;    
   double power_purity;         /* 0.00 to 0.99 */
                                
   double core_efficency;       
   
   double antimatter;
   double antimatter_max;
   
   int shield_state;            /* 0 for off, 1 for on. */
   double shield_frequency;     /* In KHz. */
   
   int selected_weapon;
   int total_weapons;           /* May not be greater than MAX_WEAPONS. */
   
   long birth_time;             /* In milliseconds. */
   long age;                    /* In milliseconds. */
   long lifespan;               /* How old before dies, less than 0 means */
                                /* never dies. */

   
   int cloak_state;             /* -1 = none, 0 = off, 1 = on */
   double cloak_strength;       /* 0.0 to 1.0, when cloak is on,      */
                                /* cloak_strength reduces visibility. */
   double visibility;           /* 0.0 to 1.0. */
   
   double shield_visibility;    /* In milliseconds. When greater than 0, */
                                /* shields are visibable.                */
   
   int damage_control;          /* 0 = off, 1 = on. */


   /* Score structure. */
   score_struct score;

   /* Permission structure. */
   permission_struct permission;

   /* Animation structure. */
   xsw_animation_struct animation;

   /* Weapons structure, limited to MAX_WEAPONS. */
   xsw_weapons_struct weapons[MAX_WEAPONS];
   
} xsw_object[MAX_OBJECTS];
   
/*
 *   Total (Topmost) Number of Objects:
 *
 *	Automatically increased to reflect topmost object number + 1
 *	whenever a new object is allocated in DBCreateObject() and
 *	DBCreateExplicitObject(). This number must always be equal or less
 *	than MAX_OBJECTS.
 */
long total_objects;    



#endif
