/* *************************************************************************
 *
 *	net.c	Network Data Transfering Functions
 *	------------------------------------------
 *	Functions:
 *
 *	int NetSendData(char *outbounddata)
 *	int NetOpenConnection(char *new_hostname, int new_portnum)
 *      int NetHandleLogin()
 *      int NetSendGlobalMessage(char *message)
 *      int NetSendDisconnect()
 *      int NetSendSetInterval()
 *	int NetSendExec(char *arg)
 *      int NetSendLocalObjectPosition()
 *      long NetHandleWhoAmI(char *arg)
 *      int NetHandleSetObjApp(char *arg)
 *      int NetHandleSetObjAppS(char *arg)
 *      int NetHandleSetObjectValues(char *arg)
 *      int NetHandleSetObjectMaximums(char *arg)
 *	int NetSendSetWeapon(int selected_weapon)
 *      int NetSendSetShields(int shield_state, double shield_frequency)
 *      int NetSendSetDmgCtl()
 *      int NetSendSetCloak()
 *	int NetSendWeaponsLock()
 *	int NetSendWeaponsUnlock()
 *	int NetSendIntercept(char *arg)
 *	int NetHandleSetName(char *arg)
 *      int NetHandleRecv()
 *
 *	---
 *
 *	NetSendData() sends out outbounddata to the swserv_socket.
 *	If the swserv_socket is invalid (not connected or -1) then;
 *	the data will not be sent, the swserv_socket will be set to -1,
 *	the connection_state will be set to 0, and the lplayer_object will
 *	be set to -1.
 *	Returns non-zero on error.
 *
 *	NetOpenConnection() opens a AF_INET SOCK_STREAM socket.
 *	Returns the socket number or -1 if the socket cannot be opened.
 *
 *
 *
 *	NetHandleRecv() uses recv() to get command from the server.
 *	This function should be called once per loop in the main()'s
 *	main event loop.   It will parse the recv()'ed data and call the
 *	appropriate Net*() function to handle it.
 *	Returns non-zero on error.
 *
 *
 *
 *
 *
 *
 *
 */

#include "../include/xsw.h"  



int NetSendData(char *outbounddata)
{
	/* Local variables. */
	int bytes_sent;
	struct timeval timeout;
	fd_set writefds;


	/* Make sure socket is opened. */
	if(swserv_socket < 0)
		return(-1);


	/* Set timeout. */
	timeout.tv_sec = 0;
	timeout.tv_usec = 2000;


	/* Check if socket is ready for send. */
	errno = 0;
	FD_ZERO(&writefds);
	FD_SET(swserv_socket, &writefds);
	select(swserv_socket + 1, NULL, &writefds, NULL, &timeout);


	/*
	 *   See if swserv_socket is connected and valid.
	 *   If not then set it as -1 so no other function uses it
	 *   accidentally.
	 */
	if(errno == EBADF)
	{
	    errno = 0;
	    fprintf(stderr, "NetSendData(): Socket invalid: Lost connection.\n");
	    swserv_socket = -1;
            lplayer_object = -1; 
            connection_state = 0;
	    return(-1);
	}


	/* Send data if everything is okay. */
	if(FD_ISSET(swserv_socket, &writefds))
	{
	    bytes_sent =
/*		send(swserv_socket, outbounddata, MAX_NET_CMD_SIZE, 0); */
		write(swserv_socket, outbounddata, MAX_NET_CMD_SIZE);

	    if(errno == ENOBUFS)
		fprintf(stderr, "errno: Network Congestion.\n");
	    if(bytes_sent < 0)
		fprintf(stderr, "send() no bytes sent.\n");
	}
	else
	{
	    fprintf(stderr, "FD_ISSET() returned false.\n");
/* Consider putting an increment for a send error count, if send error  */
/* exceeds a certain number, then automatically close socket.           */

	}

	return(0);
}



int NetOpenConnection(char *new_hostname, int new_portnum)
{

	/* Local variables. */
        int new_sock;	               /* socket number */
	struct hostent *he;            /* Host pointer. */
	struct servent *sp;            /* Service port pointer. */
	struct sockaddr_in newhostaddr;


	/* Get new hostname. */
	he = gethostbyname(new_hostname);
	if(he == NULL)
	{
		MesgWinAddMessage("gethostbyname(): Unknown Host.",
			normal_text_pixel);
		return(-1);
	}

	/* Get new socket. */
	new_sock = socket(AF_INET, SOCK_STREAM, 0);
	if(new_sock < 0)
	{
                MesgWinAddMessage("socket(): Cannot open socket.",
                        normal_text_pixel);
                return(-1);
	}

	newhostaddr.sin_family = AF_INET;		/* In hbo. */
	newhostaddr.sin_port = htons(new_portnum);	/* In nbo. */
	newhostaddr.sin_addr = *((struct in_addr *)he->h_addr);
	bzero(&(newhostaddr.sin_zero), 8);	/* Zero the rest of struct. */

	/* Open connection. */
	if(connect(new_sock, (struct sockaddr *)&newhostaddr,
		sizeof(struct sockaddr)) == -1)
	{
                MesgWinAddMessage("connect(): Cannot connect stream socket.",
                        normal_text_pixel);
                return(-1);
	}


	/* Set socket to non-blocking. */
        fcntl(new_sock, F_SETFL, O_NONBLOCK);


        /* Return the new socket number. */
        return(new_sock);
}



int NetHandleLogin()
{
	/* Local variables. */
	char sndbuf[MAX_NET_CMD_SIZE];
	char stringa[256];
	int status;


	/* Format the data to be sent. */
	sprintf(sndbuf, "%i %s;%s\n%c",
		NET_CMD_LOGIN,
		loginname,
		loginpassword,
		'\0'
	);

	/* Send the data. */
	status = NetSendData(sndbuf);

	/*
	 *   We've sent our name and password, that does not mean we are
	 *   logged in yet.   We wait for a NET_CMD_WHOAMI response from
	 *   server.   Once we recieve a NET_CMD_WHOAMI, then we are
	 *   logged in.
	 *
	 *   IMPORTANT: The connection_state will be set to 2 once we
	 *   receive a location and apperance update for lplayer_object.
	 *   See NetHandleSetObjApp() farther below.
	 */

	/* Print on the message_win that we've sent the login data. */
	if(status != 0)
	{
	    sprintf(stringa, "Error: Unable to send login.");
	    MesgWinAddMessage(stringa, normal_text_pixel);
	}
	else
	{
	    sprintf(stringa, "Sending login name and password for: %s",
		loginname);
	    MesgWinAddMessage(stringa, normal_text_pixel);
	}

	return(status);
}



int NetSendGlobalMessage(char *message)
{
	/* Local variables. */
	int x, y, z;
	char stringa[256];
	char sndbuf[MAX_NET_CMD_SIZE];
	int bytes_sent;

	/*
	 *   NET_CMD_GLOBALMESG format is as follows:
	 *
	 *	message
	 */
	sprintf(sndbuf, "%i %s\n%c",
		NET_CMD_GLOBALMESG,
		message,
		'\0'
	);
	sndbuf[MAX_NET_CMD_SIZE -1] = '\0';


	/* Send data. */
	if((NetSendData(sndbuf)) < 0)
	{
		MesgWinAddMessage(
		    "NetSendGlobalMessage(): Error sending message.",
		    normal_text_pixel);
		return(-1);
	}
	else
	{
		return(0);
	}
}



int NetSendDisconnect()
{
        /* Local variables. */
        char recbuf[MAX_NET_CMD_SIZE];
	char sndbuf[MAX_NET_CMD_SIZE];
	int timeout;
        int bytes_read = 0;
	char *strptr;


	/* Disconnect only if socket is opened. */
	if(swserv_socket > -1)
	{
	   sprintf(sndbuf, "%i\n%c", NET_CMD_LOGOUT, '\0');
	   NetSendData(sndbuf);
	}


	return(0);
}



int NetSendSetInterval()
{
	/* Local variables. */
	char sndbuf[MAX_NET_CMD_SIZE];


	/* Make sure socket to server is opened. */
	if(swserv_socket < 0)
		return(-1);


        /*
         *   NET_CMD_SETINTERVAL format is as follows:
         *
         *      interval
         */
	sprintf(sndbuf, "%i %i\n%c",
		NET_CMD_SETINTERVAL,
		net_server_update_interval,
		'\0'
	);

        sndbuf[MAX_NET_CMD_SIZE - 1] = '\0';


        /* Send it. */
        NetSendData(sndbuf);

	return(0);
}



int NetSendExec(char *arg)
{
        /* Local variables. */
        char sndbuf[MAX_NET_CMD_SIZE];


        /* Make sure socket to server is opened. */
        if(swserv_socket < 0)
                return(-1);
 
        /* 
         *   NET_CMD_EXEC format is as follows:
         *
         *      argument
         */
        sprintf(sndbuf, "%i %s\n%c",
                NET_CMD_EXEC,
                arg,
                '\0'
        );

	sndbuf[MAX_NET_CMD_SIZE - 1] = '\0';


        /* Send it. */
        NetSendData(sndbuf);

	return(0);
}



int NetSendLocalObjectPosition()
{
        /* Local variables. */
        char sndbuf[MAX_NET_CMD_SIZE];
 
 
        /*
	 *   NET_CMD_SETOBJLOCAPP format is as follows:
	 *
	 *	object_num,
	 *	type, imageset, size,
	 *	x, y, z,
	 *	heading, pitch,
	 *	velocity velocity_heading,
	 *	throttle, frame
	 */
        sprintf(sndbuf,
"%i %i\
 %i %i %i\
 %lf %lf %lf\
 %lf %lf\
 %lf %lf\
 %i %i\n%c",
                NET_CMD_SETOBJLOCAPP,
                lplayer_object,

                xsw_object[lplayer_object].type,
                xsw_object[lplayer_object].imageset,
		xsw_object[lplayer_object].size,

                xsw_object[lplayer_object].x,
                xsw_object[lplayer_object].y,
                xsw_object[lplayer_object].z,

                xsw_object[lplayer_object].heading,
                xsw_object[lplayer_object].pitch,

                xsw_object[lplayer_object].velocity,
                xsw_object[lplayer_object].velocity_heading,

                xsw_object[lplayer_object].throttle,
		xsw_object[lplayer_object].animation.current_frame,
		'\0'
        );


        /* Send it. */
        NetSendData(sndbuf);
                
                
        return(0);
}



long NetHandleWhoAmI(char *arg)
{
        /* Local variables. */
        long object_num;
        char name[MAX_NET_CMD_SIZE];
	char object_str[MAX_NET_CMD_SIZE];
	char *strptr;
	int x;
                

	/* Get name. */
	strcpy(name, arg);
	if((strptr = strstr(name, ";")) != NULL)
	{
		*strptr = '\0';

		/* Get object number. */
		strptr = strstr(arg, ";");
		strptr += 1;
		strcpy(object_str, strptr);
		object_num = atol(object_str);

		/* Make sure object_num is valid. */
		if((object_num < 0) || (object_num >= MAX_OBJECTS))
			return(-1);

		/* If the lplayer_object has not been created yet,
		 * we do *NOT* create it here.
		 * The reason is because we do not know the imageset code
		 * thus we do not have enough information to pass to
		 * DBCreateExplicitObject().
		 * The lplayer_object will be created in
		 * NetHandleSetObjApp() or one of the other appearance
	 	 * or values setting functions.
		 *
		 */

		/* Set the lplayer_object to the number givin to us. */
		lplayer_object = object_num;

		return(object_num);
	}
	else
	{
		return(-1);
	}
}



int NetHandleRecycleObject(char *arg)
{
	/* Local variables. */
	long object_num;


        /* arg must be atleast 1 character long. */
        if(strlen(arg) < 1)
                return(-1);


        /*
         *   NET_CMD_RECYCLEOBJECT format is as follows:
         *
         *      object_num
         */
	sscanf(arg, "%i",
		&object_num
	);


        /* See if object_num is valid. */
        if((object_num < 0) || (object_num >= MAX_OBJECTS))
                return(-1);

/* Additional procedures if object_num was lplayer_object? */



	/* Recycle the object. */
	DBRecycleObject(object_num);


	return(0);
}



int NetHandleSetObjApp(char *arg)
{
	/* Local variables. */
	long object_num;

	long type;
	long imageset;
	long size;

	double x, y, z;
	double heading, pitch;

	double velocity;
	double velocity_heading;

        int throttle;
	long current_frame;

	int status;


	/* arg must be atleast 1 character long. */
	if(strlen(arg) < 1)
		return(-1);

        /*
         *   NET_CMD_SETOBJLOCAPP format is as follows:
         *
         *      object_num,
         *      type, imageset, size,
         *      x, y, z,
         *      heading, pitch,
         *      velocity velocity_heading,
         *      throttle, frame
         */
	sscanf(arg,
"%i\
 %i %i %i\
 %lf %lf %lf\
 %lf %lf\
 %lf %lf\
 %i %i",
		&object_num,

		&type,
		&imageset,
		&size,

		&x,
		&y,
		&z,

		&heading,
		&pitch,

		&velocity,
		&velocity_heading,

		&throttle,
		&current_frame
	);


	/* object_num must be valid. */
	if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
		return(-1);

	/* imageset must be valid. */
	if( (imageset < 0) || (imageset >= MAX_IMAGESETS) )
		return(-1);


	/*
	 *   We don't update local player object location or apperance with
	 *   server data once we are in connection_state 2.
	 *   With the exception that the object is intercepting something.
	 */
        if( (object_num == lplayer_object) &&
            (xsw_object[object_num].intercepting_object > -1)
        )
	{
            xsw_object[object_num].heading = heading;
            xsw_object[object_num].pitch = pitch;
	}
	if( (object_num == lplayer_object) &&
            (connection_state == 2)
	)
	{
	    return(0);
	}
	else
	{
	    connection_state = 2;
	}


	/* Has object_num been created yet? */
	if(xsw_object[object_num].type <= XSW_OBJ_TYPE_GARBAGE)
	{
	    status = DBCreateExplicitObject(object_num, imageset, game_win);
	    if(status != 0)
	    {
		fprintf(stderr, "Error creating object %i.\n", object_num);
		return(-1);
	    }
	}


	/* Set new location and apperance values on object. */
        xsw_object[object_num].type = type;
        xsw_object[object_num].imageset = imageset;
	xsw_object[object_num].size = size;

        xsw_object[object_num].x = x;
        xsw_object[object_num].y = y;
        xsw_object[object_num].z = z;

        xsw_object[object_num].heading = heading;
        xsw_object[object_num].pitch = pitch;

        xsw_object[object_num].velocity = velocity;
        xsw_object[object_num].velocity_heading = velocity_heading;

        xsw_object[object_num].throttle = throttle;
        xsw_object[object_num].animation.current_frame = current_frame;


	/* Refresh the last time it was updated. */
	xsw_object[object_num].last_updated = cur_millitime;


	return(0);
}



int NetHandleSetObjAppS(char *arg)
{
        /* Local variables. */
        int status;

        long object_num;
          
        long type;
        long imageset;
        long size;
        
        double x, y, z;  
        double heading, pitch; 
        
        long current_frame;
 
 
        /* arg must be atleast 1 character long. */
        if(strlen(arg) < 1)
                return(-1);
                
                
        /*
         *   NET_CMD_SETOBJLOCAPPS format is as follows:
         *
         *      object_num,
         *      type, imageset, size,
         *      x, y, z, 
         *      heading, pitch,
         *	frame
         */
        sscanf(arg,
"%i\
 %i %i %i\
 %lf %lf %lf\
 %lf %lf\
 %i",
                &object_num,
                
                &type,
                &imageset,
                &size,
         
                &x,
                &y,
                &z,
         
                &heading,
                &pitch,
                 
                &current_frame
        );
         

        /* See if object_num is valid. */
        if((object_num < 0) || (object_num >= MAX_OBJECTS))
                return(-1);

        /* See if imageset is valid. */
        if((imageset < 0) || (imageset >= MAX_IMAGESETS)) 
                return(-1);
         

        /*
         *   We don't update local player object location or apperance with
         *   server data once we are in connection_state 2.
         *   With the exception that the object is intercepting something.
         */
        if( (object_num == lplayer_object) &&
            (xsw_object[object_num].intercepting_object > -1)
        )
        {
            xsw_object[object_num].heading = heading;
            xsw_object[object_num].pitch = pitch;
        }
        if( (object_num == lplayer_object) &&
            (connection_state == 2)
        )
        {
            return(0);
        }       
        else    
        {       
            connection_state = 2;
        }


        /* Has object_num been created yet? */
        if(xsw_object[object_num].type <= XSW_OBJ_TYPE_GARBAGE)
        {
            status = DBCreateExplicitObject(object_num, imageset, game_win);
            if(status != 0)
            {
                fprintf(stderr, "Error creating object %i.\n", object_num);
                return(-1);
            }
        }

                   
        /* Set new location and apperance values on object. */
        xsw_object[object_num].type = type;
        xsw_object[object_num].imageset = imageset;
        xsw_object[object_num].size = size;

        xsw_object[object_num].x = x;
        xsw_object[object_num].y = y;
        xsw_object[object_num].z = z;
                
        xsw_object[object_num].heading = heading;  
        xsw_object[object_num].pitch = pitch;

        xsw_object[object_num].animation.current_frame = current_frame;
         
         
        /* Refresh the last time it was updated. */
        xsw_object[object_num].last_updated = cur_millitime;
         
         
        return(0);
}



int NetHandleSetObjectValues(char *arg)
{
        /* Local variables. */
        long object_num;
        int status;

	long locked_on;
	long intercepting_object;
	double velocity;
	double velocity_heading;

	int thrust_rev_state;
	double thrust_dir;
	double thrust;
	int throttle;

	double hp;
	double power;
	double antimatter;
	int shield_state;

	int selected_weapon;
	int cloak_state;
	double cloak_strength;
	double visibility;

	int damage_control;


        /*
         * Read the data. Format is as follows:
         *
	 *   object_num,
	 *   locked_on, intercepting_object, velocity, velocity_heading,
	 *   thrust_rev_state, thrust_dir, thrust, throttle, hp, power,
	 *   antimatter, shield_state, selected_weapon, cloak_state,
	 *   cloak_strength, visibility, damage_control
	 *
         */
        sscanf(arg,
"%i\
 %i %i %lf %lf\
 %i %lf %lf %i %lf %lf\
 %lf %i %i %i\
 %lf %lf %i",

                &object_num,

                &locked_on,
                &intercepting_object,
                &velocity,
                &velocity_heading,

                &thrust_rev_state,
                &thrust_dir,
                &thrust,
                &throttle,
                &hp,
                &power,

                &antimatter,
                &shield_state,
                &selected_weapon,
                &cloak_state,

                &cloak_strength,
                &visibility,
                &damage_control
        );
             
                
        /* See if object_num is valid. */
        if((object_num < 0) || (object_num >= MAX_OBJECTS))
                return(-1);

        /* See if imageset is valid. */
/*
        if((imageset < 0) || (imageset >= MAX_IMAGESETS))
                return(-1);
*/


        /* Has object_num been created yet? */
/*
        if(xsw_object[object_num].type <= XSW_OBJ_TYPE_GARBAGE)
        {
            status = DBCreateExplicitObject(object_num, imageset, game_win);
            if(status != 0)
            {
                fprintf(stderr, "Error creating object %i.\n", object_num);
                return(-1);
            }
        }
*/       

	/* Set object values. */
	xsw_object[object_num].locked_on = locked_on;
        xsw_object[object_num].intercepting_object = intercepting_object;

	/* Skip velocity and thrust for lplayer_object. The lplayer_object */
	/* must not be affected by movement settings givin by remote       */
	/* server.                                                         */
        if(object_num != lplayer_object)
	{
            xsw_object[object_num].velocity = velocity;
            xsw_object[object_num].velocity_heading = velocity_heading;
            xsw_object[object_num].thrust_rev_state = thrust_rev_state;
            xsw_object[object_num].thrust_dir = thrust_dir;
            xsw_object[object_num].thrust = thrust;
            xsw_object[object_num].throttle = throttle;
	}
        xsw_object[object_num].hp = hp;
        xsw_object[object_num].power = power;
        xsw_object[object_num].antimatter = antimatter;
        xsw_object[object_num].shield_state = shield_state;
        xsw_object[object_num].selected_weapon = selected_weapon;
        xsw_object[object_num].cloak_state = cloak_state;
        xsw_object[object_num].cloak_strength = cloak_strength;
        xsw_object[object_num].visibility = visibility;
        xsw_object[object_num].damage_control = damage_control;


        /* Refresh the last time it was updated. */
        xsw_object[object_num].last_updated = cur_millitime;


	/* Redraw stats if this was the local object. */
	if(object_num == lplayer_object)
	{
	    DrawStats(player_stats_win);
	    DrawSensorReadout(sensor_readout_win);
	}


	return(0);
}


int NetHandleSetObjectMaximums(char *arg)
{
        /* Local variables. */
        long object_num;
        int status;

	long type;
	long imageset;
	long owner;
	long size;
	double scanner_range;
	double velocity_max;
	double thrust_power;
	double turnrate;
	double hp_max;
	double power_max;
	double power_purity;
	double core_efficency;
	double antimatter_max;
	int total_weapons;
	double visibility;



        /* 
	 * Read the data. Format is as follows:
	 *
	 *   object_num,
	 *   type, imageset, owner, size, scanner_range,
	 *   velocity_max, thrust_power, turnrate, hp_max, power_max,
	 *   power_purity, core_efficency, antimatter_max, total_weapons, visibility
	 */
        sscanf(arg,
"%i\
 %i %i %i %i %lf\
 %lf %lf %lf %lf %lf\
 %lf %lf %lf %i %lf",

                &object_num,

                &type,
                &imageset,
                &owner,
                &size,
                &scanner_range,

                &velocity_max,
                &thrust_power,
                &turnrate,
                &hp_max,
                &power_max,

                &power_purity,
                &core_efficency,
                &antimatter_max,
                &total_weapons,
                &visibility

        );


        /* See if object_num is valid. */
        if((object_num < 0) || (object_num >= MAX_OBJECTS))
                return(-1);

        /* See if imageset is valid. */
        if((imageset < 0) || (imageset >= MAX_IMAGESETS))  
                return(-1);
             

        /* Has object_num been created yet? */
        if(xsw_object[object_num].type <= XSW_OBJ_TYPE_GARBAGE)
        {
            status = DBCreateExplicitObject(object_num, imageset, game_win);
            if(status != 0)
            {
                fprintf(stderr, "Error creating object %i.\n", object_num);
                return(-1);
            }
        }


        /* Set new maximum values on object. */
        xsw_object[object_num].type = type;
        xsw_object[object_num].imageset = imageset;
        xsw_object[object_num].owner = owner;
        xsw_object[object_num].size = size;
        xsw_object[object_num].scanner_range = scanner_range;

        xsw_object[object_num].velocity_max = velocity_max; 
        xsw_object[object_num].thrust_power = thrust_power;
        xsw_object[object_num].turnrate = turnrate;
        xsw_object[object_num].hp_max = hp_max;
        xsw_object[object_num].power_max = power_max;

        xsw_object[object_num].power_purity = power_purity;
        xsw_object[object_num].core_efficency = core_efficency;
        xsw_object[object_num].antimatter_max = antimatter_max;
        xsw_object[object_num].total_weapons = total_weapons;
        xsw_object[object_num].visibility = visibility;


        /* Refresh the last time it was updated. */
        xsw_object[object_num].last_updated = cur_millitime;

	return(0);
}



int NetSendSetWeapon(int selected_weapon)
{
        /* Local variables. */
        char sndbuf[MAX_NET_CMD_SIZE];


	/* Make sure socket is opened. */
	if(swserv_socket < 0)
		return(-1);

        /* Make sure lplayer_object is valid. */
        if((lplayer_object < 0) || (lplayer_object >= MAX_OBJECTS))
                return(-1);


	/* Make sure selected_weapon is valid. */
	if( (selected_weapon < 0) || (selected_weapon >= MAX_WEAPONS) )
		return(-1);

        /*
         *   NET_CMD_SETWEAPON format is as follows:
         *
         *      selected_weapon
         */
        sprintf(sndbuf, "%i %i\n%c",
                NET_CMD_SETWEAPON,
		selected_weapon,
                '\0'
        );
 

        /* Send it. */
        NetSendData(sndbuf);


	return(0);
}



int NetSendSetShields(int shield_state, double shield_frequency)
{
        /* Local variables. */
        char sndbuf[MAX_NET_CMD_SIZE];


        /* Make sure socket is opened. */
        if(swserv_socket < 0)
                return(-1);

	/* Make sure lplayer_object is valid. */
	if((lplayer_object < 0) || (lplayer_object >= MAX_OBJECTS))
		return(-1);


        /*
         *   NET_CMD_SETSHIELDS format is as follows:
         *
	 *	shield_state, shield_frequency
         */
        sprintf(sndbuf, "%i %i %lf\n%c",
                NET_CMD_SETSHIELDS,
		shield_state,
		shield_frequency,
                '\0'
        );
                     

        /* Send it. */
        NetSendData(sndbuf);


	return(0);
}



int NetSendSetDmgCtl()
{
        /* Local variables. */
        char sndbuf[MAX_NET_CMD_SIZE];


        /* Make sure socket is opened. */
        if(swserv_socket < 0)
                return(-1);

        /* Make sure lplayer_object is valid. */
        if((lplayer_object < 0) || (lplayer_object >= MAX_OBJECTS))
                return(-1);
        
                
        /*
         *   NET_CMD_SETDMGCTL format is as follows:
         *   
         */             
        sprintf(sndbuf, "%i\n%c",
                NET_CMD_SETDMGCTL, 
                '\0'
        );
          
          
        /* Send it. */
        NetSendData(sndbuf);
        
        
        return(0);
}



int NetSendSetCloak() 
{       
        /* Local variables. */
        char sndbuf[MAX_NET_CMD_SIZE];


        /* Make sure socket is opened. */
        if(swserv_socket < 0)
                return(-1);

        /* Make sure lplayer_object is valid. */
        if((lplayer_object < 0) || (lplayer_object >= MAX_OBJECTS))
                return(-1);
         
             
        /*              
         *   NET_CMD_SETCLOAK format is as follows:
         *
         */
        sprintf(sndbuf, "%i\n%c",
                NET_CMD_SETCLOAK,
                '\0'
        );


        /* Send it. */
        NetSendData(sndbuf);


        return(0);
}



int NetSendWeaponsLock()
{
        /* Local variables. */
        char sndbuf[MAX_NET_CMD_SIZE];


        /* Make sure socket is opened. */
        if(swserv_socket < 0)
                return(-1);


        /* Make sure lplayer_object is valid. */
        if((lplayer_object < 0) || (lplayer_object >= MAX_OBJECTS))
                return(-1);


        /*
         *   NET_CMD_WEAPONSLOCK format is as follows:
         *
         */
        sprintf(sndbuf, "%i\n%c",
                NET_CMD_WEAPONSLOCK,
                '\0'
        );


        /* Send it. */
        NetSendData(sndbuf);
          
           
        return(0);
}



int NetSendWeaponsUnlock()
{
        /* Local variables. */
        char sndbuf[MAX_NET_CMD_SIZE];


        /* Make sure socket is opened. */
        if(swserv_socket < 0)
                return(-1);
 
 
        /* Make sure lplayer_object is valid. */
        if((lplayer_object < 0) || (lplayer_object >= MAX_OBJECTS))
                return(-1);
 
        
        /*
         *   NET_CMD_WEAPONSUNLOCK format is as follows:
         *
         */
        sprintf(sndbuf, "%i\n%c",
                NET_CMD_WEAPONSUNLOCK,
                '\0'
        );
                
                
        /* Send it. */
        NetSendData(sndbuf);
         
           
        return(0);
}



int NetSendIntercept(char *arg)
{
        /* Local variables. */   
        char sndbuf[MAX_NET_CMD_SIZE];
	char larg[MAX_NET_CMD_SIZE];

          
        /* Make sure socket is opened. */
        if(swserv_socket < 0)
                return(-1);
        
 
        /* Make sure lplayer_object is valid. */
        if((lplayer_object < 0) || (lplayer_object >= MAX_OBJECTS))
                return(-1);

	/* Sanitize arg and copy it to larg. */
	strncpy(larg, StringStripSpaces(arg), MAX_NET_CMD_SIZE);
	larg[MAX_NET_CMD_SIZE - 1] = '\0';

 
        /*
         *   NET_CMD_INTERCEPT format is as follows:
         *
         */
        sprintf(sndbuf, "%i %s\n%c",
                NET_CMD_INTERCEPT,
		larg,
                '\0'
        );


        /* Send it. */
        NetSendData(sndbuf);
 

        return(0);
}



int NetSendFireWeapon()
{
	/* Local variables. */
	char sndbuf[MAX_NET_CMD_SIZE];


        /* Make sure socket is opened. */
        if(swserv_socket < 0)
                return(-1);


        /* Make sure lplayer_object is valid. */
        if((lplayer_object < 0) || (lplayer_object >= MAX_OBJECTS))
                return(-1);
                
        /*
         *   NET_CMD_WEAPONSUNLOCK format is as follows:
         *
         */
        sprintf(sndbuf, "%i\n%c",
                NET_CMD_FIREWEAPON,
                '\0'
        );


	/* Send it. */
        NetSendData(sndbuf);


	return(0);
}



int NetHandleSetName(char *arg)
{
	/* Local variables. */
	char name[MAX_NAME_LENGTH];
	char stringa[MAX_NET_CMD_SIZE];
	long object_num;
	char *strptr;


	/* arg must be atleast 1 character long. */
	if(strlen(arg) < 1)
		return(-1);


        /*
         * NET_CMD_SETNAME format is as follows:
         *
         *   object_num, name
         */

	/* Must have a ';' character in arg. */
	if( (strptr = strchr(arg, ';')) == NULL )
		return(-1);

	/* Get object_num. */
	strncpy(stringa, arg, MAX_NET_CMD_SIZE);
	strptr = strchr(stringa, ';');
	*strptr = '\0';
	object_num = atol(stringa);

	/* Get name. */
	strptr = strchr(arg, ';');
	strptr += 1;
	strncpy(name, strptr, MAX_NAME_LENGTH);


	/* Make sure object_num is valid. */
	if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
		return(-1);

	/* Set new name for object. */
	strncpy(xsw_object[object_num].name, name, MAX_NAME_LENGTH);
	xsw_object[object_num].name[MAX_NAME_LENGTH - 1] = '\0';


	return(0);
}



int NethandleSetWeaponValues(char *arg)
{
	/* Local variables. */
	long object_num;
	int weapon_num;

	long type;
	long emission_type;
	long amount;
	long max;

	double power;
	double range;
	double create_power;
                        
	long delay;
	long last_used;


        /* arg must be atleast 1 character long. */
        if(strlen(arg) < 1)   
                return(-1);
        
        
        /*
         *   NET_CMD_SETWEAPONVAL format is as follows:
         *
         *      object_num, weapon_num,
         *      type, emission_type, amount, max,
         *      power, range, create_power
         *      delay, last_used
         */
        sscanf(arg,
"%i %i\
 %i %i %i %i\
 %lf %lf %lf\
 %i %i",
                &object_num,
                &weapon_num,

		&type,
		&emission_type,
		&amount,
		&max,

		&power,
		&range,
		&create_power,
                
		&delay,
		&last_used
        );


        /* Make sure object_num is valid. */
        if( (object_num < 0) || (object_num >= MAX_OBJECTS) )
                return(-1);


	/* Make sure weapon_num is valid. */
	if( (weapon_num < 0) || (weapon_num >= MAX_WEAPONS) )
                return(-1);

	/* Set weapon values. */
	xsw_object[object_num].weapons[weapon_num].type = type;
        xsw_object[object_num].weapons[weapon_num].emission_type = emission_type;
        xsw_object[object_num].weapons[weapon_num].amount = amount;
        xsw_object[object_num].weapons[weapon_num].max = max;
        xsw_object[object_num].weapons[weapon_num].power = power;
        xsw_object[object_num].weapons[weapon_num].range = range;
        xsw_object[object_num].weapons[weapon_num].create_power = create_power;
        xsw_object[object_num].weapons[weapon_num].delay = delay;

/*
 * Last used is handled locally since cur_millitime on client and server
 * may differ. last_used will be set by the client's EventHandleWeaponFire()
 * function.
 */
/*        xsw_object[object_num].weapons[weapon_num].last_used = last_used;
 */


	return(0);
}



int NetHandleRecv()
{
        /* Local variables. */
        char recbuf[MAX_NET_OVERFLOW_SIZE];
	long recbuf_cnt;
        char workbuf[MAX_NET_CMD_SIZE];
	long workbuf_cnt;

	char arg[MAX_NET_CMD_SIZE];
	int command;

        long bytes_read;
	char *strptr;
	struct timeval timeout;
	fd_set readfds;


	/* Poll socket to see if there is any incoming data to be read. */
	errno = 0;
        timeout.tv_sec = 0;
        timeout.tv_usec = 0;
	FD_ZERO(&readfds);
	FD_SET(swserv_socket, &readfds);

	select(swserv_socket + 1, &readfds, NULL, NULL, &timeout);
	if( !(FD_ISSET(swserv_socket, &readfds)) )
		return(0);


/* Check for invalid socket? */


        /* Get incoming data. */
	errno = 0;
        bytes_read = recv(swserv_socket, recbuf, MAX_NET_OVERFLOW_SIZE, 0);

	/* Did we get any data? */
        if(errno == EAGAIN)
	{
	    errno = 0;
            return(0);
	}

	/* Is bytes_read cannot be greater than MAX_NET_OVERFLOW_SIZE? */
	if(bytes_read > MAX_NET_OVERFLOW_SIZE)
		bytes_read = MAX_NET_OVERFLOW_SIZE;


	/* Set null character at the end of recbuf just to be safe. */
	recbuf[MAX_NET_OVERFLOW_SIZE - 1] = '\0';


if(print_event_bool)
{       
fprintf(stderr, "Recieved: %i bytes.\n", bytes_read); 
}           


	/* Set buffer counts to 0. */
	workbuf_cnt = 0;
	recbuf_cnt = 0;


	/* *************************************************************** */
	/* ***                  Begin Parsing recbuf                   *** */
        while(recbuf_cnt < bytes_read)
        {
            /* Fetch data for workbuf. */
            workbuf_cnt = 0;
            while(workbuf_cnt < MAX_NET_CMD_SIZE)
            {   
                if(recbuf_cnt >= bytes_read)
                {
                    workbuf[workbuf_cnt] = '\0';
                    break;
                }
                
                if(recbuf[recbuf_cnt] == '\0')
                {
                    recbuf_cnt++;
                    continue;
                }
        
                if( (recbuf[recbuf_cnt] == '\n') ||
		    (recbuf[recbuf_cnt] == '\r')
		)
                {   
                    workbuf[workbuf_cnt] = '\0';
                    recbuf_cnt++;
                    break;
                }

                workbuf[workbuf_cnt] = recbuf[recbuf_cnt];
        
                recbuf_cnt++;
                workbuf_cnt++;
            }    
            
            if(workbuf[0] == '\0')
                break;

if(print_event_bool)
{                   
fprintf(stderr, "Handling: `%s'\n", workbuf);
}



	    /* Get command command. */
	    if( (command = StringGetNetCommand(workbuf)) < 0)
	    {
		continue;
	    }

	    /* Get argument arg. */
	    strncpy(arg, StringGetNetArgument(workbuf), MAX_NET_CMD_SIZE);
	    arg[MAX_NET_CMD_SIZE - 1] = '\0';


            /* See which command to perform. */
            switch(command)
	    {
	        case NET_CMD_COMMENT:
		    /* Handle comment response. */
		    break;

	        case NET_CMD_LOGIN:
		    MesgWinAddMessage(arg, normal_text_pixel);
		    NetHandleLogin();
		    break;

                case NET_CMD_LOGOUT:
                    close(swserv_socket);
                    swserv_socket = -1;
		    lplayer_object = -1;
                    connection_state = 0;
		    MesgWinAddMessage("Disconnected.", normal_text_pixel);
                    break;

                case NET_CMD_WHOAMI:
		    NetHandleWhoAmI(arg);
                    break;

	        case NET_CMD_ECHO:
		    MesgWinAddMessage(arg, normal_text_pixel);
		    break;

	        case NET_CMD_REFRESH:
		    /* Client dosen't handle this. */
		    break;

	        case NET_CMD_SETINTERVAL:
		    /* Client dosen't handle this. */
                    break;

		case NET_CMD_EXEC:
		    /* Client dosen't handle this. */
		    break;

	        case NET_CMD_GLOBALMESG:
		    MesgWinAddMessage(arg, normal_text_pixel);
		    break;

	        case NET_CMD_GROUPMESG:
                    MesgWinAddMessage(arg, normal_text_pixel);
                    break;

                case NET_CMD_DIRECTMESG:
                    MesgWinAddMessage(arg, normal_text_pixel);
                    break;

	        case NET_CMD_CREATEOBJECT:
		    /* Handle object create. */
		    break;

	        case NET_CMD_RECYCLEOBJECT:
		    NetHandleRecycleObject(arg);
		    break;

	        case NET_CMD_SETOBJLOCAPP:
		    NetHandleSetObjApp(arg);
                    break;

	        case NET_CMD_SETOBJVALUES:
		    NetHandleSetObjectValues(arg);
                    break;

                case NET_CMD_SETOBJMAXIMUMS:
                    NetHandleSetObjectMaximums(arg);
                    break;

	        case NET_CMD_CREATEANIMATED:
		    /* Not supported yet. */
		    break;

	        case NET_CMD_SETWEAPON:
                    /* Client dosen't handle this, see NetHandleSetObjectValues(). */
                    break;

	        case NET_CMD_SETSHIELDS:
                    /* Client dosen't handle this, see NetHandleSetObjectValues(). */
                    break;

	        case NET_CMD_SETDMGCTL:
                    /* Client dosen't handle this, see NetHandleSetObjectValues(). */
                    break;

	        case NET_CMD_SETCLOAK:
                    /* Client dosen't handle this, see NetHandleSetObjectValues(). */
                    break;

	        case NET_CMD_WEAPONSLOCK:
                    /* Client dosen't handle this, see NetHandleSetObjectValues(). */
                    break;

	        case NET_CMD_WEAPONSUNLOCK:
                    /* Client dosen't handle this, see NetHandleSetObjectValues(). */
		    break;

	        case NET_CMD_INTERCEPT:
                    /* Client dosen't handle this, see NetHandleSetObjectValues(). */
                    break;

	        case NET_CMD_FIREWEAPON:
                    /* Client dosen't handle this. */
		    break;

	        case NET_CMD_SETNAME:
		    NetHandleSetName(arg);
		    break;

	        case NET_CMD_SETWEAPONVAL:
		    NethandleSetWeaponValues(arg);
		    break;

	        case NET_CMD_SETOBJLOCAPPS:
	            NetHandleSetObjAppS(arg);
		    break;

	    }

	}

        return(0);
}
