/* **********************************************************************
 *
 *	loadstatus.c	Loading Status Window
 *	-------------------------------------
 *	Functions:
 *
 *	pid_t LoadStatusWindowExec(char *message, long process_items)
 *	int LoadStatusWindowUpdate()
 *
 *	---
 *
 *	Status Window is a window that is created before
 *	toplevel to display loading information and etc before
 *	the rest of the program finishes loading so the user
 *	knows what is being done and what the delay is.
 *
 *	LoadStatusWindowExec() initializes and runs one instance
 *	of a load status window, displaying a progress bar.
 *	Returns the pid of the child process in which it forks off.
 *	The child process dies when a SIGKILL is recieved.
 *
 *	LoadStatusWindowUpdate() increases the number of items
 *	processed and increments the processed_items count.
 *	Each time a signal SIGUSR1 is recieved, the number of items
 *	processed is increased.
 *	Returns non-zero in error.
 *
 */

#include "../include/xsw.h"  



/* Images. */
#include "images/xsw_icon.xpm"
#include "images/startup_bkg.xpm"



int LSWinInit()
{
        /* Local variables. */
        int x, y, z;
        double dx, dy, dz;
        char stringa[256];
        pid_t the_child_pid;


        /* Set size of loadstatus_win window. */
        int loadstatus_win_width=320;
        int loadstatus_win_height=240;


	/* Check if display is connected. */
	if(display == NULL)
	{
	    fprintf(stderr, "LSWinInit(): Display not connected.\n");
	    return(-1);
	}



        /* Get positions for x and y to center st_toplevel in center */
        /* of display.                                               */
        x = DisplayWidth(display, scr_num);
        y = DisplayHeight(display, scr_num);
        if(x < loadstatus_win_width)
        {
                x = 0;
        }
        else
        {
                x = ( x - loadstatus_win_width ) / 2;
        }
        if(y < loadstatus_win_height)
        {
                y = 0;
        }
        else
        {
                y = ( y - loadstatus_win_height ) / 2;
        }


        /* Create the loadstatus_win. */
        loadstatus_win = XCreateWindow(
           display,
           DefaultRootWindow(display),
           x,              /* x (obsolete) */
           y,              /* y (obsolete) */
           loadstatus_win_width,      /* width. */
           loadstatus_win_height,     /* height. */
           0,              /* border_width. */
           depth,          /* depth from parent. */
           InputOutput,    /* IO option. */
           visual,
           WIN_ATTR_MASK,
           &attributes
        ); 
        
        loadstatus_pbar_win = XCreateSimpleWindow(display,
           loadstatus_win, 16, 156, 288, 18, 0, white_pix, black_pix);



	/* Load just the pixmaps needed. */ 
        xsw_icon_pixmap = readpixmapdata(xsw_icon_xpm);
        startup_bkg_pixmap = readpixmapdata(startup_bkg_xpm);
        

        /* Set size hints for window manager. */
        sizehints.flags=USPosition|PSize|PMinSize|PMaxSize;
        sizehints.x=x;
        sizehints.y=y;
        sizehints.min_width=loadstatus_win_width;
        sizehints.min_height=loadstatus_win_height;
        sizehints.max_width=loadstatus_win_width;   
        sizehints.max_height=loadstatus_win_height;
        sprintf(stringa, "%s %s", XSW_NAME, XSW_VERSION);
        XSetStandardProperties(display,
           loadstatus_win,
           stringa,             /* Window title name. */
           XSW_NAME,            /* Iconified name. */
           xsw_icon_pixmap,     /* Icon pixmap. */
           0,0,                 /* Command line arguments. */
           &sizehints           /* Sizehints. */
        );
 
           
        /* Set XEvent masks. */
        XSelectInput(display, loadstatus_win, toplevel_eventmask);
        XSelectInput(display, loadstatus_pbar_win, toplevel_eventmask);
                
                
        /* Define cursors. */
        XDefineCursor(display, loadstatus_win,
                XCreateFontCursor(display, XC_watch)
        );      

         
        /* Map loadstatus_win. */
        XMapWindow(display, loadstatus_win);
        XMapSubwindows(display, loadstatus_win);
 
         
        /* Set background pixmap for loadstatus_win. */
        XSetWindowBackgroundPixmap(display, loadstatus_win, startup_bkg_pixmap);
 

	return(0);
}



int LSWinDrawProgressBar(long items, long max_items, char *message)
{
	/* Local variables. */
	double dx, dy, dz;
	char stringa[256];


        /* Check if display is connected. */
        if(display == NULL)
        {
            fprintf(stderr, "LSWinDrawProgressBar(): Display not connected.\n");
            return(-1);
        }


	/* items must be atleast 0. */
	if(items < 0)
		items = 0;

	/* max_items must be atleast 1. */
	if(max_items < 1)
		max_items = 1;


	/* Redraw loadstatus_win. */
	sprintf(stringa, "Version: %s", XSW_VERSION);
        XSetForeground(display, the_GC, bold_text_pixel);
	XClearWindow(display, loadstatus_win);
        XDrawString(display, loadstatus_win, the_GC,
            215, 230, stringa, strlen(stringa));


	/* Redraw loadstatus_pbar_win. */
        XClearWindow(display, loadstatus_pbar_win);

        dx = items;
        dy = max_items; 
        dz = (dx / dy) * 288;

        XSetForeground(display, the_GC, normal_text_pixel);
        XFillRectangle(display, loadstatus_pbar_win, the_GC,
            0,
            0,
            dz,
            17
        );
        
        XSetForeground(display, the_GC, light_grey_pixel);
        XDrawRectangle(display, loadstatus_pbar_win, the_GC,
            0, 0, 287, 17);

        XSetForeground(display, the_GC, dark_grey_pixel);
        XDrawString(display, loadstatus_pbar_win, the_GC,
            16, 13, message, strlen(message));


	return(0);
}



void LSWinDestroy()
{
	XDestroyWindow(display, loadstatus_pbar_win);
	loadstatus_pbar_win = -1;

        XDestroyWindow(display, loadstatus_win);
        loadstatus_win = -1;

	return;
}
