/* **************************************************************************
 *
 *	images.c	Loads/Initializes Images
 *	----------------------------------------
 *	Functions:
 *
 *	int IMGLoadBasic()
 *	void IMGFreeAllImages()
 *	int IMGLoadSpecified(long imageset_code, long frames)
 *
 *	---
 *
 *	Handles the loading and freeing of imagesets.
 *
 *	IMGLoadBasic() loads basic images.
 *	Returns zero on success.
 *
 *	IMGFreeAllImages() frees all loaded images and imagesets.
 *	No return value.
 *
 */

#include "../include/xsw.h"  



/* Basic images. */

#include "images/xsw_title_sm.xpm"
#include "images/misc/unknown1.xpm"



/* Color closeness. */
#ifndef COLORCLOSENESS
   #define COLORCLOSENESS 40000
#endif  


int IMGLoadBasic()
{
        /* Local variables. */
        XpmAttributes xpmattr;  /* struct for xpm attributes. */
        XpmInfo xpminfo;        /* struct for xpm info. */
        XpmImage xpmimg;        /* struct for xpm image. */
        int status;
	Pixmap pixmapmask;
	char filename[1024];


        /* Set XPM attributes. */
        xpmattr.closeness = COLORCLOSENESS;
        xpmattr.valuemask = XpmSize | XpmCloseness;
   

	/* ********************************************************** */
	/* ***        Begin Loading Basic Images From Data        *** */


        /* XSW Title, small. */
        status = XpmCreatePixmapFromData(
                display,
                DefaultRootWindow(display),
                xsw_title_sm_xpm,
                &xsw_title_sm_pixmap,
                &pixmapmask,
                &xpmattr
        );
        if(status != XpmSuccess)
        { 
            fprintf(stderr, "IMGLoadBasic() Error reading xpm from data.\n");
        }
        /* Tell load status process we loaded an image. */
        LSWinDrawProgressBar(1, 3, "Loading basic images...");


        /* Bridge. */
        sprintf(filename, "%s/bridge.xpm", XSW_IMAGES_DIR);
        status = XpmReadFileToPixmap(
        	display,
        	DefaultRootWindow(display),
                filename,  
                &bridge_pixmap,
                &bridge_mask_pixmap,
                &xpmattr
        );
        if(status != XpmSuccess)
        { 
            fprintf(stderr, "IMGLoadBasic() Error reading: `%s'\n",
		filename);
        }
        /* Tell load status process we loaded an image. */
        LSWinDrawProgressBar(2, 3, "Loading bridge images...");


	/* Unknown imageset is considered basic and must be loaded here. */
        status = XpmCreatePixmapFromData(
                display,  
                DefaultRootWindow(display),
                unknown1_xpm,
                &xsw_imageset[IMAGESET_CODE_UNKNOWN].image[0],
                &xsw_imageset[IMAGESET_CODE_UNKNOWN].shape_mask[0],
                &xpmattr
        );
        xsw_imageset[IMAGESET_CODE_UNKNOWN].width = 48;  
        xsw_imageset[IMAGESET_CODE_UNKNOWN].height = 64;
	xsw_imageset[IMAGESET_CODE_UNKNOWN].fwidth = 48;
	xsw_imageset[IMAGESET_CODE_UNKNOWN].fheight = 64;
	/* Mark as loaded. */
	xsw_imageset[IMAGESET_CODE_UNKNOWN].load_state = 1;
        if(status != XpmSuccess)
        { 
            fprintf(stderr, "IMGLoadBasic() Error reading xpm from data.\n");
        }
        /* Tell load status process we loaded an image. */
        LSWinDrawProgressBar(3, 3, "Loading misc images...");



	return(0);
}



int IMGLoadSpecified(long imageset_code, long frames)
{
	/* Local variables. */
	long lx, ly, lz;
	char stringa[1024];
        XpmAttributes xpmattr;  /* struct for xpm attributes. */
        XpmInfo xpminfo;        /* struct for xpm info. */
        XpmImage xpmimg;        /* struct for xpm image. */
	int status;
	char filename[1024];


	/* Make sure imageset_code is valid. */
	if( (imageset_code < 0) || (imageset_code >= MAX_IMAGESETS) )
	{
		fprintf(stderr,
                 "IMGLoadSpecified(): Error: Imageset code %i out of range.\n",
		    imageset_code
		);
		return(-1);
	}

	/* Make sure frames is valid. */
	if( (frames < 1) || (frames > MAX_IMAGESET_FRAMES) )
	{
                fprintf(stderr,
                 "IMGLoadSpecified(): Error: total frames %i out of range.\n",
                     frames
                );
                return(-1);
	}


        /* Set XPM attributes. */
        xpmattr.closeness = COLORCLOSENESS;
        xpmattr.valuemask = XpmSize | XpmCloseness;


	/* Print image loading message. */
	sprintf(stringa, "Loading imageset %i...", imageset_code);
	MesgWinAddMessage(stringa, dark_grey_pixel);


	/* See which imageset to load. */
	switch(imageset_code)
	{
	    /* Unknown */
	    case IMAGESET_CODE_UNKNOWN:
		sprintf(filename, "%s/misc/unknown1.xpm", XSW_IMAGES_DIR);
		status = XpmReadFileToPixmap(display,
		    DefaultRootWindow(display),
		    filename,
		    &xsw_imageset[imageset_code].image[0],
		    &xsw_imageset[imageset_code].shape_mask[0],
		    &xpmattr
		);
		if(status != XpmSuccess)
		{
		    fprintf(stderr, "%s: Loading error.\n", filename);
		}
                xsw_imageset[imageset_code].width = 48;
                xsw_imageset[imageset_code].height = 64;
                xsw_imageset[imageset_code].fwidth = 48;
                xsw_imageset[imageset_code].fheight = 64;
		break;

            /* Buoy */
            case IMAGESET_CODE_BUOY:
                sprintf(filename, "%s/buoy/buoy01.xpm", XSW_IMAGES_DIR);
                status = XpmReadFileToPixmap(display,
                    DefaultRootWindow(display),
                    filename,
                    &xsw_imageset[imageset_code].image[0],
                    &xsw_imageset[imageset_code].shape_mask[0],
                    &xpmattr
                );
                if(status != XpmSuccess)
                {
                    fprintf(stderr, "%s: Loading error.\n", filename);
                }
                xsw_imageset[imageset_code].width = 32;
                xsw_imageset[imageset_code].height = 32;
                xsw_imageset[imageset_code].fwidth = 32;
                xsw_imageset[imageset_code].fheight = 32;
                break;

            /* Hubble Space Telescope */
            case IMAGESET_CODE_HUBBLE:
                sprintf(filename, "%s/misc/hst01.xpm", XSW_IMAGES_DIR);
                status = XpmReadFileToPixmap(display,
                    DefaultRootWindow(display),
                    filename,
                    &xsw_imageset[imageset_code].image[0],
                    &xsw_imageset[imageset_code].shape_mask[0],
                    &xpmattr 
                );
                xsw_imageset[imageset_code].width = 200;
                xsw_imageset[imageset_code].height = 150;
                xsw_imageset[imageset_code].fwidth = 200;
                xsw_imageset[imageset_code].fheight = 150;
                if(status != XpmSuccess)
                {
                    fprintf(stderr, "%s: Loading error.\n", filename);
                }
                break;

            /* Weapon Pulse Phazer Green */
            case IMAGESET_CODE_WEP_PP_G:
                sprintf(filename, "%s/weapons/pp_g_01.xpm", XSW_IMAGES_DIR);
                status = XpmReadFileToPixmap(display,
                    DefaultRootWindow(display),
                    filename,
                    &xsw_imageset[imageset_code].image[0],
                    &xsw_imageset[imageset_code].shape_mask[0],
                    &xpmattr
                );
                xsw_imageset[imageset_code].width = 16;
                xsw_imageset[imageset_code].height = 16;
                xsw_imageset[imageset_code].fwidth = 16;
                xsw_imageset[imageset_code].fheight = 16;
                if(status != XpmSuccess)
                {
                    fprintf(stderr, "%s: Loading error.\n", filename);
                }
                break;

            /* Weapon Pulse Phazer Yellow */
            case IMAGESET_CODE_WEP_PP_Y:
                sprintf(filename, "%s/weapons/pp_y_01.xpm", XSW_IMAGES_DIR);
                status = XpmReadFileToPixmap(display,
                    DefaultRootWindow(display),
                    filename,
                    &xsw_imageset[imageset_code].image[0],
                    &xsw_imageset[imageset_code].shape_mask[0],
                    &xpmattr 
                );
                xsw_imageset[imageset_code].width = 16;
                xsw_imageset[imageset_code].height = 16;
                xsw_imageset[imageset_code].fwidth = 16;
                xsw_imageset[imageset_code].fheight = 16;
                if(status != XpmSuccess)
                {
                    fprintf(stderr, "%s: Loading error.\n", filename);
                }
                break;

            /* Weapon Photon Torpedo Red */
            case IMAGESET_CODE_WEP_PT_R:
                sprintf(filename, "%s/weapons/pt_r_01.xpm", XSW_IMAGES_DIR);
                status = XpmReadFileToPixmap(display,
                    DefaultRootWindow(display),
                    filename,
                    &xsw_imageset[imageset_code].image[0],
                    &xsw_imageset[imageset_code].shape_mask[0],
                    &xpmattr 
                );
                xsw_imageset[imageset_code].width = 16;
                xsw_imageset[imageset_code].height = 16;
                xsw_imageset[imageset_code].fwidth = 16;
                xsw_imageset[imageset_code].fheight = 16;
                if(status != XpmSuccess)
                {
                    fprintf(stderr, "%s: Loading error.\n", filename);
                }
                break;


	    /* Weapon Stream Yellow */
	    case IMAGESET_CODE_WEP_STREAM_Y:
		/* Stream weapons have no images. */
                xsw_imageset[imageset_code].width = 16;
                xsw_imageset[imageset_code].height = 16;
                xsw_imageset[imageset_code].fwidth = 16;
                xsw_imageset[imageset_code].fheight = 16;
                break;    


            /* Weapon Stream Green */
            case IMAGESET_CODE_WEP_STREAM_G:
                /* Stream weapons have no images. */    
                xsw_imageset[imageset_code].width = 16; 
                xsw_imageset[imageset_code].height = 16;  
                xsw_imageset[imageset_code].fwidth = 16; 
                xsw_imageset[imageset_code].fheight = 16;
                break;


            /* Explosion 1 (Big) */
            case IMAGESET_CODE_EXPLOSION1:
		for(lx = 0, ly = 1; lx < 4; lx++, ly++)
		{
		    if(ly < 10)
		    {
                        sprintf(filename, "%s/explosions/expl_0%i.xpm",
                            XSW_IMAGES_DIR, ly
                        );
		    }
                    else
                    {
                        sprintf(filename, "%s/explosions/expl_%i.xpm",
                            XSW_IMAGES_DIR, ly
                        );
                    }
                    status = XpmReadFileToPixmap(display,
                        DefaultRootWindow(display),
                        filename,
                        &xsw_imageset[imageset_code].image[lx],
                        &xsw_imageset[imageset_code].shape_mask[lx],
                        &xpmattr
                    );
                    if(status != XpmSuccess)
                    {
                        fprintf(stderr, "%s: Loading error.\n", filename);
                    }

		}
                xsw_imageset[imageset_code].width = 200;
                xsw_imageset[imageset_code].height = 150;
                xsw_imageset[imageset_code].fwidth = 200;
                xsw_imageset[imageset_code].fheight = 150;
                break;

            /* Explosion 2 (Small) */
            case IMAGESET_CODE_EXPLOSION2:
/* Using explosion 1's frame 1 for now... */
                sprintf(filename, "%s/explosions/expl_01.xpm", XSW_IMAGES_DIR);
                status = XpmReadFileToPixmap(display,
                    DefaultRootWindow(display),
                    filename,
                    &xsw_imageset[imageset_code].image[0],
                    &xsw_imageset[imageset_code].shape_mask[0],
                    &xpmattr
                );
                if(status != XpmSuccess)
                {
                    fprintf(stderr, "%s: Loading error.\n", filename);
                }
                xsw_imageset[imageset_code].width = 200;
                xsw_imageset[imageset_code].height = 150;
                xsw_imageset[imageset_code].fwidth = 200;
                xsw_imageset[imageset_code].fheight = 150;
		break;

            /* Maqui Raider */
            case IMAGESET_CODE_MRAD:
                for(lx = 0, ly = 1; lx < 16; lx++, ly++)
                {
                    if(ly < 10)
                    {
                        sprintf(filename, "%s/mrad/mrad0%i.xpm",
                            XSW_IMAGES_DIR, ly
                        );
                    } 
                    else
                    {
                        sprintf(filename, "%s/mrad/mrad%i.xpm",
                            XSW_IMAGES_DIR, ly
                        );
                    }
                    status = XpmReadFileToPixmap(display,
                        DefaultRootWindow(display),
                        filename,
                        &xsw_imageset[imageset_code].image[lx],
                        &xsw_imageset[imageset_code].shape_mask[lx],
                        &xpmattr
                    );
                    if(status != XpmSuccess)
                    {
                        fprintf(stderr, "%s: Loading error.\n", filename);
                    }
                    
                }
                xsw_imageset[imageset_code].width = 200;
                xsw_imageset[imageset_code].height = 150;
                xsw_imageset[imageset_code].fwidth = 200;
                xsw_imageset[imageset_code].fheight = 150;
                break;

            /* Klingon Bird of Prey */
            case IMAGESET_CODE_KBOP:
                for(lx = 0, ly = 1; lx < 16; lx++, ly++)
                {
                    if(ly < 10)
                    {
                        sprintf(filename, "%s/kbop/kbop0%i.xpm",
                            XSW_IMAGES_DIR, ly
                        );
                    }
                    else
                    {
                        sprintf(filename, "%s/kbop/kbop%i.xpm",
                            XSW_IMAGES_DIR, ly
                        );
                    }
                    status = XpmReadFileToPixmap(display,
                        DefaultRootWindow(display),
                        filename,
                        &xsw_imageset[imageset_code].image[lx],
                        &xsw_imageset[imageset_code].shape_mask[lx],
                        &xpmattr
                    );
                    if(status != XpmSuccess)
                    {
                        fprintf(stderr, "%s: Loading error.\n", filename);
                    }
                      
                }
                xsw_imageset[imageset_code].width = 200;
                xsw_imageset[imageset_code].height = 150;
                xsw_imageset[imageset_code].fwidth = 200;
                xsw_imageset[imageset_code].fheight = 150;
                break;

            /* Intrepid Class */
            case IMAGESET_CODE_INTR:
                for(lx = 0, ly = 1; lx < 16; lx++, ly++)
                {    
                    if(ly < 10)
                    {
                        sprintf(filename, "%s/intr/intr0%i.xpm",
                            XSW_IMAGES_DIR, ly
                        );
                    }
                    else
                    {
                        sprintf(filename, "%s/intr/intr%i.xpm",
                            XSW_IMAGES_DIR, ly
                        );
                    }
                    status = XpmReadFileToPixmap(display,
                        DefaultRootWindow(display),
                        filename,
                        &xsw_imageset[imageset_code].image[lx],
                        &xsw_imageset[imageset_code].shape_mask[lx],
                        &xpmattr
                    );
                    if(status != XpmSuccess)
                    {
                        fprintf(stderr, "%s: Loading error.\n", filename);
                    }
                    
                }
                xsw_imageset[imageset_code].width = 200;
                xsw_imageset[imageset_code].height = 150;
                xsw_imageset[imageset_code].fwidth = 200;
                xsw_imageset[imageset_code].fheight = 150;
                break;


	    /* Klingon Vortcha Class */
            case IMAGESET_CODE_KVOR:
                for(lx = 0, ly = 1; lx < 16; lx++, ly++)
                {     
                    if(ly < 10)
                    {
                        sprintf(filename, "%s/kvor/kvor0%i.xpm",
                            XSW_IMAGES_DIR, ly
                        );
                    }
                    else
                    {
                       sprintf(filename, "%s/kvor/kvor%i.xpm",
                            XSW_IMAGES_DIR, ly
                        );
                    }
                    status = XpmReadFileToPixmap(display,
                        DefaultRootWindow(display),
                        filename,
                        &xsw_imageset[imageset_code].image[lx],
                        &xsw_imageset[imageset_code].shape_mask[lx],
                        &xpmattr
                    );
                    if(status != XpmSuccess)
                    {
                        fprintf(stderr, "%s: Loading error.\n", filename);
                    }
                        
                }
                xsw_imageset[imageset_code].width = 200;
                xsw_imageset[imageset_code].height = 150;
                xsw_imageset[imageset_code].fwidth = 200;
                xsw_imageset[imageset_code].fheight = 150;
                break;


	    /* Station - Cardassian */
	    case IMAGESET_CODE_STCAR:
                for(lx = 0, ly = 1; lx < 1; lx++, ly++)
                {
                    if(ly < 10)
		    {
                        sprintf(filename, "%s/stcar/stcar0%i.xpm",
                            XSW_IMAGES_DIR, ly
                        );
                    }
                    else
                    {
                        sprintf(filename, "%s/stcar/stcar%i.xpm",
                            XSW_IMAGES_DIR, ly
                        );
                    }
                    status = XpmReadFileToPixmap(display,
                        DefaultRootWindow(display),
                        filename,
                        &xsw_imageset[imageset_code].image[lx],
                        &xsw_imageset[imageset_code].shape_mask[lx],
                        &xpmattr
                    );
                    if(status != XpmSuccess)
                    {
                        fprintf(stderr, "%s: Loading error.\n", filename);
                    }
                          
                }
                xsw_imageset[imageset_code].width = 300;
                xsw_imageset[imageset_code].height = 225;
                xsw_imageset[imageset_code].fwidth = 300; 
                xsw_imageset[imageset_code].fheight = 225;
                break;



	    default:
		fprintf(stderr, "IMGLoadSpecified(): Unknown imageset code %i.\n",
			imageset_code);
		break;
	}



	/* Mark this imageset as loaded. */
	xsw_imageset[imageset_code].load_state = 1;


	return(0);
}



void IMGFreeAllImages()
{
	/* Local variables. */
	long x, y, z;


	/* XSW Icon */
	XFreePixmap(display, xsw_icon_pixmap);

        /* Startup Background */
        XFreePixmap(display, startup_bkg_pixmap);

	/* Small XSW Title */
        XFreePixmap(display, xsw_title_sm_pixmap);

	/* Bridge. */
        XFreePixmap(display, bridge_pixmap);
	XFreePixmap(display, bridge_mask_pixmap);


	/* Unknown */
	if(xsw_imageset[IMAGESET_CODE_UNKNOWN].load_state == 1)
	{
         XFreePixmap(display, xsw_imageset[IMAGESET_CODE_UNKNOWN].image[0]);
         XFreePixmap(display, xsw_imageset[IMAGESET_CODE_UNKNOWN].shape_mask[0]);
	}

	/* Buoy */
        if(xsw_imageset[IMAGESET_CODE_BUOY].load_state == 1)   
        {
	 XFreePixmap(display, xsw_imageset[IMAGESET_CODE_BUOY].image[0]);
	 XFreePixmap(display, xsw_imageset[IMAGESET_CODE_BUOY].shape_mask[0]);
	}

	/* Hubble Space Telescope */
        if(xsw_imageset[IMAGESET_CODE_HUBBLE].load_state == 1)   
        {
	 XFreePixmap(display, xsw_imageset[IMAGESET_CODE_HUBBLE].image[0]);
         XFreePixmap(display, xsw_imageset[IMAGESET_CODE_HUBBLE].shape_mask[0]);
	}

	/* Weapons */
        if(xsw_imageset[IMAGESET_CODE_WEP_PP_G].load_state == 1)   
        {
         XFreePixmap(display, xsw_imageset[IMAGESET_CODE_WEP_PP_G].image[0]);
         XFreePixmap(display, xsw_imageset[IMAGESET_CODE_WEP_PP_G].shape_mask[0]);
	}

        if(xsw_imageset[IMAGESET_CODE_WEP_PP_Y].load_state == 1)   
        {
         XFreePixmap(display, xsw_imageset[IMAGESET_CODE_WEP_PP_Y].image[0]);
         XFreePixmap(display, xsw_imageset[IMAGESET_CODE_WEP_PP_Y].shape_mask[0]);
	}

        if(xsw_imageset[IMAGESET_CODE_WEP_PT_R].load_state == 1)   
        {
         XFreePixmap(display, xsw_imageset[IMAGESET_CODE_WEP_PT_R].image[0]);
         XFreePixmap(display, xsw_imageset[IMAGESET_CODE_WEP_PT_R].shape_mask[0]);
	}

	/* Explosion 1 */
        if(xsw_imageset[IMAGESET_CODE_EXPLOSION1].load_state == 1)   
        {
	 for(x = 0; x < 4; x++)
	 {
	   XFreePixmap(display,
		xsw_imageset[IMAGESET_CODE_EXPLOSION1].image[x]);
	   XFreePixmap(display,
		xsw_imageset[IMAGESET_CODE_EXPLOSION1].shape_mask[x]);
	 }
	}

        /* Maqui Raider */
        if(xsw_imageset[IMAGESET_CODE_MRAD].load_state == 1)   
        {
         for(x = 0; x < 16; x++)
         {
	   XFreePixmap(display,
		xsw_imageset[IMAGESET_CODE_MRAD].image[x]);
           XFreePixmap(display,
                xsw_imageset[IMAGESET_CODE_MRAD].shape_mask[x]);
         }
	}

        /* Klingon Bird of Prey */
        if(xsw_imageset[IMAGESET_CODE_KBOP].load_state == 1)   
        {
         for(x = 0; x < 16; x++)     
         {
           XFreePixmap(display,
                xsw_imageset[IMAGESET_CODE_KBOP].image[x]);
           XFreePixmap(display,
                xsw_imageset[IMAGESET_CODE_KBOP].shape_mask[x]);
         }
	}

        /* Intrepid Class */
        if(xsw_imageset[IMAGESET_CODE_INTR].load_state == 1)
        {
         for(x = 0; x < 16; x++)
         {
           XFreePixmap(display,
                xsw_imageset[IMAGESET_CODE_INTR].image[x]);
           XFreePixmap(display,
                xsw_imageset[IMAGESET_CODE_INTR].shape_mask[x]);
         }
        }

        /* Klingon Vortcha Class */
        if(xsw_imageset[IMAGESET_CODE_KVOR].load_state == 1)
        {            
         for(x = 0; x < 16; x++)
         {
           XFreePixmap(display,
                xsw_imageset[IMAGESET_CODE_KVOR].image[x]);
           XFreePixmap(display,
                xsw_imageset[IMAGESET_CODE_KVOR].shape_mask[x]);
         }
        }



	return;
}
