/* **********************************************************************
 *
 *	draw.c		Drawing Functions
 *	---------------------------------
 *	Functions:
 *
 *      int DrawSensor(
 *         long object_num,
 *         Window w,
 *         double win_x, double win_y,
 *         double height, double width,
 *         double scan_scale
 *      )
 *      int DrawLockCursor(Window w, double x_rel, double y_rel,
 *	   double cur_size)
 *      int DrawObjectAnimated(long object_num)
 *      int DrawHeadingArrow(Window w, double arrow_heading,
 *         unsigned int x_center, unsigned int y_center)
 *      int DrawStarfield(Window w,
 *         double x_rel, double y_rel, int width, int height)
 *      int DrawObjectImageset(long object_num)
 *	int DrawGameWin()
 *	int DrawStats(Window w)
 *	int DrawSensorReadout(Window w)
 *	int DrawToplevel()
 *
 *	---
 *
 *	Draw/redraw routines to redraw windows.
 *
 *	DrawSensor() draws a sensor (scanner) display relative
 *	to object object_num on Window w (does not clear window) with
 *	translation win_x and win_y and size height and width all in
 *	XSW Screen (pixel) units. scan_scale is half of width or height
 *	(whichever is larger) divided by object_num's scanner range.
 *	Returns non-zero on error.
 *
 *	DrawLockCursor() draws a weapons lock cursor on Window w
 *	(does not clear window) centered at position x_rel and y_rel
 *	with a weapons lock cursor scale of cur_scale.   All input
 *	units must be in XSW Screen (pixel) units.
 *      Returns non-zero on error.
 *
 *	DrawObjectAnimated() draws animated object object_num.
 *      Returns non-zero on error.
 *
 *	DrawHeadingArrow() draws a heading arrow on Window w (does not
 *	clear window) with a heading of arrow_heading (in radians) centered
 *	at x_center and y_center in XSW Screen (pixel) units.
 *      Returns non-zero on error.
 *
 *	DrawStarfield() draws a starfield on Window w (does not clear
 *	window) with relative distance of x_rel and y_rel and Window w's
 *	size width and height, all input units must be in XSW Screen
 *	(pixel) units.
 *      Returns non-zero on error.
 *
 *	DrawObjectImageset() draws object object_num's imageset.
 *      Returns non-zero on error.
 *
 *	DrawGameWin() draws all objects on the game window.
 *      Returns non-zero on error.
 *
 *	DrawStats() draws stats on Window w.
 *      Returns non-zero on error.
 *
 *	DrawSensorReadout() draws scanner verbose information on
 * 	Window w.
 *      Returns non-zero on error.
 *
 *      DrawToplevel()
 *	Returns non-zero on error.
 *
 */

#include "../include/xsw.h"  



int DrawSensor
(
	long object_num,	/* Relative to this object */
	Window w,		/* Window to draw this on. */
	double win_x, double win_y,	/* Translation of scanner box */
	double height, double width,	/* Size of scanner box. */
	double scan_scale	/* Multiply all coordinates by this */
)
{
	/* Local variables. */
	long object_count;
	double rel_x, rel_y;
	double x,y;
	char stringa[128];


	/* Object must be valid. */
	if((object_num < 0) || (object_num >= MAX_OBJECTS))
		return(-1);
	if(xsw_object[object_num].type <= XSW_OBJ_TYPE_GARBAGE)
		return(-1);


	/* scan_scale may not exceed a calculation of object_num's   */
	/* scanner_range                                             */
	if(height >= width)
	{
	   if((scan_scale * xsw_object[object_num].scanner_range) <
	      (height / 2))
           {
		scan_scale = (height / 2) / xsw_object[object_num].scanner_range;
	   }
	}
	else
	{
           if((scan_scale * xsw_object[object_num].scanner_range) <
              (width / 2))
           {
                scan_scale = (width / 2) / xsw_object[object_num].scanner_range;
           }
	}
	if(scan_scale < 0)
	   scan_scale = 0;


	/* Draw maximum range being scanned. */
        if(height >= width)
	{
	   sprintf(stringa, "%.1lf",
               (height / 2) / scan_scale);
	}
	else
	{
           sprintf(stringa, "%.1lf",
               (width / 2) / scan_scale);
	}
        XSetForeground(display, the_GC, normal_text_pixel);
        XDrawString(
           display, w, the_GC,
           5 + win_x, 18 + win_y, stringa, strlen(stringa)
        );


	/* Get relative position (in XSW real units). */
	rel_x = xsw_object[object_num].x;
	rel_y = xsw_object[object_num].y;


	/* Draw each object. */
	for(object_count = 0; object_count < total_objects; object_count++)
	{
	   /* Skip garbages and errors. */
	   if(xsw_object[object_count].type <= XSW_OBJ_TYPE_GARBAGE)
	      continue;

           /* Skip outdated objects. */
           if((xsw_object[object_count].last_updated + OBJECT_OUTDATED_INTERVAL)
              < cur_millitime)
                continue;

	   /* Skip stream weapons. */
	   if(xsw_object[object_count].type == XSW_OBJ_TYPE_STREAMWEAPON)
		continue;

	   /* Is object visible enough? */
	   x = xsw_object[object_count].x - rel_x;
	   y = xsw_object[object_count].y - rel_y;
	   if(Mu3DDistance(x,y,0) >
             (xsw_object[object_num].scanner_range *
             DBGetObjectVisibility(object_count))
	   )
		continue;


	   /* Change color depending on type of object. */
	   if(xsw_object[lplayer_object].locked_on == object_count)
	        XSetForeground(display, the_GC, weapons_lock_pixel);
	   else if(xsw_object[object_count].type == XSW_OBJ_TYPE_STATIC)
		XSetForeground(display, the_GC, light_grey_pixel);
           else if(xsw_object[object_count].type == XSW_OBJ_TYPE_WEAPON)
                XSetForeground(display, the_GC, white_pix);
	   else
	        XSetForeground(display, the_GC, white_pix);

	   /* Get x position to draw point. */
	   x = (xsw_object[object_count].x - rel_x) * scan_scale;
	   x = x + (width / 2);
	   x = x + win_x;
	   if(x > (win_x + width))
		continue;
           if(x < win_x)
                continue;

           y = (xsw_object[object_count].y - rel_y) * scan_scale;
	   y *= -1;
           y = y + (height / 2);
           y = y + win_y;
           if(y > (win_y + height))
                continue;
           if(y < win_y)
                continue;


	   /* Draw the marker, x and y are center. */
           XDrawArc(
               display,
               w,
               the_GC,
               (x - 3), (y - 3),          /* x, y */
               6, 6,         /* width, height */
               0, 23040      /* angle1, angle2 */
           );
	   /* Don't draw line for the following objects. */
           if( (xsw_object[object_count].type == XSW_OBJ_TYPE_STATIC) ||
               (xsw_object[object_count].type == XSW_OBJ_TYPE_HOME) ||
               (xsw_object[object_count].type == XSW_OBJ_TYPE_AREA) ||
               (xsw_object[object_count].type == XSW_OBJ_TYPE_ANIMATED)
	   )
	      continue;

	   XDrawLine(
               display,
               w, 
               the_GC,
	       x, y,
               (x + MuPolarRotX(xsw_object[object_count].heading, 9)),
	       (y - MuPolarRotY(xsw_object[object_count].heading, 9))
	   );

	}


	return(0);
}



int DrawLockCursor(Window w, double x_rel, double y_rel, double cur_size)
{       
        /* Local variables. */
        XPoint new_xpoint[10];


        /* Multiply cur_size by 2. */
        cur_size *= 2;

	/* cur_size may not be greater than 64. */
	if(cur_size > 64)
		cur_size = 64;


	/* Set foreground color of weapons lock. */
        XSetForeground(display, the_GC, weapons_lock_pixel);


   
        /* Set the left polygon points. */
        (double)new_xpoint[0].x = (-1.15 * cur_size) + x_rel;
        (double)new_xpoint[0].y = (-1.15 * cur_size) + y_rel;
             
        (double)new_xpoint[1].x = (-1.10 * cur_size) + x_rel;
        (double)new_xpoint[1].y = (-1.20 * cur_size) + y_rel;

        (double)new_xpoint[2].x = (-0.85 * cur_size) + x_rel;
        (double)new_xpoint[2].y = (-0.90 * cur_size) + y_rel;
   
        (double)new_xpoint[3].x = (-0.85 * cur_size) + x_rel;
        (double)new_xpoint[3].y = (-0.82 * cur_size) + y_rel;
        
        (double)new_xpoint[4].x = (-1.05 * cur_size) + x_rel;
        (double)new_xpoint[4].y = (-1.04 * cur_size) + y_rel;

        
        (double)new_xpoint[5].x = (-1.05 * cur_size) + x_rel;
        (double)new_xpoint[5].y = (1.04 * cur_size) + y_rel;
        
        (double)new_xpoint[6].x = (-0.85 * cur_size) + x_rel;
        (double)new_xpoint[6].y = (0.82 * cur_size) + y_rel; 
             
        (double)new_xpoint[7].x = (-0.85 * cur_size) + x_rel;
        (double)new_xpoint[7].y = (0.90 * cur_size) + y_rel; 

        (double)new_xpoint[8].x = (-1.10 * cur_size) + x_rel;
        (double)new_xpoint[8].y = (1.20 * cur_size) + y_rel; 
   
        (double)new_xpoint[9].x = (-1.15 * cur_size) + x_rel;
        (double)new_xpoint[9].y = (1.15 * cur_size) + y_rel; 
        
        XFillPolygon(display, w, the_GC,
             new_xpoint,
             10,
             Complex,
             CoordModeOrigin
        );
        
        
        /* Set the right polygon points. */
        (double)new_xpoint[0].x = (1.15 * cur_size) + x_rel;
        (double)new_xpoint[0].y = (-1.15 * cur_size) + y_rel;

        (double)new_xpoint[1].x = (1.10 * cur_size) + x_rel; 
        (double)new_xpoint[1].y = (-1.20 * cur_size) + y_rel;
        
        (double)new_xpoint[2].x = (0.85 * cur_size) + x_rel; 
        (double)new_xpoint[2].y = (-0.90 * cur_size) + y_rel;
        
        (double)new_xpoint[3].x = (0.85 * cur_size) + x_rel; 
        (double)new_xpoint[3].y = (-0.82 * cur_size) + y_rel;
  
        (double)new_xpoint[4].x = (1.05 * cur_size) + x_rel; 
        (double)new_xpoint[4].y = (-1.04 * cur_size) + y_rel;
        
        
        (double)new_xpoint[5].x = (1.05 * cur_size) + x_rel; 
        (double)new_xpoint[5].y = (1.04 * cur_size) + y_rel;
        
        (double)new_xpoint[6].x = (0.85 * cur_size) + x_rel; 
        (double)new_xpoint[6].y = (0.82 * cur_size) + y_rel;
        
        (double)new_xpoint[7].x = (0.85 * cur_size) + x_rel; 
        (double)new_xpoint[7].y = (0.90 * cur_size) + y_rel;
        
        (double)new_xpoint[8].x = (1.10 * cur_size) + x_rel; 
        (double)new_xpoint[8].y = (1.20 * cur_size) + y_rel;
        
        (double)new_xpoint[9].x = (1.15 * cur_size) + x_rel;
        (double)new_xpoint[9].y = (1.15 * cur_size) + y_rel;
        
        XFillPolygon(display, w, the_GC,
             new_xpoint,
             10,
             Complex,
             CoordModeOrigin
        );


	return(0);
}



int DrawObjectAnimated(long object_num)
{
	/* Local variables. */
	long current_frame;


	/* We only do XSW Objects with animated imagesets. */
	if(xsw_object[object_num].type != XSW_OBJ_TYPE_ANIMATED)	
		return(-1);


        /* Clear the window. */
        XClearWindow(display, xsw_object[object_num].window);


	/* See which ImageSet to draw. */
	switch(xsw_object[object_num].imageset)
	{
	   /* Explosion Type 1 */
	   case IMAGESET_CODE_EXPLOSION1:
		/* Make sure imageset is loaded. */
                if(xsw_imageset[IMAGESET_CODE_EXPLOSION1].load_state == 0)
                        IMGLoadSpecified(IMAGESET_CODE_EXPLOSION1, 4);

		/* Increment frame? */
		if( (xsw_object[object_num].animation.last_interval +
		    xsw_object[object_num].animation.interval) < cur_millitime
		)
	        {
		    xsw_object[object_num].animation.last_interval =
			cur_millitime;
		    xsw_object[object_num].animation.current_frame++;
		}

		/* Explosion 1 has 4 frames, make sure we don't go beyond. */
		if( (xsw_object[object_num].animation.current_frame > 3) ||
		    (xsw_object[object_num].animation.current_frame < 0) )
			xsw_object[object_num].animation.current_frame = 0;

		current_frame = xsw_object[object_num].animation.current_frame;
                XSetWindowBackgroundPixmap(display,
                   xsw_object[object_num].window,
		   xsw_imageset[IMAGESET_CODE_EXPLOSION1].image[current_frame]
                );
                XShapeCombineMask(display,
                   xsw_object[object_num].window,
                   ShapeBounding,
                   0, 0,
		   xsw_imageset[IMAGESET_CODE_EXPLOSION1].shape_mask[current_frame],
                   ShapeSet
                );
                break;


           /* Explosion Type 2 */
           case IMAGESET_CODE_EXPLOSION2:
                /* Make sure imageset is loaded. */
                if(xsw_imageset[IMAGESET_CODE_EXPLOSION2].load_state == 0)
                        IMGLoadSpecified(IMAGESET_CODE_EXPLOSION2, 1);

                /* Increment frame? */
                if( (xsw_object[object_num].animation.last_interval +
                    xsw_object[object_num].animation.interval) < cur_millitime
                )
                {
                    xsw_object[object_num].animation.last_interval =
                        cur_millitime;
                    xsw_object[object_num].animation.current_frame++;
		}

                /* Explosion 2 has 1 frame, make sure we don't go beyond. */
                if( (xsw_object[object_num].animation.current_frame > 0) ||
                    (xsw_object[object_num].animation.current_frame < 0) )
                        xsw_object[object_num].animation.current_frame = 0;

                current_frame = xsw_object[object_num].animation.current_frame;
                XSetWindowBackgroundPixmap(display,
                   xsw_object[object_num].window,
                   xsw_imageset[IMAGESET_CODE_EXPLOSION2].image[current_frame]
                );
                XShapeCombineMask(display,
                   xsw_object[object_num].window,
                   ShapeBounding,
                   0, 0,
                   xsw_imageset[IMAGESET_CODE_EXPLOSION2].shape_mask[current_frame],
                   ShapeSet
                );
                break;


           /* Default to Unknown */
           default:
                /* Make sure imageset is loaded. */
		if(xsw_imageset[IMAGESET_CODE_UNKNOWN].load_state == 0)
			IMGLoadSpecified(IMAGESET_CODE_UNKNOWN, 1);

                /* Draw using xsw_imageset[IMAGESET_CODE_UNKNOWN] */
                XSetWindowBackgroundPixmap(display,
                        xsw_object[object_num].window,
                        xsw_imageset[IMAGESET_CODE_UNKNOWN].image[0]
                );
                XShapeCombineMask(display,
                        xsw_object[object_num].window,
                        ShapeBounding,
                        0, 0,
                        xsw_imageset[IMAGESET_CODE_UNKNOWN].shape_mask[0],
                        ShapeSet
                );
                break;
	}

	return(0);
}



int DrawStreamWeapon(long object_num, double x_rel, double y_rel)
{
	/* Local variables. */
        double heading; 	/* In radians. */
        double x, y;
	double angle;		/* In radians. */
	double range;		/* In XSW Screen (pixels) units. */
	double hyp;		/* In XSW Screen (pixels) units. */
	XPoint wep_point[4];


        /* We only do stream weapon XSW Objects. */
        if(xsw_object[object_num].type != XSW_OBJ_TYPE_STREAMWEAPON)
                return(-1);


	/*
	 * Drawing Stream Weapons:
	 *
	 *   Objects of type XSW_OBJ_TYPE_STREAMWEAPON do not have imagesets,
	 *   instead they are drawn with Xlib draw commands (not pixmaps).
	 */
        switch(xsw_object[object_num].imageset)
	{
	    /* Green Beam */
	    case IMAGESET_CODE_WEP_STREAM_G:

		heading = xsw_object[object_num].heading;
		range = xsw_object[object_num].size;

		/* Set points. */
		angle = atan(2 / range);
		hyp = sqrt((2*2)+(range*range));

		x = MuPolarRotX((heading - 1.5707963),2);
		y = MuPolarRotY((heading - 1.5707963),2);
		y *= -1;
		wep_point[0].x = x + x_rel;
		wep_point[0].y = y + y_rel;
                x = MuPolarRotX((heading - angle),hyp);
                y = MuPolarRotY((heading - angle),hyp);
                y *= -1;
                wep_point[1].x = x + x_rel;
                wep_point[1].y = y + y_rel;
                x = MuPolarRotX((heading + angle),hyp);
                y = MuPolarRotY((heading + angle),hyp);
                y *= -1;
                wep_point[2].x = x + x_rel;
                wep_point[2].y = y + y_rel;
                x = MuPolarRotX((heading + 1.5707963),2);
                y = MuPolarRotY((heading + 1.5707963),2);
                y *= -1;
                wep_point[3].x = x + x_rel;
                wep_point[3].y = y + y_rel;
 
		/* Draw on the game_win. */
		XSetForeground(display, the_GC, weapons_lock_pixel);
                XFillPolygon(display, game_win, the_GC,
                   wep_point,
                   4,
                   Complex,
                   CoordModeOrigin
                );

		break;




	    /* Default to Yellow Beam */
	    default:

                heading = xsw_object[object_num].heading;
                range = xsw_object[object_num].size;
 
                /* Set points. */
                angle = atan(2 / range);
                hyp = sqrt((2*2)+(range*range));
        
                x = MuPolarRotX((heading - 1.5707963),2);
                y = MuPolarRotY((heading - 1.5707963),2);
                y *= -1;
                wep_point[0].x = x + x_rel;
                wep_point[0].y = y + y_rel;
                x = MuPolarRotX((heading - angle),hyp);
                y = MuPolarRotY((heading - angle),hyp);
                y *= -1;
                wep_point[1].x = x + x_rel;
                wep_point[1].y = y + y_rel;
                x = MuPolarRotX((heading + angle),hyp);
                y = MuPolarRotY((heading + angle),hyp);
                y *= -1;
                wep_point[2].x = x + x_rel;
                wep_point[2].y = y + y_rel; 
                x = MuPolarRotX((heading + 1.5707963),2);
                y = MuPolarRotY((heading + 1.5707963),2);
                y *= -1;
                wep_point[3].x = x + x_rel;
                wep_point[3].y = y + y_rel;
                
                /* Draw on the game_win. */
                XSetForeground(display, the_GC, weapons_lock_pixel);
                XFillPolygon(display, game_win, the_GC,
                   wep_point,
                   4,
                   Complex,
                   CoordModeOrigin
                );

		break;
	}




	return(0);
}



int DrawObjectImageset(long object_num)
{
        /* Local variables. */
        double theta;
        unsigned int imageset_frame;
        

        /* Clear the window. */
        XClearWindow(display, xsw_object[object_num].window);
        

	/* ************************************************** */
	/* ***    Get imageset_frame Based on Heading     *** */
        /* See which imageset_frame to use (0 to 15). */
        theta = 0.19634954;
        for(imageset_frame = 0; imageset_frame < MAX_IMAGESET_FRAMES;
                imageset_frame++ )
        {
           if(xsw_object[object_num].heading < theta) 
                break;
           else
                theta += 0.39269908;
        }
        /* Make sure imageset_frame is not out of range. */
        if(imageset_frame >= MAX_IMAGESET_FRAMES)
                imageset_frame = 0;



	/* ************************************************** */
        /* ***            Draw Imageset Frame             *** */
        switch(xsw_object[object_num].imageset)
        {
           /* Buoy */
           case IMAGESET_CODE_BUOY:
                /* Make sure imageset is loaded. */
                if(xsw_imageset[IMAGESET_CODE_BUOY].load_state == 0)
                        IMGLoadSpecified(IMAGESET_CODE_BUOY, 1);

                XSetWindowBackgroundPixmap(display,
                        xsw_object[object_num].window,
                        xsw_imageset[IMAGESET_CODE_BUOY].image[0]
                );
                XShapeCombineMask(display,
                        xsw_object[object_num].window,
                        ShapeBounding,
                        0, 0,
                        xsw_imageset[IMAGESET_CODE_BUOY].shape_mask[0],
                        ShapeSet
                );   
                break;

	   /* Hubble Space Telescope */
           case IMAGESET_CODE_HUBBLE:
                /* Make sure imageset is loaded. */
                if(xsw_imageset[IMAGESET_CODE_HUBBLE].load_state == 0)
                        IMGLoadSpecified(IMAGESET_CODE_HUBBLE, 1);

                XSetWindowBackgroundPixmap(display,
                        xsw_object[object_num].window,
                        xsw_imageset[IMAGESET_CODE_HUBBLE].image[0]
                );
                XShapeCombineMask(display,
                        xsw_object[object_num].window,
                        ShapeBounding,
                        0, 0,
                        xsw_imageset[IMAGESET_CODE_HUBBLE].shape_mask[0],
                        ShapeSet
                );
                break;
        
           /* Weapons Fire: Pulse Yellow */
           case IMAGESET_CODE_WEP_PP_Y:
                /* Make sure imageset is loaded. */
                if(xsw_imageset[IMAGESET_CODE_WEP_PP_Y].load_state == 0)
                        IMGLoadSpecified(IMAGESET_CODE_WEP_PP_Y, 1);

                XSetWindowBackgroundPixmap(display,
                        xsw_object[object_num].window,
                        xsw_imageset[IMAGESET_CODE_WEP_PP_Y].image[0]
                );
                XShapeCombineMask(display,
                        xsw_object[object_num].window,
                        ShapeBounding,
                        0, 0,
                        xsw_imageset[IMAGESET_CODE_WEP_PP_Y].shape_mask[0],
                        ShapeSet
                );
                break;
        
           /* Weapons Fire: Pulse Green */
           case IMAGESET_CODE_WEP_PP_G:
                /* Make sure imageset is loaded. */
                if(xsw_imageset[IMAGESET_CODE_WEP_PP_G].load_state == 0)
                        IMGLoadSpecified(IMAGESET_CODE_WEP_PP_G, 1);

                XSetWindowBackgroundPixmap(display,
                        xsw_object[object_num].window,
                        xsw_imageset[IMAGESET_CODE_WEP_PP_G].image[0]
                );
                XShapeCombineMask(display,
                        xsw_object[object_num].window,
                        ShapeBounding,
                        0, 0,
                        xsw_imageset[IMAGESET_CODE_WEP_PP_G].shape_mask[0],
                        ShapeSet
                );
                break;
                
           /* Weapons Fire: Photon Torpedo */
           case IMAGESET_CODE_WEP_PT_R:
                /* Make sure imageset is loaded. */
                if(xsw_imageset[IMAGESET_CODE_WEP_PT_R].load_state == 0)
                        IMGLoadSpecified(IMAGESET_CODE_WEP_PT_R, 1);

                XSetWindowBackgroundPixmap(display,
                        xsw_object[object_num].window,
                        xsw_imageset[IMAGESET_CODE_WEP_PT_R].image[0]
                );
                XShapeCombineMask(display,
                        xsw_object[object_num].window,
                        ShapeBounding, 
                        0, 0,
                        xsw_imageset[IMAGESET_CODE_WEP_PT_R].shape_mask[0],
                        ShapeSet
                );
                break;
                        
           /* Maqui Raider */
           case IMAGESET_CODE_MRAD:
                /* Make sure imageset is loaded. */
                if(xsw_imageset[IMAGESET_CODE_MRAD].load_state == 0)
                        IMGLoadSpecified(IMAGESET_CODE_MRAD, 16);

                XSetWindowBackgroundPixmap(display,
                        xsw_object[object_num].window,
                        xsw_imageset[IMAGESET_CODE_MRAD].image[imageset_frame]
                );
                XShapeCombineMask(display,
                        xsw_object[object_num].window,
                        ShapeBounding, 
                        0, 0,
                        xsw_imageset[IMAGESET_CODE_MRAD].shape_mask[imageset_frame],
                        ShapeSet
                );
                break;
                        
           /* Klingon Bird of Prey */ 
           case IMAGESET_CODE_KBOP:
                /* Make sure imageset is loaded. */
                if(xsw_imageset[IMAGESET_CODE_KBOP].load_state == 0)
                        IMGLoadSpecified(IMAGESET_CODE_KBOP, 16);

                XSetWindowBackgroundPixmap(display,
                        xsw_object[object_num].window,
                        xsw_imageset[IMAGESET_CODE_KBOP].image[imageset_frame]
                );
                XShapeCombineMask(display,
                        xsw_object[object_num].window,
                        ShapeBounding, 
                        0, 0,
                        xsw_imageset[IMAGESET_CODE_KBOP].shape_mask[imageset_frame],
                        ShapeSet
                );
                break;

           /* Intrepid Class */
           case IMAGESET_CODE_INTR:
                /* Make sure imageset is loaded. */
                if(xsw_imageset[IMAGESET_CODE_INTR].load_state == 0)
                        IMGLoadSpecified(IMAGESET_CODE_INTR, 16);

                XSetWindowBackgroundPixmap(display,   
                        xsw_object[object_num].window,
                        xsw_imageset[IMAGESET_CODE_INTR].image[imageset_frame]
                );
                XShapeCombineMask(display,
                        xsw_object[object_num].window,
                        ShapeBounding,
                        0, 0,
                        xsw_imageset[IMAGESET_CODE_INTR].shape_mask[imageset_frame],
                        ShapeSet
                );
                break;

           /* Klingon Vortcha Class */
           case IMAGESET_CODE_KVOR:
                /* Make sure imageset is loaded. */
                if(xsw_imageset[IMAGESET_CODE_KVOR].load_state == 0)
                        IMGLoadSpecified(IMAGESET_CODE_KVOR, 16);

                XSetWindowBackgroundPixmap(display,
                        xsw_object[object_num].window,
                        xsw_imageset[IMAGESET_CODE_KVOR].image[imageset_frame]
                );
                XShapeCombineMask(display,
                        xsw_object[object_num].window,
                        ShapeBounding,
                        0, 0,
			xsw_imageset[IMAGESET_CODE_KVOR].shape_mask[imageset_frame],
                        ShapeSet
                );
                break;


	   /* Station - Cardassian */
	   case IMAGESET_CODE_STCAR:
                /* Make sure imageset is loaded. */
                if(xsw_imageset[IMAGESET_CODE_STCAR].load_state == 0)
                        IMGLoadSpecified(IMAGESET_CODE_STCAR, 1);

                XSetWindowBackgroundPixmap(display,
                        xsw_object[object_num].window,
                        xsw_imageset[IMAGESET_CODE_STCAR].image[0]
                );
                XShapeCombineMask(display,
                        xsw_object[object_num].window,
                        ShapeBounding,
                        0, 0,
                        xsw_imageset[IMAGESET_CODE_STCAR].shape_mask[0],
                        ShapeSet
                );
                break;


           /* Default to Unknown */  
           default:
                /* Make sure imageset is loaded. */
                if(xsw_imageset[IMAGESET_CODE_UNKNOWN].load_state == 0)
                        IMGLoadSpecified(IMAGESET_CODE_UNKNOWN, 1);

                XSetWindowBackgroundPixmap(display,
                        xsw_object[object_num].window,
                        xsw_imageset[IMAGESET_CODE_UNKNOWN].image[0]
                );      
                XShapeCombineMask(display,
                        xsw_object[object_num].window,
                        ShapeBounding,
                        0, 0,
                        xsw_imageset[IMAGESET_CODE_UNKNOWN].shape_mask[0],
                        ShapeSet
                );
                break;
        }
                        
        
        return(0);
}



int DrawHeadingArrow(Window w, double arrow_heading,
	unsigned int x_center, unsigned int y_center)
{

	/* Set foreground color (we need a better color!) */
        XSetForeground(display, the_GC, white_pix);


	/* Sanitize arrow_heading. */
	while(arrow_heading > 6.2831853)
		arrow_heading -= 6.2831853;
        while(arrow_heading < 0)
                arrow_heading += 6.2831853;


	/* Invert arrow_heading. */
	arrow_heading = 6.2831853 - arrow_heading;

	/* Begin drawing the arrow. */
	XDrawLine(display, w, the_GC,
		x_center - MuPolarRotX(arrow_heading + 0, 73),
		y_center - MuPolarRotY(arrow_heading + 0, 73),
		x_center - MuPolarRotX(arrow_heading + 0.03, 68),
		y_center - MuPolarRotY(arrow_heading + 0.03, 68)
	);
        XDrawLine(display, w, the_GC,
                x_center - MuPolarRotX(arrow_heading + 0, 73),
                y_center - MuPolarRotY(arrow_heading + 0, 73),
                x_center - MuPolarRotX(arrow_heading - 0.03, 68),
                y_center - MuPolarRotY(arrow_heading - 0.03, 68)
        );
  

	return(0);
}



int DrawStarfield(Window w,
    double x_rel, double y_rel, int width, int height)
{
	/* Local variables. */
        double x, y;
        int temp_x_holder;
        int temp_y_holder;

        temp_x_holder = x_rel;
        temp_y_holder = y_rel;


        /* Sanitize x_rel and y_rel. */
        while( x_rel > width )
                x_rel -= width;
        while( x_rel < 0 )
                x_rel += width;

        while( y_rel > height )
                y_rel -= height;
        while( y_rel < 0 )
                y_rel += height;


        /* Invert x_rel. */
        x_rel = width - x_rel;


	/* ******************************************************** */
	/* ***              Begin Drawing Stars                 *** */

        XSetForeground(display, the_GC, star1_pixel);
        x = temp_x_holder;
        y = temp_y_holder;
        while( x > (width * 2) )
                x -= (width * 2);
        while( x < 0 )
                x += (width * 2);
        while( y > (height * 2) )
                y -= (height * 2);
        while( y < 0 )
                y += (height * 2);
        x = width - (x / 2);
        y = y / 2;
        XDrawPoint(display, w, the_GC, x, y);


        XSetForeground(display, the_GC, star2_pixel);
        x = temp_x_holder + 540;
        y = temp_y_holder + 240;
        while( x > (width * 2) )
                x -= (width * 2);
        while( x < 0 )
                x += (width * 2);
        while( y > (height * 2) )
                y -= (height * 2);
        while( y < 0 )
                y += (height * 2);
        x = width - (x / 2);
        y = y / 2;
        XDrawPoint(display, w, the_GC, x, y);

        XSetForeground(display, the_GC, star2_pixel);
        x = temp_x_holder + 320;
        y = temp_y_holder + 840;
        while( x > (width * 2) )
                x -= (width * 2);
        while( x < 0 )
                x += (width * 2);
        while( y > (height * 2) )
                y -= (height * 2);
        while( y < 0 )
                y += (height * 2);
        x = width - (x / 2);
        y = y / 2;
        XDrawPoint(display, w, the_GC, x, y);

        XSetForeground(display, the_GC, star3_pixel);
        x = temp_x_holder + 712;
        y = temp_y_holder + 1080;
        while( x > (width * 2) )
                x -= (width * 2);
        while( x < 0 )
                x += (width * 2);
        while( y > (height * 2) )
                y -= (height * 2);
        while( y < 0 )
                y += (height * 2);
        x = width - (x / 2);
        y = y / 2;
        XDrawPoint(display, w, the_GC, x, y);


	/* BEGIN PARALLAX MODIFICATIONS --Caspian */

	/* ABOVE WAS THE TOPMOST (nearest) STAR LAYER. Here is the middle layer. */
        XSetForeground(display, the_GC, star4_pixel);
        x = temp_x_holder + 1000;
        y = temp_y_holder + 1250;
        while( x > (width * 4) )
                x -= (width * 4);
        while( x < 0 )
                x += (width * 4);
        while( y > (height * 4) )
                y -= (height * 4);
        while( y < 0 )
                y += (height * 4);
	x = width - (x / 4);
	y = y / 4;
	XDrawPoint(display, w, the_GC, x, y);


        XSetForeground(display, the_GC, star5_pixel);
        x = temp_x_holder + 729;
        y = temp_y_holder + 258;
        while( x > (width * 4) )
                x -= (width * 4);
        while( x < 0 )
                x += (width * 4);
        while( y > (height * 4) )
                y -= (height * 4);
        while( y < 0 )
                y += (height * 4);
	x = width - (x / 4);
	y = y / 4;
	XDrawPoint(display, w, the_GC, x, y);

        XSetForeground(display, the_GC, star6_pixel);
        x = temp_x_holder + 395;
        y = temp_y_holder + 572;
        while( x > (width * 4) )
                x -= (width * 4);
        while( x < 0 )
                x += (width * 4);
        while( y > (height * 4) )
                y -= (height * 4);
        while( y < 0 )
                y += (height * 4);
        x = width - (x / 4);
        y = y / 4;
        XDrawPoint(display, w, the_GC, x, y);




        /* Bottom (farthest) layer. */
        XSetForeground(display, the_GC, star7_pixel);
        x = temp_x_holder + 1900;
        y = temp_y_holder + 1450;
        while( x > (width*8) )
                x -= (width*8);
        while( x < 0 )
                x += (width*8);
        while( y > (height*8) )
                y -= (height*8);
        while( y < 0 )
                y += (height*8);
	x = width - (x / 8);
	y = y / 8;
	XDrawPoint(display, w, the_GC, x, y);

        XSetForeground(display, the_GC, star8_pixel);
        x = temp_x_holder + 1729;
        y = temp_y_holder + 1180;
        while( x > (width*8) )
                x -= (width*8);
        while( x < 0 )
                x += (width*8);
        while( y > (height*8) )
                y -= (height*8);
        while( y < 0 )
                y += (height*8);
	x = width - (x / 8);
	y = y / 8;
	XDrawPoint(display, w, the_GC, x, y);

        XSetForeground(display, the_GC, star9_pixel);
        x = temp_x_holder + 1275;
        y = temp_y_holder + 1572;
        while( x > (width*8) )
                x -= (width*8);
        while( x < 0 )
                x += (width*8);
        while( y > (height*8) )
                y -= (height*8);
        while( y < 0 )
                y += (height*8);
	x = width - (x / 8);
	y = y / 8;
	XDrawPoint(display, w, the_GC, x, y);



        return(0);
}



int DrawGameWin()
{
	/* Local variables. */
	char stringa[256];
	long object_count;
	double result;
	double x_rel, y_rel;	/* Relative from XSW real */
				/* plane in real units to */
				/* lplayer_object.        */
	double x_vtrans, y_vtrans;	/* Translation caused by       */
					/* velocity of lplayer_object. */
	long imageset;


	/* lplayer_object must be valid. */
	if((lplayer_object < 0) || (lplayer_object >= MAX_OBJECTS))
                return(-1);   


	/* *****************************************************************
	 *
	 *   Size of game_win Window is 700 by 700 pixels. All Real units
	 *   *must* be converted to Screen (pixel) units before they can be
	 *   used in any calculation here.
	 */

	/* Get relative values to be applied to all object */
	/* coordiates.                                     */
	x_rel = xsw_object[lplayer_object].x * 1000;
	y_rel = xsw_object[lplayer_object].y * 1000;

	/* Get velocity translation to be applied to all object */
	/* coordinates.                                         */
	if(xsw_object[lplayer_object].velocity_max <= 0)
	{
		xsw_object[lplayer_object].velocity_max = 1;
	}

	x_vtrans = MuPolarRotX(
		xsw_object[lplayer_object].velocity_heading,
		(xsw_object[lplayer_object].velocity / 
		xsw_object[lplayer_object].velocity_max) * 300
	);
        y_vtrans = MuPolarRotY(
                xsw_object[lplayer_object].velocity_heading,            
                (xsw_object[lplayer_object].velocity /
                xsw_object[lplayer_object].velocity_max) * 300
        );


	/* Clear window. */
	XClearWindow(display, game_win);

	/* Draw starfield on game_win. */
	DrawStarfield(game_win,
		x_rel - x_vtrans, y_rel + y_vtrans,
		700, 700
	);


	/* ********************************************************** */
	/* ***               Begin Drawing Objects                *** */
	for(object_count = 0; object_count < total_objects; object_count++)
	{
	   /* Skip error and garbage objects. */
	   if(xsw_object[object_count].type <= XSW_OBJ_TYPE_GARBAGE)
		continue;

	   /* Skip outdated objects. */
	   if((xsw_object[object_count].last_updated + OBJECT_OUTDATED_INTERVAL)
	      < cur_millitime)
	   {
		XUnmapWindow(display, xsw_object[object_count].window);
		continue;
	   }

	   /* Don't draw if object is cloaked or below 25% visibility. */
	   if(DBGetObjectVisibility(object_count) <= 0.25)
	   {
                XUnmapWindow(display, xsw_object[object_count].window);
		continue;
	   }


	   /* Is the object *way* out of the screen?           */
	   /* Do *not* take into account x_vtrans and y_vtrans */
	   result = Mu3DDistance(
	     (xsw_object[object_count].x * 1000) - (x_rel + 350),
	     (xsw_object[object_count].y * 1000) - (y_rel - 350),
	     0
	   );
/* May need to be increased for stream weapons,       */
/* since they have a long range from their center.    */
	   if(result > 1300)
	   {
		XUnmapWindow(display, xsw_object[object_count].window);
		continue;
	   }


	   /* Move the XSW Object's window. */
	   /* Except for stream weapons. */
	   if(xsw_object[object_count].type == XSW_OBJ_TYPE_STREAMWEAPON)
	   {
	     XUnmapWindow(display, xsw_object[object_count].window);
	   }
	   else
           {
	     imageset = xsw_object[object_count].imageset;
	     if( (imageset < 0) || (imageset >= MAX_IMAGESETS) )
		continue;

	     XMapWindow(display, xsw_object[object_count].window);
	     XMoveWindow(display, xsw_object[object_count].window,
	      (xsw_object[object_count].x * 1000) - (x_rel + x_vtrans) +
	      ( 350 - (xsw_imageset[imageset].fwidth / 2) ),
	      (((xsw_object[object_count].y * 1000) - (y_rel + y_vtrans)) * -1) +
	      ( 350 - (xsw_imageset[imageset].fheight / 2) )
	     );
	   }


	   /* Draw animated object (ie explosions) */
	   if(xsw_object[object_count].type == XSW_OBJ_TYPE_ANIMATED)
	   {
		DrawObjectAnimated(object_count);
	   }
	   /* Draw stream weapon object (ie phazer beams) */
	   else if(xsw_object[object_count].type == XSW_OBJ_TYPE_STREAMWEAPON)
	   {
		DrawStreamWeapon(object_count,
                   (xsw_object[object_count].x * 1000) -
                   (x_rel + x_vtrans) + 350,
                   (((xsw_object[object_count].y * 1000) -
                   (y_rel + y_vtrans)) * -1) + 350
		);
	   }
	   /* Draw regular object (everything else). */
	   else
	   {
	        DrawObjectImageset(object_count);
	   }


	   /* Draw weapons lock cursor on locked object? */
	   if(xsw_object[lplayer_object].locked_on == object_count)
	   {
		DrawLockCursor(
		   game_win,
		   (xsw_object[object_count].x * 1000) -
		   (x_rel + x_vtrans) + 350,
                   (((xsw_object[object_count].y * 1000) -
		   (y_rel + y_vtrans)) * -1) + 350,
		   xsw_object[object_count].size
		);
	   }

	   /* Draw names? */
	   if( (show_object_names > 0) &&
	       (strlen(xsw_object[object_count].name) > 0)
	   )
	   {
	      if( (xsw_object[object_count].type == XSW_OBJ_TYPE_STATIC) ||
		  (xsw_object[object_count].type == XSW_OBJ_TYPE_DYNAMIC) ||
                  (xsw_object[object_count].type == XSW_OBJ_TYPE_CONTROLLED) ||
                  (xsw_object[object_count].type == XSW_OBJ_TYPE_PLAYER) ||
                  (xsw_object[object_count].type == XSW_OBJ_TYPE_HOME) ||
                  (xsw_object[object_count].type == XSW_OBJ_TYPE_AREA)
	      )
	      {
		 XSetForeground(display, the_GC, normal_text_pixel);
		 XDrawString(display, game_win, the_GC,
		    (xsw_object[object_count].x * 1000) -
                    (x_rel + x_vtrans) + 260,
                    (((xsw_object[object_count].y * 1000 ) -
                    (y_rel + y_vtrans)) * -1) + 275,
                    xsw_object[object_count].name,
                    strlen(xsw_object[object_count].name)
                 );
              }
	   }


	}


	/* Draw direction arrow for object 0. */
	if(show_heading_arrow > 0)
	{
	   DrawHeadingArrow(game_win, xsw_object[lplayer_object].heading,
		350 - x_vtrans, 350 + y_vtrans);
	}


	/* Update FPS counter. */
        if(show_object_names > 0)
	{
	    fps_counter.fcount++;
	    if( (fps_counter.lastud + fps_counter.interval) < cur_millitime )
	    {
	        fps_counter.lfcount = fps_counter.fcount;
	        fps_counter.fcount = 0;
	        fps_counter.lastud = cur_millitime;
	    }
	    sprintf(stringa, "FPS: %i", fps_counter.lfcount);
	    XDrawString(display, game_win, the_GC,
		600, 695,
		stringa, strlen(stringa)
	    );
	}
	

	return(0);
}



int DrawStats(Window w)
{
	/* Local variables. */
	double x, y, z;
	long lx, ly, lz;
	long counta;
	char stringa[256];


	/* Clear all windows related to player_stats_win. */
	XClearWindow(display, w);


        /* lplayer_object must be valid. */
        if((lplayer_object < 0) || (lplayer_object >= MAX_OBJECTS))
                return(-1);



	/* Hull. */
        XSetForeground(display, the_GC, hull_bar_pixel);
        XDrawRectangle(display, w, the_GC,
           15, 70, 122, 15
        );
        XFillArc(display,
           w,
           the_GC,
           5, 70,         /* x, y */
           15, 15,         /* width, height */
           5760, 11520     /* angle1, angle2 */
        );
        XFillArc(display,
           w,
           the_GC,
           132, 70,         /* x, y */
           15, 15,         /* width, height */
           -5760, 11520     /* angle1, angle2 */
        );
        XSetForeground(display, the_GC, black_pix);
        XFillRectangle(display, w, the_GC,
           16, 71, 121, 14
        );
        XSetForeground(display, the_GC, hull_bar_pixel);
	if(xsw_object[lplayer_object].hp_max > 0)
	{
            x = ( xsw_object[lplayer_object].hp /
                xsw_object[lplayer_object].hp_max ) * 121;
	}
	else
	{
	    x = 0;
	}
        XFillRectangle(display, w, the_GC,
           16, 71, x, 14
        );
        XSetForeground(display, the_GC, dark_grey_pixel);
        XDrawString(
           display, w, the_GC,
           63, 83, "HULL", strlen("HULL")
        );

	/* Power */
	XSetForeground(display, the_GC, power_bar_pixel);
	XDrawRectangle(display, w, the_GC,
	   15, 93, 122, 15
	);
        XFillArc(display,
           w,  
           the_GC,
           5, 93,         /* x, y */
           15, 15,         /* width, height */
           5760, 11520     /* angle1, angle2 */
        );           
        XFillArc(display,
           w,  
           the_GC,
           132, 93,         /* x, y */
           15, 15,         /* width, height */
           -5760, 11520     /* angle1, angle2 */
        );
        XSetForeground(display, the_GC, black_pix);
	XFillRectangle(display, w, the_GC,
           16, 94, 121, 14
        );
        XSetForeground(display, the_GC, power_bar_pixel);
	if(xsw_object[lplayer_object].power_max > 0)
	{
	    x = ( xsw_object[lplayer_object].power /
                xsw_object[lplayer_object].power_max ) * 121;
	}
	else
	{
	    x = 0;
	}
        XFillRectangle(display, w, the_GC,
           16, 94, x, 14
        );
        XSetForeground(display, the_GC, dark_grey_pixel);
        XDrawString(
           display, w, the_GC,
           56, 106, "POWER", strlen("POWER")
        );

	/* Shields */
	if(xsw_object[lplayer_object].shield_state == 0)
	{
                XSetForeground(display, the_GC, shields_bar_pixel);
                XDrawRectangle(display, w, the_GC,
                   15, 116, 122, 15);
                XFillArc(display,
                   w,
                   the_GC,
                   5, 116,         /* x, y */
                   15, 15,         /* width, height */
                   5760, 11520     /* angle1, angle2 */
               );           
               XFillArc(display,
                   w,  
                   the_GC,
                   132, 116,         /* x, y */
                   15, 15,         /* width, height */
                   -5760, 11520     /* angle1, angle2 */
                );

                XSetForeground(display, the_GC, black_pix);
                XFillRectangle(display, w, the_GC,
                   16, 117, 121, 14);
		sprintf(stringa, "SH: DOWN  %.2f",
                   xsw_object[lplayer_object].shield_frequency);
		XSetForeground(display, the_GC, shields_bar_pixel);
		XDrawString(
           		display, w, the_GC,
		        20,128,stringa,strlen(stringa)
        	);
	}
	else
	{
                XSetForeground(display, the_GC, shields_bar_pixel);
                XFillRectangle(display, w, the_GC,
                   15, 116, 122, 15);
                XFillArc(display,
                   w,
                   the_GC,
                   5, 116,         /* x, y */
                   15, 15,         /* width, height */
                   5760, 11520     /* angle1, angle2 */
               );    
               XFillArc(display,
                   w, 
                   the_GC,
                   132, 116,         /* x, y */
                   15, 15,         /* width, height */
                   -5760, 11520     /* angle1, angle2 */
                );
                sprintf(stringa, "SH: UP    %.2f",
                   xsw_object[lplayer_object].shield_frequency);
                XSetForeground(display, the_GC, dark_grey_pixel);
                XDrawString(
                        display, w, the_GC,
                        20,128,stringa,strlen(stringa)
                );
	}


	/* Weapons Lock */
	if(
            (xsw_object[lplayer_object].locked_on >= 0) &&
	    (xsw_object[lplayer_object].locked_on < MAX_OBJECTS)
        )
	{
		lx = xsw_object[lplayer_object].locked_on;
		sprintf(stringa,"WL: %s (#%i)",
		   xsw_object[lx].name,
		   lx
		);
                XSetForeground(display, the_GC, weapons_lock_pixel);
                XDrawString(
                   display, w, the_GC,
                   5,161,stringa,strlen(stringa)
                );
	}


	/* Weapons. */
	ly = 194;
	for(lx = 0; lx < MAX_WEAPONS; lx++)
	{
	   if(xsw_object[lplayer_object].total_weapons <= lx)
	      break;

	   switch(xsw_object[lplayer_object].weapons[lx].emission_type)
	   {
             case WEPEMISSION_PULSE:
               if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_NULL)
                   sprintf(stringa, "%i. NULL Device", (lx + 1));
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_PHAZER_5)
                   sprintf(stringa, "%i. Stream Phazer", (lx + 1));
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_DISRUPTOR_1)
                   sprintf(stringa, "%i. Stream Disruptor", (lx + 1));
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_DISRUPTOR_5)
                   sprintf(stringa, "%i. Pulse Disruptor", (lx + 1));
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_ESPHERE_5)
                   sprintf(stringa, "%i. Expanding Sphere", (lx + 1));
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_PULSE_PHAZER_5)
                   sprintf(stringa, "%i. Pulse Phazer", (lx + 1));
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_PHOTON_TORP_5)
                   sprintf(stringa, "%i. Photons", (lx + 1));
               break;

	     case WEPEMISSION_PROJECTILE:
               if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_NULL)
                   sprintf(stringa, "%i. NULL Device: %i", (lx + 1),
                       xsw_object[lplayer_object].weapons[lx].amount);
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_PHAZER_5)
                   sprintf(stringa, "%i. Stream Phazer: %i", (lx + 1),
                       xsw_object[lplayer_object].weapons[lx].amount);
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_DISRUPTOR_1)
                   sprintf(stringa, "%i. Stream Disruptor: %i", (lx + 1),
                       xsw_object[lplayer_object].weapons[lx].amount);
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_DISRUPTOR_5)
                   sprintf(stringa, "%i. Pulse Disruptor: %i", (lx + 1),
                       xsw_object[lplayer_object].weapons[lx].amount);
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_ESPHERE_5)
                   sprintf(stringa, "%i. Expanding Sphere: %i", (lx + 1),
                       xsw_object[lplayer_object].weapons[lx].amount);
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_PULSE_PHAZER_5)
                   sprintf(stringa, "%i. Pulse Phazer: %i", (lx + 1),                  
                       xsw_object[lplayer_object].weapons[lx].amount);
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_PHOTON_TORP_5)
                   sprintf(stringa, "%i. Photons: %i", (lx + 1),
                       xsw_object[lplayer_object].weapons[lx].amount);
               break;


	     default:
               if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_NULL)
                   sprintf(stringa, "%i. NULL Device", (lx + 1));
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_PHAZER_5)
                   sprintf(stringa, "%i. Stream Phazer", (lx + 1));
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_DISRUPTOR_1)
                   sprintf(stringa, "%i. Stream Disruptor", (lx + 1));
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_DISRUPTOR_5)
                   sprintf(stringa, "%i. Pulse Disruptor", (lx + 1));
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_ESPHERE_5)
                   sprintf(stringa, "%i. Expanding Sphere", (lx + 1));
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_PULSE_PHAZER_5)
                   sprintf(stringa, "%i. Pulse Phazer", (lx + 1));
               else if(xsw_object[lplayer_object].weapons[lx].type <= WEPTYPE_PHOTON_TORP_5)
                   sprintf(stringa, "%i. Photons", (lx + 1)); 
               break;
	   }

	   if(xsw_object[lplayer_object].selected_weapon == lx)
		XSetForeground(display, the_GC, selected_weapon_pixel);
	   else
		XSetForeground(display, the_GC, unselected_weapon_pixel);

           XDrawString(
              display, w, the_GC, 
              5, ly, stringa, strlen(stringa)
           );

	   ly += 33;
	}


	/* Visibility */
        XSetForeground(display, the_GC, visibility_bar_pixel);
        XDrawRectangle(display, w, the_GC,
           15, 331, 122, 15
        );
        XFillArc(display,
           w,
           the_GC,
           5, 331,         /* x, y */
           15, 15,         /* width, height */
           5760, 11520     /* angle1, angle2 */
        );
        XFillArc(display,
           w,
           the_GC,
           132, 331,         /* x, y */
           15, 15,         /* width, height */
           -5760, 11520     /* angle1, angle2 */
        );

        XSetForeground(display, the_GC, black_pix);
        XFillRectangle(display, w, the_GC,
           16, 332, 121, 14
        );
        XSetForeground(display, the_GC, visibility_bar_pixel);
	if(xsw_object[lplayer_object].cloak_state >= 1)
	{
	   x = xsw_object[lplayer_object].visibility -
              xsw_object[lplayer_object].cloak_strength;
	}
	else
	{
	   x = xsw_object[lplayer_object].visibility;
	}
        if(x < 0)
           x = 0;
	x *= 121;
        XFillRectangle(display, w, the_GC, 
           16, 332, x, 14
        );
        XSetForeground(display, the_GC, dark_grey_pixel);
        XDrawString(
           display, w, the_GC,
           40, 344,"VISIBILITY",strlen("VISIBILITY")
        );

        /* Intercept */ 
        if(xsw_object[lplayer_object].intercepting_object >= 0)
        {
                lx = xsw_object[lplayer_object].intercepting_object;
                sprintf(stringa,"I: %s (#%i)",
                   xsw_object[lx].name,
                   lx
                );
                XSetForeground(display, the_GC, intercept_pixel);
                XDrawString(
                   display, w, the_GC,
                   5,380,stringa,strlen(stringa)
                );    
        }


        /* Throttle */
        XSetForeground(display, the_GC, throttle_bar_pixel);
        XDrawRectangle(display, w, the_GC,
           5, 400, 15, 95
        );
        XSetForeground(display, the_GC, black_pix);
        XFillRectangle(display, w, the_GC,
           6, 401, 14, 94
        ); 
        XSetForeground(display, the_GC, throttle_bar_pixel);
        y = xsw_object[lplayer_object].throttle;
	y = ( y / 100 ) * 94;
        y = 94 - y;
        XFillRectangle(display, w, the_GC,
           6, 401 + y, 14, 94 - y
        );
	if(xsw_object[lplayer_object].thrust_rev_state == 1)
	{
           XSetForeground(display, the_GC, dark_grey_pixel);
           XDrawString(
              display, w, the_GC,
              7,492,"R",strlen("R")
           );
	}

        XSetForeground(display, the_GC, throttle_bar_pixel);
        XDrawRectangle(display, w, the_GC,
           136, 400, 15, 95
        );
        XSetForeground(display, the_GC, black_pix);
        XFillRectangle(display, w, the_GC,
           137, 401, 14, 94
        );
        XSetForeground(display, the_GC, throttle_bar_pixel);
        XFillRectangle(display, w, the_GC,
           137, 401 + y, 14, 94 - y
        );
        if(xsw_object[lplayer_object].thrust_rev_state == 1)
        {
           XSetForeground(display, the_GC, dark_grey_pixel);
           XDrawString(
              display, w, the_GC,
              139,492,"R",strlen("R")
           );
        }


        /* Heading */
	sprintf(stringa, "HDG: %3.2f'",
		(xsw_object[lplayer_object].heading * 180) / PI);
	XSetForeground(display, the_GC, status_text_pixel);
        XDrawString(
           display, w, the_GC,
           25, 415, stringa, strlen(stringa)
        );

        /* Velocity */
        XSetForeground(display, the_GC, velocity_bar_pixel);
        XDrawRectangle(display, w, the_GC,
           37, 424, 82, 15
        );
        XSetForeground(display, the_GC, black_pix);
        XFillRectangle(display, w, the_GC, 
           38, 425, 81, 14
        );
        XSetForeground(display, the_GC, velocity_bar_pixel);
	if(xsw_object[lplayer_object].velocity_max > 0)
	{
            x = ( xsw_object[lplayer_object].velocity /
                xsw_object[lplayer_object].velocity_max ) * 81;
	}
	else
	{
	    x = 0;
	}
        XFillRectangle(display, w, the_GC,
           38, 425, x, 14
        ); 
        sprintf(stringa, "VEL: %.4f",
                xsw_object[lplayer_object].velocity);  
        XSetForeground(display, the_GC, dark_grey_pixel);
        XDrawString(
           display, w, the_GC,
           39, 437, stringa, strlen(stringa)
        );

        /* X Coordinates */
        sprintf(stringa, "X: %.4f",
                xsw_object[lplayer_object].x);
        XSetForeground(display, the_GC, status_text_pixel);
        XDrawString(
           display, w, the_GC,
           25, 470, stringa, strlen(stringa)
        );

        /* Y Coordinates */
        sprintf(stringa, "Y: %.4f",
                xsw_object[lplayer_object].y);
        XSetForeground(display, the_GC, status_text_pixel);
        XDrawString(
           display, w, the_GC,
           25, 485, stringa, strlen(stringa)
        );

	/* Antimatter */
        XSetForeground(display, the_GC, antimatter_bar_pixel);
        XDrawRectangle(display, w, the_GC,
           15, 520, 122, 15
        );
        XFillArc(display,
           w,
           the_GC,
           5, 520,         /* x, y */
           15, 15,         /* width, height */
           5760, 11520     /* angle1, angle2 */
        );
        XFillArc(display,
           w,
           the_GC,
           132, 520,         /* x, y */
           15, 15,         /* width, height */
           -5760, 11520     /* angle1, angle2 */
        );
        XSetForeground(display, the_GC, black_pix);
        XFillRectangle(display, w, the_GC,
           16, 521, 121, 14
        );
        XSetForeground(display, the_GC, antimatter_bar_pixel);
	if(xsw_object[lplayer_object].antimatter_max > 0)
	{
            x = ( xsw_object[lplayer_object].antimatter /
                xsw_object[lplayer_object].antimatter_max ) * 121;
	}
	else
	{
	    x = 0;
	}
        XFillRectangle(display, w, the_GC,
           16, 521, x, 14
        );
        XSetForeground(display, the_GC, dark_grey_pixel);
        XDrawString(
           display, w, the_GC,
           41, 533, "ANTIMATTER", strlen("ANTIMATTER")
        );

        /* Damage Control */
        XSetForeground(display, the_GC, dmgctl_bar_pixel);
        XDrawRectangle(display, w, the_GC,
           15, 545, 122, 15
        );
        XFillArc(display,
           w,
           the_GC,
           5, 545,         /* x, y */
           15, 15,         /* width, height */
           5760, 11520     /* angle1, angle2 */
        );
        XFillArc(display,
           w,
           the_GC,
           132, 545,         /* x, y */
           15, 15,         /* width, height */
           -5760, 11520     /* angle1, angle2 */
        );
	if(xsw_object[lplayer_object].damage_control == 1)
	{
            XFillRectangle(display, w, the_GC,
               15, 545, 122, 15
            );

            XSetForeground(display, the_GC, black_pix);
            XDrawString(
               display, w, the_GC,
               28, 558, "DAMAGE CONTROL", strlen("DAMAGE CONTROL")
            );
	}
	else
	{
            XSetForeground(display, the_GC, black_pix);
            XFillRectangle(display, w, the_GC,
               16, 546, 121, 14
	    );

            XSetForeground(display, the_GC, dmgctl_bar_pixel);
            XDrawString(
               display, w, the_GC,
               28, 557, "DAMAGE CONTROL", strlen("DAMAGE CONTROL")
            );
	}





	return(0);
}



int DrawSensorReadout(Window w)
{
	/* Local variables. */
	char stringa[256];
	long locked_object;
        double x, y, z;
        long lx, ly, lz;
        long counta;


	/* Clear Window w. */
        XClearWindow(display, w);


        /* lplayer_object must be valid. */
        if((lplayer_object < 0) || (lplayer_object >= MAX_OBJECTS))
                return(-1);


	/* Get locked_object. */
	locked_object = xsw_object[lplayer_object].locked_on;


	/* Make sure locked_object is valid. */
	if(locked_object < 0)
		return(-1);

        if(locked_object >= MAX_OBJECTS)
                return(-1);


	/* ******************************************************** */
	/* ***                 Begin Drawing                    *** */

	/* Name. */
        XSetForeground(display, the_GC, normal_text_pixel);
	sprintf(stringa, "%s(#%i)",
		xsw_object[locked_object].name,
		locked_object
	);
        XDrawString(
           display, w, the_GC,
           8, 20, stringa, strlen(stringa)
        );

	/* Distance. */
	XSetForeground(display, the_GC, bold_text_pixel);
	x = xsw_object[lplayer_object].x - xsw_object[locked_object].x;
	y = xsw_object[lplayer_object].y - xsw_object[locked_object].y;
	x = Mu3DDistance(x, y, 0);
	if(xsw_object[lplayer_object].scanner_range > 0)
	{
	    x = x / xsw_object[lplayer_object].scanner_range;
	}
	else
	{
	    x = 0;
	}
	x = (x * 135) + 8;
	XDrawLine(display, w, the_GC,
           x, 28, x, 40);
        XSetForeground(display, the_GC, normal_text_pixel);
	XDrawLine(display, w, the_GC,
	   8, 25, 8, 40);
        XDrawLine(display, w, the_GC,
           143, 25, 143, 40);
        XDrawLine(display, w, the_GC,
           8, 40, 143, 40);


	/* Hull. */
        XSetForeground(display, the_GC, hull_bar_pixel);
        XDrawRectangle(display, w, the_GC,
           15, 70, 122, 15
        );
        XFillArc(display,
           w,
           the_GC,
           5, 70,         /* x, y */
           15, 15,         /* width, height */
           5760, 11520     /* angle1, angle2 */
        );
        XFillArc(display,
           w,   
           the_GC,
           132, 70,         /* x, y */
           15, 15,         /* width, height */
           -5760, 11520     /* angle1, angle2 */
        );
        XSetForeground(display, the_GC, black_pix);
        XFillRectangle(display, w, the_GC,
           16, 71, 121, 14
        );
        XSetForeground(display, the_GC, hull_bar_pixel);
	if(xsw_object[locked_object].hp_max > 0)
	{
            x = ( xsw_object[locked_object].hp /
                xsw_object[locked_object].hp_max ) * 121;
	}
	else
	{
	    x = 0;
	}
        XFillRectangle(display, w, the_GC,
           16, 71, x, 14
        );
        XSetForeground(display, the_GC, dark_grey_pixel);
        XDrawString(
           display, w, the_GC,
           63, 83, "HULL", strlen("HULL")
        ); 

        /* Power */
        XSetForeground(display, the_GC, power_bar_pixel);
        XDrawRectangle(display, w, the_GC,
           15, 93, 122, 15
        );
        XFillArc(display,
           w,
           the_GC,
           5, 93,          /* x, y */
           15, 15,         /* width, height */
           5760, 11520     /* angle1, angle2 */
        );
        XFillArc(display,
           w,
           the_GC,
           132, 93,         /* x, y */
           15, 15,          /* width, height */
           -5760, 11520     /* angle1, angle2 */
        );
        XSetForeground(display, the_GC, black_pix);
        XFillRectangle(display, w, the_GC,
           16, 94, 121, 14
        );
        XSetForeground(display, the_GC, power_bar_pixel);
	if(xsw_object[locked_object].power_max > 0)
	{
            x = ( xsw_object[locked_object].power /
                xsw_object[locked_object].power_max ) * 121;
	}
	else
	{
	    x = 0;
	}
        XFillRectangle(display, w, the_GC,
           16, 94, x, 14
        );
        XSetForeground(display, the_GC, dark_grey_pixel);
        XDrawString(
           display, w, the_GC,
           56, 106, "POWER", strlen("POWER")
        );

        /* Shields */
        if(xsw_object[locked_object].shield_state == 0)
        {
                XSetForeground(display, the_GC, shields_bar_pixel);
                XDrawRectangle(display, w, the_GC,
                   15, 116, 122, 15);
                XFillArc(display,
                   w,
                   the_GC,
                   5, 116,         /* x, y */
                   15, 15,         /* width, height */
                   5760, 11520     /* angle1, angle2 */
               );
               XFillArc(display,
                   w,
                   the_GC,
                   132, 116,        /* x, y */
                   15, 15,          /* width, height */
                   -5760, 11520     /* angle1, angle2 */
                );
        
                XSetForeground(display, the_GC, black_pix);
                XFillRectangle(display, w, the_GC,
                   16, 117, 121, 14);
                sprintf(stringa, "SH: DOWN");
                XSetForeground(display, the_GC, shields_bar_pixel);
                XDrawString(
                        display, w, the_GC,
                        20,128,stringa,strlen(stringa)
                );
        }
        else
        {
                XSetForeground(display, the_GC, shields_bar_pixel);
                XFillRectangle(display, w, the_GC,
                   15, 116, 122, 15);
                XFillArc(display,
                   w,
                   the_GC,
                   5, 116,         /* x, y */
                   15, 15,         /* width, height */
                   5760, 11520     /* angle1, angle2 */
               );
               XFillArc(display,
                   w,
                   the_GC,  
                   132, 116,         /* x, y */
                   15, 15,           /* width, height */  
                   -5760, 11520      /* angle1, angle2 */
                );
                sprintf(stringa, "SH: UP");
                XSetForeground(display, the_GC, dark_grey_pixel);  
                XDrawString(
                        display, w, the_GC,
                        20,128,stringa,strlen(stringa)
                );   
        }

        /* X Coordinates */
        sprintf(stringa, "X: %.4f",
                xsw_object[locked_object].x);
        XSetForeground(display, the_GC, status_text_pixel);
        XDrawString(
           display, w, the_GC,
           8, 150, stringa, strlen(stringa)
        );
           
        /* Y Coordinates */
        sprintf(stringa, "Y: %.4f",
                xsw_object[locked_object].y);
        XSetForeground(display, the_GC, status_text_pixel);
        XDrawString(
           display, w, the_GC,
           8, 170, stringa, strlen(stringa)
        );

        /* Visibility */
        XSetForeground(display, the_GC, visibility_bar_pixel);
        XDrawRectangle(display, w, the_GC,
           15, 231, 122, 15
        );
        XFillArc(display,
           w,
           the_GC,
           5, 231,         /* x, y */
           15, 15,         /* width, height */
           5760, 11520     /* angle1, angle2 */
        ); 
        XFillArc(display,
           w,
           the_GC,
           132, 231,         /* x, y */
           15, 15,           /* width, height */
           -5760, 11520      /* angle1, angle2 */
        );
          
        XSetForeground(display, the_GC, black_pix);
        XFillRectangle(display, w, the_GC,
           16, 232, 121, 14   
        );
        XSetForeground(display, the_GC, visibility_bar_pixel);
        if(xsw_object[locked_object].cloak_state >= 1)
        {
           x = xsw_object[locked_object].visibility -
              xsw_object[locked_object].cloak_strength;
        }
        else
        {
           x = xsw_object[locked_object].visibility;
        }
        if(x < 0)
           x = 0;
        x *= 121;
        XFillRectangle(display, w, the_GC,
           16, 232, x, 14
        );
        XSetForeground(display, the_GC, dark_grey_pixel);
        XDrawString(
           display, w, the_GC,
           40, 244,"VISIBILITY",strlen("VISIBILITY")
        );





        /* Antimatter */
        XSetForeground(display, the_GC, antimatter_bar_pixel);
        XDrawRectangle(display, w, the_GC,
           15, 400, 122, 15
        );
        XFillArc(display,
           w,
           the_GC,
           5, 400,         /* x, y */
           15, 15,         /* width, height */
           5760, 11520     /* angle1, angle2 */
        );
        XFillArc(display,
           w,
           the_GC,
           132, 400,        /* x, y */
           15, 15,          /* width, height */
           -5760, 11520     /* angle1, angle2 */
        );
        XSetForeground(display, the_GC, black_pix);
        XFillRectangle(display, w, the_GC,
           16, 401, 121, 14
        );
        XSetForeground(display, the_GC, antimatter_bar_pixel);
	if(xsw_object[locked_object].antimatter_max > 0)
	{
            x = ( xsw_object[locked_object].antimatter /
                xsw_object[locked_object].antimatter_max ) * 121;
	}
	else
	{
	    x = 0;
	}
        XFillRectangle(display, w, the_GC,
           16, 401, x, 14
        );
        XSetForeground(display, the_GC, dark_grey_pixel);
        XDrawString(
           display, w, the_GC,
           41, 412, "ANTIMATTER", strlen("ANTIMATTER")
        );
 









	return(0);
}



int DrawToplevel()
{
	XClearWindow(display, toplevel);

	XSetWindowBackground(display,
                toplevel,
                dark_grey_pixel
        );


	return(0);
}
